//==========================================================================
//
//	title		: define functions to access hardware
//	company		: YAMAHA
//	author		: Taichi Sugiyama
//	create Data : 30/Sep/99
//
//==========================================================================
#include "define.h"


//#define	TRACE_HW
#ifdef TRACE_HW
#define	DPF(X)		cprintf("TRACE_HW:");cprintf X
#define	dprintf(X)	cprintf X
#else
#define	DPF(X)
#define	dprintf(X)
#endif


//==========================================================================
//	read BYTE native register
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//
//	return
//		read BYTE value
//
//==========================================================================
BYTE	Hw_ReadRegByte(PHWINFO phwi, DWORD dwOffset)
{
	return *(phwi->lpbNativeReg + dwOffset);
}


//==========================================================================
//	read WORD native register
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//
//	return
//		read WORD value
//
//==========================================================================
WORD	Hw_ReadRegWord(PHWINFO phwi, DWORD dwOffset)
{
	return *(phwi->lpwNativeReg + (dwOffset >> 1));
}


//==========================================================================
//	read DWORD native register
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//
//	return
//		read DWORD value
//
//==========================================================================
DWORD	Hw_ReadRegDword(PHWINFO phwi, DWORD dwOffset)
{
	return *(phwi->lpdNativeReg + (dwOffset >> 2));
}


//==========================================================================
//	write BYTE native register
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//		bData		: data
//
//	return
//
//==========================================================================
void	Hw_WriteRegByte(PHWINFO phwi, DWORD dwOffset, BYTE bData)
{
	*(phwi->lpbNativeReg + dwOffset) = bData;
}


//==========================================================================
//	write WORD native register
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//		wData		: data
//
//	return
//
//==========================================================================
void	Hw_WriteRegWord(PHWINFO phwi, DWORD dwOffset, WORD wData)
{
	*(phwi->lpwNativeReg + (dwOffset >> 1)) = wData;
}


//==========================================================================
//	write primary CODEC
//
//	argument
//		phwi		: pointer to hardware information
//		dwOffset	: offset address
//		dwData		: data
//
//	return
//
//==========================================================================
void	Hw_WriteRegDword(PHWINFO phwi, DWORD dwOffset, DWORD dwData)
{
	*(phwi->lpdNativeReg + (dwOffset >> 2)) = dwData;
}


//==========================================================================
//	AC97 link reset
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//
//==========================================================================
void	Hw_AC97LinkReset(PHWINFO phwi)
{
	BYTE	bCmd;


	bCmd = PCIReadConfigByte(&phwi->pcikey, PCIR_DSXGCTRL);
	if(bCmd & 0x03) {
		PCIWriteConfigByte(&phwi->pcikey, PCIR_DSXGCTRL, (BYTE)(bCmd & 0xFC));
		PCIWriteConfigByte(&phwi->pcikey, PCIR_DSXGCTRL, (BYTE)(bCmd | 0x03));
		PCIWriteConfigByte(&phwi->pcikey, PCIR_DSXGCTRL, (BYTE)(bCmd & 0xFC));
	}
}


//==========================================================================
//	check primary busy
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//		if busy, return 1
//
//==========================================================================
BYTE	Hw_CheckPrimaryBusy(PHWINFO phwi)
{
	DWORD	nCount = 0L;

	while(++nCount < YDSXG_AC97TIMEOUT) {
		if((Hw_ReadRegWord(phwi, YDSXGR_PRISTATUSADR) & 0x8000) == 0x0000)
			return 0;
	}
	return 1;
}


//==========================================================================
//	check secondary busy
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//		if busy, return 1
//
//==========================================================================
BYTE	Hw_CheckSecondaryBusy(PHWINFO phwi)
{
	DWORD	nCount = 0L;

	while(++nCount < YDSXG_AC97TIMEOUT) {
		if((Hw_ReadRegWord(phwi, YDSXGR_SECSTATUSADR) & 0x8000) == 0x0000)
			return 0;
	}
	return 1;
}


//==========================================================================
//	read primary CODEC
//
//	argument
//		phwi	: pointer to hardware information
//		wOffset	: offset address
//
//	return
//		read value
//
//==========================================================================
WORD	Hw_ReadPrimaryCodec(PHWINFO phwi, WORD wOffset)
{
	DWORD	i;

	Hw_WriteRegWord(phwi, YDSXGR_AC97CMDADR, (WORD)(YDSXG_AC97READCMD | wOffset));
	if(Hw_CheckPrimaryBusy(phwi))
		return YDSXG_AC97READFALSE;
	if((phwi->wDeviceID == YMF744) && (phwi->bRevisionID < 2)) {
		for(i=0; i<600; i++)
			Hw_ReadRegWord(phwi, YDSXGR_PRISTATUSDATA);
	}
	return Hw_ReadRegWord(phwi, YDSXGR_PRISTATUSDATA);
}


//==========================================================================
//	write primary CODEC
//
//	argument
//		phwi	: pointer to hardware information
//		wOffset	: offset address
//		wData	: data
//
//	return
//		if error. return not 0
//
//==========================================================================
BYTE	Hw_WritePrimaryCodec(PHWINFO phwi, WORD wOffset, WORD wData)
{
	DWORD	dwCmd;

	dwCmd = (DWORD)(YDSXG_AC97WRITECMD | wOffset);
	dwCmd <<= 16;
	dwCmd |= (DWORD)wData;
	Hw_WriteRegDword(phwi, YDSXGR_AC97CMDDATA, dwCmd);
	if (Hw_CheckPrimaryBusy(phwi))
		return 1;
	return 0;
}


//==========================================================================
//	read primary CODEC
//
//	argument
//		phwi	: pointer to hardware information
//		wOffset	: offset address
//
//	return
//		read value
//
//==========================================================================
WORD	Hw_ReadSecondaryCodec(PHWINFO phwi, WORD wOffset)
{
	WORD	wCmd;
	DWORD	i;

	if((phwi->bCodecID == 0) || (phwi->bCodecID > 3))
		return YDSXG_AC97READFALSE;
	wCmd = (WORD)(YDSXG_AC97READCMD | ((WORD)(phwi->bCodecID & 0x03) << 8) | wOffset);
	Hw_WriteRegWord(phwi, YDSXGR_AC97CMDADR, wCmd);
	if(Hw_CheckSecondaryBusy(phwi))
		return YDSXG_AC97READFALSE;
	if((phwi->wDeviceID == YMF744) && (phwi->bRevisionID < 2)) {
		for(i=0; i<600; i++)
			Hw_ReadRegWord(phwi, YDSXGR_SECSTATUSDATA);
	}
	return Hw_ReadRegWord(phwi, YDSXGR_SECSTATUSDATA);
}


//==========================================================================
//	write primary CODEC
//
//	argument
//		phwi	: pointer to hardware information
//		wOffset	: offset address
//		wData	: data
//
//	return
//		if error. return not 0
//
//==========================================================================
BYTE	Hw_WriteSecondaryCodec(PHWINFO phwi, WORD wOffset, WORD wData)
{
	DWORD	dwCmd;

	if((phwi->bCodecID == 0) || (phwi->bCodecID > 3))
		return 1;
	dwCmd = (DWORD)(YDSXG_AC97WRITECMD | ((WORD)(phwi->bCodecID & 0x03) << 8) |wOffset);
	dwCmd <<= 16;
	dwCmd |= (DWORD)wData;
	Hw_WriteRegDword(phwi, YDSXGR_AC97CMDDATA, dwCmd);
	if (Hw_CheckSecondaryBusy(phwi))
		return 1;
	return 0;
}


//==========================================================================
//	start
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//
//==========================================================================
void	Hw_Start(PHWINFO phwi)
{
	DWORD	dwMapOfRec = 0x00000000;
	DWORD	dwMapOfEffect = 0x00000000;
	DWORD	dwMode = 0x00000003;

	DPF(("Hw_Start"));

	//---- rec -------------------------------------------------------------
	if(phwi->dwHwStatus & HWDEF_REC)
		dwMapOfRec |= 0x00000001;
	if(phwi->dwHwStatus & HWDEF_ADC)
		dwMapOfRec |= 0x00000002;

	//---- AC3 -------------------------------------------------------------
	if(phwi->dwHwStatus & HWDEF_AC3) {
		dwMapOfEffect |= 0x00000018;
		dwMode |= 0x40000000;
	}

	if(dwMapOfRec)
		Hw_WriteRegDword(phwi, YDSXGR_MAPOFREC, dwMapOfRec);

	if(dwMapOfEffect)
		Hw_WriteRegDword(phwi, YDSXGR_MAPOFEFFECT, dwMapOfEffect);

	Hw_WriteRegDword(phwi, YDSXGR_MODE, dwMode);

	dprintf(("/End\n\r"));
}


//==========================================================================
//	stp
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//
//==========================================================================
void	Hw_Stop(PHWINFO phwi)
{
	DWORD	dwtmp;
	DWORD	nCount = 0L;

	DPF(("Hw_Stop"));

	Hw_WriteRegDword(phwi, YDSXGR_MODE, 0x00000000);
	Hw_WriteRegDword(phwi, YDSXGR_MAPOFREC, 0x00000000);
	Hw_WriteRegDword(phwi, YDSXGR_MAPOFEFFECT, 0x00000000);

	while(++nCount < YDSXG_WORKBITTIMEOUT) {
		dwtmp = Hw_ReadRegDword(phwi, YDSXGR_STATUS);
		if((dwtmp & 0x00000002) == 0x00000000)
			break;
	}

	dprintf(("/End\n\r"));
}


//==========================================================================
//	enable DSP
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//
//==========================================================================
void	Hw_EnableDSP(PHWINFO phwi)
{
	DPF(("Hw_EnableDSP"));

	Hw_WriteRegDword(phwi, YDSXGR_CONFIG, 0x00000001);

	dprintf(("/End\n\r"));
}


//==========================================================================
//	disable DSP
//
//	argument
//		phwi	: pointer to hardware information
//
//	return
//
//==========================================================================
void	Hw_DisableDSP(PHWINFO phwi)
{
	DWORD	dwtmp;
	DWORD	nCount = 0L;

	DPF(("Hw_DisableDSP"));

	dwtmp = Hw_ReadRegDword(phwi, YDSXGR_CONFIG);
	if (dwtmp) {
		Hw_WriteRegDword(phwi, YDSXGR_CONFIG, 0x00000000);
	}
	while(++nCount < YDSXG_WORKBITTIMEOUT) {
		dwtmp = Hw_ReadRegDword(phwi, YDSXGR_STATUS);
		if((dwtmp & 0x00000002) == 0x00000000)
			break;
	}

	dprintf(("/End\n\r"));
}
