//*****************************************************************************
//* Name:
//*      1212cfg.h
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file contains private configuration function declarations and
//*      constants.
//*
//* Modification Log:
//*
//*      1.9   08/20/97 Bill
//*      (v 1.0B6) Added Auto Sync Rec/Play data member.
//*
//*      1.8   08/05/97 Bill
//*      (v 1.0B5) Added idleMonitorOn member for tracking whether
//*      the card is in idle monitor mode.  (For making sure we don't
//*      try to update the card's ADC sensitivity while in monitor mode).
//*
//*      1.7   06/17/97 Bill
//*      (v 1.11) Added curRecBufNum.
//*
//*      1.6   06/13/97 Bill
//*      (v 1.09) Added numMsPerInt data member for wave position estimation.
//*
//*      1.5   06/03/97 Bill
//*      (v 1.05) Added fileHandle data member for debug messages.
//*
//*      1.4   03/14/97 Bill
//*      Added data members for wave device driver support.
//*
//*      1.3   12/27/96 Bill
//*      Modified to support multiple cards.
//*
//*      1.2   11/15/96 Bill
//*      Name changed to 1212cfg.h.
//*
//*      1.1   11/13/96 Bill
//*      Initial version created.
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#ifndef K1212CFG_H
#define K1212CFG_H

#ifndef  KORG1212_H
#include "KORG1212.h"
#endif
#ifndef  K1212IFPR_H
#include "1212ifpr.h"
#endif
#ifndef  K1212WAVESTRUCT_H
#include "1212wavestruct.h"
#endif


// -------------------------------------------------------------------------
// constants
// -------------------------------------------------------------------------
#define K1212_VERSION ((KORG1212_Major << 8) | KORG1212_Minor)
#define MAX_REGPATH_SIZE 256
#define K1212_SETTINGS_STRING "Settings"


// -------------------------------------------------------------------------
// Card API mode enumeration.
// -------------------------------------------------------------------------
typedef enum {
   K1212_NATIVE_API_MODE,
   K1212_WAVEDEVICE_API_MODE
} APIMode;


// ----------------------------------------------------------------------------
// the card configuration structure contains the configuration data provided
// by the configuration manager for the card.
// ----------------------------------------------------------------------------
typedef struct cardConfigStruct {
   DEVNODE           devNode;       // configuration manager's device node identifier for the
                                    //    card.  A value of zero indicates the card is not
                                    //    configured.
   DWORD             busNum;        // bus number that this card resides on
   DWORD             deviceNum;     // device number of the card on busNum

   char              registryPath[MAX_REGPATH_SIZE];

   THREADHANDLE      fillThreadHdl; // fill routine thread handle for boosting its priority.
   PVOID             fillFunc;      // this is a pointer to the ring 3 fill routine function.
   DWORD             fillParam;     // parameter to pass to the fill function

   APIHandle         apiHandle;     // identifies the API object that opened this card.
   APIMode           apiMode;       // indicates which mode the card is in - wave or custom

   int               irqNum;        // hardware IRQ number for our card
   int               irqFlags;      // properties of our IRQ reported by the config mgr.
   ULONG             irqHandle;     // handle assigned to our irq handler
   ULONG             irqCount;      // number of IRQs received from the card

   DWORD             curFillBufNum; // contains the buffer number to be filled by the fill
                                    //    routines
   DWORD             curRecBufNum;  // contains the buffer number last filled by the card
   char*             physMemBase;   // start of physical memory mapped space of the card
   char*             memBase;       // linear address referencing the phys memory mapped space
   WORD              ioBase;        // start of the I/O space for the card
   DWORD             memLength;     // memory space length reported by config mgr

   cardIntfRegs      intfRegs;      // structure containing pointers to the card's interface registers

   DWORD             cmdRetryCount; // tracks how many times we have retried sending to the
                                    //    card.
   MEMHANDLE         dspMemBufHdl;  // handle for the DSP microcode downloaded to the card
   char*             dspMemBufPtr;  // linear address of the DSP code
   char*             dspPhysBufPtr; // physical address of the DSP code

   CardState         state;         // current state of the card
   DWORD             openCount;     // keeps track of how many times the card has been opened.
                                    //    this is used as a simple method of providing unique
                                    //    card handles to applications.
   BOOL              idleMonitorOn; // indicates whether the card is in idle monitor mode.

   MEMHANDLE         sharedBufHdl;  // handle to the shared memory block
   KorgSharedBuffer* sharedBufPtr;  // points to mem block containing the record and playback
                                    //    buffers for the card.
   KorgSharedBuffer* sharedBufPhys; // physical address of the above buffer

   DWORD             numMsPerInt;   // rounded off number of milliseconds per interrupt from the
                                    //    card - based on clock rate.
   ClockSourceIndex  clkSrcRate;    // sample rate and clock source
   WORD              leftADCInSens; // ADC left channel input sensitivity
   WORD              rightADCInSens;// ADC right channel input sensitivity

   // ----------------------------------------------------------------------------
   // Every card contains k1212NumWaveDevices wave devices.  These wave devices
   // can be used when the card has been placed in wave device mode.  (By
   // successfully opening a wave device through the protected mode interface.)
   // ----------------------------------------------------------------------------
   WORD               masterVolume;       // master volume level for all wave out devices.

   HANDLE             fileHandle;         // v1.05 - file for posting debugging messages
   k1212WaveInDevice  waveInDevice [k1212NumWaveDevices];
   k1212WaveOutDevice waveOutDevice[k1212NumWaveDevices];

   BOOL               autoWaveSyncMode;   // v1.0B6 - when true, overrides individual sync
                                          //    record/playback settings.
} cardConfig;

// -------------------------------------------------------------------------
// structure for accessing bus information
// -------------------------------------------------------------------------
typedef struct	PCIAccess_s
					{
						BYTE    bBusNumber;     // Bus no 0-255
						BYTE    bDevFuncNumber; // Device # in bits 7:3 and
														// Function # in bits 2:0
						WORD    wPCIReserved;   //
					} sPCIAccess;

// -------------------------------------------------------------------------
// private cardData mutator functions
// -------------------------------------------------------------------------
void initCardData       (DWORD cardIndex);
void setDevNode         (DWORD cardIndex, DEVNODE      devNode);
void setBusNum          (DWORD cardIndex, DWORD        busNum);
void setRegistryPath    (DWORD cardIndex, const char*  regPath);
void setDeviceNum       (DWORD cardIndex, DWORD        deviceNum);
void setFillThreadHdl   (DWORD cardIndex, THREADHANDLE fillThreadHdl);
void setFillFunc        (DWORD cardIndex, PVOID        fillFunc);
void setFillParam       (DWORD cardIndex, DWORD        fillParam);
void setApiHandle       (DWORD cardIndex, APIHandle    apiHandle);
void setApiMode         (DWORD cardIndex, APIMode      apiMode);
void setIrqNum          (DWORD cardIndex, int          iIrqNum);
void setIrqHandle       (DWORD cardIndex, ULONG        iIrqHandle);
void setPhysMemBase     (DWORD cardIndex, char*        pMemBase);
void setMemBase         (DWORD cardIndex, char*        pMemBase);
void setIoBase          (DWORD cardIndex, WORD         wIoBase);
void setMemLength       (DWORD cardIndex, DWORD        dMemLength);
void setDSPMemHdl       (DWORD cardIndex, MEMHANDLE    dspMemHdl);
void setDSPMemPtr       (DWORD cardIndex, char*        dspMemPtr);
void setDSPPhysMemPtr   (DWORD cardIndex, char*        dspPhysMemPtr);
void incIrqCount        (DWORD cardIndex);
void clearIrqCount      (DWORD cardIndex);
void clearCmdRetryCount (DWORD cardIndex);
void setSharedBufferPtr (DWORD cardIndex, KorgSharedBuffer* bufPtr);
void setSharedPhyBufPtr (DWORD cardIndex, KorgSharedBuffer* bufPtr);
void setFileHandle      (DWORD cardIndex, HANDLE fileHandle);

// -------------------------------------------------------------------------
// private cardData accessor functions
// -------------------------------------------------------------------------
const char*  getRegistryPath (DWORD cardIndex);
char*        getRegPathPtr   (DWORD cardIndex);
char*        getPhysMemBase  (DWORD cardIndex);
MEMHANDLE    getDSPMemHdl    (DWORD cardIndex);
char*        getDSPMemPtr    (DWORD cardIndex);
ULONG        getIrqHandle    (DWORD cardIndex);
THREADHANDLE getFillThreadHdl(DWORD cardIndex);
PVOID        getFillFunc     (DWORD cardIndex);
DWORD        getFillParam    (DWORD cardIndex);
APIHandle    getApiHandle    (DWORD cardIndex);
APIMode      getApiMode      (DWORD cardIndex);

// -------------------------------------------------------------------------
// private configuration functions
// -------------------------------------------------------------------------
CONFIGRET   startConfiguration(DEVNODE devNode);  // sets the configuration data
                                                  // for the card specified by devNode

CONFIGRET   RebootCard(DWORD cardIndex);          // sends the reboot command and schedules
                                                  //  the finishConfiguration function.

void __stdcall finishConfiguration(VMHANDLE       hVM,   // finishes the configuration
                                   PCLIENT_STRUCT pcrs,  // after the card has been reset
                                   PVOID          refData,
                                   DWORD          extra);

void        freeDSPucodeImageMemory(DWORD cardIndex);    // makes sure the DSP ucode image
                                                         //  memory gets freed.
CONFIGRET   stopConfiguration(DEVNODE devNode);   // makes sure resources allocated for the
                                                  //    card specified by devNode are freed
CONFIGRET   hookCardIRQ(DWORD cardIndex,
                        int irqNum);              // hooks the card interrupt

void      InitSettings(DWORD cardIndex);          // inits vol/routing tables & sensitivity.
void      SaveCardSettings(DWORD cardIndex);      // saves card settings to the registry
BOOL      RetrieveCardSettings(DWORD cardIndex);  // gets card settings from the registry
void      SaveWaveSettings(DWORD cardIndex);      // saves wave device settings to the registry
BOOL      RetrieveWaveSettings(DWORD cardIndex);  // gets wave device settings from the registry

// -------------------------------------------------------------------------
// utility functions
// -------------------------------------------------------------------------
DWORD             getBufferCount(void);
DWORD             HexStrToInt(char* hexStr);
void              charTo2HexChars(char* lpszTwoHexChars, char decValue);
CONFIGRET         ParseDevIdString(char* devIdBuffer,
                                   DWORD* busNum,
                                   DWORD* deviceNum
                  );

#endif // K1212CFG_H
