/*******************************************************************/
/* Copyright 1995 Crystal Semiconductor Corp., ALL RIGHTS RESERVED */
/*   Program name: WAVE.EXE                                        */
/*   Purpose:      Wave Play/Record with DMA or PIO.               */
/*                 Reads/Writes WAVE format files direct to disk   */
/*                 Unlimited length.                               */
/*   Written by:   Richard Vail for Crystal Semiconductor          */
/*   History: 1.0  02/17/95: Created by Dick Vail                  */
/*            1.1  03/10/95: Added TIO - timer int driven pio      */
/*            1.2  04/06/95: Added volume hot keys                 */
/*            1.2  04/06/95: Added FFT display                     */
/*            1.2  04/06/95: Fixed bug in detection code           */
/*            1.5  06/21/95: Fix EQ lookup table                   */
/*                                                                 */
/*******************************************************************/

#include <ctype.h>
#include <io.h>
#include <fcntl.h>
#include <conio.h>
#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <graph.h>

typedef unsigned char BYTE;
typedef unsigned int  WORD;
typedef unsigned long DWORD;

#define PI (float)3.1415926
#define _2PI (float)2.0 * PI
#define BARSIZE 16
#define MAXBARS 32

extern int FFTpower; 	// power(2,?) = FFTsize
extern int FFTsize;		// Number of points in FFT
extern int EQNumBars;		// Number of bars in EQ display
extern int samples_per_buffer;

extern int snapshot;
extern int debug;
extern void ExitErrorMsg(char *);
extern struct _videoconfig vc;

void CreateCosTable(void);
void CreateSinTable(void);
void CreateWinTable(void);
void CreateBitTable(void);
void CreateEQBinTable(void);

long _far *FFTrData; //Real Data Array
long _far *FFTiData;	//Imaginary Data Array
long _far *FFTWindow;//Lookup Window
int _far *BitReverse;//dynamically allocated and initialized by CreateBitTable
int _far *CosLookup;	//dynamically allocated and initialized by CreateCosTable
int _far *SinLookup;	//dynamically allocated and initialized by CreateSinTable

int EQBinTable[MAXBARS][2];	//initialized dynamically by CreateEQBinTable
BYTE EQlast[MAXBARS];

BYTE EQLookup[16][16] =
{ 0,  0,  3,  5,  6,  8,  8,  9, 10, 10, 11, 11, 12, 12, 12, 13,
  0,  0,  4,  5,  7,  8,  8,  9, 10, 10, 11, 11, 12, 12, 13, 13,
  3,  4,  5,  6,  7,  8,  9,  9, 10, 10, 11, 11, 12, 12, 13, 13,
  5,  5,  6,  7,  8,  8,  9, 10, 10, 11, 11, 12, 12, 12, 13, 13,
  6,  7,  7,  8,  8,  9,  9, 10, 10, 11, 11, 12, 12, 12, 13, 13,
  8,  8,  8,  8,  9,  9, 10, 10, 11, 11, 11, 12, 12, 12, 13, 13,
  8,  8,  9,  9,  9, 10, 10, 10, 11, 11, 12, 12, 12, 13, 13, 13,
  9,  9,  9, 10, 10, 10, 10, 11, 11, 12, 12, 12, 12, 13, 13, 13,
 10, 10, 10, 10, 10, 11, 11, 11, 11, 12, 12, 12, 13, 13, 13, 13,
 10, 10, 10, 11, 11, 11, 11, 12, 12, 12, 12, 13, 13, 13, 13, 14,
 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 13, 13, 13, 13, 13, 14,
 11, 11, 11, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 14, 14,
 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 14, 14, 14,
 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 14, 14, 14, 14,
 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 14, 14, 14, 14, 14,
 13, 13, 13, 13, 13, 13, 13, 13, 13, 14, 14, 14, 14, 14, 14, 15
};

/******************************************************************/
void CreateEQBinTable(void)
{	int i;
	int start, end, inc;

	start=4;
	end=(FFTsize/2)-1;
	inc=0;
	for (i=0; i<MAXBARS; i++)
	{	if ((EQBinTable[i][0] = start + inc) > end)
			EQBinTable[i][0] = end;
		inc += ((FFTsize/FFTpower+2)/(MAXBARS-i))-1;
		if ((EQBinTable[i][1] = start + inc) > end)
			EQBinTable[i][1] = end;
		start++;
	}
	EQBinTable[MAXBARS-1][1] = end;//all the rest
	if (debug==-1)
		for (i=0; i<MAXBARS; i++)
			printf("\nI%d=%d,%d ", i, EQBinTable[i][0], EQBinTable[i][1]);
}

/******************************************************************/
// Cosine Lookup Table
// C = CosLookup[i] replaces:
// IBITR = BitReverse[i];
// ARG = 2 * pi / N   * (float)IBITR;
// C = cos(ARG) * 256;
void CreateCosTable(void)
{  int i;
	float iBtr;
	float arg;
	int BitRevFunction(int j);

	if ((CosLookup = (int _far *) _halloc((long)FFTsize, sizeof(int))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Cosine Lookup table.\n");

	for (i = 0; i < FFTsize; i++)
	{
		iBtr = (float)BitRevFunction(i);
		arg = (_2PI / (float)FFTsize	) * iBtr;
		CosLookup[i] = (int)(cos(arg) * 256.0);
	}
}

/******************************************************************/
// Sine Lookup Table
// S = SinLookup[i] replaces:
// IBITR = BitReverse[i];
// ARG = 2 * pi / N   * (float)IBITR;
// C = sin(ARG) * 256;
void CreateSinTable(void)
{  int i;
	float iBtr;
	float arg;
	int BitRevFunction(int j);

	if ((SinLookup = (int _far *) _halloc((long)FFTsize, sizeof(int))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Sin Lookup table.\n");

	for (i = 0; i < FFTsize; i++)
	{
		iBtr = (float)BitRevFunction(i);
		arg = (_2PI / (float)FFTsize	) * iBtr;
		SinLookup[i] = (int)(sin(arg) * 256.0);
	}
}

/******************************************************************/
// Blackman 4-term window for a ?? point FFT.
// Equation found on page 64 of Fredric J. Harris's
// paper 'On the use of Windows for Harmonic Analysis
// with Discrete Fourier Transform'
// NOTE: The window value of 1 was converted to 32767
// since 32768 is too big for an integer
void CreateWinTable(void)
{	int i;
	float value;

#define A0 (float)0.35875
#define A1 (float)0.48829
#define A2 (float)0.14128
#define A3 (float)0.01168

	if ((FFTWindow = (long _far *) _halloc((long)FFTsize, sizeof(long))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Window table.\n");

	// This window is from the paper "On the use of windows for harmonic
	// analysis with the Descrete fourier Transform".  The window is the
	// Blackman-Harris function found on page 66

	for (i = 0; i < FFTsize; i++)
	{
		value = A0 - (A1*(float)cos((_2PI/FFTsize)*i)) + (A2*(float)cos((_2PI/FFTsize)*(float)2.0*i)) - (A3*(float)cos((_2PI/FFTsize)*(float)3.0*i));
		FFTWindow[i] = (long)((float)32767.0 * value);
	}
}

/******************************************************************/
// Bit reverse lookup table for FFT.
// BitReverse[i] equals the bit reverse of i.
void CreateBitTable(void)
{	int i;
	int BitRevFunction(int j);

	if ((BitReverse = (int _far *) _halloc((long)FFTsize, sizeof(int))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Bit Reverse table.\n");

	for (i = 0; i < FFTsize; i++)
	{
		BitReverse[i] = BitRevFunction(i);
	}
}

/******************************************************************/
int BitRevFunction(int j)
{	int i;
	int J1;
	int iBtr;
	int Pow1;
	int Pow2;

	J1 = j;
	iBtr = 0;
	for (i = 0; i < FFTpower; i++)
	{
		Pow1 = (int)pow(2, i);
		Pow2 = (int)pow(2, FFTpower-1 - i);
		if (J1 & Pow1) iBtr = iBtr + Pow2;
	}

	return (iBtr);
};

/******************************************************************/
void FFTalloc(void)
{	int i;

	if ((FFTrData = (long _far *) _halloc((long)FFTsize, sizeof(long))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Real array.\n");
	if ((FFTiData = (long _far *) _halloc((long)FFTsize, sizeof(long))) == NULL)
		ExitErrorMsg("\nFFT Error: No memory available for Imaginary array.\n");
	for (i = 0; i < FFTsize; i++)
	{	FFTiData[i] = 0L;
		FFTrData[i] = 0L;
	}
	CreateWinTable();
	CreateBitTable();
	CreateCosTable();
	CreateSinTable();
	CreateEQBinTable();
}

/******************************************************************/
void FFTfree(void)
{
	if (FFTrData != NULL)
		_hfree(FFTrData);
	if (FFTiData != NULL)
		_hfree(FFTiData);
	if (CosLookup != NULL)
		_hfree(CosLookup);
	if (SinLookup != NULL)
		_hfree(SinLookup);
	if (FFTWindow != NULL)
		_hfree(FFTWindow);
	if (BitReverse != NULL)
		_hfree(BitReverse);
}

/******************************************************************/
int FFT(int LeftRight, BYTE _far *data, int SampleSize, int nChannels)
{	register int	k;
	register int	i;			// Loop Counter
	int   		LoopSize;	// Will be FFTsize or 2*FFTsize
	int   		Delta;		// How much to increment 'i' in our loop.  This is to skip over a channel
	int   		Start;		// Where in 'data' to start reading from
	int  _far *	iPointer;	// The 'data' array as integers
	long _far *	rPtr;
	long _far *	wPtr;

	int  N2;
	int  NU1;
	int  L;
	int  J;

	long C,S;
	long TREAL, TIMAG;
   long _far *	iPtrKN2;
   long _far *	rPtrKN2;
   long _far *	iPtrK;
   long _far *	rPtrK;
	FILE *f;

	#define STEREO_LEFT 	0
	#define STEREO_RIGHT 1
	#define MONO			2

   Start = 0;	//starting offset
	Delta = 1; //samples_per_buffer / FFTsize;	//offset between samples
	if (nChannels == 2)
   { 	Delta *= 2;
		if (LeftRight & 1)
   	 	Start = 1;//right
	}
	rPtr = FFTrData;		// Dereference the Real Array
   wPtr = FFTWindow;
   LoopSize = Delta * FFTsize;
	if (SampleSize == 2)
	{	iPointer = (int far *)data;
		for (i = Start; i < LoopSize; i+=Delta)
		{	// The values of the window function were multiplyed by
			// 32768 so divide by 37768.
			*rPtr++ = ((long)(iPointer[i]) * (*wPtr++)) >> 15L;
		}
	}
	else
	{	for (i = Start; i < LoopSize; i+=Delta)
		{	// The values of the window function were multiplyed by
			// 32768 so we need to divide by 37768. But, we want to
			// scale our 8-bit value up to a 16-bit number so we need
			// to multiply by 256.  That's why we shift right by 15-8=7.
			*rPtr++ = ((long)(data[i]) * (*wPtr++)) >> 7L;
		}
	}
	// Clear the imaginary numbers
	memset(FFTiData, 0x00, FFTsize * sizeof(long));

	if (snapshot)
	{	snapshot=0;
		if ((f = fopen("fftdata", "wb")) != NULL)
		{	for (i=0;i<FFTsize;i++)
				fprintf(f, "%ld\n", FFTrData[i]);
			fclose(f);
			printf("\nSnapshot file=FFTDATA SS=%d NC=%d LR=%d.", SampleSize, nChannels, LeftRight);
		}
	}
//now calculate the fft
   N2 = FFTsize >> 1;
   NU1 = FFTpower - 1;
	C = (long)CosLookup[0];
	S = (long)SinLookup[0];
   k = 0;
   for (L = 0; L < FFTpower; L++)
   {  while(k < (const)FFTsize)
		{  rPtrKN2 = FFTrData + N2 + k;
	      iPtrKN2 = FFTiData + N2 + k;
	      rPtrK = FFTrData + k;
	      iPtrK = FFTiData + k;
			for (i = 0; i < N2; i++)
			{	if (J != (k >> NU1))
	         {	J = (k >> NU1);
					C = (long)CosLookup[J];
	         	S = (long)SinLookup[J];
				}
				// The Cos/Sin tables were multiplyed by 256 to scale them up
				// to integer values.  So divide by 256 (>>8)
	         TREAL = ((*rPtrKN2 * C) + (*iPtrKN2 * S)) >> 8L;
	         TIMAG = ((*iPtrKN2 * C) - (*rPtrKN2 * S)) >> 8L;
	         *(rPtrKN2++) = *rPtrK - TREAL;
	         *(iPtrKN2++) = *iPtrK - TIMAG;
	         *(rPtrK++) += TREAL;
	         *(iPtrK++) += TIMAG;
				k++;
	      }
	      k += N2;
		}
      k = 0;
      NU1--;
      N2 = N2 >> 1;
   }
	return(LeftRight);
}

/******************************************************************/
void FFTdisplay(int LeftRight, int nChannels)
{	register int i, j;
	int row,col;
	__segment videoseg;
	BYTE __based(videoseg) *screen;
	BYTE textcolor;
	BYTE EQ[MAXBARS], EQvalue, EQstep, EQsn;
	int EQsignal=0, EQnoise=0, EQpeakbar=0;

	enum {PEAK			=0xD2};	//0x1E};	//0xD2};
	enum {BAR			=0xD7};	//0xBA};	//0xD7};
	enum {BLANK			=0x20};
	enum {LEFT_COLOR	=9};		//hi blue
	enum {RIGHT_COLOR	=12};		//hi red
	enum {WHITE			=15};		//hi white

	if (vc.adapter == _MDPA)	//monochrome display adapter
		videoseg=0xB000;
	else
		videoseg = 0xB800;
	EQstep=(MAXBARS/EQNumBars);
	if (LeftRight == 0)	//clear display
	{	if (nChannels == 2)	//stereo
		{	screen=(BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
			textcolor=(BYTE) _gettextcolor();
			for (i=0; i < BARSIZE; i++)
			{	for (j=0; j < ((EQNumBars*2)+2); j++)
				{	*screen++ = BLANK;
					*screen++ = textcolor;
				}
				screen -= (vc.numtextcols+(EQNumBars*2)+2) * 2;
			}
		}
		else	//mono
		{	screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
			textcolor = (BYTE) _gettextcolor();
			for (i=0; i < BARSIZE; i++)
			{	for (j=0; j < EQNumBars; j++)
				{	*screen++ = BLANK;
					*screen++ = textcolor;
				}
				screen -= (vc.numtextcols+EQNumBars) * 2;
			}
		}
		return;
	}
	// break down the FFT into bins for an equalizer bar display
	for (i=0; i < EQNumBars; i++)
	{	EQ[i] = 0;
		for (j = EQBinTable[i*EQstep][0]; j <= EQBinTable[(i*EQstep)+(EQstep-1)][1]; j++)
		{	if ((row = abs((int)(FFTrData[BitReverse[j]] >> (6L+(long)(FFTpower))))) > (BARSIZE-1))
				row = (BARSIZE-1);
			if ((col = abs((int)(FFTiData[BitReverse[j]] >> (6L+(long)(FFTpower))))) > (BARSIZE-1))
				col = (BARSIZE-1);
			if ((EQvalue = EQLookup[row][col]) > EQ[i])
				EQ[i] = EQvalue;
		}
		if (EQ[i] >= (EQlast[i]/4))	//peak display
			EQlast[i] = (EQ[i] * 4);	//new peak higher
		else
			EQlast[i]--;		//old peak decays w/time
		if (EQ[i] > EQ[EQpeakbar])	//max EQ for S/N
			EQpeakbar = i;
		EQnoise += EQ[i];
	}
	if (nChannels == 2)	//stereo
	{	if (LeftRight & 1)
			screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-EQNumBars-2)*2);//right
		else
			screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
		for (i=0; i < BARSIZE; i++)
		{	for (j=0; j < EQNumBars; j++)
			{	*screen++ = ((EQlast[j]/4)==i ? PEAK : ((EQ[j] >= i) ? BAR : BLANK));
				*screen++ = ((LeftRight & 1) ? RIGHT_COLOR : LEFT_COLOR);
			}
			screen -= (vc.numtextcols+EQNumBars) * 2;
		}
		if (LeftRight & 1)
			screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-EQNumBars-2)*2);//right
		else
			screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
	}
	else	//mono
	{	screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
		for (i=0; i < BARSIZE; i++)
		{	for (j=0; j < EQNumBars; j++)
			{	*screen++ = ((EQlast[j]/4)==i ? PEAK : ((EQ[j] >= i) ? BAR : BLANK));
				*screen++ = LEFT_COLOR;
			}
			screen -= (vc.numtextcols+EQNumBars) * 2;
		}
		screen = (BYTE __based(videoseg) *)(((vc.numtextcols*vc.numtextrows)-(2*EQNumBars)-4)*2);//left
	}

	EQnoise -= (EQsignal = EQ[EQpeakbar]);	//remove signal from total noise
	EQnoise /= (EQNumBars - 1);
	EQsn = ((EQsignal - EQnoise) * 85) / 15;
//	if (debug)
//		printf("\npeakbar=%d signal=%d noise=%d sn=%d", EQpeakbar, EQsignal, EQnoise, EQsn);
	*screen++ = 'E';
	*screen++ = WHITE;
	*screen++ = 'Q';
	*screen++ = WHITE;
	*screen++ = 's';
	*screen++ = WHITE;
	*screen++ = '/';
	*screen++ = WHITE;
	*screen++ = 'n';
	*screen++ = WHITE;
	*screen++ = '=';
	*screen++ = WHITE;
	if (EQsn <= 99)
		*screen++ = '0' + (EQsn / 10);
	else
		*screen++ = '*';
	*screen++ = WHITE;
	if (EQsn <= 99)
		*screen++ = '0' + (EQsn % 10);
	else
		*screen++ = '*';
	*screen++ = WHITE;
}

