//==========================================================================
//
//	title		: define functions to call PCI BIOS
//	company		: YAMAHA
//	author		: Taichi Sugiyama
//	create Data : 28/Sep/99
//
//==========================================================================
#include <dos.h>
#include <string.h>
#include "pcicfg.h"


//==========================================================================
//	get device number
//
//	param
//		bParam : parameter
//
//	return
//		device number
//
//==========================================================================
BYTE	DevNum(BYTE bParam)
{
	return (BYTE)(bParam >> 3);
}


//==========================================================================
//	get function number
//
//	param
//		bParam : parameter
//
//	return
//		function number
//
//==========================================================================
BYTE	FuncNum(BYTE bParam)
{
	return (BYTE)(bParam & 0x07);
}


//==========================================================================
//	get device function number
//
//	param
//		bDev	: device number
//		bFunc	: function number
//
//	return
//		(bDev << 3) | bFunc
//
//==========================================================================
BYTE	DevFunc(BYTE bDev, BYTE bFunc)
{
	return (BYTE)((bDev << 3) | bFunc);
}


//==========================================================================
//	get PCI bus number
//
//	param
//
//	return
//		PCI bus number
//
//==========================================================================
BYTE	PCIGetBus(void)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_BIOS_PRESENT;

	int386(PCI_SERVICE, &reg, &reg);

	if (reg.w.cflag & 0x0001)
		return 0;
	else {
		return reg.h.cl;
	}
 }


//==========================================================================
//	find PCI device
//
//	param
//		wVendor	: vendor ID
//		wDevice	: device ID
//		bIndex	: index
//		ppkey	: pointer to device node
//
//	return
//		0x00 if success
//
//==========================================================================
BYTE	PCIFindDevice(WORD wVendor, WORD wDevice, BYTE bIndex, PPCIKEY ppkey)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_FIND_DEVICE;
	reg.w.cx = wDevice;
	reg.w.dx = wVendor;
	reg.w.si = bIndex;

	int386(PCI_SERVICE, &reg, &reg);

	ppkey->bBus = reg.h.bh;
	ppkey->bDev = DevNum(reg.h.bl);
	ppkey->bFunc = FuncNum(reg.h.bl);

	return reg.h.ah;
}


//==========================================================================
//	read BYTE
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//
//	return
//		read data
//
//==========================================================================
BYTE	PCIReadConfigByte(PPCIKEY ppkey, WORD wAdr)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_READ_BYTE;
	reg.h.bh = ppkey->bBus;
	reg.h.bl = DevFunc(ppkey->bDev, ppkey->bFunc);
	reg.w.di = wAdr;
	int386(PCI_SERVICE, &reg, &reg);

	return reg.h.cl;
}


//==========================================================================
//	read WORD
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//
//	return
//		read data
//
//==========================================================================
WORD	PCIReadConfigWord(PPCIKEY ppkey, WORD wAdr)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_READ_WORD;
	reg.h.bh = ppkey->bBus;
    reg.h.bl = DevFunc(ppkey->bDev, ppkey->bFunc);
	reg.w.di = wAdr;

	int386(PCI_SERVICE, &reg, &reg);

	return reg.w.cx;
}


//==========================================================================
//	read DWORD
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//
//	return
//		read data
//
//==========================================================================
DWORD	PCIReadConfigDword(PPCIKEY ppkey, WORD wAdr)
{
	DWORD	dwData;


	dwData  = (DWORD)PCIReadConfigWord(ppkey, wAdr + 2) << 16;
	dwData |= (DWORD)PCIReadConfigWord(ppkey, wAdr);

	return dwData;
}


//==========================================================================
//	write BYTE
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//		bData	: data
//
//	return
//
//==========================================================================
void	PCIWriteConfigByte(PPCIKEY ppkey, WORD wAdr, BYTE bData)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_WRITE_BYTE;
	reg.h.bh = ppkey->bBus;
	reg.h.bl = DevFunc(ppkey->bDev, ppkey->bFunc);
	reg.h.cl = bData;
	reg.w.di = wAdr;

	int386(PCI_SERVICE, &reg, &reg);
}


//==========================================================================
//	write WORD
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//		wData	: data
//
//	return
//
//==========================================================================
void	PCIWriteConfigWord(PPCIKEY ppkey, WORD wAdr, WORD wData)
{
	union REGS	reg;


	reg.h.ah = PCI_FUNCTION_ID;
	reg.h.al = PCI_WRITE_WORD;
	reg.h.bh = ppkey->bBus;
	reg.h.bl = DevFunc(ppkey->bDev, ppkey->bFunc);
	reg.w.cx = wData;
	reg.w.di = wAdr;

   	int386(PCI_SERVICE, &reg, &reg);
}


//==========================================================================
//	write DWORD
//
//	param
//		ppkey	: pointer to device node
//		wAdr	: offset address
//		dwData	: data
//
//	return
//
//==========================================================================
void	PCIWriteConfigDword(PPCIKEY ppkey, WORD wAdr, DWORD dwData)
{
	PCIWriteConfigWord(ppkey, wAdr, LoW(dwData ));
	PCIWriteConfigWord(ppkey, wAdr + 2, HiW(dwData));
}
