//*****************************************************************************
//* Name:
//*      1212pm.cpp
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file implements the interface between 16 bit windows applications and 
//*      the Korg1212 VxD.  This file is compiled and archived into a library
//*      file that a 16 bit application can link in to gain access to the VxD.
//*      The application will use the 1212api.h header file to communicate with
//*      the card.
//*
//*
//* Modification Log:
//*
//*      1.3   6/09/97 Bill
//*      Added support for breaking out of a playback loop.
//*
//*      1.2   6/02/97 Bill
//*      Added support for posting messages to a file.  This is done for debugging
//*       purposes, and should be removed before final release.
//*
//*      1.1   1/29/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1996 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <mmsystem.h>
#define  PVOID void*
#ifndef  K1212API_H
#include "1212api.h"
#endif
#ifndef  K12PMAPIC_H
#include "12pmapic.h"
#endif
#ifndef  K1212APIP_H
#include "1212apip.h"
#endif
#ifndef  K1212WAVEAPI_H
#include "1212Wa~1.h"      // 1212WaveApi.h
#endif
#ifndef  K12PMAPII_H
#include "12pmapii.h"
#endif


// ----------------------------------------------------------
// Globals
// ----------------------------------------------------------
CKorg1212Api*  CKorg1212Api::_instance = 0;  // init instance variable

HFILE          ghFileHandle;            // v1.2

// ----------------------------------------------------------
// string table for identifying 1212 I/O wave devices
// ----------------------------------------------------------
char __far*   waveDevNameTable[] = {SR_K1212_WAVE_DEVICE1,
                                    SR_K1212_WAVE_DEVICE2,
                                    SR_K1212_WAVE_DEVICE3,
                                    SR_K1212_WAVE_DEVICE4,
                                    SR_K1212_WAVE_DEVICE5,
                                    SR_K1212_WAVE_DEVICE6
              };


// ----------------------------------------------------------
// string table for identifying 1212 I/O wave devices
// ----------------------------------------------------------
char __far*   waveDevSourceNameTable[] = {SR_K1212_WAVE_SOURCE1,
                                          SR_K1212_WAVE_SOURCE2,
                                          SR_K1212_WAVE_SOURCE3,
                                          SR_K1212_WAVE_SOURCE4,
                                          SR_K1212_WAVE_SOURCE5,
                                          SR_K1212_WAVE_SOURCE6
              };


// ----------------------------------------------------------
// string table for identifying the 1212 I/O card
// ----------------------------------------------------------
#define DEST_SHORT_NAME_INDEX  0
#define DEST_LONG_NAME_INDEX   1
char __far*  destNameTable[] = {
                                 "Master Volume",
                                 "Korg 1212 I/O Volume - card "
             };

// ----------------------------------------------------------
// string table for identifying wave device source names
// ----------------------------------------------------------
char __far*  mixerNameTable[] = {"Korg 1212 I/O Wave Out Mixer"};


DWORD       mixerControlMinValues[K1212_CTRL_MAX] = {
                                                      MIXCTRL_MIN_VOL,
                                                      MIXCTRL_MIN_VOL,
                                                      0
            };

DWORD       mixerControlMaxValues[K1212_CTRL_MAX] = {
                                                      MIXCTRL_MAX_VOL,
                                                      MIXCTRL_MAX_VOL,
                                                      1
            };

DWORD       mixerControlSteps[K1212_CTRL_MAX] = {
                                                   MIXCTRL_VOL_STEPS,
                                                   MIXCTRL_VOL_STEPS,
                                                   1
            };

DWORD       mixerControlTypes[K1212_CTRL_MAX] = {
                                                   MIXERCONTROL_CONTROLTYPE_VOLUME,
                                                   MIXERCONTROL_CONTROLTYPE_VOLUME,
                                                   MIXERCONTROL_CONTROLTYPE_MUTE
            };


char __far* mixerControlNames[k1212NumWaveDevices][K1212_CTRL_MAX] = {
                           "ADAT 1-2 Vol",   "ADAT 1-2 Vol",   "ADAT 1-2 Mute",
                           "ADAT 3-4 Vol",   "ADAT 3-4 Vol",   "ADAT 3-4 Mute",
                           "ADAT 5-6 Vol",   "ADAT 5-6 Vol",   "ADAT 5-6 Mute",
                           "ADAT 7-8 Vol",   "ADAT 7-8 Vol",   "ADAT 7-8 Mute",
                           "Analog 1-2 Vol", "Analog Vol",     "Analog Mute",
                           "S/PDIF Vol",     "S/PDIF Vol",     "S/PDIF Mute",
            };



WORD         stereoChannelMap[] = {LEFT_CHANNEL,      // ADAT 1
                                   RIGHT_CHANNEL,     // ADAT 2
                                   LEFT_CHANNEL,      // ADAT 3
                                   RIGHT_CHANNEL,     // ADAT 4
                                   LEFT_CHANNEL,      // ADAT 5
                                   RIGHT_CHANNEL,     // ADAT 6
                                   LEFT_CHANNEL,      // ADAT 7
                                   RIGHT_CHANNEL,     // ADAT 8
                                   LEFT_CHANNEL,      // Analog left
                                   RIGHT_CHANNEL,     // Analog right
                                   LEFT_CHANNEL,      // S/PDIF left
                                   RIGHT_CHANNEL,     // S/PDIF right
             };

// -----------------------------------------------------------------------
// forward declarations
// -----------------------------------------------------------------------
BOOL CallVxdsPMEntry(FARPROC              vxdAddress, 
                     k1212PMFunctionCodes functionCode,
                     k1212PMIn*           inBufPtr, 
                     WORD                 inBufSize,
                     k1212PMOut*          outBufPtr, 
                     WORD                 outBufSize,
                     WORD __far&          bytesReturned, 
                     WORD                 flags);
        


// -----------------------------------------------------------------------
// function for displaying a message box.  This function is called by the
// VxD when it wishes to notify the user of some (bad) condition.
// -----------------------------------------------------------------------
VOID FAR PASCAL MessageBoxFunction
(
   char __far * stringPtr
)
{
   MessageBox(NULL,
              stringPtr,
              "Korg 1212 I/O VxD",
              (MB_ICONINFORMATION | MB_OK)
   );
}




// -------------------------------------------------------------------------------------
// This function converts the passed in decimal value (ranging from 0x0 - 0xFF), to its
// equivalent string characters.  If the value is out of range, then FALSE is returned.
// twoHexChars must point to a three character array (zero terminated string).
// -------------------------------------------------------------------------------------
void charTo2HexChars(char __far* lpszTwoHexChars, char decValue)
{
   char firstDigitValue;

   lpszTwoHexChars[2] = '\0';

   firstDigitValue = (decValue >> 4);
   if (firstDigitValue > 0x9) {
      lpszTwoHexChars[0] = ((firstDigitValue - 0xA) + 'A');
   }
   else {
      lpszTwoHexChars[0] = firstDigitValue + '0';
   }

   decValue &= 0x0F;
   if (decValue > 0x9) {
      lpszTwoHexChars[1] = ((decValue - 0xA) + 'A');
   }
   else {
      lpszTwoHexChars[1] = decValue + '0';
   }
}


// ----------------------------------------------------------------------------
// CKorg1212Api::OpenDriver()
// control function for the API to the Korg1212 driver.  This function makes
// sure that only one instance of the API is created by an application.
// ----------------------------------------------------------------------------
CKorg1212Api*  CKorg1212Api::OpenDriver()
{   
   if (_instance == 0) {
      _instance = new CKorg1212Api;
   }
   return _instance;
}

// ----------------------------------------------------------------------------
// CKorg1212Api::CKorg1212Api()
// constructor for the Korg 1212 driver API object. This function creates the
// private implementation object, which handles the real initialization.
// ----------------------------------------------------------------------------
CKorg1212Api::CKorg1212Api()
{   
   _impl = new CKorg1212ApiImpl; // create the private implementation object
}


// ----------------------------------------------------------------------------
// CKorg1212ApiImpl::CKorg1212ApiImpl()
// constructor for the Korg 1212 driver API private implementation object. 
// This function will allocate resources and open a connection to the driver 
// using the CreateFile/DeviceIoControl interface
// ----------------------------------------------------------------------------
CKorg1212ApiImpl::CKorg1212ApiImpl()
{   
   SetNumCardsInSystem(0);  // initialize to specify no cards in the system

   // -----------------------------------------------------
   // init card information structures and handles.
   // -----------------------------------------------------
   for (int index = 0; index < k1212MaxCards; index++) {
      InitCardData(index);
   }

   SetVxDAddress();         // opens a connection to the driver
   SetPMInputBufPtr();      // allocates memory and assigns size and ptr members 
   SetPMOutputBufPtr();     // allocates memory and assigns size and ptr members
   RegisterAPI();           // registers our connection with the driver.
   QueryAllCards();         // obtains information about all cards in the system
                            //  from the driver.

   RegisterMsgBoxFunc((DWORD)MessageBoxFunction);    
                            // registers the message box display function with the
                            //  VxD.
}


// ----------------------------------------------------------------------------
// CKorg1212Api::SetVxdHandle()
// opens a connection to the Korg1212 driver and sets the vxdAddress data member
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::SetVxDAddress()
{
   FARPROC p;
   const char far* VxDName = KORG1212_ID_STRING;
   _asm {
      push  bx
      push  di

      mov   ax, 1684h
      xor   bx, bx
      les   di, VxDName
      int   2Fh      
      // ------------------------------------------------------------
      // if successful, es:di contains a far pointer to call the VxD
      // ------------------------------------------------------------
      mov   word ptr [p], di
      mov word ptr [p+2], es

      pop   di
      pop   bx       
   }

   vxdAddress = p;    
}


// ----------------------------------------------------------------------------
// CKorg1212Api::SetPMInputBufPtr()
// allocates space for the PM input buffer and stores a pointer to it.
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::SetPMInputBufPtr()
{
   pmInputBufSize = sizeof(k1212PMIn);
   pmInputBufPtr  = new k1212PMIn;
}


// ----------------------------------------------------------------------------
// CKorg1212Api::SetPMOutputBufPtr()
// allocates space for the PM output buffer and stores a pointer to it.
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::SetPMOutputBufPtr()
{
   pmOutputBufSize = sizeof(k1212PMOut);
   pmOutputBufPtr  = new k1212PMOut;
}


// ----------------------------------------------------------------------------
// CKorg1212Api::VerifyRefNum()
// verifies that the passed in card reference number is within bounds.
// ----------------------------------------------------------------------------
BOOL  CKorg1212ApiImpl::VerifyRefNum(k1212CardRef refNum)
{
   if ((refNum >= 0) && (refNum < GetNumCardsInSystem())) {
      return TRUE;
   }
   else {
      return FALSE;
   }
}


// ----------------------------------------------------------------------------
// CKorg1212ApiImpl::ClearCardData()
// clears out our local data structures for the specified card.
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::ClearCardData(k1212CardRef refNum)
{
   InitCardData      (refNum);
}


// ----------------------------------------------------------------------------
// CKorg1212ApiImpl::InitCardData()
// clears out our local data structures for the specified card.
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::InitCardData(k1212CardRef refNum)
{
   WORD        deviceId;
   WORD        controlId;

   CKorg1212MixControl __far* mixControlPtr;


   SetCardHandle     (refNum, 0);
   SetCardIndex      (refNum, 0);
   SetCardState      (refNum, K1212_STATE_NONEXISTENT);
   SetCardIrqNum     (refNum, 0);
   SetCardDevNode    (refNum, 0);
   SetCardBusNum     (refNum, 0);
   SetCardDevNum     (refNum, nullDeviceNumber);
   SetCardCompType   (refNum, MIXERLINE_COMPONENTTYPE_DST_SPEAKERS);

   // ------------------------------------------------------------------
   // card master volume control initialization
   // ------------------------------------------------------------------
   mixControlPtr = GetMasterVolumeControl(refNum);   
   mixControlPtr->SetControlType  (MIXERCONTROL_CONTROLTYPE_VOLUME);
   mixControlPtr->SetMinValue     (MIXCTRL_MIN_VOL);
   mixControlPtr->SetMaxValue     (MIXCTRL_MAX_VOL);
   mixControlPtr->SetNumSteps     (MIXCTRL_MAX_VOL);
   mixControlPtr->SetShortName    (destNameTable[DEST_SHORT_NAME_INDEX]);         
   mixControlPtr->SetName         (destNameTable[DEST_LONG_NAME_INDEX]);         
           
   // ------------------------------------------------------------------
   // init mixer control information
   // ------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {

      SetWaveCompType(refNum,
                      deviceId,
                      MIXERLINE_COMPONENTTYPE_SRC_WAVEOUT
      );

      // ------------------------------------------------------------------
      // client pointer list initialization
      // ------------------------------------------------------------------
      SetMixerClientPtr(refNum,
                        deviceId,
                        0
      );
      
      // ------------------------------------------------------------------
      // mixer controls initialization
      // ------------------------------------------------------------------
      for (controlId = 0; controlId < K1212_CTRL_MAX; controlId++) {
         mixControlPtr = GetMixControl(refNum,
                                       deviceId,
                                       controlId
                         );
         mixControlPtr->SetControlType  (mixerControlTypes[controlId]);
         mixControlPtr->SetMinValue     (mixerControlMinValues[controlId]);
         mixControlPtr->SetMaxValue     (mixerControlMaxValues[controlId]);
         mixControlPtr->SetNumSteps     (mixerControlSteps[controlId]);
         mixControlPtr->SetShortName    (mixerControlNames[deviceId][controlId]);         
         mixControlPtr->SetName         (mixerControlNames[deviceId][controlId]);         
      }

   }
}


// ----------------------------------------------------------------------------
// CKorg1212Api::RegisterAPI()
// Let the driver know we are here, and get a handle for our troubles.
// ----------------------------------------------------------------------------
BOOL CKorg1212ApiImpl::RegisterAPI()
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->cardIndex = 0;   // avoid the "bad index" error
   inBufPtr->deviceID  = 0;   // avoid the "bad index" error

   // ---------------------------------------------------------------------
   // let the driver know we be here (ebonics)
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(GetVxDAddress(), 
                       K1212PM_REGISTER_API,
                       inBufPtr, 
                       GetPMInputBufSize(),
                       outBufPtr, 
                       GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL
       )) {

      // ---------------------------------------------------------------------
      // if we successfully registered, store our assigned API handle.
      // ---------------------------------------------------------------------
      if (outBufPtr->returnValue == K1212_CMDRET_Success) {
         SetApiHandle(outBufPtr->apiInfo.apiHandle);
         SetNumBuffers(outBufPtr->apiInfo.sharedBufferCount);
         return TRUE;
      }
      else {
         return FALSE;
      }
   }
   else {
      return FALSE;
   }
}
      
      
// ----------------------------------------------------------------------------
// CKorg1212Api::RegisterMsgBoxFunc()
// Let the driver know where the message box display function is.
// ----------------------------------------------------------------------------
BOOL CKorg1212ApiImpl::RegisterMsgBoxFunc(DWORD funcPtr)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->cardIndex     = 0;   // avoid the "bad index" error
   inBufPtr->deviceID      = 0;   // avoid the "bad index" error
   inBufPtr->errMsgFuncPtr = funcPtr;
 
   // ---------------------------------------------------------------------
   // let the driver know we be here (ebonics)
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(GetVxDAddress(), 
                       K1212PM_ERRMSGFUNC,
                       inBufPtr, 
                       GetPMInputBufSize(),
                       outBufPtr, 
                       GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL
       )) {

      // ---------------------------------------------------------------------
      // if we successfully registered, store our assigned API handle.
      // ---------------------------------------------------------------------
      if (outBufPtr->returnValue == K1212_CMDRET_Success) {
         return TRUE;
      }
      else {
         return FALSE;
      }
   }
   else {
      return FALSE;
   }
}
      
      
// ----------------------------------------------------------------------------
// CKorg1212Api::QueryAllCards()
// opens a connection to the Korg1212 driver and sets the vxdHandle data member
// ----------------------------------------------------------------------------
void  CKorg1212ApiImpl::QueryAllCards()
{
   BOOL  cardExists;       // flags whether we already know about a card
   WORD  bytesReturned;    // number of bytes placed into the output buffer - if
                           //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // copy the card information elements over into our local array and
   // tally the number of cards present.
   // ---------------------------------------------------------------------
   DWORD curRef;
   for (int index = 0; index < k1212MaxCards; index++) {
      // ---------------------------------------------------------------------
      // assign input fields
      // ---------------------------------------------------------------------
      inBufPtr->apiHandle = (WORD)GetApiHandle();
      inBufPtr->cardIndex = index;
      inBufPtr->deviceID  = 0;   // avoid the "bad index" error
   
      // ---------------------------------------------------------------------
      // call the driver to get card info for the current index
      // ---------------------------------------------------------------------
      if (CallVxdsPMEntry(GetVxDAddress(), 
                          K1212PM_GET_CARD_INFO,
                          inBufPtr, 
                          GetPMInputBufSize(),
                          outBufPtr, 
                          GetPMOutputBufSize(),
                          bytesReturned, 
                          NULL)
          ) {

         if (outBufPtr->cardInfo.state != K1212_STATE_NONEXISTENT) {

            // ---------------------------------------------------------------------
            // make sure we don't already know about this card before storing info.
            // ---------------------------------------------------------------------
            curRef = GetNumCardsInSystem();

            cardExists = FALSE;
            for (DWORD testRef = 0; testRef < curRef; testRef++) {
               if ((GetCardBusNum(testRef) == outBufPtr->cardInfo.busNum) &&
                   (GetCardDevNum(testRef) == outBufPtr->cardInfo.deviceNum)) {
                   cardExists = TRUE;
               }
            }
            if (cardExists == TRUE) {
               continue;
            }

            // ---------------------------------------------------------------------
            // copy info into next available slot and bump the configured card count
            // ---------------------------------------------------------------------
            SetCardState      (curRef, (CardState)(outBufPtr->cardInfo.state));
            SetCardDevNode    (curRef, outBufPtr->cardInfo.devNode);
            SetCardIrqNum     (curRef, outBufPtr->cardInfo.irqNum);
            SetCardBusNum     (curRef, outBufPtr->cardInfo.busNum);
            SetCardDevNum     (curRef, outBufPtr->cardInfo.deviceNum);
            SetCardIndex      (curRef, index);
            IncNumCardsInSystem();

         }  // if card exists
      }  // if PM call
   }  // for all card indeces
}


// ----------------------------------------------------------------------------
// CKorg1212Api::~CKorg1212Api()
// destructor for the Korg 1212 driver API object.  This function releases 
// the private implementation object (which frees resources) and closes the 
// connection to the driver
// ----------------------------------------------------------------------------
CKorg1212Api::~CKorg1212Api()
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.


   _impl->RegisterMsgBoxFunc((DWORD)0);    
                          // let the VxD know it can't use us to display message
                          //     boxes anymore.
 
   // ---------------------------------------------------------------------
   // One last PM call for old time's sake! (before we get rid of everything)
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle = (WORD)_impl->GetApiHandle();

   // ---------------------------------------------------------------------
   // let the driver know we won't be talking to it anymore so it can clean
   // up in case the application didn't
   // ---------------------------------------------------------------------
   CallVxdsPMEntry(_impl->GetVxDAddress(), 
                   K1212PM_CLOSE_API,
                   inBufPtr, 
                   _impl->GetPMInputBufSize(),
                   outBufPtr, 
                   _impl->GetPMOutputBufSize(),
                   bytesReturned, 
                   NULL
   );

   // ---------------------------------------------------------------------
   // it is now safe to delete the resources
   // ---------------------------------------------------------------------
   delete _impl;    
}

// ----------------------------------------------------------------------------
// CKorg1212ApiImpl::~CKorg1212ApiImpl()
// destructor for the Korg 1212 driver API object.  This function releases 
// resources allocated for the API object.
// ----------------------------------------------------------------------------
CKorg1212ApiImpl::~CKorg1212ApiImpl()
{
   for (DWORD curRef = 0; curRef < k1212MaxCards; curRef++) {
      // do card specific cleanup here
   }
   delete[] GetPMInputBufPtr();   // free up allocated memory
   delete[] GetPMOutputBufPtr();  // "
   pmInputBufPtr  = 0;            // reset the pointers so we don't walk on anybody
   pmOutputBufPtr = 0;            // else by accident
   vxdAddress     = 0;            // "

}

                           
// -----------------------------------------------------------------------------
// *****************************************************************************
// API calls to the driver.  All calls return a K1212CmdRet value, which should
// be K1212_CMDRET_Success if all went well.  Parameters passed by reference 
// are used for returning function specific data.
// *****************************************************************************
// -----------------------------------------------------------------------------


K1212CmdRet CKorg1212Api::GetVersion (WORD __far & version)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle = (WORD)_impl->GetApiHandle();

   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_VERSION,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      version = outBufPtr->version;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


DWORD CKorg1212Api::GetNumCardsInSystem (void)
{
   return _impl->GetNumCardsInSystem();
}


K1212CmdRet CKorg1212Api::GetDeviceName (k1212CardRef  cardRefNum,
                                         WORD          devNumber,
                                         char __far*   devName)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      if (devNumber >= k1212NumWaveDevices) {
         return K1212_CMDRET_BadIndex;
      }
      else {
         wsprintf(devName,
                  "%s, %02d/%02d",
                  waveDevNameTable[devNumber],
                  _impl->GetCardBusNum(cardRefNum),
                  _impl->GetCardDevNum(cardRefNum)
         );
         return K1212_CMDRET_Success;
      }        
   }
   else {
      return K1212_CMDRET_BadIndex;
   }
}


K1212CmdRet CKorg1212Api::GetDestinationShortName (k1212CardRef cardRefNum,
                                                   char __far*  destName)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      wsprintf(destName,
               "%s",
               destNameTable[DEST_SHORT_NAME_INDEX]
      );
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadIndex;
   }
}

                                              
K1212CmdRet CKorg1212Api::GetDestinationLongName (k1212CardRef cardRefNum,
                                                   char __far*  destName)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      wsprintf(destName,
               "%s %02d/%02d",
               destNameTable[DEST_LONG_NAME_INDEX],
               _impl->GetCardBusNum(cardRefNum),
               _impl->GetCardDevNum(cardRefNum)
      );
      return K1212_CMDRET_Success;
   }
   else {
      return K1212_CMDRET_BadIndex;
   }
}

                                              
K1212CmdRet CKorg1212Api::GetSourceName (k1212CardRef cardRefNum,
                                         WORD         deviceId,
                                         char __far*  sourceName)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      if (deviceId >= k1212NumWaveDevices) {
         return K1212_CMDRET_BadIndex;
      }
      else {
         wsprintf(sourceName,
                  "%s",
                  waveDevSourceNameTable[deviceId]         
         );
         return K1212_CMDRET_Success;
      }        
   }
   else {
      return K1212_CMDRET_BadIndex;
   }
}
                                         

K1212CmdRet CKorg1212Api::GetMixerName  (char __far*   mixerName)
{
   wsprintf(mixerName,
            "%s",
            mixerNameTable[0]
   );
   return K1212_CMDRET_Success;
}


DWORD CKorg1212Api::GetDestType  (k1212CardRef cardRefNum)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      return _impl->GetCardCompType(cardRefNum);
   }
   else {
      return 0;
   }
}


DWORD CKorg1212Api::GetSourceType  (k1212CardRef cardRefNum,
                                    WORD         deviceId)
{
   if (_impl->VerifyRefNum(cardRefNum)) {
      if (deviceId >= k1212NumWaveDevices) {
         return 0;
      }
      else {
         return _impl->GetWaveCompType(cardRefNum,
                                       deviceId
                       );
      }
   }
   return 0;
}


WORD CKorg1212Api::GetNumDestMixControls (k1212CardRef cardRefNum)
{
   return 1;
}


WORD CKorg1212Api::GetNumSourceMixControls (k1212CardRef cardRefNum,
                                            WORD         deviceId)
{
   return K1212_CTRL_MAX;
}


const CKorg1212MixControl __far*  CKorg1212Api::GetMixControl (k1212CardRef cardRefNum,
                                                               WORD         deviceId,
                                                               WORD         controlId)
{
   return _impl->GetMixControl(cardRefNum,
                               deviceId,
                               controlId
                 );
}                                                 


const CKorg1212MixControl __far*  CKorg1212Api::GetMasterVolumeControl (
                                                               k1212CardRef cardRefNum
                                                )
{
   return _impl->GetMasterVolumeControl(cardRefNum);
}                                                 


K1212CmdRet CKorg1212Api::GetCardInfo (k1212CardRef                cardRefNum,
                                       k1212PublicCardInfo __far & cardInfo)
{
   return _impl->GetCardInfo(cardRefNum, cardInfo);
}


K1212CmdRet CKorg1212ApiImpl::GetCardInfo (k1212CardRef                cardRefNum,
                                           k1212PublicCardInfo __far & kciCardInfo)
{
   K1212CmdRet retCode;

   if (VerifyRefNum(cardRefNum)) {
      if ((retCode = UpdateCardInfo(cardRefNum)) == K1212_CMDRET_Success) {
         kciCardInfo.state     = GetCardState  (cardRefNum);
         kciCardInfo.irqNum    = GetCardIrqNum (cardRefNum);
         kciCardInfo.busNum    = GetCardBusNum (cardRefNum);
         kciCardInfo.deviceNum = GetCardDevNum (cardRefNum);
      }
      return retCode;
   }
   else {
      return K1212_CMDRET_BadIndex;
   }
}


K1212CmdRet CKorg1212ApiImpl::UpdateCardInfo (k1212CardRef cardRefNum)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle = (WORD)GetApiHandle();
   inBufPtr->cardIndex = GetCardIndex(cardRefNum);
   inBufPtr->deviceID  = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(GetVxDAddress(), 
                       K1212PM_GET_CARD_INFO,
                       inBufPtr, 
                       GetPMInputBufSize(),
                       outBufPtr, 
                       GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      SetCardState (cardRefNum, (CardState)outBufPtr->cardInfo.state);
      SetCardIrqNum(cardRefNum, outBufPtr->cardInfo.irqNum);
      SetCardBusNum(cardRefNum, outBufPtr->cardInfo.busNum);
      SetCardDevNum(cardRefNum, outBufPtr->cardInfo.deviceNum);
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetCardState (k1212CardRef        cardRefNum,
                                        CardState __far &   cardState)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID  = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_CARD_INFO,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      cardState = (CardState)(outBufPtr->cardInfo.state);
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::VerifyWaveInFormat (k1212CardRef        cardRefNum,
                                              WORD                deviceId,
                                              k1212WaveFormatData formatData)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;

   inBufPtr->waveFormatData.sampleRate    = formatData.sampleRate;
   inBufPtr->waveFormatData.bitsPerSample = formatData.bitsPerSample;
   inBufPtr->waveFormatData.nChannels     = formatData.nChannels;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_CHECK_INPUT_FORMAT,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }   
}


K1212CmdRet CKorg1212Api::VerifyWaveOutFormat (k1212CardRef        cardRefNum,
                                               WORD                deviceId,
                                               k1212WaveFormatData formatData)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;

   inBufPtr->waveFormatData.sampleRate    = formatData.sampleRate;
   inBufPtr->waveFormatData.bitsPerSample = formatData.bitsPerSample;
   inBufPtr->waveFormatData.nChannels     = formatData.nChannels;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_CHECK_OUTPUT_FORMAT,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }   
}


K1212CmdRet CKorg1212Api::GetDeviceInState (k1212CardRef      cardRefNum,
                                            WORD              deviceId,
                                            DWORD __far &     devState)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;

   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_WAVEIN_STATE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      devState = outBufPtr->deviceState;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }   
}


K1212CmdRet CKorg1212Api::GetDeviceOutState (k1212CardRef      cardRefNum,
                                             WORD              deviceId,
                                             DWORD __far &     devState)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;

   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_WAVEOUT_STATE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      devState = outBufPtr->deviceState;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }   
}


K1212CmdRet CKorg1212Api::EnableWaveInDevs (k1212CardRef cardRefNum)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_ENABLE_WAVEINDEVS,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::EnableWaveOutDevs (k1212CardRef cardRefNum)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_ENABLE_WAVEOUTDEVS,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::DisableWaveInDevs (k1212CardRef cardRefNum)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_DISABLE_WAVEINDEVS,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::DisableWaveOutDevs (k1212CardRef cardRefNum)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_DISABLE_WAVEOUTDEVS,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::OpenWaveInDevice (k1212CardRef                cardRefNum,
                                            WORD                        deviceId,
                                            k1212DeviceOpenData __far & devOpenData)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;
   
   inBufPtr->devOpenData.waveFormatParams.sampleRate    = devOpenData.waveFormatParams.sampleRate;
   inBufPtr->devOpenData.waveFormatParams.bitsPerSample = devOpenData.waveFormatParams.bitsPerSample;
   inBufPtr->devOpenData.waveFormatParams.nChannels     = devOpenData.waveFormatParams.nChannels;

   inBufPtr->devOpenData.driverCallbackAddress = devOpenData.driverCallbackAddress;
   inBufPtr->devOpenData.clientHandle          = devOpenData.clientHandle;
   
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_OPEN_WAVEINDEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::OpenWaveOutDevice (k1212CardRef                cardRefNum,
                                             WORD                        deviceId,
                                             k1212DeviceOpenData __far & devOpenData)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;
   
   inBufPtr->devOpenData.waveFormatParams.sampleRate    = devOpenData.waveFormatParams.sampleRate;
   inBufPtr->devOpenData.waveFormatParams.bitsPerSample = devOpenData.waveFormatParams.bitsPerSample;
   inBufPtr->devOpenData.waveFormatParams.nChannels     = devOpenData.waveFormatParams.nChannels;

   inBufPtr->devOpenData.driverCallbackAddress = devOpenData.driverCallbackAddress;
   inBufPtr->devOpenData.clientHandle          = devOpenData.clientHandle;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_OPEN_WAVEOUTDEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::CloseWaveInDevice (k1212CardRef cardRefNum,
                                             WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_CLOSE_WAVEINDEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::CloseWaveOutDevice (k1212CardRef cardRefNum,
                                              WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_CLOSE_WAVEOUTDEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::AddRecordBuffer (k1212CardRef                cardRefNum,
                                           WORD                        deviceId,
                                           k1212WaveBufferInfo __far & waveBufInfo)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;
   
   inBufPtr->waveBufferInfo.waveBufferPtr = waveBufInfo.waveBufferPtr;
   inBufPtr->waveBufferInfo.lpClient      = waveBufInfo.lpClient;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_ADD_RECORD_BUFFER,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::AddPlayBuffer (k1212CardRef                cardRefNum,
                                         WORD                        deviceId,
                                         k1212WaveBufferInfo __far & waveBufInfo)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;
   
   inBufPtr->waveBufferInfo.waveBufferPtr = waveBufInfo.waveBufferPtr;
   inBufPtr->waveBufferInfo.lpClient      = waveBufInfo.lpClient;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_ADD_PLAY_BUFFER,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::StartWaveInDevice (k1212CardRef cardRefNum,
                                             WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_START_WAVEIN_DEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::StopWaveInDevice (k1212CardRef cardRefNum,
                                            WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_INPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_STOP_WAVEIN_DEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::PauseWaveOutDevice (k1212CardRef cardRefNum,
                                              WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_PAUSE_WAVEOUT_DEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::ResumeWaveOutDevice (k1212CardRef cardRefNum,
                                               WORD         deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_RESUME_WAVEOUT_DEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::ResetWaveDevice (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           K1212WaveDeviceType deviceType)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   inBufPtr->deviceType = deviceType;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_RESET_WAVE_DEV,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::BreakLoop (k1212CardRef        cardRefNum,
                                     WORD                deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = deviceId;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_BREAKLOOP,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetWavePosition (k1212CardRef        cardRefNum,
                                           WORD                deviceId,
                                           K1212WaveDeviceType deviceType,
                                           MMTIME __far &      mmtime)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID     = deviceId;
   inBufPtr->deviceType   = deviceType;
   inBufPtr->mmTimeFormat = (DWORD)(mmtime.wType);
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_WAVE_POSITION,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      if (outBufPtr->returnValue == K1212_CMDRET_Success) {
         mmtime.wType    = (UINT)outBufPtr->mmTimeData.type;
         mmtime.u.sample = outBufPtr->mmTimeData.value;    // DWORD union for all types
      }
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetWaveMasterVolume (k1212CardRef  cardRefNum,
                                               WORD __far &  volume)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_MASTER_VOLUME,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      if (outBufPtr->returnValue == K1212_CMDRET_Success) {
         volume  = outBufPtr->wValue;
      }
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::SetWaveMasterVolume (k1212CardRef     cardRefNum,
                                               WORD             volume)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   inBufPtr->wValue       = volume;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_SET_MASTER_VOLUME,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetWaveDeviceVolume (k1212CardRef  cardRefNum,
                                               WORD          deviceId, 
                                               WORD __far &  leftVolume,
                                               WORD __far &  rightVolume)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID     = deviceId;
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_WAVEOUT_VOLUME,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      if (outBufPtr->returnValue == K1212_CMDRET_Success) {
         leftVolume  = outBufPtr->lrVals.leftValue;
         rightVolume = outBufPtr->lrVals.rightValue;
      }
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::SetWaveDeviceVolume (k1212CardRef     cardRefNum,
                                               WORD             deviceId,
                                               WORD             leftVolume,
                                               WORD             rightVolume)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle       = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex       = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID        = deviceId;
   inBufPtr->deviceType      = K1212_WAVETYPE_OUTPUT;
   inBufPtr->waveVol.leftVolume  = leftVolume;
   inBufPtr->waveVol.rightVolume = rightVolume;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_SET_WAVEOUT_VOLUME,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::QueryWaveDeviceMute (k1212CardRef cardRefNum,
                                               WORD         deviceId,
                                               WORD __far & muteStatus)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID     = deviceId;
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_QUERY_WAVEOUT_DEVICE_MUTE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      muteStatus = outBufPtr->wValue;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::MuteWaveDevice      (k1212CardRef     cardRefNum,
                                               WORD             deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID     = deviceId;
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_MUTE_WAVEOUT_DEVICE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::UnmuteWaveDevice    (k1212CardRef     cardRefNum,
                                               WORD             deviceId)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle    = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex    = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID     = deviceId;
   inBufPtr->deviceType   = K1212_WAVETYPE_OUTPUT;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_UNMUTE_WAVEOUT_DEVICE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::SetClockSourceRate (k1212CardRef     cardRefNum,
                                              ClockSourceIndex clkSrcIdx)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   inBufPtr->clkSrcIdx  = clkSrcIdx;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_SET_CLOCK_SOURCE_RATE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetClockSourceRate (k1212CardRef             cardRefNum,
                                              ClockSourceIndex __far & curClkSrcVal)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_CLOCK_SOURCE_RATE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      curClkSrcVal = (ClockSourceIndex)(outBufPtr->curClkSrc);
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::SetInputSensitivity (k1212CardRef cardRefNum,
                                               WORD         rightSens, 
                                               WORD         leftSens)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle         = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex         = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID          = 0;   // avoid the "bad index" error
   inBufPtr->lrVals.leftValue  = leftSens;
   inBufPtr->lrVals.rightValue = rightSens;
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_SET_INPUT_SENSITIVITY,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetInputSensitivity (k1212CardRef cardRefNum,
                                               WORD __far & curRightSens, 
                                               WORD __far & curLeftSens)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_INPUT_SENSITIVITY,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      curRightSens = outBufPtr->lrVals.rightValue;
      curLeftSens  = outBufPtr->lrVals.leftValue;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::GetRecordPlayLatency (WORD __far & numFrames)
{
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle = (WORD)_impl->GetApiHandle();

   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_GET_RECORD_PLAY_LATENCY,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      numFrames = outBufPtr->recPlyLatency;
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}


K1212CmdRet CKorg1212Api::AddMixerClient (k1212CardRef cardRefNum,
                                          WORD         deviceId,
                                          LPMXDCLIENT  lpMixerClient)
{
   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // it is, so add the client to the head of the list.
   // ---------------------------------------------------------------------
   lpMixerClient->lpNext = _impl->GetMixerClientPtr(cardRefNum,
                                                    deviceId
                                  );

   _impl->SetMixerClientPtr(cardRefNum,
                            deviceId,
                            lpMixerClient
          );

   return K1212_CMDRET_Success;
}

                                     
K1212CmdRet CKorg1212Api::RemoveMixerClient (k1212CardRef cardRefNum,
                                             WORD         deviceId,
                                             LPMXDCLIENT  lpMixerClient)
{
   LPMXDCLIENT tempPtr;

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // is it at the top of the list? 
   // ---------------------------------------------------------------------
   tempPtr = GetMixerClient(cardRefNum, 
                            deviceId,
                            0
             );
   if (tempPtr == lpMixerClient) {
      _impl->SetMixerClientPtr(cardRefNum,
                               deviceId,
                               tempPtr->lpNext
             );
      tempPtr->lpNext = 0;  // clear it out, now that it's no longer in the list
      return K1212_CMDRET_Success;
   }

   // ---------------------------------------------------------------------
   // if not, check the rest of the list. 
   // ---------------------------------------------------------------------
   do {
      if (lpMixerClient == tempPtr->lpNext) {
         tempPtr->lpNext       = lpMixerClient->lpNext;
         lpMixerClient->lpNext = 0;   // clear it now that it's out of the list
         return K1212_CMDRET_Success;
      }
   } while ((tempPtr = GetMixerClient(cardRefNum,
                                      deviceId,
                                      tempPtr
                       )) != 0);
   
   // ---------------------------------------------------------------------
   // if we got here, no match was found, return an error code.
   // ---------------------------------------------------------------------
   return K1212_CMDRET_BadParams;
}


LPMXDCLIENT  CKorg1212Api::GetMixerClient (k1212CardRef cardRefNum,
                                           WORD         deviceId,
                                           LPMXDCLIENT  lpMixerClient)
{
   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return 0;
   }

   if (lpMixerClient == 0) {
      return _impl->GetMixerClientPtr(cardRefNum, deviceId);
   }
   else {
      return lpMixerClient->lpNext;
   }
}


K1212CmdRet CKorg1212Api::DevNodeToCardRef (DWORD                devNode,
                                            k1212CardRef __far & cardRefNum)
{
   for (cardRefNum = 0; cardRefNum < GetNumCardsInSystem(); cardRefNum++) {
      if (_impl->GetCardDevNode(cardRefNum) == devNode) {
         return K1212_CMDRET_Success;
      }
   }
   cardRefNum = 0;
   return K1212_CMDRET_CardUninitialized;
}


K1212CmdRet CKorg1212Api::PostMsgToFile (k1212CardRef cardRefNum,
                                         char __far *   buffer,
                                         DWORD          numBytesToWrite)
{ 
   WORD bytesReturned;    // number of bytes placed into the output buffer - if
                          //    reported by the driver.  Ignored for now.

   // ---------------------------------------------------------------------
   // make sure the card reference number is valid.
   // ---------------------------------------------------------------------
   if (!_impl->VerifyRefNum(cardRefNum)) {
      return K1212_CMDRET_BadIndex;
   }

   // ---------------------------------------------------------------------
   // get working pointers to the input and output buffers for the PM call
   // ---------------------------------------------------------------------
   k1212PMIn*  inBufPtr  = _impl->GetPMInputBufPtr();     
   k1212PMOut* outBufPtr = _impl->GetPMOutputBufPtr();

   // ---------------------------------------------------------------------
   // assign input fields
   // ---------------------------------------------------------------------
   inBufPtr->apiHandle  = (WORD)_impl->GetApiHandle();
   inBufPtr->cardIndex  = _impl->GetCardIndex(cardRefNum);
   inBufPtr->deviceID   = 0;   // avoid the "bad index" error
   lstrcpyn(inBufPtr->string,
            buffer,
            39
   );
   
   // ---------------------------------------------------------------------
   // call the driver
   // ---------------------------------------------------------------------
   if (CallVxdsPMEntry(_impl->GetVxDAddress(), 
                       K1212PM_POSTMSG_TO_FILE,
                       inBufPtr, 
                       _impl->GetPMInputBufSize(),
                       outBufPtr, 
                       _impl->GetPMOutputBufSize(),
                       bytesReturned, 
                       NULL)
       ) {
      // ---------------------------------------------------------------------
      // make output assignments, and return with the driver's return code
      // ---------------------------------------------------------------------
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
   else {      // driver call failed
      return (K1212CmdRet)(outBufPtr->returnValue);
   }
}    



// -----------------------------------------------------------------------------
// *****************************************************************************
// CKorg1212Api::GetStateString()
// CKorg1212Api::GetCmdRetString()
// CKorg1212Api::GetClockSelString()
// CKorg1212Api::GetStopModeString()
// 
// These functions return descriptive strings for the given types of values and
// are used for debugging and perhaps application display purposes.
// *****************************************************************************
// -----------------------------------------------------------------------------
const char __far* CKorg1212Api::GetStateString (CardState state)
{
   switch (state) {
      case K1212_STATE_NONEXISTENT:
         return "Not configured";

      case K1212_STATE_UNINITIALIZED:
         return "Uninitialized";

      case K1212_STATE_DSP_IN_PROCESS:
         return "DSP microcode download in process";

      case K1212_STATE_DSP_COMPLETE:
         return "DSP microcode download complete";

      case K1212_STATE_READY:
         return "Card is ready but not opened";

      case K1212_STATE_OPEN:
         return "Card is opened and ready";

      case K1212_STATE_SETUP:
         return "Setup for play";

      case K1212_STATE_PLAYING:
         return "Playing";

      case K1212_STATE_MONITOR:
         return "Monitor Mode";

      case K1212_STATE_CALIBRATING:
         return "Calibrating";

      default:
         return "Unrecognized state value";
   }
}

const char __far* CKorg1212Api::GetCmdRetString (K1212CmdRet cmdRetVal)
{
   switch (cmdRetVal) {
      case K1212_CMDRET_Success:
         return "Call succeeded";

      case K1212_CMDRET_DIOCFailure:
         return "DIOC failure";

      case K1212_CMDRET_PMFailure:
         return "Protected Mode call failure";

      case K1212_CMDRET_FailUnspecified:
         return "Unspecified failure";

      case K1212_CMDRET_FailBadState:
         return "Failure due to current card state";

      case K1212_CMDRET_CardUninitialized:
         return "Card is uninitialized";

      case K1212_CMDRET_BadIndex:
         return "Bad card index/reference specified";

      case K1212_CMDRET_BadHandle:
         return "Bad card handle specified";

      case K1212_CMDRET_NoFillRoutine:
         return "No fill routine has been registered";

      case K1212_CMDRET_FillRoutineInUse:
         return "A fill routine has already been registered";

      default:
         return "Unrecognized return code value";
   }
}

const char __far* CKorg1212Api::GetClockSelString (ClockSourceIndex clkSrcIdx)
{
   switch (clkSrcIdx) {
      case K1212_CLKIDX_AdatAt44_1K:
         return "ADAT clock at 44.1 KHz";

      case K1212_CLKIDX_AdatAt48K:
         return "ADAT clock at 48 KHz";

      case K1212_CLKIDX_WordAt44_1K:
         return "S/PDIF clock at 44.1 KHz";

      case K1212_CLKIDX_WordAt48K:
         return "S/PDIF clock at 48 KHz";

      case K1212_CLKIDX_LocalAt44_1K:
         return "Local clock at 44.1 KHz";

      case K1212_CLKIDX_LocalAt48K:
         return "Local clock at 48 KHz";

      default:
         return "Unrecognized clock selector value";
   }
}

const char __far* CKorg1212Api::GetStopModeString (StopModeSelector stopMode)
{
   switch ((unsigned short)stopMode) {
      case K1212_STOP_FastAdatStop:
         return "Fast ADAT stop mode";

      case K1212_STOP_NormalStop:
         return "Normal ADAT stop mode";

      default:
         return "Unrecognized stop mode value";
   }
}


BOOL CallVxdsPMEntry (FARPROC              vxdAddress, 
                      k1212PMFunctionCodes functionCode,
                      k1212PMIn*           inBufPtr, 
                      WORD                 inBufSize,
                      k1212PMOut*          outBufPtr, 
                      WORD                 outBufSize,
                      WORD __far &         bytesReturned, 
                      WORD                 flags)
{
   k1212PmReq req;

   if (vxdAddress == NULL) {
      return FALSE;
   }
   
   req.functionCode = functionCode; 
   req.inBufSize    = inBufSize; 
   req.outBufSize   = outBufSize;   
   _fmemcpy(&(req.inBuf),
            inBufPtr,
            inBufSize
   );
   _fmemcpy(&(req.outBuf),
            outBufPtr,
            outBufSize
   );

   _asm lea di, [req]
   _asm mov ax, ss
   _asm mov es, ax

   vxdAddress();

   bytesReturned = req.bytesReturned;
   _fmemcpy(outBufPtr,
            &(req.outBuf),
            outBufSize
   );
   if (req.outBuf.returnValue == K1212_CMDRET_Success) {
      return TRUE;
   }
   else {
      return FALSE;
   }
}                
