//*****************************************************************************
//* Name:
//*      1212xwav.c - Wave device module for VxD KORG1212
//*
//* Project:
//*      1212 I/O VxD
//*
//* Author:
//*      Bill Jenkins
//*
//* Description:
//*      This file implements the wave device buffer translation functions.
//*
//*
//* Modification Log:
//*
//*      1.8   08/26/97 Bill
//*      (v 1.0B6) Put in fix for S/PDIF playback/recording.
//*
//*      1.7   06/17/97 Bill
//*      (v 1.11) Final attempt at cleaning up wave position messages before
//*      initial release.
//*
//*      1.6   06/10/97 Bill
//*      (v 1.09) Removed 1.04 remnants.  New wave device member lastIsrTime is
//*      now updated.
//*
//*      1.5   06/10/97 Bill
//*      (v 1.08) Record and playback routines did not update the sample
//*      played/recorded count appropriately for the last buffer.  11K and 22K
//*      record counts did not factor in the sample rate.
//*
//*      1.4   05/15/97 Bill
//*      (v 1.04) Modified playback routines to use the wave output device's 
//*      preFillBufIndex data member instead of the card's curFillBufNum data
//*      member to determine which buffer to fill.  preFillBufIndex also gets
//*      gets incremented by the output routine.
//*
//*      1.3   05/15/97 Bill
//*      (v 1.02) Added low pass filter to 22K and 11K playback routines.  
//*
//*      1.2   05/15/97 Bill
//*      (v 1.01) Corrected volume control error for 22K 8-bit stereo playback.  
//*
//*      1.1   03/17/97 Bill
//*      Initial version created.  
//*
//*
//* Copyright (c) 1997 Korg, Inc.
//* All rights reserved.
//*
//* This program is protected as an unpublished work under the U.S.
//* copyright laws.  The above copyright notice is not intended to
//* effect a publication of this work.
//*
//* This program is the confidential and proprietary information of
//* Korg and all its subsidiaries.
//*****************************************************************************

#include <vtoolsc.h>
#include <vmm.h>
#include <debug.h>
#include <vmmreg.h>

#ifndef  K1212CFG_H
#include "1212cfg.h"
#endif
#ifndef  K1212CARD_H
#include "1212card.h"
#endif
#ifndef  K1212INTF_H
#include "1212intf.h"
#endif
#ifndef  K1212WAVE_H
#include "1212wave.h"
#endif
#ifndef  K1212WAVEAPI_H
#include "1212WaveApi.h"
#endif

// ----------------------------------------------------------------------------
// thunk for the VM priority event and forward declaration of the event
// handler.  (v1.11)  externed here.
// ----------------------------------------------------------------------------
extern PriorityVMEvent_THUNK  RlsBufferOutThunk;
extern PriorityVMEvent_THUNK  RlsBufferInThunk;
extern VOID  __stdcall        RlsWaveOutBufferCallback(VMHANDLE       hVM, 
                                                       PVOID          Refdata, 
                                                       PCLIENT_STRUCT pcrs, 
                                                       DWORD          Flags
                              );
extern VOID  __stdcall        RlsWaveInBufferCallback (VMHANDLE       hVM, 
                                                       PVOID          Refdata, 
                                                       PCLIENT_STRUCT pcrs, 
                                                       DWORD          Flags
                              );

// ----------------------------------------------------------------------------
// (v1.02)
// The following arrays and functions are used to implement the low pass filter
// for 11K to 44K playback conversion.
// ----------------------------------------------------------------------------
long  lp1Coef[NUM_11TO44_WAVEOUT_SAMPLES_STORED + 1][NUM_11TO44_WAVEOUT_FILTER_COEF_PER_SAMPLE] = {
                 0,     0,     0, 65536,     0,     0,     0,     0,
               715,  2952,  9388, 58482, 18143,  5308,  1526,   349,
               686,  3037,  9958, 40261, 40261,  9958,  3037,   686,
               349,  1526,  5308, 18143, 58482,  9388,  2952,   715
      };

WORD  prevSampleSubscript11[NUM_11TO44_WAVEOUT_SAMPLES_STORED]
                           [NUM_11TO44_WAVEOUT_SAMPLES_STORED] = {
               0, 1, 2, 3, 4, 5, 6,
               1, 2, 3, 4, 5, 6, 0,
               2, 3, 4, 5, 6, 0, 1,
               3, 4, 5, 6, 0, 1, 2,
               4, 5, 6, 0, 1, 2, 3,
               5, 6, 0, 1, 2, 3, 4,
               6, 0, 1, 2, 3, 4, 5,
      };

   // -----------------------------------------------------------------------------------
   // void  CalculateLp1Samples
   // --------------------------------
   // this function takes the current sample value, along with the previous three values
   // and calculates the next four samples to output.
   // NOTE: As a side effect, the previous sample array and subscript "pointer" in the
   //       wave output device are updated for the next sample.
   // -----------------------------------------------------------------------------------
void  CalculateLp1Samples
(
   K1212Sample          currentLeftSample,   // current sample value
   K1212Sample          currentRightSample,  // current sample value
   k1212WaveOutDevice*  waveOutDevPtr,       // current wave output device we are filtering
   K1212Sample*         resultLeftSamples,   // stores four calculated left channel samples
   K1212Sample*         resultRightSamples   // stores four calculated right channel samples
)
{
   register long         sum;
   register WORD         sampleNum;   
   register K1212Sample  sampleA, sampleB, sampleC, sampleD, sampleE, sampleF, sampleG;
   
   // ---------------------------------------------------------------------------------
   // Do LEFT samples first
   // ---------------------------------------------------------------------------------
   sampleA = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][0]];
   sampleB = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][1]];
   sampleC = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][2]];
   sampleD = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][3]];
   sampleE = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][4]];
   sampleF = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][5]];
   sampleG = waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][6]];

   // ---------------------------------------------------------------------------------
   // calculate all but the last value, which is taken directly from the sample array
   // ---------------------------------------------------------------------------------
   for (sampleNum = 1; sampleNum <= NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED; sampleNum++) {
      sum =   (lp1Coef[sampleNum][1] * sampleB)
            + (lp1Coef[sampleNum][3] * sampleD)
            + (lp1Coef[sampleNum][4] * sampleE)
            + (lp1Coef[sampleNum][6] * sampleG)
            - (lp1Coef[sampleNum][0] * sampleA)
            - (lp1Coef[sampleNum][2] * sampleC)
            - (lp1Coef[sampleNum][5] * sampleF)
            - (lp1Coef[sampleNum][7] * currentLeftSample);

      resultLeftSamples[sampleNum] = (sum >> 16);
   }
   resultLeftSamples[0] = sampleD;
   
   // ---------------------------------------------------------------------------------
   // Now do RIGHT sample
   // ---------------------------------------------------------------------------------
   sampleA = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][0]];
   sampleB = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][1]];
   sampleC = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][2]];
   sampleD = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][3]];
   sampleE = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][4]];
   sampleF = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][5]];
   sampleG = waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][6]];

   // ---------------------------------------------------------------------------------
   // calculate all but the last value, which is taken directly from the sample array
   // ---------------------------------------------------------------------------------
   for (sampleNum = 1; sampleNum <= NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED; sampleNum++) {
      sum =   (lp1Coef[sampleNum][1] * sampleB)
            + (lp1Coef[sampleNum][3] * sampleD)
            + (lp1Coef[sampleNum][4] * sampleE)
            + (lp1Coef[sampleNum][6] * sampleG)
            - (lp1Coef[sampleNum][0] * sampleA)
            - (lp1Coef[sampleNum][2] * sampleC)
            - (lp1Coef[sampleNum][5] * sampleF)
            - (lp1Coef[sampleNum][7] * currentRightSample);

      resultRightSamples[sampleNum] = (sum >> 16);
   }
   resultRightSamples[0] = sampleD;
   
   // ---------------------------------------------------------------------------------
   // update the wave device array and subscript index
   // ---------------------------------------------------------------------------------
   waveOutDevPtr->prevLeftSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][0]] 
         = currentLeftSample;
   waveOutDevPtr->prevRightSample[prevSampleSubscript11[waveOutDevPtr->curSampleSubscript][0]] 
         = currentRightSample;
   waveOutDevPtr->curSampleSubscript++;
   if (waveOutDevPtr->curSampleSubscript >= NUM_11TO44_WAVEOUT_SAMPLES_STORED) {
      waveOutDevPtr->curSampleSubscript = 0;
   }
}


// ----------------------------------------------------------------------------
// (v1.02)
// The following arrays and functions are used to implement the low pass filter
// for 22K to 44K playback conversion.
// ----------------------------------------------------------------------------
long  lp2Coef[NUM_22TO44_WAVEOUT_FILTER_COEF_PER_SAMPLE] = {
               5960, 8344, 13907,  41722, 41722, 13907, 8344, 5960
      };

WORD  prevSampleSubscript22[NUM_22TO44_WAVEOUT_SAMPLES_STORED]
                           [NUM_22TO44_WAVEOUT_SAMPLES_STORED] = {
               0, 1, 2, 3, 4, 5, 6,
               1, 2, 3, 4, 5, 6, 0,
               2, 3, 4, 5, 6, 0, 1,
               3, 4, 5, 6, 0, 1, 2,
               4, 5, 6, 0, 1, 2, 3,
               5, 6, 0, 1, 2, 3, 4,
               6, 0, 1, 2, 3, 4, 5,
      };

   // -----------------------------------------------------------------------------------
   // void  CalculateLp2Samples
   // --------------------------------
   // this function implements the 22K to 44K filter.  The filter takes the current 
   // sample value, along with the previous seven values and calculates the next two 
   // samples to output.  The filter is implemented based on the following:
   //
   //  Given eight samples - A, B, C, D, E, F, G, and H, where H is the most recent
   //  sample, output point 1 will be point D, output point 2 will be generated by
   //  
   //  output2 = -a1*A + a2*B - a3*C + a4*D + a5*E - a6*F + a7*G - a8*H
   //
   //  a1-a8 are coefficients stored in the lp2Coef array.
   //
   // NOTE: As a side effect, the previous sample array and subscript "pointer" in the
   //       wave output device are updated for the next sample.
   // -----------------------------------------------------------------------------------
void  CalculateLp2Samples
(
   K1212Sample          currentLeftSample,   // current sample value
   K1212Sample          currentRightSample,  // current sample value
   k1212WaveOutDevice*  waveOutDevPtr,       // current wave output device we are filtering
   K1212Sample*         resultLeftSamples,   // stores two calculated left channel samples
   K1212Sample*         resultRightSamples   // stores two calculated right channel samples
)
{
   register long         sum;
   register WORD         sampleNum;   
   
   // ---------------------------------------------------------------------------------
   // Do LEFT samples first
   // calculate all but the last value, which is taken directly from the sample array
   // ---------------------------------------------------------------------------------
   sum =   (lp2Coef[1] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][1]])
         + (lp2Coef[3] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][3]])
         + (lp2Coef[4] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][4]])
         + (lp2Coef[6] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][6]])
         - (lp2Coef[0] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][0]])
         - (lp2Coef[2] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][2]])
         - (lp2Coef[5] * waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][5]])
         - (lp2Coef[7] * currentLeftSample);

      resultLeftSamples[1] = (sum >> 16);   // return interpolated sample 

   // --------------------------------------------------------
   // return fixed sample
   // --------------------------------------------------------
   resultLeftSamples[0] = waveOutDevPtr->prevLeftSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][3]];
   
   // ---------------------------------------------------------------------------------
   // Now do RIGHT sample
   // calculate all but the last value, which is taken directly from the sample array
   // ---------------------------------------------------------------------------------
   sum =   (lp2Coef[1] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][1]])
         + (lp2Coef[3] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][3]])
         + (lp2Coef[4] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][4]])
         + (lp2Coef[6] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][6]])
         - (lp2Coef[0] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][0]])
         - (lp2Coef[2] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][2]])
         - (lp2Coef[5] * waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][5]])
         - (lp2Coef[7] * currentRightSample);

      resultRightSamples[1] = (sum >> 16);      // return interpolated sample

   // --------------------------------------------------------
   // return fixed sample
   // --------------------------------------------------------
   resultRightSamples[0] = waveOutDevPtr->prevRightSample
                              [prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][3]];
   
   // ---------------------------------------------------------------------------------
   // update the wave device array and subscript index
   // ---------------------------------------------------------------------------------
   waveOutDevPtr->prevLeftSample[prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][0]] 
         = currentLeftSample;
   waveOutDevPtr->prevRightSample[prevSampleSubscript22[waveOutDevPtr->curSampleSubscript][0]] 
         = currentRightSample;
   waveOutDevPtr->curSampleSubscript++;
   if (waveOutDevPtr->curSampleSubscript >= NUM_22TO44_WAVEOUT_SAMPLES_STORED) {
      waveOutDevPtr->curSampleSubscript = 0;
   }
}




// ----------------------------------------------------------------------------
// The following function manages the movement of data between the card's
// shared buffers and the wave device driver buffers.  It is invoked each
// time the card requests the next buffer to be filled (and the card is in
// wave mode).
// ----------------------------------------------------------------------------
void WaveDeviceFillRoutineRec (WORD cardIndex); 
void WaveDeviceFillRoutinePlay(WORD cardIndex);

void WaveDeviceFillRoutine(WORD cardIndex)
{
   WaveDeviceFillRoutineRec (cardIndex);
   WaveDeviceFillRoutinePlay(cardIndex);
}

void WaveDeviceFillRoutineRec(WORD cardIndex)
{
   WORD                deviceId;
   k1212WaveInDevice*  waveInDevPtr;
   DWORD               timeStamp;
   K1212MemWaveInRet*  bufRetPtr;
   K1212MemWaveInRet*  prevBufRetPtr;

   timeStamp = Get_System_Time();

   // ----------------------------------------------------------------------------
   // check the state of each device.  For those devices that are active, call
   // their fill routine.  For the ones that are not, call the silence routine.
   // ----------------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {

      waveInDevPtr = getWaveInDevPtr(cardIndex,
                                     deviceId
                     );
      if (waveInDevPtr != 0) {
         switch (waveInDevPtr->state) {
            case K1212_WAVEINSTATE_RECORDING:
               (*(BufXlateWaveInFuncPtr)
                 waveInDevPtr->translateFunctionPtr)(cardIndex,
                                                     waveInDevPtr,
                                                     kPlayBufferFrames,
                                                     waveInDevPtr->spdifDevice
                               );
               waveInDevPtr->lastIsrTime = timeStamp;
               break;

            case K1212_WAVEINSTATE_STOP_PENDING:
               (*(BufXlateWaveInFuncPtr)
                 waveInDevPtr->translateFunctionPtr)(cardIndex,
                                                     waveInDevPtr,
                                                     waveInDevPtr->lastEstimatedOffset,
                                                     waveInDevPtr->spdifDevice
                               );
               FinishWaveInStop(cardIndex,
                                deviceId,
                                waveInDevPtr
               );
               break;
         }
      }
   }
}


void WaveDeviceFillRoutinePlay(WORD cardIndex)
{
   WORD                deviceId;
   k1212WaveOutDevice* waveOutDevPtr;
   DWORD               timeStamp;
   K1212MemWaveOutRet* bufRetPtr;
   K1212MemWaveOutRet* prevBufRetPtr;

   timeStamp = Get_System_Time();

   // ----------------------------------------------------------------------------
   // check the state of each device.  For those devices that are active, call
   // their fill routine.  For the ones that are not, call the silence routine.
   // ----------------------------------------------------------------------------
   for (deviceId = 0; deviceId < k1212NumWaveDevices; deviceId++) {

      waveOutDevPtr = getWaveOutDevPtr(cardIndex,
                                       deviceId
                      );
      if (waveOutDevPtr != 0) {

         switch (waveOutDevPtr->state) {
            case K1212_WAVEOUTSTATE_PLAYING:
               (*(BufXlateWaveOutFuncPtr)
                waveOutDevPtr->translateFunctionPtr)(cardIndex,
                                                     waveOutDevPtr,
                                                     waveOutDevPtr->spdifDevice
                               );
               if ((waveOutDevPtr->waveBufferQueue  != 0) ||
                   (waveOutDevPtr->numBufsPrefilled != 0)) {
                  waveOutDevPtr->lastIsrTime = timeStamp;
               }
               break;
   
               default:
                  (*playXlateFunctions[WAVFMT_UNSUPPORTED])(cardIndex,
                                                            waveOutDevPtr,
                                                            waveOutDevPtr->spdifDevice
                    );
                  break;
         }
         
         // ----------------------------------------------------------------------------
         // now check to see whether this device has any buffers to be returned. (v1.11)
         // ----------------------------------------------------------------------------
         START_CRITICAL
            bufRetPtr = waveOutDevPtr->retBufQueue;
            prevBufRetPtr = 0;
            while (bufRetPtr != 0) {
               if (bufRetPtr->buffer.delayCount == 0) {
                  // ------------------------------------------------------------------
                  // free the buffer.  Always schedule the event so that we can still 
                  // manipulate the linked list without having to worry about the
                  // event modifying things in the meantime.
                  // ------------------------------------------------------------------
                  Call_Priority_VM_Event(HIGH_PRI_DEVICE_BOOST,
                                         waveOutDevPtr->vmHandle,
                                         (PEF_ALWAYS_SCHED | PEF_WAIT_FOR_STI | PEF_WAIT_NOT_CRIT),
                                         &(bufRetPtr->buffer),
                                         RlsWaveOutBufferCallback,
                                         0,                          // no timeout specified
                                         &RlsBufferOutThunk
                  );
                  if (prevBufRetPtr != 0) {
                     prevBufRetPtr->next = bufRetPtr->next;
                  }
                  else {
                     waveOutDevPtr->retBufQueue = bufRetPtr->next;
                  }
               }
               else {                     
                  bufRetPtr->buffer.delayCount--;
                  prevBufRetPtr = bufRetPtr;
               }
   
               bufRetPtr = bufRetPtr->next;     // goto next in the list
            }
         END_CRITICAL
      }
   }
}



// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// The following functions translate from the card's shared buffers to wave
// buffers for recording.  The name of each function describes the wave format
// it is converting to.
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
#define VERIFY_PARAMS                               \
   if ( (cardIndex    >= k1212MaxCards) ||          \
        (waveDevicePtr == 0) ) {                    \
      return;                                       \
   }

void recordXlateSilence
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   // -----------------------------------------------------------------
   // do nothing.  this is an unsupported format or device that is not 
   // recording.
   // -----------------------------------------------------------------
}


void recordXlate48000Stereo16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;

   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(K1212Sample)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount++) {

            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(K1212Sample)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += sampleCount;     // v1.08
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // Handle special case for S/PDIF (v1.0B6)
            // --------------------------------------------------------------------
            if (spdifDevice == SPDIF_TRUE) {
               *waveSamplePtr       = *((K1212Sample*)cardSamplePtr);
               *(waveSamplePtr + 1) = *((K1212Sample*)(cardSamplePtr + sizeof(K1212SpdifSample)));
            }
            else {
               *((DWORD*)waveSamplePtr) = *((DWORD*)cardSamplePtr);
            }
            numBytesRecordedInBuf += sizeof(DWORD);
            cardSamplePtr         += sizeof(KorgAudioFrame);      // move to next frame
            (char*)waveSamplePtr  += sizeof(DWORD);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += numSamplesToRecord;
      }
   }
}


void recordXlate48000Stereo8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   if (spdifDevice == SPDIF_TRUE) {
      frameJump = (sizeof(KorgAudioFrame) - sizeof(K1212SpdifSample));
   }
   else {
      frameJump = (sizeof(KorgAudioFrame) - sizeof(K1212Sample));
   }

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(char)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount++) {

            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(char)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += sampleCount;
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // --------------------------------------------------------------------
            *waveSamplePtr = (char)((*((K1212Sample*)cardSamplePtr) >> 8) + PCM_8BIT_MIDPT);
            waveSamplePtr++;
 
            if (spdifDevice == SPDIF_TRUE) {
              cardSamplePtr += sizeof(K1212Sample);                    // move to next sample
            }
            else {
              cardSamplePtr += sizeof(K1212SpdifSample);               // move to next spdif sample
            }

            *waveSamplePtr = (char)((*((K1212Sample*)cardSamplePtr) >> 8) + PCM_8BIT_MIDPT);  
            waveSamplePtr++;
            numBytesRecordedInBuf += 2;
            cardSamplePtr         += frameJump;                      // move to next frame
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += numSamplesToRecord;
      }
   }
}

void recordXlate48000Mono16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;

   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(K1212Sample));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount++) {

            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - sizeof(K1212Sample));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += sampleCount;
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // just the left sample is copied for the mono case.
            // --------------------------------------------------------------------
            *waveSamplePtr = *((K1212Sample*)cardSamplePtr);
            numBytesRecordedInBuf += sizeof(K1212Sample);
            cardSamplePtr         += sizeof(KorgAudioFrame);      // move to next frame
            (char*)waveSamplePtr  += sizeof(K1212Sample);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += numSamplesToRecord;
      }
   }
}

void recordXlate48000Mono8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(char));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount++) {

            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (sizeof(char)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += sampleCount;
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next sample into the buffer and update our counts.
            // only the left sample is moved in mono.
            // --------------------------------------------------------------------
            *waveSamplePtr = (char)((*((K1212Sample*)cardSamplePtr) >> 8) + PCM_8BIT_MIDPT);
            waveSamplePtr++;
            numBytesRecordedInBuf++;
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += numSamplesToRecord;
      }
   }
}

void recordXlate44100Stereo16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   recordXlate48000Stereo16(cardIndex,
                            waveDevicePtr,
                            numSamplesToRecord,
                            spdifDevice
   );
}

void recordXlate44100Stereo8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   recordXlate48000Stereo8(cardIndex,
                           waveDevicePtr,
                           numSamplesToRecord,
                           spdifDevice
   );
}

void recordXlate44100Mono16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   recordXlate48000Mono16(cardIndex,
                          waveDevicePtr,
                          numSamplesToRecord,
                          spdifDevice
   );
}

void recordXlate44100Mono8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   recordXlate48000Mono8(cardIndex,
                         waveDevicePtr,
                         numSamplesToRecord,
                         spdifDevice
   );
}

void recordXlate22050Stereo16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          leftSampleSum;           
   DWORD          rightSampleSum;           
   DWORD          sampleSize;

   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   if (spdifDevice == SPDIF_TRUE) {
      sampleSize = sizeof(K1212SpdifSample);
   }
   else {
      sampleSize = sizeof(K1212Sample);
   }

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(K1212Sample)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 2) {

            // --------------------------------------------------------------------
            // sampleCount += 2 since we average two samples to handle 22 -> 44 KHz
            // conversion...
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(K1212Sample)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 2);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // --------------------------------------------------------------------
            leftSampleSum   = *((K1212Sample*)cardSamplePtr);
            rightSampleSum  = *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);        // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);        // move to next frame

            *waveSamplePtr = (K1212Sample)(leftSampleSum >> 1);  // sample = ((s1 + s2) / 2)
            waveSamplePtr++;
            *waveSamplePtr = (K1212Sample)(rightSampleSum >> 1); // sample = ((s1 + s2) / 2)
            waveSamplePtr++;

            numBytesRecordedInBuf += sizeof(DWORD);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 2);
      }
   }
}

void recordXlate22050Stereo8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          leftSampleSum;           
   DWORD          rightSampleSum;           
   DWORD          sampleSize;

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   if (spdifDevice == SPDIF_TRUE) {
      sampleSize = sizeof(K1212SpdifSample);
   }
   else {
      sampleSize = sizeof(K1212Sample);
   }

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(char)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 2) {

            // --------------------------------------------------------------------
            // sampleCount += 2 since we average two samples to convert from 22 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(char)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 2);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // --------------------------------------------------------------------
            leftSampleSum   = *((K1212Sample*)cardSamplePtr);
            rightSampleSum  = *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);          // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);          // move to next frame

            // sample = ((s1 + s2) / 2) >> 8   then convert to unsigned 
            *waveSamplePtr = (char)((leftSampleSum >> 9) + PCM_8BIT_MIDPT);
            waveSamplePtr++;
            // sample = ((s1 + s2) / 2) >> 8   then convert to unsigned
            *waveSamplePtr = (char)((rightSampleSum >> 9) + PCM_8BIT_MIDPT);  
            waveSamplePtr++;

            numBytesRecordedInBuf += 2;
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 2);
      }
   }
}

void recordXlate22050Mono16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          sampleSum;           

   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(K1212Sample));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 2) {

            // --------------------------------------------------------------------
            // sampleCount += 2 since we average two samples to convert from 22 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - sizeof(K1212Sample));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 2);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // just the left sample is copied for the mono case.
            // --------------------------------------------------------------------
            sampleSum      = *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame

            *waveSamplePtr = (K1212Sample)(sampleSum >> 1);    // sample = (s1 + s2) / 2

            numBytesRecordedInBuf += sizeof(short);
            (char*)waveSamplePtr  += sizeof(short);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 2);
      }
   }
}

void recordXlate22050Mono8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          sampleSum;           

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(char));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 2) {

            // --------------------------------------------------------------------
            // sampleCount += 2 since we average two samples to convert from 22 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - sizeof(char));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 2);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next sample into the buffer and update our counts.
            // only the left sample is moved in mono.
            // --------------------------------------------------------------------
            sampleSum      = *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame

            // sample = ((s1+s2) / 2) >> 8    then convert to unsigned
            *waveSamplePtr = (char)((sampleSum >> 9) + PCM_8BIT_MIDPT);  

            waveSamplePtr++;
            numBytesRecordedInBuf++;
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 2);
      }
   }
}

void recordXlate11025Stereo16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          leftSampleSum;           
   DWORD          rightSampleSum;
   DWORD          sampleSize;
   
   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   if (spdifDevice == SPDIF_TRUE) {
      sampleSize = sizeof(K1212SpdifSample);
   }
   else {
      sampleSize = sizeof(K1212Sample);
   }

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(K1212Sample)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 4) {

            // --------------------------------------------------------------------
            // sampleCount += 4 since we average four samples to handle 11 -> 44 KHz
            // conversion...
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(K1212Sample)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 4);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // --------------------------------------------------------------------
            leftSampleSum   = *((K1212Sample*)cardSamplePtr);
            rightSampleSum  = *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame

            *waveSamplePtr = (K1212Sample)(leftSampleSum >> 2);  // sample = (s1+s2+s3+s4) / 4)
            waveSamplePtr++;
            *waveSamplePtr = (K1212Sample)(rightSampleSum >> 2); // sample = (s1+s2+s3+s4) / 4)
            waveSamplePtr++;

            numBytesRecordedInBuf += sizeof(DWORD);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 4);
      }
   }
}

void recordXlate11025Stereo8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          leftSampleSum;           
   DWORD          rightSampleSum;
   DWORD          sampleSize;

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   if (spdifDevice == SPDIF_TRUE) {
      sampleSize = sizeof(K1212SpdifSample);
   }
   else {
      sampleSize = sizeof(K1212Sample);
   }

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - (2 * sizeof(char)));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 4) {

            // --------------------------------------------------------------------
            // sampleCount += 4 since we average four samples to convert from 11 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - (2 * sizeof(char)));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 4);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // both right and left samples are moved in one DWORD move.
            // --------------------------------------------------------------------
            leftSampleSum   = *((K1212Sample*)cardSamplePtr);
            rightSampleSum  = *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame
            leftSampleSum  += *((K1212Sample*)cardSamplePtr);
            rightSampleSum += *((K1212Sample*)(cardSamplePtr + sampleSize));
            cardSamplePtr  += sizeof(KorgAudioFrame);         // move to next frame

            // sample = ((s1+s2+s3+s4)/4) >> 8   then convert to unsigned
            *waveSamplePtr = (char)((leftSampleSum >> 10) + PCM_8BIT_MIDPT);  
            waveSamplePtr++;
            // sample = ((s1+s2+s3+s4)/4) >> 8   then convert to unsigned
            *waveSamplePtr = (char)((rightSampleSum >> 10) + PCM_8BIT_MIDPT);
            waveSamplePtr++;

            numBytesRecordedInBuf += 2;
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 4);
      }
   }
}

void recordXlate11025Mono16
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          sampleSum;           

   register DWORD        numBytesRecordedInBuf;
   register char*        cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(K1212Sample));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 4) {

            // --------------------------------------------------------------------
            // sampleCount += 4 since we average four samples to convert from 11 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - sizeof(K1212Sample));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 4);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next samples into the buffer and update our counts.
            // just the left sample is copied for the mono case.
            // --------------------------------------------------------------------
            sampleSum      = *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame

            *waveSamplePtr = (K1212Sample)(sampleSum >> 2);    // sample = (s1+s2+s3+s4)/4

            numBytesRecordedInBuf += sizeof(K1212Sample);
            (char*)waveSamplePtr  += sizeof(K1212Sample);
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 4);
      }
   }
}

void recordXlate11025Mono8
(
   WORD               cardIndex,
   k1212WaveInDevice* waveDevicePtr,
   DWORD              numSamplesToRecord,
   DWORD              spdifDevice
)
{
   DWORD          bufLength;
   DWORD          bufFullThreshold;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          sampleSum;           

   register DWORD numBytesRecordedInBuf;
   register char* cardSamplePtr;
   register char* waveSamplePtr;    // use this for more efficient looping - reset
                                    //    the wave device's pointer after the loop
                                    //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to receive the
   // data
   // -----------------------------------------------------------------
   waveHdrPtr = waveDevicePtr->waveBufferQueue;
   if (waveHdrPtr != 0) {

      waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
      if (waveSamplePtr != 0) {

         // ------------------------------------------------------------
         // setup local variables for more efficient looping
         // ------------------------------------------------------------
         bufLength             = waveDevicePtr->bufLength;
         numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
         bufFullThreshold      = (bufLength - sizeof(char));

         // ---------------------------------------------------------------------
         // move data from the card buffer into the wave buffer(s) 
         // get a pointer to the card data to copy.  This pointer will be offset
         // by the stereo pair that it refers to and incremented an audio frame
         // at a time.
         // ---------------------------------------------------------------------
         cardSamplePtr = (char*)getRecBufXAddress(cardIndex,
                                                  getCurRecBufNum(cardIndex)
                                );
         cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
         for (sampleCount = 0; sampleCount < numSamplesToRecord; sampleCount += 4) {

            // --------------------------------------------------------------------
            // sampleCount += 4 since we average four samples to convert from 11 to
            // 44 KHz.
            // --------------------------------------------------------------------
            // see if we've filled the current buffer.  If we have, release the
            // buffer back to the application and reset our variables for the new
            // buffer.  If there are no more buffers to receive data, exit.
            // --------------------------------------------------------------------
            if (numBytesRecordedInBuf > bufFullThreshold) {
               waveHdrPtr->dwBytesRecorded = numBytesRecordedInBuf;
               ReleaseWaveInBuffer(cardIndex,
                                   waveDevicePtr
               );
               waveHdrPtr       = waveDevicePtr->waveBufferQueue;
               waveSamplePtr    = (char*)waveDevicePtr->curWaveSamplePtr;
               bufFullThreshold = (waveDevicePtr->bufLength - sizeof(char));
               if ((waveHdrPtr == 0) || (waveSamplePtr == 0) || (bufLength == 0)) {
                  waveDevicePtr->numSamplesRecorded += (sampleCount / 4);
                  return;
               }
               numBytesRecordedInBuf = waveDevicePtr->numBytesRecordedInBuf;
            }

            // --------------------------------------------------------------------
            // move the next sample into the buffer and update our counts.
            // only the left sample is moved in mono.
            // --------------------------------------------------------------------
            sampleSum      = *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame
            sampleSum     += *((K1212Sample*)cardSamplePtr);
            cardSamplePtr += sizeof(KorgAudioFrame);           // move to next frame

            // sample = ((s1+s2+s3+s4)/4) >> 8    then convert to unsigned
            *waveSamplePtr = (char)((sampleSum >> 10) + PCM_8BIT_MIDPT);           

            waveSamplePtr++;
            numBytesRecordedInBuf++;
         } // for all samples

         // -----------------------------------------------------------------------
         // update the wave device's variables with our local count and pointer
         // -----------------------------------------------------------------------
         waveDevicePtr->curWaveSamplePtr       = waveSamplePtr;
         waveDevicePtr->numBytesRecordedInBuf  = numBytesRecordedInBuf;
         waveDevicePtr->numSamplesRecorded    += (numSamplesToRecord / 4);
      }
   }
}



// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// The following functions translate from the playback wave buffers to the 
// card's shared buffers.  The name of each function describes the wave format
// it is converting from.
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
void playXlateSilence
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
            DWORD sampleCount;
            DWORD samplesToAdd;
   register char* cardSamplePtr;

   VERIFY_PARAMS

   cardSamplePtr = (char*)getPlayBufXAddress(cardIndex,
                                             getCurFillBufNum(cardIndex)
                          );
   cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
   if (spdifDevice == SPDIF_TRUE) {
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {
         *(K1212SpdifSample*)cardSamplePtr                              = 0;
         *(K1212SpdifSample*)(cardSamplePtr + sizeof(K1212SpdifSample)) = 0;
         cardSamplePtr += sizeof(KorgAudioFrame);
      }
   }
   else {
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {
         *(DWORD*)cardSamplePtr = 0;
         cardSamplePtr += sizeof(KorgAudioFrame);
      }
   }

   // ----------------------------------------------------------------------
   // if the device is paused, then don't update any of the counts.
   // ----------------------------------------------------------------------
   if (waveDevicePtr->state == K1212_WAVEOUTSTATE_PAUSED) {
      return;
   }

   // ----------------------------------------------------------------------
   // check whether the device has run out of buffers and is now playing 
   // back the residual due to buffer prefill when the device was started.
   // if so, decrement the number of prefilled buffers count and update the
   // number of samples played, since it is still really playing samples and
   // has not reached the silence yet.  (v1.11)
   // ----------------------------------------------------------------------
   switch (waveDevicePtr->numBufsPrefilled) {
      default:
         waveDevicePtr->numBufsPrefilled--;
         break;

      case 0:
         break;
   }

   if (waveDevicePtr->numSamplesPrefilled != 0) {
      samplesToAdd = (kPlayBufferFrames / FramesPerSample(waveDevicePtr->format));
      if (waveDevicePtr->numSamplesPrefilled < samplesToAdd) {
         waveDevicePtr->numSamplesPlayed    += waveDevicePtr->numSamplesPrefilled;
         waveDevicePtr->numSamplesPrefilled  = 0;
      }
      else {
         waveDevicePtr->numSamplesPlayed    += samplesToAdd;
         waveDevicePtr->numSamplesPrefilled -= samplesToAdd;
      }
   }
}

void adjustSamplesPlayedForPartialBuffer
(
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               frameCount
)
{
   DWORD remainingSamplesToAdd;
   DWORD framesPerSample;
   DWORD samplesPlayed;

   framesPerSample       = FramesPerSample(waveDevicePtr->format);
   samplesPlayed         = (frameCount / framesPerSample);
   remainingSamplesToAdd = ((kPlayBufferFrames - frameCount) / framesPerSample);
   
   if (remainingSamplesToAdd > waveDevicePtr->numSamplesPrefilled) {
      waveDevicePtr->numSamplesPlayed    += (waveDevicePtr->numSamplesPrefilled + samplesPlayed);
      waveDevicePtr->numSamplesPrefilled  = 0;
   }
   else {
      waveDevicePtr->numSamplesPlayed    += (remainingSamplesToAdd + samplesPlayed);
      waveDevicePtr->numSamplesPrefilled -= remainingSamplesToAdd;
   }
}


void playXlate48000Stereo16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   short          leftSampleValue;
   short          rightSampleValue;
   long           lResult;
   DWORD          frameJump;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(K1212Sample));
      frameJump = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
      
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {

         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
            
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * (*waveSamplePtr++));
            leftSampleValue  = (lResult >> 16);

            lResult          = (waveDevicePtr->rightVolumeLevel * (*waveSamplePtr++));
            rightSampleValue = (lResult >> 16);
         }
         else {
            leftSampleValue = *waveSamplePtr++;
            rightSampleValue = *waveSamplePtr++;
         }

         *cardSamplePtr = leftSampleValue;
         (char*)cardSamplePtr += sampleSize;
         *cardSamplePtr = rightSampleValue;

         numBytesRemainingInBuf  -= (2 * sizeof(K1212Sample));
         (char*)cardSamplePtr    += frameJump;      // move to next frame
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += kPlayBufferFrames;
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate48000Stereo8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register char*        waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(char));
      frameJump = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;

      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {

         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr    == 0)    || 
                (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {

            lResult = (waveDevicePtr->leftVolumeLevel * 
                       (K1212Sample)(((*waveSamplePtr) - PCM_8BIT_MIDPT) << 8));

            (*cardSamplePtr) = (lResult >> 16);
            numBytesRemainingInBuf--;
            (char*)cardSamplePtr += sampleSize;
            waveSamplePtr++;

            lResult          = (waveDevicePtr->rightVolumeLevel * 
                                (K1212Sample)(((*waveSamplePtr) - PCM_8BIT_MIDPT) << 8));
            (*cardSamplePtr) = (lResult >> 16);
         }
         else {
            *cardSamplePtr = (((*waveSamplePtr) - PCM_8BIT_MIDPT) << 8);    // copy left sample
            numBytesRemainingInBuf--;
            (char*)cardSamplePtr += sampleSize;
            waveSamplePtr++;

            *cardSamplePtr = (((*waveSamplePtr) - PCM_8BIT_MIDPT) << 8);    // copy right sample
         }
         numBytesRemainingInBuf--;
         (char*)cardSamplePtr += frameJump;
         waveSamplePtr++;
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += kPlayBufferFrames;
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice

      );
   }
}

void playXlate48000Mono16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {

         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < sizeof(K1212Sample) ) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // in this case, the same sample is copied into both left and right 
         // channels in the card's buffer
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult        = (waveDevicePtr->leftVolumeLevel * (*waveSamplePtr));
            *cardSamplePtr = (lResult >> 16);
            (char*)cardSamplePtr += sampleSize;
            lResult        = (waveDevicePtr->rightVolumeLevel * (*waveSamplePtr));
            *cardSamplePtr = (lResult >> 16);
         }
         else {
            *cardSamplePtr = *waveSamplePtr;
            (char*)cardSamplePtr += sampleSize;
            *cardSamplePtr = *waveSamplePtr;
         }
         numBytesRemainingInBuf -= sizeof(K1212Sample);
         (char*)cardSamplePtr   += frameJump;
         (char*)waveSamplePtr   += sizeof(K1212Sample);
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += kPlayBufferFrames;
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate48000Mono8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register char*        waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
        
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount++) {

         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf <= 0) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // in this case, the same sample is copied into both left and right 
         // channels in the card's buffer
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult        = (waveDevicePtr->leftVolumeLevel * 
                              (K1212Sample)((*waveSamplePtr - PCM_8BIT_MIDPT) << 8));
            *cardSamplePtr = (lResult >> 16);
            (char*)cardSamplePtr += sampleSize;
            lResult        = (waveDevicePtr->rightVolumeLevel * 
                              (K1212Sample)((*waveSamplePtr - PCM_8BIT_MIDPT) << 8));
            *cardSamplePtr = (lResult >> 16);
         }
         else {
            *cardSamplePtr = ((*waveSamplePtr - PCM_8BIT_MIDPT) << 8);
            (char*)cardSamplePtr += sampleSize;
            *cardSamplePtr = ((*waveSamplePtr - PCM_8BIT_MIDPT) << 8);
         }
         numBytesRemainingInBuf--;
         (char*)cardSamplePtr += frameJump;
         waveSamplePtr++;
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += kPlayBufferFrames;
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate44100Stereo16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   playXlate48000Stereo16(cardIndex,
                          waveDevicePtr,
                          spdifDevice
   );
}

void playXlate44100Stereo8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   playXlate48000Stereo8(cardIndex,
                         waveDevicePtr,
                         spdifDevice
   );
}

void playXlate44100Mono16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   playXlate48000Mono16(cardIndex,
                        waveDevicePtr,
                        spdifDevice
   );
}

void playXlate44100Mono8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   // --------------------------------------------------------------------
   // 44.1 vs 48 - there is no difference.  Only the frequency of these
   // calls changes
   // --------------------------------------------------------------------
   playXlate48000Mono8(cardIndex,
                       waveDevicePtr,
                       spdifDevice
   );
}

void playXlate22050Stereo16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;
   DWORD          sampleSize;

   K1212Sample    leftSampleValue, rightSampleValue;

   K1212Sample    leftSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(K1212Sample));
      frameJump              = sizeof(KorgAudioFrame);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
        
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 2) {

         // --------------------------------------------------------------------
         // sampleCount += 2, since we are converting 22K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into locals and update our counts.
         // --------------------------------------------------------------------
         leftSampleValue  = (*waveSamplePtr);
         waveSamplePtr++;
         rightSampleValue = (*waveSamplePtr);
         waveSamplePtr++;
         numBytesRemainingInBuf -= (2 * sizeof(K1212Sample));

         // --------------------------------------------------------------------
         // adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * leftSampleValue);
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * rightSampleValue);
            rightSampleValue = (lResult >> 16);
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp2Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the samples to the card.
         // --------------------------------------------------------------------
         *cardSamplePtr                                     = leftSamples[0];
         *(K1212Sample*)((char*)cardSamplePtr + sampleSize) = rightSamples[0];
         (char*)cardSamplePtr                              += frameJump;     // goto next frame

         *cardSamplePtr                                     = leftSamples[1];
         *(K1212Sample*)((char*)cardSamplePtr + sampleSize) = rightSamples[1];
         (char*)cardSamplePtr                              += frameJump;     // goto next frame
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 2);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate22050Stereo8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;
   
   K1212Sample    leftSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   K1212Sample    leftSampleValue, rightSampleValue;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register char*        waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(char));
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
        
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 2) {

         // --------------------------------------------------------------------
         // sampleCount += 2, since we are converting 22K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr    == 0)    || 
                (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // --------------------------------------------------------------------
         leftSampleValue  = (*waveSamplePtr - PCM_8BIT_MIDPT) << 8;
         waveSamplePtr++;
         rightSampleValue = (*waveSamplePtr - PCM_8BIT_MIDPT) << 8;
         waveSamplePtr++;

         // --------------------------------------------------------------------
         // adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * leftSampleValue);
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * rightSampleValue);
            rightSampleValue = (lResult >> 16);
         }
         numBytesRemainingInBuf -= 2;

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp2Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the samples to the card.
         // --------------------------------------------------------------------
         *cardSamplePtr = leftSamples[0];
         (char*)cardSamplePtr += sampleSize; 
         *cardSamplePtr = rightSamples[0];
         (char*)cardSamplePtr += frameJump;     // goto next frame

         *cardSamplePtr = leftSamples[1];
         (char*)cardSamplePtr += sampleSize; 
         *cardSamplePtr = rightSamples[1];
         (char*)cardSamplePtr += frameJump;     // goto next frame

      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 2);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate22050Mono16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;

   K1212Sample    leftSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   K1212Sample    leftSampleValue;
   K1212Sample    rightSampleValue;
   long           lResult;
   DWORD          sampleSize;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 2) {

         // --------------------------------------------------------------------
         // sampleCount += 2, since we are converting 22K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < sizeof(K1212Sample) ) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // Adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * (*waveSamplePtr));
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * (*waveSamplePtr));
            rightSampleValue = (lResult >> 16);
         }
         else {
            leftSampleValue = rightSampleValue = *waveSamplePtr;
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp2Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the samples to the card.
         // --------------------------------------------------------------------
         *cardSamplePtr = leftSamples[0];
         (char*)cardSamplePtr += sampleSize;
         *cardSamplePtr = rightSamples[0];
         (char*)cardSamplePtr += frameJump;  // next frame

         *cardSamplePtr = leftSamples[1];
         (char*)cardSamplePtr += sampleSize;
         *cardSamplePtr = rightSamples[1];
         (char*)cardSamplePtr += frameJump;  // next frame

         numBytesRemainingInBuf -= sizeof(K1212Sample);
         (char*)waveSamplePtr   += sizeof(K1212Sample);
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 2);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate22050Mono8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           lResult;

   K1212Sample    leftSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_22TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   K1212Sample    leftSampleValue;
   K1212Sample    rightSampleValue;
   unsigned char  adjustedSample;
   DWORD          sampleSize;

   register long           numBytesRemainingInBuf;
   register K1212Sample*   cardSamplePtr;
   register unsigned char* waveSamplePtr;    // use this for more efficient looping - reset
                                             //    the wave device's pointer after the loop
                                             //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 2) {

         // --------------------------------------------------------------------
         // sampleCount += 2, since we are converting 22K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf <= 0) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // in this case, the same sample is copied into both left and right 
         // channels in the card's buffer
         // --------------------------------------------------------------------
         adjustedSample   = (*waveSamplePtr - PCM_8BIT_MIDPT);
         leftSampleValue  = (adjustedSample << 8);
         rightSampleValue = leftSampleValue;

         // --------------------------------------------------------------------
         // Adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * leftSampleValue); 
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * rightSampleValue);
            rightSampleValue = (lResult >> 16);
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp2Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );
         
         // --------------------------------------------------------------------
         // write the samples to the card.
         // --------------------------------------------------------------------
         *cardSamplePtr = leftSamples[0];
         (char*)cardSamplePtr += sampleSize;
         *cardSamplePtr = rightSamples[0];
         (char*)cardSamplePtr += frameJump;        // next frame
         *cardSamplePtr = leftSamples[1];
         (char*)cardSamplePtr += sampleSize;
         *cardSamplePtr = rightSamples[1];
         (char*)cardSamplePtr += frameJump;        // next frame
         numBytesRemainingInBuf--;
         waveSamplePtr++;
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 2);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate11025Stereo16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           repeatCount;
   long           lResult;
   DWORD          sampleSize;

   K1212Sample    leftSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   K1212Sample    leftSampleValue;
   K1212Sample    rightSampleValue;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(K1212Sample));
      frameJump              = sizeof(KorgAudioFrame);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 4) {

         // --------------------------------------------------------------------
         // sampleCount += 4, since we are converting 11K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // Adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * (*waveSamplePtr)); 
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * (*(waveSamplePtr + 1)) ); 
            rightSampleValue = (lResult >> 16);
         }
         else {
            leftSampleValue  = (*waveSamplePtr);
            rightSampleValue = (*(waveSamplePtr + 1));
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp1Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the filtered samples to the card.
         // --------------------------------------------------------------------
         for (repeatCount = 0; repeatCount < 4; repeatCount++) {
            *cardSamplePtr                                     = leftSamples[repeatCount];
            *(K1212Sample*)((char*)cardSamplePtr + sampleSize) = rightSamples[repeatCount];
            (char*)cardSamplePtr                              += frameJump;     // goto next frame
         }

         numBytesRemainingInBuf  -= sizeof(DWORD);
         (char*)waveSamplePtr    += sizeof(DWORD);
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 4);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate11025Stereo8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   DWORD          stereoSampleSize;
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           repeatCount;
   long           lResult;
   DWORD          sampleSize;

   K1212Sample    leftSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   K1212Sample    leftSampleValue, rightSampleValue;

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register char*        waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   char* - 8 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      stereoSampleSize       = (2 * sizeof(char));
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 4) {

         // --------------------------------------------------------------------
         // sampleCount += 4, since we are converting 11K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < stereoSampleSize) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr    == 0)    || 
                (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // both right and left samples are moved in one DWORD move.
         // we repeat each sample twice to convert from 22K to 44K until
         // the DSP guru dude blesses us with the golden algorithm.
         // --------------------------------------------------------------------
         leftSampleValue  = (*waveSamplePtr++ - PCM_8BIT_MIDPT) << 8;   // must be post increment
         rightSampleValue = (*waveSamplePtr++ - PCM_8BIT_MIDPT) << 8;
         numBytesRemainingInBuf -= 2;                                   // we just read two bytes from
                                                                        //   the buffer

         // --------------------------------------------------------------------
         // Adjust volume level if volume control is enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * leftSampleValue); 
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * rightSampleValue); 
            rightSampleValue = (lResult >> 16);
         }


         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp1Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the filtered samples to the card.
         // --------------------------------------------------------------------
         for (repeatCount = 0; repeatCount < 4; repeatCount++) {
            *cardSamplePtr        = leftSamples[repeatCount];
            (char*)cardSamplePtr += sampleSize;
            *cardSamplePtr        = rightSamples[repeatCount];
            (char*)cardSamplePtr += frameJump;     // goto next frame
         }
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 4);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate11025Mono16
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           repeatCount;
   long           lResult;
   DWORD          sampleSize;
   K1212Sample    leftSampleValue;
   K1212Sample    rightSampleValue;

   K1212Sample    leftSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register K1212Sample* waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (K1212Sample*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 4) {

         // --------------------------------------------------------------------
         // sampleCount += 4, since we are converting 11K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf < sizeof(K1212Sample) ) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // in this case, the same sample is copied into both left and right 
         // channels in the card's buffer.  Adjust volume if volume control is
         // enabled.
         // --------------------------------------------------------------------
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult     = (waveDevicePtr->leftVolumeLevel * (*waveSamplePtr)); 
            leftSampleValue = (lResult >> 16);
            lResult     = (waveDevicePtr->rightVolumeLevel * (*waveSamplePtr)); 
            rightSampleValue = (lResult >> 16);
         }
         else {
            leftSampleValue = rightSampleValue = (*waveSamplePtr);
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp1Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );

         // --------------------------------------------------------------------
         // write the filtered samples to the card.
         // --------------------------------------------------------------------
         for (repeatCount = 0; repeatCount < 4; repeatCount++) {
            *cardSamplePtr = leftSamples[repeatCount];
            (char*)cardSamplePtr += sampleSize;                 
            *cardSamplePtr = rightSamples[repeatCount];
            (char*)cardSamplePtr += frameJump;
         }
         numBytesRemainingInBuf -= sizeof(K1212Sample);
         (char*)waveSamplePtr   += sizeof(K1212Sample);
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 4);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}

void playXlate11025Mono8
(
   WORD                cardIndex,
   k1212WaveOutDevice* waveDevicePtr,
   DWORD               spdifDevice
)
{
   PWAVEHDR       waveHdrPtr;
   DWORD          sampleCount;
   DWORD          frameJump;
   long           repeatCount;
   K1212Sample    sampleValue;
   K1212Sample    leftSampleValue;
   K1212Sample    rightSampleValue;
   long           lResult;
   DWORD          sampleSize;

   K1212Sample    leftSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];
   K1212Sample    rightSamples[NUM_11TO44_WAVEOUT_SAMPLES_CALCULATED + 1];

   register long         numBytesRemainingInBuf;
   register K1212Sample* cardSamplePtr;
   register char*        waveSamplePtr;    // use this for more efficient looping - reset
                                           //    the wave device's pointer after the loop
                                           //   K1212Sample* - 16 bit PCM

   VERIFY_PARAMS

   // -----------------------------------------------------------------
   // make sure all the buffer pointers are in place to send the
   // samples to the card buffers
   // -----------------------------------------------------------------
   waveHdrPtr    = waveDevicePtr->waveBufferQueue;
   waveSamplePtr = (char*)waveDevicePtr->curWaveSamplePtr;
   if ((waveHdrPtr != 0) && (waveSamplePtr != 0)) {

      // ------------------------------------------------------------
      // setup local variables for more efficient looping
      // ------------------------------------------------------------
      if (spdifDevice == SPDIF_TRUE) {
         sampleSize = sizeof(K1212SpdifSample);
      }
      else {
         sampleSize = sizeof(K1212Sample);
      }
      numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;
      frameJump              = (sizeof(KorgAudioFrame) - sampleSize);

      // ---------------------------------------------------------------------
      // move data from the card buffer into the wave buffer(s) 
      // get a pointer to the card data to copy.  This pointer will be offset
      // by the stereo pair that it refers to and incremented an audio frame
      // at a time.
      // ---------------------------------------------------------------------
      cardSamplePtr = (K1212Sample*)getPlayBufXAddress(cardIndex,
                                                       getCurFillBufNum(cardIndex)
                                    );
      (char*)cardSamplePtr += waveDevicePtr->stereoPairOffset;
         
      for (sampleCount = 0; sampleCount < kPlayBufferFrames; sampleCount += 4) {

         // --------------------------------------------------------------------
         // sampleCount += 4, since we are converting 11K to 44K
         // --------------------------------------------------------------------
         // see if we've used up the current buffer.  If we have, release the
         // buffer back to the application and reset our variables for the new
         // buffer.  If there are no more buffers to send data, fill the 
         // remaining portions of the card buffer with silence.
         // --------------------------------------------------------------------
         if (numBytesRemainingInBuf <= 0) {
            ReleaseWaveOutBuffer(cardIndex,
                                 waveDevicePtr
            );
            waveHdrPtr             = waveDevicePtr->waveBufferQueue;
            waveSamplePtr          = waveDevicePtr->curWaveSamplePtr;
            numBytesRemainingInBuf = (long)waveDevicePtr->remainingBytesInBuf;

            if ((waveHdrPtr == 0) || (waveSamplePtr == 0)) {

               // --------------------------------------------------------------
               // no more wave buffers, fill the remainder with silence and exit
               // --------------------------------------------------------------
               adjustSamplesPlayedForPartialBuffer(waveDevicePtr, sampleCount);
               while (sampleCount < kPlayBufferFrames) {
                  *((DWORD*)cardSamplePtr) = 0;
                  if (spdifDevice == SPDIF_TRUE) {
                     *((DWORD*)cardSamplePtr + 1) = 0;
                  }
                  (char*)cardSamplePtr    += sizeof(KorgAudioFrame);
                  sampleCount++;
               }
               return;
            }
         }

         // --------------------------------------------------------------------
         // move the next samples into the buffer and update our counts.
         // in this case, the same sample is copied into both left and right 
         // channels in the card's buffer.  Adjust volume level if volume 
         // control is enabled.
         // --------------------------------------------------------------------
         sampleValue = ((*waveSamplePtr - PCM_8BIT_MIDPT) << 8);
         if (waveDevicePtr->volumeControlEnabled == TRUE) {
            lResult          = (waveDevicePtr->leftVolumeLevel * sampleValue);
            leftSampleValue  = (lResult >> 16);
            lResult          = (waveDevicePtr->rightVolumeLevel * sampleValue);
            rightSampleValue = (lResult >> 16);
         }
         else {
            leftSampleValue = rightSampleValue = sampleValue;
         }

         // --------------------------------------------------------------------
         // filter the samples.
         // --------------------------------------------------------------------
         CalculateLp1Samples(leftSampleValue,
                             rightSampleValue,
                             waveDevicePtr,
                             leftSamples,
                             rightSamples
         );
         
         // --------------------------------------------------------------------
         // write the filtered samples to the card.
         // --------------------------------------------------------------------
         for (repeatCount = 0; repeatCount < 4; repeatCount++) {
            *cardSamplePtr = leftSamples[repeatCount];
            (char*)cardSamplePtr += sampleSize;
            *cardSamplePtr = rightSamples[repeatCount];
            (char*)cardSamplePtr += frameJump;  // next frame
         }
         numBytesRemainingInBuf--;
         waveSamplePtr++;
      } // for all samples

      // -----------------------------------------------------------------------
      // update the wave device's variables with our local count and pointer
      // -----------------------------------------------------------------------
      waveDevicePtr->curWaveSamplePtr    = waveSamplePtr;
      waveDevicePtr->remainingBytesInBuf = (DWORD)numBytesRemainingInBuf;
      waveDevicePtr->numSamplesPlayed   += (kPlayBufferFrames / 4);
   }
   else {
      playXlateSilence(cardIndex,      // no buffer, so fill with silence
                       waveDevicePtr,
                       spdifDevice
      );
   }
}


// ----------------------------------------------------------------------------
// The following arrays hold the buffer translation function pointers for both
// record and playback.  They are indexed by k1212WaveFormat.
// ----------------------------------------------------------------------------
BufXlateWaveInFuncPtr recordXlateFunctions[] = {
                                                recordXlateSilence,

                                                recordXlate48000Stereo16,
                                                recordXlate48000Stereo8,
                                                recordXlate48000Mono16,
                                                recordXlate48000Mono8,

                                                recordXlate44100Stereo16,
                                                recordXlate44100Stereo8,
                                                recordXlate44100Mono16,
                                                recordXlate44100Mono8,

                                                recordXlate22050Stereo16,
                                                recordXlate22050Stereo8,
                                                recordXlate22050Mono16,
                                                recordXlate22050Mono8,

                                                recordXlate11025Stereo16,
                                                recordXlate11025Stereo8,
                                                recordXlate11025Mono16,
                                                recordXlate11025Mono8,
                      };

BufXlateWaveOutFuncPtr playXlateFunctions[] = {
                                                playXlateSilence,

                                                playXlate48000Stereo16,
                                                playXlate48000Stereo8,
                                                playXlate48000Mono16,
                                                playXlate48000Mono8,

                                                playXlate44100Stereo16,
                                                playXlate44100Stereo8,
                                                playXlate44100Mono16,
                                                playXlate44100Mono8,

                                                playXlate22050Stereo16,
                                                playXlate22050Stereo8,
                                                playXlate22050Mono16,
                                                playXlate22050Mono8,

                                                playXlate11025Stereo16,
                                                playXlate11025Stereo8,
                                                playXlate11025Mono16,
                                                playXlate11025Mono8,
                       };
                                       
