// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Intel Corporation
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#ifndef _MINWAVE_H_
#define _MINWAVE_H_


/*****************************************************************************
 * Constants
 *****************************************************************************
 */

/*****************************************************************************
 * Structures and Definitions
 */

// This structure is used to translate the TopoPins member to a (registered)
// system pin member or vice versa.
typedef struct
{
    WavePins    PinDef;
    int         PinNr;
} tPinTransTable;


// This structure is used to translate the TopoNodes member to a (registered)
// system node member or vice versa.
typedef struct
{
    WaveNodes   NodeDef;
    int         NodeNr;
} tNodeTransTable;

// max. number of connections
const int WAVE_MAX_CONNECTIONS = 0x80;

/*****************************************************************************
 * Forward References
 *****************************************************************************
 */
class CPortWavePci;
class CMiniportWaveICHStream;


extern NTSTATUS CreateMiniportWaveICHStream
(
    OUT     CMiniportWaveICHStream **   pWaveIchStream,
    IN      PUNKNOWN                    pUnknown,
    IN      POOL_TYPE                   PoolType
);

/*****************************************************************************
 * Classes
 *****************************************************************************
 */

/*****************************************************************************
 * CMiniportWaveICH
 *****************************************************************************
 * ICH wave PCI miniport.  This object is associated with the device and is
 * created when the device is started.  The class inherits IMiniportWavePci
 * so it can expose this interface, CUnknown so it automatically gets
 * reference counting and aggregation support, and IPowerNotify for ACPI
 * power management notification.
 */
//class CMiniportWaveICH : public IMiniportWavePci,
class CMiniportWaveICH : public IMiniportWaveICH,
                         public IPowerNotify,
                         public CUnknown
{
private:
    // The stream class accesses a lot of private member variables.
    // A better way would be to abstract the access through member
    // functions which on the other hand would produce more overhead
    // both in CPU time and programming time.
    friend class CMiniportWaveICHStream;

    //
    // CMiniportWaveICH private variables
    //
    CMiniportWaveICHStream *Streams[WAVE_PIN_TOP_ELEMENT];
    PPORTWAVEPCI        m_Port;           // Port driver object
    PADAPTERCOMMON      m_AdapterCommon;  // Adapter common object
    PINTERRUPTSYNC      m_InterruptSync;  // Interrupt Sync
    PDMACHANNEL         m_DmaChannel;     // Bus master support

    PMINIPORTTOPOLOGYICH m_pTopology;   // Topology manager object

    DEVICE_POWER_STATE  m_PowerState;   // advanced power control

    PSERVICEGROUP       m_pServiceGroupWave;

    short               m_sDeviceNumber;
    short               m_sBoardIndex;

    PPCFILTER_DESCRIPTOR     FilterDescriptor;       // Filter Descriptor
    PPCPIN_DESCRIPTOR        PinDescriptors;         // Pin Descriptors
    PPCNODE_DESCRIPTOR       NodeDescriptors;        // Node Descriptors
    PPCCONNECTION_DESCRIPTOR ConnectionDescriptors;  // Connection Descriptors

    tPinTransTable  *pstPinTrans;   // pin translation table
    tNodeTransTable *pstNodeTrans; // node translation table

    WORD                    m_wChannelNumber;    // ## FM (07/02/2000)
    WORD                    m_wChannelCount;

	DWORD				m_PinDataRangesIndex;

    BOOL                m_bConnectInterrupt;

    GUID                m_guidDevName;

	CProtocol* m_pDsp;

    // Set the Wave Device Number.
    void SetDeviceNumber
    (
        IN      short   DeviceNumber,
        IN      short   BoardIndex
    );

    /*************************************************************************
     * CMiniportWaveICH methods
     *************************************************************************
     * These are private member functions used internally by the object.  See
     * MINWAVE.CPP for specific descriptions.
     */

    //
    // Checks and connects the miniport to the resources.
    //
    NTSTATUS ProcessResources
    (
        IN   PRESOURCELIST     ResourceList
    );

    //
    // Tests the data format but not the sample rate.
    //
    NTSTATUS TestDataFormat
    (
        IN PKSDATAFORMAT Format,
        IN WavePins      Pin
    );

    // Test for standard sample rate support and fill the data range information
    // in the structures below.
    NTSTATUS BuildDataRangeInformation(   );

    // builds the topology.
    NTSTATUS BuildTopology
    (
        IN      ULONG   PmDeviceIndex,
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber,
        IN      ULONG   PmMaxWaveInOutNumber
    );

    // registers (builds) the pins
    NTSTATUS BuildPinDescriptors
    (
        IN      ULONG   PmDeviceIndex,
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber,
        IN      ULONG   PmMaxWaveInOutNumber
    );

    // registers (builds) the nodes
    NTSTATUS BuildNodeDescriptors
    (
        IN      ULONG   PmDeviceIndex,
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber,
        IN      ULONG   PmMaxWaveInOutNumber
    );

    // registers (builds) the connection between the pin, nodes.
    NTSTATUS BuildConnectionDescriptors
    (
        IN      ULONG   PmDeviceIndex,
        IN      ULONG   PmWaveInNumber,
        IN      ULONG   PmWaveOutNumber,
        IN      ULONG   PmMaxWaveInOutNumber
    );


#if (DBG)
    // dumps the topology. you can specify if you want to dump pins, nodes,
    // connections (see debug.h).
    void DumpTopology (void);
#endif

/*************************************************************************

    TransNodeNrToNodeDef

*************************************************************************/

    // translates the system node id to a TopoNode.
    WaveNodes TransNodeNrToNodeDef (IN int Node)
    {
#if (DBG)
        WaveNodes   n;

        n = pstNodeTrans[Node].NodeDef;
        // check for invalid translation. could be caused by a connection
        // to a not registered node or wrong use of nodes.
        if (n == NODE_INVALID)
        {
            DOUT (DBG_ERROR, ("Invalid node nr %u.", Node));
        }
        return n;
#else
        return pstNodeTrans[Node].NodeDef;
#endif
    };

/*************************************************************************

    TransNodeDefToNodeNr

*************************************************************************/

    // translates a TopoNode to a system node id.
    int TransNodeDefToNodeNr (IN WaveNodes Node)
    {
#if (DBG)
        int n;

        // check for invalid translation. could be caused by a connection
        // to a not registered node or wrong use of nodes.
        n = pstNodeTrans[Node].NodeNr;
        if (n == -1)
        {
            DOUT (DBG_ERROR, ("Invalid WaveNode %u.", Node));
        }
        return n;
#else
        return pstNodeTrans[Node].NodeNr;
#endif
    };

/*************************************************************************

    TransPinNrToPinDef

*************************************************************************/

    // translates a system pin id to a TopoPin.
    WavePins TransPinNrToPinDef (IN int Pin)
    {
#if (DBG)
        WavePins p;

        p = pstPinTrans[Pin].PinDef;
        // check for invalid translation. could be caused by a connection
        // to a not registered pin or wrong use of pins.
        if (p == PIN_INVALID)
        {
            DOUT (DBG_ERROR, ("Invalid pin nr %u.", Pin));
        }
        return p;
#else
        return pstPinTrans[Pin].PinDef;
#endif
    };

/*************************************************************************

    TransPinDefToPinNr

*************************************************************************/

    // translates a TopoPin to a system pin id.
    int TransPinDefToPinNr (IN WavePins Pin)
    {
#if (DBG)
        int p;

        p = pstPinTrans[Pin].PinNr;
        // check for invalid translation. could be caused by a connection
        // to a not registered pin or wrong use of pins.
        if (p == -1)
        {
            DOUT (DBG_ERROR, ("Invalid WavePin %u.", Pin));
        }
        return p;
#else
        return pstPinTrans[Pin].PinNr;
#endif
    };

/*************************************************************************

    SetNodeTranslation

*************************************************************************/

    // sets one table entry for translation.
    void SetNodeTranslation (IN int NodeNr, IN WaveNodes NodeDef , IN ULONG line)
    {
        pstNodeTrans[NodeNr].NodeDef = WaveNodes( NodeDef + line * WAVE_NODE_TOP_ELEMENT );
        pstNodeTrans[NodeDef + line * WAVE_NODE_TOP_ELEMENT].NodeNr = NodeNr;
    };

/*************************************************************************

    SetPinTranslation

*************************************************************************/

    // sets one table entry for translation.
    void SetPinTranslation (IN int PinNr, IN WavePins PinDef, IN ULONG line)
    {
        pstPinTrans[PinNr].PinDef = WavePins( PinDef + line * WAVE_PIN_TOP_ELEMENT );
        pstPinTrans[PinDef + line * WAVE_PIN_TOP_ELEMENT].PinNr = PinNr;
    };

public:
	
    /*************************************************************************
     * The following two macros are from STDUNK.H.  DECLARE_STD_UNKNOWN()
     * defines inline IUnknown implementations that use CUnknown's aggregation
     * support.  NonDelegatingQueryInterface() is declared, but it cannot be
     * implemented generically.  Its definition appears in MINWAVE.CPP.
     * DEFINE_STD_CONSTRUCTOR() defines inline a constructor which accepts
     * only the outer unknown, which is used for aggregation.  The standard
     * create macro (in MINWAVE.CPP) uses this constructor.
     */
    DECLARE_STD_UNKNOWN ();
    DEFINE_STD_CONSTRUCTOR (CMiniportWaveICH);
    ~CMiniportWaveICH ();

    //
    // Include IMiniportWavePci (public/exported) methods
    //
    IMP_IMiniportWavePci;

    //
    // IPowerNotify methods
    //
    IMP_IPowerNotify;

    /*************************************************************************
     * IMiniportWaveICH methods
     */
    STDMETHODIMP_(void) MyVery1stInit
    (
        IN      USHORT  DeviceNumber,
        IN      USHORT  BoardIndex,
        IN      USHORT  PmChannelFirst,
        IN      USHORT  PmChannelCount,
        IN      LPGUID  PmDeviceNameGuid,
        IN      BOOL    PmConnectInterrupt
    );

    // returns the system pin id's for wave out, wave in .
    STDMETHODIMP_(NTSTATUS) GetPhysicalConnectionPins
    (
        IN ULONG  Index,
        OUT PULONG  Wave
    );

    STDMETHODIMP_(void) SetTopologyRef
    (
        IN  PMINIPORTTOPOLOGYICH    PmTopologyRef
    );

    STDMETHODIMP_(NTSTATUS) PropertyHandlerComponentId
    (
        IN  PPCPROPERTY_REQUEST PropertyRequest
    );

    STDMETHODIMP_(NTSTATUS) PropertyHandlerProposedFormat
    (
        IN PPCPROPERTY_REQUEST      PropertyRequest
    );

    //
    // This static functions is the interrupt service routine which is
    // not stream related, but services all streams at once.
    //
    static NTSTATUS InterruptServiceRoutine
    (
        IN      PINTERRUPTSYNC  InterruptSync,
        IN      PVOID           StaticContext
    );

    NTSTATUS NotifyStream( BOOL PmIsRecord, DWORD PmAudio, BOOL PmDirectCall );

	//
    // This is the property handler for KSPROPERTY_AUDIO_CHANNEL_CONFIG of the
    // DAC node.
    //
    static NTSTATUS PropertyChannelConfig
    (
        IN      PPCPROPERTY_REQUEST PropertyRequest
    );



};

typedef CMiniportWaveICH *PCMiniportWaveICH;


VOID  DSNDNotifyStream(	IN LPVOID PmParentIchWave,
						IN BOOL   PmIsRecord,
						IN DWORD  PmAudio,
						IN BOOL   PmDirectCall );

#endif          // _MINWAVE_H_

