// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//  Portions Copyright (c) 1998-1999 Intel Corporation
//  Portions Copyright (c) 1998-1999 Microsoft Corporation. 
//
// **************************************************************************

#include "shared.h"

#include "PIOCommands.h"

#include "mintopo.h"

// max. number of connections
const int TOPO_MAX_CONNECTIONS = 0x80;


/*****************************************************************************
 * Topology miniport tables
 */

/*****************************************************************************
 * PinDataRangesBridge
 *****************************************************************************
 * Structures indicating range of valid format values for bridge pins.
 */
static KSDATARANGE PinDataRangesAnalogBridge[] =
{
   {
      sizeof(KSDATARANGE),
      0,
      0,
      0,
      STATICGUIDOF(KSDATAFORMAT_TYPE_AUDIO),
      STATICGUIDOF(KSDATAFORMAT_SUBTYPE_ANALOG),
      STATICGUIDOF(KSDATAFORMAT_SPECIFIER_NONE)
   }
};

/*****************************************************************************
 * PinDataRangePointersBridge
 *****************************************************************************
 * List of pointers to structures indicating range of valid format values
 * for audio bridge pins.
 */
static PKSDATARANGE PinDataRangePointersAnalogBridge[] =
{
    (PKSDATARANGE)&PinDataRangesAnalogBridge[0]
};


/*****************************************************************************
 * PropertiesVolume
 *****************************************************************************
 * Properties for volume controls.
 */
static PCPROPERTY_ITEM PropertiesVolume[] =
{
    { 
        &KSPROPSETID_Audio, 
        KSPROPERTY_AUDIO_VOLUMELEVEL,
        KSPROPERTY_TYPE_GET | KSPROPERTY_TYPE_SET | KSPROPERTY_TYPE_BASICSUPPORT,
        CMiniportTopologyICH::PropertyHandler_Level
    },
    {
        &KSPROPSETID_Audio,
        KSPROPERTY_AUDIO_CPU_RESOURCES,
        KSPROPERTY_TYPE_GET,
        CMiniportTopologyICH::PropertyHandler_CpuResources
    }
};

/*****************************************************************************
 * AutomationVolume
 *****************************************************************************
 * Automation table for volume controls.
 */
DEFINE_PCAUTOMATION_TABLE_PROP (AutomationVolume, PropertiesVolume);

/*****************************************************************************
 * PropertiesMute
 *****************************************************************************
 * Properties for mute controls.
 */
static PCPROPERTY_ITEM PropertiesMute[] =
{
    { 
        &KSPROPSETID_Audio, 
        KSPROPERTY_AUDIO_MUTE,
        KSPROPERTY_TYPE_GET | KSPROPERTY_TYPE_SET,
        CMiniportTopologyICH::PropertyHandler_OnOff
    },
    {
        &KSPROPSETID_Audio,
        KSPROPERTY_AUDIO_CPU_RESOURCES,
        KSPROPERTY_TYPE_GET,
        CMiniportTopologyICH::PropertyHandler_CpuResources
    }
};

/*****************************************************************************
 * AutomationMute
 *****************************************************************************
 * Automation table for mute controls.
 */
DEFINE_PCAUTOMATION_TABLE_PROP (AutomationMute, PropertiesMute);

/*****************************************************************************
 * PropertiesMux
 *****************************************************************************
 * Properties for mux controls.
 */
static PCPROPERTY_ITEM PropertiesMux[] =
{
    { 
        &KSPROPSETID_Audio, 
        KSPROPERTY_AUDIO_MUX_SOURCE,
        KSPROPERTY_TYPE_GET | KSPROPERTY_TYPE_SET,
        CMiniportTopologyICH::PropertyHandler_Ulong
    },
    {
        &KSPROPSETID_Audio,
        KSPROPERTY_AUDIO_CPU_RESOURCES,
        KSPROPERTY_TYPE_GET,
        CMiniportTopologyICH::PropertyHandler_CpuResources
    }
};

/*****************************************************************************
 * AutomationMux
 *****************************************************************************
 * Automation table for mux controls.
 */
DEFINE_PCAUTOMATION_TABLE_PROP (AutomationMux, PropertiesMux);


#pragma code_seg("PAGE")

/*****************************************************************************
 * CreateMiniportTopologyICH
 *****************************************************************************
 * Creates a topology miniport object for the ICH audio adapter.  This uses a
 * macro from STDUNK.H to do all the work.
 */
NTSTATUS CreateMiniportTopologyICH
(
    OUT     PUNKNOWN *  Unknown,
    IN      REFCLSID,
    IN      PUNKNOWN    UnknownOuter    OPTIONAL,
    IN      POOL_TYPE   PoolType
)
{
    PAGED_CODE ();

    ASSERT (Unknown);

    STD_CREATE_BODY_WITH_TAG (CMiniportTopologyICH, Unknown, UnknownOuter, PoolType, PCX_MINTOPO_TAG);
}

/*****************************************************************************
 * CMiniportTopologyICH::NonDelegatingQueryInterface
 *****************************************************************************
 * Obtains an interface.  This function works just like a COM QueryInterface
 * call and is used if the object is not being aggregated.
 * We basically just check any GUID we know and return this object in case we
 * know it.
 */
STDMETHODIMP_(NTSTATUS) CMiniportTopologyICH::NonDelegatingQueryInterface
(
    IN      REFIID  Interface,
    OUT     PVOID * Object
)
{
    PAGED_CODE ();

    ASSERT (Object);

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::NonDelegatingQueryInterface]"));

    // Is it IID_IUnknown?
    if (IsEqualGUIDAligned (Interface, IID_IUnknown))
    {
        *Object = (PVOID)(PUNKNOWN)this;
    }
    else
    // or IID_IMiniport ...
    if (IsEqualGUIDAligned (Interface, IID_IMiniport))
    {
        *Object = (PVOID)(PMINIPORT)this;
    }
    else
    // or IID_IMiniportTopology ...
    if (IsEqualGUIDAligned (Interface, IID_IMiniportTopology))
    {
        *Object = (PVOID)(PMINIPORTTOPOLOGY)this;
    }
    else
    // or maybe our IID_IMiniportTopologyICH ...
    if (IsEqualGUIDAligned (Interface, IID_IMiniportTopologyICH))
    {
        *Object = (PVOID)(PMINIPORTTOPOLOGYICH)this;
    }
    else
    {
        // nothing found, must be an unknown interface.
        *Object = NULL;
        return STATUS_INVALID_PARAMETER;
    }

    //
    // We reference the interface for the caller.
    //
    ((PUNKNOWN)(*Object))->AddRef ();
    return STATUS_SUCCESS;
}

/*****************************************************************************
 * CMiniportTopologyICH::~CMiniportTopologyICH
 *****************************************************************************
 * Destructor.
 */
CMiniportTopologyICH::~CMiniportTopologyICH ()
{
    PAGED_CODE ();

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::~CMiniportTopologyICH]"));

    // release all the stuff that we had referenced or allocated.
    if (Port)
    {
        Port->Release ();
        Port = NULL;
    }

    if (AdapterCommon)
    {
        AdapterCommon->Release ();
        AdapterCommon = NULL;
    }

    if (FilterDescriptor)
    {
        DDK_FREE_WITH_TAG (FilterDescriptor, PCX_MINTOPO_TAG);
        FilterDescriptor = NULL;
    }

    if (ConnectionDescriptors)
    {
        DDK_FREE_WITH_TAG (ConnectionDescriptors, PCX_MINTOPO_TAG);
        ConnectionDescriptors = NULL;
    }

    if (NodeDescriptors)
    {
        DDK_FREE_WITH_TAG (NodeDescriptors, PCX_MINTOPO_TAG);
        NodeDescriptors = NULL;
    }

    if (PinDescriptors)
    {
        DDK_FREE_WITH_TAG (PinDescriptors, PCX_MINTOPO_TAG);
        PinDescriptors = NULL;
    }

    if (pstNodeTrans)
    {
        DDK_FREE_WITH_TAG (pstNodeTrans, PCX_MINTOPO_TAG);
        pstNodeTrans = NULL;
    }

    if (pstPinTrans)
    {
        DDK_FREE_WITH_TAG (pstPinTrans, PCX_MINTOPO_TAG);
        pstPinTrans = NULL;
    }
}

/*****************************************************************************
 * CMiniportTopologyICH::DataRangeIntersection
 *****************************************************************************
 * Is defined in the IMiniportTopology interface. We just return
 * STATUS_NOT_IMPLEMENTED and portcls will use a default handler for this.
 */
STDMETHODIMP_(NTSTATUS) CMiniportTopologyICH::DataRangeIntersection
(
    IN      ULONG           /*PinId*/,
    IN      PKSDATARANGE    /*DataRange*/,
    IN      PKSDATARANGE    /*MatchingDataRange*/,
    IN      ULONG           /*OutputBufferLength*/,
    OUT     PVOID           /*ResultantFormat     OPTIONAL*/,
    OUT     PULONG          /*ResultantFormatLength*/
)
{
    return STATUS_NOT_IMPLEMENTED;
};

/*****************************************************************************
 * CMiniportTopologyICH::MyVery1stInit
 *****************************************************************************
 * Initializes the miniport indexes
 */

STDMETHODIMP_(void) CMiniportTopologyICH::MyVery1stInit
(
    IN      USHORT      PmDeviceIndex,
    IN      USHORT      PmBoardIndex,
    IN      USHORT      PmChannelFirst,
    IN      USHORT      PmChannelCount,
    IN      LPGUID      PmDeviceNameGuid
)
{
    PAGED_CODE ();

    m_sDeviceNumber = PmDeviceIndex;
    m_wBoardIndex   = PmBoardIndex;
    m_wChannelNumber = PmChannelFirst;
    m_wChannelCount = PmChannelCount;
    m_guidDevName   = *PmDeviceNameGuid;
}

/*****************************************************************************
 * CMiniportTopologyICH::Init
 *****************************************************************************
 * Initializes the miniport.
 * We initialize the translation tables, add reference to the port driver and
 * build the topology.
 */
STDMETHODIMP_(NTSTATUS) CMiniportTopologyICH::Init
(
    IN      PUNKNOWN        UnknownAdapter,
    IN      PRESOURCELIST   /*ResourceList*/,
    IN      PPORTTOPOLOGY   Port_
)
{
    PAGED_CODE ();

    ASSERT (UnknownAdapter);
    ASSERT (Port_);

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::Init]"));

    //
    // Set the copy protect flag to FALSE.
    //
    m_bCopyProtectFlag = FALSE;
    m_bDigitalOutputDisable = FALSE;

    m_bIsOutputPipeReady		= FALSE;
    m_bIsInputPipeReady			= FALSE;

    DOUT (DBG_TEST, ("CMiniportTopologyICH::Init\n"));

    //
    // AddRef() is required because we are keeping this pointer.
    //
    Port = Port_;
    Port->AddRef ();

    //
    // get the IAdapterCommon interface from the adapter.
    //
    NTSTATUS ntStatus = UnknownAdapter->QueryInterface (IID_IAdapterCommon,
                                                       (PVOID *) &AdapterCommon);

    // build the topology (means register pins, nodes, connections).
    if (NT_SUCCESS (ntStatus) && AdapterCommon)
    {
        ULONG l_ulWaveInNumber = AdapterCommon->GetWaveInNumber();
        ULONG l_ulWaveOutNumber = AdapterCommon->GetWaveOutNumber();

        ntStatus = BuildTopology (l_ulWaveInNumber, l_ulWaveOutNumber);
    }

    if (!NT_SUCCESS (ntStatus))
    {
        //
        // clean up our mess
        //

        // clean up AdapterCommon
        if (AdapterCommon)
        {
            AdapterCommon->Release ();
            AdapterCommon = NULL;
        }

        // release the port
        Port->Release ();
        Port = NULL;
    }
    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::GetDescription
 *****************************************************************************
 * Gets/returns the topology to the system.
 */
STDMETHODIMP_(NTSTATUS) CMiniportTopologyICH::GetDescription
(
    OUT     PPCFILTER_DESCRIPTOR *  OutFilterDescriptor
)
{
    PAGED_CODE ();

    ASSERT (OutFilterDescriptor);

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::GetDescription]"));

#if (DBG)
    // Dump it here. The system requests the topology only once.
    DumpTopology ();
#endif

    if (FilterDescriptor)
    {
        *OutFilterDescriptor = FilterDescriptor;
        return STATUS_SUCCESS;
    }
    else
        return STATUS_DEVICE_CONFIGURATION_ERROR;
}

#if (DBG)
/*****************************************************************************
 * CMiniportTopologyICH::DumpTopology
 *****************************************************************************
 * Dumps the topology for debugging.
 * See the defines at the beginning of this file?
 */
void CMiniportTopologyICH::DumpTopology (void)
{
    PAGED_CODE ();

    if (FilterDescriptor)
    {
        // dump the pins
        DOUT (DBG_PINS, ("TOPOLOGY MINIPORT PINS"));
        for(ULONG index = 0; index < FilterDescriptor->PinCount; index++)
        {
            DOUT (DBG_PINS, ("  %2d %s", index,
                             TopoPinStrings[TransPinNrToPinDef (index) % PIN_TOP_ELEMENT]));
        }

        // dump the nodes
        DOUT (DBG_NODES, ("TOPOLOGY MINIPORT NODES"));
        for(index = 0; index < FilterDescriptor->NodeCount; index++)
        {
            DOUT (DBG_NODES, ("  %2d %s", index,
                              NodeStrings[TransNodeNrToNodeDef (index) %  NODE_TOP_ELEMENT]));
        }

        // dump the connections
        DOUT (DBG_CONNS, ("TOPOLOGY MINIPORT CONNECTIONS"));
        for(index = 0; index < FilterDescriptor->ConnectionCount; index++)
        {
            ULONG FromNode = FilterDescriptor->Connections[index].FromNode;
            ULONG FromNodePin = FilterDescriptor->Connections[index].FromNodePin;
            ULONG ToNode = FilterDescriptor->Connections[index].ToNode;
            ULONG ToNodePin = FilterDescriptor->Connections[index].ToNodePin;

            DOUT (DBG_CONNS, ("  %2d (%d,%d)->(%d,%d)", index,
                FromNode,
                FromNodePin,
                ToNode,
                ToNodePin));
        }
    }
}
#endif


/*****************************************************************************
 * Miniport Topology               V = Volume, M = Mute, L = Loudness, A = AGC
 *====================             T = Treble, B = Bass, 0-9 = PinNr of node
 *
 * PCBEEP ---> V ---> M -----------------------------\
 * PHONE  ---> V ---> M ----------------------------\ \
 *                                                   \ \
 * WaveOut ------> V --> M ---------------> 1+-----+  \ \
 * MIC1 or MIC2 --> L --> A --> V --> M --> 2| SUM |   \ \
 * LineIn -------> V --> M ---------------> 3|     |    \ \
 * CD     -------> V --> M ---------------> 4|     |0--->SUM--> T --> B --> L --> A +-> V --> M (MasterOut)
 * Video  -------> V --> M ---------------> 5|     |                                |
 * AUX    -------> V --> M ---------------> 6|     |                                +-> V --> M (Headphone)
 * 3D Depth  -------> V --> M ------------> 7|     |
 * 3D Center -------> V --> M ------------> 8+-----+
 *
 *
 * virt. Pin: Tone mix mono   ---> V --> M ---> 7+-------+
 * virt. Pin: Tone mix stereo ---> V --> M ---> 6|       |
 * Phone                      ---> V --> M ---> 8|   M   |
 * Mic (after AGC)            ---> V --> M ---> 1|       |0-----> (WaveIn)
 * LineIn                     ---> V --> M ---> 5|   U   |
 * CD                         ---> V --> M ---> 2|       |
 * Video                      ---> V --> M ---> 3|   X   |
 * AUX                        ---> V --> M ---> 4+-------+
 *
 *
 * virt. Pin: 3D mix mono ---> V ---> M ---> 1+-----+
 *                                            | MUX |0----> (MonoOut)
 * Mic (after AGC)        ---> V ---> M ---> 2+-----+
 *
 *
 * Mic (after AGC) ----> V ----> M -----> (MicIn)
 *
 *----------------------------------------------------------------------------
 *
 * As you can see, the exposed topology is somewhat different from the real AC97
 * topology. This is because the system that translates the topology to "mixer
 * lines" gets confused if it has to deal with all the mess. So we have to make it
 * plain and simple for the system.
 * Some issues block us from exposing a nice plain and simple topology. The prg.
 * which displayes the "Volume Control Panel" (sndvol32) does _only_ display
 * Volumes, Mutes (only one "in a row"), Treble, Bass, Loudness and AGC under
 * Advanced control panel. We don't have 3D controls, and before we go into a
 * Muxer, there has to be Volume controls in front.
 * So what do we do?
 * 1) We fake 3D controls as Volume controls. The Mutes represent 3D bypass and
 *    3D on/off
 * 2) All inputs (including the 3D controls) go staight into a SUM. Important is
 *    that there are not 2 Volumes, Mutes in a row, e.g. ---> V ---> M ---> V ---> M
 *    In that case, only one Volume/Mute would be displayed.
 * 3) We can't make a connection from the tone controls to the Wave In muxer (even
 *    with Volumes in front), so we create fake pins that we name user friendly.
 *    Same with the connection from the 3D mixer to the Mono output.
 * 4) We discard all supermixer controls that would convert stereo to mono or vice
 *    versa. Instead, we just connect the lines and when the control is queried we
 *    fail a right channel request (mono has only left channel).
 * 5) We have to make virtual volume and mute controls in front of each muxer.
 *    As you can see, these controls can be mono or stereo and there is only one
 *    HW register for them, so we have to cache the values and prg. the register
 *    each time the select changes or the selected volume control changes.
 */

/*****************************************************************************
 * CMiniportTopologyICH::BuildTopology
 *****************************************************************************
 * Builds the topology descriptors based on hardware configuration info
 * obtained from the adapter.
 */
NTSTATUS CMiniportTopologyICH::BuildTopology
(
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber
)
{
    PAGED_CODE ();

    NTSTATUS ntStatus = STATUS_SUCCESS;

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::BuildTopology]"));

    // allocate our filter descriptor
    FilterDescriptor = (PPCFILTER_DESCRIPTOR) DDK_ALLOC_WITH_TAG (PagedPool,
                        sizeof(PCFILTER_DESCRIPTOR), PCX_MINTOPO_TAG);
    if (FilterDescriptor)
    {
        // clear out the filter descriptor
        RtlZeroMemory (FilterDescriptor, sizeof(PCFILTER_DESCRIPTOR));

        // build the pin list
        ntStatus = BuildPinDescriptors (PmWaveInNumber,
                                        PmWaveOutNumber);

        if (NT_SUCCESS (ntStatus))
        {
            // build the node list
            ntStatus = BuildNodeDescriptors (PmWaveInNumber,
                                             PmWaveOutNumber);
            if (NT_SUCCESS (ntStatus))
            {
                // build the connection list
                ntStatus = BuildConnectionDescriptors (PmWaveInNumber,
                                                       PmWaveOutNumber);
            }
        }
    }
    else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

    // that's whatever one of these build... functions returned.
    return ntStatus;
}

/*****************************************************************************
 * CMiniportTopologyICH::BuildPinDescriptors
 *****************************************************************************
 * Builds the topology pin descriptors.
 */
NTSTATUS CMiniportTopologyICH::BuildPinDescriptors
(
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.
#define INIT_PIN( pin, lineNumber, pinptr, category, name, index )      \
    pinptr->KsPinDescriptor.Category = (GUID*) category;    \
    pinptr->KsPinDescriptor.Name = (GUID*) name;            \
    SetPinTranslation (index++, pin , lineNumber );                       \
    pinptr++

    PAGED_CODE ();

    ULONG               Index;
    PPCPIN_DESCRIPTOR   CurrentPin;

    int MemorySize;

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::BuildPinDescriptors]"));

    MemorySize = PIN_TOP_ELEMENT * sizeof(tPinTranslationTable);

    pstPinTrans = (tPinTranslationTable*) (DDK_ALLOC_WITH_TAG(NonPagedPool,MemorySize, PCX_MINTOPO_TAG));
    //
    // initialize translation tables
    //
    if( !pstPinTrans)
        return STATUS_INSUFFICIENT_RESOURCES;

    ULONG i;
    for (i = 0; i < PIN_TOP_ELEMENT ; i++)
    {

        pstPinTrans[i].PinDef = PIN_INVALID;
        pstPinTrans[i].PinNr = -1;
    }

    // allocate our descriptor memory
    MemorySize = (PIN_WAVEOUT_NUMBER) * sizeof(PCPIN_DESCRIPTOR);
    MemorySize += (PIN_WAVEIN_NUMBER) * sizeof(PCPIN_DESCRIPTOR);
    PinDescriptors = PPCPIN_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool, MemorySize, PCX_MINTOPO_TAG));
    if (!PinDescriptors)
        return STATUS_INSUFFICIENT_RESOURCES;

    //
    // set default pin descriptor parameters
    //
    RtlZeroMemory (PinDescriptors, MemorySize);

    // spend some more time and set the pin descriptors to expected values.
    for (CurrentPin = PinDescriptors, Index = 0;
         Index < (ULONG) PIN_TOP_ELEMENT ;
         CurrentPin++, Index++)
    {
        CurrentPin->KsPinDescriptor.DataRangesCount = SIZEOF_ARRAY(PinDataRangePointersAnalogBridge);
        CurrentPin->KsPinDescriptor.DataRanges      = PinDataRangePointersAnalogBridge;
        CurrentPin->KsPinDescriptor.DataFlow        = KSPIN_DATAFLOW_IN;
        CurrentPin->KsPinDescriptor.Communication   = KSPIN_COMMUNICATION_NONE;
    }

    //
    // modify the individual pin descriptors
    //
    CurrentPin  = PinDescriptors;
    Index       = 0;

    BOOL bIsVISTA = IoIsWdmVersionAvailable(6, 0x00);   // Windows VISTA

    // ************** ALL WAVE OUT **************************

    if ( m_sDeviceNumber < PmWaveOutNumber )
    {
        ULONG lineOut =0;

        // add the PIN_WAVEOUT_SOURCE pin descriptor (not optional)

        // already initialized :
        // CurrentPin->KsPinDescriptor.DataFlow = KSPIN_DATAFLOW_IN;
        //
        INIT_PIN (PIN_WAVEOUT_SOURCE,
                  lineOut,
                  CurrentPin,
                  &KSCATEGORY_AUDIO,
                  NULL,
                  Index);

		// add the PIN_LINEOUT pin descriptor (not optional)
		CurrentPin->KsPinDescriptor.DataFlow = KSPIN_DATAFLOW_OUT;
		//

        if ( bIsVISTA )   // Windows VISTA
        {
			DOUT (DBG_WARNING, ("[BuildPinDescriptors] LINE_OUT : %x %x %x %x%x%x%x%x%x%x%x\n", m_guidDevName.Data1, m_guidDevName.Data2, m_guidDevName.Data3, m_guidDevName.Data4[0], m_guidDevName.Data4[1], m_guidDevName.Data4[2], m_guidDevName.Data4[3], m_guidDevName.Data4[4], m_guidDevName.Data4[5], m_guidDevName.Data4[6], m_guidDevName.Data4[7]));

		    INIT_PIN (PIN_LINEOUT,
				      lineOut,
				      CurrentPin,
				      &KSNODETYPE_LINE_CONNECTOR,	// TEST
	    		      &m_guidDevName,
				      Index);
        }
        else
        {
		    INIT_PIN (PIN_LINEOUT,
				      lineOut,
				      CurrentPin,
    			      &KSNODETYPE_SPEAKER,
	    		      &KSAUDFNAME_VOLUME_CONTROL,
				      Index);
        }
	}

    // ************** ALL WAVE IN **************************

    ULONG lineIn =0;

    if ( m_sDeviceNumber < PmWaveInNumber )
    {
        CurrentPin->KsPinDescriptor.DataFlow = KSPIN_DATAFLOW_IN;

        if ( bIsVISTA )   // Windows VISTA
        {
			DOUT (DBG_WARNING, ("[BuildPinDescriptors] LINE_IN : %x %x %x %x%x%x%x%x%x%x%x\n", m_guidDevName.Data1, m_guidDevName.Data2, m_guidDevName.Data3, m_guidDevName.Data4[0], m_guidDevName.Data4[1], m_guidDevName.Data4[2], m_guidDevName.Data4[3], m_guidDevName.Data4[4], m_guidDevName.Data4[5], m_guidDevName.Data4[6], m_guidDevName.Data4[7]));

		    INIT_PIN (PIN_DIGITALIN,
				  lineIn,
				  CurrentPin,
				  &KSNODETYPE_LINE_CONNECTOR,
				  &m_guidDevName,
				  Index);
        }
        else
        {
		    INIT_PIN (PIN_DIGITALIN,
				  lineIn,
				  CurrentPin,
				  &KSNODETYPE_DIGITAL_AUDIO_INTERFACE,
				  NULL,
				  Index);
        }

        // add the PIN_WAVEIN_DEST pin descriptor (not optional)
        CurrentPin->KsPinDescriptor.DataFlow = KSPIN_DATAFLOW_OUT;
		//
        INIT_PIN (PIN_WAVEIN_DEST,
                  lineIn,
                  CurrentPin,
                  &KSCATEGORY_AUDIO,
                  NULL,
                  Index);
    }

    // add the pin descriptor information to the filter descriptor
    FilterDescriptor->PinCount = Index;
    FilterDescriptor->PinSize = sizeof (PCPIN_DESCRIPTOR);
    FilterDescriptor->Pins = PinDescriptors;


    return STATUS_SUCCESS;

#undef INIT_PIN
}

/*****************************************************************************
 * CMiniportTopologyICH::BuildNodeDescriptors
 *****************************************************************************
 * Builds the topology node descriptors.
 */
NTSTATUS CMiniportTopologyICH::BuildNodeDescriptors
(
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.
#define INIT_NODE( node, lineNumber, nodeptr, type, name, automation, index )   \
    nodeptr->Type = (GUID*) type;                                   \
    nodeptr->Name = (GUID*) name;                                   \
    nodeptr->AutomationTable = automation;                          \
    SetNodeTranslation (index++, node, lineNumber);                             \
    nodeptr++

    PAGED_CODE ();

    NTSTATUS ntStatus = STATUS_SUCCESS;
    int MemorySize;

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::BuildNodeDescriptors]"));

    MemorySize = NODE_TOP_ELEMENT * sizeof(tNodeTranslationTable);
    pstNodeTrans = (tNodeTranslationTable*) (DDK_ALLOC_WITH_TAG(NonPagedPool,MemorySize, PCX_MINTOPO_TAG));

    if( ! pstNodeTrans)
        return STATUS_INSUFFICIENT_RESOURCES;

    //
    // initialize translation tables
    //
    ULONG i;
    for (i = 0; i < NODE_TOP_ELEMENT ; i++)
    {
        pstNodeTrans[i].NodeDef = NODE_INVALID;
        pstNodeTrans[i].NodeNr = -1;
    }

    // allocate our descriptor memory
    MemorySize = (NODE_WAVEOUT_NUMBER) * sizeof(PCNODE_DESCRIPTOR);
    MemorySize += (NODE_WAVEIN_NUMBER) * sizeof(PCNODE_DESCRIPTOR);

    // clear out the cached settings for nodes
    RtlZeroMemory (stMultiNodeCache, NODE_TOP_ELEMENT);

    NodeDescriptors = PPCNODE_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool,MemorySize, PCX_MINTOPO_TAG));
    if (NodeDescriptors)
    {
        PPCNODE_DESCRIPTOR  CurrentNode = NodeDescriptors;
        ULONG               Index = 0;

        //
        // set default node descriptor parameters
        //
        RtlZeroMemory (NodeDescriptors, MemorySize);

        // We don't have loopback mode currently. It is only used for testing anyway.

        // ************** ALL WAVE OUT **************************

        if ( m_sDeviceNumber < PmWaveOutNumber )
        {
            ULONG lineOut =0;

			// add the NODE_MASTEROUT_VOLUME node
			INIT_NODE (NODE_MASTEROUT_VOLUME,
					   lineOut,
					   CurrentNode,
					   &KSNODETYPE_VOLUME,
					   &KSAUDFNAME_MASTER_VOLUME,
					   &AutomationVolume,
					   Index);

			// add the NODE_MASTEROUT_MUTE node
			INIT_NODE (NODE_MASTEROUT_MUTE,
					   lineOut,
					   CurrentNode,
					   &KSNODETYPE_MUTE,
					   &KSAUDFNAME_MASTER_MUTE,
					   &AutomationMute,
					   Index);
		}

        // ************** ALL WAVE IN **************************

        ULONG lineIn =0;

        if ( m_sDeviceNumber < PmWaveInNumber )
        {
			// add the NODE_DIGITALIN_VOLUME node
			INIT_NODE (NODE_DIGITALIN_VOLUME,
					   lineIn,
					   CurrentNode,
					   &KSNODETYPE_VOLUME,
					   &KSAUDFNAME_WAVE_VOLUME,
					   &AutomationVolume,
					   Index);

            // add the NODE_INPUT_SELECT node
		    INIT_NODE (NODE_INPUT_SELECT,
				       lineIn,
				       CurrentNode,
				       &KSNODETYPE_MUX,
				       &KSAUDFNAME_RECORDING_SOURCE,
				       &AutomationMux,
				       Index);
        }

        // add the nodes to the filter descriptor
        FilterDescriptor->NodeCount = Index;
        FilterDescriptor->NodeSize = sizeof(PCNODE_DESCRIPTOR);
        FilterDescriptor->Nodes = NodeDescriptors;

    }
    else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

    return ntStatus;

#undef INIT_NODE
}

/*****************************************************************************
 * CMiniportTopologyICH::BuildConnectionDescriptors
 *****************************************************************************
 * Builds the topology connection descriptors.
 */
NTSTATUS CMiniportTopologyICH::BuildConnectionDescriptors
(
    IN      ULONG   PmWaveInNumber,
    IN      ULONG   PmWaveOutNumber
)
{
// Improvement would be to not use a Macro, use (inline) function instead.

// for filter pin to node connections
#define INIT_FN_CONN( cptr, fpin, tnode, tpin )         \
    cptr->FromNode = KSFILTER_NODE;                     \
    cptr->FromNodePin = TransTopoPinDefToPinNr (fpin);      \
    cptr->ToNode = TransTopoNodeDefToNodeNr (tnode);        \
    cptr->ToNodePin = tpin;                             \
    cptr++,ConnectionCount++

// for node to node connections
#define INIT_NN_CONN( cptr, fnode, fpin, tnode, tpin )  \
    cptr->FromNode = TransTopoNodeDefToNodeNr (fnode);      \
    cptr->FromNodePin = fpin;                           \
    cptr->ToNode = TransTopoNodeDefToNodeNr (tnode);        \
    cptr->ToNodePin = tpin;                             \
    cptr++,ConnectionCount++



// for node to filter pin connections
#define INIT_NF_CONN( cptr, fnode, fpin, tpin )         \
    cptr->FromNode = TransTopoNodeDefToNodeNr (fnode);      \
    cptr->FromNodePin = fpin;                           \
    cptr->ToNode = KSFILTER_NODE;                       \
    cptr->ToNodePin = TransTopoPinDefToPinNr (tpin);        \
    cptr++,ConnectionCount++

    PAGED_CODE ();

    NTSTATUS    ntStatus            = STATUS_SUCCESS;
    ULONG       ConnectionCount     = 0;

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::BuildConnectionDescriptors]"));

    // allocate our descriptor memory
    ConnectionDescriptors = PPCCONNECTION_DESCRIPTOR (DDK_ALLOC_WITH_TAG (PagedPool,
                            TOPO_MAX_CONNECTIONS * sizeof(PCCONNECTION_DESCRIPTOR), PCX_MINTOPO_TAG));
    if (ConnectionDescriptors)
    {
        PPCCONNECTION_DESCRIPTOR  CurrentConnection = ConnectionDescriptors;

        // ************** ALL WAVE OUT **************************

        if ( m_sDeviceNumber < PmWaveOutNumber )
        {
            ULONG lineOut = 0;

            // build the wave out (coming in) path

            ULONG l_ulPinOffset = lineOut * PIN_TOP_ELEMENT;
            ULONG l_ulNodeOffset = lineOut * NODE_TOP_ELEMENT;

            // Build WaveOUT => MAIN_MIX path
            //

			// PIN_WAVEOUT_SOURCE -> NODE_MASTEROUT_VOLUME
			INIT_FN_CONN (CurrentConnection, 
						  TopoPins(PIN_WAVEOUT_SOURCE + l_ulPinOffset), 
						  TopoNodes(NODE_MASTEROUT_VOLUME + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_IN);

			// NODE_WAVEOUT_VOLUME -> NODE_MASTEROUT_MUTE
			INIT_NN_CONN (CurrentConnection, 
						  TopoNodes(NODE_MASTEROUT_VOLUME + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_OUT, 
						  TopoNodes(NODE_MASTEROUT_MUTE + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_IN);

			// NODE_MASTEROUT_MUTE -> PIN_LINEOUT
			INIT_NF_CONN (CurrentConnection, 
						  TopoNodes(NODE_MASTEROUT_MUTE + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_OUT, 
						  TopoPins(PIN_LINEOUT + l_ulPinOffset));
		}

        // ************** ALL WAVE IN **************************

        ULONG lineIn = 0;

        if ( m_sDeviceNumber < PmWaveInNumber )
        {
            ULONG l_ulPinOffset = lineIn * PIN_TOP_ELEMENT;
            ULONG l_ulNodeOffset = lineIn * NODE_TOP_ELEMENT;

            // Build Digital -> input select path
            //

            // PIN_DIGITALIN -> NODE_DIGITALIN_VOLUME
			INIT_FN_CONN (CurrentConnection, 
						  TopoPins(PIN_DIGITALIN + l_ulPinOffset), 
						  TopoNodes(NODE_DIGITALIN_VOLUME + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_IN);

            // NODE_DIGITALIN_VOLUME -> NODE_INPUT_SELECT
			INIT_NN_CONN (CurrentConnection, 
						  TopoNodes(NODE_DIGITALIN_VOLUME + l_ulNodeOffset), 
						  KSNODEPIN_STANDARD_OUT, 
						  TopoNodes(NODE_INPUT_SELECT + l_ulNodeOffset), 
						  KSNODEPIN_SUM_MUX_IN + 0);

            // NODE_INPUT_SELECT -> PIN_WAVEIN_DEST
			INIT_NF_CONN (CurrentConnection, 
						  TopoNodes(NODE_INPUT_SELECT + l_ulNodeOffset), 
						  KSNODEPIN_SUM_MUX_OUT, 
						  TopoPins(PIN_WAVEIN_DEST + l_ulPinOffset));
		}

        // add the connections to the filter descriptor
        FilterDescriptor->ConnectionCount = ConnectionCount;
        FilterDescriptor->Connections = ConnectionDescriptors;
    } else
    {
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

    return ntStatus;

#undef INIT_NN_CONN
#undef INIT_FN_CONN
#undef INIT_NF_CONN
}

/*****************************************************************************
 * CMiniportTopologyICH::GetPhysicalConnectionPins
 *****************************************************************************
 * Returns the system pin IDs of the bridge pins that are connected with the
 * wave miniport.
 * If one pin is not used, the value is -1, that could only happen for MinInDest.
 */
STDMETHODIMP CMiniportTopologyICH::GetPhysicalConnectionPins
(
    IN ULONG  Index,
    OUT PULONG  Wave
)
{
    PAGED_CODE ();

    ASSERT (Wave);

    // set the pin IDs.
    *Wave = TransTopoPinDefToPinNr (TopoPins(Index));

    DOUT (DBG_PRINT, ("[CMiniportTopologyICH::GetPhysicalConnectionPins %d : %d]",Index, *Wave ));

    return STATUS_SUCCESS;
}


/*****************************************************************************
/*****************************************************************************
/*****************************************************************************
 */
#pragma code_seg()

