// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#ifndef __LXPROTOCOL_CPP__
#define __LXPROTOCOL_CPP__
#endif

#include "drvdbg.h"
#include "drvdef.h"
#include "lxprotocol.h"

#ifdef SIMLXES
extern "C" {
extern DWORD		Sim_PLX_Registers[];
}
#endif

//////////////////////////////////////////////////////////////////////
// Construction  (currently called only once at driver startup)
//
CLXProtocol::CLXProtocol( PDSP_INFO PmpDspInfo, CPIOCommands *PmpPIO) : CProtocol( PmpDspInfo, PmpPIO) 
{
	DOUT (DBG_PROT_INIT, ("CLXProtocol::CLXProtocol() PmpPIO = %p\n",PmpPIO));

    WORD	i = 0;

    ASSERT(m_PIO);

	// Initialize the chaining of STREAM_INFO data
    // for both out and in streams	
    // ********************************************
	BZERO2( m_Out_Stream_Info_Array, STREAM_INFO, MAX_OUTSTREAM);
	BZERO2( m_In_Stream_Info_Array,  STREAM_INFO, MAX_INSTREAM);

	m_PCMOnlyGranularity = MICROBLAZE_IBL_DEFAULT; // default for LX boards

	m_K	= 0;

	m_Pipe_In_StartedMask		=0;
	m_Pipe_Out_StartedMask		=0;
    m_Pipe_In_WaitToggleMask    =0;
    m_Pipe_Out_WaitToggleMask   =0;
	m_Stream_In_MutedMask		=0;
	m_Stream_Out_MutedMask		=0;

    // Prompt Board for IOs (same whatever software is loaded)

    WORD LcOutputs = 0;
    WORD LcInputs = 0;

#ifndef SIMLXES
    m_PIO->PIOGetNbIOs(&LcOutputs, &LcInputs);
#else
    LcOutputs = 32;
    LcInputs = 32;
#endif

    // Stereo number of IO's -> mono
    LcOutputs *= 2;
    LcInputs *= 2;

    ASSERT(LcOutputs<=MAX_CARD_OUTPUTS);
    ASSERT(LcOutputs>=2);
    ASSERT(LcInputs<=MAX_CARD_INPUTS);
    ASSERT(LcInputs>=2);

    // Convert audio numbers into masks with one bit set
    // for each audio present
    // Physical audio are named from 0 to n - 1
    // Virtual audio are named from n to n + p - 1
    // ---------------------------------------------------

    for ( i = 0 ; i < LcOutputs ; i++ )  {
        m_pDSPInfo->dsMaskVPOut |= UTIMask64bit(i); // assuming the board only has 1 dsp CHIP
    }
    for ( i = 0 ; i < LcInputs ; i++ ) 
        m_pDSPInfo->dsMaskVPIn  |= UTIMask64bit(i);

	m_pDSPInfo->dsMaxBoardOutStream = (BYTE)LcOutputs;
    m_pDSPInfo->dsMaxPipeOutStream  = (BYTE)LcInputs;

	m_pDSPInfo->dsManagedPhysicalOutputNumber	= LcOutputs;
	m_pDSPInfo->dsManagedPhysicalInputNumber	= LcInputs;
	m_pDSPInfo->dsManagedVirtualOutputNumber	= 0;
	m_pDSPInfo->dsManagedVirtualInputNumber		= 0;
    m_pDSPInfo->dsMaskVVOut			= 0;
    m_pDSPInfo->dsMaskVVIn			= 0;


    // Initialize the chaining of STREAM_INFO data
    // for both out and in streams
    // ********************************************
    for ( i = 0; i < (MAX_OUTSTREAM - 1) ; i++ ) {
        m_Out_Stream_Info_Array[i].siAdrNextStream = &m_Out_Stream_Info_Array[i + 1];
    }

    m_Out_Stream_Info_Array[MAX_OUTSTREAM-1].siAdrNextStream = NULL;
    m_StreamOutFreeList = m_Out_Stream_Info_Array;

    for ( i = 0; i < (MAX_INSTREAM - 1) ; i++ ) {
        m_In_Stream_Info_Array[i].siAdrNextStream = &m_In_Stream_Info_Array[i + 1];
    }
    
    m_In_Stream_Info_Array[MAX_INSTREAM-1].siAdrNextStream = NULL;
    m_StreamInFreeList = m_In_Stream_Info_Array;

    // must be init later, when the version of the dsp software is checked
    //
    m_PlayFormats   = 0;
    m_RecordFormats = 0;

	m_PCMOnlyGranularity = MICROBLAZE_IBL_MIN;
	
} // End of regular Constructor



WORD	CLXProtocol:: IDiag_HaltEmbedded()
{
	WORD        LcRet = SUCCESS;
	KIRQL       LcOldIrql;

	if  ((m_Pipe_Out_StartedMask == (DWORDLONG)0) && (m_Pipe_In_StartedMask == (DWORDLONG)0 )) 
	{
		if (IDiag_IsDspRunning())
		{

		// park microblaze

		/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

		// Fill CMD_RMH_INFO structure
		InitMessage( CMD_00_INFO_DEBUG );

		m_Rmh.Cmd[0] |= 0xAA;

		LcRet = SendMessage(  ) ;

		m_pDSPInfo->dsEtat = 0;			// microblaze parked
		m_pDSPInfo->dsFatalError = 0;	// reset fatal error
 
		/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

		}
		return LcRet ;	

	}
	
	return ED_FREE_DSP_REFUSED;
}



//**************************************************************************
//
// WORD CLXProtocol::IInit_GetVersionAndFeatures( IN BYTE PmSoftNum, PGENERAL_INFO PmGenInfo  )
//
// Return value:
// *************
//
// TRUE if the DSP software version matches the driver's one
//
//**************************************************************************
WORD   CLXProtocol:: IInit_GetVersionAndFeatures( BYTE PmSoftNum, PGENERAL_INFO PmGenInfo, PDWORD PmDspVersion )
{
	WORD        LcRet = SUCCESS;
    DWORD       LcExpectedVersion ;
    DWORD       LcDspVersion ;

	LcExpectedVersion =	PmGenInfo->PreferredFirmwareVersion;

	DOUT (DBG_PROT_INIT, ("CLXProtocol::IInit_GetVersionAndFeatures() PmSoftNum = %x\n",PmSoftNum));

    KIRQL LcOldIrql;
    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill message structure
    InitMessage( CMD_01_GET_SYS_CFG );

	LcRet = SendMessage();

	LcDspVersion	= m_Rmh.Stat[1];

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	if(PmDspVersion) *PmDspVersion = LcDspVersion;

	if (PmGenInfo == NULL) return LcRet; // only update DSP version, after firmware update.

    if ( LcRet == SUCCESS )
    {
        switch ( PmGenInfo->PDriverInfo->drDoNotCheckForVersionState ) {

        case 'B' :
        case '-' :
            LcDspVersion		&= 0xFFFF00;
            LcExpectedVersion	&= 0xFFFF00;
            break;

        case 'R' :
            LcDspVersion		&= 0xFF0000;
            LcExpectedVersion	&= 0xFF0000;
            break;

        default  :
            break;
        }

        if ( LcDspVersion != LcExpectedVersion )
        {
            LcRet = ED_DSP_VERSION_MISMATCH ;
        }

        // retrieve Play/Rec features
        // done here because we may have to handle alternate DSP files.
        //
        if( LcRet == SUCCESS )
        {
            LcRet = vio_SupportedPRFeatures();
        }

        // Init the EtherSound sample rate
        //
        if( LcRet == SUCCESS )
        {
            DWORD LcFrequency;
            BOOLEAN LcDummy;
            LcRet = IClock_BoardGetClockFrequency( &LcFrequency, &LcDummy);
            if(LcRet == SUCCESS)    m_PIO->PIOSetActualClockFrequency(LcFrequency);
        }
    }
    else
    {
        LcRet = ED_DSP_CORRUPTED ;
    }

    EXIT_PCX_ERROR( LcRet );

	return LcRet;
	
} // IInit_GetVersionAndFeatures

//**************************************************************************
//
// WORD CLXProtocol::IDiag_HandleBoardIBL(IN PTARGET_INFO)
//
//**************************************************************************
WORD CLXProtocol::IDiag_HandleBoardIBL( INOUT LPIBL_RESP_INFO PmPtrIBLRep )
{
	WORD        LcRet			= SUCCESS   ;
	WORD        LcSetIBL ;
    KIRQL		LcOldIrql;

	// last value set, or default.
	//
	WORD        LcSnappedIBL	= m_PCMOnlyGranularity;
	
	DOUT (DBG_PROT_VIO, ("CLXProtocol::IDiag_HandleBoardIBL"));

	PmPtrIBLRep->iblMax			= MICROBLAZE_IBL_MAX;
	PmPtrIBLRep->iblMin			= MICROBLAZE_IBL_MIN;
	PmPtrIBLRep->iblGranularity = 0xFFFF;	// does not apply, only discrete values are supported

	// tester s'il s'agit du get/set
	//
	LcSetIBL = PmPtrIBLRep->iblSamples;

	if (LcSetIBL)		// Set 
	{
		LcSnappedIBL = MICROBLAZE_IBL_MIN;

		while ( (LcSnappedIBL < LcSetIBL) &&  (LcSnappedIBL < MICROBLAZE_IBL_MAX) ) { LcSnappedIBL *=2; };

		/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);
		// Fill CMD_RMH_INFO structure
		InitMessage( CMD_02_SET_GRANULARITY );

		m_Rmh.Cmd[0] |= LcSnappedIBL ;  
		
		// Send RMH
		LcRet = SendMessage();
		
		/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

		if (LcSnappedIBL != LcSetIBL) 
		{
			LcRet = WD_IBL_CLIPPED;

			DOUT(DBG_WARNING, ("WARNING IBL clipped : (Min %d) < (Set %d)  < (Max%d)\n",
										PmPtrIBLRep->iblMin     ,
										PmPtrIBLRep->iblSamples ,
										PmPtrIBLRep->iblMax ));

		}
	}

	PmPtrIBLRep->iblSamples = LcSnappedIBL;
	
	// update m_PCMOnlyGranularity
	//
	m_PCMOnlyGranularity = LcSnappedIBL;

	DOUT (DBG_PROT_VIO, ("HandleBoardIBL() -> m_PCMOnlyGranularity = %x\n", m_PCMOnlyGranularity ));
	
	return LcRet ;
}





//**************************************************************************
//
// WORD CLXProtocol::IDiag_DspSetTimerInterrupt(WORD PmSamples)
//
// Input Parameters:
// *****************
// PmSamples = nbr d'ech pour lequel une IRQ est leve. 
//
// Return value:
// *************
//
// SUCCESS if no error, error code otherwise
//
//**************************************************************************
WORD CLXProtocol::IDiag_DspSetTimerInterrupt(WORD PmSamples)
{
	WORD	LcRet = SUCCESS;
    KIRQL LcOldIrql;
    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	do{
		if(PmSamples == 0)				// switch off
		{
			if(m_TimerIT_Refs != 1)
			{
				if(m_TimerIT_Refs > 0) m_TimerIT_Refs --;
				LcRet = SUCCESS;
				break;
			}
			m_TimerIT_Refs = 0;
		}
		else							// switch on
		{
			if(m_TimerIT_Refs++ > 0)
			{
				LcRet = SUCCESS;
				break;
			}
			m_TimerIT = 0;
		}

        if( (PmSamples == 0) || (m_PCMOnlyGranularity == 0) ) {
            m_K = 0;
        } else {
            m_K = PmSamples / m_PCMOnlyGranularity;
        }

		// Fill CMD_RMH_INFO structure
		InitMessage( CMD_03_SET_TIMER_IRQ );

		m_Rmh.Cmd[0] |=  ( m_K & MASK_TIMER_K);

		// Send RMH
		LcRet = SendMessage();

		DOUT (DBG_PRINT, ("IDiag_DspSetTimerInterrupt() m_K = %d, Return = %d\n", m_K, LcRet));

	}while(FALSE);

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	DOUT (DBG_PROT_VIO, ("IDiag_DspSetTimerInterrupt() m_TimerIT_Refs = %d\n", m_TimerIT_Refs));

	return LcRet ;
} // IDiag_DspSetTimerInterrupt

//////////////////////////////////////////////////////////////////////
// End of DIAG and INIT
//////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
// FLOW
//////////////////////////////////////////////////////////////////////

#define PIPE_INFO_TO_CMD(capture, audio) ((DWORD)((DWORD)(audio) | ((capture) ? ID_IS_CAPTURE : 0L)) << ID_OFFSET)

//**************************************************************************
//
// Fill the CMD_RMH_INFO: Pipe definition command
// and send this RMH.
//
//**************************************************************************
WORD CLXProtocol::IFlow_PipeDef(	IN BYTE PmAttributes,
                                    IN WORD PmFirstAudio,
                                    IN BYTE PmNbMaxFlux,
									IN WORD PmNbAudioIO,
									IN DWORD PmChannelMask,
									IN WORD PmManagement,
									IN PFRAME_INFO PmFrameInfo,
                                    IN BOOLEAN PmIsAPINP,
									OUT PHANDLE_PIPE PmPHdlPipe,
									OUT PWORD PmNbAllocatedStreams )
{
	WORD			LcRet = SUCCESS;
    PSTREAM_INFO	LcCurrentStream  = NULL;
    BOOLEAN			capture;

    *PmPHdlPipe = NULL;

	DOUT (DBG_PROT_FLOW, ("CLXProtocol::IFlow_PipeDef() \n"));

    capture = ((PmAttributes & PIPE_PLAYREC_MASK) == OPER_REC);

	if ( PmNbMaxFlux != 1 )
	{
				DOUT (DBG_ERROR, ("CLXProtocol::IFlow_PipeDef() ED_ALLOCATE_STREAM_IMPOSSIBLE (GSM) @line %d \n",__LINE__));
				return ED_ALLOCATE_STREAM_IMPOSSIBLE;
    };

    // Select the relevant pool of free STREAM_INFO structs
    // *****************************************************
	if ( capture )		LcCurrentStream = m_StreamInFreeList;
    else		        LcCurrentStream = m_StreamOutFreeList;

    if(LcCurrentStream != NULL)  
    {
      LcCurrentStream->siAdrPreviousStream = NULL ;
    }
    else
    {
      DOUT (DBG_ERROR, ("CLXProtocol::IFlow_PipeDef() LcCurrentStream == NULL\n"));
      return ED_ALLOCATE_STREAM_IMPOSSIBLE;
    }

	// initialize the entry
	// *********************
	LcCurrentStream->siAdrBuffer = NULL;
	LcCurrentStream->siAdrNextBufferToGive = NULL;
	LcCurrentStream->siReleasedBuffers  = 0;
	LcCurrentStream->siFirstAudioNum	= 0;			 
	LcCurrentStream->siFirstAudioValid	= FALSE;		 
	LcCurrentStream->siAPINP            = PmIsAPINP;
    LcCurrentStream->siPCMSampleSize    = 2 * PmNbAudioIO; // init with something different from 0

    KIRQL LcOldIrql;
    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill CMD_RMH_INFO structure
    InitMessage( CMD_06_ALLOCATE_PIPE );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmFirstAudio);

	m_Rmh.Cmd[0] |=  PmNbAudioIO;

	LcRet = SendMessage(  ) ;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if ( LcRet == SUCCESS )
    {
		*PmNbAllocatedStreams = 1 ;
		*PmPHdlPipe = (HANDLE_PIPE) LcCurrentStream;

		LcCurrentStream->siFirstAudioValid	= TRUE;
		LcCurrentStream->siFirstAudioNum	= PmFirstAudio;

		if ( capture )  m_StreamInFreeList  = LcCurrentStream->siAdrNextStream;
		else			m_StreamOutFreeList = LcCurrentStream->siAdrNextStream;
    }

    return ( LcRet );
}		// DefPipe

// **************************************************************************
//
// WORD IFlow_PipeRelease( IN PTARGET_INFO, IN BOOLEAN )
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTgPipe:      Identify the target
//  BOOLEAN      PmReleaseAll:  Whether to forward the request to the board
//
// Output Paramaters:
// ******************
//
// Return value:
// *************
//
// O if no error, error otherwise
//
//**************************************************************************
//
// Fill the CMD_RMH_INFO: Pipe free command
// and send this RMH.
//
//**************************************************************************
WORD CLXProtocol::IFlow_PipeRelease(	IN  PTARGET_INFO PmPtrTgPipe,
                                        IN  BOOLEAN      PmCommit )
{
    WORD           LcRet    = SUCCESS   ;
    PSTREAM_INFO   LcCurrentStream  = (PSTREAM_INFO)PmPtrTgPipe->tgVioHandle ;
	
	BOOLEAN		capture = (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY) ? FALSE : TRUE;


    // Free all the stream info structures by appending them to the
    // the relevant list of free structs
    // *************************************************************

	DOUT (DBG_PROT_FLOW, ("CLXProtocol::IFlow_PipeRelease()\n"));


    if ( LcCurrentStream != NULL )
    {
        if (!capture)
        {
            LcCurrentStream->siAdrNextStream = m_StreamOutFreeList;
            m_StreamOutFreeList = (PSTREAM_INFO) PmPtrTgPipe->tgVioHandle;
        }
        else
        {
            LcCurrentStream->siAdrNextStream = m_StreamInFreeList;
            m_StreamInFreeList = (PSTREAM_INFO) PmPtrTgPipe->tgVioHandle;
        }
    }

    if ( PmCommit )
    {
		DOUT (DBG_PROT_FLOW, ("CLXProtocol::IFlow_PipeRelease() -> DSP \n"));

		KIRQL LcOldIrql;
	    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);
		
		// Fill CMD_RMH_INFO structure
		InitMessage( CMD_07_RELEASE_PIPE );
		
        m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

        // Send RMH
		LcRet = SendMessage(  ) ;

		/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);
    }

    return ( LcRet );
} // IFlow_PipeRelease

//**************************************************************************
//
// WORD IFlow_PipeSampleCount( IN PTARGET_INFO, OUT PPCX_TIME PmSampleCount)
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTgPipe:Identify the target
//
// Output Paramaters:
// ******************
//
//  PPCX_TIME PmSampleCount: The sample count for the corresponding pipe
//
// Return value:
// *************
//
// O if no error, error otherwise
//
//**************************************************************************
//
//
//**************************************************************************
WORD CLXProtocol::IFlow_PipeSampleCount(
    IN  PTARGET_INFO    PmPtrTgPipe     ,
    OUT PPCX_TIME       PmSampleCount   )
{
    WORD                LcRet = SUCCESS   ;

	DOUT (DBG_PROT_FLOW, ("CLXProtocol::IFlow_PipeSampleCount()\n"));

	KIRQL LcOldIrql;
    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill CMD_RMH_INFO structure
    InitMessage( CMD_0A_GET_PIPE_SPL_COUNT );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(PmPtrTgPipe->tgCaracPipeVoie & OPER_REC, PmPtrTgPipe->tgPipe);

	m_Rmh.LgStat = 2; // need all words here !

    LcRet = SendMessage(  ) ;

    // Process the reply
    // 
    *PmSampleCount  =	( (DWORDLONG)(m_Rmh.Stat[0] & MASK_SPL_COUNT_HI ) << 32 )	// hi part
						+	m_Rmh.Stat[1] ;									// lo part
				
					// + (m_Rmh.Stat[2] % m_PCMOnlyGranularity) ;			// current XES ptr position

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}

//////////////////////////////////////////////////////////////////////
// End of FLOW
//////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
// Private Methods
//////////////////////////////////////////////////////////////////////

void CLXProtocol::InitMessage( cmd_mb_opcodes op_code )
{
									// Fill CMD_RMH_INFO structure
									// ---------------------------
                                    m_Rmh.Index   = (BYTE)op_code ; // for dbg purpose
									m_Rmh.LgStat  = m_DspCommands[op_code].dcStatusLength;
									m_Rmh.DspStat = m_DspCommands[op_code].dcStatusType;
									m_Rmh.LgCmd   = m_DspCommands[op_code].dcCmdLength;
									m_Rmh.Cmd[0]  = m_DspCommands[op_code].dcCodeOp;
                                    BZERO2( &(m_Rmh.Cmd[1]), DWORD, (MB_REG_CMD_MAX-1)); 
};


WORD CLXProtocol::SendMessage(  ) 
	{
		WORD    LcRet   = SUCCESS   ;
        BOOLEAN LcNoCheck = FALSE;

		if( m_FatalDspError )
		{
			EXIT_PCX_ERROR( m_FatalDspError );
		}

#ifndef SIMLXES
        LcRet = m_PIO->PIOSendMessage( &m_Rmh );
#else
		extern WORD PIOSendMessage( CMD_RMH_INFO* Rmh ); 
		LcRet = PIOSendMessage( &m_Rmh );
#endif

#ifdef DBG_VIEWER
        if((LcRet != SUCCESS) | (g_ulDebugOut & DBG_PIO_DSP))
        {
            int i;
	        DOUT (DBG_ERROR, ("PIOSendMessage %s = %x\n", m_DspCommands[m_Rmh.Index].dcOpName, LcRet));
	        DOUT (DBG_ERROR, ("  .Index = %x\n", m_Rmh.Index));
	        DOUT (DBG_ERROR, ("  .LgCmd = %x\n", m_Rmh.LgCmd));
	        DOUT (DBG_ERROR, ("  .LgStat = %x\n", m_Rmh.LgStat));
            for(i=0; i<m_Rmh.LgCmd; i++)
	        DOUT (DBG_ERROR, ("  .Cmd[%x] = %x\n", i, m_Rmh.Cmd[i] ));
            for(i=0; i<m_Rmh.LgStat; i++)
	        DOUT (DBG_ERROR, ("  .Stat[%x] = %x\n", i, m_Rmh.Stat[i] ));
	        DOUT (DBG_ERROR, ("  .DspStat = %x\n", m_Rmh.DspStat));
        }
#endif

		if (   ( LcRet == ED_DSP_TIMED_OUT )
			|| ( LcRet == ED_DSP_CRASHED ) )
		{
#ifndef SIMLXES
			APHKeepDspError( m_pDSPInfo, LcRet );
#endif
			m_FatalDspError = LcRet;
		}

		return( LcRet );
} // SendMessage

// **************************************************************************
//
// STATIC WORD TogglePipeState( IN BYTE, IN BYTE, IN BYTE, IN PTARGET_INFO
//                              IN PTIME_CODE_START_INFO )
//
// Input Parameters:
// *****************
//
//  BYTE                  PmState:          Requested state (START_STATE/PAUSE_STATE)
//  BYTE                  PmCondition:      Toggle condition (START_PAUSE_*)
//  BYTE                  PmNbPipe:         Number of pipes
//  PTARGET_INFO          PmTgPipe:         Identify the target
//  PTIME_CODE_START_INFO PmPTimeCodeStart: Parameters for start on time-code
//                                          (is NULL if PmCondition is different
//                                           from START_PAUSE_ON_TIME_CODE)
//
// Return value:
// *************
//
// SUCCESS if no error, error otherwise
//
// **************************************************************************
//
// Toggle start/pause state synchronously for a list of pipes. The
// final status of these pipes will be PmState, unless PmCondition
// is different from START_PAUSE_IMMEDIATE. In this case, the toggle condition is external
// and the final status of the pipes is actually unknown.
//
// Pipes that are already in requested state are not affected
//
// Before switch to start, send a prepare request to the DSP
//
// FS - 19/02/1997
//  Change LcPipeMask and LcWaitingPipeMask to a bi-dimensional array
// to allow 32 in-pipes + 32 out-pipes to start synchronously
//  Change the decoding of PipeIndex (the mask cannot be a reference
// but only the corresponding target)
//
// Notice: PmCondition = START_PAUSE_ON_TIME_CODE and PmState = PAUSE_STATE
//         is not supported.
// **************************************************************************
//static int test = 0;
WORD CLXProtocol::vio_TogglePipeState(
    IN BYTE         PmState,
    IN BYTE         PmCondition,
    IN BYTE         PmNbPipe,
    IN PTARGET_INFO PmPtrTgPipe,
    IN PTIME_CODE_START_INFO PmPTimeCodeStart)
{
	WORD			i;
    WORD			LcRet = SUCCESS ;
	KIRQL			LcOldIrql;
    DWORDLONG		LcToggle_In_PipeTargetMask = 0;     // 64 bits target mask
    DWORDLONG		LcToggle_Out_PipeTargetMask = 0;    // 64 bits target mask
    DWORDLONG		LcWaiting_In_PipeTargetMask = 0;    // 64 bits target mask
    DWORDLONG		LcWaiting_Out_PipeTargetMask = 0;   // 64 bits target mask
    DWORDLONG		LcToggleError_In_Mask = 0;          // 64 bits target mask
    DWORDLONG		LcToggleError_Out_Mask = 0;         // 64 bits target mask
	BOOLEAN			LcManyPipes = TRUE;
	BOOLEAN			capture = FALSE;
	BOOLEAN			toggle_error = FALSE;
	PTARGET_INFO	LcTgPipe = PmPtrTgPipe;
    TARGET_INFO     LcWaitPipe;

	if (PmCondition == START_PAUSE_ON_TIME_CODE)	return ED_UNAVAILABLE_FEATURE;
	
	if (PmNbPipe == 1)	LcManyPipes = FALSE;

	for ( i = 0 ; i <PmNbPipe ; i++, LcTgPipe++)
	{
		ASSERT(LcTgPipe);

		if (LcTgPipe->tgCaracPipeVoie == OPER_PLAY)
		{
			LcToggle_Out_PipeTargetMask |= UTIMask64bit(LcTgPipe->tgPipe);
		}
		else
		{
			LcToggle_In_PipeTargetMask  |= UTIMask64bit(LcTgPipe->tgPipe);
			capture = TRUE;	// if only one pipe, it is a capture pipe
		}
	}

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    if (PmState == PAUSE_STATE)
	{
		LcToggle_Out_PipeTargetMask &= m_Pipe_Out_StartedMask;
		LcToggle_In_PipeTargetMask  &= m_Pipe_In_StartedMask;
	}
	else
	{
		LcToggle_Out_PipeTargetMask &= ~m_Pipe_Out_StartedMask;
		LcToggle_In_PipeTargetMask  &= ~m_Pipe_In_StartedMask;
	}

	if ((LcToggle_Out_PipeTargetMask | LcToggle_In_PipeTargetMask) == 0)
    {
	    /*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);
        return SUCCESS; // nothing to do
    }

    // need to verify that old state has finished to toggle ?
    LcWaiting_Out_PipeTargetMask = LcToggle_Out_PipeTargetMask & m_Pipe_Out_WaitToggleMask;
    LcWaiting_In_PipeTargetMask  = LcToggle_In_PipeTargetMask  & m_Pipe_In_WaitToggleMask;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    i = 0;
    while (LcWaiting_Out_PipeTargetMask)
    {
        if( LcWaiting_Out_PipeTargetMask & ((DWORDLONG) 1 ) )
        {
            LcWaitPipe.tgCaracPipeVoie = OPER_PLAY;
            LcWaitPipe.tgPipe = i;

            LcRet = IEvent_WaitPipeState( &LcWaitPipe, (PmState == PAUSE_STATE) ? PSTATE_RUN : PSTATE_IDLE);
            if(LcRet != SUCCESS)
            {
                LcToggleError_Out_Mask |= UTIMask64bit(i);
                toggle_error = TRUE;
            }
        }
        i++;
        LcWaiting_Out_PipeTargetMask >>= 1;
    }
    i = 0;
    while (LcWaiting_In_PipeTargetMask)
    {
        if( LcWaiting_In_PipeTargetMask & ((DWORDLONG) 1 ) )
        {
            LcWaitPipe.tgCaracPipeVoie = OPER_REC;
            LcWaitPipe.tgPipe = i;

            LcRet = IEvent_WaitPipeState( &LcWaitPipe, (PmState == PAUSE_STATE) ? PSTATE_RUN : PSTATE_IDLE);
            if(LcRet != SUCCESS)
            {
                LcToggleError_In_Mask |= UTIMask64bit(i);
                toggle_error = TRUE;
            }
        }
        i++;
        LcWaiting_In_PipeTargetMask >>= 1;
    }

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    if (toggle_error == FALSE)
    {
	InitMessage(CMD_0B_TOGGLE_PIPE_STATE);

	if (LcManyPipes == FALSE) 
	{
        m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);
	}
	else
	{
		m_Rmh.Cmd[0] |= MASK_CMD_LISTE;

		m_Rmh.LgCmd = 5 ; // all words sent
		
		m_Rmh.Cmd[1] = (DWORD)(LcToggle_Out_PipeTargetMask >> 32);          // out.hi
		m_Rmh.Cmd[2] = (DWORD)(LcToggle_Out_PipeTargetMask & 0xFFFFFFFF);   // out.lo;
		m_Rmh.Cmd[3] = (DWORD)(LcToggle_In_PipeTargetMask >> 32);           // in.hi
		m_Rmh.Cmd[4] = (DWORD)(LcToggle_In_PipeTargetMask & 0xFFFFFFFF);    // in.lo;
	}

//        if((PmState == PAUSE_STATE) || ((test++) & 0x3) != 0x3)
//        {
    LcRet = SendMessage(  ) ;
//        }
//        else
//        {
//          DOUT(DBG_WARNING, ("******* NO CMD_0B_TOGGLE_PIPE_STATE(%d)%c\n", PmPtrTgPipe->tgPipe, capture ? 'R' : 'P'));
//        }

    }
    else    // there was a toggle_error
    {
        // TODO
        DbgPrint("LXESWDM : START PLAY PIPE ERROR START=0x%I64x  WAIT=0x%I64x TOGGLE=0x%I64x\n", m_Pipe_Out_StartedMask, m_Pipe_Out_WaitToggleMask, LcToggle_Out_PipeTargetMask);
        DbgPrint("LXESWDM : START REC  PIPE ERROR START=0x%I64x  WAIT=0x%I64x TOGGLE=0x%I64x\n", m_Pipe_In_StartedMask, m_Pipe_In_WaitToggleMask, LcToggle_In_PipeTargetMask);

        LcRet = SUCCESS;

        // one of the last toggles did not work ! so don't toggle again here
        //
	    if (LcManyPipes == FALSE) 
        {
            DOUT(DBG_WARNING, ("### TOGGLE ERROR(pipe %d) -> %s\n", PmPtrTgPipe->tgPipe, PmState == PAUSE_STATE ? "PAUSE" : "RUN"));
            DOUT(DBG_WARNING, ("### donot CMD_0B_TOGGLE_PIPE_STATE(%d)%c\n", PmPtrTgPipe->tgPipe,
                capture ? 'R' : 'P'));
            // one pipe : do nothing
        }
        else
	    {
            DWORDLONG LcPartialToggle_In_Mask  = LcToggle_In_PipeTargetMask  & ~LcToggleError_In_Mask;
            DWORDLONG LcPartialToggle_Out_Mask = LcToggle_Out_PipeTargetMask & ~LcToggleError_Out_Mask;

            if((LcPartialToggle_Out_Mask != 0) || (LcPartialToggle_In_Mask !=0))
            {
    	        InitMessage(CMD_0B_TOGGLE_PIPE_STATE);

		        m_Rmh.Cmd[0] |= MASK_CMD_LISTE;

		        m_Rmh.LgCmd = 5 ; // all words sent

		        m_Rmh.Cmd[1] = (DWORD)(LcPartialToggle_Out_Mask >> 32);         // out.hi
		        m_Rmh.Cmd[2] = (DWORD)(LcPartialToggle_Out_Mask & 0xFFFFFFFF);  // out.lo;
		        m_Rmh.Cmd[3] = (DWORD)(LcPartialToggle_In_Mask >> 32);          // in.hi
		        m_Rmh.Cmd[4] = (DWORD)(LcPartialToggle_In_Mask & 0xFFFFFFFF);   // in.lo;

                LcRet = SendMessage(  ) ;
            }
	    }
    }

    // need wait for run state before pipepause !
    // need wait for pause state before pipestop !
    m_Pipe_Out_WaitToggleMask |= LcToggle_Out_PipeTargetMask;
    m_Pipe_In_WaitToggleMask  |= LcToggle_In_PipeTargetMask;

    if( LcRet == SUCCESS )  // update states
    {
	    if (PmState == START_STATE)
	    {
		    m_Pipe_Out_StartedMask	|= LcToggle_Out_PipeTargetMask ;
		    m_Pipe_In_StartedMask	|= LcToggle_In_PipeTargetMask;
	    }
	    else
	    {
		    m_Pipe_Out_StartedMask	&= ~LcToggle_Out_PipeTargetMask;
		    m_Pipe_In_StartedMask	&= ~LcToggle_In_PipeTargetMask;
	    }
    }

    /*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}

// **************************************************************************
//
// WORD CLXProtocol::IClock_BoardGetClockFrequency( ... )
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmPtrTarget: Pointer onto the target
//
// Output Parameters:
// *****************
//
//  PDWORD PmFrequencyPtr:    Storage for the result frequency
//  PDWORD PmPFirstChange:    Storage: tell whether it's the first time the DSP
//                            see a frequency change.
//
// Return value:
// *************
//
// 0 if no error, error otherwise
//
// **************************************************************************
//
// Tell the board that its audio input has been modified and that the DSP
// need to resynchronize its FIFO
//
// **************************************************************************
WORD CLXProtocol::IClock_BoardGetClockFrequency(	OUT PDWORD          PmFrequencyPtr,
													OUT PBOOLEAN        PmPFirstChange,
													OUT PDWORD			PmPRawData	/* = NULL*/)
{
    WORD	LcRet = SUCCESS ;
    DWORD	LcFrequency = 0 ;
	KIRQL	LcOldIrql;
	WORD    LcFreqRaw	= 0;
	WORD    LcFreq		= 0;

    *PmFrequencyPtr = 0;
    *PmPFirstChange = FALSE;

    // If a Dsp on a board is not loaded or crashed
    // this is useless to ask it resynchronizing
    // ---------------------------------------------
    if ( !IDiag_IsDspRunning( ) )
    {
        return SUCCESS ;
    }

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill CMD_RMH_INFO structure
    InitMessage( CMD_01_GET_SYS_CFG );

    LcRet = SendMessage(  ) ;

    if ( LcRet == SUCCESS )
    {
        LcFreqRaw = (WORD)( m_Rmh.Stat[0] >> FREQ_FIELD_OFFSET );
        LcFreq =	LcFreqRaw & XES_FREQ_COUNT8_MASK;

		if		((LcFreq <	XES_FREQ_COUNT8_48_MAX) || (LcFreq > XES_FREQ_COUNT8_44_MIN)) 
													LcFrequency = 0;		// unknown
		else if ( LcFreq >= XES_FREQ_COUNT8_44_MAX) LcFrequency = 44100;	// 44.1
		else										LcFrequency = 48000;	// 48.0 
    }

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	if ( PmPRawData ) *PmPRawData = LcFreqRaw;	// keep PM/M, WCK info. 

    // multiply with frequency ratio
    DWORD LcFrequencyRatio = m_PIO->PIOGetFrequencyRatio();
    *PmFrequencyPtr = LcFrequency * LcFrequencyRatio;

    DOUT ( DBG_PRINT, ("IClock_BoardGetClockFrequency = %d * %d = %d Hz.\n", LcFrequency, LcFrequencyRatio, *PmFrequencyPtr ));

	return LcRet;
} // IClock_BoardGetClockFrequency

//**************************************************************************
//
// WORD CLXProtocol::I_StopPipe( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmTgPipe:       Identify the target
//
//Output Paramaters:
//******************
//
//Return value:
//*************
//
// O if no error, error otherwise
//
//**************************************************************************
//
//
//**************************************************************************
WORD CLXProtocol::IFlow_PipeStop( IN PTARGET_INFO PmPtrTgPipe )
{
	WORD			LcRet = SUCCESS;
    PSTREAM_INFO    LcPtStream;
    BOOLEAN			capture = (PmPtrTgPipe->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;
    KIRQL LcOldIrql;

    // be sure that pipes have toggled !
    //
	IEvent_WaitEndOfCoding( PmPtrTgPipe );

    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_09_STOP_PIPE );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

    LcRet = SendMessage(  ) ;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    // Reset all streams
    // ------------------
    LcPtStream = (PSTREAM_INFO) PmPtrTgPipe->tgVioHandle;

    // Purge buffers if any
    // ---------------------
    if (capture)     vio_StreamPurgeRecBuffers( LcPtStream );
    else			 vio_StreamPurgeBuffers( LcPtStream );

    LcPtStream->siReleasedBuffers = 0;

    return ( LcRet );
}

//**************************************************************************
//
// WORD CLXProtocol::IFlow_PipeState( IN PTARGET_INFO, OUT PBYTE PmState)
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTgPipe:  Identify the target pipe
//
// Output Parameters:
// ******************
//
//  PBYTE PmState:         The current state of the pipe : 1 start, 0 pause
//
// Return value:
// *************
//
// SUCCESS if no error, error code otherwise
//
//**************************************************************************
WORD CLXProtocol::IFlow_PipeState(
    IN  PTARGET_INFO PmPtrTgPipe,
    OUT PBYTE        PmState )
{
    WORD            LcRet = SUCCESS  ;
    BOOLEAN         capture = (PmPtrTgPipe->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;
    KIRQL LcOldIrql;
    
    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_0A_GET_PIPE_SPL_COUNT );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

    LcRet = SendMessage(  ) ;

    if ( LcRet == SUCCESS )
    {
        *PmState = (BYTE)( ( m_Rmh.Stat[0] >> PSTATE_OFFSET)  & 0x0F ); // PSTATE_IDLE, PSTATE_RUN, PSTATE_PURGE, PSTATE_ACQUIRE or PSTATE_CLOSING
    }
	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}

//**************************************************************************
//
// WORD CLXProtocol::IFlow_StreamStart( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmTgPipe:       Identify the target
//
//Output Paramaters:
//******************
//
//Return value:
//*************
//
// O if no error, error otherwise
//
//**************************************************************************
//
//
//**************************************************************************
WORD CLXProtocol::IFlow_StreamStart( IN PTARGET_INFO PmPtrTgPipe )
{
	WORD        LcRet = SUCCESS;
	DWORDLONG	LcMask  = UTIMask64bit( PmPtrTgPipe->tgPipe );
	BOOLEAN		capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;
	KIRQL       LcOldIrql;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);
	
	// init 
	InitMessage( CMD_13_SET_STREAM_STATE );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);
	
	// param
	m_Rmh.Cmd[0] |= SSTATE_RUN;
	
	LcRet = SendMessage(  ) ;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}


//**************************************************************************
//
// WORD CLXProtocol::IFlow_StreamPause( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmTgPipe:       Identify the target
//
//Return value:
//*************
//
// O if no error, error otherwise
//
//**************************************************************************
WORD CLXProtocol::IFlow_StreamPause( IN PTARGET_INFO PmPtrTgPipe, BOOLEAN PmDrain /*= FALSE */)
{
	BOOLEAN		capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;
	KIRQL       LcOldIrql;
	WORD		LcRet;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	// init 
	InitMessage( CMD_13_SET_STREAM_STATE );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

    // param
	m_Rmh.Cmd[0] |= SSTATE_PAUSE;
	
	LcRet = SendMessage(  ) ;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	return LcRet;	
}


//**************************************************************************
//
// WORD CLXProtocol::IFlow_StreamStop( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmTgPipe:       Identify the target
//
//Return value:
//*************
//
// O if no error, error otherwise
//
//**************************************************************************
WORD CLXProtocol::IFlow_StreamStop( IN PTARGET_INFO PmPtrTgPipe )
{
    WORD        LcRet       = SUCCESS ;
	BOOLEAN		capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;
	KIRQL       LcOldIrql;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	// init 
	InitMessage( CMD_13_SET_STREAM_STATE );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

	// param
	m_Rmh.Cmd[0] |= SSTATE_STOP;
	
	LcRet = SendMessage(  ) ;

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    // Init the stream
    // ---------------

	if (capture)
    {
		vio_StreamPurgeRecBuffers( (PSTREAM_INFO)(PmPtrTgPipe->tgVioHandle));
	}
	else
	{
		vio_StreamPurgeBuffers( (PSTREAM_INFO)(PmPtrTgPipe->tgVioHandle));
	}

    return ( LcRet );
}


//**************************************************************************
//
// WORD  CLXProtocol::IFlow_StreamInSetFormat( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmPtrTgPipe:       Identify the target
//
//**************************************************************************
//
// Set the format of a stream
//
// FS - 05/02/97
// WARNING: As of today, padding bit is not yet managed,
//              neither the use of a new coder
//
// FH - 19/03/97 BUG correction: the SetOutStreamFormat opcode is different
//                               to the SetInStreamFormat opcode !!
//
//**************************************************************************
WORD  CLXProtocol::IFlow_StreamInSetFormat(
    IN PTARGET_INFO PmPtrTgPipe ,
    IN DWORD        PmFmtParams,
    IN DWORD        PmHeaderHiLo,
    IN DWORD        PmChannelMask,
	IN DWORD        PmSamplesPerChannel)
{
   WORD        LcRet = SUCCESS;
   WORD        LcOffset = 0 ;
   KIRQL       LcOldIrql;
   WORD		   LcNbChannels;


   BOOLEAN LcHasHeader  = (PmFmtParams & (1<<BIT_FMP_HEADER) )  ? TRUE : FALSE;
   BOOLEAN LcHasSuffix = (PmFmtParams & (1<<BIT_FMP_SD)     )  ? TRUE : FALSE;

   // on redcode ;)..il faudra changer ca au niveau de l'api...
   BOOLEAN LcIs16bits = (PmHeaderHiLo & HEADER_FMT_16BITS)  ? TRUE : FALSE;
   BOOLEAN LcIsIntel  = (PmHeaderHiLo & HEADER_FMT_INTEL)  ? TRUE : FALSE;

    if ( LcHasHeader || LcHasSuffix )
	{
		EXIT_PCX_ERROR( ED_FORMAT_NOT_SUPPORTED );
    }

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill CMD_RMH_INFO structure
    InitMessage( CMD_0C_DEF_STREAM );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(TRUE, PmPtrTgPipe->tgPipe);

	if (LcIs16bits)	
		m_Rmh.Cmd[0] |=	(STREAM_FMT_16b << STREAM_FMT_OFFSET);

	if (LcIsIntel)	
		m_Rmh.Cmd[0] |=	(STREAM_FMT_intel << STREAM_FMT_OFFSET);

	if (PmSamplesPerChannel)	
		m_Rmh.Cmd[0] |=	MASK_STREAM_IS_ASIO;
	
	LcNbChannels = (WORD)( PmFmtParams & 0x1FF );

	// give nb of ch -1 
	m_Rmh.Cmd[0] |=	(LcNbChannels -1 ) ;

    LcRet = SendMessage(  ) ;
	 
	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if( LcRet == SUCCESS )
    {
        DWORD LcSampleSize = LcIs16bits ? 2 : 3;
        PSTREAM_INFO	pStream;

        LcSampleSize *= LcNbChannels ;

        pStream = (PSTREAM_INFO)(PmPtrTgPipe->tgVioHandle);
		
		ASSERT(MAX_STREAM_BUFFER<=SI_MAX_STREAM_BUFFER); ASSERT(pStream);
	
		pStream->siPCMSampleSize = LcSampleSize;
    }

    return ( LcRet );
}

//**************************************************************************
//
// WORD  CLXProtocol::IFlow_StreamOutSetFormat( IN PTARGET_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO PmPtrTgPipe:       Identify the target
//
//**************************************************************************
//
// Set the format of a stream
//
//**************************************************************************
	WORD  CLXProtocol::IFlow_StreamOutSetFormat(
    IN PTARGET_INFO PmPtrTgPipe ,
    IN DWORD        PmFmtParams,
    IN DWORD        PmHeaderHiLo,
    IN DWORD        PmChannelMask,
	IN DWORD        PmSamplesPerChannel
	)
{
   WORD     LcRet = SUCCESS;
   WORD     LcOffset = 0 ;
   KIRQL    LcOldIrql;
   WORD		LcNbChannels;

   BOOLEAN LcHasHeader  = (PmFmtParams & (1<<BIT_FMP_HEADER) )  ? TRUE : FALSE;
   BOOLEAN LcHasSuffix = (PmFmtParams & (1<<BIT_FMP_SD)     )  ? TRUE : FALSE;
   //BOOLEAN LcIsMultich = (PmFmtParams & (1<<BIT_FMP_MULTICHANNEL))  ? TRUE : FALSE;

   // on redcode ;)..il faudra changer ca au niveau de l'api...
   BOOLEAN LcIs16bits = (PmHeaderHiLo & HEADER_FMT_16BITS)  ? TRUE : FALSE;
   BOOLEAN LcIsIntel  = (PmHeaderHiLo & HEADER_FMT_INTEL)  ? TRUE : FALSE;

    if ( LcHasHeader || LcHasSuffix || (PmChannelMask > 0x3))
	{
		// no mpeg, suffix, or multichannel for now
		EXIT_PCX_ERROR( ED_FORMAT_NOT_SUPPORTED );
    }

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    // Fill CMD_RMH_INFO structure
    InitMessage( CMD_0C_DEF_STREAM );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(FALSE, PmPtrTgPipe->tgPipe);

	if (LcIs16bits)	
		m_Rmh.Cmd[0] |=	(STREAM_FMT_16b << STREAM_FMT_OFFSET);

	if (LcIsIntel)	
		m_Rmh.Cmd[0] |=	(STREAM_FMT_intel << STREAM_FMT_OFFSET);
	
	if (PmSamplesPerChannel)	
		m_Rmh.Cmd[0] |=	MASK_STREAM_IS_ASIO;

	// give nb of ch.
	//
	LcNbChannels = (WORD)( PmFmtParams & 0x1FF );

	m_Rmh.Cmd[0] |=	( LcNbChannels - 1 );

	if (LcNbChannels == 1)
	{
	// support mono --> stereo
	// si le flux est mono, on va gerer le mapping en fonction de PmChannelMask
	// par defaut, il faut se comporter comme sur PCX, c-a-d copier le flux mono vers 
	// les deux audios du pipe stereo.
	//
		switch (PmChannelMask)
		{
			case SPEAKER_FRONT_LEFT:	break;	// default for embedded, no mapping
			case SPEAKER_FRONT_RIGHT:	
				m_Rmh.Cmd[0] |= MASK_STREAM_HAS_MAPPING;
				m_Rmh.Cmd[1]  = 1;				// one mapping descriptor [stream_ch | ES_ch]
				m_Rmh.Cmd[1] |= 0x0001 << 8;	// stream0 -> pipe1
				m_Rmh.LgCmd = 2;
				break;
			default: // 0 or center, or stereo
				m_Rmh.Cmd[0] |= MASK_STREAM_HAS_MAPPING;
				m_Rmh.Cmd[1]  = 2;				// two mappings descriptors [stream_ch | ES_ch]
				m_Rmh.Cmd[1] |= 0x0100 << 8;	// stream0 -> pipe1 and stream0 -> pipe0
				m_Rmh.LgCmd = 2;
				break;
		}
	}

    LcRet = SendMessage(  ) ;
	 
	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if( LcRet == SUCCESS )
    {
        DWORD LcSampleSize = LcIs16bits ? 2 : 3;
        PSTREAM_INFO	pStream;

        LcSampleSize *= LcNbChannels;

        pStream = (PSTREAM_INFO)(PmPtrTgPipe->tgVioHandle);
		
		ASSERT(MAX_STREAM_BUFFER<=SI_MAX_STREAM_BUFFER); ASSERT(pStream);
	
		pStream->siPCMSampleSize = LcSampleSize;
    }

    return ( LcRet );
} // IFlow_StreamOutSetFormat


/**
*	LEVEL Interface :	Set Mutes
*
*	Setting a Mute :
*			LcLevelAudioInfo.gaiHasMuteLevel = TRUE;
*			LcLevelAudioInfo.gaiMuteLevel = (WORD)( *PmPOnOff ? 1 : 0 );
*
*	@see Topology, PLEVEL_AUDIO_INFO
*	@return SUCCESS or error code.
*/
WORD	CLXProtocol::ILevel_AudioSetDigitalLevel(	IN PTARGET_INFO        PmPtrTgPipe,
													IN DWORDLONG           PmAudioMask,
													IN PLEVEL_AUDIO_INFO   PmPtrLevelInf )  		
{
	WORD        LcRet = SUCCESS;
	KIRQL       LcOldIrql;
	BOOLEAN		capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;
	DWORDLONG	LcMask  = PmAudioMask;

    if ( PmPtrLevelInf->gaiHasDigitalLevel
        +PmPtrLevelInf->gaiHasMonitorLevel
        +PmPtrLevelInf->gaiHasM1Level
        +PmPtrLevelInf->gaiHasM2Level)	 return ED_UNAVAILABLE_FEATURE;

    if ((PmPtrLevelInf->gaiHasMuteLevel) && (LcMask))
	{
		if (capture) 
		{
			if (PmPtrLevelInf->gaiMuteLevel) 
				m_Stream_In_MutedMask |= LcMask;
			else
				m_Stream_In_MutedMask &= ~LcMask;
		}
		else
		{
			if (PmPtrLevelInf->gaiMuteLevel) 
				m_Stream_Out_MutedMask |= LcMask;
			else
				m_Stream_Out_MutedMask &= ~LcMask;
		}
	
		/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);
		
		// Fill CMD_RMH_INFO structure
		InitMessage( CMD_0D_SET_MUTE );

		if (capture) 
		{
			m_Rmh.Cmd[1] = (DWORD)(m_Stream_In_MutedMask >> (DWORDLONG) 32);
			m_Rmh.Cmd[2] = (DWORD)(m_Stream_In_MutedMask &  (DWORDLONG) 0xFFFFFFFF);
		}
		else
		{
			m_Rmh.Cmd[1] = (DWORD)(m_Stream_Out_MutedMask >> (DWORDLONG) 32);
			m_Rmh.Cmd[2] = (DWORD)(m_Stream_Out_MutedMask &  (DWORDLONG) 0xFFFFFFFF);
		}

		LcRet = SendMessage(  )  ;

		/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);
	}

    return LcRet;
} // ILevel_AudioSetDigitalLevel


/**
*	LEVEL Interface :	Get VU and Peak Meters
*
*	@param	 PmIsInput	    
*	@param	 PmAudioMask	
*	@param	 PmPtrVuPicMeterInf pointer to the meter infos
*
*	@see API : GetAudioVuMeterSubFct
*	@return SUCCESS or error code.
*/  
WORD    CLXProtocol::ILevel_AudioGetVuPicMeter(	IN BOOLEAN      PmIsInput,
										        IN DWORDLONG            PmAudioMask,
										        IN WORD                 PmNbAudio,
										        OUT PVU_PIC_METER_INFO  PmPtrVuPicMeterInf )
{
    WORD    LcRet = SUCCESS;
	KIRQL   LcOldIrql;
    WORD    LcAudioID = 0;

	DWORD	s1,s2,s3,s4; /*four sample values*/

	VU_PIC_METER_INFO*	LcPtrVuPicMeterInf = PmPtrVuPicMeterInf;


    while( PmAudioMask )
    {
        if(PmAudioMask & (DWORDLONG)(0x0F))
        {
	        /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);
		        
	        // Fill CMD_RMH_INFO structure
	        InitMessage( CMD_12_GET_PEAK );

            m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(PmIsInput, LcAudioID);

	        LcRet = SendMessage(  )  ;

            if ( LcRet == SUCCESS )
            {
				s1 = m_Peak2Sample[  m_Rmh.Stat[0]       & 0x0F   ] ;
				s2 = m_Peak2Sample[ (m_Rmh.Stat[0] >> 4) & 0x0F   ] ;
				s3 = m_Peak2Sample[ (m_Rmh.Stat[0] >> 8) & 0x0F   ] ;
				s4 = m_Peak2Sample[ (m_Rmh.Stat[0] >>12) & 0x0F   ] ;
            }
			else
			{
				s1 = s2 = s3 = s4 = 0;
			}

	        /*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

			if(PmAudioMask & 0x01)
			{
				LcPtrVuPicMeterInf->vpmiVuMeter =  s1;
				LcPtrVuPicMeterInf->vpmiPicMeter = s1;
				LcPtrVuPicMeterInf->vpmiSaturation = (s1 == 0x7FFFFF) ? TRUE : FALSE;
				LcPtrVuPicMeterInf++;
			}

			if(PmAudioMask & 0x02)
			{
				LcPtrVuPicMeterInf->vpmiVuMeter =  s2;
				LcPtrVuPicMeterInf->vpmiPicMeter = s2;
				LcPtrVuPicMeterInf->vpmiSaturation = (s2 == 0x7FFFFF) ? TRUE : FALSE;
				LcPtrVuPicMeterInf++;
			}

			if(PmAudioMask & 0x04)
			{
				LcPtrVuPicMeterInf->vpmiVuMeter =  s3;
				LcPtrVuPicMeterInf->vpmiPicMeter = s3;
				LcPtrVuPicMeterInf->vpmiSaturation = (s3 == 0x7FFFFF) ? TRUE : FALSE;
				LcPtrVuPicMeterInf++;
			}

			if(PmAudioMask & 0x08)
			{
				LcPtrVuPicMeterInf->vpmiVuMeter =  s4;
				LcPtrVuPicMeterInf->vpmiPicMeter = s4;
				LcPtrVuPicMeterInf->vpmiSaturation = (s4 == 0x7FFFFF) ? TRUE : FALSE;
				LcPtrVuPicMeterInf++;
			}
        }

        PmAudioMask >>= 4;
        LcAudioID += 4;
    }
    return ( LcRet );

}; // ILevel_AudioGetVuPicMeter

/**
*	LEVEL Interface :	Get Audio Levels
*
*	@param	 PmIsInput	
*	@param	 PmAudioID	the number of the audio (0 to 63 on LX6464ES)
*	@param	 PmPtrLevelInf  OUT	pointer to the array of Info Structures
*
*	@see API : GetOutAudioLevelsSubFct
*	@return SUCCESS or error code.
*/  
WORD    CLXProtocol::ILevel_AudioGetLevels(	IN BOOLEAN      PmIsInput,
									        IN DWORDLONG            PmAudioMask,
									        IN WORD                 PmNbAudio,
											OUT PLEVEL_AUDIO_INFO  PmPtrLevelInf )
{
	DWORDLONG	LcMask;
	BOOLEAN		IsMuted;
    WORD    LcAudioID = 0;

    while( PmAudioMask )
    {
        if(PmAudioMask & 0x01)
        {
			LcMask = UTIMask64bit( LcAudioID );
			if (PmIsInput)
				IsMuted = ( LcMask & m_Stream_In_MutedMask ) ? 1 : 0;
			else
				IsMuted = ( LcMask & m_Stream_Out_MutedMask ) ? 1 : 0;

			PmPtrLevelInf->gaiMuteLevel = IsMuted;
        }
        PmAudioMask >>= 1;
        LcAudioID++;
    } // while 

    return SUCCESS;
}  // ILevel_AudioGetLevels



//**************************************************************************
//
// WORD  CLXProtocol::IFlow_StreamGetState( IN PTARGET_INFO,
//                          IN WORD,
//                          OUT PSTATE_STREAM_INFO )
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO            PmPtrTgPipe :    Identify the target
//  WORD                    PmNbStream :     Number of stream
//
//Output Parameters:
//*****************
//
//  PSTATE_STREAM_INFO      PmPtrStreamInf : Identify the state, and hour for each stream
//
// FH: creation
//
//**************************************************************************
// This function suppose:
//
//**************************************************************************
WORD    CLXProtocol::IFlow_StreamGetState(
    IN PTARGET_INFO         PmPtrTgPipe,
    IN WORD                 PmNbStream,
    OUT PSTATE_STREAM_INFO  PmPtrStreamInf )
{
	WORD        LcRet = SUCCESS;
	KIRQL       LcOldIrql;
	BOOLEAN		capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_0E_GET_STREAM_SPL_COUNT );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

    m_Rmh.LgStat = 1;

    LcRet = SendMessage( );

    PmPtrStreamInf->Start = (m_Rmh.Stat[0] & SF_START ) ? START_STATE : PAUSE_STATE;
	
	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    PmPtrStreamInf->Hour = 0; // start time not supported

    return ( LcRet );
}


// **************************************************************************
//
// WORD  CLXProtocol::IFlow_StreamSamplePosition( IN PTARGET_INFO,
//                             IN WORD,
//                             OUT PPCX_TIME )
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO            PmPtrTgPipe :    Identify the target
//  WORD                    PmNbStream :     Number of stream
//
// Output Parameters:
// *****************
//
//  PPCX_TIME               PmPtrStreamInf : Array of sample counts
//                                           for each stream
//
//**************************************************************************
//
// This function assumes that the PmPtrStreamInf array holds as many
// entries as specified by PmNbStream
//
//**************************************************************************
WORD    CLXProtocol::IFlow_StreamSamplePosition(
    IN  PTARGET_INFO    PmPtrTgPipe     ,
    IN  WORD            PmNbStream      ,
    OUT PPCX_TIME       PmPtrStreamInf  )
{
	WORD	LcRet = SUCCESS;
	KIRQL	LcOldIrql;
	BOOLEAN	capture	= (PmPtrTgPipe->tgCaracPipeVoie == OPER_PLAY ) ? FALSE : TRUE;
    PCX_TIME LcBytePos;

	ASSERT(PmNbStream==1);

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_0E_GET_STREAM_SPL_COUNT );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTgPipe->tgPipe);

    LcRet = SendMessage( );

	LcBytePos = ((PCX_TIME)(m_Rmh.Stat[0] & MASK_SPL_COUNT_HI) << 32) + m_Rmh.Stat[1];
	
	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if( *PmPtrStreamInf == 123 )
    {
        // ByteCount requested !
        *PmPtrStreamInf = LcBytePos;
    }
    else
    {
        // SampleCount requested
        PSTREAM_INFO	pStream;
        pStream = (PSTREAM_INFO)(PmPtrTgPipe->tgVioHandle);

        if( pStream->siPCMSampleSize == 0 ) EXIT_PCX_ERROR( ED_INVALID_PIPE );

        *PmPtrStreamInf = LcBytePos / pStream->siPCMSampleSize;
    }

    return ( LcRet );
} // IFlow_StreamSamplePosition


WORD CLXProtocol::IEvent_WaitPipeState( IN PTARGET_INFO PmTarget, IN BYTE PmState )
{
    WORD LcRet;
    BYTE LcCurrentState;

    // vider le IBuffer : max 2*PCMOnlyGranularity = 2*1024 at 44100 = < 50 ms : timeout 50 ms
    for( int i=0; i<50; i++ )
    {
        LcRet = IFlow_PipeState(PmTarget, &LcCurrentState);

        if( LcRet != SUCCESS )
            return LcRet;

        if( LcCurrentState == PmState )
            return SUCCESS;

        DOUT(DBG_WARNING, ("### IFlow_PipeState(%d)%c waitfor:%d current:%d\n", PmTarget->tgPipe,
            PmTarget->tgCaracPipeVoie == OPER_PLAY ? 'P' : 'R',
            PmState, LcCurrentState));

        PCX_WAIT_AT_LEAST_MILLI_SEC(1);
    }
    EXIT_PCX_ERROR( ED_BOARD_TIME_OUT );
}


/*	
*	Wait for the time of a pseudo-frame, bo be sure a setting has been taken into account
*	or the current sound transfer is done.
*/
VOID	CLXProtocol::IEvent_WaitEndOfCoding( IN PTARGET_INFO PmTarget )
{
    DWORDLONG LcPipeMask = UTIMask64bit( PmTarget->tgPipe );
    WORD LcRet = SUCCESS;
	KIRQL LcOldIrql;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    if( PmTarget->tgCaracPipeVoie == OPER_PLAY)
    {
        // pipe needs to be paused before
        ASSERT( (m_Pipe_Out_StartedMask & LcPipeMask) == 0 );

        if( (m_Pipe_Out_WaitToggleMask & LcPipeMask) == 0 )
            LcRet = WD_WCC; // nothing to wait
        else
            m_Pipe_Out_WaitToggleMask &= ~LcPipeMask;
    }
    else
    {
        // pipe needs to be paused before
        ASSERT( (m_Pipe_In_StartedMask & LcPipeMask) == 0 );

        if( (m_Pipe_In_WaitToggleMask & LcPipeMask) == 0 )
            LcRet = WD_WCC; // nothing to wait
        else
            m_Pipe_In_WaitToggleMask &= ~LcPipeMask;
    }

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if( LcRet != SUCCESS )  return;

    IEvent_WaitPipeState( PmTarget, PSTATE_IDLE );
}


WORD	CLXProtocol::IEvent_WaitEndOfStartPipe( IN PTARGET_INFO PmTarget )
{
    WORD LcRet = WD_WCC;
	KIRQL LcOldIrql;

    DWORDLONG LcPipeMask = UTIMask64bit( PmTarget->tgPipe );
    BOOLEAN is_play = (PmTarget->tgCaracPipeVoie == OPER_PLAY);

    /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    if (is_play)
    {
        // pipe needs to be started before
        ASSERT( (m_Pipe_Out_StartedMask & LcPipeMask) != 0 );

        if( (m_Pipe_Out_WaitToggleMask & LcPipeMask) == 0 )
            LcRet = SUCCESS; // nothing to wait
        else
            m_Pipe_Out_WaitToggleMask &= ~LcPipeMask;
    }
    else
    {
        // pipe needs to be paused before
        ASSERT( (m_Pipe_In_StartedMask & LcPipeMask) != 0 );

        if( (m_Pipe_In_WaitToggleMask & LcPipeMask) == 0 )
            LcRet = SUCCESS; // nothing to wait
        else
            m_Pipe_In_WaitToggleMask &= ~LcPipeMask;
    }

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    if (LcRet == SUCCESS)
        return SUCCESS;

    LcRet = IEvent_WaitPipeState( PmTarget, PSTATE_RUN );

    if (LcRet != SUCCESS)
    {
        // ERROR the pipe did not start
        //

        /*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

        if (is_play)
        {
            m_Pipe_Out_StartedMask &= ~LcPipeMask;
            // TODO !
            DbgPrint("LXESWDM : START PIPE(P %d) ERROR (mask=0x%I64x)\n", PmTarget->tgPipe, m_Pipe_Out_StartedMask);
        }
        else
        {
            m_Pipe_In_StartedMask &= ~LcPipeMask;
            // TODO !
            DbgPrint("LXESWDM : START PIPE(R %d) ERROR (mask=0x%I64x)\n", PmTarget->tgPipe, m_Pipe_In_StartedMask);
        }

	    /*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);
    }

    return LcRet;
}


//**************************************************************************
//
// WORD CLXProtocol::IEvent_AsynchronousEvents ( IN PTARGET_INFO  )
//
//Input Parameters:
//*****************
//      PTARGET_INFO *PmPtTarget : Pointer to a target which describes a pipe
//                              - tgCarte, tgDsp, TgPipe, tgVioHandlePipe
//
//Output Paramaters:
//******************
//
//     PBOOLEAN      PmPEndStartOnTimeCode : if TRUE, indicates that the start on
//                                           time-code conditions are filled.
//     PBOOLEAN      PmPFrequencyChange    : if TRUE, indicates that the frequency
//                                           of the board changed (UER mode).
//
//Return value:
//*************
//
// 0 if no error, error otherwise
//
//**************************************************************************
//
// Read the async. events that have occured on the target, and dispatch
// the errors in the driver structures. If any async. command is waiting
// for an error, then trigger it.
//
//**************************************************************************

WORD CLXProtocol::IEvent_AsynchronousEvents(  OUT PBOOLEAN         PmPEndStartOnTimeCode,
                                              OUT PBOOLEAN         PmPFrequencyChange,
                                              OUT PDWORDLONG       PmPNotifiedOutPipeMask,
                                              OUT PDWORDLONG       PmPNotifiedInPipeMask,
											  OUT PES_DRV_RESP	   PmES_Response)
{
    DWORD  LcIrqSrc = PCX_IRQ_NONE ;
    WORD   LcRet = SUCCESS ;
	KIRQL  LcOldIrql;

	// 
    // Default values.
    //
    *PmPEndStartOnTimeCode = FALSE;
    *PmPFrequencyChange    = FALSE;
    *PmPNotifiedOutPipeMask = 0;
    *PmPNotifiedInPipeMask  = 0;
	 PmES_Response->ulStatus = WD_NO_MORE_DATA;  

    // reading the PLX mailbox never crashes, unless PCI is dead....
    LcIrqSrc = InterlockedExchange(&m_SrcIT, PCX_IRQ_NONE);

    DOUT (DBG_PROT_EVENT, ("IEvent_TestITSrc() SrcIT! = %x\n", LcIrqSrc));

    // nothing to do ? PortCls may have called a Stream::Service routine.
    //
    if (LcIrqSrc == PCX_IRQ_NONE) return SUCCESS;
 
    // The frequency has changed on the board.
    //
    if ( LcIrqSrc & MASK_SYS_STATUS_FREQ )   *PmPFrequencyChange = TRUE;

    if ( LcIrqSrc & MASK_SYS_ASYNC_EVENTS )
    {
	//	BOOLEAN	LcHasErr		= (LcIrqSrc & MASK_SYS_STATUS_ERROR)? TRUE : FALSE ;
		BOOLEAN	LcHasUnderrun	= (LcIrqSrc & MASK_SYS_STATUS_URUN)	? TRUE : FALSE ;
		BOOLEAN	LcHasOverrun	= (LcIrqSrc & MASK_SYS_STATUS_ORUN )? TRUE : FALSE ;
		BOOLEAN	LcEBPending_OUT = (LcIrqSrc & MASK_SYS_STATUS_EOBO ) ? TRUE : FALSE ;
		BOOLEAN	LcEBPending_IN  = (LcIrqSrc & MASK_SYS_STATUS_EOBI ) ? TRUE : FALSE ;

		DWORD			error_code = SUCCESS;
		TARGET_INFO     LcTarget;

  		/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

		//	On peut optimiser pour ne pas lire les evenements vides
		//	les mots de rponse sont dans l'ordre suivant : 
		//	Stat[0]	mot de status gnral
		//	Stat[1]	fin de buffer OUT pF
		//	Stat[2]	fin de buffer OUT pf
		//	Stat[3]	fin de buffer IN pF
		//	Stat[4]	fin de buffer IN pf
		//	Stat[5]	underrun poid fort
		//	Stat[6]	underrun poid faible
		//	Stat[7]	overrun poid fort
		//	Stat[8]	overrun poid faible

		InitMessage(CMD_04_GET_EVENT);

		// m_Rmh.lgStatus = 0, on ne lit rien, mais on execute la commande
		// pour que le reset ait lieu

		if			(LcEBPending_IN)	m_Rmh.LgStat = 5;
		else 	if  (LcEBPending_OUT)	m_Rmh.LgStat = 3;
		else 	if  (LcHasOverrun)		m_Rmh.LgStat = 9;
		else 	if  (LcHasUnderrun)		m_Rmh.LgStat = 7;
	//	else 	if  (LcHasErr)			m_Rmh.LgStat = 2;

		LcRet = SendMessage() ;

        if ( LcRet != SUCCESS )
        {
		    /*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);
            return LcRet;
        }

		error_code			= m_Rmh.Stat[1];

		DWORDLONG OrunMask	= ( (DWORDLONG)m_Rmh.Stat[7] << 32) + m_Rmh.Stat[8];
		DWORDLONG UrunMask	= (	(DWORDLONG)m_Rmh.Stat[5] << 32) + m_Rmh.Stat[6];
		DWORDLONG PipeOutMask;
		BYTE	  pipe_idx;

        if(LcEBPending_IN)
        {
            *PmPNotifiedInPipeMask = (	(DWORDLONG)m_Rmh.Stat[3] << 32) + m_Rmh.Stat[4];
            DOUT(DBG_DMA, ("IEvent_AsynchronousEvents : LcEBPending_IN = %I64x\n", *PmPNotifiedInPipeMask));
        }
        if(LcEBPending_OUT)
        {
            *PmPNotifiedOutPipeMask = (	(DWORDLONG)m_Rmh.Stat[1] << 32) + m_Rmh.Stat[2];
            DOUT(DBG_DMA, ("IEvent_AsynchronousEvents : LcEBPending_OUT = %I64x\n", *PmPNotifiedOutPipeMask));
        }

		/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

        LcTarget.tgMaskFlux      = 1;

#ifndef SIMLXES
        LcTarget.tgCarte = m_PIO->PIOGetBoardIndex();
#endif
	
		// check pipe out
		//
		if (LcHasUnderrun) 
		{
			LcTarget.tgCaracPipeVoie = OPER_PLAY ;

			for ( pipe_idx = 0; pipe_idx <	m_pDSPInfo->dsManagedPhysicalOutputNumber ; pipe_idx++)
			{
				PipeOutMask			= UTIMask64bit( pipe_idx );
				LcTarget.tgPipe     = pipe_idx;

				// check for underrun
				if ( PipeOutMask & UrunMask )		APHKeepPipeError( &LcTarget, EB_STREAM_UNDERRUN );
			}
		}

		// check pipe in
		//
		if (LcHasOverrun) 
		{
			LcTarget.tgCaracPipeVoie = OPER_REC ;
			for ( pipe_idx = 0; pipe_idx <	m_pDSPInfo->dsManagedPhysicalInputNumber ; pipe_idx++)
			{
				PipeOutMask			= UTIMask64bit( pipe_idx );
				LcTarget.tgPipe     = pipe_idx;
				// check for underrun
				if ( PipeOutMask & OrunMask )		APHKeepPipeError( &LcTarget, EB_STREAM_UNDERRUN );
			}
		}
	} // async events

    return( LcRet ) ;
}



/*	This will ask for the buffer status on the given pipe.
*	@warning	on LXES, all finished buffers are freed at once with this command
*				and must be notified to the application (no "More" bit)
*
*	@param	PmNbNeededPtr		IN Pointer to the variable that receives the number of Needed buffers.
*	@param	PmNbFreedPtr		IN Pointer to the variable that receives the number of Freed buffers.
*	@param	PmBufferSizeArray	IN array of at least MAX_STREAM_BUFFERS DWORDS, that receive the size of the freed capture buffers.
*/
WORD	CLXProtocol::vio_InitSoundTransferLX(	IN PTARGET_INFO PmPtrTarget,
												OUT PDWORD      PmNbNeededPtr,
												OUT PDWORD      PmNbFreedPtr,
												OUT PDWORD      PmBufferSizeArray)
{
	WORD        LcRet = SUCCESS;
	KIRQL       LcOldIrql;
    BOOLEAN     capture = (PmPtrTarget->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;

	*PmNbNeededPtr = 0;
	*PmNbFreedPtr = 0;


	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	InitMessage( CMD_08_ASK_BUFFERS );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTarget->tgPipe);

    LcRet = SendMessage( );

    if ( LcRet == SUCCESS )
    {
        WORD    i       ;

		for ( i = 0; i < MAX_STREAM_BUFFER ; i++) 
		{
			if		(m_Rmh.Stat[i] & (BF_EOB << BUFF_FLAGS_OFFSET))			/*finished*/
			{
				*PmNbFreedPtr	+= 1;
				if (PmBufferSizeArray)	PmBufferSizeArray[i] = m_Rmh.Stat[i] & MASK_DATA_SIZE;
			}
			else if( (m_Rmh.Stat[i] & (BF_VALID << BUFF_FLAGS_OFFSET)) == 0)	/*free*/
				*PmNbNeededPtr  += 1;
		}
	}

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}


/**
*	Send a buffer for play or capture on a LXES card
*
*   @param	IN	PmPtrTarget
*   @param	IN	PmBuffDataLength
*   @param	IN	PmBuffAddressLo
*   @param	IN	PmBuffAddressHi		set to zero if 64bit transfer not supported, or HIGH part is zero
*   @param	IN	PmPauseRequest
*   @param	IN	PmNotifyBuffer
*   @param	IN	PmMisc
*	@param	OUT PmBufferIndex
*	@see	vio_StreamGiveBufferNP
*/
WORD CLXProtocol::vio_StreamGiveBufferLX(
    IN	PTARGET_INFO     PmPtrTarget,
    IN	DWORD            PmBuffDataLength,
    IN	DWORD            PmBuffAddressLo,
    IN	DWORD            PmBuffAddressHi,
    IN	BYTE             PmPauseRequest,
    IN	BOOLEAN          PmNotifyBuffer,
    IN	DWORD            PmMisc,
	OUT	PBYTE			 PmBufferIndex)
{
	WORD			LcRet = SUCCESS;
	KIRQL			LcOldIrql;
    BOOLEAN			capture = (PmPtrTarget->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;
	PSTREAM_INFO	pStream;


    DOUT (DBG_PROT_VIO, ("vio_StreamGiveBufferNP() @%x len(%d) ",PmBuffAddressLo ,PmBuffDataLength ));

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	InitMessage( CMD_0F_UPDATE_BUFFER );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTarget->tgPipe);

	// params
    //
    if ( PmPauseRequest & DC_DIFFERED_DELAY )       m_Rmh.Cmd[0] |= BF_PAUSE;
    if ( PmNotifyBuffer )                           m_Rmh.Cmd[0] |= BF_NOTIFY_EOB;
    if ( PmMisc & BUFFER_MISC_CIRCULAR_BUFFER )     m_Rmh.Cmd[0] |= BF_CIRCULAR;

	m_Rmh.Cmd[1] = PmBuffDataLength & MASK_DATA_SIZE ;	// datalength
	m_Rmh.Cmd[2] = PmBuffAddressLo;						// addresse PC, lo
	
	if (PmBuffAddressHi)
	{
		m_Rmh.LgCmd = 4;
		m_Rmh.Cmd[3]	 = PmBuffAddressHi;
		m_Rmh.Cmd[0]	|= BF_64BITS_ADR;
	}

    LcRet = SendMessage(  );

	// Bind the PC-Address with the Buffer Index for the card
	//
	if (LcRet == SUCCESS)
	{
		pStream = (PSTREAM_INFO)(PmPtrTarget->tgVioHandle);
		
		ASSERT(MAX_STREAM_BUFFER<=SI_MAX_STREAM_BUFFER); ASSERT(pStream);
	
		pStream->siBufferAddr2Index[m_Rmh.Stat[0] /*buffer index*/] = PmBuffAddressLo;
		
		if (PmBufferIndex)	*PmBufferIndex = (BYTE)m_Rmh.Stat[0] ;
	}

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	return LcRet;
}

//**************************************************************************
//
//  WORD CLXProtocol::vio_FreeBufferAndGetSizeNP(IN PTARGET_INFO,IN WORD,OUT PDWORD)
//
//Input Parameters:
//*****************
//
//  PTARGET_INFO    PmPtrTarget:            Identify the target
//  WORD            PmCurrentStream:        Identify the stream number
//
//Output Parameters:
//*****************
//
//  PDWORD          PmBufferSizePtr:        ptr to current buffer size (in bytes)
//
//Return value:
//*************
//
//  SUCCESS, or an error otherwise.
//
//**************************************************************************
//
//**************************************************************************
WORD CLXProtocol::vio_FreeBufferAndGetSizeNP(
    IN  PTARGET_INFO    PmPtrTarget,
    IN  WORD            PmCurrentStream,
    OUT PDWORD          PmBufferSizePtr)
{
	WORD        LcRet = SUCCESS;
    BOOLEAN     capture = (PmPtrTarget->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;
	KIRQL       LcOldIrql;

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_11_CANCEL_BUFFER );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTarget->tgPipe);
	m_Rmh.Cmd[0] |= MASK_BUFFER_ID;	// ask for the current buffer : the microblaze will seek for it 

    LcRet = SendMessage( );

    if ( LcRet == SUCCESS )
    {
        // Get the size of the freed buffer (in bytes)
        //
        *PmBufferSizePtr = m_Rmh.Stat[0] & MASK_DATA_SIZE ; // remove flags
    }

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    return ( LcRet );
}


WORD CLXProtocol::vio_SoundTransferPlayNP(
    IN  PTARGET_INFO    PmTarget,
    IN  BOOLEAN         PmEndOfPlayRequested,
    IN  WORD            PmNbStreams)
{
    PSTREAM_INFO        LcPtStream;
    LPPLAY_REQ_INFO     LcPtBuffer;
    LPPLAY_REQ_INFO     LcPtrNextBufferToGive = NULL;
    DWORD               LcAddressToTransfer;
    DWORD               LcLengthToTransfer;

    DWORD               LcNeededBuff;
    DWORD               LcFreedBuff;
    WORD                LcRet;

#ifdef SIMLXES
	static WORD		LcBuffIdx = 0;
#endif

    // Ask the DSP to know if there is a buffer
    // to send and a buffer to free.

    LcRet = vio_InitSoundTransferLX(PmTarget, &LcNeededBuff, &LcFreedBuff, NULL /*play*/) ;


    if ( LcRet != SUCCESS ) return( LcRet );

    // Point to the first stream
    // *************************
    LcPtStream = (PSTREAM_INFO) PmTarget->tgVioHandle;

    // Must handle pipes with no stream !
    //
    if ( LcPtStream == NULL ) return LcRet;

	while ( LcFreedBuff--)
	{
		        // Retrieve first sent buffer pointer for the stream ( and not released )
                // ----------------------------------------------------------------------
                LcPtBuffer = (LPPLAY_REQ_INFO)LcPtStream->siAdrBuffer;

                if ( LcPtBuffer != NULL )
                {
                    if( LcPtBuffer->plqPause & DC_DIFFERED_DELAY )
                    {
                        // A buffer with a pause has finished : Notify if wished
                        APHEndPlayPipe( PmTarget );
                    }
                    // Switch to next stream buffer:
                    // Memorize the next buffer to free.
                    // This pointer could point to NULL if we are going to free
                    // the last buffer.
                    // ----------------------------
                    LcPtStream->siAdrBuffer = LcPtBuffer->plqForDriver.dprAdrNextBlk;

                    DOUT(DBG_PROT_BUFFER, ("### FREE PLAY BUFFER %d (pause = %x)\n", LcPtBuffer->plqMBlazeBufIdx, LcPtBuffer->plqPause));

                    // Release the buffer just finished
                    // ---------------------------------
                    vio_FreeBuffer( OPER_PLAY,
                                    LcPtStream,
                                    (LPBC_HEADER_INFO)LcPtBuffer );
                }
                else
                {
                    DOUT (DBG_ERROR, ("CNPProtocol::vio_SoundTransferPlayNP() FREE NULL BUFF !!!"));
                    EXIT_PCX_ERROR( ED_INVALID_BUFFER );
                }
	}   // LcFreedBuff

	while ( LcNeededBuff--)
	{
		// Memorize the buffer pointer to send
		// -----------------------------------
		LcPtrNextBufferToGive = (LPPLAY_REQ_INFO)LcPtStream->siAdrNextBufferToGive;
		
		if ( LcPtrNextBufferToGive != NULL )
		{
			// Get the next buffer pointer to give
			//------------------------------------
			LcPtStream->siAdrNextBufferToGive =
				(LPBC_HEADER_INFO)(LcPtrNextBufferToGive->plqForDriver.dprAdrNextBlk);
			
			if ( LcPtStream->siAdrNextBufferToGive == NULL )
			{
				// Remove the corresponding bit from LcNeededBuffMask
				// because we will not complete the next DSP request
				// --------------------------------------------------
				LcNeededBuff = 0;
			}
			
			// Read the address of storage and the available length
			// ----------------------------------------------------
#ifndef SIMLXES
			
		LcRet = BUFGetBufferInfoForTransNP( LcPtrNextBufferToGive->plqBuffNum,
											&LcPtrNextBufferToGive->plqForDriver.dprHandleBuffer,
											0L,
											LcPtrNextBufferToGive->plqDataLength,
											&LcLengthToTransfer,
											&LcAddressToTransfer );
#else
		LcRet = SUCCESS; LcLengthToTransfer = LcPtrNextBufferToGive->plqDataLength; LcAddressToTransfer = 0x12300000 + LcBuffIdx++;
#endif

			if ( LcRet != SUCCESS ) return( LcRet );

			BYTE LcBufferIndex;	// la carte remonte le n du buffer 

			// Give the requested buffer for the stream
			// -----------------------------------------
			LcRet = vio_StreamGiveBufferLX( PmTarget,
                                            LcLengthToTransfer,
                                            LcAddressToTransfer, // @ lo 
											0,				     // @ hi
											LcPtrNextBufferToGive->plqPause,
											TRUE, // notify EOB
											LcPtrNextBufferToGive->plqMisc,
											&LcBufferIndex );

			if ( LcRet != SUCCESS ) return( LcRet );

            DOUT(DBG_PROT_BUFFER, ("### GIVE PLAY BUFFER %d : @=%p, len=%d (pause=%x)\n", LcBufferIndex, LcAddressToTransfer, LcLengthToTransfer, LcPtrNextBufferToGive->plqPause));

            LcPtrNextBufferToGive->plqMBlazeBufIdx = (DWORD)LcBufferIndex;
		} 
	} // LcNeededBuff

    return( SUCCESS );
}


WORD	CLXProtocol::vio_SoundTransferRecordNP( IN  PTARGET_INFO    PmTarget,
												IN  BOOLEAN         PmFinishing,
												IN  WORD            PmNbStreams) 
{
    PSTREAM_INFO        LcPtStream;
    LPRECORD_REQ_INFO   LcPtBuffer;
    LPRECORD_REQ_INFO   LcPtrNextBufferToGive = NULL;
    DWORD               LcAddressToTransfer;
    DWORD               LcLengthToTransfer;
	DWORD               LcFreedBuffLen[MAX_STREAM_BUFFER];
    DWORD               LcNeededBuff;
    DWORD               LcFreedBuff;
    WORD                LcRet;

    // Ask the DSP to know if there is a buffer
    // to send and a buffer to free.

    LcRet = vio_InitSoundTransferLX(PmTarget, &LcNeededBuff, &LcFreedBuff, LcFreedBuffLen) ;

    if ( LcRet != SUCCESS ) return( LcRet );

    // Point to the first stream
    // *************************
    LcPtStream = (PSTREAM_INFO) PmTarget->tgVioHandle;

    // Must handle pipes with no stream !
    //
    if ( LcPtStream == NULL ) return LcRet;

	while ( LcFreedBuff--)
	{
		// Retrieve first sent buffer pointer for the stream ( and not released )
        // ----------------------------------------------------------------------
        LcPtBuffer = (LPRECORD_REQ_INFO)LcPtStream->siAdrBuffer;

        if ( LcPtBuffer != NULL )
        {
            // Switch to next stream buffer:
            // Memorize the next buffer to free.
            // This pointer could point to NULL if we are going to free
            // the last buffer.
            // ----------------------------
            LcPtStream->siAdrBuffer = LcPtBuffer->rcqForDriver.dprAdrNextBlk;

            // Update the buffer size
            //
            ASSERT( LcPtBuffer->rcqMBlazeBufIdx < MAX_STREAM_BUFFER );
            DOUT(DBG_PROT_BUFFER, ("FREE REC BUF %d len= %d\n", LcPtBuffer->rcqMBlazeBufIdx , LcFreedBuffLen[LcPtBuffer->rcqMBlazeBufIdx]));

            LcPtBuffer->rcqForDriver.dprCurrentLength = LcFreedBuffLen[LcPtBuffer->rcqMBlazeBufIdx] ;

            // Release the buffer just finished
            // ---------------------------------
            vio_FreeBuffer( OPER_REC,
                            LcPtStream,
                            (LPBC_HEADER_INFO)LcPtBuffer );
        }
        else
        {
            DOUT (DBG_ERROR, ("CNPProtocol::vio_SoundTransferPlayNP() FREE NULL BUFF !!!"));
            EXIT_PCX_ERROR( ED_INVALID_BUFFER );
        }
	}   // LcFreedBuff

	while ( LcNeededBuff--)
	{
		// Memorize the buffer pointer to send
		// -----------------------------------
		LcPtrNextBufferToGive = (LPRECORD_REQ_INFO)LcPtStream->siAdrNextBufferToGive;
		
		if ( LcPtrNextBufferToGive != NULL )
		{
			// Get the next buffer pointer to give
			//------------------------------------
			LcPtStream->siAdrNextBufferToGive =
				(LPBC_HEADER_INFO)(LcPtrNextBufferToGive->rcqForDriver.dprAdrNextBlk);
			
			if ( LcPtStream->siAdrNextBufferToGive == NULL )
			{
				// Remove the corresponding bit from LcNeededBuffMask
				// because we will not complete the next DSP request
				// --------------------------------------------------
				LcNeededBuff = 0;
			}
			
			// Read the address of storage and the available length
			// ----------------------------------------------------
#ifndef SIMLXES
			
		LcRet = BUFGetBufferInfoForTransNP( LcPtrNextBufferToGive->rcqBuffNum,
											&LcPtrNextBufferToGive->rcqForDriver.dprHandleBuffer,
											0L,
											LcPtrNextBufferToGive->rcqDataLength,
											&LcLengthToTransfer,
											&LcAddressToTransfer );
#else
		LcRet = SUCCESS; LcLengthToTransfer = LcPtrNextBufferToGive->rcqDataLength; LcAddressToTransfer = 0x20123456;
#endif

			if ( LcRet != SUCCESS ) return( LcRet );

			BYTE LcBufferIndex;	// la carte remonte le n du buffer 

			// Give the requested buffer for the stream
			// -----------------------------------------
			LcRet = vio_StreamGiveBufferLX( PmTarget,
                                            LcLengthToTransfer,
                                            LcAddressToTransfer, // @ lo 
											0,				     // @ hi
											0,
											TRUE, // notify EOB
											LcPtrNextBufferToGive->rcqMisc,
											&LcBufferIndex );

			if ( LcRet != SUCCESS ) return( LcRet );

            DOUT(DBG_PROT_BUFFER, ("### GIVE  REC BUFFER %d : @=%p, len=%d\n", LcBufferIndex, LcAddressToTransfer, LcLengthToTransfer));

            LcPtrNextBufferToGive->rcqMBlazeBufIdx = (DWORD)LcBufferIndex;
		} 
	} // LcNeededBuff

    // If End of record requested, purge buffer
    //
    if ( PmFinishing )
    {
        // get the current size of the buffer only for the current buffer !
        BOOL LcCurrentDone = FALSE;

        for (  ; ; )
        {
            DWORD LcBufferSize = 0;

            // If no buffer remains, then we are done
            //
            if ( LcPtStream->siAdrBuffer == NULL ) break ;

            // If we reach the buffer that are not sent yet, then we are done
            //
            if ( LcPtStream->siAdrBuffer ==
                 LcPtStream->siAdrNextBufferToGive ) break ;

            // Retrieve first sent buffer pointer ( and not released )
            //
            LcPtBuffer = (LPRECORD_REQ_INFO) LcPtStream->siAdrBuffer;

            if( LcCurrentDone == FALSE )
            {
                // Update the buffer size
                //
                LcRet = vio_FreeBufferAndGetSizeNP(
                    PmTarget,
                    0,              // current stream
                    &LcBufferSize);

                if ( LcRet != SUCCESS )
                {
                    // Set the buffer size to zero in the user atempts to read
                    // the size (by ignoring the error for example).
                    //
                    LcPtBuffer->rcqForDriver.dprCurrentLength = 0L ;
                    return( LcRet );
                }

                LcCurrentDone = TRUE;
            }

            DOUT(DBG_PROT_BUFFER, ("FINISHING : FREE REC BUF %d len=%d of %d\n", LcPtBuffer->rcqMBlazeBufIdx , LcBufferSize, LcPtBuffer->rcqDataLength));

            LcPtBuffer->rcqForDriver.dprCurrentLength = LcBufferSize ;

            // Switch to next stream buffer:
            //
            LcPtStream->siAdrBuffer = LcPtBuffer->rcqForDriver.dprAdrNextBlk;

            // Release the buffer just finished
            //
            vio_FreeBuffer( OPER_REC,
                            LcPtStream,
                            (LPBC_HEADER_INFO)LcPtBuffer );
        }
    }
    // endif
    // *****

    return SUCCESS ;
}


// **************************************************************************
//
//  WORD CLXProtocol::vio_StreamCancelBuffer(IN PTARGET_INFO,IN WORD,IN DWORD,IN DWORD,IN BYTE)
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO    PmPtrTarget:            Identify the target
//  WORD            PmCurrentStream:        Identify the stream number
//
// Return value:
// *************
//
//  SUCCESS, or an error otherwise.
//
// **************************************************************************
//
WORD CLXProtocol::vio_StreamCancelBuffer(
    IN PTARGET_INFO     PmPtrTarget,
    IN WORD             PmCurrentStream,
    IN DWORD            PmBuffDataLength,
    IN DWORD            PmBuffAddress )
{
	WORD        LcRet = SUCCESS;
	KIRQL       LcOldIrql;
    BOOLEAN		capture = (PmPtrTarget->tgCaracPipeVoie & OPER_REC) ? TRUE : FALSE;
	PSTREAM_INFO pStream = (PSTREAM_INFO)(PmPtrTarget->tgVioHandle);
    BOOL bFound = FALSE;
    DWORD       k;

    ASSERT(MAX_STREAM_BUFFER<=SI_MAX_STREAM_BUFFER);
    ASSERT(pStream);

    for(k= 0; k<MAX_STREAM_BUFFER; k++)
    {
	    if( pStream->siBufferAddr2Index[k /*buffer index*/] == PmBuffAddress )
        {
            bFound = TRUE;
            break;
        }
    }
#pragma message("verifier k == plqMBlazeBufIdx ?")

    if( ! bFound )  EXIT_PCX_ERROR( WD_CANNOT_CANCEL );

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

    InitMessage( CMD_11_CANCEL_BUFFER );

    m_Rmh.Cmd[0] |= PIPE_INFO_TO_CMD(capture, PmPtrTarget->tgPipe);
	m_Rmh.Cmd[0] |= k;

    LcRet = SendMessage( );

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

	return LcRet;
}

//**************************************************************************
//
// VOID CLXProtocol::IBuffer_CancelStreamBuffer()
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTgPipe:    Identify the target stream
//  LPBC_HEADER_INFO PmPtBuffer : Points to the request bloc of the souncd buffer
//
// Output Paramaters:
// ******************
//
//**************************************************************************
//
// Removes a sound buffer for a stream.
//
//**************************************************************************
//
// FS - 02/04/1997
// function added to manage IRP cancellation (NT only)
// This function simply removes the buffer from the stream, with no extra
// processing.
//
//**************************************************************************
VOID    CLXProtocol::IBuffer_CancelStreamBuffer(	IN  PTARGET_INFO        PmPtrTgPipe  ,
													IN  LPBC_HEADER_INFO    PmPtBuffer  )
{
    PSTREAM_INFO            LcPtStream;
    LPPLAY_REQ_INFO         LcPtBuffer;
    LPPLAY_REQ_INFO         LcPtPrevBuffer;
    DWORD                   LcStreamMask;
	KIRQL                   LcOldIrql;

/*    // ## FS (26/11/97) - bug fix for FA#79
    // First of all, stop the stream. This is essential on a NP board
    // IMPORTANT: we cannot use the VIOStopStream function. Doing so
    // would complete all the IRPs corresponding to the stream's
    // buffers including the IRP we are cancelling: this would
    // ineluctably end into a blue screen for a
    // MULTIPLE_IRP_COMPLETE_REQUEST
    // -------------------------------------------------------------

	/*>>*/ KeAcquireSpinLock (m_MsgLock, &LcOldIrql);

	// Prepare the command ...
    //
 /*   vio_InitRmh( CMD_STOP_STREAM );

    // ## FM (08/03/2000) -- BUG FIX (was missing!)
    //
    if ( PmPtrTgPipe->tgCaracPipeVoie == OPER_REC ) {
        m_Rmh.Cmd[0] |= 0x000800;
    }

    m_Rmh.Cmd[0] |= ( PmPtrTgPipe->tgPipe << FIELD_SIZE);
    m_Rmh.Cmd[1] = PmPtrTgPipe->tgMaskFlux;

    // Actually send the RMH
    //
    (VOID) vio_SendMessage(  );

	/*<<*/ KeReleaseSpinLock (m_MsgLock, LcOldIrql);

    // Once the stream is stopped, we are sure that the DSP
    // will no more tell us that it needs buffers for
    // this stream. Hence, we can now safely remove the buffer
    // from the chained list of the stream
    // -------------------------------------------------------

    // Calculate the Stream pointer
    // ****************************
    LcPtStream = (PSTREAM_INFO) PmPtrTgPipe->tgVioHandle;
    LcStreamMask = PmPtrTgPipe->tgMaskFlux >> 1;

    while (LcStreamMask)
    {
        LcPtStream = LcPtStream->siAdrNextStream;
        LcStreamMask >>= 1;
    }

    // Retrieve the buffer in the stream buffer list
    // ----------------------------------------------
    LcPtBuffer = (LPPLAY_REQ_INFO)LcPtStream->siAdrBuffer;

    // Just in case...
    if ( LcPtBuffer == NULL)
    {
        return ;
    }

	LcPtPrevBuffer = NULL;	// prefast warning
    while (   ( LcPtBuffer != (LPPLAY_REQ_INFO)PmPtBuffer )
           && ( (LcPtBuffer->plqForDriver).dprAdrNextBlk != NULL ) )
    {
        LcPtPrevBuffer = LcPtBuffer ;
        LcPtBuffer = (LPPLAY_REQ_INFO) LcPtBuffer->plqForDriver.dprAdrNextBlk;
    }

    // Remove the buffer
    // -----------------
    if ( LcPtBuffer == (LPPLAY_REQ_INFO)PmPtBuffer )
    {
        if ( LcPtBuffer != (LPPLAY_REQ_INFO)LcPtStream->siAdrBuffer )
        {
			// Just in case...(prefast warning)
			if ( LcPtPrevBuffer == NULL)
			{
				return ;
			}
            (LcPtPrevBuffer->plqForDriver).dprAdrNextBlk =
                                (LcPtBuffer->plqForDriver).dprAdrNextBlk ;
        }
        else
        {
            LcPtStream->siAdrBuffer = (LcPtBuffer->plqForDriver).dprAdrNextBlk ;
        }

        // CCA (05/06/2000) -- FA N120
        // Update siAdrNextBufferToGive field if it points to the deleted buffer.
        //
        if ( LcPtBuffer == (LPPLAY_REQ_INFO)LcPtStream->siAdrNextBufferToGive )
            LcPtStream->siAdrNextBufferToGive = (LcPtBuffer->plqForDriver).dprAdrNextBlk ;
    }
}	// IBuffer_CancelStreamBuffer


BOOLEAN CLXProtocol::IEvent_AcknowledgeIRQ( BOOLEAN* PmTimerOccured, BOOLEAN* PmAsyncPending, BOOLEAN* PmAsyncESCMD )
{
	BOOLEAN LcRet = FALSE;
	DWORD LcTimerIT;
	DWORD LcAsyncIT;

#ifndef SIMLXES
	DWORD LcIrqSrc = m_PIO->PIOTestAndAckIt();
#else
	DWORD LcIrqSrc = Sim_PLX_Registers[9];
#endif

	if(LcIrqSrc != PCX_IRQ_NONE)
	{
		LcAsyncIT = LcIrqSrc & MASK_SYS_ASYNC_EVENTS; // + EtherSound response (set by xilinx) + EOB

		if( LcAsyncIT & MASK_SYS_STATUS_ESA )
        {
            LcAsyncIT &= ~MASK_SYS_STATUS_ESA;

            *PmAsyncESCMD = TRUE;
        }

        if(LcIrqSrc & MASK_SYS_STATUS_CMD_DONE)
        {
#ifndef SIMLXES
            LONG lOld = InterlockedExchange( &m_PIO->m_SendMessageLocked, 0);
#ifdef DBG_VIEWER
            if(lOld != 1)
            {
                DOUT(DBG_WARNING, ("**** WARNING IT MASK_SYS_STATUS_CMD_DONE with m_SendMessageLocked=%d\n", lOld));
            }
#endif
#endif
        }

		if (m_K) 
		{
			LcTimerIT = (LcIrqSrc & MASK_SYS_TIMER_COUNT) / m_K;  // Timer Counter of K*IRQ_XES 
			if(m_TimerIT != LcTimerIT)
			{
				DOUT(DBG_PROT_EVENT, ("TIMER_EVENT_OCCURED!\n"));
				*PmTimerOccured = TRUE;
				m_TimerIT = LcTimerIT;
			}
		}

		if(LcAsyncIT)	
		{
			DOUT(DBG_PROT_EVENT, ("ASYNC_EVENT_PENDING!\n"));
			*PmAsyncPending = TRUE;
			m_SrcIT = LcAsyncIT;
		}
		LcRet = TRUE;
	}

	return LcRet;
}  // IEvent_AcknowledgeIRQ 


#ifdef DBG_VIEWER
#define CMD_NAME(a) a
#else
#define CMD_NAME(a) NULL
#endif


/**
*	@brief	Initialization and control data for the Microblaze interface
*	
*	- OpCode:		the opcode field of the command set at the proper offset
*	- CmdLength		the number of command words
*	- StatusType	offset in the status registers : 0 means that the return value may be different from SUCCESS, and must be read
*	- StatusLength	the number of status words (in addition to the return value)
*/
DSP_CMD_INFO	CLXProtocol::m_DspCommands[] = 
{
    // ----------------------------------------------------------------------------------------
    // OpCode      |							CmdLength   |   StatusType  |    StatusLength
    // ----------------------------------------------------------------------------------------
    { (CMD_00_INFO_DEBUG << OPCODE_OFFSET )			,1 /*custom*/   ,1		,0 /**/		        , CMD_NAME("INFO_DEBUG") },
	{ (CMD_01_GET_SYS_CFG << OPCODE_OFFSET )		,1 /**/         ,1      ,2 /**/		        , CMD_NAME("GET_SYS_CFG") },
    { (CMD_02_SET_GRANULARITY << OPCODE_OFFSET )	,1 /**/         ,1      ,0 /**/		        , CMD_NAME("SET_GRANULARITY") },
	{ (CMD_03_SET_TIMER_IRQ << OPCODE_OFFSET )		,1 /**/         ,1      ,0 /**/		        , CMD_NAME("SET_TIMER_IRQ") },
    { (CMD_04_GET_EVENT << OPCODE_OFFSET )			,1 /**/         ,1      ,0 /*up to 10*/     , CMD_NAME("GET_EVENT") },
    { (CMD_05_GET_PIPES << OPCODE_OFFSET )			,1 /**/         ,1      ,2 /*up to 4*/      , CMD_NAME("GET_PIPES") },
    { (CMD_06_ALLOCATE_PIPE << OPCODE_OFFSET )		,1 /**/         ,0      ,0 /**/		        , CMD_NAME("ALLOCATE_PIPE") },
	{ (CMD_07_RELEASE_PIPE << OPCODE_OFFSET )		,1 /**/         ,0      ,0 /**/		        , CMD_NAME("RELEASE_PIPE") },
    { (CMD_08_ASK_BUFFERS << OPCODE_OFFSET )		,1 /**/         ,1      ,MAX_STREAM_BUFFER  , CMD_NAME("ASK_BUFFERS") },
	{ (CMD_09_STOP_PIPE << OPCODE_OFFSET )			,1 /**/         ,0      ,0 /*up to 2*/      , CMD_NAME("STOP_PIPE") },
    { (CMD_0A_GET_PIPE_SPL_COUNT << OPCODE_OFFSET )	,1 /**/			,1      ,1 /*up to 2*/      , CMD_NAME("GET_PIPE_SPL_COUNT") },
	{ (CMD_0B_TOGGLE_PIPE_STATE << OPCODE_OFFSET )  ,1 /*up to 5*/  ,1      ,0 /**/		        , CMD_NAME("TOGGLE_PIPE_STATE") },
	{ (CMD_0C_DEF_STREAM << OPCODE_OFFSET )			,1 /*up to 4*/	,1      ,0 /**/		        , CMD_NAME("DEF_STREAM") },
    { (CMD_0D_SET_MUTE  << OPCODE_OFFSET )			,3 /**/			,1      ,0 /**/		        , CMD_NAME("SET_MUTE") },
	{ (CMD_0E_GET_STREAM_SPL_COUNT << OPCODE_OFFSET ),1/**/			,1      ,2 /**/		        , CMD_NAME("GET_STREAM_SPL_COUNT") },
    { (CMD_0F_UPDATE_BUFFER << OPCODE_OFFSET )		,3 /*up to 4*/	,0      ,1 /**/		        , CMD_NAME("UPDATE_BUFFER") },
    { (CMD_10_GET_BUFFER << OPCODE_OFFSET )			,1 /**/			,1      ,4 /**/		        , CMD_NAME("GET_BUFFER") },
	{ (CMD_11_CANCEL_BUFFER << OPCODE_OFFSET )		,1 /**/			,1      ,1 /*up to 4*/      , CMD_NAME("CANCEL_BUFFER") },
    { (CMD_12_GET_PEAK << OPCODE_OFFSET )			,1 /**/		    ,1      ,1 /**/		        , CMD_NAME("GET_PEAK") },
    { (CMD_13_SET_STREAM_STATE << OPCODE_OFFSET )	,1 /**/		    ,1      ,0 /**/		        , CMD_NAME("SET_STREAM_STATE") },
};

DWORD	CLXProtocol::m_Peak2Sample[] = 
{
	0x00000109, // -90.308dB
    0x0000083B, // -72.247dB
	0x000020C4,	// -60.205dB
	0x00008273,	// -48.030dB
	0x00020756, // -36.005dB
	0x00040C37, // -30.001dB
	0x00081385, // -24.002dB
	0x00101D3F, // -18.000dB
	0x0016C310,	// -15.000dB
	0x002026F2,	// -12.001dB
	0x002D6A86, // -9.000dB
	0x004026E6,	// -6.004dB
	0x005A9DF6, // -3.000dB
	0x0065AC8B,	// -2.000dB
	0x00721481, // -1.000dB
	0x007FFFFF, // FS
};

