// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************


// *************************************************************************
//                  Status Pipe subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd6( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Status command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd6(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM6 );
}

// *************************************************************************
//
// STATIC VOID PipeSampleCountFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// ****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Parameters:
// *****************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Process the sample count command.
//  Update the sample count command response bloc.
//
// *************************************************************************
STATIC VOID PipeSampleCountFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_STATUS_INFO          LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;
    LPSAMPLE_COUNTING_RESP_INFO LcLPtrRep;
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    TARGET_INFO                 LcTargetInf;
    BYTE                        LcAppIndex;
    WORD                        LcRet,LcCurrentBoard;
    WORD                        LcPipeIndex;
    PCX_TIME                    LcSampleCount;
    PVOIE_INFO                  LcVoieInfPtr;
    PPIPE_INFO                  LcPipeInfPtr;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_STATUS_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRep       = (LPSAMPLE_COUNTING_RESP_INFO)PmBlocRetour;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // FS - 15/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof( SAMPLE_COUNTING_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =   sizeof( SAMPLE_COUNTING_RESP_INFO )
                              - sizeof( RESP_HEADER_INFO );


    // Retrieve application index
    // --------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );


    // Retrieve and verify pipe masks
    // -------------------------------
    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcLPtrReq->scqOutputPipeMask64.QuadPart,
                                        LcLPtrReq->scqInputPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPipeInfPtr,
                                        &LcVoieInfPtr,
                                        &LcTargetInf );

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    LcCurrentBoard = LcTargetInf.tgCarte;

	LcRet = APH_Protocol_Array[LcCurrentBoard]->IFlow_PipeSampleCount( &LcTargetInf, &LcSampleCount );

    if ( !( LcRet & ERROR_MASK ) )
    {
        LcLPtrRep->scpSampleCount = LcSampleCount;
    }

    LcLPtrRep->RespHeader.rhCptr = LcRet;
    return ;
}

// *************************************************************************
//
// STATIC VOID PipeStateFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Status command response bloc
//
// *************************************************************************
//
//  Tell whether a pipe is started or paused
//
// *************************************************************************
STATIC VOID PipeStateFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPPIPE_STATUS_INFO      LcLPtrReq;
    LPBC_HEADER_INFO        LcLPtrReqHeader;
    LPPIPE_STATE_RESP_INFO  LcLPtrResp;
    LPRESP_HEADER_INFO      LcLPtrRespHeader;
    TARGET_INFO             LcTarget;
    PVOIE_INFO              LcVoieInfPtr;
    PPIPE_INFO              LcPipeInfPtr;
    BYTE                    LcAppIndex;
    WORD                    LcPipeIndex;
    WORD                    LcRet;
    BYTE                    LcState;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_STATUS_INFO)      PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)        PmBlocRequete;
    LcLPtrResp      = (LPPIPE_STATE_RESP_INFO)  PmBlocRetour;
    LcLPtrRespHeader= (LPRESP_HEADER_INFO)      PmBlocRetour;

    // FS - 15/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof( PIPE_STATE_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRespHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRespHeader->rhSize =   sizeof( PIPE_STATE_RESP_INFO )
                   - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );


    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcLPtrReq->scqOutputPipeMask64.QuadPart,
                                        LcLPtrReq->scqInputPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPipeInfPtr,
                                        &LcVoieInfPtr,
                                        &LcTarget );

	WORD LcCurrentBoard = LcTarget.tgCarte;


    if ( LcRet != SUCCESS )
    {
        LcLPtrRespHeader->rhCptr = LcRet;
        return;
    }

    // Ask the DSP for the pipe state
    // ------------------------------
    LcRet = CUR_PROTOCOL_PTR->IFlow_PipeState( &LcTarget, &LcState );

    if ( !( LcRet & ERROR_MASK ) )
    {
        // Send the state back to the user
        // -------------------------------
        LcLPtrResp->pspStart = (LcState == PSTATE_IDLE) ? 0 : 1;
    }

    LcLPtrRespHeader->rhCptr = LcRet ;
    return;
}

// *************************************************************************
//
// STATIC VOID     PipeGetNotificationFct( IN LPDWORD, OUT LPDWORD );
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Status command response bloc
//
// *************************************************************************
//
//  Read a notification of differed command for the pipe or tell if none
// is available
//
// *************************************************************************
STATIC VOID     PipeGetNotificationFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPPIPE_NOTIFICATION_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO                LcLPtrReqHeader;
    LPPIPE_NOTIFICATION_RESP_INFO   LcLPtrResp;
    LPRESP_HEADER_INFO              LcLPtrRespHeader;
    TARGET_INFO             LcTarget;
    PVOIE_INFO              LcVoieInfPtr;
    PPIPE_INFO              LcPipeInfPtr;
    BYTE                    LcAppIndex;
    WORD                    LcPipeIndex;
    WORD                    LcRet;
    WORD                    LcCommandCode;
    PCX_TIME                LcCommandTime;
    BYTE                    LcCommandDelay;
    DWORD                   LcCommandParam1;
    DWORD                   LcCommandParam2;
    BOOLEAN                 LcLast;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_NOTIFICATION_REQ_INFO)    PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)                PmBlocRequete;
    LcLPtrResp      = (LPPIPE_NOTIFICATION_RESP_INFO)   PmBlocRetour;
    LcLPtrRespHeader= (LPRESP_HEADER_INFO)              PmBlocRetour;

    // FS - 15/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof( PIPE_NOTIFICATION_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRespHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRespHeader->rhSize =   sizeof( PIPE_NOTIFICATION_RESP_INFO )
                   - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );


    LcRet = ChkPmOneInOrOutPipeMask(
                                LcAppIndex,
                                LcLPtrReq->pnqOutPipeMask64.QuadPart,
                                LcLPtrReq->pnqInPipeMask64.QuadPart,
                                &LcPipeIndex,
                                &LcPipeInfPtr,
                                &LcVoieInfPtr,
                                &LcTarget );

	WORD LcCurrentBoard = LcTarget.tgCarte;


    if ( LcRet != SUCCESS )
    {
        LcLPtrRespHeader->rhCptr = LcRet;
        return;
    }

    // Ask the DSP for the pipe state
    // ------------------------------
    LcRet = CUR_PROTOCOL_PTR->IEvent_PipeNotification(	&LcTarget       ,
														&LcCommandCode  ,
														&LcCommandTime  ,
														&LcCommandDelay ,
														&LcCommandParam1,
														&LcCommandParam2,
														&LcLast         );


    if ( (!LcRet) && LcLast )
    {
        // ## FS (16/04/1998) -- FA #152
        // If a command has been notified but no read
        // has occured yet for the pipe, return a
        // notification anyway for an unknown command
        // ------------------------------------------
        if (   ( LcPipeInfPtr->piNotifiedNotReadYet )
            && ( LcCommandCode == 0 ) )
        {
            // Fake an unknown command notification
            // ------------------------------------
            LcCommandCode = UNKNOWN_COMMAND_NOTIFIED ;
            LcCommandTime = 0 ;
            LcCommandDelay = 0 ;
            LcCommandParam1 = 0 ;
            LcCommandParam2 = 0 ;
        }

        // Reset the flag of waiting notifications
        // ---------------------------------------
        LcPipeInfPtr->piPendingNotify = FALSE ;
    }

    // ## FS (16/04/1998) -- FA # 152
    // At least one getnotification has been done, there
    // is no more need for a notification of an unknown command
    // ---------------------------------------------------------
    LcPipeInfPtr->piNotifiedNotReadYet = FALSE ;

    LcLPtrResp->pnrOutPipeMask64 = LcLPtrReq->pnqOutPipeMask64;
    LcLPtrResp->pnrInPipeMask64  = LcLPtrReq->pnqInPipeMask64;

    // Command Opcode
    LcLPtrResp->pnrCommandCode      = LcCommandCode;

    // differed param of the command
    LcLPtrResp->pnrDiffered         = LcCommandDelay;

    // notification time
    LcLPtrResp->pnrHour             = LcCommandTime;

    // 1st param of the command
    LcLPtrResp->pnrParam1           = LcCommandParam1;

    // 2nd param of the command (buffer tag)
    LcLPtrResp->pnrParam2           = LcCommandParam2;

    LcLPtrRespHeader->rhCptr = LcRet ;
    return;
}

// *************************************************************************
//
// STATIC VOID     PipeGetClockFct( IN LPDWORD, OUT LPDWORD );
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Status command response bloc
//
// *************************************************************************
//
//  Read the clock information for the pipe.
//
// *************************************************************************
STATIC VOID     PipeGetClockFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPPIPE_GET_CLOCK_REQ_INFO   LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;
    LPPIPE_GET_CLOCK_RESP_INFO  LcLPtrResp;
    LPRESP_HEADER_INFO          LcLPtrRespHeader;
    TARGET_INFO                 LcTarget;
    PVOIE_INFO                  LcVoieInfPtr;
    PPIPE_INFO                  LcPipeInfPtr;
    PBOARD_INFO                 LcPBoard;
    DWORD                       LcFrequency = 0;
    WORD                        LcPipeIndex;
    BYTE                        LcAppIndex;
    WORD                        LcRet;
    WORD                        LcCurrentBoard;
    SRC_CONFIG                  LcSrcConfig;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_GET_CLOCK_REQ_INFO)       PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)                PmBlocRequete;
    LcLPtrResp      = (LPPIPE_GET_CLOCK_RESP_INFO)      PmBlocRetour;
    LcLPtrRespHeader= (LPRESP_HEADER_INFO)              PmBlocRetour;

    // FS - 15/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof( PIPE_GET_CLOCK_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRespHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRespHeader->rhSize =   sizeof( PIPE_GET_CLOCK_RESP_INFO )
                   - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Only one pipe must be specified
    // -------------------------------
    if(LcAppIndex == DHSAppHandle)
    {
        LcRet = DHSGetBoardAudioFromOnePipe(
                                LcLPtrReq->scqOutputPipeMask64.QuadPart,
                                LcLPtrReq->scqInputPipeMask64.QuadPart,
                                &LcTarget.tgCarte,
                                &LcTarget.tgAudio );
    }
    else
    {
        LcRet = ChkPmOneInOrOutPipeMask(
                                LcAppIndex,
                                LcLPtrReq->scqOutputPipeMask64.QuadPart,
                                LcLPtrReq->scqInputPipeMask64.QuadPart,
                                &LcPipeIndex,
                                &LcPipeInfPtr,
                                &LcVoieInfPtr,
                                &LcTarget );
    }

    //if pipe not specified, board and audio are not known!
    if( LcRet != SUCCESS )
    {
        LcLPtrRespHeader->rhCptr = LcRet;
        return;
    } // forceread

    // Retrieve the board description
    // corresponding to this pipe
    // ------------------------------
    LcPBoard = &(APH_Board_Info_Array[LcTarget.tgCarte]);

    // Fill in the reply block from the board entry
    // --------------------------------------------
    LcLPtrResp->pcrOutputMask64.QuadPart = LcPBoard->biPipeClockMask[NO_MASK_PIPE_OUT];
    LcLPtrResp->pcrInputMask64.QuadPart  = LcPBoard->biPipeClockMask[NO_MASK_PIPE_IN];

    LcRet = APHGetPipeClock( LcLPtrReq->scqForceActual, 
                             &LcTarget,
                             &LcLPtrResp->pcrSource,
                             &LcLPtrResp->pcrSync,
                             &LcLPtrResp->pcrFrequency );

	LcCurrentBoard = LcTarget.tgCarte;

    if( LcLPtrResp->pcrSource == CLOCK_TYPE_UER_SYNCHRO )
    {
        LcLPtrResp->pcrMisc = (WORD) CUR_COMMANDS_PTR->PIOGetActualClkInputNum();
    }
    else
    {
        LcLPtrResp->pcrMisc = 0;
    }

	// if a SRC exist on this Audio (Input and Output audio use the same SRC! ):
	//
	if(SUCCESS == CUR_COMMANDS_PTR->PIOGetConfigSRC( LcTarget.tgAudio, &LcSrcConfig, &(LcLPtrResp->pcrSRCRatio)))
	{
        LcLPtrResp->pcrFlags    =  (LcSrcConfig == SRC_off) ? 0 : SRC_IN_USE;

		if (  ((LcLPtrReq->scqInputPipeMask64.QuadPart != 0) &&  (LcSrcConfig != SRC_out))
            ||(LcSrcConfig != SRC_in) // LcLPtrReq->scqOutputPipeMaskLo != 0
           )
           LcLPtrResp->pcrFlags    |=   SRC_AVAILABLE;
	}

    // only tell ratio if the SRC is actually set for this audio
    //
    if (LcLPtrResp->pcrFlags !=  (SRC_AVAILABLE|SRC_IN_USE)) 
        LcLPtrResp->pcrSRCRatio = 0;
    
    LcLPtrRespHeader->rhCptr = LcRet ;
    return;
}

// *************************************************************************
//                  Audio Status subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd7( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Audio Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Audio Status command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd7(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM7 );
}

// *************************************************************************
//
// STATIC VOID GetAudioVuMeterSubFct( IN BYTE, IN DWORD, IN DWORD, IN WORD,
//                                    OUT LPVU_METER_RESP_INFO, OUT PWORD )
//
// Input Parameters:
// *****************
//
//  BYTE    PmAppHandle:    Application handle
//  DWORD   PmOutPipeMask:  Mask of output pipes
//  DWORD   PmInPipeMask:   Mask of input pipes
//  WORD    PmAudioMask:    Mask of pipe audios
//
// Output Paramaters:
// ******************
//
//  LPVU_METER_RESP_INFO    PmLPtrRepVuMeter:   Vu/peak meter descriptors
//  PWORD                   PmNbAudioPtr:       Number of vu/peak meter
//                                              descriptors filled in
//
// Return Value:
// *************
//
// SUCCESS if the vu/peak meters have been retrieved successfully, an error
// code otherwise
//
// *************************************************************************
STATIC WORD GetAudioVuMeterSubFct(
    IN  BYTE                    PmAppHandle,
    IN  DWORDLONG               PmOutPipeMask,
    IN  DWORDLONG               PmInPipeMask,
    IN  WORD                    PmAudioMask,
    OUT LPVU_METER_RESP_INFO    PmLPtrRepVuMeter,
    OUT PWORD                   PmNbAudioPtr )
{
    TARGET_INFO                 LcTargetInf;
    STATIC VU_PIC_METER_INFO    LcVuPicMeterInf[MAX_BOARD_AUDIO];
    PVU_PIC_METER_INFO          LcPtrVuPicMeterInf;
    WORD                        LcNbAudio = 0, LcPipeIndex = 0;
    DWORDLONG                   LcOkAudioMask = 0;
    WORD                        i, LcRet ;
    DWORDLONG                   LcAudioMask;
    BYTE                        LcAppIndex;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;
    BOOLEAN                     LcIsInput;

    LcAppIndex = DecodeHandle( PmAppHandle );

    if(LcAppIndex == DHSAppHandle)
    {
        return DHSGetAudioVuMeter(  PmOutPipeMask,
                                    PmInPipeMask,
                                    PmAudioMask,
                                    PmLPtrRepVuMeter,
                                    PmNbAudioPtr );
    }

    // Verify pipe specified
    // Check if it is an Input or Output command
    // and retrieve the pipe index
    //------------------------------------------
    LcRet = ChkPmOneInOrOutPipeMask(
                            LcAppIndex,
                            PmOutPipeMask,
                            PmInPipeMask,
                            &LcPipeIndex,
                            &LcPipeInfPtr,
                            &LcVoieInfPtr,
                            &LcTargetInf );

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        return LcRet;
    }

    // Select the right table for a scan for all pipe's audios
    // -------------------------------------------------------
    if ( LcTargetInf.tgCaracPipeVoie == OPER_PLAY )
    {
        LcIsInput = FALSE;
    }
    else
    {
        LcIsInput = TRUE;
    }

    // Let's scan for audio allocated by this pipe
    // -------------------------------------------

    LcOkAudioMask = LcPipeInfPtr->piPhysAudioMask & (((DWORDLONG)PmAudioMask) << LcPipeInfPtr->piNumPipeForDsp);

    LcAudioMask = LcOkAudioMask;
    LcNbAudio = 0;
    while ( LcAudioMask )
    {
        if( LcAudioMask & (DWORDLONG)1 )
            LcNbAudio++;
        LcAudioMask >>= 1;
    }

    LcPtrVuPicMeterInf = &LcVuPicMeterInf[0];

    // ## FS (11/12/97) -- bug fix for FA #91
    // the application may send a bad pipe audio mask that does not
    // match any ouput. In this case, we should better report an
    // error
    // ------------------------------------------------------------
    if ( LcNbAudio == 0 )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
    }
    else
    {
        LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetVuPicMeter(LcIsInput,
															LcOkAudioMask,
															LcNbAudio,
															LcPtrVuPicMeterInf );
        if ( LcRet != SUCCESS )
        {
            // No audio meters returned
            // in error case
            // ------------------------
            LcNbAudio = 0;
        }
    }

    // Fill output parameters
    // -----------------------
    *PmNbAudioPtr = LcNbAudio ;

    for ( i=0; i<LcNbAudio; i++, LcPtrVuPicMeterInf++, PmLPtrRepVuMeter++ )
    {
        PmLPtrRepVuMeter->vmSaturation = LcPtrVuPicMeterInf->vpmiSaturation;
        PmLPtrRepVuMeter->vmVuMeter    = LcPtrVuPicMeterInf->vpmiVuMeter;
        PmLPtrRepVuMeter->vmPeakMeter  = LcPtrVuPicMeterInf->vpmiPicMeter;
    }

    return LcRet;
}


// *************************************************************************
//
// STATIC VOID AudioVPMeterFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
// FH Creation
//
// *************************************************************************
STATIC VOID AudioVPMeterFct(    IN LPDWORD  PmBlocRequete,
                                OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPVU_METER_RESP_INFO        LcLPtrRepVuMeter;
    LPAUDIO_VUMETER_REQ_INFO    LcLPtrReqHeader;
    WORD                        LcRet;
    WORD                        LcNbAudio = 0 ;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRepVuMeter = (LPVU_METER_RESP_INFO)( LcLPtrRepHeader + 1 );

    LcLPtrReqHeader = (LPAUDIO_VUMETER_REQ_INFO)PmBlocRequete;

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbAudio*sizeof(VU_METER_RESP_INFO)) )
    {
        LcRet = ED_REPLY_BLOC_TOO_SHORT;
    }
    else
    {
        // Actually process the request
        // ----------------------------
        LcRet = GetAudioVuMeterSubFct(
                        (LcLPtrReqHeader->Header).hdHandle,
                        LcLPtrReqHeader->avqOutputPipeMask64.QuadPart,
                        LcLPtrReqHeader->avqInputPipeMask64.QuadPart,
                        LcLPtrReqHeader->avqAudioMask,
                        LcLPtrRepVuMeter,
                        &LcNbAudio );
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbAudio*sizeof(VU_METER_RESP_INFO) );

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID GetOutAudioLevelsSubFct( IN BYTE, IN DWORD, IN WORD,
//                                       OUT LPOUT_AUDIO_GET_LEVELS_RESP_INFO,
//                                       OUT PWORD )
//
// Input Parameters:
// *****************
//
//  BYTE    PmAppHandle:    Application handle
//  DWORD   PmPipeMask:     Mask of output pipes
//  WORD    PmAudioMask:    Mask of pipe audios
//
// Output Paramaters:
// ******************
//
//  LPOUT_AUDIO_GET_LEVELS_RESP_INFO PmLPtrAudio:   Audio levels descriptors
//  PWORD                           PmNbAudioPtr:   Number of descriptors
//                                                   filled in
//
// Return Value:
// *************
//
// SUCCESS if the levels have been retrieved successfully, an error
// code otherwise
//
// *************************************************************************
STATIC WORD GetOutAudioLevelsSubFct(
    IN  BYTE                                PmAppHandle,
    IN  DWORDLONG                           PmPipeMask,
    IN  WORD                                PmAudioMask,
    OUT LPOUT_AUDIO_GET_LEVELS_RESP_INFO    PmLPtrAudio,
    OUT PWORD                               PmNbAudioPtr )
{
    TARGET_INFO                         LcTargetInf;
    STATIC LEVEL_AUDIO_INFO             LcLevelAudioInf[MAX_AUDIO_PIPE];
    PLEVEL_AUDIO_INFO                   LcPtrLevelAudioInf;
    WORD                                i, LcRet;
    WORD                                LcAudioMask, LcLevel ;
    DWORDLONG                           LcOkAudioMask = 0;
    WORD                                LcNbAudio = 0;
    BYTE                                LcAppIndex;
    WORD                                LcOutPipeIndex;
    PPIPE_INFO                          LcPipeInfPtr;
    PVOIE_INFO                          LcVoieInfPtr;
    LPOUT_AUDIO_GET_LEVELS_RESP_INFO    LcLPtrAudio;
    BOOLEAN                             LcIsInput;

    // Initialize output params
    // ------------------------
    *PmNbAudioPtr = 0;

    LcAppIndex = DecodeHandle( PmAppHandle );

    if( LcAppIndex == DHSAppHandle )
    {
        return DHSGetOutAudioLevels(
                        PmPipeMask,
                        PmAudioMask,
                        PmLPtrAudio,
                        PmNbAudioPtr);
    }
    // a get is not forbidden even if this audio is controlled by DHS

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneOutputPipeMask( LcAppIndex,
                                    PmPipeMask,
                                    &LcOutPipeIndex,
                                    &LcPipeInfPtr,
                                    &LcVoieInfPtr,
                                    &LcTargetInf );

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        return LcRet;
    }

    LcIsInput = FALSE;

    // Let's scan for audio allocated by this pipe and
    // apply the set level command if in the scope of request mask
    // -------------------------------------------------------------
    LcAudioMask = PmAudioMask ;
    LcLPtrAudio = PmLPtrAudio;
    LcNbAudio = 0;

    for ( i = 0 ; i < LcPipeInfPtr->piNbAudio ; i++ )
    {
        if ( LcAudioMask == 0 ) break ;

        // Get levels only if audio is physical, owned by the appli and
        // the pipe, and match the given mask
        // -------------------------------------------------------------
        if (  ( LcVoieInfPtr[i].Nat == 0 )
            && ( UTIMaskWord( LcVoieInfPtr[i].AppAudioNum ) & LcAudioMask  ) )
        {
            ASSERT ( LcVoieInfPtr[i].IndexAppli != 0 );
            ASSERT ( LcVoieInfPtr[i].NumPipe == LcOutPipeIndex );

            // Get analog level
            // *****************
            LcRet = CUR_COMMANDS_PTR->PIOGetAudioAnalogLevel(   LcVoieInfPtr[i].AudioNum,
                                                                OPER_PLAY,
                                                                &LcLevel );

            // ## FS -- bug fix for FA #77 - we should not return
            // WD_NO_HARDWARE_SUPPORT on a digital-only board
            // -----------------------------------------------
            if ( LcRet & ERROR_MASK )
            {
               return LcRet;
            }

            // Store the level
            // ---------------
            LcLPtrAudio->olpAnalogLevel = LcLevel ;
            LcLPtrAudio++;
            LcNbAudio++;

            LcAudioMask &= ~UTIMaskWord( LcVoieInfPtr[i].AppAudioNum ) ; // BUG FH

            // Build a audio Mask with the autorised audio IO
            //-----------------------------------------------
            LcOkAudioMask |= UTIMask64bit( LcVoieInfPtr[i].AudioNum );

        }  // if this audio is concerned

    } // for all audio outputs

    LcPtrLevelAudioInf = &LcLevelAudioInf[0];

    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetLevels(	LcIsInput,
														LcOkAudioMask,
														LcNbAudio,
														LcPtrLevelAudioInf );

    if ( LcRet == SUCCESS )
    {
        // Fill response block
        // -------------------
        for ( i=0, LcLPtrAudio = PmLPtrAudio; i<LcNbAudio; i++, LcPtrLevelAudioInf++, LcLPtrAudio++ )
        {
            // Get digital level
            // *****************
            LcLPtrAudio->olpDigitalLevel = LcPtrLevelAudioInf->gaiDigitalLevel;
            // Get monitoring level
            // ********************
            LcLPtrAudio->olpMonitoringLevel = LcPtrLevelAudioInf->gaiMonitorLevel;
            // Get mute
            // ********
            LcLPtrAudio->olpMute = LcPtrLevelAudioInf->gaiMuteLevel;
            // Get monitoring mutes
            // *********************
            LcLPtrAudio->olpMonitoringMute1 = LcPtrLevelAudioInf->gaiM1Level;
            LcLPtrAudio->olpMonitoringMute2 = LcPtrLevelAudioInf->gaiM2Level;
        }

        *PmNbAudioPtr = LcNbAudio;
    }

    return LcRet ;
}

// *************************************************************************
//
// STATIC VOID GetOutAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get all levels for output audio
//
//  FH Modification: Add digital levels processing
//
// *************************************************************************
STATIC VOID GetOutAudioLevelsFct(
    IN LPDWORD  PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPOUT_AUDIO_GET_LEVELS_REQ_INFO     LcLPtrReq;
    LPBC_HEADER_INFO                    LcLPtrReqHeader;
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    LPOUT_AUDIO_GET_LEVELS_RESP_INFO    LcLPtrAudio ;
    WORD                                LcRet;
    WORD                                LcNbAudio;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPOUT_AUDIO_GET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrAudio     = (LPOUT_AUDIO_GET_LEVELS_RESP_INFO)(LcLPtrRepHeader + 1);

    LcNbAudio = 0 ;

    // Actually process the request
    // ----------------------------
    LcRet = GetOutAudioLevelsSubFct(
                        LcLPtrReqHeader->hdHandle,
                        LcLPtrReq->olqOutputPipeMask64.QuadPart,
                        LcLPtrReq->olqAudioMask,
                        LcLPtrAudio,
                        &LcNbAudio );

    // ## FS (11/12/97) -- bug fix for FA #91
    // the application may send a bad pipe audio mask that does not
    // match any ouput. In this case, we should better report an
    // error
    // ------------------------------------------------------------
    if ( LcNbAudio == 0 )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
    }

    // update size of reply block
    // --------------------------
    LcLPtrRepHeader->rhSize = LcNbAudio * sizeof(OUT_AUDIO_GET_LEVELS_RESP_INFO);
    LcLPtrRepHeader->rhCptr = LcRet ;

    return;
}

// *************************************************************************
//
// STATIC VOID GetInAudioLevelsSubFct( IN BYTE, IN DWORDLONG, IN WORD,
//                                       OUT LPIN_AUDIO_GET_LEVELS_RESP_INFO,
//                                       OUT PWORD )
//
// Input Parameters:
// *****************
//
//  BYTE    PmAppHandle:    Application handle
//  DWORDLONG PmPipeMask:   Mask of input pipes
//  WORD    PmAudioMask:    Mask of pipe audios
//
// Output Paramaters:
// ******************
//
//  LPIN_AUDIO_GET_LEVELS_RESP_INFO PmLPtrAudio:    Audio levels descriptors
//  PWORD                           PmNbAudioPtr:   Number of descriptors
//                                                   filled in
//
// Return Value:
// *************
//
// SUCCESS if the levels have been retrieved successfully, an error
// code otherwise
//
// *************************************************************************
STATIC WORD GetInAudioLevelsSubFct(
    IN  BYTE                                PmAppHandle,
    IN  DWORDLONG                           PmPipeMask,
    IN  WORD                                PmAudioMask,
    OUT LPIN_AUDIO_GET_LEVELS_RESP_INFO     PmLPtrAudio,
    OUT PWORD                               PmNbAudioPtr )
{
    TARGET_INFO                     LcTargetInf;
    LEVEL_AUDIO_INFO                 LcLevelAudioInf[MAX_AUDIO_PIPE];
    PLEVEL_AUDIO_INFO                LcPtrLevelAudioInf;
    WORD                            i, LcRet ;
    WORD                            LcAudioMask ;
    WORD                            LcOkAudioMask = 0;
    WORD                            LcNbAudio = 0;
    BYTE                            LcAppIndex;
    WORD                            LcInPipeIndex;
    PPIPE_INFO                      LcPipeInfPtr;
    PVOIE_INFO                      LcVoieInfPtr;
    LPIN_AUDIO_GET_LEVELS_RESP_INFO LcLPtrAudio ;

    // Initialize output params
    // ------------------------
    *PmNbAudioPtr = 0;

    LcAppIndex = DecodeHandle( PmAppHandle );

    if( LcAppIndex == DHSAppHandle )
    {
        return DHSGetInAudioLevels(
                        PmPipeMask,
                        PmAudioMask,
                        PmLPtrAudio,
                        PmNbAudioPtr);
    }

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneInputPipeMask(  LcAppIndex,
                                    PmPipeMask,
                                    &LcInPipeIndex,
                                    &LcPipeInfPtr,
                                    &LcVoieInfPtr,
                                    &LcTargetInf );

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        return LcRet;
    }

    // Let's scan for audio allocated by this pipe and
    // apply the set level command if in the scope of request mask
    // -------------------------------------------------------------
    LcAudioMask = PmAudioMask;
    LcLPtrAudio = PmLPtrAudio;
    LcNbAudio = 0;

    for ( i = 0 ; i < LcPipeInfPtr->piNbAudio ; i++ )
    {
        if ( LcAudioMask == 0 ) break ;

        // Get levels only if audio is physical, owned by the appli and
        // the pipe, and match the given mask
        // -------------------------------------------------------------
        if (  ( LcVoieInfPtr[i].Nat == 0 )
            && ( UTIMaskWord( LcVoieInfPtr[i].AppAudioNum ) & LcAudioMask  ) )
        {
            ASSERT ( LcVoieInfPtr[i].IndexAppli != 0 );
            ASSERT ( LcVoieInfPtr[i].NumPipe == LcInPipeIndex );

            // Get analog level
            // ----------------
            // Get analog/micro input level information
            // ----------------------------------------
            if( CUR_COMMANDS_PTR->PIOGetInAudioLevels(  LcVoieInfPtr[i].AudioNum,
                                                        &LcLPtrAudio->ilpAnalogLevel,
                                                        &LcLPtrAudio->ilpAnalogMute,
                                                        &LcLPtrAudio->ilpMicroLevel,
                                                        &LcLPtrAudio->ilpMicroMute) != SUCCESS )
            {
                LcLPtrAudio->ilpAnalogLevel = 0;
                LcLPtrAudio->ilpAnalogMute  = 0;
                LcLPtrAudio->ilpMicroLevel  = 0;
                LcLPtrAudio->ilpMicroMute   = 0;
            }

            LcLPtrAudio++;

            LcAudioMask &= ~UTIMaskWord( LcVoieInfPtr[i].AppAudioNum ) ; // BUG FH

            // Build a audio Mask with the autorised audio IO
            //-----------------------------------------------
            LcOkAudioMask |= UTIMask64bit( LcVoieInfPtr[i].AudioNum );
            LcNbAudio++;

        }  // if this audio is concerned

    } // for all audio inputs

    LcPtrLevelAudioInf = &LcLevelAudioInf[0];

    LcRet = CUR_PROTOCOL_PTR->ILevel_AudioGetLevels(	TRUE, // LcIsInput
														LcOkAudioMask,
														LcNbAudio,
														LcPtrLevelAudioInf );
    if ( LcRet == SUCCESS )
    {
        // Fill response block
        // -------------------
        *PmNbAudioPtr = LcNbAudio;
        for ( i=0, LcLPtrAudio = PmLPtrAudio; i<LcNbAudio; i++, LcPtrLevelAudioInf++, LcLPtrAudio++ )
        {
            // Get digital level
            // ------------------
            LcLPtrAudio->ilpDigitalLevel = LcPtrLevelAudioInf->gaiDigitalLevel;
        }

        *PmNbAudioPtr = LcNbAudio;
    }

    return LcRet;
}

// *************************************************************************
//
// STATIC VOID GetInAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get all levels for output audio
//
//  FH creation
//
// *************************************************************************
STATIC VOID GetInAudioLevelsFct(
    IN LPDWORD  PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_GET_LEVELS_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO                LcLPtrReqHeader;
    LPRESP_HEADER_INFO              LcLPtrRepHeader;
    LPIN_AUDIO_GET_LEVELS_RESP_INFO LcLPtrAudio ;
    WORD                            LcRet;
    WORD                            LcNbAudio;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPIN_AUDIO_GET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrAudio     = (LPIN_AUDIO_GET_LEVELS_RESP_INFO)(LcLPtrRepHeader + 1);

    LcNbAudio = 0;

    // Actually process the request
    // ----------------------------
    LcRet = GetInAudioLevelsSubFct(
                        LcLPtrReqHeader->hdHandle,
                        LcLPtrReq->ilqInPipeMask64.QuadPart,
                        LcLPtrReq->ilqAudioMask,
                        LcLPtrAudio,
                        &LcNbAudio );

    // update size of reply block
    // --------------------------
    LcLPtrRepHeader->rhSize = LcNbAudio * sizeof(IN_AUDIO_GET_LEVELS_RESP_INFO);
    LcLPtrRepHeader->rhCptr = LcRet ;

    // ## FS (11/12/97) -- bug fix for FA #91
    // the application may send a bad pipe audio mask that does not
    // match any ouput. In this case, we should better report an
    // error
    // ------------------------------------------------------------
    if ( LcNbAudio == 0 )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
    }

    return;
}


// *************************************************************************
//
// STATIC VOID GetAudioUerSubFct( IN BYTE, IN DWORD, IN WORD,
//                                 OUT LPUER_RESP_INFO, OUT PWORD )
//
// Input Parameters:
// *****************
//
//  BYTE    PmAppHandle:    Application handle
//  DWORD   PmInPipeMask:   Mask of input pipes
//  WORD    PmAudioMask:    Mask of pipe audios
//
// Output Paramaters:
// ******************
//
//  LPUER_RESP_INFO     PmUerDescPtr:   Audio UER descriptors
//  PWORD               PmNbAudioPtr:   Number of descriptors filled in
//
// Return Value:
// *************
//
// SUCCESS if the Uer registers have been retrieved successfully, an error
// code otherwise
//
// *************************************************************************
STATIC WORD GetAudioUerSubFct(
    IN  BYTE            PmAppHandle,
    IN  DWORDLONG       PmPipeMask,
    IN  WORD            PmAudioMask,
    OUT LPUER_RESP_INFO PmUerDescPtr,
    OUT PWORD           PmNbAudioPtr )
{
    BYTE                LcAppIndex      ;
    WORD                j, LcError      ;
    WORD                LcInPipeIndex   ;
    DWORD               LcInAudioMask   ;
    WORD                LcInAudio, LcInAudioIndex;
    BOOLEAN             LcFound         ;
    TARGET_INFO         LcTarget        ;
    BYTE                LcData, LcMode  ;
    PPIPE_INFO          LcPipeInfPtr    ;
    PVOIE_INFO          LcVoieInfPtr    ;
    WORD                LcNbAudio       = 0 ;

    LcAppIndex = DecodeHandle( PmAppHandle );

    if(LcAppIndex == DHSAppHandle)
    {
        return DHSGetAudioUer(  PmPipeMask,
                                PmAudioMask,
                                PmUerDescPtr,
                                PmNbAudioPtr );
    }

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcError = ChkPmOneInputPipeMask(    LcAppIndex      ,
                                        PmPipeMask    ,
                                        &LcInPipeIndex  ,
                                        &LcPipeInfPtr    ,
                                        &LcVoieInfPtr    ,
                                        &LcTarget);

    if ( LcError != SUCCESS )
    {
        return LcError;
    }

    // retrieve in audio mask
    // ************************
    LcInAudioMask = (DWORD) PmAudioMask ;

    // verify this mask specify at least one audio
    // **********************************************
    if ( LcInAudioMask == 0 )
    {
        EXIT_PCX_ERROR( ED_INVALID_PIPE_AUDIO );
    }

    LcNbAudio = 0;

    // For all audios specified
    // *************************
    while ( ( LcInAudioMask != 0 ) && ( LcError == SUCCESS ) )
    {
        LcInAudio = UTIDWMask2Word( LcInAudioMask ) ;

        // FS - (18/04/97) - infinite loop ending into WD_REPLY_TRUNCATED
        LcInAudioMask &= ~UTIMaskDWord( LcInAudio ) ;

        // does the audio actually belongs to the pipe ?
        // **********************************************
        // FS - (18/04/97) - actual upper limit is MAX_AUDIO_PIPE
        if ( LcInAudio >= MAX_BOARD_AUDIO )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // seek for this audio
        // ********************

        // FH BUG CORRECTION 12/03/97
        LcFound = FALSE ;
        for ( j = 0 ; (j < LcPipeInfPtr->piNbAudio) && !LcFound ; j ++ )
        {
            ASSERT( LcVoieInfPtr[j].NumPipe == LcInPipeIndex );
            ASSERT( LcVoieInfPtr[j].IndexAppli != 0 );

            if ( LcVoieInfPtr[j].AppAudioNum == LcInAudio )
            {
                LcFound = TRUE;
                LcInAudioIndex = j;
            }
        }

        if ( !LcFound )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        WORD	LcCurrentBoard;
		DWORD	LcCBitFrequ;
		BYTE    LcUerExtra;
        
        // Prepare a target to ask for this information
        // ********************************************
        LcCurrentBoard     = LcTarget.tgCarte;
        LcTarget.tgAudio   = LcVoieInfPtr[LcInAudioIndex].AudioNum ;

        // Retrieve the UER information
        // ****************************
        LcError = CUR_COMMANDS_PTR->PIOGetAudioUer(LcTarget.tgAudio, &LcData, &LcMode, &LcCBitFrequ, &LcUerExtra );

        // Fill in the reply bloc
        // **********************
        PmUerDescPtr->urpData      = LcData ;
        PmUerDescPtr->urpMode      = LcMode ;
		PmUerDescPtr->urpExtra     = LcUerExtra;
        PmUerDescPtr->urpFrequency = LcCBitFrequ;

        // Set position to next descriptor
        // *******************************
        PmUerDescPtr++ ;
        LcNbAudio++ ;
    }
    // For all audios specified
    // *************************

    if ( LcError == SUCCESS )
    {
        *PmNbAudioPtr = LcNbAudio;
    }

    return LcError;
}

// *************************************************************************
//
// STATIC VOID GetAudioUerFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get UER register
//
// *************************************************************************
STATIC VOID GetAudioUerFct(
    IN LPDWORD  PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPBC_HEADER_INFO    LcReqHeaderPtr  ;
    LPUER_REQ_INFO      LcRequestPtr    ;
    LPRESP_HEADER_INFO  LcRepHeaderPtr  ;
    LPUER_RESP_INFO     LcUerDescPtr    ;
    WORD                LcNbAudio       = 0 ;
    WORD                LcRet;

    // Initialisation of pointers
    LcReqHeaderPtr  = (LPBC_HEADER_INFO) PmBlocRequete ;
    LcRequestPtr    = (LPUER_REQ_INFO) PmBlocRequete ;
    LcRepHeaderPtr  = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LcUerDescPtr    = (LPUER_RESP_INFO)( LcRepHeaderPtr + 1 );


    // Actually process the request
    // ----------------------------
    LcRet = GetAudioUerSubFct(
                        LcReqHeaderPtr->hdHandle,
                        LcRequestPtr->urqInputPipeMask64.QuadPart,
                        LcRequestPtr->urqAudioMask,
                        LcUerDescPtr,
                        &LcNbAudio );

    // ## FS (11/12/97) -- bug fix for FA #91
    // the application may send a bad pipe audio mask that does not
    // match any ouput. In this case, we should better report an
    // error
    // ------------------------------------------------------------
    if ( LcNbAudio == 0 )
    {
        LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
    }

    // update size of reply block
    // --------------------------
    LcRepHeaderPtr->rhSize = LcNbAudio * sizeof(UER_RESP_INFO);
    LcRepHeaderPtr->rhCptr = LcRet ;

    LcRepHeaderPtr->rhCptr = LcRet ;
}


// *************************************************************************
//                  Stream Status subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd8( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Stream Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Stream Status command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd8(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM8 );
}


// *************************************************************************
//
// STATIC VOID StreamVPMeter( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the Stream Vu-Meter command.
//
// *************************************************************************
//
//  FH Creation
//
// *************************************************************************
STATIC VOID StreamVPMeter(
                            IN  LPDWORD PmBlocRequete,
                            OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPSTREAM_VU_METER_REQ_INFO  LcLPtrReqHeader;
    LPSTREAM_VU_METER_RESP_INFO LcLPtrStreamVuPicMeterInf;
    TARGET_INFO                 LcTargetInf;
    WORD                        LcNbStream = 0, LcPipeIndex = 0;
    WORD                        i, LcRet ;
    BYTE                        LcAppIndex;
    DWORD                       LcStreamMask;
    DWORDLONG                   LcAudioMask;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;
    WORD                        LcNbPipeChannel = 0;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrStreamVuPicMeterInf = (LPSTREAM_VU_METER_RESP_INFO)( LcLPtrRepHeader + 1 );

    LcLPtrReqHeader = (LPSTREAM_VU_METER_REQ_INFO)PmBlocRequete;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->Header.hdHandle );

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneOutputPipeMask( LcAppIndex,
                                    LcLPtrReqHeader->svqOutPipeMask64.QuadPart,
                                    &LcPipeIndex,
                                    &LcPipeInfPtr,
                                    &LcVoieInfPtr,
                                    &LcTargetInf);

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return ;
    }

    // Retrieve and verify pipe audios mask
    // ------------------------------------------
    LcAudioMask =  LcLPtrReqHeader->svqAudioMask & LcPipeInfPtr->piPipeAudioMask ;
    if ( LcAudioMask != LcLPtrReqHeader->svqAudioMask )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_PIPE_AUDIO );
        return;
    }

    while (LcAudioMask)
    {
        if (LcAudioMask & 0x1) LcNbPipeChannel++;
        LcAudioMask >>=1;
    }

    // Retrieve and verify stream mask
    // ------------------------------------------
    LcStreamMask =  LcLPtrReqHeader->svqStreamMask & LcPipeInfPtr->piStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Filter the effective streams
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = LcStreamMask;

    // counting the number of streams
    // -------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbPipeChannel*LcNbStream*sizeof(LPSTREAM_VU_METER_RESP_INFO)) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbPipeChannel*LcNbStream*sizeof(STREAM_VU_METER_RESP_INFO) );

    // Send a request to the DSP
    // --------------------------
    LcRet = CUR_PROTOCOL_PTR->ILevel_StreamGetVuMeter(	&LcTargetInf,
														LcNbStream * LcNbPipeChannel,
                                                        LcLPtrReqHeader->svqAudioMask,
														LcLPtrStreamVuPicMeterInf );

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID StreamStateFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the Stream Status command.
//
// *************************************************************************
//
//  FH Creation
//
// *************************************************************************
STATIC VOID StreamStateFct(
                            IN  LPDWORD PmBlocRequete,
                            OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPSTREAM_STATE_REQ_INFO     LcLPtrReqHeader;
    LPSTREAM_STATE_RESP_INFO    LcLPtrStreamInf;
    TARGET_INFO                 LcTargetInf;
    STATE_STREAM_INFO           LcStateStreamInf[MAX_STREAM_PIPE];
    PSTATE_STREAM_INFO          LcPtrStateStreamInf;
    WORD                        LcNbStream = 0, LcPipeIndex = 0;
    WORD                        i, LcRet ;
    BYTE                        LcAppIndex;
    DWORD                       LcStreamMask;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrStreamInf = (LPSTREAM_STATE_RESP_INFO)( LcLPtrRepHeader + 1 );

    LcLPtrReqHeader = (LPSTREAM_STATE_REQ_INFO)PmBlocRequete;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->Header.hdHandle );

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcLPtrReqHeader->ssqOutPipeMask64.QuadPart,
                                        LcLPtrReqHeader->ssqInPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPipeInfPtr,
                                        &LcVoieInfPtr,
                                        &LcTargetInf);

	WORD LcCurrentBoard = LcTargetInf.tgCarte;


    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return ;
    }

    // Retrieve and verify stream mask
    // -------------------------------
    // ## FS (08/08/1997) - correct silent record
    // Filter the effective streams
    // ------------------------------------------
    LcStreamMask =  LcLPtrReqHeader->ssqStreamMask
                  & LcPipeInfPtr->piStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Fill TARGET_INFO structure
    // --------------------------

    // ## FS (02/07/1997) - correct silent record
    // Filter the effective streams
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = LcStreamMask;

    // counting the number of streams
    // -------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbStream*sizeof(LPSTREAM_STATE_RESP_INFO)) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbStream*sizeof(STREAM_STATE_RESP_INFO) );

    // Send a request to the DSP
    // --------------------------
    LcPtrStateStreamInf = &LcStateStreamInf[0];
    
	LcRet = CUR_PROTOCOL_PTR->IFlow_StreamGetState(	&LcTargetInf,
													LcNbStream,
													LcPtrStateStreamInf );

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Fill response block
    // -------------------
    for ( i=0; i<LcNbStream; i++, LcLPtrStreamInf++, LcPtrStateStreamInf++ )
    {
        LcLPtrStreamInf->sspStart = LcPtrStateStreamInf->Start;
        LcLPtrStreamInf->sspHour  = LcPtrStateStreamInf->Hour;
    }

    LcLPtrRepHeader->rhCptr = SUCCESS ;
    return;
}

// *************************************************************************
//
// STATIC VOID StreamSampleOrByteFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the Stream Sample or Byte Count command.
//
// *************************************************************************
//
//  FH Creation
//
// *************************************************************************
STATIC VOID StreamSampleOrByteFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour,
	IN	BOOLEAN PmIsByte			
	)
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPSTREAM_SAMPLE_REQ_INFO    LcLPtrReqHeader;
    LPSTREAM_SAMPLE_RESP_INFO   LcLPtrStreamInf;
    TARGET_INFO                 LcTargetInf;
    PCX_TIME                    LcSampleStreamInf[MAX_STREAM_PIPE];
    PPCX_TIME                   LcPtrSampleStreamInf;
    WORD                        LcNbStream = 0, LcPipeIndex = 0;
    WORD                        i, LcRet ;
    BYTE                        LcAppIndex;
    DWORD                       LcStreamMask;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrStreamInf = (LPSTREAM_SAMPLE_RESP_INFO)( LcLPtrRepHeader + 1 );

    LcLPtrReqHeader = (LPSTREAM_SAMPLE_REQ_INFO)PmBlocRequete;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->Header.hdHandle );

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcLPtrReqHeader->ssqOutPipeMask64.QuadPart,
                                        LcLPtrReqHeader->ssqInPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPipeInfPtr,
                                        &LcVoieInfPtr,
                                        &LcTargetInf);

	WORD LcCurrentBoard = LcTargetInf.tgCarte;


    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return ;
    }

    // Retrieve and verify stream mask
    // Filter the effective streams
    // ------------------------------------------
    LcStreamMask =  LcLPtrReqHeader->ssqStreamMask
                  & LcPipeInfPtr->piStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Fill TARGET_INFO structure
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = LcStreamMask;

    // counting the number of streams
    // -------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbStream*sizeof(STREAM_SAMPLE_RESP_INFO)) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbStream*sizeof(STREAM_SAMPLE_RESP_INFO) );

    // Send a request to the DSP
    // --------------------------
	
    // MBR (2003/05/23) it's the driver that counts bytes for playback now
    if( (PmIsByte == FALSE) || (LcTargetInf.tgCaracPipeVoie != OPER_PLAY) )
    {
		// check if the PCXTIME shall be a bytecount instead: 
		LcPtrSampleStreamInf = &LcSampleStreamInf[0];
		if ( PmIsByte == TRUE ) *LcPtrSampleStreamInf = 123;

		LcRet = CUR_PROTOCOL_PTR->IFlow_StreamSampleCount(	&LcTargetInf,
															LcNbStream,
															LcPtrSampleStreamInf );
		if ( LcRet != SUCCESS )
		{
			LcLPtrRepHeader->rhCptr = LcRet ;
			return;
		}

		// Fill response block
		// -------------------
		for ( i=0; i<LcNbStream; i++, LcLPtrStreamInf++, LcPtrSampleStreamInf++ )
		{
			LcLPtrStreamInf->sspHour = *LcPtrSampleStreamInf;
		}
    }
    else
    {
        LcStreamMask = LcTargetInf.tgMaskFlux;

        // Fill response block
        // -------------------
        for ( i=0; i<MAX_STREAM_PIPE; i++ )
        {
            if( (LcStreamMask >> i) & 1 )
            {
                LcLPtrStreamInf->sspHour = TbOutStreamBytePos[LcPipeIndex][i];
                LcLPtrStreamInf++;
            }
        }
    }

    LcLPtrRepHeader->rhCptr = SUCCESS ;
    return;
};


STATIC VOID StreamSampleFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    StreamSampleOrByteFct( PmBlocRequete, PmBlocRetour, FALSE );
	return;
};

STATIC VOID StreamByteFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    StreamSampleOrByteFct( PmBlocRequete, PmBlocRetour, TRUE );
	return;
};

// *************************************************************************
//
// STATIC VOID StreamPositionFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//  Process the Stream Playback position.
// *************************************************************************
STATIC VOID StreamPositionFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour
	)
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPSTREAM_SAMPLE_REQ_INFO    LcLPtrReqHeader;
    LPSTREAM_SAMPLE_RESP_INFO   LcLPtrStreamInf;
    TARGET_INFO                 LcTargetInf;
    PCX_TIME                    LcSampleStreamInf[MAX_STREAM_PIPE];
    PPCX_TIME                   LcPtrSampleStreamInf;
    WORD                        LcNbStream = 0, LcPipeIndex = 0;
    WORD                        i, LcRet ;
    BYTE                        LcAppIndex;
    DWORD                       LcStreamMask;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrStreamInf = (LPSTREAM_SAMPLE_RESP_INFO)( LcLPtrRepHeader + 1 );
    LcLPtrReqHeader = (LPSTREAM_SAMPLE_REQ_INFO)PmBlocRequete;
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->Header.hdHandle );

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcLPtrReqHeader->ssqOutPipeMask64.QuadPart,
                                        LcLPtrReqHeader->ssqInPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPipeInfPtr,
                                        &LcVoieInfPtr,
                                        &LcTargetInf);

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return ;
    }

    // Retrieve and verify stream mask
    // Filter the effective streams
    // ------------------------------------------
    LcStreamMask =  LcLPtrReqHeader->ssqStreamMask
                  & LcPipeInfPtr->piStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Fill TARGET_INFO structure
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = LcStreamMask;

    // counting the number of streams
    // -------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbStream*sizeof(STREAM_SAMPLE_RESP_INFO)) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbStream*sizeof(STREAM_SAMPLE_RESP_INFO) );

    // Send a request to the DSP
    // --------------------------
	
	// check if the PCXTIME shall be a bytecount instead: 
	LcPtrSampleStreamInf = &LcSampleStreamInf[0];

	LcRet = CUR_PROTOCOL_PTR->IFlow_StreamSamplePosition(	&LcTargetInf,
														LcNbStream,
														LcPtrSampleStreamInf );
	if ( LcRet != SUCCESS )
	{
		LcLPtrRepHeader->rhCptr = LcRet ;
		return;
	}

	// Fill response block
	// -------------------
	for ( i=0; i<LcNbStream; i++, LcLPtrStreamInf++, LcPtrSampleStreamInf++ )
	{
		LcLPtrStreamInf->sspHour = *LcPtrSampleStreamInf;
	}

    LcLPtrRepHeader->rhCptr = SUCCESS ;
    return;
}; // StreamPositionFct



// *************************************************************************
//
// STATIC VOID StreamLevelsFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the get out Stream levels command.
//
// *************************************************************************
STATIC VOID StreamLevelsFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    LPGET_OUT_STREAM_LEVELS_REQ_INFO    LcLPtrReqHeader;
    LPGET_OUT_STREAM_LEVELS_RESP_INFO   LcLPtrStreamInf;

    STATIC LEVEL_STREAM_INFO    LcLevelInf[MAX_STREAM_PIPE];
    TARGET_INFO                 LcTargetInf;
    PPIPE_INFO                  LcPipeInfPtr;
    PVOIE_INFO                  LcVoieInfPtr;

    DWORD                       LcStreamMask;
    WORD                        LcNbStream = 0, LcOutPipeIndex = 0;
    WORD                        i, LcRet ;
    WORD                        LcCurrentBoard;
    BYTE                        LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrStreamInf = (LPGET_OUT_STREAM_LEVELS_RESP_INFO)( LcLPtrRepHeader + 1 );

    LcLPtrReqHeader = (LPGET_OUT_STREAM_LEVELS_REQ_INFO)PmBlocRequete;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->Header.hdHandle );

    // verify this mask specify one pipe ...
    // ... and only one pipe
    // and check the pipe actually belongs to the application
    // ******************************************************
    LcRet = ChkPmOneOutputPipeMask(    LcAppIndex,
                                       LcLPtrReqHeader->oslqOutPipeMask64.QuadPart,
                                       &LcOutPipeIndex,
                                       &LcPipeInfPtr,
                                       &LcVoieInfPtr,
                                       &LcTargetInf);

	LcCurrentBoard = LcTargetInf.tgCarte;


    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return ;
    }

    // Retrieve and verify stream mask
    // Filter the effective streams
    // ------------------------------------------
    LcStreamMask =  LcLPtrReqHeader->oslqStreamMask
                  & LcPipeInfPtr->piStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Fill TARGET_INFO structure
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = LcStreamMask;

    // counting the number of streams
    // -------------------------------
    while (LcStreamMask)
    {
        LcNbStream += (WORD) (LcStreamMask & 0x1);
        LcStreamMask >>= 1 ;
    }

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->Header.hdBrSize <
         (sizeof(RESP_HEADER_INFO) + LcNbStream*sizeof(GET_OUT_STREAM_LEVELS_RESP_INFO)) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Process the response block size
    // -------------------------------
    LcLPtrRepHeader->rhSize = ( LcNbStream*sizeof(GET_OUT_STREAM_LEVELS_RESP_INFO) );

    // Send a request to the DSP
    // --------------------------
    LcRet = CUR_PROTOCOL_PTR->ILevel_StreamOutGetLevels(	&LcTargetInf,
															LcNbStream,
															LcLevelInf );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Fill response block
    // -------------------
    for ( i = 0 ; i < LcNbStream ; i++ )
    {
        LcLPtrStreamInf->oslrLeftChannelToOutput1DigitalLevel =
            LcLevelInf[i].sliPanLevelCgV1;
        LcLPtrStreamInf->oslrLeftChannelToOutput2DigitalLevel =
            LcLevelInf[i].sliPanLevelCgV2;
        LcLPtrStreamInf->oslrRightChannelToOutput1DigitalLevel =
            LcLevelInf[i].sliPanLevelCdV1;
        LcLPtrStreamInf->oslrRightChannelToOutput2DigitalLevel =
            LcLevelInf[i].sliPanLevelCdV2;
        LcLPtrStreamInf->oslrDigitalLevel1 = LcLevelInf[i].sliLevelStream1;
        LcLPtrStreamInf->oslrDigitalLevel2 = LcLevelInf[i].sliLevelStream2;
        LcLPtrStreamInf->oslrMute1 = LcLevelInf[i].sliMuteStream1;
        LcLPtrStreamInf->oslrMute2 = LcLevelInf[i].sliMuteStream2;

        LcLPtrStreamInf++;
    }

    LcLPtrRepHeader->rhCptr = SUCCESS ;
    return;
}

// *************************************************************************
//
// STATIC VOID GetOutStreamEffects( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//
// *************************************************************************
//
//  Process the command: get the effects parameters on output stream.
//
// *************************************************************************
//
STATIC VOID GetOutStreamEffects(
                            IN  LPDWORD PmBlocRequete   ,
                            OUT LPDWORD PmBlocRetour    )
{
    LPGET_OUT_STREAM_EFFECTS_REQ_INFO   LcLPtrReq;
    LPBC_HEADER_INFO                    LcLPtrReqHeader;
    LPGET_OUT_STREAM_EFFECTS_RESP_INFO  LcLPtrRep;
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    TARGET_INFO                         LcTargetInf;
    BYTE                                LcAppIndex ;
    WORD                                LcRet ;
    WORD                                LcPipeIndex ;
    DWORDLONG                           LcPipeMask;
    WORD                                LcStreamIndex ;
    DWORD                               LcStreamMask;
	WORD								LcCurrentBoard = 0;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPGET_OUT_STREAM_EFFECTS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRep       = (LPGET_OUT_STREAM_EFFECTS_RESP_INFO)PmBlocRetour;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof( GET_OUT_STREAM_EFFECTS_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =   sizeof( GET_OUT_STREAM_EFFECTS_RESP_INFO )
                              - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );
    LcPipeMask = LcLPtrReq->gseqOutPipeMask64.QuadPart;

    // Verify pipes specified
    // -----------------------
    LcRet = ChkPmManyOutputPipeMask( LcAppIndex,
                                     LcPipeMask );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Verify that only one pipe is specified
    // --------------------------------------
    LcPipeIndex = UTI64Mask2Word( LcPipeMask ) ;
    LcPipeMask &= ~UTIMask64bit( LcPipeIndex );
    if ( LcPipeMask != 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_PIPE );
        return;
    }

    // Fill target_info
    // ----------------
    LcTargetInf.tgCaracPipeVoie = OPER_PLAY;
    LcTargetInf.tgDifferedType  = 0;
    LcTargetInf.tgHour          = 0;
    LcTargetInf.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
    LcTargetInf.tgPipe          = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;

    // ## FS (02/07/1997) - correct silent record
    // Filter the effective streams
    // ------------------------------------------
    LcTargetInf.tgMaskFlux = (  LcLPtrReq->gseqStreamMask
                              & TbPipeOutInfo[LcPipeIndex].piStreamMask );

    // If the specified streams do not exist
    // give up and return an error
    // -------------------------------------
    if ( LcTargetInf.tgMaskFlux == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Verify that only one stream is specified
    // ----------------------------------------
    LcStreamMask  = LcTargetInf.tgMaskFlux;
    LcStreamIndex = UTIDWMask2Word( LcStreamMask ) ;
    LcStreamMask &= ~UTIMaskDWord( LcStreamIndex );
    if ( LcStreamMask != 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    LcRet = CUR_PROTOCOL_PTR->IFlow_StreamOutGetEffects(	&LcTargetInf,
															LcLPtrReq->gseqEffectId,															
															LcLPtrRep->gserParam );
    LcLPtrRepHeader->rhCptr = LcRet;
}


// *************************************************************************
//                  Board Audio Status subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd12( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Board Audio Status command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Board Audio Status command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd12(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM12 );
}


// *************************************************************************
//
// STATIC VOID GetInAudioCByteFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
STATIC VOID GetInAudioCByteFct(  IN LPDWORD  PmBlocRequete,
                                OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_GET_CBYTE_REQ_INFO	LcReq;
    LPIN_AUDIO_GET_CBYTE_RESP_INFO	LcResp;
    LPBC_HEADER_INFO                LcReqHeader;
    LPRESP_HEADER_INFO              LcRespHeader;
	WORD							LcCurrentBoard;
	WORD							LcRet;

    // Initialisation of pointers
    // ---------------------------
    LcReq        = (LPIN_AUDIO_GET_CBYTE_REQ_INFO)PmBlocRequete;
    LcReqHeader  = (LPBC_HEADER_INFO)PmBlocRequete;
    LcResp       = (LPIN_AUDIO_GET_CBYTE_RESP_INFO)PmBlocRetour;
    LcRespHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcReqHeader->hdBrSize < sizeof( *LcResp ) )
    {
        LOAD_PCX_ERROR( LcRespHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcRespHeader->rhSize = sizeof( *LcResp ) - sizeof( *LcRespHeader );

	LcCurrentBoard = LcReq->iacqAudioInf.paBoard;

	if( CUR_COMMANDS_PTR == NULL )
	{
        LOAD_PCX_ERROR( LcRespHeader->rhCptr, ED_INVALID_BOARD );
        return;
	}

	LcRet = CUR_COMMANDS_PTR->PIOGetAudioCByte(	LcReq->iacqAudioInf.paAudioNum,
												LcReq->iacqByteID,
												&LcResp->iacpValidity,
												&LcResp->iacpByteValue);
    LcRespHeader->rhCptr = LcRet ;
    return;
}

// *************************************************************************
//
// STATIC VOID BoardAudioVPMeterFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
STATIC VOID BoardAudioVPMeterFct(    IN LPDWORD  PmBlocRequete,
                                OUT LPDWORD PmBlocRetour )
{
    LPBOARD_AUDIO_VUMETER_RESP_INFO LcLPtrRep;
    LPRESP_HEADER_INFO              LcLPtrRepHeader;
    LPBOARD_AUDIO_VUMETER_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO                LcLPtrReqHeader;
    PVOIE_INFO                      LcPtVoie;
    DWORD                           LcOutPipeMask;
    DWORD                           LcInPipeMask;
    WORD                            LcRet = SUCCESS;
    WORD                            LcNbAudio = 0 ;
    BYTE                            LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRep = (LPBOARD_AUDIO_VUMETER_RESP_INFO)PmBlocRetour;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrReq = (LPBOARD_AUDIO_VUMETER_REQ_INFO)PmBlocRequete;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof(BOARD_AUDIO_VUMETER_RESP_INFO) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =  sizeof(BOARD_AUDIO_VUMETER_RESP_INFO)
                             - sizeof(RESP_HEADER_INFO);

    // Verify pipe specified
    // Check if it is an Input or Output command
    // and retrieve the pipe index
    //------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->bavqAudioInf).paBoard,
                            (LcLPtrReq->bavqAudioInf).paDsp,
                            (LcLPtrReq->bavqAudioInf).paAudioNum,
                            (LcLPtrReq->bavqAudioInf).paAudioAttributes,
                            &LcPtVoie );

    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                if ( (LcLPtrReq->bavqAudioInf).paAudioAttributes & AUDIO_IN )
                {
                    LcOutPipeMask = 0;
                    LcInPipeMask = UTIMaskDWord( LcPtVoie->NumPipe );
                }
                else
                {
                    LcOutPipeMask = UTIMaskDWord( LcPtVoie->NumPipe );
                    LcInPipeMask = 0;
                }

                // Actually process the request
                // ----------------------------
                LcRet = GetAudioVuMeterSubFct( LcLPtrReqHeader->hdHandle,
                                               LcOutPipeMask,
                                               LcInPipeMask,
                                               UTIMaskWord( LcPtVoie->AppAudioNum ),
                                               &(LcLPtrRep->bavrVuMeter),
                                               &LcNbAudio );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID GetOutBoardAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get levels for an output audio
//
// *************************************************************************
STATIC VOID GetOutBoardAudioLevelsFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPOUT_BOARD_AUDIO_GET_LEVELS_REQ_INFO   LcLPtrReq;
    LPBC_HEADER_INFO                        LcLPtrReqHeader;
    LPRESP_HEADER_INFO                      LcLPtrRepHeader;
    LPOUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO  LcLPtrRep ;
    PVOIE_INFO                              LcPtVoie;
    WORD                                    LcRet ;
    WORD                                    LcNbAudio;
    BYTE                                    LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPOUT_BOARD_AUDIO_GET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRep       = (LPOUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO)PmBlocRetour;


    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof(OUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =  sizeof(OUT_BOARD_AUDIO_GET_LEVELS_RESP_INFO)
                             - sizeof(RESP_HEADER_INFO);

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Verify pipe specified
    // Check if it is an Input or Output command
    // and retrieve the pipe index
    //------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->oblqAudioInf).paBoard,
                            (LcLPtrReq->oblqAudioInf).paDsp,
                            (LcLPtrReq->oblqAudioInf).paAudioNum,
                            (LcLPtrReq->oblqAudioInf).paAudioAttributes,
                            &LcPtVoie );


    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                // Actually process the request
                // ----------------------------
                LcRet = GetOutAudioLevelsSubFct( LcLPtrReqHeader->hdHandle,
                                                 UTIMask64bit( LcPtVoie->NumPipe ),
                                                 UTIMaskWord( LcPtVoie->AppAudioNum ),
                                                 &(LcLPtrRep->oblrLevels),
                                                 &LcNbAudio );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID GetInBoardAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get all levels for a board output audio
//
// *************************************************************************
STATIC VOID GetInBoardAudioLevelsFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_BOARD_AUDIO_GET_LEVELS_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO                        LcLPtrReqHeader;
    LPRESP_HEADER_INFO                      LcLPtrRepHeader;
    LPIN_BOARD_AUDIO_GET_LEVELS_RESP_INFO   LcLPtrRep ;
    PVOIE_INFO                              LcPtVoie;
    WORD                                    LcRet ;
    WORD                                    LcNbAudio;
    BYTE                                    LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPIN_BOARD_AUDIO_GET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRep       = (LPIN_BOARD_AUDIO_GET_LEVELS_RESP_INFO)PmBlocRetour;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof(IN_BOARD_AUDIO_GET_LEVELS_RESP_INFO) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =  sizeof(IN_BOARD_AUDIO_GET_LEVELS_RESP_INFO)
                             - sizeof(RESP_HEADER_INFO);

    // Verify pipe specified
    // Check if it is an Input or Output command
    // and retrieve the pipe index
    //------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->iblqAudioInf).paBoard,
                            (LcLPtrReq->iblqAudioInf).paDsp,
                            (LcLPtrReq->iblqAudioInf).paAudioNum,
                            (LcLPtrReq->iblqAudioInf).paAudioAttributes,
                            &LcPtVoie );


    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                // Actually process the request
                // ----------------------------
                LcRet = GetInAudioLevelsSubFct( LcLPtrReqHeader->hdHandle,
                                                UTIMask64bit( LcPtVoie->NumPipe ),
                                                UTIMaskWord( LcPtVoie->AppAudioNum ),
                                                &(LcLPtrRep->iblrLevels),
                                                &LcNbAudio );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID GetBoardAudioUerFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Status command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Status command response bloc
//
// *************************************************************************
//
//  Get UER register for a board output
//
// *************************************************************************
STATIC VOID GetBoardAudioUerFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPBOARD_UER_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO        LcLPtrReqHeader;
    LPRESP_HEADER_INFO      LcLPtrRepHeader;
    LPBOARD_UER_RESP_INFO   LcLPtrRep ;
    PVOIE_INFO              LcPtVoie;
    WORD                    LcRet ;
    WORD                    LcNbAudio;
    BYTE                    LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPBOARD_UER_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRep       = (LPBOARD_UER_RESP_INFO)PmBlocRetour;


    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReqHeader->hdBrSize < sizeof(BOARD_UER_RESP_INFO) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize =  sizeof(BOARD_UER_RESP_INFO)
                             - sizeof(RESP_HEADER_INFO);

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Verify pipe specified
    // Check if it is an Input or Output command
    // and retrieve the pipe index
    //------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->buqAudioInf).paBoard,
                            (LcLPtrReq->buqAudioInf).paDsp,
                            (LcLPtrReq->buqAudioInf).paAudioNum,
                            (LcLPtrReq->buqAudioInf).paAudioAttributes,
                            &LcPtVoie );


    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                // Actually process the request
                // ----------------------------
                LcRet = GetAudioUerSubFct( LcLPtrReqHeader->hdHandle,
                                           UTIMask64bit( LcPtVoie->NumPipe ),
                                           UTIMaskWord( LcPtVoie->AppAudioNum ),
                                           &(LcLPtrRep->burUer),
                                           &LcNbAudio );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}



