// *************************************************************************
//
//  COPYRIGHT 1996-1999 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************
//
//  AH_GENE.C
//
//  Project:                :   PCX***_E Driver
//
//  File description        :   part of API_HDL.C (code of general functions)
//
//  Author                  :   FH / FS / FM
//
//  Creation date           :   07/08/96
//  Last modification date  :   $Date: 6/06/07 4:51p $ by $Author: Mbr $
//
// **************************************************************************
//
//  API_HDL.C is splitted into 4 subparts. This file contains the code and
// only the code of the general commands of the driver.
//
// **************************************************************************

#define MIN( a, b )     ( (a) < (b) ? (a) : (b) )

// *************************************************************************
//
// STATIC VOID DspDebugFct( PmBlocRequete, PmBlocRetour)
//
//Input Parameters:
// ****************
//
//
//Output Paramaters:
// *****************
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : DSP trace request
//  LPDWORD PmBlocRetour  : DSP trace reply if any
//
//
//Return value:
// ************
//
// *************************************************************************
STATIC VOID DspDebugFct( INOUT LPDWORD PmBlocRequete,
             INOUT LPDWORD PmBlocRetour )
{
    LPDSP_TRACE_REQ_INFO    LcLPtrRequest;
    LPDSP_TRACE_RESP_INFO   LcLPtrReply;
	WORD LcCurrentBoard;

    LcLPtrRequest = (LPDSP_TRACE_REQ_INFO)  PmBlocRequete;
    LcLPtrReply   = (LPDSP_TRACE_RESP_INFO) PmBlocRetour;

    LcCurrentBoard = LcLPtrRequest->dspCarteNum;

	if( (LcCurrentBoard > MAX_BOARD) ||
		(CUR_PROTOCOL_PTR == NULL) ) {
		LcLPtrReply->rhCptr = ED_INVALID_BOARD;
		return;
	}

    LcLPtrReply->rhCptr = CUR_PROTOCOL_PTR->IDiag_DSPTrace( LcLPtrRequest->dspRequest,
                                                            LcLPtrRequest->dspParam1,
                                                            LcLPtrRequest->dspParam2,
                                                            (LPDWORD)(LcLPtrRequest + 1),
                                                            (LPDWORD)(LcLPtrReply + 1) );
} // DspDebugFct


// *************************************************************************
//
// STATIC VOID BlocTransfer( ... )
//
//Input Parameters:
// ****************
//
//  LPBYTE    PmPBufferSrc : pointer to the buffer to be copied.
//  DWORD     PmSizeSrc    : size (in bytes) of the buffer to be copied.
//  DWORD     PmBlocNumber : number of the bloc to be copied (begin from 0).
//                           The size of a bloc is DRIVER_TRACE_RESP_LEN bytes.
//
//Output Paramaters:
// *****************
//
//  LPBYTE    PmPBufferDst : pointer to the buffer where the bytes are copied.
//  LPWORD    PmPBufferLen : number of bytes copied.
//  LPWORD    PmPRemaining : number of bytes remaining to be copied.
//
//Return value:
// ************
//
// *************************************************************************
STATIC VOID BlocTransfer(
    IN    LPBYTE    PmPBufferSrc,
    IN    DWORD     PmSizeSrc,
    IN    DWORD     PmBlocNumber,
    OUT   LPBYTE    PmPBufferDst,
    OUT   LPWORD    PmPBufferLen,
    OUT   LPWORD    PmPRemaining )
{
    DWORD i;
    DWORD LcFirst = PmBlocNumber * DRIVER_TRACE_RESP_LEN;
    DWORD LcLast  = MIN( LcFirst + DRIVER_TRACE_RESP_LEN, PmSizeSrc );

    *PmPBufferLen = 0;
    *PmPRemaining = 0;

    for ( i = LcFirst ; i < LcLast ; i++ )
    {
        PmPBufferDst[i-LcFirst] = PmPBufferSrc[i];
    }

    if ( LcFirst < LcLast )
    {
        *PmPBufferLen = (WORD)(LcLast - LcFirst);
        *PmPRemaining = (WORD)(PmSizeSrc - LcLast);
    }
}

// *************************************************************************
//
// STATIC VOID DriverDebugFct( PmBlocRequete, PmBlocRetour)
//
//Input Parameters:
// ****************
//
//
//Output Paramaters:
// *****************
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : DRIVER trace request
//  LPDWORD PmBlocRetour  : DRIVER trace reply if any
//
//
//Return value:
// ************
//
// *************************************************************************

#define VERIFY_TYPE_AND_INDEX( TbIn, TbOut )                 \
        switch ( LcType )                                    \
        {                                                    \
        case 0:                                              \
            if ( LcInd >= MAX_DRV_INPIPE )                   \
            {                                                \
                LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS ); \
            }                                                \
            LcPInfo = TbIn;                                  \
            break;                                           \
                                                             \
        case 1:                                              \
            if ( LcInd >= MAX_DRV_OUTPIPE )                  \
            {                                                \
                LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS ); \
            }                                                \
                                                             \
            LcPInfo = TbOut;                                 \
            break;                                           \
                                                             \
        default:                                             \
            LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );     \
            break;                                           \
        }

STATIC VOID DriverDebugFct(
    INOUT LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    LPDRIVER_TRACE_REQ_INFO    LcLPtrRequest;
    LPDRIVER_TRACE_RESP_INFO   LcLPtrReply;
    WORD                       LcRet = SUCCESS;
    WORD                       LcBlocNumber;

    LcLPtrRequest = (LPDRIVER_TRACE_REQ_INFO)  PmBlocRequete;
    LcLPtrReply   = (LPDRIVER_TRACE_RESP_INFO) PmBlocRetour;
    LcBlocNumber  = LcLPtrRequest->driParam1;

    switch ( LcLPtrRequest->driRequest )
    {
    case REQUEST_DRIVER_INFO:
        BlocTransfer( (LPBYTE) (&GeneralDriverInfo),
                      sizeof( DRIVER_INFO ),
                      LcBlocNumber,
                      LcLPtrReply->driBuffer,
                      & (LcLPtrReply->driBufferLen),
                      & (LcLPtrReply->driRemaining) );
        break;

    case REQUEST_BOARD_INFO:
    {
        WORD LcBoardNum = LcLPtrRequest->driParam2;

        if ( LcBoardNum >= MAX_BOARD )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
        }
        else
        {
            BlocTransfer( (LPBYTE) (APH_Board_Info_Array + LcBoardNum),
                          sizeof( BOARD_INFO ),
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_BOARD_LEVEL_INFO:
    {
        WORD  LcCurrentBoard = LcLPtrRequest->driParam2;
        PBYTE LcPBufferSrc;
        WORD  LcSizeSrc;

        if( (LcCurrentBoard > MAX_BOARD) ||
			(CUR_COMMANDS_PTR == NULL ) )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_BOARD );
            break;
        }

        LcRet = CUR_COMMANDS_PTR->PIOGetBoardLevelInfo( & LcPBufferSrc, & LcSizeSrc );

        if ( LcRet == SUCCESS )
        {
            BlocTransfer( LcPBufferSrc,
                          LcSizeSrc,
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_APPLI_INFO:
    {
        WORD LcAppliInd = LcLPtrRequest->driParam2;

        if ( LcAppliInd >= MAX_PCX_HANDLE )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
        }
        else
        {
            BlocTransfer( (LPBYTE) (TbAppliInfo + LcAppliInd),
                          sizeof( APPLI_INFO ),
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_VOIE_INFO:
    {
        WORD       LcInd  = LcLPtrRequest->driParam2;
        WORD       LcType = LcLPtrRequest->driParam3;
        WORD       LcBoard= LcLPtrRequest->driParam4; // TODO
        PVOIE_INFO LcPInfo;

        switch ( LcType )
        {
        case 0:
            if( (LcBoard >= MAX_BOARD) || (LcInd >= MAX_BOARD_INPUTS) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
            }
            LcPInfo = TbVoieInInfo[LcBoard];
            break;

        case 1:
            if( (LcBoard >= MAX_BOARD) || (LcInd >= MAX_BOARD_OUTPUTS) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
            }

            LcPInfo = TbVoieOutInfo[LcBoard];
            break;

        default:
            LOAD_PCX_ERROR( LcRet, ED_INVALID_ADDRESS );
            break;
        }

        if ( LcRet == SUCCESS )
        {
            BlocTransfer( (LPBYTE) ( LcPInfo + LcInd ),
                          sizeof( VOIE_INFO ),
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_PIPE_INFO:
    {
        WORD       LcInd  = LcLPtrRequest->driParam2;
        WORD       LcType = LcLPtrRequest->driParam3;
        PPIPE_INFO LcPInfo;

        VERIFY_TYPE_AND_INDEX( TbPipeInInfo, TbPipeOutInfo )

        if ( LcRet == SUCCESS )
        {
            BlocTransfer( (LPBYTE) ( LcPInfo + LcInd ),
                          sizeof( PIPE_INFO ),
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_STREAM_INFO:
    {
        DWORD LcAdr =   ( ((DWORD)( LcLPtrRequest->driParam2 )) << 16 )
                      | LcLPtrRequest->driParam3;
        WORD  LcCurrentBoard = LcLPtrRequest->driParam2;
        PBYTE LcPBufferSrc;
        WORD  LcSizeSrc;

        if( (LcCurrentBoard > MAX_BOARD) ||
			(CUR_PROTOCOL_PTR == NULL ) )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_BOARD );
            break;
        }

        LcRet = CUR_PROTOCOL_PTR->IDiag_StreamGetInfo( LcAdr, &LcPBufferSrc, &LcSizeSrc );

        if ( LcRet == SUCCESS )
        {
            BlocTransfer( LcPBufferSrc,
                          LcSizeSrc,
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_GEN_BUFFER_INFO:
    {
        GEN_BUFFER_INFO LcGenBufferInfo;

        BUFGetFeatures( & LcGenBufferInfo.gbBuffTotalNb,
                        & LcGenBufferInfo.gbBuffAvailNb,
                        & LcGenBufferInfo.gbBuffSize );

        BlocTransfer( (PBYTE) (& LcGenBufferInfo),
                      sizeof( GEN_BUFFER_INFO ),
                      LcBlocNumber,
                      LcLPtrReply->driBuffer,
                      & (LcLPtrReply->driBufferLen),
                      & (LcLPtrReply->driRemaining) );
        break;
    }

    case REQUEST_BUFFER_INFO:
    {
        WORD  LcBufInd = LcLPtrRequest->driParam2;
        PBYTE LcPBufferSrc;
        WORD  LcSizeSrc;

        LcRet = BUFGetBufferInfoForDebug( LcBufInd,
                                          & LcPBufferSrc,
                                          & LcSizeSrc );
        if ( LcRet == SUCCESS )
        {
            BlocTransfer( LcPBufferSrc,
                          LcSizeSrc,
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_PHYS_MAP_INFO:
    {
        WORD  LcBufInd = LcLPtrRequest->driParam2;
        PBYTE LcPBufferSrc;
        WORD  LcSizeSrc;

        LcRet = BUFGetPhysMapInfoForDebug( LcBufInd,
                                           & LcPBufferSrc,
                                           & LcSizeSrc );
        if ( LcRet == SUCCESS )
        {
            BlocTransfer( LcPBufferSrc,
                          LcSizeSrc,
                          LcBlocNumber,
                          LcLPtrReply->driBuffer,
                          & (LcLPtrReply->driBufferLen),
                          & (LcLPtrReply->driRemaining) );
        }
        break;
    }

    case REQUEST_FORCE:

#ifdef DBG_VIEWER
#include "ah_forceio.c"
#endif

    default:
        LOAD_PCX_ERROR( LcRet, ED_UNAVAILABLE_FEATURE) ;
        break;
    }

    LcLPtrReply->RespHeader.rhCptr = LcRet;
}

// *************************************************************************
//
// STATIC VOID DspRecoverFatalErr( PmBlocRequete, PmBlocRetour)
//
//Input Parameters:
// ****************
//
//
//Output Paramaters:
// *****************
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : DSP Recover fatal DSP error request
//  LPDWORD PmBlocRetour  : DSP Recover fatal DSP error reply
//
//
// FH 25/02/97 V3 Management
// *************************************************************************
STATIC VOID DspRecoverFatalErr( INOUT LPDWORD PmBlocRequete,
                                INOUT LPDWORD PmBlocRetour )
{
    LPDSP_ERROR_REQ_INFO    LcLPtrRequest;
    LPDSP_ERROR_RESP_INFO   LcLPtrReply;
    LPRESP_HEADER_INFO      LcLPtrRepHeader;
    WORD                    LcBoardNum;
    WORD                    LcDspNum;

    // Initialisation of pointers
    LcLPtrRequest = (LPDSP_ERROR_REQ_INFO)  PmBlocRequete;
    LcLPtrReply   = (LPDSP_ERROR_RESP_INFO) PmBlocRetour;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcBoardNum = LcLPtrRequest->derqCarteNum;
    LcDspNum = LcLPtrRequest->derqDspNum;

    // Let's process some controls
    // ---------------------------
    if ( LcBoardNum >= MAX_BOARD )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_BOARD);
        return;
    }

    if ( LcDspNum >= MAX_BOARD_DSP )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_DSP );
        return;
    }

    LcLPtrReply->derspDspError = APH_Board_Info_Array[LcBoardNum].biTbDspInfo[LcDspNum].dsFatalError;

    LcLPtrRepHeader->rhCptr = SUCCESS;
    return;
}

// *************************************************************************
//
// STATIC VOID GetApplicationAudioOwner( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// ****************
//
// Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : Get Application Audio Owner request
//
// Output Parameters:
// *****************
//
//  LPDWORD PmBlocRetour  : Get Application Audio Owner reply
//
//
// FH 28/02/97 CREATION V3 Management
// *************************************************************************
STATIC VOID GetApplicationAudioOwner( INOUT LPDWORD PmBlocRequete,
                                      INOUT LPDWORD PmBlocRetour )
{
    LPAPP_AUDIO_OWNER_REQ_INFO    LcLPtrRequest         ;
    LPAPP_AUDIO_OWNER_RESP_INFO   LcLPtrReply           ;
    LPRESP_HEADER_INFO            LcLPtrRepHeader       ;
    BYTE                          LcApplicationAudioOwner;
    WORD                          LcRet                 ;
    PVOIE_INFO                    LcPtVoie              ;

    // Initialisation of pointers
    LcLPtrRequest = (LPAPP_AUDIO_OWNER_REQ_INFO)  PmBlocRequete;
    LcLPtrReply   = (LPAPP_AUDIO_OWNER_RESP_INFO) PmBlocRetour;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Search the VOIE_INFO entry, if any, for this audio
    // --------------------------------------------------
    LcRet = LookupBoardAudio(
                            LcLPtrRequest->AudioInf.paBoard,
                            LcLPtrRequest->AudioInf.paDsp,
                            LcLPtrRequest->AudioInf.paAudioNum,
                            LcLPtrRequest->AudioInf.paAudioAttributes,
                            &LcPtVoie );

    // ## FS (27/03/1998) -- FA #149 the DSP may be not loaded
    // this is not an error
    // --------------------------------------------------------
    if (   (LcRet != SUCCESS )
        && ( LcRet != ED_INVALID_DSP_SOFTWARE ) )   // This is not an error
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Is the requested audio allaocted ?
    // ----------------------------------
    if ( LcPtVoie == NULL )
    {
        // This audio is available
        // -----------------------
        LcLPtrReply->AppliInf.adHandle  = 0 ;
    }
    else
    {
        // This audio is already allocated to an application
        // -------------------------------------------------

		ASSERT(LcPtVoie->IndexAppli >= 1);

        PAPPLI_INFO     LcAppliPtr = &(TbAppliInfo[LcPtVoie->IndexAppli - 1]);

        LcApplicationAudioOwner = EncodeHandle( (BYTE)(LcPtVoie->IndexAppli - 1) );

        if ( LcAppliPtr->Carac & AI_PRIVATE_ATTRIB_MASK )
        {
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, WD_PRIVATE_DATA );

            if (   ((LcLPtrRequest->Header).hdHandle != DRIVER_KEY )
                || ((LcLPtrRequest->Header).hdHandle != LcApplicationAudioOwner ) )
            {
                LcLPtrReply->AppliInf.adHandle  = 0 ;
                return ;
            }
        }

        // It's Ok let's fill command response
        // ------------------------------------
        LcLPtrReply->AppliInf.adHandle        = LcApplicationAudioOwner;
        LcLPtrReply->AppliInf.adAttribut      = LcAppliPtr->Carac;
        LcLPtrReply->AppliInf.adWindowHandle  = LcAppliPtr->WindowHandle;
        LcLPtrReply->AppliInf.adWindowMessage = LcAppliPtr->WindowMessage;

        // ## FS (08/12/1998) -- Add Unicode support
        // -----------------------------------------
        MEMCPY(&((LcLPtrReply->AppliInf).adAppName),
               &(LcAppliPtr->TbNomAppli),
               sizeof(APP_STR_INFO));
    }

    return;
}


// *************************************************************************
//
// STATIC VOID InvalidCmd1( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd1(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM1 );
}

// *************************************************************************
//
// STATIC VOID AudioReleaseSubFct( IN BYTE, IN BYTE )
//
// Input Parameters:
// *****************
//
//  BYTE    PmAppIndex:     the application index
//  BYTE    PmPipeNum:      the index of the pipe owning the audio
//
// *************************************************************************
//
// Manage the audio release for the pipe definition / release Fct
//
//  FH: Bug correction 11/02/97
//  With the PmPipeNum parameter, we can't know if it is an ouput pipe
//  or an input pipe, so we add an another parameter: PmInputPipe, which is TRUE
//  for an input pipe.
//
// *************************************************************************
STATIC VOID AudioReleaseSubFct( IN BYTE     PmAppIndex,
                                IN BYTE     PmPipeNum,
                                IN BYTE     PmBoardNum,
                                IN BOOLEAN  PmInputPipe )
{
    PDSP_INFO   LcPDsp ;
    WORD        i;
    PVOIE_INFO  LcVoieInfo;

    ASSERT( PmBoardNum < MAX_BOARD );

    if ( PmInputPipe == FALSE )
    {
        // Release any output audio allocated to this pipe and application
        // ----------------------------------------------------------------
        for ( i = 0 ; i < MAX_BOARD_OUTPUTS ; i++ )
        {
            LcVoieInfo = &TbVoieOutInfo[PmBoardNum][i];

            if (  ( LcVoieInfo->IndexAppli == (PmAppIndex + 1) )
                && ( LcVoieInfo->NumPipe == PmPipeNum ) )
            {
                // Release the DSP audio output
                // -----------------------------
                LcPDsp = &(APH_Board_Info_Array[PmBoardNum].biTbDspInfo[0]) ;

                // ## FS (04/11/97) - Fix for bug FA #70
                // AudioNum is valid for both physical and
                // virtual audios
                // -------------------------------------------

                ASSERT( LcVoieInfo->AudioNum == i );

                if (LcVoieInfo->Nat == 0)      // Physical audio ?
                {
                    LcPDsp->dsMaskVPOut |= UTIMask64bit( i );
                }
                else                                // virtual audio then
                {
                    LcPDsp->dsMaskVVOut |= UTIMask64bit( i );
                }

                // Free the entry
                // ---------------
                LcVoieInfo->IndexAppli = 0;
            }
        }
    }
    else
    {
        // Release any input audio allocated to this pipe and application
        // ----------------------------------------------------------------
        for ( i = 0 ; i < MAX_BOARD_INPUTS ; i++ )
        {
            LcVoieInfo = &TbVoieInInfo[PmBoardNum][i];

            if (  ( LcVoieInfo->IndexAppli == (PmAppIndex + 1) )
                && ( LcVoieInfo->NumPipe == PmPipeNum ) )
            {
                // Release the DSP audio input
                // -----------------------------
                LcPDsp = &(APH_Board_Info_Array[PmBoardNum].biTbDspInfo[0]) ;

                // ## FS (04/11/97) - Fix for bug FA #70
                // AudioNum is valid for both physical and
                // virtual audios
                // -------------------------------------------

                ASSERT( LcVoieInfo->AudioNum == i );

                if (LcVoieInfo->Nat == 0)      // Physical audio ?
                {
                    LcPDsp->dsMaskVPIn |= UTIMask64bit( i );
                }
                else                                // virtual audio then
                {
                    LcPDsp->dsMaskVVIn |= UTIMask64bit( i );
                }

                // Free the entry
                // ---------------
                LcVoieInfo->IndexAppli = 0;
            }
        }
    }

    return;
}

// *************************************************************************
//
// STATIC VOID AudioAllocateSubFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete:  General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour:   General command request bloc
//  PBYTE   PmPipeNum:      The first audio number if allocation succeeds
//
// *************************************************************************
//
// Manage the audio allocation for the pipe definition Fct
//
// *************************************************************************
STATIC VOID AudioAllocateSubFct(
    IN LPDWORD  PmBlocRequete,
    OUT LPDWORD PmBlocRetour,
    IN  BYTE    PmPipeIndex,
    OUT PBYTE   PmBoardNum,
    OUT PWORD   PmFirstAudio )
{
    LPPIPE_DEF_REQ_INFO LcLPtrReq;
    LPPIPE_DEF_RESP_INFO LcLPtrRep;
    LPPIPE_STREAM_INFO  LcPtrStream;
    LPPIPE_AUDIO_INFO   LcPtrAudio;
    PVOIE_INFO          LcPtVoieInfo ;
    PDSP_INFO           LcPtrDsp;
    TARGET_INFO         LcTarget;
    BOOLEAN             LcFound, LcFirstDone;
    BYTE                i;
    WORD                j, LcAudioNum;
    BYTE                LcAppIndex;
    BYTE                LcCurrentBoard;
    WORD                LcOper ;
    PDWORDLONG          LcPDspFreeAudioMask ;
    BOOLEAN             LcIsCapture;

    LcLPtrReq = (LPPIPE_DEF_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPPIPE_DEF_RESP_INFO)PmBlocRetour;

    LcAppIndex = DecodeHandle( LcLPtrReq->Header.hdHandle );

    LcOper =        (LcLPtrReq->pdqAttributes & PIPE_PLAYREC_MASK) ;
    LcIsCapture =   ( LcOper == OPER_PLAY ) ? FALSE : TRUE;

    // Step #1: skip the streams requested
    // ----------------------------------------

    LcPtrStream = (LPPIPE_STREAM_INFO)(LcLPtrReq + 1);

    LcPtrAudio = (LPPIPE_AUDIO_INFO)(LcPtrStream + LcLPtrReq->pdqMaxStream);

    // ----------------------------------------
    // Step #2: check all parameters, and audio
    //          availability, and allocate
    // ----------------------------------------

    if (  LcLPtrReq->pdqAudioNum == 0 )
    {
        LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_ALLOCATE_AUDIO_IMPOSSIBLE );
        return;
    }
    LcTarget.tgCarte = LcCurrentBoard = LcPtrAudio[0].paBoard ;

    if (  ( LcCurrentBoard >= MAX_BOARD )
        || ( APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_BOARD );
        return ;
    }

    LcPtVoieInfo = LcIsCapture ? TbVoieInInfo[LcCurrentBoard] : TbVoieOutInfo[LcCurrentBoard];

    // First audio is not found yet
    // first audio number will be the pipe number
    // -------------------------------------------
    LcFirstDone = FALSE ;

    // For each audio requested
    // -------------------------
    for ( i = 0 ; i < LcLPtrReq->pdqAudioNum ; i++ )
    {
        // ----------------------------------------
        // Step #2.1: Check these parameters
        // ----------------------------------------

        // The board must be the same for all audios
        // ----------------------------------------
        if ( LcCurrentBoard != LcPtrAudio[i].paBoard )
        {
            LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_BOARD );

            if ( LcFirstDone )
            {
                AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
            }
            return ;
        }

        // Check the attributes correctness
        // ---------------------------------
        if ( (LcPtrAudio[i].paAudioAttributes & ~ALL_AUDIO_ATTRIB_MASK) != 0 )
        {
            LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_UNKNOWN_AUDIO_ATTRIB );

            if ( LcFirstDone )
            {
                AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
            }
            return ;
        }

        // Prepare a target, since we have to convert the board
        // audio num into a dsp one
        // ------------------------------------------------------

        LcFound = FALSE ;

        if ( (LcPtrAudio[i].paAudioAttributes & AUDIO_PHYSICAL) == 0 )
        {
            // Virtual audio
            // ------------------

            if ( i == 0 )
            {
                // Shortcuts to driver structures decribing the board and dsp
                // ----------------------------------------------------------
                LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0]);

                // Fixed limit to 16 applis instead of 32
                //    Was UTIMaskWord instead of UTIMaskDWord
                // ---
                // The dsp must be loaded and running
                // -----------------------------------
                if (	( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsEtat == 0 )
                    || !( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsMaskIndexAppli
                         & UTIMaskDWord( (WORD)LcAppIndex ) ) )
                {
                    LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_DSP );

                    if ( LcFirstDone )
                    {
                        AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                    }
                    return ;
                }

                if ( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsFatalError != 0 )
                {
                    LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_DSP_CRASHED );

                    if ( LcFirstDone )
                    {
                        AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                    }
                    return ;
                }

            }
            else if ( LcPtrAudio[i].paDsp )
            {
                // only 1 DSP supported 
                // ----------------------------------
                LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_DSP );

                if ( LcFirstDone )
                {
                    AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                }
                return ;
            }

            // ----------------------------------------
            // Step #2.2Virt: check audio availability
            // ----------------------------------------
            if ( LcOper == OPER_PLAY )
                LcPDspFreeAudioMask = &(LcPtrDsp->dsMaskVVOut) ;
            else
                LcPDspFreeAudioMask = &(LcPtrDsp->dsMaskVVIn) ;

            if ( !(  UTIMask64bit(LcPtrAudio[i].paAudioNum) & *LcPDspFreeAudioMask ) )
            {
                // the audio is already allocated or does not exist
                // -------------------------------------------------
                LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_PIPE_AUDIO );

                if ( LcFirstDone )
                {
                    AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                }
                return ;
            }

            // ----------------------------------------
            // Step #2.3Virt: find a free entry for this audio
            // ----------------------------------------

            j = LcPtrAudio[i].paAudioNum;

            if ( j < (LcIsCapture ? MAX_BOARD_INPUTS : MAX_BOARD_OUTPUTS) )
            {
                if ( LcPtVoieInfo[j].IndexAppli == 0 )
                {
                    // Get the audio number ( assuming only 1 DSP )
                    // -----------------------
                    LcAudioNum = LcPtrAudio[i].paAudioNum;

                    // Audio is no more available
                    // ----------------------------
                    LcFound = TRUE ;
                    *LcPDspFreeAudioMask &=  ~UTIMask64bit( LcAudioNum );

                    // If 1st audio allocated, let's remember it and
                    // assign the future pipe number
                    // -----------------------------------
                    if ( !LcFirstDone )
                    {
                        LcFirstDone = TRUE;
                        *PmFirstAudio = j ;
                    }

                    // initialize the entry
                    // ---------------------
                    LcPtVoieInfo[j].Nat = 1;     // virtual audio
                    LcPtVoieInfo[j].AppAudioNum = i;
                    LcPtVoieInfo[j].AudioNum = LcAudioNum;
                    LcPtVoieInfo[j].IndexAppli = LcAppIndex + 1;
                    LcPtVoieInfo[j].NumPipe = PmPipeIndex;

                }
            }  // find a free entry

           // Verify the allocation has been done actually
           // ---------------------------------------------
           if ( !LcFound )
           {
               LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_ALLOCATE_AUDIO_IMPOSSIBLE );

               if ( LcFirstDone )
               {
                   AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
               }
               return ;
           }
        }
        else        // This is a physical audio then
        {
            // ----------------------------------------
            // Step #2.2Phys: check audio availability
            // ----------------------------------------

          if ( i == 0 )
            {
                // Shortcuts to driver structures describing the board and dsp
                // ----------------------------------------------------------
                LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0]);

                // Fixed limit to 16 applis instead of 32
                //    Was UTIMaskWord instead of UTIMaskDWord
                // ---
                // The dsp must be loaded and running
                // -----------------------------------
                if (   ( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsEtat == 0 )
                    || !( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0].dsMaskIndexAppli
                    & UTIMaskDWord( (WORD)LcAppIndex ) ) )
                {
                    LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_DSP );

                    if ( LcFirstDone )
                    {
                        AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                    }
                    return ;
                }
            }

            // now, let's check the audio is available
            // ----------------------------------------
            if ( LcOper == OPER_PLAY )
              LcPDspFreeAudioMask = &(LcPtrDsp->dsMaskVPOut) ;
            else
              LcPDspFreeAudioMask = &(LcPtrDsp->dsMaskVPIn) ;

            if ( !( UTIMask64bit(LcPtrAudio[i].paAudioNum) & *LcPDspFreeAudioMask ) )
            {
                // the audio is already allocated or does not exist
                // -------------------------------------------------
                LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_INVALID_PIPE_AUDIO );

                if ( LcFirstDone )
                {
                    AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                }
                return ;
            }

            // ----------------------------------------
            // Step #2.3Phys: find a free entry for this audio
            // ----------------------------------------

            j = LcPtrAudio[i].paAudioNum;

            if ( j < (LcIsCapture ? MAX_BOARD_INPUTS : MAX_BOARD_OUTPUTS) )
            {
                if ( LcPtVoieInfo[j].IndexAppli == 0 )
                {
                    LcAudioNum = LcPtrAudio[i].paAudioNum;

                    // Audio is no more available
                    // ----------------------------
                    LcFound = TRUE ;
                    *LcPDspFreeAudioMask &= ~UTIMask64bit( LcAudioNum );

                    // If 1st audio allocated, let's remember it and
                    // assign the future pipe number
                    // -----------------------------------
                    if ( !LcFirstDone )
                    {
                        LcFirstDone = TRUE;
                        *PmFirstAudio = j ;
                    }

                    // initialize the entry
                    // ---------------------
                    LcPtVoieInfo[j].Nat = 0;
                    LcPtVoieInfo[j].AppAudioNum = i;
                    LcPtVoieInfo[j].AudioNum = LcAudioNum;
                    LcPtVoieInfo[j].IndexAppli = LcAppIndex + 1;
                    LcPtVoieInfo[j].NumPipe = PmPipeIndex;

                }
            }  // find a free entry

            // Verify the allocation has been done actually
            // ---------------------------------------------
            if ( !LcFound )
            {
                LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_ALLOCATE_AUDIO_IMPOSSIBLE );

                if ( LcFirstDone )
                {
                    AudioReleaseSubFct( LcAppIndex, PmPipeIndex, LcCurrentBoard, LcIsCapture );
                }
                return ;
            }

        } // if virtual audio else physical

    } // Allocation for each audio

    if ( !LcFirstDone )
    {
        LOAD_PCX_ERROR( (LcLPtrRep->RespHeader).rhCptr, ED_ALLOCATE_AUDIO_IMPOSSIBLE );
    }

    *PmBoardNum = LcCurrentBoard;

}  // ************ end of AudioAllocateSubFct() *******************


// *************************************************************************
//
// STATIC VOID PipeRemoveSubFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
//  FH Modification clean PipeUerOutMask of BOARD_INFO structure
//
//
// *************************************************************************
//
//  Process the pipe definition command.
//  Update the pipe definition command response bloc.
//
// *************************************************************************
//
// ## FS (04/09/1998) -- FA #175, FA #203, FA #207
// In order to manage error cases in a much more readable and safer
// way, the loop for each pipe will be handled at only one place, that
// is at the API level.
//
// For the moment, we will not enforce a parameter check for only one pipe
//
// *************************************************************************
STATIC VOID PipeRemoveSubFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_REM_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO     LcLPtrReqHeader;
    LPRESP_HEADER_INFO   LcLPtrRepHeader;
    TARGET_INFO          LcTarget;
    DWORDLONG            LcPipeMask ;
    BYTE                 LcAppIndex;
    BYTE                 LcCurrentBoard, LcDsp;
    BYTE                 i, LcPipeIndex;
    BYTE                 LcMaxPipe;
    BOOLEAN              LcErrorFound;
    WORD                 LcRet;
    PPIPE_INFO           LcPtPipe ;
    PVOIE_INFO           LcPtAudio ;
    PBOARD_INFO          LcPtrBoard;
    BOOLEAN              LcIsCapture;

    // 10/02/97 - Use pseudo reply block under NT
    LPADD_DESC_INFO      LcAddrDescTab      = NULL ;

    LcLPtrReq = (LPPIPE_REM_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // 10/02/97 - Use pseudo reply block under NT
    LcAddrDescTab    = (LPADD_DESC_INFO)
                            ((LPANY_UNMAPPING_REPLY_INFO)LcLPtrRepHeader)->urAddrDescriptors;

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // No error encountered yet
    // -------------------------
    LcErrorFound = FALSE ;

    // For all the pipes
    // -----------------
    for ( i = 0 ; i < 2 ; i++ )
    {
        if( i == 0 )
        {
            LcPipeMask = LcLPtrReq->prqOutMask64.QuadPart;
            LcPtPipe   = TbPipeOutInfo ;
            LcMaxPipe  = MAX_DRV_OUTPIPE ;
            LcTarget.tgCaracPipeVoie = OPER_PLAY;
            LcIsCapture= FALSE;
        } else {
            LcPipeMask = LcLPtrReq->prqInMask64.QuadPart;
            LcPtPipe   = TbPipeInInfo ;
            LcMaxPipe  = MAX_DRV_INPIPE ;
            LcTarget.tgCaracPipeVoie = OPER_REC;
            LcIsCapture= TRUE;
        }

        LcRet = SUCCESS ;

        while ( LcPipeMask != 0 )
        {
            LcPipeIndex = (BYTE) UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( (WORD) LcPipeIndex );

            // When max pipe is reached, no need to go on with this mask
            // ----------------------------------------------------------
            ASSERT( LcPipeIndex < LcMaxPipe );

            // Pipe index is also the index of the first audio of the pipe
            // Check the pipe is actually owned by the application (normally
            // already checked, but let's make sure...
            // -----------------------------------------------------------
            // already done by ChkPmManyInOrOutPipeMask
            // and will be tested once more by AudioReleaseSubFct
            if ( 1 ) // LcPtAudio[LcPipeIndex].IndexAppli == (LcAppIndex + 1) )
            {
                LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
                ASSERT( LcCurrentBoard < MAX_BOARD );
                LcDsp   = 0;

                // Verify the pipe has been unsynchronized
                // ---------------------------------------

                // ## FS (06/01/98) -- check the board is still
                // here (for Pocket support) -- fix for FA#96
                // --------------------------------------------
                if (   ( APH_Board_Info_Array[LcCurrentBoard].biCarac == CARAC_USED )
                    && ( LcPtPipe[LcPipeIndex].piClockSource != CLOCK_TYPE_NONE ) )
                {
                    LOAD_PCX_ERROR( LcRet, ED_SUPPRESS_PIPE_REFUSED );
                }
                else
                {
                    LcTarget.tgPipe         = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
                    LcTarget.tgVioHandle    = LcPtPipe[LcPipeIndex].piVioHandle;

                    // ## FS (08/10/1997) -- do not send any command to the
                    // board if the board does no more exists of if
                    // DSP has crashed or timed out. In these cases, we
                    // simply bypass the pipe release message in order
                    // to release all driver ressources anyway.
                    // -----------------------------------------------------

                    // unsynchronized pipe can be removed
                    // ----------------------------------

                    // ## FS (08/10/97) -- check the board is still
                    // here (for Pocket support)
                    // --------------------------------------------

					BOOLEAN LcTest =  (( APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED )
									|| ( APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[LcDsp].dsFatalError != 0 ) );

                        // Board already removed or crashed,
                        // do not send forward the request,
                        // but still release the driver resources
                        // --------------------------------------
                    LcRet = CUR_PROTOCOL_PTR->IFlow_PipeRelease( &LcTarget, (!LcTest) );

                }

                if ( LcRet == SUCCESS )
                {
                    // Free the buffers allocated to the pipe
                    // ---------------------------------------
                    BUFFreeBuffers(
                            &LcPtPipe[LcPipeIndex].piHandleBuffers   ,
                            LcAddrDescTab                           ) ;

                    // resets audio levels
                    // ----------------------------------------
                    PipeResetLevelsSubFct(   LcPtPipe[LcPipeIndex].piResetAnaLevelsAction,
                                            LcPipeIndex,
                                            LcIsCapture );

                    // Free the audios allocated to the pipe
                    // ---------------------------------------
                    AudioReleaseSubFct(
                             LcAppIndex         ,
                             LcPipeIndex        ,
                             LcCurrentBoard     ,
                             LcIsCapture) ;

                    // Remove the current output pipe from
                    // PipeUerOutMask of BOARD_INFO structure
                    // --------------------------------------
                    if ( LcIsCapture == FALSE )
                    {
                        DWORD       k   ;
                        PDSP_INFO   LcDspInfoPtr ;

                        // Get the corresponding board structure
                        LcPtrBoard = &(APH_Board_Info_Array[LcCurrentBoard]);

                        // Get the corresponding DSP structure
                        LcDspInfoPtr = &((LcPtrBoard->biTbDspInfo)[LcDsp]);

                        for ( k = 0 ; k < MAX_CONTEXT ; k ++ )
                        {
                            LcDspInfoPtr->dsContextMaskOutPipes[k] &= ~UTIMask64bit( LcPipeIndex );

                            // If nobody uses the context, then flags it as such
                            //
                            if ( !LcDspInfoPtr->dsContextMaskOutPipes[k] )
                            {
                                LcDspInfoPtr->dsContextID[k] = 0    ;
                            }
                        }
                    }
                    else
                    {
                        // do not change audio input settings if DHS controlled
                        //
                        if( LcIsCapture && APHChkIsAudioDHSControlled( LcCurrentBoard, LcPtPipe[LcPipeIndex].piNumPipeForDsp, TRUE ) )
                        {
                            LcRet = SUCCESS;
                        }
                        else
                        {
                            // release possible source reservation on matching audios
                            //
                            DWORDLONG LcAudioMask = APH_Get_Pipe_Audio_Mask( LcPipeIndex );

                            LcRet = CUR_PROTOCOL_PTR->IResource_Source_Release_Audios(LcAudioMask);
                        }
                    }

                    // Zeroes the entry, including the piNbAudio field
                    // ------------------------------------------------
                    BZERO2( &LcPtPipe[LcPipeIndex], PIPE_INFO, 1 );

                }
                else if ( !LcErrorFound )
                {
                    // Remember this error and continue
                    // releasing as many pipes as possible
                    // ------------------------------------
                    LcErrorFound = TRUE ;
                    LcLPtrRepHeader->rhCptr = LcRet;
                }

            } // if the pipe belongs to the application

        }  // for all pipes in the mask

    } // for all masks

}

// *************************************************************************
//
// STATIC VOID PipeRemoveFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
//  FH Modification clean PipeUerOutMask of BOARD_INFO structure
//
//
// *************************************************************************
//
//  Process the pipe definition command.
//  Update the pipe definition command response bloc.
//
// *************************************************************************
//
// ## FS (04/09/1998) -- FA #175, FA #203, FA #207
// In order to manage error cases in a much more readable and safer
// way, the loop for each pipe will be handled at only one place, that
// is at the API level.
//
// For the moment, we will not enforce a parameter check for only one pipe
//
// *************************************************************************
STATIC VOID PipeRemoveFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_REM_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO     LcLPtrReqHeader;
    LPRESP_HEADER_INFO   LcLPtrRepHeader;
    WORD                 LcRet;
    BYTE                 LcAppIndex;

    // 10/02/97 - Use pseudo reply block under NT
    LPADD_DESC_INFO      LcAddrDescTab      = NULL ;

    LcLPtrReq = (LPPIPE_REM_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRepHeader->rhSize = 0 ;

    // 10/02/97 - Use pseudo reply block under NT
    LcAddrDescTab    = (LPADD_DESC_INFO)
                            ((LPANY_UNMAPPING_REPLY_INFO)LcLPtrRepHeader)->urAddrDescriptors;
    BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // DHS application pipe handling
    //
    if(LcAppIndex == DHSAppHandle)
    {
        DHSPipeRemove(LcLPtrReq, LcLPtrRepHeader);
        return;
    }

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcLPtrReq->prqOutMask64.QuadPart,
                                        LcLPtrReq->prqInMask64.QuadPart );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    PipeRemoveSubFct( PmBlocRequete, PmBlocRetour );
}


// *************************************************************************
//
// STATIC VOID PipeDefFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour : General command request bloc
//
// Return value:
// *************
//
//  SUCCESS if the allocations complete successfully, an error code
// otherwise
//
// *************************************************************************
//
//  Allocation of a pipe + audio + buffers
//
// *************************************************************************
STATIC VOID PipeDefFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_DEF_REQ_INFO  LcLPtrReq;
    LPPIPE_DEF_RESP_INFO LcLPtrRep;
    LPBC_HEADER_INFO     LcLPtrReqHeader;
    LPRESP_HEADER_INFO   LcLPtrRepHeader;
    LPPIPE_STREAM_INFO   LcPtrStream;
    LPADD_DESC_INFO      LcPtrAddr;
    PPIPE_INFO           LcPtPipe;
    STATIC FRAME_INFO    LcTbFrameInfo[MAX_STREAM_PIPE];
    BYTE                 LcPipeIndex;
    BYTE                 LcAppIndex;
    WORD                 i;
    WORD                 LcRet, LcRepSize;
    DWORD                LcAllocatedBuffers;
    HANDLE_BUFFER        LcHandleBuffers;
    DWORD                LcPlayFormats, LcRecordFormats ;
    BOOLEAN              LcIsCapture;
    WORD                 LcFirstAudio;
    BYTE                 LcAttributes;
    BYTE                 LcCurrentBoard = 0;


    // 10/02/97 - Use pseudo reply block under NT
    LPADD_DESC_INFO      LcAddrDescTab      = NULL ;


    LcLPtrRep = (LPPIPE_DEF_RESP_INFO)PmBlocRetour;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrReq = (LPPIPE_DEF_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    // 10/02/97 - Use pseudo reply block under NT
    LcAddrDescTab = (LPADD_DESC_INFO)( LcLPtrRep + 1 );

    BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );

    // Get the application index from handle
    // --------------------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // ----------------------------------------
    // Step #0: Parameters checks
    // ----------------------------------------

    // Verify the pipe attributes correctness
    // ---------------------------------------
    if ( (LcLPtrReq->pdqAttributes & ~ALL_PIPE_ATTRIB_MASK) != 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_UNKNOWN_PIPE_ATTRIB );
        return;
    }

    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    LcRepSize = sizeof( PIPE_DEF_RESP_INFO );
    if ( LcLPtrReqHeader->hdBrSize < LcRepSize )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    LcLPtrRepHeader->rhSize = LcRepSize - sizeof( RESP_HEADER_INFO );

    // DHS application pipe handling
    //
    if(LcAppIndex == DHSAppHandle)
    {
        DHSPipeDef(LcLPtrReq, LcLPtrRep);
        return;
    }

    // ----------------------------------------
    // Step #0.99: find free pipe entry
    // ----------------------------------------

    // It is an input or output pipe ?
    LcAttributes = LcLPtrReq->pdqAttributes & ( PIPE_OFFLINE_MASK | PIPE_PLAYREC_MASK);
    LcIsCapture = (LcAttributes & PIPE_PLAYREC_MASK) ? TRUE : FALSE;

    int LcMaxPipe;
    if (LcIsCapture)
    {
        LcMaxPipe = MAX_DRV_INPIPE;
        LcPtPipe = TbPipeInInfo;
    }
    else
    {
        LcMaxPipe = MAX_DRV_OUTPIPE;
        LcPtPipe = TbPipeOutInfo;
    }
    // find free entry in the pipe array
    //
    for( LcPipeIndex=0; LcPipeIndex<LcMaxPipe; LcPipeIndex++ )
    {
        if( LcPtPipe[LcPipeIndex].piNbAudio == 0)
            break;
    }
    if( LcPipeIndex == LcMaxPipe )
    {
        // no free entry found !
        //
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_ALLOCATE_AUDIO_IMPOSSIBLE );
        return;
    }

    // ----------------------------------------
    // Step #1: allocate the buffers requested
    // ----------------------------------------
    LcRet = BUFAllocateBuffers( LcLPtrReq->pdqBuffersNum    ,
                                LcLPtrReq->pdqBufferSize    ,
                                &LcAllocatedBuffers         ,
                                &LcHandleBuffers            );

    // if no error occurred, then let's fill the
    // address descriptors for each buffer actually allocated
    // ------------------------------------------------------
    if ( LcRet & ERROR_MASK )
    {
        LcLPtrRep->RespHeader.rhCptr = LcRet;
        return;
    }

    for ( i = 0, LcPtrAddr = (LPADD_DESC_INFO)(LcLPtrRep + 1) ;
          ( i < LcAllocatedBuffers ) ;
           i++ )
    {
        // Check the reply bloc will be long enough to hold the reply
        // if not, stop allocating since the buffer will be lost
        // ----------------------------------------------------------
        LcRepSize += sizeof( ADD_DESC_INFO );

        if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
            break;
        }
        (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

        // Update _ADD_DESC_INFO structure
        // -------------------------------
        BUFFillAddressInfo(
                    i                       ,
                    &LcHandleBuffers        ,
                    (LcLPtrReq->Header).hdWorld,
                    LcPtrAddr               );
        LcPtrAddr++;

    }   // end for ( each buffer allocated )


    // Fill the rest of the reply block and
    // set a warning if not the whole amount of buffers
    // has been allocated
    // ------------------------------------------------

    LcLPtrRep->pdpBuffersNum = (WORD)LcAllocatedBuffers;

    if ( LcAllocatedBuffers < LcLPtrReq->pdqBuffersNum )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_UNDER_ALLOCATE_BUFFER );
    }

    // ----------------------------------------
    // Step #2: allocate the audio requested
    // ----------------------------------------

    AudioAllocateSubFct( PmBlocRequete, PmBlocRetour, LcPipeIndex, &LcCurrentBoard, &LcFirstAudio );

    if ( LcLPtrRep->RespHeader.rhCptr & ERROR_MASK )
    {
        // Audio allocation failed, release the buffers
        // ---------------------------------------------
        BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );
        BUFFreeBuffers( &LcHandleBuffers, LcAddrDescTab );

        // Pass the error code through
        // ----------------------------
        return ;
    }

    // ----------------------------------------
    // Step #3: check the streams requested
    // ----------------------------------------

    LcPtrStream = (LPPIPE_STREAM_INFO)(LcLPtrReq + 1);

    for ( i = 0 ;
          i < LcLPtrReq->pdqMaxStream ;
          i++, LcPtrStream++ )
    {

        LcTbFrameInfo[i].fiFormat = LcPtrStream->psFormat;
        LcTbFrameInfo[i].fiHSize  = LcPtrStream->psBitRate;

    }

    // -----------------------------------------------------
    // Step #3bis: check stream formats are
    //            compatible to the 1st DSP
    // -----------------------------------------------------
    // We assume here that only one DSP is concerned
    // ----------------------------------------------

    if( LcIsCapture == FALSE ) {
			//MBR (2003/05/23) reset bytecounter
			BZERO2( TbOutStreamBytePos[LcPipeIndex], PCX_TIME, MAX_STREAM_PIPE );
    } 

    LcRet           = SUCCESS;    // allow buffer underallocation

    CUR_PROTOCOL_PTR->IDiag_GetSupportedPRFeatures( &LcPlayFormats ,&LcRecordFormats );
    
    if (LcLPtrReq->pdqAttributes & PIPE_PLAYREC_MASK)
    {
        // Record pipe, test the appropriate format mask
        // ---------------------------------------------
        for ( i = 0 ; i < LcLPtrReq->pdqMaxStream ; i++ )  {
            if (   (LcTbFrameInfo[i].fiFormat & LcRecordFormats) != LcTbFrameInfo[i].fiFormat )
            {
                // Match error, at least one format is not supported
                // -------------------------------------------------
                LOAD_PCX_ERROR( LcRet, ED_FORMAT_NOT_SUPPORTED );
                break;
            }
        }
    }
    else
    {
        // Play pipe, test the appropriate format mask
        // ---------------------------------------------
        for ( i = 0 ; i < LcLPtrReq->pdqMaxStream ; i++ )  {
            if (   (LcTbFrameInfo[i].fiFormat & LcPlayFormats) != LcTbFrameInfo[i].fiFormat )
            {
                // Match error, at least one format is not supported
                // -------------------------------------------------
                LOAD_PCX_ERROR( LcRet, ED_FORMAT_NOT_SUPPORTED );
                break;
            }
        }
    }

    /*
    */
    if ( LcRet != SUCCESS )
    {
        BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );
        BUFFreeBuffers( &LcHandleBuffers, LcAddrDescTab );
        AudioReleaseSubFct( LcAppIndex, LcPipeIndex, LcCurrentBoard, LcIsCapture );

        // Pass the error code through
        // ----------------------------
        LcLPtrRep->RespHeader.rhCptr = LcRet;
        return ;
    }

    // ----------------------------------------
    // Step #4: resets audio levels
    // ----------------------------------------

    // ## FM (02/05/1999) -- Allows the user to specify whever or not analogic
    //                       levels must be reset when creating or deleting a
    //                       pipe. This can be done either globally through an
    //                       option in the .ini file, or locally for each pipe.
    //
    // store for removefct
    LcPtPipe[LcPipeIndex].piResetAnaLevelsAction = GeneralDriverInfo.drResetAnalogLevelsAction;
    
    if( LcLPtrReq->pdqManagement & P_DO_NOT_RESET_ANALOG_LEVELS) {
        // override

        LcPtPipe[LcPipeIndex].piResetAnaLevelsAction = RESET_GAIN_NO ;
    }

    // ----------------------------------------
    // Step #5: send the request to the DSP
    // ----------------------------------------

    LcRet = CUR_PROTOCOL_PTR->IFlow_PipeDef(	IN LcAttributes,
                                                IN LcFirstAudio,
                                                LcLPtrReq->pdqMaxStream             ,
												LcLPtrReq->pdqAudioNum              ,
                                                LcLPtrReq->pdqChannelMask           ,
												LcLPtrReq->pdqManagement            ,
												LcTbFrameInfo                       ,
                                                TRUE                                , // from APINP
												&LcPtPipe[LcPipeIndex].piVioHandle  ,
												&LcLPtrRep->pdpMaxAllocatedStream  
                                                );
    if ( LcRet != SUCCESS )
    {
        BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );
        BUFFreeBuffers( &LcHandleBuffers, LcAddrDescTab );
        AudioReleaseSubFct( LcAppIndex, LcPipeIndex, LcCurrentBoard, LcIsCapture );

        // Pass the error code through
        // ----------------------------
        LcLPtrRep->RespHeader.rhCptr = LcRet;
        return ;
    }

    // ----------------------------------------
    // Step #6: fill reply block
    // ----------------------------------------

    // Fill Response bloc: Output Pipe Mask
    // ************************************
    LcLPtrRep->pdpPipeMask64.QuadPart = UTIMask64bit( LcPipeIndex );

    // NumPipe = handle of this pipe for the application

    // Fill the corresponding output pipe structure
    // ********************************************
    LcPtPipe[LcPipeIndex].piNumPipeForDsp = (BYTE)LcFirstAudio;
    LcPtPipe[LcPipeIndex].piNumBoard = (BYTE)LcCurrentBoard;
    LcPtPipe[LcPipeIndex].piNbAudio = LcLPtrReq->pdqAudioNum;
    LcPtPipe[LcPipeIndex].piNbMaxFlux = (BYTE) LcLPtrRep->pdpMaxAllocatedStream;

    // build stream mask

    LcPtPipe[LcPipeIndex].piStreamMask = 0 ;
    for ( i = 0 ; i < LcLPtrRep->pdpMaxAllocatedStream; i ++ )
    {
        LcPtPipe[LcPipeIndex].piStreamMask <<= 1 ;
        LcPtPipe[LcPipeIndex].piStreamMask |= 1 ;
    }

    // build audio mask
    for ( i = 0 ; i < LcPtPipe[LcPipeIndex].piNbAudio; i ++ )
    {
        LcPtPipe[LcPipeIndex].piPipeAudioMask <<= 1 ;
        LcPtPipe[LcPipeIndex].piPipeAudioMask |= 1 ;
    }

    LcPtPipe[LcPipeIndex].piPhysAudioMask = (DWORDLONG)(LcPtPipe[LcPipeIndex].piPipeAudioMask) << LcFirstAudio ;

    LcPtPipe[LcPipeIndex].piPipeIsOffline = ( ( LcLPtrReq->pdqAttributes & PIPE_OFFLINE_MASK ) != 0 );

    LcPtPipe[LcPipeIndex].piHandleBuffers = LcHandleBuffers;
    LcPtPipe[LcPipeIndex].piAdrBlkReq = NULL;
    LcPtPipe[LcPipeIndex].piConversionEnded = FALSE ; // FS - 04/03/1997

    LcPtPipe[LcPipeIndex].piAdrNotifyReq = NULL;      // FS - 25/04/1997
    LcPtPipe[LcPipeIndex].piPendingNotify = FALSE ;         // for notified commands
    LcPtPipe[LcPipeIndex].piNotifiedNotReadYet = FALSE ;    // for notified commands

    // if one of this audios is controlled by DHS,
    // adjust its digital and monitoring levels and mutes
    for ( i = 0 ; i < LcPtPipe[LcPipeIndex].piNbAudio; i ++ )
    {
        if( APHChkIsAudioDHSControlled( LcCurrentBoard, LcFirstAudio + i, LcIsCapture ) )
        {
            APHNewPipeSetDHSLevels( LcCurrentBoard, LcFirstAudio + i, LcIsCapture );
        }
    }
    // end DHS
}


// *************************************************************************
//
// STATIC VOID DspDeallocSsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
// Try to deallocate application resources
//
// *************************************************************************
STATIC VOID DspDeallocSsFct(IN LPDWORD PmBlocRequete,
                            OUT LPDWORD PmBlocRetour )
{
    LPDSP_USE_REQ_INFO   LcLPtrReq;
    LPRESP_HEADER_INFO   LcLPtrRep;
    LPBC_HEADER_INFO     LcLPtrReqHeader;
    PDSP_INFO            LcPtrDsp;
    WORD                 LcCurrentBoard, LcDsp;
    BYTE                 LcAppIndex;
    WORD                 k;
    BOOLEAN              LcCanBeReleased;
    DWORD                LcBoardMask, LcDspMask ;

    LcLPtrRep = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReq = (LPDSP_USE_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    // Get application index in driver data structures
    LcAppIndex = DecodeHandle( (LcLPtrReq->Header).hdHandle );

    LcBoardMask = (DWORD) LcLPtrReq->duqBoardMask ;

    while ( LcBoardMask != 0 )
    {
        // Get board number and verify it
        // -------------------------------
        LcCurrentBoard = UTIDWMask2Word( LcBoardMask );
        LcBoardMask &= ~UTIMaskDWord( LcCurrentBoard );
        if ( LcCurrentBoard >= MAX_BOARD )
        {
            LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_BOARD );
            continue;
        }

        LcDspMask = (DWORD) LcLPtrReq->duqDspMask ;

        while ( LcDspMask != 0 )
        {
            // Get Dsp number and verify it
            // -----------------------------
            LcDsp   = UTIDWMask2Word( LcDspMask );
            LcDspMask &= ~UTIMaskDWord( LcDsp );
            if ( LcDsp >= MAX_BOARD_DSP )
            {
                LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_DSP );
                continue;
            }

            // Process deallocate command
            // **************************
            LcCanBeReleased = TRUE;

            // #29/01/97 #FH bug correction:
            // If the Dsp was still used by an application,
            // an other application wich does'nt use this DSP
            // was authorized to deallocate it.

            // Verify if the application use this DSP
            // ---------------------------------------
            LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[LcDsp]) ;
            if ( LcPtrDsp->dsMaskIndexAppli != 0L )
            {
                if ( (LcPtrDsp->dsMaskIndexAppli & UTIMaskDWord(LcAppIndex)) == 0L )
                {
                    // fill the response bloc
                    // **********************
                    LcCanBeReleased = FALSE;
                    LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_FREE_DSP_REFUSED );
                    continue;
                }
            }

            // Seeking for allocated ressources ( Input stream )
            // if any exists, return an error
            // --------------------------------------------------
            for ( k = 0; k < MAX_BOARD_INPUTS; k++ )
            {
                if ( TbVoieInInfo[LcCurrentBoard][k].IndexAppli == (LcAppIndex + 1) )
                {
                    // fill the response bloc
                    // **********************
                    LcCanBeReleased = FALSE;
                    LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_FREE_DSP_REFUSED );
                    continue;
                }
            }

            // Seeking for allocated ressources ( Output stream )
            // if any exists, return an error
            // ---------------------------------------------------
            for ( k = 0; k < MAX_BOARD_OUTPUTS; k++ )
            {
                if ( TbVoieOutInfo[LcCurrentBoard][k].IndexAppli == (LcAppIndex + 1) )
                {
                    // fill the response bloc
                    // **********************
                    LcCanBeReleased = FALSE;
                    LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_FREE_DSP_REFUSED );
                    continue;
                }
            }

            if ( LcCanBeReleased )
            {
                // **********************************
                // It's Ok, all application resources
                // has been deallocated
                // **********************************
                LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[LcDsp]) ;

                // Substract the handle application from DSP structure
                // handle application
                // ----------------------------------------------------
                LcPtrDsp->dsMaskIndexAppli &= ~UTIMaskDWord( LcAppIndex );

                // FS - 18/11/96
                // If no more application are using this DSP
                // and it has crashed,reset its running state
                // mark it as free
                // ------------------------------------------
                if (   ( LcPtrDsp->dsMaskIndexAppli == 0 )
                    && ( LcPtrDsp->dsFatalError != 0 ) )
                {
                    LcPtrDsp->dsEtat = 0;
                    LcPtrDsp->dsFatalError = 0;
                }
            }

        }
        // end while there is a DSP
    }
    // end while there is a board
}

// in nphdwdm.cpp
//
extern GENERAL_INFO m_GeneralInfo ;

// *************************************************************************
//
// STATIC VOID DspManageFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour : General command request bloc
//
// *************************************************************************
//
//  Process the Dsp manage command.
//  Update the Dsp manage command response bloc.
//
// *************************************************************************
STATIC VOID DspManageFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPDSP_USE_REQ_INFO   LcLPtrReq      ;
    LPRESP_HEADER_INFO   LcLPtrRep      ;
    LPBC_HEADER_INFO     LcLPtrReqHeader;
    PBOARD_INFO          LcPtrBoard     ;
    LPBUFF_DESC_INFO     LcLPtrBuffDsp  ;
    BYTE                 LcAppIndex     ;
    WORD                 LcCurrentBoard, LcDsp ;
    PDSP_INFO            LcPtrDsp       ;
    WORD                 k              ;
    BOOL                 LcRetLoad      ;
    WORD                 LcRet          = SUCCESS;
    // 10/02/97 - Use pseudo reply block under NT
    LPADD_DESC_INFO      LcAddrDescTab  = NULL ;

    LcLPtrRep = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReq = (LPDSP_USE_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    // 10/02/97 - Use pseudo reply block under NT
    LcAddrDescTab    = (LPADD_DESC_INFO)
                            ((LPANY_UNMAPPING_REPLY_INFO)LcLPtrRep)->urAddrDescriptors;
    BZERO2( LcAddrDescTab, ADD_DESC_INFO, 1 );

    LcLPtrRep->rhSize = 0;

    // Get application index
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Get the Carte structure pointer
    LcPtrBoard = APH_Board_Info_Array;

    // Get board number and verify it
    // -------------------------------
    LcCurrentBoard = UTIDWMask2Word( (DWORD) LcLPtrReq->duqBoardMask );

    if( (LcCurrentBoard >= MAX_BOARD) ||
		(APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_BOARD );
        return;
    }

    // Get Dsp number and verify it
    // -----------------------------
    LcDsp   = UTIDWMask2Word( (DWORD) LcLPtrReq->duqDspMask );

    if ( LcDsp >= MAX_BOARD_DSP )
    {
        LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_DSP );
        return;
    }

    // Let's check whether it is a use / load / release command
    // -----------------------------------------------------------
    if ( LcLPtrReq->duqNbBuff == CMD_DSP_DEALLOC )
    {
        // ******************
        // Deallocate command
        // ******************
        DspDeallocSsFct( PmBlocRequete, PmBlocRetour );
    }
    else if ( LcLPtrReq->duqNbBuff == 0 )
    {
        // ***********
        // Use command
        // ***********

        LcPtrDsp = &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[LcDsp]) ;

        ASSERT(LcDsp==0);

        // If DSP is already loaded and running then store
        // the use command otherwise returns an error
        // ------------------------------------------------------
        if ( (CUR_PROTOCOL_PTR) && (LcPtrDsp->dsEtat > 0) && (LcPtrDsp->dsFatalError == 0) )
        {
            // We can use a DSP that do not have Time Code
            // activated yet
            // -------------------------------------------
            if ( ( LcLPtrReq->duqDspFeatures & DSP_CAN_TIME_CODE_MASK ) == DSP_CAN_TIME_CODE_MASK )
            {
                // Authorize the time-code on the board DSP.
                // Only one DSP may manage a TimeCode signal
                //
                LcRet = CUR_PROTOCOL_PTR->IClock_ManageSignal( SMPTE_PRESENT_MASK );
            }
            else
            {
                // Invalidate TimeCode if not requested by any application
                //
                if ( ( LcPtrDsp->dsMaskIndexAppli == 0 ) || ( LcPtrDsp->dsMaskIndexAppli == UTIMaskDWord( (WORD)LcAppIndex ) ) )
                {
                    // Software invalidate the DSP handlers for LTC
                    // and Hardware disconnect TimeCode from the board
                    //
                    (void) CUR_PROTOCOL_PTR->IClock_ManageSignal( 0 );
                }

            }

            // Add the application handle to DSP users field
            // ---------------------------------------------
            LcPtrDsp->dsMaskIndexAppli |= UTIMaskDWord( (WORD)LcAppIndex );

            LcLPtrRep->rhCptr = LcRet ;

        }
        else
        {
            LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_USE_DSP_REFUSED );
            return;
        }
    }
    else if ( (LcLPtrReq->duqNbBuff > 0) &&
              (LcLPtrReq->duqNbBuff < MAX_BUFF_LIMIT) )
    {
        // ****************
        // Download command
        // ****************

        LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_LOAD_DSP_IMPOSSIBLE );
        return;

    } // Download command

}


// *************************************************************************
//
// STATIC VOID DspResourcesFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the DSP resources usage get command
//
// *************************************************************************
STATIC VOID DspResourcesFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPDSP_RESOURCES_REQ_INFO		LcLPtrReq      ;
    LPDSP_RESOURCES_RESP_INFO		LcLPtrRespInfo ;
    LPRESP_HEADER_INFO				LcLPtrRep      ;
    LPBC_HEADER_INFO				LcLPtrReqHeader;
    PBOARD_INFO          LcPtrBoard     ;
    TARGET_INFO          LcTargetInf    ;
    WORD                 LcCurrentBoard, LcDsp ;
    PDSP_INFO            LcPtrDsp       ;
    WORD                 LcRet          = SUCCESS;
	
    LcLPtrRep		= (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrRespInfo	= (LPDSP_RESOURCES_RESP_INFO)PmBlocRetour;

    LcLPtrReq = (LPDSP_RESOURCES_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    LcLPtrRep->rhSize = 0;

    // Get the Carte structure pointer
    LcPtrBoard = APH_Board_Info_Array;

	// Get board number and verify it
    // -------------------------------
    LcCurrentBoard = UTIDWMask2Word(LcLPtrReq->drqBoardMask);
	
    if( (LcCurrentBoard >= MAX_BOARD) || (CUR_PROTOCOL_PTR == NULL) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_BOARD );
        return;
    }
	
    // Fill target info structure
    // **************************
    LcTargetInf.tgCarte = LcCurrentBoard;

	LcRet = CUR_PROTOCOL_PTR->IDiag_DspResourcesUsage( LcLPtrRespInfo );

    if (LcRet)
	{
		LOAD_PCX_ERROR( LcLPtrRep->rhCptr, LcRet );
		return;
	}

	LcLPtrRep->rhCptr = LcRet;

	return;
}


// *************************************************************************
//
// STATIC VOID GetDspBuffersFct( IN LPDWORD, OUT LPDWORD )
//
// Input  Parameters:
// ******************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output  Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  Retrieves information about addresses of reserved buffers
//
// *************************************************************************
STATIC VOID GetDspBuffersFct(
    IN LPDWORD  PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPDSP_BUFF_RESP_INFO LcPtrRep;
    LPDSP_BUFF_REQ_INFO  LcPtrReq;
    WORD                 i, LcRepSize;
    LPADD_DESC_INFO      LcPtrAddr;

    LcPtrReq = (LPDSP_BUFF_REQ_INFO) PmBlocRequete ;
    LcPtrRep = (LPDSP_BUFF_RESP_INFO) PmBlocRetour ;

    // Check the reply bloc will be long enough to hold the reply
    // if not, stop allocating since the buffer will be lost
    // ----------------------------------------------------------
    LcRepSize = sizeof( DSP_BUFF_RESP_INFO );
    if ( LcPtrReq->Header.hdBrSize < LcRepSize )
    {
        LcPtrRep->RespHeader.rhCptr = ED_REPLY_BLOC_TOO_SHORT;
        return;
    }

    (LcPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

    LcPtrRep->dbReservedBuffers = 0;
    return;
}

// *************************************************************************
//
// STATIC VOID DriverPathFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
//
//Return value:
// ************
//
// *************************************************************************
//
//  Process the Dsp filename command.
//  Update the Dsp filename command response bloc.
//
// *************************************************************************
STATIC VOID DriverPathFct (
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPDRIVER_PATH_RESP_INFO LcLPtrRep;
    LPDRIVER_PATH_REQ_INFO  LcLPtrReq;
    LPSTR                   LcLPtrStr;
    LPSTR                   LcDrvStr;
    BYTE                    LcStringSize;

    LcLPtrRep = (LPDRIVER_PATH_RESP_INFO)PmBlocRetour;
    LcLPtrReq = (LPDRIVER_PATH_REQ_INFO)PmBlocRequete;

    // FS - 15/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( DRIVER_PATH_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    (LcLPtrRep->RespHeader).rhSize = sizeof( DRIVER_PATH_RESP_INFO )
                   - sizeof( RESP_HEADER_INFO );

    // Copy path to the DSP files
    // --------------------------
    for ( LcStringSize = 0, LcDrvStr = GeneralDriverInfo.drPath,
                              LcLPtrStr = &(LcLPtrRep->dnpFileName[0])  ;
          (*LcDrvStr != '\0') && (LcStringSize < MAX_DSP_FILE_NAME_LEN) ;
          LcStringSize++, LcDrvStr ++, LcLPtrStr ++                     )
    {
        *LcLPtrStr = *LcDrvStr ;
    }

    // NULL-terminate the string
    // --------------------------
    if ( LcStringSize < MAX_DSP_FILE_NAME_LEN - 1 )
    {
        LcLPtrStr[LcStringSize] = L'\0';
    }
    else
    {
        LcLPtrStr[MAX_DSP_FILE_NAME_LEN - 1] = L'\0';
        LcStringSize --;
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
    }

    // Fill the size of the PATH+NAME DSP file
    LcLPtrRep->dnpSize = LcStringSize;
}


// *************************************************************************
//
// STATIC VOID ListAppliFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
//
//Return value:
// ************
//
// *************************************************************************
//
//  Process the application list command.
//  Update the Application list command response bloc.
//
// *************************************************************************
// FH 25/20/97 V3 updating
// *************************************************************************
STATIC VOID ListAppliFct(   INOUT LPDWORD PmBlocRequete,
                            INOUT LPDWORD PmBlocRetour )
{
    LPAPP_LIST_REQ_INFO  LcLPtrReq;
    LPAPP_LIST_RESP_INFO LcLPtrRep;
    LPAPP_DESC           LcLPtrApp;
    WORD                 i;
    BYTE                 LcNbAppliFind;
    WORD                 LcRepSize = 0;     // Cumulative size of reply
    BOOLEAN              LcPrivateHandle = FALSE;
    BOOLEAN              LcAuthorizedHandle = FALSE;


    LcLPtrReq = (LPAPP_LIST_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPAPP_LIST_RESP_INFO)PmBlocRetour;

    // FS - 14/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReq->alqHeader.hdBrSize < sizeof( APP_LIST_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    LcRepSize = sizeof( APP_LIST_RESP_INFO );
    (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

    LcLPtrApp = (LPAPP_DESC)(LcLPtrRep + 1);

    // Get application handle
    // ----------------------
    if ( LcLPtrReq->alqHeader.hdHandle == DRIVER_KEY )
    {
        LcPrivateHandle = TRUE;
        LcAuthorizedHandle = TRUE;
    }

    for ( i=0, LcNbAppliFind = 0 ; i < MAX_PCX_HANDLE ; i++ )
    {
        // If the application handle is DRIVER_KEY ( the powerfull handle )
        // we seek for all applications ( public and private ), otherwise
        // we only seek for public one.
        // ------------------------------------------------------------------
        if ( LcPrivateHandle == FALSE )
        {
            LcAuthorizedHandle =
              ((TbAppliInfo[i].Carac & AI_PRIVATE_ATTRIB_MASK) == 0) ? TRUE : FALSE;
        }

        if ( (TbAppliInfo[i].Carac & AI_ALLOCATED_MASK) && LcAuthorizedHandle )
        {
            // One more application encountered
            // --------------------------------
            LcNbAppliFind++;

            // FS - 14/10/1996
            // Check the reply bloc is long enough to hold one more appli
            // ----------------------------------------------------------
            if ( LcLPtrReq->alqHeader.hdBrSize < (LcRepSize + sizeof( APP_DESC )) )
            {
                LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
                continue;
            }

            LcRepSize += sizeof( APP_DESC );
            (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

            LcLPtrApp->adHandle = EncodeHandle( (BYTE) i );


            // Copy application attribut: public(= 0) or private(= 1)
            // this information is important with the magic handle
            // -----------------------------------------------------
            LcLPtrApp->adAttribut =
               ((TbAppliInfo[i].Carac & AI_PRIVATE_ATTRIB_MASK) == 0) ? APP_PUBLIC : APP_PRIVATE;

            if(TbAppliInfo[i].Carac & AI_DHS_ATTRIB_MASK)
                LcLPtrApp->adAppInfoEx = APP_INFO_EX_DHS;

            // ## FS (08/12/1998) -- Add Unicode support
            // -----------------------------------------
            // copy the application name
            //
            MEMCPY(&(LcLPtrApp->adAppName),
                   &(TbAppliInfo[i].TbNomAppli),
                   sizeof(APP_STR_INFO));

            // copy the window handle and window message
            // -----------------------------------------
            LcLPtrApp->adWindowHandle = TbAppliInfo[i].WindowHandle;
            LcLPtrApp->adWindowMessage = TbAppliInfo[i].WindowMessage;

            LcLPtrApp++;
        }
    }

    // Fill the number of public application
    LcLPtrRep->alpNbHandle = LcNbAppliFind;

}

// *************************************************************************
//
// STATIC VOID ListPipeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// ******************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  Process the command 'List of Pipes' and returns the list of pipes
// defined for a given application handle
//
// *************************************************************************
// FH 25/20/97 V3 updating
// *************************************************************************
STATIC VOID ListPipeFct(    IN  LPDWORD PmBlocRequete,
                            OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_LIST_REQ_INFO    LcLPtrReq   ;
    LPPIPE_LIST_RESP_INFO   LcLPtrRep   ;
    LPPIPE_DESC_INFO        LcPPipe     ;
    WORD                    i, k        ;
    WORD                    LcAppNum    ;
    WORD                    LcPipeIndex ;
    BYTE                    LcPipesNum  ;
    WORD                    LcRepSize   = 0;     // Cumulative size of reply
    DWORDLONG               LcMaskPipes ;
    DWORD                   LcNbPipe    ;
    PPIPE_INFO              LcPtPipe    ;
    PVOIE_INFO              LcPtVoie    ;
    LPBYTE                  LcPtPipeNum ;
    BOOLEAN                 LcAuthorizedHandle;

    LcLPtrReq = (LPPIPE_LIST_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPPIPE_LIST_RESP_INFO)PmBlocRetour;

    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( PIPE_LIST_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    LcRepSize = sizeof( PIPE_LIST_RESP_INFO );
    (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);


    // Get the application handle on which we want pipe list
    // -----------------------------------------------------
    LcAppNum = DecodeHandle( LcLPtrReq->plqHandle );

    //{ OCT : check LcAppNum
    if(LcAppNum >= MAX_PCX_HANDLE)
    {
        LcLPtrRep->RespHeader.rhCptr = ED_INVALID_PCX_HANDLE;
        return;
    }
    //} OCT

    // Get requested application handle
    // if = 0,          we can return only public application pipe list
    // if = DRIVER_KEY, we return application pipe list even for private application
    // ------------------------------------------------------------------------------
    if ( LcLPtrReq->Header.hdHandle == DRIVER_KEY )
    {
        LcAuthorizedHandle = TRUE;
    }
    else LcAuthorizedHandle =
            ((TbAppliInfo[LcAppNum].Carac & AI_PRIVATE_ATTRIB_MASK) == FALSE) ? TRUE : FALSE;

    if (  ( LcAppNum >= MAX_PCX_HANDLE )
        || ( TbAppliInfo[LcAppNum].Carac == 0 )
        || !LcAuthorizedHandle )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_PCX_HANDLE );
        return;
    }

    // Points to the 1st reply structure
    // ---------------------------------
    LcPPipe = (LPPIPE_DESC_INFO)(LcLPtrRep + 1);

    // FS : 13/01/1997
    // Same processing for both Out and In pipes : duplicate code
    // removed. Loop twice instead : out pipes first, then in pipes
    // ------------------------------------------------------------
    for ( k = 0 ; k < 2 ; k++ )
    {
        if( k == 0 ) {
            LcPtPipe = &TbPipeOutInfo[0]        ;
            LcNbPipe = MAX_DRV_OUTPIPE          ;
            LcPtPipeNum = &(LcLPtrRep->plpNbOut);
        } else {
            LcPtPipe = &TbPipeInInfo[0]         ;
            LcNbPipe = MAX_DRV_INPIPE           ;
            LcPtPipeNum = &(LcLPtrRep->plpNbIn) ;
            break;
        }

        // Handle checked, let's retrieve the pipes owned by this application
        // along a full scan of the VOIE_INFO in/out structures
        // Build a mask of pipes
        // -------------------------------------------------------------------
        LcMaskPipes = 0;
        LcPipesNum = 0;

        for ( i = 0 ; i < LcNbPipe ; i++ )
        {
            if( LcPtPipe[i].piNbAudio == 0 )
                continue;

            BYTE LcCurrentBoard = LcPtPipe[i].piNumBoard;
            BYTE LcFirstAudio = LcPtPipe[i].piNumPipeForDsp;

            ASSERT( LcCurrentBoard < MAX_BOARD );
            if( k == 0 ) {
                ASSERT( LcFirstAudio < MAX_BOARD_OUTPUTS );
                LcPtVoie = &TbVoieOutInfo[LcCurrentBoard][LcFirstAudio];
            } else {
                ASSERT( LcFirstAudio < MAX_BOARD_INPUTS );
                LcPtVoie = &TbVoieInInfo[LcCurrentBoard][LcFirstAudio];
            }
            ASSERT( i == LcPtVoie->NumPipe );

            // Seeking for audio owned by this appli
            // -------------------------------------
            if ( LcPtVoie->IndexAppli == (LcAppNum + 1) )
            {
                LcMaskPipes |= UTIMask64bit( i );
            }
        }

        // Read the mask of pipes and retrieve data from
        // corresponding PIPE_INFO structure
        // ----------------------------------------------
        while ( LcMaskPipes != 0 )
        {
            // One more pipe encountered
            // --------------------------
            LcPipesNum++;
            LcPipeIndex = UTI64Mask2Word( LcMaskPipes );

            // remove this pipe from the mask
            // -------------------------------
            LcMaskPipes &= ~UTIMask64bit( LcPipeIndex );

            // Check the reply block is long enough to hold the information
            // -------------------------------------------------------------
            LcRepSize += sizeof( PIPE_DESC_INFO );
            if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
            {
                LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
                continue;
            }

            (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

            // Copy the relevant information
            // ------------------------------
            LcPPipe->pdPipeMask64.QuadPart = UTIMask64bit( LcPipeIndex );
            LcPPipe->pdNbIO = LcPtPipe[LcPipeIndex].piNbAudio       ;
            LcPPipe->pdMaxStream = LcPtPipe[LcPipeIndex].piNbMaxFlux;
            LcPPipe->pdPipeAttrib =  ( k==0 ) ? PIPE_OUT : PIPE_IN;
            LcPPipe->pdPipeAttrib |= ( LcPtPipe[LcPipeIndex].piPipeIsOffline == FALSE ) ? PIPE_ONLINE : PIPE_OFFLINE;

            // next description if any
            // ------------------------
            LcPPipe ++;

        } // while there are out pipes found

        // Register the total number of in/out pipes
        // ---------------------------------------
        *LcPtPipeNum = LcPipesNum;
    }
    // end of loop for out then in pipes
    // ---------------------------------

}

// *************************************************************************
//
// STATIC VOID ListSyncedPipesFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// ******************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  Process the command 'List of Sync'ed Pipes' and returns the list of pipes
// synchronized all together for a given application handle
//
// *************************************************************************
STATIC  VOID    ListSyncedPipesFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour    )
{
    LPSYNCED_PIPE_LIST_REQ_INFO    LcLPtrReq   ;
    LPSYNCED_PIPE_LIST_RESP_INFO   LcLPtrRep   ;
    LPSYNCED_PIPE_DESC_INFO        LcPPipe     ;
    PPIPE_INFO                     LcPtPipe    ;
    PVOIE_INFO                     LcPtVoie    ;
    PDWORDLONG                     LcPPipeMaskDone   ;
    DWORDLONG                      LcOutPipeMaskDone = 0 ;
    DWORDLONG                      LcInPipeMaskDone  = 0 ;
    DWORD                          LcNbPipe    ;
    DWORDLONG                      LcMaskPipes ;
    DWORDLONG                      LcThisPipeMask    ;
    WORD                           LcThisPipeIndex   ;
    WORD                           LcOper      ;
    WORD                           i, k        ;
    WORD                           LcAppNum    ;
    WORD                           LcRepSize   = 0;     // Cumulative size of reply

    LcLPtrReq = (LPSYNCED_PIPE_LIST_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPSYNCED_PIPE_LIST_RESP_INFO)PmBlocRetour;

    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof(SYNCED_PIPE_LIST_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    LcRepSize = sizeof( SYNCED_PIPE_LIST_RESP_INFO );
    (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);
    LcLPtrRep->splpNbOfSyncedGroups = 0;

    // Get the application handle on which we want pipe list
    // -----------------------------------------------------
    LcAppNum = DecodeHandle( LcLPtrReq->splqHandle );
    if (  ( LcAppNum >= MAX_PCX_HANDLE ) || ( TbAppliInfo[LcAppNum].Carac == 0 ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_PCX_HANDLE );
        return;
    }

    // Points to the 1st reply structure
    // ---------------------------------
    LcPPipe = (LPSYNCED_PIPE_DESC_INFO)(LcLPtrRep + 1);

    // FS : 13/01/1997
    // Same processing for both Out and In pipes : duplicate code
    // removed. Loop twice instead : out pipes first, then in pipes
    // ------------------------------------------------------------
    for ( k = 0 ; k < 2 ; k++ )
    {
        if( k == 0 ) {
            LcPtPipe        = &TbPipeOutInfo[0] ;
            LcNbPipe        = MAX_DRV_OUTPIPE   ;
            LcPPipeMaskDone = &LcOutPipeMaskDone;
            LcOper          = OPER_PLAY         ;
        } else {
            LcPtPipe        = &TbPipeInInfo[0]  ;
            LcNbPipe        = MAX_DRV_INPIPE    ;
            LcPPipeMaskDone = &LcInPipeMaskDone ;
            LcOper          = OPER_REC          ;
        }

        // Handle checked, let's retrieve the pipes owned by this application
        // along a full scan of the VOIE_INFO in/out structures
        // Build a mask of pipes
        // -------------------------------------------------------------------
        LcMaskPipes = 0;

        for ( i = 0 ; i < LcNbPipe ; i++ )
        {
            if( LcPtPipe[i].piNbAudio == 0 )
                continue;

            BYTE LcCurrentBoard = LcPtPipe[i].piNumBoard;
            BYTE LcFirstAudio = LcPtPipe[i].piNumPipeForDsp;

            ASSERT( LcCurrentBoard < MAX_BOARD );
            if( k == 0 ) {
                ASSERT( LcFirstAudio < MAX_BOARD_OUTPUTS );
                LcPtVoie = &TbVoieOutInfo[LcCurrentBoard][LcFirstAudio];
            } else {
                ASSERT( LcFirstAudio < MAX_BOARD_INPUTS );
                LcPtVoie = &TbVoieInInfo[LcCurrentBoard][LcFirstAudio];
            }
            ASSERT( i == LcPtVoie->NumPipe );

            // Seeking for audio owned by this appli
            // -------------------------------------
            if ( LcPtVoie->IndexAppli == (LcAppNum + 1) )
            {
                LcMaskPipes |= UTIMask64bit( i );
            }
        }

        // Read the mask of pipes and retrieve data from
        // corresponding PIPE_INFO structure
        // ----------------------------------------------
        while ( LcMaskPipes != 0 )
        {
            // One more pipe encountered
            // --------------------------
            LcThisPipeIndex = UTI64Mask2Word( LcMaskPipes );
            LcThisPipeMask = UTIMask64bit( LcThisPipeIndex );

            // remove this pipe from the mask
            // -------------------------------
            LcMaskPipes &= ~LcThisPipeMask;

            // If pipe has been handled already, skip it
            // -----------------------------------------
            if ( LcThisPipeMask & (*LcPPipeMaskDone) )
            {
                continue;
            }

            // Check the reply block is long enough to hold the information
            // -------------------------------------------------------------
            LcRepSize += sizeof( SYNCED_PIPE_DESC_INFO );
            if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
            {
                LcLPtrRep->RespHeader.rhCptr = WD_REPLY_TRUNCATED;
                continue;
            }
            (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

            // Copy the relevant information
            // ------------------------------
            (LcLPtrRep->splpNbOfSyncedGroups)++ ;
            if ( LcPtPipe[LcThisPipeIndex].piClockSource != CLOCK_TYPE_NONE )
            {
                // Pipe synchronized, describe the group
                //
                LcPPipe->spdOutPipeMask64.QuadPart = LcPtPipe[LcThisPipeIndex].piPipeOutClockMask ;
                LcPPipe->spdInPipeMask64.QuadPart  = LcPtPipe[LcThisPipeIndex].piPipeInClockMask ;
            }
            else
            {
                // Pipe unsynchronized, describe this pipe alone
                //
                LcPPipe->spdOutPipeMask64.QuadPart = ( LcOper == OPER_PLAY ) ? LcThisPipeMask : 0 ;
                LcPPipe->spdInPipeMask64.QuadPart  = ( LcOper == OPER_PLAY ) ? 0 : LcThisPipeMask ;
            }

            // Remember the pipes have been encountered
            // ----------------------------------------
            LcOutPipeMaskDone |= LcPPipe->spdOutPipeMask64.QuadPart ;
            LcInPipeMaskDone  |= LcPPipe->spdInPipeMask64.QuadPart ;

            // next description if any
            // ------------------------
            LcPPipe ++;

        } // while there are out pipes found

    }
    // end of loop for out then in pipes
    // ---------------------------------

}

// *************************************************************************
//
// STATIC VOID SystemFeatureFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the System feature command.
//
// *************************************************************************
STATIC VOID SystemFeatureFct(
    INOUT LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    LPSYSTEM_REQ_INFO       LcLPtrReq;
    LPSYSTEM_RESP_INFO      LcLPtrRep;
    WORD                    i;

    LcLPtrReq = (LPSYSTEM_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPSYSTEM_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( SYSTEM_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    (LcLPtrRep->RespHeader).rhSize = sizeof( SYSTEM_RESP_INFO )
                   - sizeof( RESP_HEADER_INFO );

    // Fill up the System feature structure
    // with the General Driver Info
    // -------------------------------------
    LcLPtrRep->sfRelease = GeneralDriverInfo.drRelease;
    LcLPtrRep->sfNVers   = GeneralDriverInfo.drNver;
    LcLPtrRep->sfTVers   = GeneralDriverInfo.drTver;

    // Retrieve the buffer general information
    // ----------------------------------------
    BUFGetFeatures( &LcLPtrRep->sfAllocBuff,
                    &LcLPtrRep->sfAvailBuff,
                    &LcLPtrRep->sfBuffSize );

    LcLPtrRep->sfTypeMem = MEM_NT_NONPAGED;

    // Compute the mask of boards detected and managed by the driver
    // --------------------------------------------------------------
    LcLPtrRep->sfPresentBoard = 0;
    for ( i = 0; i < MAX_BOARD; i++ )
    {
    if ( APH_Board_Info_Array[i].biCarac == CARAC_USED )
        LcLPtrRep->sfPresentBoard |= UTIMaskWord( i );
    }

}

// *************************************************************************
//
// STATIC VOID BoardHardFeatureFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the Hardware feature command for a given board.
//
// *************************************************************************
STATIC VOID BoardHardFeatureFct(
    INOUT LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    LPBOARD_HARD_REQ_INFO       LcLPtrReq;
    LPBOARD_HARD_RESP_INFO      LcLPtrRep;
    PBOARD_INFO                 LcPBoard;
    WORD                        i, LcCurrentBoard;

    LcLPtrReq = (LPBOARD_HARD_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPBOARD_HARD_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( BOARD_HARD_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    (LcLPtrRep->RespHeader).rhSize = sizeof( BOARD_HARD_RESP_INFO )
                   - sizeof(RESP_HEADER_INFO);

    // Verify that the board requested is valid
    //  and actually managed by the driver
    // -----------------------------------------
    if(LcLPtrReq->bhrBoardMask == 0)
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }
    LcCurrentBoard = UTIDWMask2Word( (DWORD) LcLPtrReq->bhrBoardMask );

    if (   ( LcCurrentBoard >= MAX_BOARD )
        || ( APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED ) )
    {
        LcLPtrRep->bhfBoardType = UNKNOWN_BOARD;
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }

    // Board is valid
    // ---------------
    LcPBoard = &(APH_Board_Info_Array[LcCurrentBoard]);

    // Copy the board description into the reply bloc
    // ----------------------------------------------

    LcLPtrRep->bhfBoardType     = CUR_COMMANDS_PTR->PIOGetType();
    LcLPtrRep->bhfExtendedType  = CUR_COMMANDS_PTR->PIOGetTypeNoAlias( );
    LcLPtrRep->bhfBus           = CUR_COMMANDS_PTR->PIOGetBusType();
    LcLPtrRep->bhfSn            = CUR_COMMANDS_PTR->PIOGetUERSyncNumber();
    LcLPtrRep->bhfNbWcl         = CUR_COMMANDS_PTR->PIOGetWCKSyncNumber();
    LcLPtrRep->bhfMiscFeatures  = CUR_COMMANDS_PTR->PIOGetMiscFeatures();
    LcLPtrRep->bhfMiscFeatures2 = CUR_COMMANDS_PTR->PIOGetMiscFeatures2();
    LcLPtrRep->bhfOptions       = CUR_COMMANDS_PTR->PIOGetOptions();
    LcLPtrRep->bhfDspNum        = 1;    //  LcPBoard->biNbDsp;

	VIOGetMACAddress( LcCurrentBoard, (PBYTE)LcLPtrRep->bhfMACAddress );

    CUR_COMMANDS_PTR->PIOGetESConfig( &LcLPtrRep->bhfEtherSound0 );

	DWORD	dummy1;	// no need for freq info;
	BOOLEAN	dummy2; // no need for firstchange info;
	    
	CUR_PROTOCOL_PTR->IClock_BoardGetClockFrequency( &dummy1, &dummy2, &LcLPtrRep->bhfEtherSound1 );
	
    // Send back the CSID stored in BOARD_INFO
    //
    LcLPtrRep->bhfCSID          = LcPBoard->biCSID;

    // Copy the I/O address ranges for the board
    // ------------------------------------------

    LcLPtrRep->bhfIO[0].ioStart = PtrToUlong(LcPBoard->portMEM);
    LcLPtrRep->bhfIO[0].ioEnd   = PtrToUlong(LcPBoard->portMEM) + LcPBoard->portMEMsize;
    LcLPtrRep->bhfIO[1].ioStart = PtrToUlong(LcPBoard->portPLX);
    LcLPtrRep->bhfIO[1].ioEnd   = PtrToUlong(LcPBoard->portPLX) + LcPBoard->portPLXsize;
    // does not exist any more
    //LcLPtrRep->bhfIO[2].ioStart = PtrToUlong(LcPBoard->portDSP);
    //LcLPtrRep->bhfIO[2].ioEnd   = PtrToUlong(LcPBoard->portDSP) + LcPBoard->portDSPsize;

    // IRQ number used by the board
    // -----------------------------
    LcLPtrRep->bhfIrqNum = (WORD)LcPBoard->biNumIrq;

    // Retrieve Phys. Audio output/input description
    // ----------------------------------------------
    for ( i = 0; i < MAX_BOARD_OUTPUTS ; i++ )
    {
        LcLPtrRep->bhfOutAudio[i] = (BYTE) CUR_COMMANDS_PTR->PIOGetDescAudioOut(i);

        // MBR(01/07/04) : Support for ciPhysicalOutFeature2:
		LcLPtrRep->bhfOutAudio2[i] = CUR_COMMANDS_PTR->PIOGetPhysOutFeature2(i);
    }

    for ( i = 0; i < MAX_BOARD_INPUTS ; i++ )
    {
        WORD LcDescAudioIn = CUR_COMMANDS_PTR->PIOGetDescAudioIn(i);
        LcLPtrRep->bhfInAudio[i] = (BYTE)(LcDescAudioIn & 0x00FF);
        LcLPtrRep->bhfExtraInAudio[i] = (BYTE)(LcDescAudioIn>>8);

        // MBR(29/05/02) : Support for ciPhysicalInFeature2:
		LcLPtrRep->bhfInAudio2[i] = CUR_COMMANDS_PTR->PIOGetPhysInFeature2(i);
    }

    // check if this entry point fully describes the board.
    //
    WORD nb_ins, nb_outs;

    CUR_COMMANDS_PTR->PIOGetNbIOs( &nb_outs, &nb_ins);

    if ((nb_outs > MAX_BOARD_OUTPUTS) || (nb_ins > MAX_BOARD_INPUTS))
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_MORE_DATA );
}

// *************************************************************************
//
// STATIC VOID BoardSoftFeatureFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the Software feature command for a given (Board, Dsp).
//
// *************************************************************************
STATIC VOID BoardSoftFeatureFct(
    INOUT LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    LPBOARD_SOFT_REQ_INFO   LcLPtrReq;
    LPBOARD_SOFT_RESP_INFO  LcLPtrRep;
    WORD                    LcRepSize;
    PBOARD_INFO             LcPBoard;
    PDSP_INFO               LcPBoardDsp;
    LPDRV_DSP_INFO          LcPDsp;
    LPDSP_SOFT_INFO         LcPDspSoft;
    DWORD                   LcAppMask;
    WORD                    LcBoardNum;

    LcLPtrReq = (LPBOARD_SOFT_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPBOARD_SOFT_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    LcRepSize = sizeof( BOARD_SOFT_RESP_INFO );
    if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

    // Verify that the board requested is valid
    //  and actually managed by the driver
    // -----------------------------------------
    LcBoardNum = UTIDWMask2Word( (DWORD) LcLPtrReq->bsrBoardMask );
    if ( ( LcBoardNum >= MAX_BOARD )
    || ( APH_Board_Info_Array[LcBoardNum].biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }

    // Board is valid
    // ---------------
    LcPBoard = &(APH_Board_Info_Array[LcBoardNum]);

    // Fill up the fixed-size part of the reply block
    // -----------------------------------------------
    LcLPtrRep->bsfDspNum = 1; //LcPBoard->biNbDsp;

    // Fill as many DSP structure as DSP on the board
    // ----------------------------------------------
    
    LcPDsp = (LPDRV_DSP_INFO)(LcLPtrRep + 1);

  /*  for ( i = 0,  ;LcPDsp = ..
      i < LcPBoard->biNbDsp ;
      i++ )
    {*/
        // Check the reply block is large enough to hold
        // the remaining information
        // ------------------------------------------------
        LcRepSize += sizeof( DRV_DSP_INFO );
        if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
            return;
        }
        (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof( RESP_HEADER_INFO );

        // Copy general DSP Info
        // ----------------------
        LcPBoardDsp = &(LcPBoard->biTbDspInfo[0]);

        LcPDsp->ddiDspState = LcPBoardDsp->dsEtat;

        // BTW, check the DSP software is actually running
        // if not, then mark it as crashed
        // ------------------------------------------------
        if (  ( LcPBoardDsp->dsEtat != 0 )
            && ( LcPBoardDsp->dsFatalError != 0 ) )
            LcPDsp->ddiDspState = DSP_STATE_CRASHED;

        // Compute the number of user applications for the
        // mask in DSP_INFO structure
        // ------------------------------------------------
        LcPDsp->ddiAppNum = 0;
        LcAppMask = LcPBoardDsp->dsMaskIndexAppli;
        while ( LcAppMask != 0 )
        {
            // there is one more application using the Dsp
            // --------------------------------------------
            LcPDsp->ddiAppNum++;

            // remove this application from the mask
            // --------------------------------------
            LcAppMask &= ~UTIMaskDWord( UTIDWMask2Word( LcAppMask ) );
        }

        // Copy the software DSP info if this DSP is loaded
        // and running (no fatal error has occured since last load)
        // ---------------------------------------------------------
        if (  ( LcPBoardDsp->dsEtat != 0 )
            && ( LcPBoardDsp->dsFatalError == 0 ) )
        {
            LcPDspSoft = (LPDSP_SOFT_INFO)(LcPDsp + 1);

            // Check the reply block is large enough to hold
            // the remaining information
            // ------------------------------------------------
            LcRepSize += sizeof( DSP_SOFT_INFO );
            if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
            {
                LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, WD_REPLY_TRUNCATED );
                return;
            }
            (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof( RESP_HEADER_INFO );

            // Copy software DSP capabilities
            // -------------------------------
            LcPDspSoft->dsiManagedPhysOut = LcPBoardDsp->dsManagedPhysicalOutputNumber;
            LcPDspSoft->dsiManagedVirtOut = LcPBoardDsp->dsManagedVirtualOutputNumber;
            LcPDspSoft->dsiManagedPhysIn =  LcPBoardDsp->dsManagedPhysicalInputNumber;
            LcPDspSoft->dsiManagedVirtIn =  LcPBoardDsp->dsManagedVirtualInputNumber;
            LcPDspSoft->dsiPhysOut = (DWORD)LcPBoardDsp->dsMaskVPOut;
            LcPDspSoft->dsiVirtOut = (DWORD)LcPBoardDsp->dsMaskVVOut;
            LcPDspSoft->dsiPhysIn =  (DWORD)LcPBoardDsp->dsMaskVPIn;
            LcPDspSoft->dsiVirtIn =  (DWORD)LcPBoardDsp->dsMaskVVIn;
            LcPDspSoft->dsiMaxBoardOutStream = LcPBoardDsp->dsMaxBoardOutStream;
            LcPDspSoft->dsiMaxPipeOutStream = LcPBoardDsp->dsMaxPipeOutStream;
            LcPDspSoft->dsiFirmwareVersion = LcPBoardDsp->dsDspVersion;

           // LcPDsp = (LPDRV_DSP_INFO)(LcPDspSoft + 1);

        }       // if DSP is loaded and running
       /* else
        {
            LcPDsp++;
        }       // else DSP is either not loaded or crashed

     } // for each DSP on board*/

}

// *************************************************************************
//
// STATIC VOID BoardSyncClocksFct(INOUT LPDWORD , INOUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// *****************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the get board synchro clock inputs command
//
// *************************************************************************
STATIC VOID BoardSyncClocksFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPGET_BOARD_CLOCK_REQ_INFO  LcLPtrReq;
    LPGET_BOARD_CLOCK_RESP_INFO LcLPtrRep;
    WORD                        LcRepSize;
    PBOARD_CLOCK_INFO           LcBoardClockInfo;
    WORD                        LcCurrentBoard;
    WORD                        LcRet;
    SHORT                       LcLastBoardClockIndex;
    WORD                        LcSynchroNum;
    int                         LcNumberOfUERSync;

    LcLPtrReq = (LPGET_BOARD_CLOCK_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPGET_BOARD_CLOCK_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    LcRepSize = sizeof( GET_BOARD_CLOCK_RESP_INFO );

    if ( LcLPtrReq->Header.hdBrSize < LcRepSize )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    (LcLPtrRep->RespHeader).rhSize = LcRepSize - sizeof(RESP_HEADER_INFO);

    // Let's zeroe the reply
    // ----------------------
    BZERO2( LcLPtrRep->bcpClock, BOARD_CLOCK_INFO, MAX_BOARD_CLOCK_INPUTS );

    LcLPtrRep->bcpNbClockInputs = 0;

    // Verify that the board requested is valid
    //  and actually managed by the driver
    // -----------------------------------------
    LcCurrentBoard = UTIDWMask2Word( (DWORD) LcLPtrReq->bcrBoardMask );

    if (   ( LcCurrentBoard >= MAX_BOARD )
        || ( APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }

    // Read the Uer Synchro input
    // --------------------------

    LcNumberOfUERSync = (int) CUR_COMMANDS_PTR->PIOGetUERSyncNumber();

    LcLastBoardClockIndex = -1;

    if( LcNumberOfUERSync )
    {
		LcNumberOfUERSync--;
        // One more descriptor is significative
        //
        LcLastBoardClockIndex = BOARD_CLOCK_UER ;

        LcBoardClockInfo = &LcLPtrRep->bcpClock[BOARD_CLOCK_UER];
        LcBoardClockInfo->bcType = CLOCK_TYPE_UER_SYNCHRO;

        LcRet = CUR_COMMANDS_PTR->PIOReadUerSync(
                0, // LcSynchroNum,
                &(LcBoardClockInfo->bcSync),
                &(LcBoardClockInfo->bcFrequency));

        if ( LcRet != SUCCESS )
        {
            LcLPtrRep->RespHeader.rhCptr = LcRet ;
        }
    }

    // Read the Word Clock Synchro input
    // ---------------------------------
    if ( CUR_COMMANDS_PTR->PIOGetWCKSyncNumber() )
    {
        // One more descriptor is significative
        //
        LcLastBoardClockIndex = BOARD_CLOCK_WCL ;

        LcBoardClockInfo = &LcLPtrRep->bcpClock[BOARD_CLOCK_WCL];
        LcBoardClockInfo->bcType = CLOCK_TYPE_WORD_CLOCK;

        LcRet = CUR_COMMANDS_PTR->PIOReadWordClockSync(
                0, // LcSynchroNum,
                &(LcBoardClockInfo->bcSync),
                &(LcBoardClockInfo->bcFrequency));

        if ( LcRet != SUCCESS )
        {
            LcLPtrRep->RespHeader.rhCptr = LcRet ;
        }
    }

    // Read the Word Clock Synchro input
    // ---------------------------------
    if ( CUR_COMMANDS_PTR->PIOGetVideoSyncNumber() )
    {
        // One more descriptor is significative
        //
        LcLastBoardClockIndex = BOARD_CLOCK_VIDEO ;

        LcBoardClockInfo = &LcLPtrRep->bcpClock[BOARD_CLOCK_VIDEO];
        LcBoardClockInfo->bcType = CLOCK_TYPE_VIDEO;

        LcRet = CUR_COMMANDS_PTR->PIOReadVideoSync(
			0, // LcSynchroNum,
			&(LcBoardClockInfo->bcSync),
			&(LcBoardClockInfo->bcFrequency));

        if ( LcRet != SUCCESS )
        {
            LcLPtrRep->RespHeader.rhCptr = LcRet ;
        }
    }

    // Read the Ethersound Clock Synchro input
    // ---------------------------------
    if ( CUR_COMMANDS_PTR->PIOGetEthersoundSyncNumber() )
    {
        // One more descriptor is significative
        //
        LcLastBoardClockIndex = BOARD_CLOCK_ETHERSOUND ;

        LcBoardClockInfo = &LcLPtrRep->bcpClock[BOARD_CLOCK_ETHERSOUND];
        LcBoardClockInfo->bcType = CLOCK_TYPE_ETHERSOUND;

#pragma message("TODO - BoardSyncClocksFct(CLOCK_TYPE_ETHERSOUND)")
		LcBoardClockInfo->bcSync = DRV_SYNC_PRESENT;
        LcBoardClockInfo->bcFrequency = CUR_COMMANDS_PTR->PIOGetActualClockFrequency();
    }

    // Read the remaining Uer Synchro inputs
    // ------------------------------------
    LcSynchroNum = 0;

    while ( LcNumberOfUERSync-- )
    {
            // One more descriptor is significative
            //
			LcLastBoardClockIndex = BOARD_CLOCK_UER_IN1 + LcSynchroNum ;

            LcBoardClockInfo = &LcLPtrRep->bcpClock[LcLastBoardClockIndex];
            LcBoardClockInfo->bcType = CLOCK_TYPE_UER_SYNCHRO;

            LcSynchroNum++;	// start PIOReadUerSync with LcSynchroNum = 1 !

            LcRet = CUR_COMMANDS_PTR->PIOReadUerSync(
                LcSynchroNum,
                &(LcBoardClockInfo->bcSync),
                &(LcBoardClockInfo->bcFrequency));

            if ( LcRet != SUCCESS )
            {
                LcLPtrRep->RespHeader.rhCptr = LcRet ;
            }
    }

	LcLPtrRep->bcpNbClockInputs = LcLastBoardClockIndex + 1;
}

// ************************************************************************
//
// STATIC VOID RegDeregFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Paramaters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
//
//Return value:
// ************
//
// *************************************************************************
//
//  Process the Register\Deregister command.
//  Update the Register\Deregister command response bloc.
//
// *************************************************************************
// FH 25/02/97:   V3 updating
//
// ## FS (04/09/1998) -- FA #175, FA #203, FA #207
// In order to manage error cases in a much more readable and safer
// way, the release loop for each pipe will be handled at only one place, that
// is at the API level.
//
// *************************************************************************
STATIC VOID RegDeregFct(INOUT LPDWORD PmBlocRequete,
            INOUT LPDWORD PmBlocRetour)
{
    LPREG_REQ_INFO LcLPtrReq;
    PBOARD_INFO    LcPtrBoard;
    LPREG_REP_INFO LcLPtrRep;

    BYTE           i,j;
    BOOLEAN        LcFound;
    DWORD          LcMaskHandleCou;

    LcLPtrReq = (LPREG_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPREG_REP_INFO)PmBlocRetour;

    // FS - 14/10/1996
    // Check the reply bloc will be long enough to hold the reply
    // ----------------------------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( REG_REP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    // Reply Bloc is long enough, set length of returned data
    (LcLPtrRep->RespHeader).rhSize =  sizeof(REG_REP_INFO)
                    - sizeof(RESP_HEADER_INFO);

    // Get the Carte structure pointer
    LcPtrBoard = APH_Board_Info_Array;

    // Parsing Register / Deregister command request bloc
    if ( LcLPtrReq->Header.hdHandle == REGISTER_HANDLE )
    {
        // *************************
        // It's a Register command
        // ***********************
        DOUT(DBG_PRINT, ("Register"));

        // this test is probably useless since the API
        // most surely tests the right MAX_SIZE
        // ------------------------------------------------
        if ( LcLPtrReq->rgqNameSize > MAX_APPNAME_SIZE )
        {
            LcLPtrRep->rgpHandle = 0;
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_APPLICATION_NAME_SIZE );
            return ;
        }

        // Verify the attributes correctness
        // ----------------------------------
        if ( (LcLPtrReq->rgqAttributes & ~ALL_APPLICATION_ATTRIB_MASK) != 0 )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_UNKNOWN_APPLICATION_ATTRIB );
            return ;
        }

        // seek for a free Application structure
        // *************************************
        for ( i=0, LcFound = FALSE ;
              ( i < MAX_PCX_HANDLE ) && ( LcFound == FALSE ) ;
              i++ )
        {
            if ( TbAppliInfo[i].Carac == FREE_ENTRY )
            {
                LcFound = TRUE;
                TbAppliInfo[i].Carac = AI_ALLOCATED_MASK;

                if ( LcLPtrReq->rgqAttributes & PRIVATE_HANDLE_MASK )
                {
                    TbAppliInfo[i].Carac |= AI_PRIVATE_ATTRIB_MASK;
                }

                // copy the application name
                //
                MEMCPY(&(TbAppliInfo[i].TbNomAppli),
                    &(LcLPtrReq->rgqAppName),
                    sizeof(APP_STR_INFO));

                // Copy the handle window and message window
                // ******************************************
                TbAppliInfo[i].WindowHandle = LcLPtrReq->rgqWindowHandle;
                TbAppliInfo[i].WindowMessage = LcLPtrReq->rgqWindowMessage;

                // fill the response bloc
                // **********************

                // New Handle of the query application
                LcLPtrRep->rgpHandle = EncodeHandle( i );

                LcLPtrRep->RespHeader.rhCptr = SUCCESS;

                // DHS application registered
                //
                if(LcLPtrReq->rgqAppInfoEx & APP_INFO_EX_DHS)
                {
                    DHSAppHandle = i;
                    TbAppliInfo[i].Carac |= AI_DHS_ATTRIB_MASK;
                }
            }
        }

        if( LcFound == FALSE )
        {
            // this value indicates that the command has been processed
            // but with an error
            LcLPtrRep->rgpHandle = 0;
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_DECL_APP_IMPOSSIBLE );
        }
    }
    else
    {
        // *************************
        // It's a Deregister command
        // *************************
        WORD    LcAppliIndex;
        BYTE    LcFirstAudio;
        BYTE    LcCurrentBoard;

        LcAppliIndex = DecodeHandle( LcLPtrReq->Header.hdHandle );

        // Get the corresponding application handle mask
        // ---------------------------------------------
        LcMaskHandleCou = UTIMaskDWord( LcAppliIndex );

        // ***********************************************
        // RAZ the corresponding Voie (input and output),
        // Pipe (input and output) and stream structures
        // ***********************************************
        for ( i = 0 ; i < MAX_DRV_INPIPE ; i++ )
        {
            if ( TbPipeInInfo[i].piNbAudio == 0 )
                continue;

            LcFirstAudio = TbPipeInInfo[i].piNumPipeForDsp;
            LcCurrentBoard = TbPipeInInfo[i].piNumBoard;

            ASSERT( LcFirstAudio < MAX_BOARD_INPUTS );
            ASSERT( LcCurrentBoard < MAX_BOARD );
            ASSERT( TbVoieInInfo[LcCurrentBoard][LcFirstAudio].IndexAppli != 0 );

            if ( TbVoieInInfo[LcCurrentBoard][LcFirstAudio].IndexAppli == (LcAppliIndex + 1) )
            {
                // Set no sampling clock for all the pipes
                // clocked with this one
                // ## FM (12/10/98) -- This is now done in the API
                // -----------------------------------
                if ( TbPipeInInfo[i].piClockSource != CLOCK_TYPE_NONE )
                {
                    // ## FS
                    //
                    DOUT(DBG_ERROR, ("CLOCK_TYPE_NONE Pbm at unregister time"));
                }

                // free the pipe if allocated
                // --------------------------

                BZERO2( &TbPipeInInfo[i], PIPE_INFO, 1 );
                BZERO2( &TbVoieInInfo[LcCurrentBoard][LcFirstAudio], VOIE_INFO, TbPipeInInfo[i].piNbAudio );
            }
        }

        for ( i = 0 ; i < MAX_DRV_OUTPIPE ; i++ )
        {
            if ( TbPipeOutInfo[i].piNbAudio == 0 )
                continue;

            LcFirstAudio = TbPipeOutInfo[i].piNumPipeForDsp;
            LcCurrentBoard = TbPipeOutInfo[i].piNumBoard;

            ASSERT( LcFirstAudio < MAX_BOARD_OUTPUTS );
            ASSERT( LcCurrentBoard < MAX_BOARD );
            ASSERT( TbVoieOutInfo[LcCurrentBoard][LcFirstAudio].IndexAppli != 0 );

            if ( TbVoieOutInfo[LcCurrentBoard][LcFirstAudio].IndexAppli == (LcAppliIndex + 1) )
            {
                // Free a possibly waiting end of play
                // and buffers if pipe exists
                // release the clocks held on the board
                // ------------------------------------

                // Set no sampling clock for all the pipes
                // clocked with this one
                // ## FM (12/10/98) -- This is now done in the API
                // -----------------------------------
                if ( TbPipeOutInfo[i].piClockSource != CLOCK_TYPE_NONE )
                {
                    // ## FS
                    //
                    DOUT(DBG_ERROR, ("CLOCK_TYPE_NONE Pbm at unregister time"));
                }


                // free the pipe if allocated
                // --------------------------

                BZERO2( &TbVoieOutInfo[LcCurrentBoard][LcFirstAudio], VOIE_INFO, TbPipeOutInfo[i].piNbAudio );
                BZERO2( &TbPipeOutInfo[i], PIPE_INFO, 1 );
            }
        }

        // Substract this handle from DSP structure handle application
        // -----------------------------------------------------------
        for ( i=0 ; i < MAX_BOARD; i++, LcPtrBoard++ )
        {
            for ( j=0 ; j < MAX_BOARD_DSP ; j++ )
            {
                if ( (LcPtrBoard->biTbDspInfo[j].dsMaskIndexAppli
                       & LcMaskHandleCou) != 0 )
                {
                     LcPtrBoard->biTbDspInfo[j].dsMaskIndexAppli &=
                                                           (~LcMaskHandleCou);
                }
            }
        }

        // Release a waiting error if any
        // ------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReq != NULL )
        {
            LPWAIT_ERROR_REQ_INFO   LcLPtrErrReq ;
            LPWAIT_ERROR_RESP_INFO  LcLPtrErrRepHeader ;

            // Update and free the Response block
            // ----------------------------------
            LcLPtrErrReq =
                (LPWAIT_ERROR_REQ_INFO) TbAppliInfo[LcAppliIndex].AdrBlkReq ;
            LcLPtrErrRepHeader =
                (LPWAIT_ERROR_RESP_INFO)LcLPtrErrReq->weqForDriver.deeAdrBlkResp;

            SignalAsyncProcessed( (LPDWORD) LcLPtrErrRepHeader                 ,
                                  (PDWORD) LcLPtrErrReq->weqForDriver.deeIrp   );
        }


        // Release a wait for board pnp events if any
        // --------------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReqBoardPnp != NULL )
        {
            APHCancelBoardPnpEvent( LcAppliIndex );
        }

        // Release a wait for GPIO events if any
        // -------------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReqGpio != NULL )
        {
            APHCancelGpioEvent( LcAppliIndex );
        }

        // Close the GPIOs opened by the application.
        // ------------------------------------------
        for ( i = 0 ; i < MAX_BOARD; i++ )
        {
            if ( APH_Board_Info_Array[i].biGpioAppliIndex == LcAppliIndex )
            {
                APH_Board_Info_Array[i].biGpioAppliIndex = MAX_PCX_HANDLE;
                APH_Board_Info_Array[i].biGpioEvent      = FALSE;
                APH_Board_Info_Array[i].biGpioNotifyMask = 0x00000000;   // no Input bits to notify
            }
        }

        // unregister DHS app
        if(TbAppliInfo[LcAppliIndex].Carac & AI_DHS_ATTRIB_MASK)
        {
            ASSERT(DHSAppHandle == LcAppliIndex);
            DHSInit(TRUE);
            DOUT(DBG_PRINT, ("DHS app unregister OK !\n"));
        }

        // Free the corresponding application structure
        // --------------------------------------------
        BZERO2( &TbAppliInfo[LcAppliIndex], APPLI_INFO, 1 );

        // fill the response bloc
        // **********************
        LcLPtrRep->rgpHandle = 0;
        LcLPtrRep->RespHeader.rhCptr = SUCCESS;
    }

}


// *************************************************************************
//
// STATIC VOID EffectContextFct(INOUT LPDWORD , INOUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : General command request bloc
//
// Output Parameters:
// ******************
//
//  LPDWORD PmBlocRetour  : General command response bloc
//
//
// Return value:
// ************
//
// *************************************************************************
//
//  Process the Load Effect Context command.
//  Update the reply bloc.
//
// *************************************************************************
// *************************************************************************
STATIC VOID EffectContextFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader ;

    // Initialisation of pointers
    // --------------------------
    LcLPtrRepHeader  = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhCptr = ED_UNAVAILABLE_FEATURE;
    return;
}

// *************************************************************************
//
// STATIC VOID TimeCodeFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the TIME_CODE command for a given board.
//
// *************************************************************************
STATIC VOID TimeCodeFct(
    INOUT  LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    PTIME_CODE_REQ_INFO    LcLPtrReq;
    PTIME_CODE_RESP_INFO   LcLPtrRep;
    WORD                   LcCurrentBoard;
    BOOLEAN                LcNewTimeCode = FALSE ;
    BOOLEAN                LcBackward = FALSE ;
    BOOLEAN                LcConfigInProgress = FALSE ;

    LcLPtrReq = (PTIME_CODE_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (PTIME_CODE_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( TIME_CODE_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    (LcLPtrRep->RespHeader).rhSize = sizeof( TIME_CODE_RESP_INFO )
                   - sizeof(RESP_HEADER_INFO);

    // Verify that the board requested is valid
    //  and actually managed by the driver
    // -----------------------------------------
    LcCurrentBoard = UTIDWMask2Word( (DWORD) LcLPtrReq->tcoBoardMask );

    if (   ( LcCurrentBoard >= MAX_BOARD )
        || ( APH_Board_Info_Array[LcCurrentBoard].biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }

    switch ( LcLPtrReq->tcoRequest )
    {
    case TIME_CODE_CMD_GET:

        LcLPtrRep->RespHeader.rhCptr = CUR_PROTOCOL_PTR->IClock_TimeCodeGet(	&LcNewTimeCode,
																				&LcBackward,
																				&LcConfigInProgress,
																				&(LcLPtrRep->tcoPcxTime),
																				&(LcLPtrRep->tcoTimeCode) );
        LcLPtrRep->tcoNewTimeCode      = (DWORD) LcNewTimeCode ;
        LcLPtrRep->tcoBackward         = (DWORD) LcBackward ;
        LcLPtrRep->tcoConfigInProgress = (DWORD) LcConfigInProgress ;
        break;

    default:
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_CMD_FAM1 );
        break;
    }
}

// *************************************************************************
//
// STATIC VOID GPIOFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the GPIO command for a given board.
//
// ASSUMPTION: ReadWriteGPIO ensure that LcLPtrReq->gpioTabInfo is related to
// the same board and that LcLPtrReq->gpioTabSize > 0.
//
// *************************************************************************
STATIC VOID GPIOFct(
    INOUT  LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
    PGPIO_REQ_INFO      LcLPtrReq;
    PGPIO_RESP_INFO     LcLPtrRep;
    WORD                LcCurrentBoard;
	DWORD				LcPIOMask = 0;
	DWORD				LcPIOData = 0;
	WORD				i;
    WORD                LcAppIndex;
    PWORD               LcPAppIndex;
    PBOARD_INFO         LcPBoardInfo;

    LcLPtrReq = (PGPIO_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (PGPIO_RESP_INFO)PmBlocRetour;

    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( GPIO_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }

    LcLPtrRep->RespHeader.rhSize = sizeof( GPIO_RESP_INFO ) - sizeof(RESP_HEADER_INFO);

    // Verify that the board requested is valid
    //  and actually managed by the driver
    // -----------------------------------------
    LcCurrentBoard = LcLPtrReq->gpioTabInfo[0].gpioBoardNumber;

    LcPBoardInfo = &APH_Board_Info_Array[LcCurrentBoard];

    if (   ( LcCurrentBoard >= MAX_BOARD )
        || ( LcPBoardInfo->biCarac != CARAC_USED ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }

    LcAppIndex = DecodeHandle( LcLPtrReq->Header.hdHandle );

    // Pointer to the index of the application that opens this GPIOs.
    LcPAppIndex = &(LcPBoardInfo->biGpioAppliIndex);
  
    switch ( LcLPtrReq->gpioRequest )
    {
    case GPIO_CMD_WRITE:

        // -----------------------------------------------------------
        // The GPIOs of the board are not opened by that application.
        // -----------------------------------------------------------

        if ( (*LcPAppIndex) != LcAppIndex )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_GPIO_NOT_OPENED );
        }
        else
        {
        // ------------------------------------------------------
        // The GPIOs of the board are opened by that application.
        // ------------------------------------------------------

			for ( i = 0 ; i < LcLPtrReq->gpioTabSize ; i++ )
            {
				// one bit set to 1 represent the same IO number to be set or reset according
				// to the same bit number in the second mask.
				LcPIOMask |= 1 << LcLPtrReq->gpioTabInfo[i].gpioNum ;
				LcPIOData |= ( LcLPtrReq->gpioTabInfo[i].gpioData ) << ( LcLPtrReq->gpioTabInfo[i].gpioNum ) ;
			}
		    LcPIOMask &= GPIO_NP2_WRITE_MASK ;
		    LcLPtrRep->RespHeader.rhCptr = CUR_COMMANDS_PTR->PIOWriteGPIO(LcPIOMask,LcPIOData);
        }
        break;

    case GPIO_CMD_READ:

        // -----------------------------------------------------------
        // The GPIOs of the board are not opened by that application.
        // -----------------------------------------------------------

        if ( (*LcPAppIndex) != LcAppIndex )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_GPIO_NOT_OPENED );
        }
        else
        {
        // ------------------------------------------------------
        // The GPIOs of the board are opened by that application.
        // ------------------------------------------------------

            // Read the GPIOs.
            // ---------------
			for ( i = 0 ; i < LcLPtrReq->gpioTabSize ; i++ )
            {
				// one bit set to 1 represent the IO number to be read
				LcPIOMask  |= 1 << LcLPtrReq->gpioTabInfo[i].gpioNum ;
			}
		    LcPIOMask &= GPIO_NP2_READ__MASK ;

            // look if there is unread data in the Fifo
            if( LcPBoardInfo->biGpioFifoReadIndex != LcPBoardInfo->biGpioFifoWriteIndex )
            {
                LcPIOData = LcPBoardInfo->biGpioFifo[LcPBoardInfo->biGpioFifoReadIndex];
                LcPBoardInfo->biGpioFifoReadIndex ++;
                if( LcPBoardInfo->biGpioFifoReadIndex >= GPIO_NP2_FIFO_SIZE )
                {
                    LcPBoardInfo->biGpioFifoReadIndex = 0;
                }
                LcPIOData &= LcPIOMask;

                // if there rests data in the Fifo we return WD_MORE_DATA !
                if(LcPBoardInfo->biGpioFifoReadIndex == LcPBoardInfo->biGpioFifoWriteIndex)
                {
                    LcLPtrRep->RespHeader.rhCptr = SUCCESS;
                }
                else
                {
                    LcLPtrRep->RespHeader.rhCptr = WD_MORE_DATA;
                }
            }
            else    // otherwise read on the port
            {
		        LcLPtrRep->RespHeader.rhCptr = CUR_COMMANDS_PTR->PIOReadGPIO(LcPIOMask, &LcPIOData);
            }

            // Fill the response bloc.
            // -----------------------
			for ( i = 0 ; i < LcLPtrReq->gpioTabSize ; i++ )
            {
				MEMCPY( LcLPtrRep->gpioTabInfo + i, LcLPtrReq->gpioTabInfo + i, sizeof(ELEM_GPIO_INFO) );
				if ( LcPIOData & (1<<LcLPtrReq->gpioTabInfo[i].gpioNum) )
						LcLPtrRep->gpioTabInfo[i].gpioData = 1 ;
				else	LcLPtrRep->gpioTabInfo[i].gpioData = 0 ;
			}
        }
        break;

    case GPIO_CMD_OPEN:

        // ------------------------------------------
        // The GPIOs of the board are already opened.
        // ------------------------------------------
              
        if ( (*LcPAppIndex) != MAX_PCX_HANDLE )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_GPIO_ALREADY_OPENED );
        }
        else
        if ( LcAppIndex >= MAX_PCX_HANDLE )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_PCX_HANDLE );
        }
        else
        {
        // ----------------------------------------------
        // The GPIOs of the board are not already opened.
        // ----------------------------------------------

			for ( i = 0 ; i < LcLPtrReq->gpioTabSize ; i++ )
            {
				// one bit set to 1 represent the same IO number to be set or reset according
				// to the same bit number in the second mask.
				LcPIOMask  |= 1<<LcLPtrReq->gpioTabInfo[i].gpioNum ;
				if (LcLPtrReq->gpioTabInfo[i].gpioType == 1)    // IN GPIO
                {
				    LcPIOData |= 1<<(LcLPtrReq->gpioTabInfo[i].gpioNum) ;
                }
			}

			LcLPtrRep->RespHeader.rhCptr = CUR_COMMANDS_PTR->PIOInitGPIO(LcPIOMask,LcPIOData);

            if ( LcLPtrRep->RespHeader.rhCptr == SUCCESS )
            {
                // --------------------------------------------
                // This GPIOs are now owned by the application.
                // --------------------------------------------

                (*LcPAppIndex) = LcAppIndex;

                // No event for that GPIO bits.
                LcPBoardInfo->biGpioEvent = FALSE;

                LcPBoardInfo->biGpioNotifyLastRead = ~GPIO_NP2_READ__MASK;    // invalid value

                // Initialise the Input Fifo
                LcPBoardInfo->biGpioFifoReadIndex = 0;
                LcPBoardInfo->biGpioFifoWriteIndex = 0;

                // Store the GPIO Input bits that could be notified
                LcPBoardInfo->biGpioNotifyMask = (LcPIOData & GPIO_NP2_READ__MASK);
            }
        }	
        break;

	case GPIO_CMD_CLOSE:

        // ---------------------------------------------------------------
        // The GPIOs of the board are not opened, or are opened but not by
        // that application.
        // ---------------------------------------------------------------
      
        if ( (*LcPAppIndex) != LcAppIndex )
        {
            LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_GPIO_NOT_OPENED );
        }
        else
        {	
        // ------------------------------------------------------
        // The GPIOs of the board are opened by that application.
        // ------------------------------------------------------

            // Set all the GPIOs to INPUT to have a hardware safe state.
			(VOID)CUR_COMMANDS_PTR->PIOInitGPIO(GPIO_NP2_READ__MASK,GPIO_NP2_READ__MASK);

            LcLPtrRep->RespHeader.rhCptr = SUCCESS;

            // ------------------------------------------------------------
            // The GPIOs of the board are now not more owned by the application.
            // ------------------------------------------------------------
            (*LcPAppIndex) = MAX_PCX_HANDLE;

            // No more Input Bits to be notified
            LcPBoardInfo->biGpioNotifyMask = 0x00000000;

            // No event for that board.
            LcPBoardInfo->biGpioEvent = FALSE;
        }
		break;

    case GPIO_CMD_FEATURE:
        {
            DWORD LcReadAccess;
            DWORD LcWriteAccess;
            DWORD LcProgrammable;

            LcLPtrRep->RespHeader.rhCptr = CUR_COMMANDS_PTR->PIOGetGPIOFeatures(&LcReadAccess, &LcWriteAccess, &LcProgrammable);

            if(SUCCESS == LcLPtrRep->RespHeader.rhCptr)
            {
                MEMCPY( LcLPtrRep->gpioTabInfo, LcLPtrReq->gpioTabInfo, sizeof(ELEM_GPIO_INFO) );

                LcLPtrRep->gpioTabInfo[0].gpioUnused1 = LcReadAccess;
                LcLPtrRep->gpioTabInfo[0].gpioUnused2 = LcWriteAccess;
                LcLPtrRep->gpioTabInfo[0].gpioUnused3 = LcProgrammable;
            }
        }
		break;

    default:
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_GPIO_CMD );
        break;
    }
}


// *************************************************************************
//
// STATIC VOID IBLFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the IBL Set/Get commands.
//
// *************************************************************************
STATIC VOID IBLFct(
    INOUT  LPDWORD PmBlocRequete,
    INOUT  LPDWORD PmBlocRetour )
{
    LPIBL_REQ_INFO		LcLPtrReq;
    LPIBL_RESP_INFO		LcLPtrRep;
    WORD				LcCurrentBoard;
    TARGET_INFO			LcTargetInf;
    PBOARD_INFO			LcPBoardInfo;
	
    LcLPtrReq = (LPIBL_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPIBL_RESP_INFO)PmBlocRetour;
	
    // Check the reply block is large enough
    // --------------------------------------
    if ( LcLPtrReq->Header.hdBrSize < sizeof( IBL_RESP_INFO ) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_REPLY_BLOC_TOO_SHORT );
        return;
    }
    (LcLPtrRep->RespHeader).rhSize = sizeof( IBL_RESP_INFO ) - sizeof( RESP_HEADER_INFO );
	
	
	// Get board number and verify it
    // -------------------------------
    LcCurrentBoard = UTIDWMask2Word(LcLPtrReq->iblBoardMask);
    LcPBoardInfo = &APH_Board_Info_Array[LcCurrentBoard];
	
    if( (LcCurrentBoard >= MAX_BOARD ) || (CUR_PROTOCOL_PTR == NULL) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->RespHeader.rhCptr, ED_INVALID_BOARD );
        return;
    }
	
	LcLPtrRep->iblSamples		=  LcLPtrReq->iblSamples ;
	LcLPtrRep->iblMin			=  0;
    LcLPtrRep->iblMax			=  0;
    LcLPtrRep->iblGranularity	=  0;

    // adjust and set IBL.
	//
	LcLPtrRep->RespHeader.rhCptr = CUR_PROTOCOL_PTR->IDiag_HandleBoardIBL( LcLPtrRep );
	
}


// *************************************************************************
//
// STATIC VOID ESHWConfSetFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  handles the ESHardwareConfSet command.
//
// *************************************************************************
STATIC VOID ESHWConfSetFct(
    INOUT  LPDWORD PmBlocRequete,
    INOUT  LPDWORD PmBlocRetour )
{
    LES_HW_CFG_REQ_INFO LcLPtrReq;
    LPRESP_HEADER_INFO  LcLPtrRep;
    WORD				LcCurrentBoard;

    LcLPtrReq = (LES_HW_CFG_REQ_INFO)PmBlocRequete;
    LcLPtrRep = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Get board number and verify it
    // -------------------------------
    LcCurrentBoard = UTIDWMask2Word(LcLPtrReq->ehcBoardMask);

    if( (LcCurrentBoard >= MAX_BOARD ) || (CUR_COMMANDS_PTR == NULL) )
    {
        LOAD_PCX_ERROR( LcLPtrRep->rhCptr, ED_INVALID_BOARD );
        return;
    }

    LcLPtrRep->rhCptr = CUR_COMMANDS_PTR->PIOUpdateESConfig( LcLPtrReq->ehcEtherSound0 );
}


// *************************************************************************
//
// STATIC VOID BoostThreadFct(INOUT LPDWORD , INOUT LPDWORD )
//
//Input / Output Parameters:
// *************************
//
//  LPDWORD PmBlocRequete : General command request bloc
//  LPDWORD PmBlocRetour  : General command response bloc
//
// *************************************************************************
//
//  boosts actual thread to priority 31.
//
// *************************************************************************
STATIC VOID BoostThreadFct(
    INOUT LPDWORD PmBlocRequete,
    INOUT LPDWORD PmBlocRetour )
{
	LPSYSTEM_REQ_INFO   LcLPtrReq = (LPSYSTEM_REQ_INFO)PmBlocRequete;
	LPRESP_HEADER_INFO  LcLPtrRep = (LPRESP_HEADER_INFO)PmBlocRetour;

    DWORD               LcPriority = LcLPtrReq->srqZero;

    if(LcPriority > HIGH_PRIORITY)  LcPriority = HIGH_PRIORITY;
    //DbgPrint("PCXBoostThisThread called with internal para = %d\n", LcPriority);
    KeSetPriorityThread( KeGetCurrentThread(), (KPRIORITY)LcPriority );

    LcLPtrRep->rhCptr = SUCCESS;
    return;
}



VOID APH_CleanUpAppliInfo()
{
    WORD LcAppliIndex;

    for (LcAppliIndex=0; LcAppliIndex < MAX_PCX_HANDLE; LcAppliIndex++)
    {
        // Release a waiting error if any
        // ------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReq != NULL )
        {
            LPWAIT_ERROR_REQ_INFO  LcLPtrErrReq = (LPWAIT_ERROR_REQ_INFO) TbAppliInfo[LcAppliIndex].AdrBlkReq ;
            LPWAIT_ERROR_RESP_INFO LcLPtrErrRep =( LPWAIT_ERROR_RESP_INFO) LcLPtrErrReq->weqForDriver.deeAdrBlkResp;

            LcLPtrErrRep->RespHeader.rhT = PROCESSED ;

            WNTSetEvent((PDWORD) LcLPtrErrReq->weqForDriver.deeIrp, STATUS_DELETE_PENDING );
        }

        // Release a wait for board pnp events if any
        // --------------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReqBoardPnp != NULL )
        {
            LPWAIT_BOARD_PNP_REQ_INFO  LcLPtrReq = (LPWAIT_BOARD_PNP_REQ_INFO) TbAppliInfo[LcAppliIndex].AdrBlkReqBoardPnp ;
            LPWAIT_BOARD_PNP_RESP_INFO LcLPtrRep = (LPWAIT_BOARD_PNP_RESP_INFO) LcLPtrReq->wpqForDriver.deeAdrBlkResp ;

            // Fill Response bloc
            // ******************
            LcLPtrRep->RespHeader.rhCptr = SUCCESS ;
            LcLPtrRep->wprEvent = BOARD_EVENT_NOTHING ;
            LcLPtrRep->wprBoardMask = 0 ;

            WNTSetEvent((PDWORD) LcLPtrReq->wpqForDriver.deeIrp, STATUS_DELETE_PENDING );
        }

        // Release a wait for GPIO events if any
        // -------------------------------------
        if ( TbAppliInfo[LcAppliIndex].AdrBlkReqGpio != NULL )
        {
            LPWAIT_GPIO_REQ_INFO  LcLPtrReq = (LPWAIT_GPIO_REQ_INFO) TbAppliInfo[LcAppliIndex].AdrBlkReqGpio ;
            LPWAIT_GPIO_RESP_INFO LcLPtrRep = (LPWAIT_GPIO_RESP_INFO) LcLPtrReq->wpqForDriver.deeAdrBlkResp ;

            // Fill Response bloc
            // ******************
            LcLPtrRep->RespHeader.rhCptr = SUCCESS ;
            LcLPtrRep->wgrBoardMask = 0; // Arbitrary value.

            WNTSetEvent((PDWORD) LcLPtrReq->wpqForDriver.deeIrp, STATUS_DELETE_PENDING );
        }
    }

    BZERO2(TbAppliInfo, APPLI_INFO, MAX_PCX_HANDLE);
}


