// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// *************************************************************************
//
// STATIC WORD SetOutAudioLevelsSubFct( IN PTARGET_INFO,
//                                      IN DWORDLONG,
//                                      IN WORD,
//                                      IN LPOUT_AUDIO_SET_LEVEL_REQ_INFO )
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTargetPtr : Target board for the command
//  DWORDLONG PmPipeMask     : Output pipe masks
//  WORD  PmAudioMask        : pipe audio
//
// Output Parameters:
// ******************
//
//  none
//
// Return value:
// *************
//
// SUCCESS if it completes successfully, an error code otherwise
//
// *************************************************************************
//
//  Set any levels for output audio
//
// *************************************************************************
STATIC WORD SetOutAudioLevelsSubFct(
    IN  PTARGET_INFO                    PmTargetPtr,
    IN  DWORDLONG                       PmPipeMask,
    IN  WORD                            PmAudioMask,
    IN  LPOUT_AUDIO_SET_LEVEL_REQ_INFO  PmLevelInfo )
{
    LEVEL_AUDIO_INFO                     LcLevelAudioInf;
    WORD                                LcAudioMask;
    BOOLEAN                             LcIsSetAudioNumLevel;
    WORD                                LcPipeIndex ;
    WORD                               LcCurrentBoard ;
	WORD                                i, LcRet;
    DWORDLONG                           LcNumAudioMask = 0;

    // Verify if the command has a delayed effect and try to set analog level,
    // in this case raise an error
    //------------------------------------------------------------------------
    if (   PmTargetPtr->tgDifferedType
        && (PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_ANALOG_MASK) )
    {
        EXIT_PCX_ERROR( ED_DIFFERED_CMD_REFUSED );
    }

    // Prepare digital levels and digital mutes processing if selected
    // ---------------------------------------------------------------
    LcIsSetAudioNumLevel = (PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_NUM_LEVEL) ? TRUE : FALSE;

    if ( LcIsSetAudioNumLevel )
    {
        if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_DIGITAL_MASK )
        {
            LcLevelAudioInf.gaiHasDigitalLevel = TRUE;
            LcLevelAudioInf.gaiDigitalLevel = PmLevelInfo->oasliDigitalLevel;
        }
        else LcLevelAudioInf.gaiHasDigitalLevel = FALSE;

        if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MONITOR_MASK )
        {
            LcLevelAudioInf.gaiHasMonitorLevel = TRUE;
            LcLevelAudioInf.gaiMonitorLevel = PmLevelInfo->oasliMonitorLevel;
        }
        else LcLevelAudioInf.gaiHasMonitorLevel = FALSE;

        if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_MASK )
        {
            LcLevelAudioInf.gaiHasMuteLevel = TRUE;
            LcLevelAudioInf.gaiMuteLevel = PmLevelInfo->oasliMute;
        }
        else LcLevelAudioInf.gaiHasMuteLevel = FALSE;

        if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_M1_MASK )
        {
            LcLevelAudioInf.gaiHasM1Level = TRUE;
            LcLevelAudioInf.gaiM1Level = PmLevelInfo->oasliMonitorMute1;
        }
        else LcLevelAudioInf.gaiHasM1Level = FALSE;

        if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MUTE_M2_MASK )
        {
            LcLevelAudioInf.gaiHasM2Level = TRUE;
            LcLevelAudioInf.gaiM2Level = PmLevelInfo->oasliMonitorMute2;
        }
        else LcLevelAudioInf.gaiHasM2Level = FALSE;
    }

    
	// For all the pipes
    // -----------------
    do
    {
        LcPipeIndex = UTI64Mask2Word( PmPipeMask ) ;
        PmPipeMask &= ~UTIMask64bit( LcPipeIndex );

        PmTargetPtr->tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;

        PVOIE_INFO LcVoieInfo = &TbVoieOutInfo[LcCurrentBoard][TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp];

        // Let's scan for audio allocated by this pipe and
        // apply the set level command if in the scope of request mask
        // -------------------------------------------------------------
        LcAudioMask = PmAudioMask ;
        for ( i = 0 ; i < TbPipeOutInfo[LcPipeIndex].piNbAudio ; i++ )
        {
            if ( LcAudioMask == 0 ) break ;

            ASSERT ( LcVoieInfo[i].IndexAppli != 0 );
            ASSERT ( LcVoieInfo[i].NumPipe == LcPipeIndex );

            // Set levels only if audio is physical, owned by the appli and
            // the pipe, and match the given mask
            // -------------------------------------------------------------
            if (  ( LcVoieInfo[i].Nat == 0 )
                && ( UTIMaskWord( LcVoieInfo[i].AppAudioNum ) & LcAudioMask  ) )
            {
                PmTargetPtr->tgAudio = LcVoieInfo[i].AudioNum ;

                // this audio is controlled by DHS !
                if( APHChkIsAudioDHSControlled(LcCurrentBoard, PmTargetPtr->tgAudio, FALSE) )
                {
                    EXIT_PCX_ERROR( ED_AUDIO_MANAGED_BY_DHS_PANEL );
                }

                // Set analog level if selected
                // ------------------------------
                if ( PmLevelInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_ANALOG_MASK )
                {
                    LcRet = CUR_COMMANDS_PTR->PIOSetAudioAnalogLevel(   PmTargetPtr->tgAudio,
                                                                        OPER_PLAY,
                                                                        PmLevelInfo->oasliAnalogLevel );
                    if ( LcRet != SUCCESS )
                        return LcRet ;
                }

                LcAudioMask &= ~UTIMaskWord( LcVoieInfo[i].AppAudioNum ) ;

                // Build a audio Mask with the autorised audio IO
                // for digital level processing
                //-----------------------------------------------
                LcNumAudioMask |= UTIMask64bit( LcVoieInfo[i].AudioNum );

            }  // if this audio is concerned

        } // for all audio outputs on a given pipe

        // ------------------------------
        // Set digital level if selected
        // ------------------------------
        if ( LcIsSetAudioNumLevel )
        {
            LcRet = CUR_PROTOCOL_PTR->ILevel_AudioSetDigitalLevel(	PmTargetPtr,
																	LcNumAudioMask,
																	&LcLevelAudioInf );
            if ( LcRet != SUCCESS )
            {
                return LcRet;
            }

            // RAZ digital audio mask
            // ----------------------
            LcNumAudioMask = 0;

        } // End of digital level processing

    } while( PmPipeMask ); // for all the pipes of the corresponding mask

    return SUCCESS;
}


// *************************************************************************
//
// STATIC WORD SetInAudioLevelsSubFct( IN PTARGET_INFO,
//                                     IN DWORDLONG,
//                                     IN WORD,
//                                     IN LPOUT_AUDIO_SET_LEVEL_REQ_INFO )
//
// Input Parameters:
// *****************
//
//  PTARGET_INFO PmTargetPtr : Target board for the command
//  DWORDLONG PmPipeMask     : Input pipe masks
//  WORD  PmAudioMask        : pipe audio
//
// Output Parameters:
// ******************
//
//  none
//
// Return value:
// *************
//
// SUCCESS if it completes successfully, an error code otherwise
//
// *************************************************************************
//
//  Set any levels for input audio
//
// *************************************************************************
//
// *IMPORTANT* this function assumes that :
//   - A pipe is on only one card
//   - All the audio I/O belong to the same DSP
//
// *************************************************************************
STATIC WORD SetInAudioLevelsSubFct(
    IN  PTARGET_INFO                    PmTargetPtr,
    IN  DWORDLONG                       PmPipeMask,
    IN  WORD                            PmAudioMask,
    IN  LPIN_AUDIO_SET_LEVEL_REQ_INFO   PmLevelInfo )
{
    LEVEL_AUDIO_INFO                     LcLevelAudioInf;
    WORD                                LcAudioMask;
    WORD                                LcPipeIndex ;
    WORD                                LcCurrentBoard = 0 ;
    WORD                                i, LcRet;
    DWORDLONG                           LcNumAudioMask = 0;

    // Verify if the command has a delayed effect and try to set analog level,
    // in this case raise an error
    //------------------------------------------------------------------------
    if (   PmTargetPtr->tgDifferedType
        && (PmLevelInfo->iasliPar1 & ~IN_AUDIO_SET_LEVEL_DIGITAL_MASK) )
    {
        EXIT_PCX_ERROR( ED_DIFFERED_CMD_REFUSED );
    }

    // Prepare digital level processing if selected
    // --------------------------------------------
    if ( PmLevelInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
    {
        LcLevelAudioInf.gaiHasDigitalLevel = TRUE;
        LcLevelAudioInf.gaiDigitalLevel = PmLevelInfo->iasliDigitalLevel;
    }
    else LcLevelAudioInf.gaiHasDigitalLevel = FALSE;

    LcLevelAudioInf.gaiHasMonitorLevel = FALSE;
    LcLevelAudioInf.gaiHasMuteLevel = FALSE;
    LcLevelAudioInf.gaiHasM1Level = FALSE;
    LcLevelAudioInf.gaiHasM2Level = FALSE;

    // For all the pipes
    // -----------------
    do
    {
        LcPipeIndex = UTI64Mask2Word( PmPipeMask ) ;
        PmPipeMask &= ~UTIMask64bit( LcPipeIndex );

        PmTargetPtr->tgCarte = LcCurrentBoard = TbPipeInInfo[LcPipeIndex].piNumBoard;
//        PmTargetPtr->tgDsp   = 0;

        PVOIE_INFO LcVoieInfo = &TbVoieInInfo[LcCurrentBoard][TbPipeInInfo[LcPipeIndex].piNumPipeForDsp];

        // Let's scan for audio allocated by this pipe and
        // apply the set level command if in the scope of request mask
        // -------------------------------------------------------------
        LcAudioMask = PmAudioMask ;

        for ( i = 0 ; i < TbPipeInInfo[LcPipeIndex].piNbAudio ; i++ )
        {
            if ( LcAudioMask == 0 ) break ;

            ASSERT( LcVoieInfo[i].IndexAppli != 0 );
            ASSERT( LcVoieInfo[i].NumPipe == LcPipeIndex );

            // Set levels only if audio is physical, owned by the appli and
            // the pipe, and match the given mask
            // -------------------------------------------------------------
            if (  ( LcVoieInfo[i].Nat == 0 )
                && ( UTIMaskWord( LcVoieInfo[i].AppAudioNum ) & LcAudioMask  ) )
            {
                PmTargetPtr->tgAudio = LcVoieInfo[i].AudioNum ;

                // this audio is controlled by DHS !
                if( APHChkIsAudioDHSControlled(LcCurrentBoard, PmTargetPtr->tgAudio, TRUE) )
                {
                    EXIT_PCX_ERROR( ED_AUDIO_MANAGED_BY_DHS_PANEL );
                }

                // Set analog level if selected
                // ------------------------------
                if ( PmLevelInfo->iasliPar1 & ~IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
                {
                    LcRet = CUR_COMMANDS_PTR->PIOSetInAudioLevels(  PmTargetPtr->tgAudio, PmLevelInfo );

                    if ( LcRet != SUCCESS )
                    {
                        return LcRet;
                    }
                }

                LcAudioMask &= ~UTIMaskWord( LcVoieInfo[i].AppAudioNum ) ;

                // Build a audio Mask with the autorised audio IO
                // for digital level processing
                //-----------------------------------------------
                LcNumAudioMask |= UTIMask64bit( LcVoieInfo[i].AudioNum );

            }  // if this audio is concerned

        } // for all audio outputs on a given pipe

    // ------------------------------
    // Set digital level if selected
    // ------------------------------

    if ( PmLevelInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
    {
        LcRet = CUR_PROTOCOL_PTR->ILevel_AudioSetDigitalLevel(	PmTargetPtr,
																LcNumAudioMask,
																&LcLevelAudioInf );
        if ( LcRet != SUCCESS )
        {
            return LcRet;
        }

        // RAZ digital audio mask
        // ----------------------
        LcNumAudioMask = 0;

        } // End of digital level processing

    } while( PmPipeMask ); // for all pipes of the corresponding mask

    return SUCCESS;
}

// *************************************************************************
//                  Board Audio IO Control subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd2( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd2(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LcLPResp->rhCptr = ED_INVALID_CMD_FAM2 ;
}

// *************************************************************************
//
// STATIC VOID SetOutBoardAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Set any levels for output audio
//
// *************************************************************************
STATIC VOID SetOutBoardAudioLevelsFct(
        IN  LPDWORD PmBlocRequete,
        OUT LPDWORD PmBlocRetour )
{
    LPOUT_BOARD_AUDIO_SET_LEVELS_REQ_INFO   LcLPtrReq;
    LPBC_HEADER_INFO                        LcLPtrReqHeader;
    LPRESP_HEADER_INFO                      LcLPtrRepHeader;
    TARGET_INFO                             LcTargetInf;
    WORD                                    LcRet   = SUCCESS ;
    BYTE                                    LcAppIndex;
    PVOIE_INFO                              LcPtVoie;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPOUT_BOARD_AUDIO_SET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( OUT_BOARD_AUDIO_SET_LEVELS_REQ_INFO)
                             - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->obslqAudio).paBoard,
                            (LcLPtrReq->obslqAudio).paDsp,
                            (LcLPtrReq->obslqAudio).paAudioNum,
                            (LcLPtrReq->obslqAudio).paAudioAttributes,
                            &LcPtVoie );

    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                // Fill target_info
                // ----------------
                LcTargetInf.tgCaracPipeVoie = OPER_PLAY ;
                LcTargetInf.tgDifferedType = LcLPtrReq->obslqDiffered;
                LcTargetInf.tgHour = LcLPtrReq->obslqHour ;

                // Actually process the request
                // ----------------------------
                LcRet = SetOutAudioLevelsSubFct( &LcTargetInf,
                                                 UTIMask64bit( LcPtVoie->NumPipe ),
                                                 UTIMaskWord( LcPtVoie->AppAudioNum ),
                                                 &(LcLPtrReq->obslqLevelInfo) );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID SetInBoardAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// FH: Creation
//
// *************************************************************************
//
//  Set any levels for input audio
//
// *************************************************************************
STATIC VOID SetInBoardAudioLevelsFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_BOARD_AUDIO_SET_LEVELS_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO                        LcLPtrReqHeader;
    LPRESP_HEADER_INFO                      LcLPtrRepHeader;
    TARGET_INFO                             LcTargetInf;
    PVOIE_INFO                              LcPtVoie;
    WORD                                    LcRet   = SUCCESS ;
    BYTE                                    LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq = (LPIN_BOARD_AUDIO_SET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( IN_BOARD_AUDIO_SET_LEVELS_REQ_INFO)
                 - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Check the pipe is actually owned by the application
    // --------------------------------------------------------
    LcRet = LookupBoardAudio(
                            (LcLPtrReq->ibslqAudio).paBoard,
                            (LcLPtrReq->ibslqAudio).paDsp,
                            (LcLPtrReq->ibslqAudio).paAudioNum,
                            (LcLPtrReq->ibslqAudio).paAudioAttributes,
                            &LcPtVoie );

    if ( LcRet == SUCCESS )
    {
        // Bug fix for FA #95
        // ------------------
        if ( LcPtVoie == NULL )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
        }
        else
        {
            if ( LcPtVoie->IndexAppli != (LcAppIndex + 1) )
            {
                LOAD_PCX_ERROR( LcRet, ED_INVALID_PIPE_AUDIO );
            }
            else
            {
                // Fill target_info
                // ----------------
                LcTargetInf.tgCaracPipeVoie = OPER_REC ;
                LcTargetInf.tgDifferedType = LcLPtrReq->ibslqDiffered;
                LcTargetInf.tgHour = LcLPtrReq->ibslqHour ;

                // Actually process the request
                // ----------------------------
                LcRet = SetInAudioLevelsSubFct( &LcTargetInf,
                                                UTIMask64bit( LcPtVoie->NumPipe ),
                                                UTIMaskWord( LcPtVoie->AppAudioNum ),
                                                &(LcLPtrReq->ibslqLevelInfo) );
            }
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet;
    return;
}


// *************************************************************************
//                  Pipe Audio IO Control subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd4( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Audio Control command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Audio Control command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd4(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM4 );
}


// *************************************************************************
//
// STATIC VOID SetOutAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Set any levels for output audio
//
// *************************************************************************
// This function supposes :
//   - A pipe is on only one card
//   - All the audio I/O belong to the same DSP
//
// *************************************************************************
STATIC VOID SetOutAudioLevelsFct(
        IN  LPDWORD PmBlocRequete,
        OUT LPDWORD PmBlocRetour )
{
    LPOUT_AUDIO_SET_LEVELS_REQ_INFO     LcLPtrReq;
    LPBC_HEADER_INFO                    LcLPtrReqHeader;
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    TARGET_INFO                         LcTargetInf;
    WORD                                LcRet   = SUCCESS ;
    BYTE                                LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPOUT_AUDIO_SET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( OUT_AUDIO_SET_LEVELS_REQ_INFO)
                             - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    if(LcAppIndex == DHSAppHandle)
    {
        LcLPtrRepHeader->rhCptr = DHSSetOutAudioLevels(LcLPtrReq);
        return;
    }
    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyOutputPipeMask( LcAppIndex, LcLPtrReq->oslqPipeMask64.QuadPart );
    if ( LcRet == SUCCESS )
    {
        // Fill target_info
        // ----------------
        LcTargetInf.tgCaracPipeVoie = OPER_PLAY ;
        LcTargetInf.tgDifferedType = LcLPtrReq->oslqDiffered;
        LcTargetInf.tgHour = LcLPtrReq->oslqHour ;

        // Actually process the request
        // ----------------------------
        LcRet = SetOutAudioLevelsSubFct( &LcTargetInf,
                                         LcLPtrReq->oslqPipeMask64.QuadPart,
                                         LcLPtrReq->oslqAudioMask,
                                         &(LcLPtrReq->oslqLevelInfo) );
    }

    LcLPtrRepHeader->rhCptr = LcRet ;
    return;
}


// *************************************************************************
//
// STATIC VOID SetInAudioLevelsFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// FH: Creation
//
// *************************************************************************
//
//  Set any levels for input audio
//
// *************************************************************************
// This function supposes :
//   - A pipe is on only one card
//   - All the audio I/O belong to the same DSP
//
// *************************************************************************
STATIC VOID SetInAudioLevelsFct(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_SET_LEVELS_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO                LcLPtrReqHeader;
    LPRESP_HEADER_INFO              LcLPtrRepHeader;
    TARGET_INFO                     LcTargetInf;
    WORD                            LcRet   = SUCCESS ;
    BYTE                            LcAppIndex;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq = (LPIN_AUDIO_SET_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( IN_AUDIO_SET_LEVELS_REQ_INFO)
                 - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    if(LcAppIndex == DHSAppHandle)
    {
        LcLPtrRepHeader->rhCptr = DHSSetInAudioLevels(LcLPtrReq);
        return;
    }

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInputPipeMask( LcAppIndex, LcLPtrReq->islqPipeMask64.QuadPart );
    if ( LcRet == SUCCESS )
    {
        // Fill target_info
        // ----------------
        LcTargetInf.tgCaracPipeVoie = OPER_REC ;
        LcTargetInf.tgDifferedType = LcLPtrReq->islqDiffered;
        LcTargetInf.tgHour = LcLPtrReq->islqHour ;

        // Actually process the request
        // ----------------------------
        LcRet = SetInAudioLevelsSubFct( &LcTargetInf,
                                         LcLPtrReq->islqPipeMask64.QuadPart,
                                         LcLPtrReq->islqAudioMask,
                                         &(LcLPtrReq->islqLevelInfo) );
    }

    LcLPtrRepHeader->rhCptr = LcRet;
    return;
}


// *************************************************************************
//
// STATIC VOID SetInPipeAudioEffect( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// MBR: Creation
//
// *************************************************************************
//
//  Set any effects for input audio
//
// *************************************************************************
// This function supposes :
//   - A pipe is on only one card
//   - All the audio I/O belong to the same DSP
//
// *************************************************************************
STATIC VOID     SetInPipeAudioEffect(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_EFFECT_REQ_INFO  LcLPtrRequest;
    LPRESP_HEADER_INFO          LcLPtrRetHeader;
    WORD                        LcRet   = SUCCESS ;
    BYTE                        LcAppIndex;
    WORD                        LcPipeIndex;
    DWORD                       LcPipeMask = 0x0;
    WORD                        LcAudioMask;
    BYTE                        LcCurrentBoard;
    WORD        i, LcAudioNum;
    PPIPE_INFO                  LcPipeInfo;
    PVOIE_INFO                  LcVoieInfo;
    TARGET_INFO                 LcTarget;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRequest   = (LPIN_AUDIO_EFFECT_REQ_INFO)  PmBlocRequete;
    LcLPtrRetHeader = (LPRESP_HEADER_INFO)          PmBlocRetour;

    LcAppIndex = DecodeHandle( LcLPtrRequest->Header.hdHandle );
    LcPipeMask =  LcLPtrRequest->iaqHandle.iaqInPipeMask;
    LcAudioMask = LcLPtrRequest->iaqInAudioMask ;

    // test if only one pipe
    //
    LcRet = ChkPmOneInputPipeMask(  LcAppIndex,
                                    LcPipeMask,
                                    &LcPipeIndex,
                                    &LcPipeInfo,
                                    &LcVoieInfo,
                                    &LcTarget );
    if ( LcRet == SUCCESS )
    {
        LcCurrentBoard = LcPipeInfo->piNumBoard;

        // Let's scan for audio allocated by this pipe and
        // apply the set level command if in the scope of request mask
        // -------------------------------------------------------------
        
        for ( i = 0 ; i < LcPipeInfo->piNbAudio ; i++ )
        {
            if ( LcAudioMask == 0 ) break ;

            ASSERT( LcVoieInfo[i].IndexAppli != 0 );
            ASSERT( LcVoieInfo[i].NumPipe == LcPipeIndex );

            // Set effect only if audio is physical, owned by the appli and
            // the pipe, and match the given mask
            // -------------------------------------------------------------
            if ( UTIMaskWord( LcVoieInfo[i].AppAudioNum ) & LcAudioMask )
            {
                LcAudioNum = LcVoieInfo[i].AudioNum ;

                // Set effect if possible
                // ------------------------
                switch( LcLPtrRequest->iaqIdentifier)
                {
                case HW_EFFECT_NONE:
                case HW_EFFECT_COMPRESSOR_LIMITER:
                    if(LcLPtrRequest->iaqCommand == EFFECT_CMD_SET )
                    {
                        LcRet = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
                    } else // remove, get not allowed
                    {
                        LcLPtrRequest->iaqIdentifier = HW_EFFECT_NONE;
                        LcRet = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
                    } 
                    break;
                    
                    LcRet = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
                    break;
                    
                case HW_EFFECT_MAXIMIZER:
                case HW_EFFECT_EQUALIZER:
                    LcRet = CUR_PROTOCOL_PTR->IFlow_AudioInputEffect(LcAudioNum, LcLPtrRequest);
                    break;
                default:
                    ASSERT(0);
                }
                
                LcAudioMask &= ~UTIMaskWord( LcVoieInfo[i].AppAudioNum ) ;
            }               // if this audio is concerned
        }                   // for all audio inputs on a given pipe
    }

    LcLPtrRetHeader->rhCptr = LcRet;
    return;
} //SetInPipeAudioEffect

// *************************************************************************
//
// STATIC VOID SetInBoardAudioParam( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// MBR: Creation
//
// *************************************************************************
//
//  Set any parameters(Mic48V) for input audio
//
// *************************************************************************
STATIC VOID     SetInBoardAudioParam(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_PARAM_REQ_INFO   LcLPtrRequest;
    LPRESP_HEADER_INFO          LcLPtrRetHeader;
    TARGET_INFO                 LcTargetInf;
    DWORD                       LcBoardMask;
    WORD                        LcRet   = SUCCESS ;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRequest   = (LPIN_AUDIO_PARAM_REQ_INFO)   PmBlocRequete;
    LcLPtrRetHeader = (LPRESP_HEADER_INFO)          PmBlocRetour;
    
    if( LcLPtrRequest->iapqValidationMask != IN_AUDIO_SET_PARAM_MIC48V_MASK )
    {
	    LOAD_PCX_ERROR( LcLPtrRetHeader->rhCptr, ED_BAD_VALIDATION_MASK);
	    return;
    }

    LcBoardMask = LcLPtrRequest->iapqInBoardMask;

    BZERO2( &LcTargetInf, TARGET_INFO, 1 );

    // For all the boards
    // ------------------
    while ( LcBoardMask && (LcRet == SUCCESS) )
    {
        WORD LcAudioMask = LcLPtrRequest->iapqInAudioMask;
        WORD LcCurrentBoard;
        
        LcTargetInf.tgCarte = LcCurrentBoard = UTIDWMask2Word( LcBoardMask );

        LcBoardMask &= ~UTIMaskDWord( LcCurrentBoard );

        while ( LcAudioMask && (LcRet == SUCCESS) )
        {
            WORD LcAudioNum = UTIDWMask2Word( LcAudioMask );

            LcAudioMask &= ~UTIMaskDWord( LcAudioNum );

            // Set 48V phantom power supply on/off permanently
            // ------------------------------
            LcTargetInf.tgAudio = LcAudioNum;

            LcRet = CUR_COMMANDS_PTR->PIOEnablePhantomPowerSupply( LcAudioNum, (BOOLEAN)(LcLPtrRequest->iapqMic48V != FALSE), TRUE );
        }
    }

    LcLPtrRetHeader->rhCptr = LcRet;
    return;
}


// *************************************************************************
//
// STATIC VOID SetInBoardAudioEffect( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Paramaters:
// ******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// MBR: Creation
//
// *************************************************************************
//
//  Set any effects for input audio
//
// *************************************************************************
STATIC VOID     SetInBoardAudioEffect(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPIN_AUDIO_EFFECT_REQ_INFO  LcLPtrRequest;
    LPRESP_HEADER_INFO          LcLPtrRetHeader;
    WORD                        LcCurrentBoard;
    TARGET_INFO                 LcTargetInf;
    WORD                        LcRet   = SUCCESS ;
    WORD                        LcAudioNum;
    PVOIE_INFO                  LcPtVoie = NULL;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrRequest   = (LPIN_AUDIO_EFFECT_REQ_INFO)  PmBlocRequete;
    LcLPtrRetHeader = (LPRESP_HEADER_INFO)          PmBlocRetour;
    
    BZERO2( &LcTargetInf, TARGET_INFO, 1 );
    
    LcTargetInf.tgCarte = LcCurrentBoard = LcLPtrRequest->iaqHandle.iaqInBoardNum;
    LcTargetInf.tgAudio = LcAudioNum     = UTIDWMask2Word( LcLPtrRequest->iaqInAudioMask );
    
    LcRet = LookupBoardAudio(   (BYTE)LcCurrentBoard,
                                (BYTE)0,    // DspNum
                                (BYTE)LcAudioNum,
                                AUDIO_PHYSICAL | AUDIO_IN,
                                &LcPtVoie);
    
    
    // accept ED_INVALID_DSP_SOFTWARE (-> when DSP is not yet loaded)
    if( (LcRet != SUCCESS) && (LcRet != ED_INVALID_DSP_SOFTWARE) )    {
        LcLPtrRetHeader->rhCptr = LcRet;
        return;
    }

    // if there is a pipe on this audio, check if it is used by the same app!
    // if there is no pipe on this audio it's OK !
    if( LcPtVoie && (LcPtVoie->IndexAppli != LcLPtrRequest->Header.hdHandle) )
    {
        LOAD_PCX_ERROR( LcLPtrRetHeader->rhCptr, ED_INVALID_PIPE_AUDIO );
        return;
    }
    
    // Set effect if possible
    // ------------------------
    switch( LcLPtrRequest->iaqIdentifier)
    {
    case HW_EFFECT_NONE:
    case HW_EFFECT_COMPRESSOR_LIMITER:
        if(LcLPtrRequest->iaqCommand == EFFECT_CMD_SET )
        {
            LcLPtrRetHeader->rhCptr = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
        } else // remove, get not allowed
        {
            LcLPtrRequest->iaqIdentifier = HW_EFFECT_NONE;
            LcLPtrRetHeader->rhCptr = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
        } 
        break;

        LcLPtrRetHeader->rhCptr = CUR_COMMANDS_PTR->PIOSetInAudioEffects( LcAudioNum, LcLPtrRequest );
        break;

    case HW_EFFECT_MAXIMIZER:
    case HW_EFFECT_EQUALIZER:
        LcLPtrRetHeader->rhCptr = CUR_PROTOCOL_PTR->IFlow_AudioInputEffect(LcAudioNum, LcLPtrRequest);
        LcLPtrRetHeader->rhCptr = SUCCESS; 

        break;
    default:
        ASSERT(0);
    }

    return;

} //SetInBoardAudioEffect
// *************************************************************************
//                  Pipe Stream Control subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd5( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocReauete : Stream Control command response bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Stream Control command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd5(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM5 );
}


// *************************************************************************
//
// STATIC VOID StartStreamFct( IN LPDWORD , OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the start stream command.
//  Update the start stream command response bloc.
//
// *************************************************************************
STATIC  VOID    StartStreamFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPSTREAM_CMD_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    TARGET_INFO            LcTargetInf;
    BYTE                   LcAppIndex;
    BYTE                   LcCurrentBoard;
    DWORDLONG              LcPipeMask ;
    DWORD                  LcStreamMask ;
    WORD                   LcPipeIndex;
    WORD                   LcRet;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq       = (LPSTREAM_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve  application handle
    // ------------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe masks
    // -------------------------------
    LcPipeMaskOut = LcLPtrReq->scqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->scqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcLPtrReq->scqStreamMask ;

    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        // For each pipe specified
        // ------------------------
        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe index
            // --------------------
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // Seek the corresponding output pipe
            // -----------------------------------
            // LcPtVoie[LcPipeIndex].IndexAppli and LcPtVoie[LcPipeIndex].NumPipe already tested
            // in ChkPmManyInOrOutPipeMask !
            {
                // Fill TARGET_INFO structure
                LcTargetInf.tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
                LcTargetInf.tgCaracPipeVoie = LcOper;
                LcTargetInf.tgPipe  = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
                LcTargetInf.tgAudio = LcTargetInf.tgPipe;

                // ## FS (02/07/1997) - correct silent record
                // Filter the effective streams
                // ------------------------------------------
                LcTargetInf.tgMaskFlux = (  LcStreamMask
                                          & LcPtPipe[LcPipeIndex].piStreamMask );

                // If the specified streams do not exist
                // give up and return an error
                // -------------------------------------
                if ( LcTargetInf.tgMaskFlux == 0 )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
                    return;
                }

                LcTargetInf.tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;
                LcTargetInf.tgDifferedType = LcLPtrReq->scqDiffered;
                LcTargetInf.tgHour = LcLPtrReq->scqHour ;

                LcRet = CUR_PROTOCOL_PTR->IFlow_StreamStart( &LcTargetInf );

                LcLPtrRepHeader->rhCptr = LcRet;
            }

        } // for each pipe of the mask
    }
    // for output pipes then for input pipes
}


// *************************************************************************
//
// STATIC VOID PauseStreamFct(IN LPDWORD , OUT LPDWORD )
//
//Input Parameters:
// ****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// ********************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the pause stream command.
//  Update the pause stream command response bloc.
//
// *************************************************************************
STATIC  VOID    PauseStreamFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPSTREAM_CMD_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    TARGET_INFO            LcTargetInf;
    BYTE                   LcAppIndex;
    BYTE                   LcCurrentBoard;
    DWORDLONG              LcPipeMask ;
    DWORD                  LcStreamMask ;
    WORD                   LcPipeIndex;
    WORD                   LcRet;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq       = (LPSTREAM_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // ----------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe masks
    // -------------------
    LcPipeMaskOut = LcLPtrReq->scqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->scqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcLPtrReq->scqStreamMask ;

    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        // For each pipe specified
        // ------------------------
        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe index
            // -------------------
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // Seek the corresponding output pipe
            // -----------------------------------
            // LcPtVoie[LcPipeIndex].IndexAppli and LcPtVoie[LcPipeIndex].NumPipe already tested
            // in ChkPmManyInOrOutPipeMask !
            {

                // Fill TARGET_INFO structure
                LcTargetInf.tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
                LcTargetInf.tgCaracPipeVoie = LcOper;
                LcTargetInf.tgPipe = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
                LcTargetInf.tgAudio = LcTargetInf.tgPipe;

                // ## FS (02/07/1997) - correct silent record
                // Filter the effective streams
                // ------------------------------------------
                LcTargetInf.tgMaskFlux = (  LcStreamMask
                                          & LcPtPipe[LcPipeIndex].piStreamMask );
                // If the specified streams do not exist
                // give up and return an error
                // -------------------------------------
                if ( LcTargetInf.tgMaskFlux == 0 )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
                    return;
                }

                LcTargetInf.tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;
                LcTargetInf.tgDifferedType = LcLPtrReq->scqDiffered;
                LcTargetInf.tgHour = LcLPtrReq->scqHour ;

                LcRet = CUR_PROTOCOL_PTR->IFlow_StreamPause( &LcTargetInf );

                LcLPtrRepHeader->rhCptr = LcRet;
            }

        } // for each pipe of the mask

    }
    // for output pipes then input pipes

}


// *************************************************************************
//
// STATIC VOID StopStreamFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// ********************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the stop stream command.
//  Update the stop stream command response bloc.
//
// *************************************************************************
STATIC  VOID    StopStreamFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPSTREAM_CMD_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    TARGET_INFO            LcTargetInf;
    BYTE                   LcAppIndex;
    BYTE                   LcCurrentBoard;
    DWORDLONG              LcPipeMask ;
    DWORD                  LcStreamMask ;
    WORD                   LcPipeIndex;
    WORD                   LcRet;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq       = (LPSTREAM_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe masks
    // -------------------
    LcPipeMaskOut = LcLPtrReq->scqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->scqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcLPtrReq->scqStreamMask ;

    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        // For each pipe specified
        // ------------------------
        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe index
            // -------------------
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // Seek the corresponding output pipe
            // -----------------------------------
            // LcPtVoie[LcPipeIndex].IndexAppli and LcPtVoie[LcPipeIndex].NumPipe already tested
            // in ChkPmManyInOrOutPipeMask !
            {
                

                // Fill TARGET_INFO structure
                LcTargetInf.tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
                LcTargetInf.tgCaracPipeVoie = LcOper;
                LcTargetInf.tgPipe = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
                LcTargetInf.tgAudio = LcTargetInf.tgPipe;

                // ## FS (02/07/1997) - correct silent record
                // Filter the effective streams
                // ------------------------------------------
                LcTargetInf.tgMaskFlux = (  LcStreamMask
                                          & LcPtPipe[LcPipeIndex].piStreamMask );
                // If the specified streams do not exist
                // give up and return an error
                // -------------------------------------
                if ( LcTargetInf.tgMaskFlux == 0 )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
                    return;
                }

                LcTargetInf.tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;

                LcRet = CUR_PROTOCOL_PTR->IFlow_StreamStop( &LcTargetInf );

                LcLPtrRepHeader->rhCptr = LcRet;

                if ( LcRet != SUCCESS )
                {
                    APHKeepDspError( &(APH_Board_Info_Array[LcCurrentBoard].biTbDspInfo[0]), LcRet );
                }

                if ( LcOper == OPER_PLAY )
                {
                    // MBR (2003/05/23) it's the driver that counts the playback bytes now
                    // reset them here
                    int  j;

                    for(j=0; j<MAX_STREAM_PIPE; j++)
                    {
                        if( (LcTargetInf.tgMaskFlux >> j) & 1 )
                        {
                            TbOutStreamBytePos[LcPipeIndex][j] = 0;
                        }
                    }
                }
            }

        } // for each pipe of the mask

    }
    // for output pipes then input pipes

}

// *************************************************************************
//
// STATIC VOID SetOutStreamSettings( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the set out stream parameters command.
//
// *************************************************************************
STATIC VOID SetOutStreamSettings(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPOUT_STREAM_SET_REQ_INFO  LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    TARGET_INFO            LcTargetInf  ;
    BYTE                   LcAppIndex   ;
    BYTE                   LcCurrentBoard;
    DWORDLONG              LcPipeMask   ;
    DWORD                  LcStreamMask ;
    WORD                   LcPipeIndex;
    WORD                   LcRet = SUCCESS ;

    // Initialisation of pointers
    LcLPtrReq = (LPOUT_STREAM_SET_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve and verify pipe mask
    // -------------------------------
    LcPipeMask = LcLPtrReq->ssq.PipeMask64.QuadPart ;

    LcRet = ChkPmManyOutputPipeMask( LcAppIndex, LcPipeMask );

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcLPtrReq->ssq.StreamMask ;

    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // For each pipe specified
    // ------------------------
    while ( (LcPipeMask != 0) && (LcRet == SUCCESS) )
    {
        // Retrieve pipe index
        // -------------------------------
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // Get the corresponding output pipe audios
        // ----------------------------------------
        // Fill TARGET_INFO structure
        LcTargetInf.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
        LcTargetInf.tgCaracPipeVoie = OPER_PLAY;
        LcTargetInf.tgPipe = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;
        LcTargetInf.tgAudio = LcTargetInf.tgPipe;

        // ## FS (02/07/1997) - correct silent record
        // Filter the effective streams
        // ------------------------------------------
        LcTargetInf.tgMaskFlux = (  LcStreamMask
                                  & TbPipeOutInfo[LcPipeIndex].piStreamMask );

        // If the specified streams do not exist
        // give up and return an error
        // -------------------------------------
        if ( LcTargetInf.tgMaskFlux == 0 )
        {
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
            return;
        }

        LcTargetInf.tgVioHandle = TbPipeOutInfo[LcPipeIndex].piVioHandle ;
        LcTargetInf.tgHour = LcLPtrReq->ssq.Hour ;
        LcTargetInf.tgDifferedType = LcLPtrReq->ssq.Differed;

        if (   ( LcRet == SUCCESS )
            && ( LcLPtrReq->ssq.Par1 & OUT_STREAM_SET_PARAM_FORMAT_MASK ) )
        {
            LcRet = CUR_PROTOCOL_PTR->IFlow_StreamOutSetFormat(
                    &LcTargetInf,
                    LcLPtrReq->ssq.DeCodeForm1,
                    LcLPtrReq->ssq.DeCodeForm2,
                    LcLPtrReq->ssq.dwChannelMask,
                    LcLPtrReq->ssq.dwSamplesPerChannel
                    );
        }

        if (   ( LcRet == SUCCESS )
            && ( LcLPtrReq->ssq.Par1 & OUT_STREAM_SET_PARAM_AUTOMAT_MASK ) )
        {
            register BOOLEAN    LcAutomation ;

            LcAutomation =
                (LcLPtrReq->ssq.Automation == OUT_STREAM_SET_SYNC_LEVEL1_OFF )
                ? TRUE : FALSE ;
            LcRet = CUR_PROTOCOL_PTR->IFlow_StreamOutSetParameters( &LcTargetInf, LcAutomation );
        }

        LcLPtrRepHeader->rhCptr = LcRet;

    } // for each pipe
}


// *************************************************************************
//
// STATIC VOID SetOutStreamLevels( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//
// FH Creation
// *************************************************************************
//
//  Process the set digital levels and digital mutes out stream command.
//
// *************************************************************************
//
// This function supposes :
//   - A pipe is on only one card
//
// *************************************************************************
STATIC VOID SetOutStreamLevels(
                            IN  LPDWORD PmBlocRequete   ,
                            OUT LPDWORD PmBlocRetour    )
{
    LPOUT_STREAM_LEVELS_REQ_INFO        LcLPtrReq;
    LPBC_HEADER_INFO                    LcLPtrReqHeader;
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    TARGET_INFO                         LcTargetInf;
    LEVEL_STREAM_INFO                   LcLevelInf;
    BYTE                                LcAppIndex ;
    WORD                                LcPipeIndex ;
    WORD                                LcRet ;
    WORD                                LcCurrentBoard = 0 ;
	DWORDLONG                           LcPipeMask;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq = (LPOUT_STREAM_LEVELS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( OUT_STREAM_LEVELS_REQ_INFO)
                 - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );
    LcPipeMask = LcLPtrReq->slqOutPipeMask64.QuadPart;

    // Verify pipes specified
    // -----------------------
    LcRet = ChkPmManyOutputPipeMask(
                            LcAppIndex,
                            LcPipeMask );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Fill target_info
    // ----------------
    LcTargetInf.tgCaracPipeVoie = OPER_PLAY;
    LcTargetInf.tgDifferedType = LcLPtrReq->slqDiffered;
    LcTargetInf.tgHour = LcLPtrReq->slqHour ;

    // Digital levels and digital mute stream processing
    // -------------------------------------------------

    // 1. Process panoramic Levels
    // ------------------------------------------------

    // Select the stream levels/mutes requested by user
    // ------------------------------------------------
    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_PAN_LEVEL_CGV1 )
        LcLevelInf.sliHasPanLevelCgV1 = TRUE;
    else
        LcLevelInf.sliHasPanLevelCgV1 = FALSE;

    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_PAN_LEVEL_CGV2 )
        LcLevelInf.sliHasPanLevelCgV2 = TRUE;
    else
        LcLevelInf.sliHasPanLevelCgV2 = FALSE;

    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_PAN_LEVEL_CDV1 )
        LcLevelInf.sliHasPanLevelCdV1 = TRUE;
    else
        LcLevelInf.sliHasPanLevelCdV1 = FALSE;

    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_PAN_LEVEL_CDV2 )
        LcLevelInf.sliHasPanLevelCdV2 = TRUE;
    else
        LcLevelInf.sliHasPanLevelCdV2 = FALSE;

    // Write their values (even if not selected hence not significant)
    // ---------------------------------------------------------------
    LcLevelInf.sliPanLevelCgV1 = LcLPtrReq->slqLeftChannelToOutput1DigitalLevel;
    LcLevelInf.sliPanLevelCgV2 = LcLPtrReq->slqLeftChannelToOutput2DigitalLevel;
    LcLevelInf.sliPanLevelCdV1 = LcLPtrReq->slqRightChannelToOutput1DigitalLevel;
    LcLevelInf.sliPanLevelCdV2 = LcLPtrReq->slqRightChannelToOutput2DigitalLevel;

    // 2. Process stream global Levels
    // -------------------------------

    // Select the stream levels/mutes requested by user
    // ------------------------------------------------
    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_LEVEL_STREAM1 )
        LcLevelInf.sliHasLevelStream1 = TRUE;
    else
        LcLevelInf.sliHasLevelStream1 = FALSE;

    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_LEVEL_STREAM2 )
        LcLevelInf.sliHasLevelStream2 = TRUE;
    else
        LcLevelInf.sliHasLevelStream2 = FALSE;

    // Write their values (even if not selected hence not significant)
    // ---------------------------------------------------------------
    LcLevelInf.sliLevelStream1 = LcLPtrReq->slqDigitalLevel1;
    LcLevelInf.sliLevelStream2 = LcLPtrReq->slqDigitalLevel2;

    // 3. Process stream global mutes
    // ------------------------------

    // Select the stream levels/mutes requested by user
    // ------------------------------------------------
    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_MUTE_STREAM1 )
        LcLevelInf.sliHasMuteStream1 = TRUE;
    else
        LcLevelInf.sliHasMuteStream1 = FALSE;

    if ( LcLPtrReq->slqPar1 & OUT_STREAM_SET_MUTE_STREAM2 )
        LcLevelInf.sliHasMuteStream2 = TRUE;
    else
        LcLevelInf.sliHasMuteStream2 = FALSE;

    // Write their values (even if not selected hence not significant)
    // ---------------------------------------------------------------
    LcLevelInf.sliMuteStream1 = LcLPtrReq->slqMute1;
    LcLevelInf.sliMuteStream2 = LcLPtrReq->slqMute2;

    // For all the pipes
    // -----------------
    do
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask ) ;
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        LcTargetInf.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
        LcTargetInf.tgPipe = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;

        // ## FS (02/07/1997) - correct silent record
        // Filter the effective streams
        // ------------------------------------------
        LcTargetInf.tgMaskFlux = (  LcLPtrReq->slqStreamMask
                                  & TbPipeOutInfo[LcPipeIndex].piStreamMask );
        // If the specified streams do not exist
        // give up and return an error
        // -------------------------------------
        if ( LcTargetInf.tgMaskFlux == 0 )
        {
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
            return;
        }

        LcRet = CUR_PROTOCOL_PTR->ILevel_StreamOutSetLevels( &LcTargetInf, &LcLevelInf );
        if ( LcRet != SUCCESS )
        {
            LcLPtrRepHeader->rhCptr = LcRet;
            return;
        }

    } while( LcPipeMask ); // for all pipes of the corresponding mask
}


// *************************************************************************
//
// STATIC VOID SetOutStreamEffects( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//
// *************************************************************************
//
//  Process the command: set the effects parameters on output stream.
//
// *************************************************************************
//
STATIC VOID SetOutStreamEffects(
                            IN  LPDWORD PmBlocRequete   ,
                            OUT LPDWORD PmBlocRetour    )
{
    LPSET_OUT_STREAM_EFFECTS_REQ_INFO   LcLPtrReq;
    LPBC_HEADER_INFO                    LcLPtrReqHeader;
    LPRESP_HEADER_INFO                  LcLPtrRepHeader;
    TARGET_INFO                         LcTargetInf;
    BYTE                                LcAppIndex ;
    WORD                                LcRet ;
    WORD                                LcPipeIndex ;
    WORD                                LcCurrentBoard ;
    DWORDLONG                           LcPipeMask;

    // Initialisation of pointers
    // ---------------------------
    LcLPtrReq       = (LPSET_OUT_STREAM_EFFECTS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    LcLPtrRepHeader->rhSize = sizeof( SET_OUT_STREAM_EFFECTS_REQ_INFO )
                            - sizeof( RESP_HEADER_INFO );

    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );
    LcPipeMask = LcLPtrReq->sseqOutPipeMask64.QuadPart;

    // Verify pipes specified
    // -----------------------
    LcRet = ChkPmManyOutputPipeMask( LcAppIndex,
                                     LcPipeMask );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Fill target_info
    // ----------------
    LcTargetInf.tgCaracPipeVoie = OPER_PLAY;
    LcTargetInf.tgDifferedType  = LcLPtrReq->sseqDiffered;
    LcTargetInf.tgHour          = LcLPtrReq->sseqHour ;

    // For all the pipes
    // -----------------
    do
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        LcTargetInf.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
        LcTargetInf.tgPipe = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;

        // ## FS (02/07/1997) - correct silent record
        // Filter the effective streams
        // ------------------------------------------
        LcTargetInf.tgMaskFlux = (  LcLPtrReq->sseqStreamMask
                                  & TbPipeOutInfo[LcPipeIndex].piStreamMask );

        // If the specified streams do not exist
        // give up and return an error
        // -------------------------------------
        if ( LcTargetInf.tgMaskFlux == 0 )
        {
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_STREAM );
            return;
        }

        LcRet = CUR_PROTOCOL_PTR->IFlow_StreamOutSetEffects(	&LcTargetInf,
																LcLPtrReq->sseqEffectId,																
																LcLPtrReq->sseqParam );
        if ( LcRet != SUCCESS )
        {
            LcLPtrRepHeader->rhCptr = LcRet;
            return;
        }

    } while ( LcPipeMask ); // for all pipes of the corresponding mask
}


// *************************************************************************
//
// STATIC VOID SetInStreamSettings( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the set out stream parameters command.
//
// *************************************************************************
STATIC VOID SetInStreamSettings(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO                LcReqHeaderPtr  ;
    LPIN_STREAM_SETTINGS_REQ_INFO   LcRequestPtr    ;
    LPRESP_HEADER_INFO              LcRepHeaderPtr  ;
    BYTE    LcAppIndex          ;
    WORD    LcCurrentBoard      ;
    DWORDLONG LcPipeMask        ;
    DWORD   LcStreamMask        ;
    WORD    LcPipeIndex         ;
    WORD    LcRet               = SUCCESS ;
    TARGET_INFO LcTarget        ;

    // Initialisation of pointers
    LcRequestPtr   = (LPIN_STREAM_SETTINGS_REQ_INFO)PmBlocRequete;
    LcReqHeaderPtr = (LPBC_HEADER_INFO)PmBlocRequete;
    LcRepHeaderPtr = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Retrieve and verify pipe mask
    // -------------------------------
    LcPipeMask = LcRequestPtr->isq.PipeMask64.QuadPart ;
    LcRet = ChkPmManyInputPipeMask( LcAppIndex, LcPipeMask );
    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet ;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcRequestPtr->isq.StreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
        return;
    }

    while ( ( LcPipeMask != 0 ) && (LcRet == SUCCESS) )
    {
        // Retrieve pipe index
        // -------------------------------
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

        // Fill TARGET_INFO structure
        LcTarget.tgCarte = LcCurrentBoard = TbPipeInInfo[LcPipeIndex].piNumBoard;
        LcTarget.tgCaracPipeVoie = OPER_REC;
        LcTarget.tgPipe = TbPipeInInfo[LcPipeIndex].piNumPipeForDsp;
        LcTarget.tgAudio = LcTarget.tgPipe;

        // correct silent record
        // Filter the effective streams
        // ------------------------------------------
        LcTarget.tgMaskFlux = (  LcStreamMask & TbPipeInInfo[LcPipeIndex].piStreamMask );
        // If the specified streams do not exist
        // give up and return an error
        // -------------------------------------
        if ( LcTarget.tgMaskFlux == 0 )
        {
            LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
            return;
        }

        LcTarget.tgVioHandle = TbPipeInInfo[LcPipeIndex].piVioHandle ;
        LcTarget.tgHour = LcRequestPtr->isq.Hour ;
        LcTarget.tgDifferedType = LcRequestPtr->isq.Differed;

        if ( LcRequestPtr->isq.Par1 & IN_STREAM_SET_PARAM_FORMAT_MASK )
        {
            LcRet = CUR_PROTOCOL_PTR->IFlow_StreamInSetFormat(
                    &LcTarget,
                    LcRequestPtr->isq.DeCodeForm1,
                    LcRequestPtr->isq.DeCodeForm2,
                    LcRequestPtr->isq.dwChannelMask,
                    LcRequestPtr->isq.dwSamplesPerChannel
                    );
        }
        else if ( LcRequestPtr->isq.Par1 != 0 )
        {
            LOAD_PCX_ERROR( LcRet, ED_BAD_VALIDATION_MASK );
        }

        LcRepHeaderPtr->rhCptr = LcRet;

    }

}


// *************************************************************************
//
// STATIC VOID SetOutStreamLCurve( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the set out stream parameters command.
//
// *************************************************************************
// ## FS (08/12/1997) -- add selction of level stream F1
// and use the same interface as for VIOSetOutStreamLevels
// *************************************************************************
STATIC VOID SetOutStreamLCurve(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO                LcReqHeaderPtr  ;
    LPOUT_STREAM_LCURVE_REQ_INFO    LcRequestPtr    ;
    LPRESP_HEADER_INFO              LcRepHeaderPtr  ;
    BYTE    LcAppIndex          ;
    BYTE    LcCurrentBoard      ;
    DWORDLONG LcPipeMask        ;
    DWORD   LcStreamMask        ;
    WORD    LcPipeIndex         ;
    WORD    LcRet               = SUCCESS ;
    TARGET_INFO         LcTarget;
    LEVEL_STREAM_INFO   LcLevelInf;

    // Initialisation of pointers
    LcRequestPtr   = (LPOUT_STREAM_LCURVE_REQ_INFO)PmBlocRequete;
    LcReqHeaderPtr = (LPBC_HEADER_INFO)PmBlocRequete;
    LcRepHeaderPtr = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Retrieve and verify pipe mask
    // -------------------------------
    LcPipeMask = LcRequestPtr->oslqOutPipeMask64.QuadPart ;
    LcRet = ChkPmManyOutputPipeMask( LcAppIndex, LcPipeMask );
    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet ;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcRequestPtr->oslqStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
        return;
    }

    while ( ( LcPipeMask != 0 ) && (LcRet == SUCCESS) )
    {
        // Retrieve pipe index
        // -------------------------------
        LcPipeIndex = UTI64Mask2Word( LcPipeMask );
        LcPipeMask &= ~UTIMask64bit( LcPipeIndex );


        // Fill TARGET_INFO structure
        LcTarget.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
        LcTarget.tgCaracPipeVoie = OPER_PLAY;
        LcTarget.tgPipe = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;

        // Filter the effective streams
        // ------------------------------------------
        LcTarget.tgMaskFlux = (  LcStreamMask  & TbPipeOutInfo[LcPipeIndex].piStreamMask );
        LcTarget.tgVioHandle = TbPipeOutInfo[LcPipeIndex].piVioHandle ;
        LcTarget.tgHour = LcRequestPtr->oslqHour ;
        LcTarget.tgDifferedType = LcRequestPtr->oslqDiffered;

        // Fill in a structure that identifies requested levels
        // ----------------------------------------------------
        if ( LcRequestPtr->oslqPar1 & OUT_STREAM_SET_LEVEL_STREAM1 )
            LcLevelInf.sliHasLevelStream1 = TRUE;
        else
            LcLevelInf.sliHasLevelStream1 = FALSE;

        if ( LcRequestPtr->oslqPar1 & OUT_STREAM_SET_LEVEL_STREAM2 )
            LcLevelInf.sliHasLevelStream2 = TRUE;
        else
            LcLevelInf.sliHasLevelStream2 = FALSE;

        LcLevelInf.sliCurveNbLevels = LcRequestPtr->oslqLevelNumber;
        LcLevelInf.sliCurveStep = LcRequestPtr->oslqStep;
        LcLevelInf.sliCurveValues = LcRequestPtr->oslqLevelValue;

        // Actually process the request and sen it to the board
        // ----------------------------------------------------
        LcRet = CUR_PROTOCOL_PTR->ILevel_StreamOutSetLevelCurve( &LcTarget, &LcLevelInf );
        if ( LcRet != SUCCESS )
        {
            LcRepHeaderPtr->rhCptr = LcRet;
            return;
        }

    }

}

// *************************************************************************
//
// STATIC VOID NotifyStreamTimeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the notify stream time command.
//
// *************************************************************************
STATIC VOID NotifyStreamTimeFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO                LcReqHeaderPtr  ;
    LPNOTIFY_STREAM_TIME_REQ_INFO   LcRequestPtr    ;
    LPRESP_HEADER_INFO              LcRepHeaderPtr  ;
    BYTE                            LcAppIndex;
    PPIPE_INFO                      LcPtPipe;
    DWORDLONG                       LcPipeMask, LcPipeMaskIn, LcPipeMaskOut;
    DWORD                           LcStreamMask;
    SHORT                           LcPipeIndex;
    WORD                            LcNbPipe = 0 ;
    BYTE                            LcCurrentBoard;
    WORD                            i;
    WORD                            LcRet;
    BYTE                            LcOper;

    // Initialisation of pointers
    LcRequestPtr   = (LPNOTIFY_STREAM_TIME_REQ_INFO)PmBlocRequete;
    LcReqHeaderPtr = (LPBC_HEADER_INFO)PmBlocRequete;
    LcRepHeaderPtr = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Retrieve pipe mask
    // ******************
    LcPipeMaskOut = LcRequestPtr->nstqOutPipeMask64.QuadPart;
    LcPipeMaskIn = LcRequestPtr->nstqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // --------------------------------
    LcStreamMask = LcRequestPtr->nstqStreamMask ;
    if ( LcStreamMask == 0 )
    {
        LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Remember differed time (only once is necessary)
    // ------------------------------------------------
    TbTargetInf[0].tgDifferedType = LcRequestPtr->nstqDiffered;
    TbTargetInf[0].tgHour = LcRequestPtr->nstqHour;

    // ## FS (08/09/1997) -- bug fix for FA #55
    // Remember how many pipes have been encountered
    // ---------------------------------------------
    LcNbPipe = 0 ;

    // For all the pipes concerned by the command, verify the command is ok
    // --------------------------------------------------------------------
    for (i=0; i<2; i++)
    {
        switch(i)
        {
          case 0 : LcPipeMask = LcPipeMaskOut;
               LcPtPipe = &TbPipeOutInfo[0];
               LcOper      = OPER_PLAY;
               break;

          case 1 : LcPipeMask = LcPipeMaskIn;
               LcPtPipe = &TbPipeInInfo[0];
               LcOper      = OPER_REC;
               break;
        }

        while ( LcPipeMask != 0 )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // Processes the corresponding output pipe
            // ---------------------------------------
            

            // Fill TARGET_INFO structure
            TbTargetInf[LcNbPipe].tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
            TbTargetInf[LcNbPipe].tgCaracPipeVoie = LcOper;
            TbTargetInf[LcNbPipe].tgPipe =  LcPtPipe[LcPipeIndex].piNumPipeForDsp;
            TbTargetInf[LcNbPipe].tgAudio = TbTargetInf[LcNbPipe].tgPipe;

            // Filter the effective streams
            // ------------------------------------------
            TbTargetInf[LcNbPipe].tgMaskFlux = (  LcStreamMask
                                                & LcPtPipe[LcPipeIndex].piStreamMask );
            // If the specified streams do not exist
            // give up and return an error
            // -------------------------------------
            if ( TbTargetInf[LcNbPipe].tgMaskFlux == 0 )
            {
                LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
                return;
            }

            LcNbPipe++;

        }
    }
	if(LcNbPipe)
	{
		LcRet = CUR_PROTOCOL_PTR->IEvent_StreamNotifyTime( TbTargetInf, LcNbPipe );
	}
    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet;
        return;
    }

}

// *************************************************************************
//
// STATIC VOID CancelStreamBufFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the cancel stream buffers command.
//
// *************************************************************************
STATIC VOID CancelStreamBufFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO                LcReqHeaderPtr  ;
    LPCANCEL_BUFFER_REQ_INFO        LcRequestPtr    ;
    LPRESP_HEADER_INFO              LcRepHeaderPtr  ;
    LPCANCEL_BUFFER_RESP_INFO       LcReplyPtr      ;
    TARGET_INFO                     LcTargetInf;
    PPIPE_INFO                      LcPtPipe;
    PVOIE_INFO                      LcPtVoie;
    DWORD                           LcStreamMask;
    WORD                            LcPipeIndex;
    WORD                            LcCurrentBoard;
    WORD                            LcRet;
    DWORD                           LcNbBuffers;
    BYTE                            LcAppIndex;

    // Initialisation of pointers
    LcRequestPtr   = (LPCANCEL_BUFFER_REQ_INFO)PmBlocRequete;
    LcReqHeaderPtr = (LPBC_HEADER_INFO)PmBlocRequete;
    LcReplyPtr     = (LPCANCEL_BUFFER_RESP_INFO)PmBlocRetour;
    LcRepHeaderPtr = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmOneInOrOutPipeMask(    LcAppIndex,
                                        LcRequestPtr->cbqOutPipeMask64.QuadPart,
                                        LcRequestPtr->cbqInPipeMask64.QuadPart,
                                        &LcPipeIndex,
                                        &LcPtPipe,
                                        &LcPtVoie,
                                        &LcTargetInf );

	LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // and filter the effective streams
    // --------------------------------
    LcStreamMask = LcRequestPtr->cbqStreamMask ;
    LcTargetInf.tgMaskFlux = (  LcStreamMask
                              & LcPtPipe[LcPipeIndex].piStreamMask );

    // If the specified streams do not exist
    // give up and return an error
    // -------------------------------------
    if ( LcTargetInf.tgMaskFlux == 0 )
    {
        LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Actually try to remove the buffer and following
    // from the stream queue
    // -----------------------------------------------
    LcRet = CUR_PROTOCOL_PTR->IBuffer_CutBuffersQueued( &LcTargetInf, LcRequestPtr->cbqBuffNum, &LcNbBuffers );

    // Fill reply block
    // ----------------
    LcRepHeaderPtr->rhCptr = LcRet;
    LcReplyPtr->cbrCancelledNumber = LcNbBuffers;
}


// *************************************************************************
//
// STATIC VOID CancelBufPauseFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the cancel pause at end of buffer command.
//
// *************************************************************************
STATIC VOID CancelBufPauseFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO                LcReqHeaderPtr  ;
    LPCANCEL_PAUSE_REQ_INFO         LcRequestPtr    ;
    LPRESP_HEADER_INFO              LcRepHeaderPtr  ;
    TARGET_INFO                     LcTargetInf;
    PPIPE_INFO                      LcPtPipe;
    PVOIE_INFO                      LcPtVoie;
    DWORD                           LcStreamMask;
    WORD                            LcPipeIndex;
    WORD                            LcRet;
    BYTE                            LcAppIndex;

    // Initialisation of pointers
    LcRequestPtr   = (LPCANCEL_PAUSE_REQ_INFO)PmBlocRequete;
    LcReqHeaderPtr = (LPBC_HEADER_INFO)PmBlocRequete;
    LcRepHeaderPtr = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmOneOutputPipeMask(    LcAppIndex,
                                       LcRequestPtr->cbpqOutPipeMask64.QuadPart,
                                       &LcPipeIndex,
                                       &LcPtPipe,
                                       &LcPtVoie,
                                       &LcTargetInf );

	WORD LcCurrentBoard = LcTargetInf.tgCarte;

    if ( LcRet != SUCCESS )
    {
        LcRepHeaderPtr->rhCptr = LcRet;
        return;
    }

    // Retrieve and verify stream mask
    // and filter the effective streams
    // --------------------------------
    LcStreamMask = LcRequestPtr->cbpqStreamMask ;
    LcTargetInf.tgMaskFlux = (  LcStreamMask
                              & LcPtPipe->piStreamMask );

    // If the specified streams do not exist
    // give up and return an error
    // -------------------------------------
    if ( LcTargetInf.tgMaskFlux == 0 )
    {
        LOAD_PCX_ERROR( LcRepHeaderPtr->rhCptr, ED_INVALID_STREAM );
        return;
    }

    // Actually try to remove the buffer and following
    // from the stream queue
    // -----------------------------------------------
    LcRet = CUR_PROTOCOL_PTR->IBuffer_BypassPauseAtBufferEnd( &LcTargetInf, LcRequestPtr->cbpqBuffNum );

    // Fill reply block
    // ----------------
    LcRepHeaderPtr->rhCptr = LcRet;
}


// *************************************************************************
//                  Pipe Control subfunctions
// *************************************************************************

// *************************************************************************
//
// STATIC VOID InvalidCmd3( PmBlocRequete, PmBlocRetour)
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Pipe Control command request bloc
//
// Output Parameters:
// **************************
//
//  LPDWORD PmBlocRetour  : Pipe Control command response bloc
//
// *************************************************************************
//
//  Common entry points for an invalid or not implemented yet command
// within the family
//
// *************************************************************************
STATIC VOID InvalidCmd3(
    IN  LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPRESP_HEADER_INFO  LcLPResp;

    LcLPResp = (LPRESP_HEADER_INFO) PmBlocRetour ;
    LOAD_PCX_ERROR( LcLPResp->rhCptr, ED_INVALID_CMD_FAM3 );
}

// *************************************************************************
//
// STATIC VOID StartPipeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the start pipe command.
//  Update the start pipe command response bloc.
//
// *************************************************************************
STATIC  VOID    StartPipeFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPPIPE_CMD_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    BYTE                   LcAppIndex, LcNbPipe;
    WORD                   LcRet, LcPipeIndex;
    DWORDLONG              LcPipeMask;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // ----------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe masks
    // -------------------
    LcPipeMaskOut = LcLPtrReq->spqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->spqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // When at least one pipe is started, no start on time-code is authorized
    // because a pause/stop on the started pipe will generate an IRQA that
    // would triggered the start on time-code.
    // ----------------------------------------------------------------------
    if (    (    ( StartedPipeInMask  != 0 )
              || ( StartedPipeOutMask != 0 ) )
         && ( LcLPtrReq->spqAttributes == START_PAUSE_ON_TIME_CODE ) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_START_IN_PROGRESS );
        return;
    }

    // When a start on time code is in progress, and until the start is effective
    // or canceled, no other start is authorized, because the IRQA
    // send for this new start will trigger the start on time code too.
    // ----------------------------------------------------------------------
    if ( StartOnTimeCodeInProgress == TRUE )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_START_ON_TIME_CODE_IN_PROGRESS );
        return;
    }

    // Check that the board that must verify the time-code start condition
    // is managed by the driver
    // -------------------------------------------------------------------
    if ( LcLPtrReq->spqAttributes == START_PAUSE_ON_TIME_CODE )
    {
        BYTE LcBoard = LcLPtrReq->spqTimeCodeStart.tcsBoardNum;

        if (    ( LcBoard >= MAX_BOARD )
             || ( APH_Board_Info_Array[LcBoard].biCarac != CARAC_USED ) )
        {
            LOAD_PCX_ERROR( LcRet, ED_INVALID_BOARD );
            return;
        }
    }

    // Remember how many pipes have been encountered
    // ---------------------------------------------
    LcNbPipe = 0 ;
    WORD LcCurrentBoard = 0;

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe
            // -------------
            // FB : 11/12/1996 : Ajout du test appartenance  l'application
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // "check pipes : no synchro inter carte" !
            // if on different cards, don't use
            if(LcNbPipe == 0)
            {
                LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
            }
            else
            {
                if( LcCurrentBoard != LcPtPipe[LcPipeIndex].piNumBoard )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_PIPE );
                    return;
                }
            }

            // Fill TARGET_INFO structure
            TbTargetInf[LcNbPipe].tgCarte       = LcCurrentBoard;
            TbTargetInf[LcNbPipe].tgCaracPipeVoie = LcOper;
            TbTargetInf[LcNbPipe].tgPipe		=  LcPtPipe[LcPipeIndex].piNumPipeForDsp;
            TbTargetInf[LcNbPipe].tgAudio		=  TbTargetInf[LcNbPipe].tgPipe;
            TbTargetInf[LcNbPipe].tgVioHandle	=  LcPtPipe[LcPipeIndex].piVioHandle ;
            LcNbPipe++;

        } // for each pipe in the mask

    }
    // for output then input pipes

    ASSERT(LcNbPipe != 0);

    LcRet = CUR_PROTOCOL_PTR->IFlow_PipeStart( TbTargetInf,LcNbPipe,LcLPtrReq->spqAttributes, & (LcLPtrReq->spqTimeCodeStart) );
    //LcRet = TogglePipeState( START_STATE, LcLPtrReq->spqAttributes, LcNbPipe, TbTargetInf, & (LcLPtrReq->spqTimeCodeStart), NULL );

    if ( LcRet == SUCCESS )
    {
        // Add the pipe in and out mask to the masks giving all the
        // started pipe.
        // --------------------------------------------------------
        StartedPipeInMask  |= LcPipeMaskIn;
        StartedPipeOutMask |= LcPipeMaskOut;

        // If a start on time code is asked, remember it.
        // ----------------------------------------------
        if ( LcLPtrReq->spqAttributes == START_PAUSE_ON_TIME_CODE )
        {
            StartOnTimeCodeInProgress = TRUE;
            StartOnTimeCodeBoardNum   = LcLPtrReq->spqTimeCodeStart.tcsBoardNum;
            StartOnTimeCodeInMask     = LcPipeMaskIn;
            StartOnTimeCodeOutMask    = LcPipeMaskOut;
        }
    }

    LcLPtrRepHeader->rhCptr = LcRet;
}

// *************************************************************************
//
// STATIC VOID PausePipeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the pause pipe command.
//  Update the pause pipe command response bloc.
//
// *************************************************************************
STATIC VOID PausePipeFct(
    IN LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPPIPE_CMD_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    BYTE                   LcAppIndex, LcNbPipe;
    WORD                   LcRet, LcPipeIndex;
    DWORDLONG              LcPipeMask;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq = (LPPIPE_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // -------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe mask
    // ------------------
    LcPipeMaskOut = LcLPtrReq->spqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->spqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // When a start on time code is in progress, and until the start is effective
    // or canceled, no pause is authorized, because the IRQA
    // send for this pause will trigger the start on time code too.
    // ----------------------------------------------------------------------
    if ( StartOnTimeCodeInProgress == TRUE )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_START_ON_TIME_CODE_IN_PROGRESS );
        return;
    }

    // Remember how many pipes have been encountered
    // ---------------------------------------------
    LcNbPipe = 0 ;
    WORD LcCurrentBoard = 0;

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe
            // -------------
            // FB : 11/12/1996 : Ajout du test appartenance  l'application
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // "check pipes : no synchro inter carte" !
            // if on different cards, don't use
            if(LcNbPipe == 0)
            {
                LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
            }
            else
            {
                if( LcCurrentBoard != LcPtPipe[LcPipeIndex].piNumBoard )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_PIPE );
                    return;
                }
            }

            // Fill TARGET_INFO structure
            TbTargetInf[LcNbPipe].tgCarte = LcCurrentBoard;
            TbTargetInf[LcNbPipe].tgCaracPipeVoie = LcOper;
            TbTargetInf[LcNbPipe].tgPipe =  LcPtPipe[LcPipeIndex].piNumPipeForDsp;
            TbTargetInf[LcNbPipe].tgAudio = TbTargetInf[LcNbPipe].tgPipe;
            TbTargetInf[LcNbPipe].tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;
            LcNbPipe++;

        } // for each pipe of the mask

    }
    // for output then input pipes

    ASSERT(LcNbPipe != 0);

    LcRet = CUR_PROTOCOL_PTR->IFlow_PipePause( TbTargetInf, LcNbPipe, LcLPtrReq->spqAttributes );
    //LcRet = TogglePipeState( PAUSE_STATE, LcLPtrReq->spqAttributes, LcNbPipe, TbTargetInf, NULL );

    LcLPtrRepHeader->rhCptr = LcRet;

}


// *************************************************************************
//
// STATIC VOID StopPipeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the stop pipe command.
//  Update the stop pipe command response bloc.
//
// *************************************************************************
STATIC VOID StopPipeFct(
    IN LPDWORD PmBlocRequete,
    OUT LPDWORD PmBlocRetour )
{
    LPPIPE_CMD_REQ_INFO    LcLPtrReq;
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    LPPIPE_CMD_RESP_INFO   LcLPtrReply;
    BYTE                   LcAppIndex, LcNbPipe;
    BYTE                   LcFirstRecordPipe = 0;
    WORD                   LcCurrentBoard;
    WORD                   i, LcRet;
    DWORDLONG              LcPipeMask;
    PAPPLI_INFO            LcPtAppli;
    WORD                   LcPipeIndex;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq       = (LPPIPE_CMD_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReply     = (LPPIPE_CMD_RESP_INFO)PmBlocRetour;

    // Retrieve application handle
    // -----------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );
    LcPtAppli = &(TbAppliInfo[LcAppIndex]);

    // Retrieve pipe mask
    // ------------------
    LcPipeMaskOut = LcLPtrReq->spqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->spqInPipeMask64.QuadPart;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Retrieve the pipe in and out mask to the masks giving all the
    // started pipes.
    // -------------------------------------------------------------
    StartedPipeInMask  &= ~LcPipeMaskIn;
    StartedPipeOutMask &= ~LcPipeMaskOut;

    // A start on time code is in progress. That means the start
    // conditions are not filled.
    // ---------------------------------------------------------
    if ( StartOnTimeCodeInProgress == TRUE )
    {
        // Retrieve the pipe in and out mask to the masks giving all the
        // started pipes on time code.
        // -------------------------------------------------------------
        StartOnTimeCodeInMask  &= ~LcPipeMaskIn;
        StartOnTimeCodeOutMask &= ~LcPipeMaskOut;

        if (    ( StartOnTimeCodeInMask  == 0 )
             && ( StartOnTimeCodeOutMask == 0 ) )
        {
            // Cancel the start on time-code because no more pipes want
            // to start on it.
            // --------------------------------------------------------
            LcCurrentBoard = StartOnTimeCodeBoardNum;

            (void)  CUR_PROTOCOL_PTR->IClock_TimeCodeConfig(	  TRUE,   // Cancel
																  0,      // NbOfTimeCode
																  0,      // SamplesCount
																  NULL ); // TabTimeCode

            StartOnTimeCodeInProgress = FALSE;
            StartOnTimeCodeBoardNum   = 0;
            StartOnTimeCodeInMask     = 0;
            StartOnTimeCodeOutMask    = 0;
        }
    }

    // Remember how many pipes have been encountered
    // ---------------------------------------------
    LcNbPipe = 0 ;
    LcCurrentBoard = 0;

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;

            // FS - 19/02/1997 - purge stopped record pipes
            LcFirstRecordPipe = LcNbPipe ;

            break;
        }

        while ( LcPipeMask != 0 )
        {
            // Retrieve pipes
            // --------------
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // "check pipes : no synchro inter carte" !
            // if on different cards, don't use
            if(LcNbPipe == 0)
            {
                LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
            }
            else
            {
                if( LcCurrentBoard != LcPtPipe[LcPipeIndex].piNumBoard )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_INVALID_PIPE );
                    return;
                }
            }

            // Fill TARGET_INFO structure
            TbTargetInf[LcNbPipe].tgCarte = LcCurrentBoard;
            TbTargetInf[LcNbPipe].tgCaracPipeVoie = LcOper;
            TbTargetInf[LcNbPipe].tgPipe = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
            TbTargetInf[LcNbPipe].tgAudio = TbTargetInf[LcNbPipe].tgPipe;
            TbTargetInf[LcNbPipe].tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;

            // Free any effect context used by the pipe
            // ----------------------------------------
            if ( LcOper == OPER_PLAY )
            {
                PBOARD_INFO LcBoardInfoPtr ;
                PDSP_INFO   LcDspInfoPtr ;
                DWORD       k ;

                LcBoardInfoPtr  = &(APH_Board_Info_Array[LcCurrentBoard]);
                LcDspInfoPtr    = &((LcBoardInfoPtr->biTbDspInfo)[0]);
                for ( k = 0 ; k < MAX_CONTEXT ; k ++ )
                {
                    LcDspInfoPtr->dsContextMaskOutPipes[k] &= ~UTIMask64bit( LcPipeIndex );

                    // If nobody uses the context, then flags it as such
                    //
                    if ( !LcDspInfoPtr->dsContextMaskOutPipes[k] )
                    {
                        LcDspInfoPtr->dsContextID[k] = 0    ;
                    }
                }
            }

            LcNbPipe++;

        } // for each pipe of the mask

    }
    // for output then input pipes

    // First do a pause on all pipes
    // ------------------------------

    ASSERT(LcNbPipe != 0);

    LcRet = CUR_PROTOCOL_PTR->IFlow_PipePause( TbTargetInf, LcNbPipe, START_PAUSE_IMMEDIATE );
    //LcRet = TogglePipeState( PAUSE_STATE, START_PAUSE_IMMEDIATE, LcNbPipe, TbTargetInf, NULL );

    // Get the sample count on the 1st pipe
    // ------------------------------------
    if ( LcRet == SUCCESS )
    {
        PCX_TIME    LcPipeTime;

        LcRet = CUR_PROTOCOL_PTR->IFlow_PipeSampleCount( TbTargetInf, &LcPipeTime );
        LcLPtrReply->sprHour = LcPipeTime ;
    }

    if ( LcRet == SUCCESS )
    {
        // FS - 19/02/1997 - purge stopped record pipes
        LcPipeMask = LcPipeMaskIn ;
        for ( i = LcFirstRecordPipe ; i < LcNbPipe ; i ++ )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

		    CUR_PROTOCOL_PTR->IEvent_WaitEndOfCoding( &TbTargetInf[i] );

            LcRet = CUR_PROTOCOL_PTR->IBuffer_TransferSoundPipe(
                        &(TbTargetInf[i]),
                        TRUE,               // purge and force transfer
                        TbPipeInInfo[LcPipeIndex].piNbMaxFlux );

            if ( LcRet != SUCCESS )
            {
                PBOARD_INFO LcBoardInfoPtr ;
                PDSP_INFO   LcDspInfoPtr ;
                LcBoardInfoPtr  = &(APH_Board_Info_Array[TbTargetInf[i].tgCarte]);
                LcDspInfoPtr    = &((LcBoardInfoPtr->biTbDspInfo)[0]);

                APHKeepDspError( LcDspInfoPtr, LcRet );
            }
        }

        // For each pipe, let's stop it, and reset it
        // -------------------------------------------
        for ( i = 0 ; i < LcNbPipe ; i ++ )
        {
            LcCurrentBoard = TbTargetInf[i].tgCarte;

            LcRet = CUR_PROTOCOL_PTR->IFlow_PipeStop( &(TbTargetInf[i]) );
            if ( LcRet != SUCCESS )
            {
                LcLPtrRepHeader->rhCptr = LcRet;
            }

            // Purge waiting end of play if any
            // Purge any effect context loaded
            // ---------------------------------
            if ( TbTargetInf[i].tgCaracPipeVoie == OPER_PLAY )
            {
                APHEndPlayPipe( &(TbTargetInf[i]) );
            }
            else
            {
                APHEndRecordPipe( &(TbTargetInf[i]) );
            }

            // Purge wait for notification
            // ----------------------------
            // ## FS (25/11/97) -- No need to register a
            // notification request. This is simply a flsuh of
            // any wait for notification request
            // bug fix for FA#78
            // ------------------------------------------------------
            APHNotifyPipe( &(TbTargetInf[i]), FALSE );
        }

        DOUT(DBG_PRINT, ("StopPipeFct : EndOfErrors"));

        // Purge waiting errors if any
        // ----------------------------
        // ## FS (18/02/1998) -- FA #130
        // Remove useless (and moreover bogus) lines assigning LcPtAppli
        // -------------------------------------------------------------
        LcPipeMask = LcPipeMaskOut ;
        for ( i = 0 ; i < LcFirstRecordPipe ; i ++ )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            TbPipeOutInfo[LcPipeIndex].piError        = SUCCESS;
            TbPipeOutInfo[LcPipeIndex].piErrorPipe    = 0;
            TbPipeOutInfo[LcPipeIndex].piErrorStream  = 0;

            // MBR (2003/05/23) it's the driver that counts the playback bytes now
            // reset the counters for all the streams here
            BZERO2( TbOutStreamBytePos[LcPipeIndex], PCX_TIME, MAX_STREAM_PIPE );
        }
        LcPipeMask = LcPipeMaskIn ;
        for ( i = LcFirstRecordPipe ; i < LcNbPipe ; i ++ )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // FS - 04/03/1997
            TbPipeInInfo[LcPipeIndex].piConversionEnded = FALSE ;

            TbPipeInInfo[LcPipeIndex].piError       = SUCCESS;
            TbPipeInInfo[LcPipeIndex].piErrorPipe   = 0;
            TbPipeInInfo[LcPipeIndex].piErrorStream = 0;
        }
    }
    else
    {
        LcLPtrRepHeader->rhCptr = LcRet;
    }

}


// *************************************************************************
//
// STATIC VOID SetOutPipeSettings( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//  Process the set out pipe parameters command.
//
//  FH Creation
// *************************************************************************
//
// This function assumes :
//   - There is only one ressource UER for the entire card
//   - A pipe is on only one card
//
// *************************************************************************
STATIC VOID SetOutPipeSettings(
                                IN LPDWORD PmBlocRequete,
                                OUT LPDWORD PmBlocRetour    )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPOUT_PIPE_SET_REQ_INFO     LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;
    TARGET_INFO                 LcTargetInf;

    BYTE                        LcAppIndex;
    WORD                        LcPipeIndex;
    DWORDLONG                   LcPipeMaskOut;
    BOOLEAN                     LcParaUerOut     = FALSE;
    BOOLEAN                     LcParaMpegOut    = FALSE;
    BOOLEAN                     LcParaBackwardOut= FALSE;
    BOOLEAN                     LcParaUerExtra   = FALSE;
    WORD                        LcRet;
    WORD                        LcCurrentBoard = 0;

    // Initialisation of pointers
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReq = (LPOUT_PIPE_SET_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    // Retrieve application index
    // **************************
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    if(LcAppIndex == DHSAppHandle)
    {
        LcLPtrRepHeader->rhCptr = DHSSetOutPipeSettings(LcLPtrReq);
        return;
    }

    // Retrieve pipe mask and verify at least one pipe is specified
    // ************************************************************
    LcPipeMaskOut = LcLPtrReq->opqOutPipeMask64.QuadPart ;
    LcRet = ChkPmManyOutputPipeMask( LcAppIndex, LcPipeMaskOut );

    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    // Fill target_info
    // ----------------
    LcTargetInf.tgCaracPipeVoie = OPER_PLAY ;
    LcTargetInf.tgDifferedType  = LcLPtrReq->opqDiffered;
    LcTargetInf.tgHour          = LcLPtrReq->opqHour ;

    if ( LcLPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_UER_MASK )
    {
        LcLPtrReq->opqUer &= UER_MODE_MASK;
        
        // Verify UER parameters values
        // ----------------------------
        if ( ( LcLPtrReq->opqUer != UER_CONSUMER     ) &&
             ( LcLPtrReq->opqUer != UER_PROFESSIONAL ) &&
             ( LcLPtrReq->opqUer != UER_NO_MODE )    )
        {
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_BAD_UER_MODE );
            return;
        }

        LcParaUerOut = TRUE;
    }

    if ( LcLPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_UER_EXTRA_MASK )
    {
		if( (LcLPtrReq->opqUerExtra & UER_SRC_OUT_MASK) ||
			(LcLPtrReq->opqUerExtra & UER_ASYNC_DATA) )
		{
			// TODO : SRC OUT not yet implemented
			// use 0x04 or 0x10 as mask ?
			//
			// Test frequency ? return ED_INVALID_FREQUENCY ?
			//
            LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_SRC_UNAVAILABLE );
            return;
		}
        // verify bit-mask
        LcLPtrReq->opqUerExtra &= ~UER_MODE_MASK ;
        LcParaUerExtra = TRUE;
    }

    if ( LcLPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_MPEG_MASK )
    {
        LcParaMpegOut = TRUE;
    }

    if ( LcLPtrReq->opqPar1 & OUT_PIPE_SET_PARAM_BACKWARD_MASK )
    {
        LcParaBackwardOut = TRUE;
    }

    // for all pipes specified
    // ***********************
    while ( LcPipeMaskOut != 0 )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMaskOut );
        LcPipeMaskOut &= ~UTIMask64bit( LcPipeIndex );

        LcTargetInf.tgCarte = LcCurrentBoard = TbPipeOutInfo[LcPipeIndex].piNumBoard;
        LcTargetInf.tgPipe  = TbPipeOutInfo[LcPipeIndex].piNumPipeForDsp;
        LcTargetInf.tgAudio = LcTargetInf.tgPipe ;

        PVOIE_INFO LcVoieOutInfo = &TbVoieOutInfo[LcCurrentBoard][LcTargetInf.tgAudio];

        // Process set UER request
        // -----------------------
        if ( LcParaUerOut || LcParaUerExtra )
        {
            // for all audios
            for(int i=0; i<TbPipeOutInfo[LcPipeIndex].piNbAudio; i++)
            {
                BYTE    LcUerParam;
                WORD    LcAudioNum;

                // all audios of the pipe
                ASSERT( LcVoieOutInfo[i].NumPipe == LcPipeIndex );
                ASSERT( LcVoieOutInfo[i].IndexAppli == (LcAppIndex + 1) );

                LcAudioNum = LcVoieOutInfo[i].AudioNum;

                // combine information opqUer and opqUerExtra !!
                if( LcParaUerOut )
                {
                    LcUerParam = LcLPtrReq->opqUer;

                    if( LcParaUerExtra )
                    {
                        LcUerParam |= LcLPtrReq->opqUerExtra;
                    }
                    else    // if no extra info, get old params for stereo, copyright, ...
                    {
                        LcUerParam |= CUR_COMMANDS_PTR->PIOGetAudioUerOutMode(LcAudioNum) & ~UER_MODE_MASK;
                    }
                }
                else    // no LcParaUerOut
                {
                    LcUerParam  = CUR_COMMANDS_PTR->PIOGetAudioUerOutMode(LcAudioNum) & UER_MODE_MASK;
                    LcUerParam |= LcLPtrReq->opqUerExtra;
                }
                // dont allow UER_NO_MODE and extra parameters
                if( (LcUerParam & UER_MODE_MASK) == UER_NO_MODE )
                {
                    LcUerParam = UER_NO_MODE;
                }

                // -----------------------------------------------------------------------
                // FH bug correction:
                // If only one Pipe use the UER on the board, we can change UER parameters
                // for this pipe (without call this command with NO_UER parameter).
                // -----------------------------------------------------------------------

                DOUT(DBG_PRINT, ("### SetOutPipeSett Board %d Audio %d param %x", LcCurrentBoard, LcAudioNum, LcUerParam));

                // If the board still use UER with an another frequency
                // or with an another UER type: error
                // ----------------------------------------------------
                if ( LcUerParam != UER_NO_MODE )
                {
                    DWORD          LcFrequency;

                    // this audio is controlled by DHS !
                    if( APHChkIsAudioDHSControlled(LcCurrentBoard, LcAudioNum, FALSE) )
                    {
                        LOAD_PCX_ERROR(LcLPtrRepHeader->rhCptr, ED_AUDIO_MANAGED_BY_DHS_PANEL);
                        return;
                    }

                    // get the current frequency
                    LcFrequency = CUR_COMMANDS_PTR->PIOGetActualClockFrequency();

                    // If the frequency is unknown we use the frequency given by the user.
                    // Otherwise, we use the computed frequency.
                    // ----------------------------------------------------------
                    if ( LcFrequency == 0 )
                    {
                       // TODO - Test of LcLPtrReq->opqFrequency ?
                       LcFrequency = LcLPtrReq->opqFrequency;
                    }

                    // It's OK, let's process the command
                    // ----------------------------------
                    LcRet = CUR_COMMANDS_PTR->PIOSetParamOutPipe(   LcAudioNum,
                                                                    LcFrequency,
                                                                    LcUerParam );

                    if ( LcRet != SUCCESS ) {
                        LcLPtrRepHeader->rhCptr = LcRet;
                        return;
                    }

                } // mode
            }
        } // UERout ou UER extra

        // Process set MPEG request
        // ------------------------
        // ## FM (09/23/97) -- Take into account the backward mode
        //
        if ( LcParaMpegOut || LcParaBackwardOut )
        {
            LcRet = CUR_PROTOCOL_PTR->IFlow_PipeOutSetParam(
                &LcTargetInf,
                LcParaMpegOut,
                LcLPtrReq->opqMpeg,
                LcParaBackwardOut,
                LcLPtrReq->opqBackward) ;

            if ( LcRet != SUCCESS )
            {
                LcLPtrRepHeader->rhCptr = LcRet;
                return;
            }
        }

    } // End for each pipe

    LcLPtrRepHeader->rhCptr = SUCCESS ;
}


// *************************************************************************
//
// STATIC VOID SetOutPipeSettings( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//  Process the set out pipe parameters command.
//
//  FH Creation
// *************************************************************************
//
// This function assumes :
//
// *************************************************************************
STATIC VOID SetOutPipeEffects(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPOUT_PIPE_EFFECTS_REQ_INFO LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;
    TARGET_INFO                 LcTargetInf;
    WORD                        LcExpectedNumberOfParams;
    BYTE                        LcAppIndex;
    WORD                        LcPipeIndex;
    DWORDLONG                   LcPipeMaskOut;
    WORD                        LcRet = SUCCESS ;
    WORD                        LcCurrentBoard ;

    // Initialisation of pointers
    // --------------------------
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReq = (LPOUT_PIPE_EFFECTS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    LcLPtrRepHeader->rhCptr = ED_UNAVAILABLE_FEATURE;
    return;
}



// *************************************************************************
//
// STATIC VOID SetInPipeSettings( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
//  Process the set input pipe parameters command.
//
// *************************************************************************
//
// This function assumes :
//   - A pipe is on only one card
//
// *************************************************************************
STATIC VOID SetInPipeSettings(
    IN LPDWORD PmBlocRequete    ,
    OUT LPDWORD PmBlocRetour    )
{
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    LPIN_PIPE_SET_REQ_INFO     LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;

    BYTE                        LcAppIndex;
    WORD                        LcPipeIndex;
    DWORDLONG                   LcPipeMaskIn;
    WORD                        LcRet;
	WORD						LcCurrentBoard;
	DWORDLONG					LcCurrentAudioMask;
	BYTE						LcUER_SRC;

    // Initialisation of pointers
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;
    LcLPtrReq = (LPIN_PIPE_SET_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;

    // Retrieve application index
    // **************************
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe mask and verify at least one pipe is specified
    // ************************************************************
    LcPipeMaskIn = LcLPtrReq->ipqInPipeMask64.QuadPart;

    /* BAD VALIDATION BITS ************************************************************/
    if( ( LcLPtrReq->ipqPar1 & ~( IN_PIPE_SET_PARAM_SOURCE_MASK | IN_PIPE_SET_PARAM_MIC_48V_MASK | IN_PIPE_SET_PARAM_UER_SRC_MASK ) ) ||
        ( LcLPtrReq->ipqPar2 != 0 ) )
    {
        LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_BAD_VALIDATION_MASK );
        return;
    }

    /* SRC HANDLING ************************************************************/
	if( LcLPtrReq->ipqPar1 & IN_PIPE_SET_PARAM_UER_SRC_MASK )
	{
		LcUER_SRC = LcLPtrReq->ipqADProperties & (IN_PROPERTIES_UER_SYNC | IN_PROPERTIES_UER_ASYNC);
		// combination of both flags forbidden
		if( LcUER_SRC == (IN_PROPERTIES_UER_SYNC | IN_PROPERTIES_UER_ASYNC) )
		{
			LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_BAD_VALIDATION_MASK );
			return;
		}
	}
	else	LcUER_SRC = 0;	// automatic SRC assignment

    if(LcAppIndex == DHSAppHandle)
    {
        if( LcLPtrReq->ipqPar1 & IN_PIPE_SET_PARAM_MIC_48V_MASK )
            LcLPtrRepHeader->rhCptr = WD_NO_HARDWARE_SUPPORT;   // TODO
        else
            LcLPtrRepHeader->rhCptr = DHSSetInPipeSettings( LcPipeMaskIn,
                                                            LcLPtrReq->ipqPar1,
                                                            LcLPtrReq->ipqSource,
                                                            LcUER_SRC );
        return;
    }

    /* 
    CHECK PIPE/APP HANDLES
    */
    LcRet = ChkPmManyInputPipeMask( LcAppIndex, LcPipeMaskIn );

    if ( LcRet != SUCCESS ) {
        LcLPtrRepHeader->rhCptr = LcRet ;
        return;
    }

    /* 
    PROCESS EACH PIPE
    */

	LcLPtrRepHeader->rhCptr = SUCCESS;

    while ( LcPipeMaskIn != 0 )
    {
        LcPipeIndex = UTI64Mask2Word( LcPipeMaskIn );
        LcPipeMaskIn &= ~UTIMask64bit( LcPipeIndex );

        // Prepare a target
        // ---------------------------------
        LcCurrentBoard = TbPipeInInfo[LcPipeIndex].piNumBoard ;

        WORD LcTargetAudio = TbPipeInInfo[LcPipeIndex].piNumPipeForDsp ;

        if ( LcLPtrReq->ipqPar1 & (IN_PIPE_SET_PARAM_SOURCE_MASK | IN_PIPE_SET_PARAM_UER_SRC_MASK) )
       /* INPUT SOURCE SETTINGS ****************************************************************/
       {
            BYTE LcSource;
            // Build the Mask of pipe audios
            //
            LcCurrentAudioMask = APH_Get_Pipe_Audio_Mask(LcPipeIndex);

            // audio is DHS controlled, forbidden for normal applications
            if( APHChkIsAudioDHSControlled( LcCurrentBoard, LcTargetAudio, TRUE ) )
            {
			    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_AUDIO_MANAGED_BY_DHS_PANEL );
			    return;
            }

            if( LcLPtrReq->ipqPar1 & IN_PIPE_SET_PARAM_SOURCE_MASK )
            {
                LcSource = LcLPtrReq->ipqSource;

                if ( LcSource == DATA_FROM_NO_INPUT )
                {
                    /* 
                        HANDLE RELEASE SOURCE FOR PIPE AUDIOS
                    */
                    CUR_PROTOCOL_PTR->IResource_Source_Release_Audios(LcCurrentAudioMask);
                    continue;
                }
            }
            else    // IN_PIPE_SET_PARAM_UER_SRC_MASK
            {
                if( LcUER_SRC == 0 )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_BAD_VALIDATION_MASK );
                    return;
                }

                LcSource = CUR_COMMANDS_PTR->PIOGetOneAudioSource(LcTargetAudio);

                if( (LcSource != DATA_FROM_DIGITAL_SYNCHRO) &&
                    (LcSource != DATA_FROM_DIGITAL_DATA) &&
                    (LcSource != DATA_FROM_SYNC_CONNECTOR) )
                {
                    LOAD_PCX_ERROR( LcLPtrRepHeader->rhCptr, ED_BAD_SOURCE );
                    return;
                }
            }

            /* 
            CHECK SOURCE COMPATIBILITY
            prevent "source[2n] != source [2n+1]"
			and
            ADDREF SOURCE FOR PIPE AUDIOS
            */
            LcRet = CUR_PROTOCOL_PTR->IResource_Source_Change_Audios( LcCurrentAudioMask, LcSource, LcUER_SRC );
            if (LcRet != SUCCESS)
			{
				LcLPtrRepHeader->rhCptr = LcRet;	// warning possible -> continue !
				if(LcRet & ERROR_MASK) break;
			}
        }

        /* OTHER SETTINGS ****************************************************************/
        if( LcLPtrReq->ipqPar1 & IN_PIPE_SET_PARAM_MIC_48V_MASK )
        {
            BOOLEAN LcEnable = (BOOLEAN) ((LcLPtrReq->ipqADProperties & IN_PROPERTIES_MIC_48V) != 0);

            // enable/disable 48V power supply for Microphone input
            LcRet = CUR_COMMANDS_PTR->PIOEnablePhantomPowerSupply( LcTargetAudio, 0, LcEnable );
            if (LcRet != SUCCESS)
			{
				LcLPtrRepHeader->rhCptr = LcRet;
				if(LcRet & ERROR_MASK) break;
			}
        }
    } // foreach pipe

    return;
}


// *************************************************************************
//
// STATIC VOID OfflineLoopFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the offline loop pipe command.
//  Update the offline loop pipe command response bloc.
//
// *************************************************************************
STATIC VOID OfflineLoopFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPOFFLINE_LOOPS_REQ_INFO    LcRequestPtr    ;
    LPBC_HEADER_INFO            LcReqHeaderPtr  ;
    LPRESP_HEADER_INFO          LcReplyPtr      ;
    LPLOOPS_DESC_INFO           LcLoopDescPtr   ;
    BYTE                        LcAppIndex      ;
    BYTE                        LcLoopNum       ;
    BYTE                        i, j            ;
    WORD                        LcError         ;
    WORD                        LcCurrentBoard;
    DWORDLONG                   LcOutPipeMask, LcInPipeMask ;
    WORD                        LcOutPipeIndex, LcInPipeIndex ;
    DWORD                       LcOutAudioMask, LcInAudioMask ;
    WORD                        LcOutAudioIndex = 0;
    WORD                        LcInAudioIndex = 0;
    WORD                        LcOutAudio, LcInAudio ;
    BOOLEAN                     LcFound         ;
    TARGET_INFO                 LcTarget        ;
    PPIPE_INFO                  LcPipeInfOutPtr, LcPipeInfInPtr ;
    PVOIE_INFO                  LcVoieInfOutPtr, LcVoieInfInPtr ;

    // Initialisation of pointers
    // ***************************
    //  Correction bug: FH
    //  after:
    //  LcReplyPtr      = (LPRESP_HEADER_INFO)PmBlocRequete ;   ??
    //  LcRequestPtr = (LPOFFLINE_LOOPS_REQ_INFO)PmBlocRetour;  ??
    //  LcReqHeaderPtr  = (LPBC_HEADER_INFO)PmBlocRetour;       ??



    LcReplyPtr      = (LPRESP_HEADER_INFO)PmBlocRetour ;
    LcRequestPtr    = (LPOFFLINE_LOOPS_REQ_INFO)PmBlocRequete ;
    LcReqHeaderPtr  = (LPBC_HEADER_INFO)PmBlocRequete;

    // Set reply size
    // **************
    LcReplyPtr->rhSize = 0 ;

    // Retrieve application index
    // **************************
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Retrieve the number of loop backs specified
    // *******************************************
    LcLoopNum = LcRequestPtr->olqNbLoops ;

    // init in case of...
    LcCurrentBoard = 0;

    // No error occured yet
    // *******************
    LcError = SUCCESS ;

    // Get position of first loop descriptor
    // *************************************
    LcLoopDescPtr = (LPLOOPS_DESC_INFO)( LcRequestPtr + 1 ) ;

    // process each loop descriptor
    // ****************************
    for ( i = 0 ; i < LcLoopNum ; i++, LcLoopDescPtr++ )
    {
        // retrieve out pipe mask
        // **********************
        LcOutPipeMask = LcLoopDescPtr->ldOutPipeMask64.QuadPart ;

        // verify this mask specify one pipe ...
        // ... and only one pipe
        // and check the pipe actually belongs to the application
        // **********************************************
        LcError = ChkPmOneOutputPipeMask(   LcAppIndex,
                                            LcOutPipeMask,
                                            &LcOutPipeIndex,
                                            &LcPipeInfOutPtr,
                                            &LcVoieInfOutPtr,
                                            &LcTarget );
        if ( LcError != SUCCESS )
        {
            break ;
        }

        // Remember the board and DSP for this pipe
        // ****************************************
        LcCurrentBoard = LcPipeInfOutPtr->piNumBoard ;

        // retrieve out audio mask
        // ************************
        LcOutAudioMask = (DWORD) LcLoopDescPtr->ldOutMask ;

        // verify this mask specify one audio ...
        // **********************************************
        if ( LcOutAudioMask == 0 )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        LcOutAudio = UTIDWMask2Word( LcOutAudioMask ) ;

        // ... and only one audio
        // **********************************************
        if ( LcOutAudioMask & ~UTIMaskDWord( LcOutAudio ) )
        {
            LOAD_PCX_ERROR( LcError, ED_MANY_AUDIOS_REFUSED );
            break ;
        }

        // does the audio actually belongs to the pipe ?
        // **********************************************
        if ( LcOutAudio >= LcPipeInfOutPtr->piNbAudio )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // seek for this audio
        // ********************
        LcFound = FALSE ;
        for ( j = 0 ; (j < LcPipeInfOutPtr->piNbAudio) && !LcFound ; j ++ )
        {
            // FS - 26/03/1997 - Add check for entry validity (IndexAppli)
            // ------------------------------------------------------------
            ASSERT( LcVoieInfOutPtr[j].NumPipe == LcOutPipeIndex );
            ASSERT( LcVoieInfOutPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            LcFound = ( LcVoieInfOutPtr[j].AppAudioNum == LcOutAudio ) ;
        }

        if ( !LcFound )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // FH not used 12/03/97
        // LcOutAudioIndex = j ;

        // retrieve in pipe mask
        // *********************
        LcInPipeMask = LcLoopDescPtr->ldInPipeMask64.QuadPart ;

        // verify this mask specify one pipe ...
        // ... and only one pipe
        // and check the pipe actually belongs to the application
        // **********************************************
        LcError = ChkPmOneInputPipeMask(    LcAppIndex,
                                            LcInPipeMask,
                                            &LcInPipeIndex,
                                            &LcPipeInfInPtr,
                                            &LcVoieInfInPtr,
                                            &LcTarget );
        if ( LcError != SUCCESS )
        {
            break ;
        }

        // Verify the board and DSP for this pipe
        // (they'd better be the same as output pipe
        // ******************************************
        if ( LcCurrentBoard != LcPipeInfInPtr->piNumBoard )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_BOARD );
            break ;
        }

        // retrieve in audio mask
        // ************************
        LcInAudioMask = (DWORD) LcLoopDescPtr->ldInMask ;

        // verify this mask specify one audio ...
        // **********************************************
        if ( LcInAudioMask == 0 )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        LcInAudio = UTIDWMask2Word( LcInAudioMask ) ;

        // ... and only one audio
        // **********************************************
        if ( LcInAudioMask & ~UTIMaskDWord( LcInAudio ) )
        {
            LOAD_PCX_ERROR( LcError, ED_MANY_AUDIOS_REFUSED );
            break ;
        }

        // does the audio actually belongs to the pipe ?
        // **********************************************
        if ( LcInAudio >= LcPipeInfInPtr->piNbAudio )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // seek for this audio
        // ********************
        LcFound = FALSE ;
        for ( j = 0 ; (j < LcPipeInfInPtr->piNbAudio) && !LcFound ; j ++ )
        {
            ASSERT( LcVoieInfInPtr[j].NumPipe == LcInPipeIndex );
            ASSERT( LcVoieInfInPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            LcFound = ( LcVoieInfInPtr[j].AppAudioNum == LcInAudio ) ;
        }

        if ( !LcFound )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }
    }

    if ( LcError != SUCCESS )
    {
        // Set error code returned
        // ***********************
        LcReplyPtr->rhCptr = LcError ;

        return ;
    }

    // All parameters are checked ok
    // Process the command and actually connect the audios
    // ---------------------------------------------------

    // Get position of first loop descriptor
    // *************************************
    LcLoopDescPtr = (LPLOOPS_DESC_INFO)( LcRequestPtr + 1 ) ;

    // process each loop descriptor
    // ****************************
    for ( i = 0 ; i < LcLoopNum ; i++, LcLoopDescPtr++ )
    {
        // Retrieve output pipe index
        // **************************
        LcOutPipeIndex = UTI64Mask2Word( LcLoopDescPtr->ldOutPipeMask64.QuadPart );

        // retrieve out Pipe audio number
        // ******************************
        LcOutAudioMask = (DWORD) LcLoopDescPtr->ldOutMask ;
        LcOutAudio = UTIDWMask2Word( LcOutAudioMask ) ;

        // retrieve DSP audio number
        // *************************
        LcCurrentBoard = TbPipeOutInfo[LcOutPipeIndex].piNumBoard;
        LcVoieInfOutPtr = &TbVoieOutInfo[LcCurrentBoard][TbPipeOutInfo[LcOutPipeIndex].piNumPipeForDsp];

        // FH bug correction 12/02/97
        for ( j = 0 ; j < TbPipeOutInfo[LcOutPipeIndex].piNbAudio; j ++ )
        {
            ASSERT( LcVoieInfOutPtr[j].NumPipe == LcOutPipeIndex );
            ASSERT( LcVoieInfOutPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            if ( LcVoieInfOutPtr[j].AppAudioNum == LcOutAudio )
            {
                LcOutAudioIndex = j ;
                break;
            }
        }

        // Retrieve input pipe index
        // **************************
        LcInPipeIndex = UTI64Mask2Word( LcLoopDescPtr->ldInPipeMask64.QuadPart );

        // retrieve in audio number for pipe
        // *********************************
        LcInAudioMask = (DWORD) LcLoopDescPtr->ldInMask ;
        LcInAudio = UTIDWMask2Word( LcInAudioMask ) ;


        // retrieve DSP audio number
        // *************************
        LcVoieInfInPtr = &TbVoieInInfo[LcCurrentBoard][TbPipeInInfo[LcInPipeIndex].piNumPipeForDsp];

        // FH bug correction 12/02/97
        for ( j = 0 ; j < TbPipeInInfo[LcInPipeIndex].piNbAudio; j ++ )
        {
            ASSERT( LcVoieInfInPtr[j].NumPipe == LcInPipeIndex );
            ASSERT( LcVoieInfInPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            if ( LcVoieInfInPtr[j].AppAudioNum == LcInAudio )
            {
                LcInAudioIndex = j ;
                break;
            }
        }

        // Send the request
        // ----------------
        LcError = CUR_PROTOCOL_PTR->IFlow_AudioConnect(	LcVoieInfOutPtr[LcOutAudioIndex].AudioNum  ,
														LcVoieInfInPtr[LcInAudioIndex].AudioNum    );

        // Give up if an error occurs
        // --------------------------
        if ( LcError != SUCCESS )
        {
            break ;
        }
    }

    // Set error code returned
    // ***********************
    LcReplyPtr->rhCptr = LcError ;

}

// *************************************************************************
//
// STATIC VOID MonitoringChangeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the change monitoring source (pipe) command.
//  Update the change monitoring source (pipe) command response bloc.
//
// *************************************************************************
STATIC VOID MonitoringChangeFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO            LcReqHeaderPtr  ;
    LPRESP_HEADER_INFO          LcReplyPtr      ;
    LPMONIT_CHANGES_REQ_INFO    LcRequestPtr    ;
    LPCHANGE_DESC_INFO          LcChangeDescPtr ;
    BYTE                        LcAppIndex      ;
    BYTE                        LcNbChanges     ;
    BYTE                        i, j            ;
    WORD                        LcError         ;
    WORD                        LcCurrentBoard  ;
    DWORDLONG                   LcOutPipeMask, LcInPipeMask ;
    WORD                        LcOutPipeIndex, LcInPipeIndex ;
    DWORD                       LcOutAudioMask, LcInAudioMask ;
    WORD                        LcOutAudioIndex = 0;
    WORD                        LcInAudioIndex = 0;
    WORD                        LcOutAudio, LcInAudio ;
    BOOLEAN                     LcFound         ;
    TARGET_INFO                 LcTarget        ;
    PPIPE_INFO                  LcPipeInfOutPtr, LcPipeInfInPtr ;
    PVOIE_INFO                  LcVoieInfOutPtr, LcVoieInfInPtr ;

    // Pointers initialisation
    // ***********************
    LcReplyPtr      = (LPRESP_HEADER_INFO)PmBlocRetour ;
    LcRequestPtr    = (LPMONIT_CHANGES_REQ_INFO)PmBlocRequete ;
    LcReqHeaderPtr  = (LPBC_HEADER_INFO)PmBlocRequete;

    // Set reply size
    // **************
    LcReplyPtr->rhSize = 0 ;

    // Retrieve application index
    // **************************
    LcAppIndex = DecodeHandle( LcReqHeaderPtr->hdHandle );

    // Retrieve the number of structures specified
    // *******************************************
    LcNbChanges = LcRequestPtr->mcqNbChanges ;

    // init in case of..
    LcCurrentBoard = 0;

    // No error occured yet
    // *******************
    LcError = SUCCESS ;

    // Get position of first structure
    // *******************************
    LcChangeDescPtr = (LPCHANGE_DESC_INFO)( LcRequestPtr + 1 ) ;

    if( LcAppIndex == DHSAppHandle )
    {
        LcReplyPtr->rhCptr = DHSChangeMonitoringSrc( LcChangeDescPtr, LcNbChanges );
        return ;
    }

    // process each structure
    // **********************
    for ( i = 0 ; i < LcNbChanges ; i++, LcChangeDescPtr++ )
    {
        // retrieve out pipe mask
        // **********************
        LcOutPipeMask = LcChangeDescPtr->cdOutPipeMask64.QuadPart ;

        // verify this mask specify one pipe ...
        // ... and only one pipe
        // and check the pipe actually belongs to the application
        // **********************************************
        LcError = ChkPmOneOutputPipeMask(   LcAppIndex,
                                            LcOutPipeMask,
                                            &LcOutPipeIndex,
                                            &LcPipeInfOutPtr,
                                            &LcVoieInfOutPtr,
                                            &LcTarget );
        if ( LcError != SUCCESS )
        {
            break ;
        }

        // Remember the board and DSP for this pipe
        // ****************************************
        LcCurrentBoard = LcPipeInfOutPtr->piNumBoard ;

        // retrieve out audio mask
        // ************************
        LcOutAudioMask = (DWORD) LcChangeDescPtr->cdOutMask ;

        // verify this mask specify one audio ...
        // **********************************************
        if ( LcOutAudioMask == 0 )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        LcOutAudio = UTIDWMask2Word( LcOutAudioMask ) ;

        // ... and only one audio
        // **********************************************
        if ( LcOutAudioMask & ~UTIMaskDWord( LcOutAudio ) )
        {
            LOAD_PCX_ERROR( LcError, ED_MANY_AUDIOS_REFUSED );
            break ;
        }

        // does the audio actually belongs to the pipe ?
        // **********************************************
        if ( LcOutAudio >= LcPipeInfOutPtr->piNbAudio )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // seek for this audio
        // ********************
        LcFound = FALSE ;

        for ( j = 0 ; j < LcPipeInfOutPtr->piNbAudio ; j ++ )
        {
            // FS - 26/03/1997 - Add check for entry validity (IndexAppli)
            // ------------------------------------------------------------
            ASSERT( LcVoieInfOutPtr[j].NumPipe == LcOutPipeIndex );
            ASSERT( LcVoieInfOutPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            LcFound = ( LcVoieInfOutPtr[j].AppAudioNum == LcOutAudio ) ;
            if( LcFound )
                break;
        }

        if ( !LcFound )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // if this audio is controlled by DHS : error
        // test only audio out. audio in is always possible then !
        if( APHChkIsAudioDHSControlled(LcCurrentBoard, LcVoieInfOutPtr[j].AudioNum, FALSE) )
        {
            LOAD_PCX_ERROR( LcError, ED_AUDIO_MANAGED_BY_DHS_PANEL );
            break ;
        }

        // retrieve in pipe mask
        // *********************
        LcInPipeMask = LcChangeDescPtr->cdInPipeMask64.QuadPart ;

        // verify this mask specify one pipe ...
        // ... and only one pipe
        // and check the pipe actually belongs to the application
        // **********************************************
        LcError = ChkPmOneInputPipeMask(    LcAppIndex,
                                            LcInPipeMask,
                                            &LcInPipeIndex,
                                            &LcPipeInfInPtr,
                                            &LcVoieInfInPtr,
                                            &LcTarget );
        if ( LcError != SUCCESS )
        {
            break ;
        }

        // Verify the board and DSP for this pipe
        // (they'd better be the same as output pipe)
        // ******************************************
        if ( LcCurrentBoard != LcPipeInfInPtr->piNumBoard )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_BOARD );
            break ;
        }

        // retrieve in audio mask
        // ************************
        LcInAudioMask = (DWORD) LcChangeDescPtr->cdInMask ;

        // verify this mask specify one audio ...
        // **********************************************
        if ( LcInAudioMask == 0 )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        LcInAudio = UTIDWMask2Word( LcInAudioMask ) ;

        // ... and only one audio
        // **********************************************
        if ( LcInAudioMask & ~UTIMaskDWord( LcInAudio ) )
        {
            LOAD_PCX_ERROR( LcError, ED_MANY_AUDIOS_REFUSED );
            break ;
        }

        // does the audio actually belongs to the pipe ?
        // **********************************************
        if ( LcInAudio >= LcPipeInfInPtr->piNbAudio )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }

        // seek for this audio
        // ********************
        LcFound = FALSE ;

        for ( j = 0 ; (j < LcPipeInfInPtr->piNbAudio) && !LcFound ; j ++ )
        {
            ASSERT( LcVoieInfInPtr[j].NumPipe == LcOutPipeIndex );
            ASSERT( LcVoieInfInPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            LcFound = ( LcVoieInfInPtr[j].AppAudioNum == LcInAudio ) ;
            if(LcFound)
                break;
        }

        if ( !LcFound )
        {
            LOAD_PCX_ERROR( LcError, ED_INVALID_PIPE_AUDIO );
            break ;
        }
    }

    if ( LcError != SUCCESS )
    {
        // Set error code returned
        // ***********************
        LcReplyPtr->rhCptr = LcError ;

        return ;
    }

    // All parameters are checked ok
    // Process the command and actually change the monitoring source
    // -------------------------------------------------------------

    // Get position of first loop descriptor
    // *************************************
    LcChangeDescPtr = (LPCHANGE_DESC_INFO)( LcRequestPtr + 1 ) ;

    // process each loop descriptor
    // ****************************
    for ( i = 0 ; i < LcNbChanges ; i++, LcChangeDescPtr++ )
    {
        // Retrieve output pipe index
        // **************************
        LcOutPipeIndex = UTI64Mask2Word( LcChangeDescPtr->cdOutPipeMask64.QuadPart );

        // retrieve out Pipe audio number
        // ******************************
        LcOutAudioMask = (DWORD) LcChangeDescPtr->cdOutMask ;
        LcOutAudio     = UTIDWMask2Word( LcOutAudioMask ) ;

        // retrieve DSP audio number
        // *************************
        LcCurrentBoard = TbPipeOutInfo[LcOutPipeIndex].piNumBoard;
        LcVoieInfOutPtr = &TbVoieOutInfo[LcCurrentBoard][TbPipeOutInfo[LcOutPipeIndex].piNumPipeForDsp];

        for ( j = 0 ; j < TbPipeOutInfo[LcOutPipeIndex].piNbAudio; j ++ )
        {
            ASSERT( LcVoieInfOutPtr[j].NumPipe == LcOutPipeIndex );
            ASSERT( LcVoieInfOutPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            if ( LcVoieInfOutPtr[j].AppAudioNum == LcOutAudio )
            {
                LcOutAudioIndex = j ;
                break;
            }
        }

        // Retrieve input pipe index
        // **************************
        LcInPipeIndex = UTI64Mask2Word( LcChangeDescPtr->cdInPipeMask64.QuadPart );

        // retrieve in audio number for pipe
        // *********************************
        LcInAudioMask = (DWORD) LcChangeDescPtr->cdInMask ;
        LcInAudio     = UTIDWMask2Word( LcInAudioMask ) ;

        // retrieve DSP audio number
        // *************************
        LcVoieInfInPtr = &TbVoieInInfo[LcCurrentBoard][TbPipeInInfo[LcInPipeIndex].piNumPipeForDsp];

        for ( j = 0 ; j < TbPipeInInfo[LcInPipeIndex].piNbAudio; j ++ )
        {
            ASSERT( LcVoieInfInPtr[j].NumPipe == LcInPipeIndex );
            ASSERT( LcVoieInfInPtr[j].IndexAppli == ( LcAppIndex + 1 ) );

            if ( LcVoieInfInPtr[j].AppAudioNum == LcInAudio )
            {
                LcInAudioIndex = j ;
                break;
            }
        }

        // Send the request
        // ----------------
        LcError = CUR_PROTOCOL_PTR->IFlow_ChangeMonitoringSource(	LcVoieInfOutPtr[LcOutAudioIndex].AudioNum  ,
																	LcVoieInfInPtr[LcInAudioIndex].AudioNum    );

        // Give up if an error occurs
        // --------------------------
        if ( LcError != SUCCESS )
        {
            break ;
        }
    }

    // Set error code returned
    // ***********************
    LcReplyPtr->rhCptr = LcError ;

}


// *************************************************************************
//
// STATIC VOID NotifyPipeTimeFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the notify pipe time command.
//
// *************************************************************************
STATIC VOID NotifyPipeTimeFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPNOTIFY_PIPE_TIME_REQ_INFO LcLPtrReq;
    LPBC_HEADER_INFO            LcLPtrReqHeader;
    LPRESP_HEADER_INFO          LcLPtrRepHeader;
    BYTE                        LcAppIndex;
    PPIPE_INFO                  LcPtPipe;
    DWORDLONG                   LcPipeMask, LcPipeMaskIn, LcPipeMaskOut;
    SHORT                       LcPipeIndex;
    WORD                        LcNbPipe = 0 ;
    BYTE                        LcCurrentBoard;
    WORD                        i;
    WORD                        LcRet;
    BYTE                        LcOper;

    // Pointers initialisation
    // ***********************
    LcLPtrReq       = (LPNOTIFY_PIPE_TIME_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application index
    // **************************
    LcAppIndex  = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe mask
    // ******************
    LcPipeMaskOut = LcLPtrReq->nptqOutPipeMask64.QuadPart ;
    LcPipeMaskIn = LcLPtrReq->nptqInPipeMask64.QuadPart ;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Remember differed time (only once is necessary)
    // ------------------------------------------------
    TbTargetInf[0].tgDifferedType = LcLPtrReq->nptqDiffered;
    TbTargetInf[0].tgHour = LcLPtrReq->nptqHour;

    // ## FS (08/09/1997) -- bug fix for FA #55
    // Remember how many pipes have been encountered
    // ---------------------------------------------
    LcNbPipe = 0 ;

    // For all the pipes concerned by the command, verify the command is ok
    // --------------------------------------------------------------------
    for (i=0; i<2; i++)
    {
        switch(i)
        {
          case 0 : LcPipeMask = LcPipeMaskOut;
               LcPtPipe = &TbPipeOutInfo[0];
               LcOper      = OPER_PLAY;
               break;

          case 1 : LcPipeMask = LcPipeMaskIn;
               LcPtPipe = &TbPipeInInfo[0];
               LcOper      = OPER_REC;
               break;
        }

        while ( LcPipeMask != 0 )
        {
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );

            // Processes the corresponding output pipe
            // ---------------------------------------
            TbTargetInf[LcNbPipe].tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;

            // Fill TARGET_INFO structure
            TbTargetInf[LcNbPipe].tgCaracPipeVoie = LcOper;
            TbTargetInf[LcNbPipe].tgPipe = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
            TbTargetInf[LcNbPipe].tgAudio = TbTargetInf[LcNbPipe].tgPipe;

            LcNbPipe++;

        }
    }

	if(LcNbPipe)
	{
		LcRet = CUR_PROTOCOL_PTR->IEvent_PipeNotifyTime( TbTargetInf, LcNbPipe );
	}
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

}


// *************************************************************************
//
// STATIC VOID PurgeDCmdsFct( IN LPDWORD, OUT LPDWORD )
//
// Input Parameters:
// *****************
//
//  LPDWORD PmBlocRequete : Control command request bloc
//
// Output Parameters:
// *******************
//
//  LPDWORD PmBlocRetour  : Control command response bloc
//
// *************************************************************************
//
//  Process the purge differed cmds command.
//
// *************************************************************************
STATIC VOID PurgeDCmdsFct(
    IN  LPDWORD PmBlocRequete   ,
    OUT LPDWORD PmBlocRetour    )
{
    LPBC_HEADER_INFO       LcLPtrReqHeader;
    LPPURGE_DCMDS_REQ_INFO LcLPtrReq;
    LPRESP_HEADER_INFO     LcLPtrRepHeader;
    BYTE                   LcAppIndex;
    TARGET_INFO            LcTargetInf;
    BYTE                   LcCurrentBoard;
    DWORDLONG              LcPipeMask ;
    DWORD                  LcStreamMask ;
    WORD                   LcPipeIndex;
    WORD                   LcRet;
    DWORDLONG              LcPipeMaskOut, LcPipeMaskIn;
    WORD                   i;
    BYTE                   LcOper;
    PPIPE_INFO             LcPtPipe;

    // Initialisation of pointers
    LcLPtrReq   = (LPPURGE_DCMDS_REQ_INFO)PmBlocRequete;
    LcLPtrReqHeader = (LPBC_HEADER_INFO)PmBlocRequete;
    LcLPtrRepHeader = (LPRESP_HEADER_INFO)PmBlocRetour;

    // Retrieve application handle
    // --------------------------------
    LcAppIndex = DecodeHandle( LcLPtrReqHeader->hdHandle );

    // Retrieve pipe and stream masks
    // ------------------------------
    LcPipeMaskOut = LcLPtrReq->pdqOutPipeMask64.QuadPart;
    LcPipeMaskIn  = LcLPtrReq->pdqInPipeMask64.QuadPart;
    LcStreamMask  = LcLPtrReq->pdqParam1 ;

    // Check all the pipe are actually owned by the application
    // --------------------------------------------------------
    LcRet = ChkPmManyInOrOutPipeMask(   LcAppIndex,
                                        LcPipeMaskOut,
                                        LcPipeMaskIn );
    if ( LcRet != SUCCESS )
    {
        LcLPtrRepHeader->rhCptr = LcRet;
        return;
    }

    // Seek for output pipes first, then for input pipes
    // -------------------------------------------------
    for ( i = 0 ; i < 2 ; i ++ )
    {
        switch( i )
        {
        case 0 :
            LcPipeMask  = LcPipeMaskOut;
            LcPtPipe    = &TbPipeOutInfo[0];
            LcOper      = OPER_PLAY;
            break;

        case 1 :
            LcPipeMask  = LcPipeMaskIn;
            LcPtPipe    = &TbPipeInInfo[0];
            LcOper      = OPER_REC;
            break;
        }

        // For each pipe specified
        // ------------------------
        while ( LcPipeMask != 0 )
        {
            // Retrieve pipe index
            // -------------------
            LcPipeIndex = UTI64Mask2Word( LcPipeMask );
            LcPipeMask &= ~UTIMask64bit( LcPipeIndex );


            // Seek the corresponding pipe
            // ---------------------------
            // The LcPtVoie[LcPipeIndex].IndexAppli and LcPtVoie[LcPipeIndex].NumPipe are
            // already tested by ChkPmManyInOrOutPipeMask
            {
                // Fill TARGET_INFO structure
                LcTargetInf.tgCarte = LcCurrentBoard = LcPtPipe[LcPipeIndex].piNumBoard;
                LcTargetInf.tgCaracPipeVoie = LcOper;
                LcTargetInf.tgPipe = LcPtPipe[LcPipeIndex].piNumPipeForDsp;
                LcTargetInf.tgAudio = LcTargetInf.tgPipe;

                // Filter the effective streams
                // ------------------------------------------
                LcTargetInf.tgMaskFlux = (  LcStreamMask
                                          & LcPtPipe[LcPipeIndex].piStreamMask );

                LcTargetInf.tgVioHandle = LcPtPipe[LcPipeIndex].piVioHandle ;

                if ( LcStreamMask == 0 )
                {
                    LcRet = CUR_PROTOCOL_PTR->IEvent_PipePurgeDCmds( &LcTargetInf,
                                               LcLPtrReq->pdqDiffered,
                                               LcLPtrReq->pdqCommandCode,
                                               LcLPtrReq->pdqParam2 );
                }
                else
                {
                    LcRet = CUR_PROTOCOL_PTR->IEvent_StreamPurgeDCmds( &LcTargetInf,
                                                 LcLPtrReq->pdqDiffered,
                                                 LcLPtrReq->pdqCommandCode,
                                                 LcLPtrReq->pdqParam2 );
                }

                if ( LcRet != SUCCESS )
                {
                    LcLPtrRepHeader->rhCptr = LcRet;
                    return;
                }
            }

        } // for each pipe of the mask

    }
    // for output pipes then input pipes
}

