// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

#include "ESIOCommands.h"
#include "xx_protocol.h"
#include "if_drv_mb.h"
#include "LXES_registers.h"
//#include "eeprom.h"

#define CHIPSC_RESET_XILINX (1L<<16)    // PGO pour USERo dans le registre pci_0x06/loc_0xEC
#define CHIPSC_GPI_USERI    (1L<<17)    // PGI pour USERi

#define IRQCS_ACTIVE_PCIDB  0x00002000L         // Bit n 13
#define IRQCS_ENABLE_PCIIRQ 0x00000100L         // Bit n 08
#define IRQCS_ENABLE_PCIDB  0x00000200L         // Bit n 09

#define L2PCIDB_IRQ         0x00000001L         // Bit n 00

#define MBOX0_HF5           0x00000001L         // Bit n 00
#define MBOX0_HF4           0x00000002L         // Bit n 01
#define MBOX0_BOOT_HERE     0x00800000L         // Bit n 23


#define CSES_TIMEOUT        100     // microseconds
#define CSES_CE             0x0001
#define CSES_BROADCAST      0x0002
#define CSES_UPDATE_LDSV    0x0004

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CESIOCommands::CESIOCommands(IOPCIDevice* pIOPCIDevice, PKSPIN_LOCK pMsgSpinLock) : CPIOCommands(pIOPCIDevice)
{
    m_MsgLockPtr = pMsgSpinLock;

    m_bliBusType                  = PCIBUS;
    m_bliBoardTypeNoAlias         = (BYTE) m_dwBoardVersion;

    m_piMiscFeatures    = DIRECT_MEMORY_ACCESS_PRESENT_MASK;
    m_piType            = LXES_BOARD ;
    m_piStereoOutNumber = 32;
    m_piStereoInNumber  = 32;

	m_piSn = 0;
    m_piNbWcl = 0;
    m_piEtherSoundSync  = 1;

    m_ConfES = 0;
    m_FrequencyRatio = 1;   // 44.1 or 48 kHz

    PHYSICAL_ADDRESS LcPhysAdd;
    LcPhysAdd.QuadPart = (ULONG_PTR)pIOPCIDevice->AddressDSP;

    m_WindowSize = MIN(pIOPCIDevice->SizeDSP, 0x500 * sizeof(DWORD));

    LPDWORD LcRegisterBase = (LPDWORD)MmMapIoSpace(
            LcPhysAdd,
            m_WindowSize,
            MmNonCached);

    DOUT(DBG_PRINT, ("MmMapIoSpace DSP (size %x) %I64x -> %p\n", m_WindowSize, LcPhysAdd, LcRegisterBase));

    // Attention : the registers are incremented by 4 * RegisterOffset (as they are pointers to DWORD !)
    m_aLXES[eReg_BASE]      = LcRegisterBase;
    m_aLXES[eReg_CSM]       = LcRegisterBase + Reg_CSM;
    m_aLXES[eReg_CRM1]      = LcRegisterBase + Reg_CRM1;
    m_aLXES[eReg_CRM2]      = LcRegisterBase + Reg_CRM2;
    m_aLXES[eReg_CRM3]      = LcRegisterBase + Reg_CRM3;
    m_aLXES[eReg_CRM4]      = LcRegisterBase + Reg_CRM4;
    m_aLXES[eReg_CRM5]      = LcRegisterBase + Reg_CRM5;
    m_aLXES[eReg_CRM6]      = LcRegisterBase + Reg_CRM6;
    m_aLXES[eReg_CRM7]      = LcRegisterBase + Reg_CRM7;
    m_aLXES[eReg_CRM8]      = LcRegisterBase + Reg_CRM8;
    m_aLXES[eReg_CRM9]      = LcRegisterBase + Reg_CRM9;
    m_aLXES[eReg_CRM10]     = LcRegisterBase + Reg_CRM10;
    m_aLXES[eReg_CRM11]     = LcRegisterBase + Reg_CRM11;
    m_aLXES[eReg_CRM12]     = LcRegisterBase + Reg_CRM12;

    m_aLXES[eReg_ICR]       = LcRegisterBase + Reg_ICR;
    m_aLXES[eReg_CVR]       = LcRegisterBase + Reg_CVR;
    m_aLXES[eReg_ISR]       = LcRegisterBase + Reg_ISR;
    m_aLXES[eReg_RXHTXH]    = LcRegisterBase + Reg_RXHTXH;
    m_aLXES[eReg_RXMTXM]    = LcRegisterBase + Reg_RXMTXM;
    m_aLXES[eReg_RHLTXL]    = LcRegisterBase + Reg_RHLTXL;
    m_aLXES[eReg_RESETDSP]  = LcRegisterBase + Reg_RESETDSP;

    m_aLXES[eReg_CSUF]      = LcRegisterBase + Reg_CSUF;
    m_aLXES[eReg_CSES]      = LcRegisterBase + Reg_CSES;
    m_aLXES[eReg_CRESMSB]   = LcRegisterBase + Reg_CRESMSB;
    m_aLXES[eReg_CRESLSB]   = LcRegisterBase + Reg_CRESLSB;
    m_aLXES[eReg_ADMACESMSB]= LcRegisterBase + Reg_ADMACESMSB;
    m_aLXES[eReg_ADMACESLSB]= LcRegisterBase + Reg_ADMACESLSB;
    m_aLXES[eReg_CONFES]    = LcRegisterBase + Reg_CONFES;
}


CESIOCommands::~CESIOCommands()
{
    if(m_aLXES[eReg_BASE])
    {
        MmUnmapIoSpace(m_aLXES[eReg_BASE], m_WindowSize);
    }
}

/*
    called in InitDsp, after DspLoad and ResetBoard
*/
BOOL  CESIOCommands::PIOTestConfig( INOUT PWORD PmDspType )
{
    if(PmDspType) *PmDspType = 0xfff;

    // TEST if we have access to Xilinx/MicroBlaze
    //
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CSM], 0);

    ULONG test = READ_REGISTER_ULONG(m_aLXES[eReg_CSM]);
    if(test != 0)
    {
        DOUT(DBG_ERROR, ("Problem : INIT ESIOCommands Reg_CSM = %x\n", test));

        UnivOUTPL(ePLX_PCICR, 1);   // PCI9056_SPACE0_REMAP

        test = READ_REGISTER_ULONG(m_aLXES[eReg_CSM]);
        if(test != 0)
        {
            DOUT(DBG_ERROR, ("ERROR : INIT ESIOCommands Reg_CSM = %x\n", test));
            return FALSE;
        }
    }

    return( TRUE );
} // TestConfig


WORD CESIOCommands::PIOUpdateESConfig( DWORD PmConfES )
{
	// read boot ES Config

    ULONG LcConfES = READ_REGISTER_ULONG(m_aLXES[eReg_CONFES]);
	ULONG LcESVer  = (LcConfES >> INTERPRETER_VERS_OFFSET) & INTERPRETER_VERS_MASK ;

	LOGFileTrace("-> ES (read) kernel type : %s\n", ( (LcESVer == INTERPRETER_V2) ? "legacy EtherSound, or early ES100 prototype." : "ES100.")) ;
	LOGFileTrace("-> ES (read) sample rate : %d\n",	  (LcConfES >> FREQ_RATIO_OFFSET ) & 0x0F );

    m_FrequencyRatio = ( PmConfES >> FREQ_RATIO_OFFSET ) & 0x0F ;

    if( (m_FrequencyRatio != FREQ_RATIO_SINGLE_MODE) &&
        (m_FrequencyRatio != FREQ_RATIO_DUAL_MODE) &&
        (m_FrequencyRatio != FREQ_RATIO_QUAD_MODE) )
    {
        m_FrequencyRatio = FREQ_RATIO_SINGLE_MODE;
    }

    DOUT(DBG_PRINT, ("PIOUpdateESConfig : ConfES Frequency ratio = %d\n", m_FrequencyRatio));

    DWORD LcRegistryInputNumber  = (PmConfES >> IOCR_INPUTS_OFFSET) & 0xFF;
    DWORD LcRegistryOutputNumber = (PmConfES >> IOCR_OUTPUTS_OFFSET) & 0xFF;

    DOUT(DBG_PRINT, ("PIOUpdateESConfig : ConfES remove IN  = %d\n", LcRegistryInputNumber));
    DOUT(DBG_PRINT, ("PIOUpdateESConfig : ConfES remove OUT = %d\n", LcRegistryOutputNumber));

    // merge de la partie registry avec la partie lue sur le xilinx. 
	//
	PmConfES = (LcConfES & CONFES_READ_PART_MASK) | ( PmConfES & CONFES_WRITE_PART_MASK);

    // update IO number of IOCR with frequency ratio

    WORD LcConfES_In  = (WORD)(m_piStereoInNumber * 2 * m_FrequencyRatio);
    WORD LcConfES_Out = (WORD)(m_piStereoOutNumber * 2 * m_FrequencyRatio);

    if(LcConfES_In > MAX_CARD_INPUTS)
    {
        LcConfES_In = MAX_CARD_INPUTS;
        m_piStereoInNumber = (WORD)(MAX_CARD_INPUTS / (m_FrequencyRatio * 2));
    }
    if(LcConfES_Out > MAX_CARD_OUTPUTS)
    {
        LcConfES_Out = MAX_CARD_OUTPUTS;
        m_piStereoOutNumber = (WORD)(MAX_CARD_OUTPUTS / (m_FrequencyRatio * 2));
    }

    DOUT(DBG_PRINT, ("PIOUpdateESConfig : ConfES add IN  = %d\n", LcConfES_In));
    DOUT(DBG_PRINT, ("PIOUpdateESConfig : ConfES add OUT = %d\n", LcConfES_Out));
    DOUT(DBG_PRINT, ("PIOUpdateESConfig : m_piStereoInNumber  = %d\n", m_piStereoInNumber));
    DOUT(DBG_PRINT, ("PIOUpdateESConfig : m_piStereoOutNumber = %d\n", m_piStereoOutNumber));

    m_ConfES =  PmConfES |
                (LcConfES_In	<< IOCR_INPUTS_OFFSET) | 
                (LcConfES_Out	<< IOCR_OUTPUTS_OFFSET);

    // write it to the card !
    // this actually kicks the ES xilinx, the first time since poweron.
	// the MAC address in the Reg_ADMACESMSB Reg_ADMACESLSB registers 
	// is not ready before this is done, and the bit 2 in Reg_CSES is set.
	//
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CONFES], m_ConfES);

    for (int i = 0; i < m_piStereoOutNumber; i++)
    {
        m_piDescAudioOut[i] = AUDIO_MORE_FEATURES_MASK;
        m_bPhysOutFeature2[i] = AUDIO_ETHERSOUND_PRESENT_MASK;
    }
    for (int i = 0; i < m_piStereoInNumber; i++)
    {
        m_piDescAudioIn[i] = AUDIO_MORE_FEATURES_MASK;
        m_bPhysInFeature2[i] = AUDIO_ETHERSOUND_PRESENT_MASK;
    }

//	DbgPrint( "m_ConfES = %lx\n", m_ConfES);


    return SUCCESS;
}//

/**
*		lecture de l'adresse MAC depuis le xilinx ES
*/
WORD  CESIOCommands::PIOGetMACAddress( PBYTE PmMACAddressPtr /* array of 6 bytes */)
{
    BYTE i;
    BYTE LcRead;
	WORD LcRet = SUCCESS;

	ASSERT(PmMACAddressPtr);   

    // test if load/save operation recently occured
	//
	DWORD l_reg = READ_REGISTER_ULONG(m_aLXES[eReg_CSES]);

	// or cash not yet initialized
	// 
	if (	( l_reg & CSES_UPDATE_LDSV ) 
		 || ( (m_piMACAddressMSB + m_piMACAddressLSB) == 0) 
		)
	{
		m_piMACAddressMSB = READ_REGISTER_ULONG(m_aLXES[eReg_ADMACESMSB]) & 0x00FFFFFF;
		m_piMACAddressLSB = READ_REGISTER_ULONG(m_aLXES[eReg_ADMACESLSB]) & 0x00FFFFFF;
	}

	PmMACAddressPtr[5] = ((PCHAR)(&m_piMACAddressLSB))[0];	
	PmMACAddressPtr[4] = ((PCHAR)(&m_piMACAddressLSB))[1];	
	PmMACAddressPtr[3] = ((PCHAR)(&m_piMACAddressLSB))[2];	
	PmMACAddressPtr[2] = ((PCHAR)(&m_piMACAddressMSB))[0];
	PmMACAddressPtr[1] = ((PCHAR)(&m_piMACAddressMSB))[1];
	PmMACAddressPtr[0] = ((PCHAR)(&m_piMACAddressMSB))[2];

	return LcRet;

}; //PIOGetMACAddress 


VOID  CESIOCommands::PIOGetESConfig( PDWORD PmConfES )
{
    if(PmConfES) *PmConfES = m_ConfES;
}

BOOL  CESIOCommands::HALWaitForBitValue_Timeout_DWordReg(
    IN PCXPORT           Pm32bitPort,
    IN DWORD             PmBitToTest,
    IN BOOLEAN           PmValue,
    IN WORD              PmTimeout)
{
    DWORD           LcReg       ;
    DWORD           LcTestRet   ;
    DWORD           LcTimeout   ;
    DWORD           LcElapsed   = 0L        ;
    DWORD           LcRetry     = 0L        ;

    LcTestRet = ( PmValue ) ? PmBitToTest : (DWORD) 0 ;

    LcTimeout = (DWORD) PmTimeout ;
    LcTimeout *= 55000L         ;   // In microsecond

    do
    {
        PCX_WAIT_AT_LEAST_MICRO_SEC(LcRetry)     ;
        LcReg = UnivINPL(Pm32bitPort) ;       // DWORD operation

        LcElapsed += LcRetry ;

        // ## FM (09/06/97) -- On attend de plus en plus longtemps pour ne pas
        //                     trop saturer l'interface PCI, mais on ne dpasse
        //                     pas 50 us pour ne pas pnaliser les requtes un
        //                     peu lente (StopPipe par exemple). En effet, si
        //                     une requte mais N ms pour passer HF3  1, et que
        //                     l'on fait le test  N-delta, alors il faudra
        //                     attendre N ms de plus pour voir HF3  1.
        //

        if ( LcRetry == 0L ) LcRetry = 1L;
        else if ( LcRetry < 50 ) LcRetry *= 2L  ;
    } while( ( (LcReg & PmBitToTest) != LcTestRet) && ( LcElapsed < LcTimeout ) );

    return ( (LcElapsed < LcTimeout) ? TRUE : FALSE );
} // HALWaitForBitValue_Timeout_DWordReg


#define IRQCS_ACTIVE_PCIDB  0x00002000L         // Bit n 13
#define IRQCS_ENABLE_PCIIRQ 0x00000100L         // Bit n 08
#define IRQCS_ENABLE_PCIDB  0x00000200L         // Bit n 09

VOID CESIOCommands::PIODisableIrq( )
{
    DWORD   LcDataReg = UnivINPL( ePLX_IRQCS);

    // Set the Doorbell and PCI interrupt enable bits to 0.
    //
    LcDataReg &= ~(IRQCS_ENABLE_PCIIRQ | IRQCS_ENABLE_PCIDB) ;

    UnivOUTPL( ePLX_IRQCS,LcDataReg );
}

VOID CESIOCommands::PIOEnableIrq( )
{
    DWORD   LcDataReg;
	
	UnivOUTPL(ePLX_L2PCIDB, 0xFFFFFFFF) ;

	LcDataReg = UnivINPL( ePLX_IRQCS);

    // Set the Doorbell and PCI interrupt enable bits to 1.
    //
    LcDataReg |= (IRQCS_ENABLE_PCIIRQ | IRQCS_ENABLE_PCIDB) ;

    UnivOUTPL( ePLX_IRQCS,LcDataReg );
}


#define IRQCS_ACTIVE_PCIDB  0x00002000L         // Bit n 13
DWORD CESIOCommands::PIOTestAndAckIt()
{
    // Read IRQCS
    DWORD   LcDataReg   = UnivINPL(ePLX_IRQCS);

    // Test if PCI Doorbell interrupt is active
    //
    if ( LcDataReg & IRQCS_ACTIVE_PCIDB )
    {
		LcDataReg = PCX_IRQ_NONE;
        DWORD temp;
        while (temp = UnivINPL(ePLX_L2PCIDB))
        {
            // RAZ interrupt
            LcDataReg |= temp;
            UnivOUTPL(ePLX_L2PCIDB,temp) ;
        }
        return LcDataReg;
    }
    return PCX_IRQ_NONE;
}


WORD CESIOCommands::PIOSetClock( DWORD PmFreq, WORD PmClockType, DWORD PmSyncInputNum, BYTE PmClockFormat )
{
    if( (PmSyncInputNum != 0 ) ||
        (PmClockType != CLOCK_TYPE_ETHERSOUND) )
    {
        EXIT_PCX_ERROR(ED_SET_CLOCK_IMPOSSIBLE);
    }

    // m_piActualClkFrequency updated externally with PIOSetActualClockFrequency()
    m_piActualClkFrequency = 0;
    m_piActualClkSource = PmClockType;
	m_piActualClkInputNum = PmSyncInputNum;

    return SUCCESS;
}


DWORD CESIOCommands::HALDefaultConfig( )
{
    HALXilinxReset(TRUE);

    // program clock
    PIOSetClock( 0, CLOCK_TYPE_ETHERSOUND, 0, 0);

    return SUCCESS;
}

BOOL CESIOCommands::HALXilinxReset( BOOL PmFirstXilinx )
{
    DWORD	LcPLXRegister;

    LcPLXRegister = UnivINPL(ePLX_CHIPSC) ;  

    // ACTIVER LE RESET DU XILINX pendant 2ms
    //
    LcPLXRegister &= ~CHIPSC_RESET_XILINX;
    UnivOUTPL( ePLX_CHIPSC,LcPLXRegister);

    PCX_WAIT_AT_LEAST_MILLI_SEC(1);

    UnivOUTPL(ePLX_MBOX3, 0);

    PCX_WAIT_AT_LEAST_MILLI_SEC(1);

    LcPLXRegister |= CHIPSC_RESET_XILINX;
    UnivOUTPL( ePLX_CHIPSC,LcPLXRegister);

    // DESACTIVER LE RESET DU XILINX pendant 1 s
    //
    LARGE_INTEGER LcPerfFreq;
    LARGE_INTEGER LcTime1 = KeQueryPerformanceCounter(&LcPerfFreq);

    int iLoop;
    for( iLoop=0; iLoop<100; iLoop++)
    {
        PCX_WAIT_AT_LEAST_MILLI_SEC(10);

        DWORD dwMBox3 = UnivINPL(ePLX_MBOX3);
        if( dwMBox3 != 0 )
        {
            break;
        }
    }

	WRITE_REGISTER_ULONG(m_aLXES[eReg_CSM], 0) ; // clear MR 

    LARGE_INTEGER LcTime2 = KeQueryPerformanceCounter(NULL);
    LONGLONG LcTime = LcTime2.QuadPart - LcTime1.QuadPart;
    if(LcTime)
    {
        LcTime *= 1000000;
        LcTime /= LcPerfFreq.QuadPart;
        LOGFileTrace("RESET XILINX : time = %I64d us (%d loops)\n", LcTime, iLoop);
    }

	// le xilinx ES peut ne pas etre encore pret, on attend...
	//
	PCX_WAIT_AT_LEAST_MILLI_SEC(600);

	return TRUE;
} // HALXilinxReset


DWORD CESIOCommands::PIOReadPLXMailBox( BYTE PmMailBoxNumber)
{
    DWORD LcRet;
    PCXPORT LcPort;

#ifdef DBG_VIEWER
    LARGE_INTEGER LcPerfFreq;
    LARGE_INTEGER LcTime1 = KeQueryPerformanceCounter(&LcPerfFreq);
#endif

    switch(PmMailBoxNumber)
    {
    case 0 :
         ASSERT(0); // reserved for HF flags 
         return 0xdeadbeef;
    case 1 :
        LcPort = ePLX_MBOX1;    break;
    case 2 :
        LcPort = ePLX_MBOX2;    break;
    case 3 :
        LcPort = ePLX_MBOX3;    break;
    case 4 :
        LcPort = ePLX_MBOX4;    break;
    case 5 :
        LcPort = ePLX_MBOX5;    break;
    case 6 :
        LcPort = ePLX_MBOX6;    break;
    case 7 :
        LcPort = ePLX_MBOX7;    break;
    default:
        return 0xdeadbeef;
    } // sw

    LcRet = UnivINPL(LcPort) ;

#ifdef DBG_VIEWER
    LARGE_INTEGER LcTime2 = KeQueryPerformanceCounter(NULL);
    LONGLONG LcTime = LcTime2.QuadPart - LcTime1.QuadPart;
    if(LcTime)
    {
        LcTime *= 1000000;
        LcTime /= LcPerfFreq.QuadPart;
        DOUT(DBG_TEST, ("PIOReadPLXMailBox( %d ) = %I64d us\n", PmMailBoxNumber, LcTime));
    }
#endif

    return LcRet;

} // PIOReadPLXMailBox


BOOL CESIOCommands::PIOWritePLXMailBox( BYTE PmMailBoxNumber, DWORD PmData)
{
    PCXPORT LcPort;
    
#ifdef DBG_VIEWER
    LARGE_INTEGER LcPerfFreq;
    LARGE_INTEGER LcTime1 = KeQueryPerformanceCounter(&LcPerfFreq);
#endif
    
    switch(PmMailBoxNumber)
    {
    case 0 :
         ASSERT(0); // reserved for HF flags 
         return FALSE;
    case 1 :
        LcPort = ePLX_MBOX1;    break;
    case 2 :
      //  LcPort = ePLX_MBOX2;    break;
         ASSERT(0); // reserved for Pipe States : the DSP keeps an image of it 
         return FALSE;
    case 3 :
        LcPort = ePLX_MBOX3;    break;
    case 4 :
        LcPort = ePLX_MBOX4;    break;
    case 5 :
        LcPort = ePLX_MBOX5;    break;
    case 6 :
        LcPort = ePLX_MBOX6;    break;
    case 7 :
        LcPort = ePLX_MBOX7;    break;
    default:
        ASSERT(0); 
        return 0xdeadbeef;
    } // sw

    UnivOUTPL(LcPort, PmData) ;

#ifdef DBG_VIEWER
    LARGE_INTEGER LcTime2 = KeQueryPerformanceCounter(NULL);
    LONGLONG LcTime = LcTime2.QuadPart - LcTime1.QuadPart;
    if(LcTime)
    {
        LcTime *= 1000000;
        LcTime /= LcPerfFreq.QuadPart;
        DOUT(DBG_TEST, ("PIOWritePLXMailBox( %d, %lx ) = %I64d us\n", PmMailBoxNumber, PmData, LcTime ));
    }
#endif
    
    return TRUE;

} // PIOWritePLXMailBox


// sleep 500 - 100 = 400 times 100us -> the timeout is >= 40 ms
//
#define XILINX_TIMEOUT_MICROSEC 40000
#define XILINX_POLL_NO_SLEEP    100
#define XILINX_POLL_ITERATIONS  500

WORD CESIOCommands::PIOSendMessage(  INOUT PCMD_RMH_INFO     PmPtrCmdRmh)
{
    WORD LcRet = ED_DSP_TIMED_OUT;

    if( ! m_EnableSendMessage )
    {
        DOUT(DBG_ERROR, ("ERROR PIOSendMessage(%d) not enabled\n", PmPtrCmdRmh->Index));
        return ED_DSP_BUSY;
    }
    // test Reg_CMS
    if( (READ_REGISTER_ULONG(m_aLXES[eReg_CSM]) & (Reg_CSM_MC | Reg_CSM_MR)) != 0 )
    {
        DOUT(DBG_ERROR, ("ERROR PIOSendMessage eReg_CSM = %x\n", *m_aLXES[eReg_CSM]));
        return ED_DSP_BUSY;
    }

    ASSERT(PmPtrCmdRmh->LgCmd != 0);
    ASSERT(PmPtrCmdRmh->LgCmd < SIZE_MAX_CMD);

    // write command
    WRITE_REGISTER_BUFFER_ULONG(m_aLXES[eReg_CRM1], PmPtrCmdRmh->Cmd, PmPtrCmdRmh->LgCmd);

    ASSERT(m_SendMessageLocked == 0);
    m_SendMessageLocked = 1;

    // MicoBlaze gogogo
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CSM], Reg_CSM_MC);

    DWORD dwLoop = 0;   // timeout 40 ms
    while ( (dwLoop < XILINX_TIMEOUT_MICROSEC) && (m_SendMessageLocked != 0) )
    {
        PCX_WAIT_AT_LEAST_MICRO_SEC(1);
        dwLoop++;
    }
    if(m_SendMessageLocked == 0)
    {
        // in Debug mode verify Reg_CSM_MR
       // ASSERT(READ_REGISTER_ULONG(m_aLXES[eReg_CSM]) & Reg_CSM_MR);

        // command finished, read status
        if (PmPtrCmdRmh->DspStat == 0)	// this command may return an error code
        {
            LcRet = (WORD) READ_REGISTER_ULONG(m_aLXES[eReg_CRM1]);
        }
        else
        {
            LcRet = SUCCESS;
        }
    }
    else
    {
        DOUT(DBG_WARNING, ("TIMEOUT PIOSendMessage ! Interrupts disabled ?\n"));
        m_SendMessageLocked = 0;

        // attente bit Reg_CSM_MR
        for( int i = 0; i < XILINX_POLL_ITERATIONS; i++ )
        {
            if(READ_REGISTER_ULONG(m_aLXES[eReg_CSM]) & Reg_CSM_MR)
            {
                // command finished, read status
                if (PmPtrCmdRmh->DspStat == 0)	// this command may return an error code
                {
                    LcRet = (WORD) READ_REGISTER_ULONG(m_aLXES[eReg_CRM1]);
	            }
                else
                {
                    LcRet = SUCCESS;
                }
                break;
            }
            if( i > XILINX_POLL_NO_SLEEP )
            {
                PCX_WAIT_AT_LEAST_MICRO_SEC(100);
            }
        }
    }
    if( (LcRet & ERROR_VALUE) == 0 )
    {
        // read response
        if(PmPtrCmdRmh->LgStat)
        {
            ASSERT( PmPtrCmdRmh->LgStat < (SIZE_MAX_STATUS-1) );

            READ_REGISTER_BUFFER_ULONG(m_aLXES[eReg_CRM2], PmPtrCmdRmh->Stat, PmPtrCmdRmh->LgStat);
        }
    }

    // clear Reg_CSM_MR
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CSM], 0);

    return LcRet;
}


WORD CESIOCommands::HALSetClockDependency( IN BYTE PmClockType )
{
    switch ( PmClockType ) {
    case INDEPENDANT_CLOCK :
        break;

    case MASTER_CLOCK:
    case SLAVE_CLOCK:
    default:
        DOUT(DBG_ERROR, ("ERROR HALSetClockDependency\n"));
        break ;
    }

    return SUCCESS;
} // HALSetClockDependency


BOOL CESIOCommands::PIOCheckESPipeline()
{
    // wait for Xilinx
    for (int i=0; i<CSES_TIMEOUT; i++)
    {
        DWORD reg = READ_REGISTER_ULONG(m_aLXES[eReg_CSES]);

		// le bit CSES_UPDATE_LDSV est  1 ds que le macprog est pret.
		// il re-passe  0 lorsque le premier read a t fait. 
		// pour l'instant on retire le test car ce bit passe a 1 environ 200  400 ms 
		// aprs que le registre confES  t crit (kick du xilinx ES).
		//         
		// On ne teste que le bit CE.
		// 
		if( (reg & CSES_CE) == 0)
        {
            return TRUE;
        }

        PCX_WAIT_AT_LEAST_MICRO_SEC(1);
    }
    return FALSE;   // timeout
}

BOOL CESIOCommands::PIOWriteCRES(DWORD dwCresMsb, DWORD dwCresLsb, BOOL bBroadcast )
{
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CRESMSB], dwCresMsb);
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CRESLSB], dwCresLsb);

	//DbgPrint("ES write cmd MSB = %lx", dwCresMsb);
	//DbgPrint("ES write cmd LSB = %lx", dwCresLsb);

    ULONG cses = CSES_CE;
    if(bBroadcast) cses |= CSES_BROADCAST;

    // go xilinx go
    //
    WRITE_REGISTER_ULONG(m_aLXES[eReg_CSES], cses);

	//DbgPrint("ES write CSES = %lx", cses);

    return TRUE;
}

BOOL CESIOCommands::PIOReadCRES(PDWORD addrCresMsb, DWORD nbDword )
{
    ASSERT(addrCresMsb != NULL);
    ASSERT(nbDword == 4);

	READ_REGISTER_BUFFER_ULONG(m_aLXES[eReg_CRESMSB], addrCresMsb, nbDword);

/*	DbgPrint ("ES PIOReadCRES resp MSB = %lx\n", *addrCresMsb); 
	DbgPrint ("ES PIOReadCRES resp LSB = %lx\n", *(addrCresMsb+1)); 
	DbgPrint ("ES PIOReadCRES mac MSB = %lx\n", *(addrCresMsb+2)); 
	DbgPrint ("ES PIOReadCRES mac LSB = %lx\n", *(addrCresMsb+3));*/ 


    return TRUE;
}
