// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// Include files
// *************
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <malloc.h>
#include <process.h>
#include <string.h>
#include <math.h>
#include <crtdbg.h> /*for ASSERT*/

#include "apidef.h"


#include "ifapidrv.h"  // interface between API and driver

#include "util.h"               // utilities making it easier to
                                // manipulate masks

#include "pcxapi_e.h"    // prototypes of functions

#include "common.h"             // common objects shared between all
                                // API family functions

// ********************************
// DIAGNOSTIC FUNCTIONS PROTOTYPES
// ********************************

// API Internals functions
// -----------------------
#include "genasync.h"

// Local defines
// **************

#ifdef _DEBUG
#define NPNTAPITRACE(x)     OutputDebugString(_T(x))
#else
#define NPNTAPITRACE(x)
#endif //_DEBUG

// get the flavor number of a board type from the CSID
//
#define EXTRACT_FLAVOR_NUM(csid)    ((csid) & 0x000000ff)
// Local typedefs
// ***************

// handle type used by an application to hold all request/reply
// command blocks used in asynchronous commands
//
typedef struct _APPLI_CMD_BLK_INFO
{
    PCX_HANDLE              acbAppHandle;       // Application handle

    ANY_ASYNC_REQUEST_INFO  acbWER;             // One request/reply block ..
    ANY_ASYNC_REPLY_INFO    acbWEP;             // .. to wait for errors

#define acbWaitErrorRequest     acbWER.aarWaitError
#define acbWaitErrorReply       acbWEP.aapWaitError

    // Actually supported only on Windows95, but defined for any Win32
    //
    ANY_ASYNC_REQUEST_INFO  acbWPR;             // One request/reply block ..
    ANY_ASYNC_REPLY_INFO    acbWPP;             // .. to wait for BoardPnp events

#define acbWaitBoardRequest     acbWPR.aarWaitBoardPnp
#define acbWaitBoardReply       acbWPP.aapWaitBoardPnp

    ANY_ASYNC_REQUEST_INFO  acbWGR;             // One request/reply block ..
    ANY_ASYNC_REPLY_INFO    acbWGP;             // .. to wait for GPIO events

#define acbWaitGpioRequest      acbWGR.aarWaitGpio
#define acbWaitGpioReply        acbWGP.aapWaitGpio

#ifdef WIN32
    OVERLAPPED              acbWaitErrorOverlapped;
    OVERLAPPED              acbWaitBoardPnpOverlapped;
    OVERLAPPED              acbWaitGpioOverlapped;
#endif

} APPLI_CMD_BLK_INFO, *PAPPLI_CMD_BLK_INFO, FAR *LPAPPLI_CMD_BLK_INFO;

// handle type used by an application to hold all request/reply
// command blocks used in in asynchronous commands on pipes
//
typedef struct _WAIT_CMD_BLK_INFO {

    PCX_HANDLE                  wcbAppHandle;       // Application handle
    DWORD                       wcbOutPipeMask;     // the pipe mask
    DWORD                       wcbInPipeMask;      // the pipe mask
    WORD                        wcbNbBuff;          // How many buffers are granted to the pipe
    DWORD                       wcbMaxBuffSize;     // Max buffer size allowed = min buff size of all allocated buffers

    DWORD                       wcbNbStreams;       // How many streams managed by the pipe

    LPANY_ASYNC_REQUEST_INFO    wcbWNR;             // One request/reply block ..
    LPANY_ASYNC_REPLY_INFO      wcbWNP;             // .. to wait for notification

#define wcbWaitNotifyRequest    wcbWNR->aarWaitNotification
#define wcbWaitNotifyReply      wcbWNP->aapWaitNotification

#ifdef WIN32
    OVERLAPPED                  wcbWaitNotifyOverlapped;
#endif

    LPANY_ASYNC_REQUEST_INFO    wcbAAR;             // One request/reply block ..
    LPANY_ASYNC_REPLY_INFO      wcbAAP;             // .. per allocated buffer

#define wcbRequestTab           wcbAAR
#define wcbReplyTab             wcbAAP

#ifdef WIN32
    LPOVERLAPPED                wcbOverlappedTab;   // Overlapped structures
#endif

    LPANY_ASYNC_REQUEST_INFO    wcbWEPR;            // One request/reply block ..
    LPANY_ASYNC_REPLY_INFO      wcbWEPP;            // .. per allocated buffer
    LPANY_ASYNC_REQUEST_INFO    wcbWERR;            // One request/reply block ..
    LPANY_ASYNC_REPLY_INFO      wcbWERP;            // .. per allocated buffer

#define wcbWaitEndPlayRequest   wcbWEPR->aarWaitEndOfPlay
#define wcbWaitEndPlayReply     wcbWEPP->aapWaitEndOfPlay
#define wcbWaitEndRecordRequest wcbWERR->aarWaitEndOfRecord
#define wcbWaitEndRecordReply   wcbWERP->aapWaitEndOfRecord

#ifdef WIN32
    OVERLAPPED                  wcbWaitEndPlayOverlapped;
    OVERLAPPED                  wcbWaitEndRecordOverlapped;
#endif

} WAIT_CMD_BLK_INFO, *PWAIT_CMD_BLK_INFO, FAR *LPWAIT_CMD_BLK_INFO;

typedef struct _BUFFER_TO_LOAD_INFO
{
    WORD                btlBufferNumber;
    WORD                btlUnused1;
    DWORD               btlDataLength;
  // ## FS (26/07/00) -- Room needed to compile for Wave16
  // DWORD               btlUnused2;
} BUFFER_TO_LOAD_INFO,* PBUFFER_TO_LOAD_INFO;

typedef struct _DSP_FILE_INFO {
    BYTE                dfBoardType;
    BYTE                dfExtendedBoardType;
    BYTE                dfSoftwareIndex;
    BYTE                dfFirmwareIndex; // Only if differs from SoftwareIndex
} DSP_FILE_INFO, *PDSP_FILE_INFO;

typedef struct _DSP_CAPABILITY_INFO
{
    LPTSTR              dciFileName;
    DWORD               dciFeatures;
    DWORD               dciPlayFormats;
    DWORD               dciRecordFormats;
    DWORD               dciPlayEffects;
    DWORD               dciRecordEffects;
    BOOLEAN             dciCheckForHardware;
  // ## FS (26/07/00) -- Room needed to compile for Wave16
  //    DWORD               dciUnused2;
  //    DWORD               dciUnused3;
} DSP_CAPABILITY_INFO, *PDSP_CAPABILITY_INFO;

typedef struct _ASSOC_BOARD_FLAVOR_INFO {
    WORD            abBoardType;
    WORD            abExtendedBoardType;
    WORD            abFlavorNum;    // ## FM (08/27/98) -- Add support for NP boards with PLX chip
    LPTSTR          abFlavorName;
} ASSOC_BOARD_FLAVOR_INFO, *PASSOC_BOARD_FLAVOR_INFO;

// Local variables
// ****************

#if !defined(WIN32)
#  pragma data_seg("SHAREDDATA")
#endif

  STATIC WAIT_CMD_BLK_INFO    WaitCmds[MAX_OUTPIPE+MAX_INPIPE];

  STATIC APPLI_CMD_BLK_INFO   WaitErrorCmds[MAX_PCX_HANDLE];

  // WAIT_CMD_BLK_INFO exceeds 64 kB, create seperate arrays (problem with 16 bit DLL)
  //
  // Requests for wcbWaitNotifyRequest, wcbWaitEndPlayRequest and wcbWaitEndRecord
  // Reply for wcbWaitNotifyReply, wcbWaitEndPlayReply and wcbWaitEndReply
  //
  STATIC  ANY_ASYNC_REPLY_INFO      WaitCmdsNofifResp[MAX_OUTPIPE+MAX_INPIPE];
  STATIC  ANY_ASYNC_REPLY_INFO      WaitCmdsPlayEndResp[MAX_OUTPIPE+MAX_INPIPE];
  STATIC  ANY_ASYNC_REPLY_INFO      WaitCmdsRecEndResp[MAX_OUTPIPE+MAX_INPIPE];

#if !defined(WIN32)
#  pragma data_seg()
#endif

// Reference list of board type flavors and their official
// names. No more line connections listed : all connections are balanced !!!
// 1st flavor is described by the 1st entry matching the
// board type, 2nd flavor is described by the 2nd entry
// matching the board type, and so on...
// -------------------------------------------------------
STATIC ASSOC_BOARD_FLAVOR_INFO  AssocBoardFlavor[] =
{
    { LXES,     LX6464ES,       1,      _T("LX6464ES") },
    { LXES,     LX6464ES,       2,      _T("LX6464ES-CAE") },
};
#define MAX_BOARD_FLAVOR_ENTRIES   ( sizeof( AssocBoardFlavor ) / sizeof( AssocBoardFlavor[0] ) )

// Reference list of DSP softwares and their capabilities
// ------------------------------------------------------
STATIC DSP_CAPABILITY_INFO  DspCapabilities[] =
{
    // SoftwareIndex: 0
    { _T(""),                   // Reserved (useful to get only path name)
      0,                                // Features
      0,                                // Play formats
      0,                                // Record formats
      0,                                // Play effects
      0,                                // Record effects
      FALSE,                            // Faking frequency
    },
    // SoftwareIndex: 1 -- LX6464ES
    { _T("LX6464ES"),        // Software file name
      DSP_CAN_PLAY_MASK                 // Features
          | DSP_CAN_RECORD_MASK,
      DSP_CAN_PR_LIN_16_MASK            // Play formats
          | DSP_CAN_PR_LIN_24_MASK,
      DSP_CAN_PR_LIN_16_MASK            // Record formats
          | DSP_CAN_PR_LIN_24_MASK,
      0,                                // Play effects
      0,                                // Record effects
      FALSE,                            // Faking frequency
    }
};
#define DSP_CAPABILITIES_INFO_ENTRIES   ( sizeof( DspCapabilities ) / sizeof( DspCapabilities[0] ) )

STATIC DSP_FILE_INFO DspFiles[] =
{
//    BoardType      BoardVersion SoftwareIndex FirmwareIndex
    { LXES,          LX6464ES,    1,            1   },
};
#define DSP_FILE_INFO_ENTRIES ( sizeof( DspFiles ) / sizeof( DspFiles[0] ) )

typedef struct _BOARD_LEVEL_INFO
{
    BYTE               blBoardType;            // ie: PCX80...
    BYTE               blExtendedBoardType;    // ie: LX6464ES
    BYTE               blFlavorNum;
    LEVEL_FEATURE_INFO blLevelFeatures;

} BOARD_LEVEL_INFO, FAR *LPBOARD_LEVEL_INFO;

// Default value for LEVEL_FEATURE_INFO.blFlavorNum
//
#define ALL_FLAVOR                  ((BYTE)-1)

// Value of LEVEL_FEATURE_INFO.lfReserved
//
#define NULL_LEVEL_FEATURE_RESERVED { 0, 0 }

// Value used to set a board -in the TabLevelFeature array- who have
// no line input/micro input/line output audio features. All the boards
// managed by the driver must have at least one entry in the
// TabLevelFeature array, otherwise the function PCXGetBoardLevelsFeatures
// returns an error.
//
#define NULL_LEVEL_FEATURE_CARAC                                  \
        {                                                         \
            FALSE,                                                \
            LEVEL_FEATURE_INPUT_LINE_MASK,                        \
            (FLOAT)    0.0,                                       \
            (FLOAT)    0.0,                                       \
            (FLOAT)    0.0,                                       \
            (FLOAT)    0.0,                                       \
            NULL_LEVEL_FEATURE_RESERVED                           \
        }

STATIC BOARD_LEVEL_INFO TabLevelFeature[] =
{
    {
        LXES,
        LX6464ES,   // ExtendedBoardType
        ALL_FLAVOR,
        NULL_LEVEL_FEATURE_CARAC
    },
};

#define BOARD_LEVEL_INFO_ENTRIES ( sizeof( TabLevelFeature ) / sizeof( TabLevelFeature[0] ) )

// Default level (in dB).
//
#define MPEG_EQUAL_DEFAULT_LEVEL            (0.0f)

// Constants used to translate a level into a coeff:
// coeff = a * b^(level/c), level in dB
//
#define MPEG_EQUAL_A                        (1 << 20)
#define MPEG_EQUAL_B                        10
#define MPEG_EQUAL_C                        20

// Minimum and maximum values for a coeff.
//
#define MPEG_EQUAL_COEFF_MIN_VALUE          0x00000000
#define MPEG_EQUAL_COEFF_MAX_VALUE          0x00ffffff

// For the user, the spectrum is devided into MPEG_EQUAL_MAX_BANDS bands.
// Each user band is described by a MPEG_EQUAL_INFO record.
//
// For the DSP, the spectrum is divided into MAX_EQUALIZATION_MPEG_COEFF bands.
//
// Each time the user gives a level for a user band, we calculate a level for
// each DSP band included in the user band, so that we obtain the following
// result:
//        level
//          ^
//          |
//  2*(L/2) |     ***               (case of a user band with 3 DSP band)
//          |                                           meMinIndex = j+0
//  1*(L/2) | ***     ***                               meMedIndex = j+1
//          |                                           meMaxIndex = j+2
//  0*(L/2) +---------------------> DSP band number
//            j+0 j+1 j+2
//
//        level
//          ^
//          |
//  2*(L/2) |     *** ***           (case of a user band with 4 DSP band)
//          |                                           meMinIndex = j+0
//  1*(L/2) | ***         ***                           meMedIndex = j+1
//          |                                           meMaxIndex = j+3
//  0*(L/2) +---------------------> DSP band number
//
//            j+0 j+1 j+2 j+3
//
// To obtain this result, we use the following formula:
// if i <= meMedIndex
//    Li = (i - meMinIndex + 1) * delta
// if i >= meMedIndex
//    Li = (meMaxIndex - i + 1) * delta
// where
//   delta = L / ( ( (meMaxIndex-meMinIndex) div 2 ) + 1 )
//   L is the user level
//
// Before being given to the DSP, the levels associated with each DSP band are
// translated into coefficient.
typedef struct _MPEG_EQUAL_INFO
{
    DWORD   meMinIndex;    // in [0,MAX_EQUALIZATION_MPEG_COEFF[
    DWORD   meMedIndex;    // in [0,MAX_EQUALIZATION_MPEG_COEFF[
    DWORD   meMaxIndex;    // in [0,MAX_EQUALIZATION_MPEG_COEFF[
} MPEG_EQUAL_INFO, * PMPEG_EQUAL_INFO;

// This array is indexed by meBandNumber.
static MPEG_EQUAL_INFO TabMpegEqual[MPEG_EQUAL_MAX_BANDS] =
{
    // MPEG_EQUAL_BASS_BAND
    {
      0,                                // meMinIndex
      0,                                // meMedIndex
      0,                                // meMaxIndex
    },
    // MPEG_EQUAL_MEDIUM_BAND
    {
      1,                                // meMinIndex
      2,                                // meMedIndex
      3,                                // meMaxIndex
    },
    // MPEG_EQUAL_TREBLE_BAND
    {
      4,                                // meMinIndex
      10,                               // meMedIndex
      31,                               // meMaxIndex
    },
};

// ************************************************************************* //
//                                                                           //
//  PROTOTYPE FUNCTIONS                                                      //
//                                                                           //
// ************************************************************************* //

STATIC BOOLEAN  LookupBoardFlavor(IN BYTE,IN BYTE,IN BYTE,OUT PASSOC_BOARD_FLAVOR_INFO *);
STATIC WORD     GetSystemFeature(OUT PSYSTEM_FEATURES_INFO);
//STATIC WORD   GenericBoardNameString(IN WORD,IN WORD,INOUT LPTSTR);


// ************************************************************************* //
//                                                                           //
//  LOCAL FUNCTIONS                                                          //
//                                                                           //
// ************************************************************************* //

// ****************************************************************************
// STATIC BOOLEAN LookupBoardFlavor()
// **********************************
//
// Input parameters :
// *******************
//
//  BYTE PmBoardType:   the board type identifier
//  WORD PmCSID:        the board CSID (or a fake one for non-PCI board)
//
// Output parameters :
// *******************
//
//  PASSOC_BOARD_FLAVOR_INFO *PmBoardFlavorPtr:  a descriptor of the type flavor
//
// Return value :
// **************
//
//  TRUE if this flavor has been identified, FALSE otherwise
//
// ****************************************************************************
//
// Returns to the board flavor characteristics (e.g. balanced line connectors
// for an LCM220 B)
//
// ****************************************************************************
STATIC  BOOLEAN LookupBoardFlavor(
    IN  BYTE                        PmBoardType,
    IN  BYTE                        PmExtendedBoardType,
    IN  BYTE                        PmFlavorNum,
    OUT PASSOC_BOARD_FLAVOR_INFO    *PmBoardFlavorPtr )
{
    register INT    i;

    // ## FM (08/27/98) -- Add support for NP boards with PLX chip:
    // Now scan the AssocBoardFlavor array for an entry matching
    // the couple (PmBoardType, PmFlavorNum).
    // -------------------------------------------------------
    *PmBoardFlavorPtr = NULL ;

    for ( i = 0 ; i < MAX_BOARD_FLAVOR_ENTRIES ; i++ )
    {
        if (    (AssocBoardFlavor[i].abBoardType == PmBoardType)
             && (AssocBoardFlavor[i].abExtendedBoardType == PmExtendedBoardType)
             && (AssocBoardFlavor[i].abFlavorNum == PmFlavorNum) )
        {
            *PmBoardFlavorPtr = &(AssocBoardFlavor[i]);
            break;
        }
    }

    return( *PmBoardFlavorPtr != NULL );
}


// ****************************************************************************
// STATIC WORD GetNumberOfBoardInMask(IN WORD PmMask)
// **************************************************
//
// Input parameters :
// *******************
//
//  WORD PmMask : Mask of the present boards
//
// Return value :
// **************
//
//  The number present boards. 
//
// ****************************************************************************
STATIC WORD GetNumberOfBoardInMask(IN WORD PmMask)
{
    WORD i=0,j=0;
    for(;i<16;i++){
      if((1<<i) & PmMask) j++;  
    }
    return j;
}


// ****************************************************************************
// STATIC WORD GetNumFrom( IN WORD PmIndex, IN WORD PmMask )
// *********************************************************
//
// Input parameters :
// *******************
//
//  WORD PmIndex : Index of the a present boards
//  WORD PmMask  : Mask of the present boards
//
// Return value :
// **************
//
//  Position in the bit mask of the present boards. -1 if no board present. 
//
// ****************************************************************************
STATIC WORD GetNumFrom( IN WORD PmIndex, IN WORD PmMask )
{
    WORD i,j=0;
    for(i=0;i<16;i++)
    {
        if ( (1<<i) & PmMask ) j++;
        if ( j == (PmIndex+1) ) return i;
    }
    return (WORD)-1;
}


// ****************************************************************************
// STATIC WORD GetSystemFeature()
// ******************************
//
// Output parameters :
// *******************
//
// PSYSTEM_FEATURES_INFO PmSystemInfo : Description of system features of the driver
//
// Return value :
// **************
//          SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Returns to the application the system characteristics of the driver.
//
// ****************************************************************************
STATIC WORD GetSystemFeature(
    OUT PSYSTEM_FEATURES_INFO PmSystemInfo )
{
    LPSYSTEM_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPSYSTEM_RESP_INFO  LcPReply = 0;     // will point to the reply


    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    SYSTEM_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( SYSTEM_REQ_INFO ),
                    sizeof( SYSTEM_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    if ( !(LcPReply->RespHeader.rhCptr & ERROR_MASK) )
    {
        // ## FS (09/06/1997) - make sure the application will not read
        //                      rubbish data
        // ------------------------------------------------------------
        MEMSET2( PmSystemInfo, 0, SYSTEM_FEATURES_INFO, 1 );

        PmSystemInfo->siReleaseNumber   = LcPReply->sfRelease;
        PmSystemInfo->siVersionNumber   = LcPReply->sfNVers;
        PmSystemInfo->siVersionType     = LcPReply->sfTVers;

        PmSystemInfo->siPatchLevel            = 'a' + PCX_API_VERSION_SUBRELEASE;
        PmSystemInfo->siAllocatedBufferNumber = (WORD)LcPReply->sfAllocBuff ;
        PmSystemInfo->siAvailableBufferNumber = (WORD)LcPReply->sfAvailBuff ;
        PmSystemInfo->siBufferSize            = LcPReply->sfBuffSize ;
        PmSystemInfo->siMemoryType            = LcPReply->sfTypeMem ;
        PmSystemInfo->siPresentBoardMask      = LcPReply->sfPresentBoard ;
    }


    PmSystemInfo->siAvailableInfo = PCX_SYSTEM_INFO_AVAILABLE;
    PmSystemInfo->siPresentBoard  = GetNumberOfBoardInMask(LcPReply->sfPresentBoard);
    PmSystemInfo->siMissingBoard  = 0;

    return( LcPReply->RespHeader.rhCptr );
}


// ************************************************************************************
// WORD PCXGetBoardLogicalNumber
//
// Input parameters :
// ******************
// - PmMaxPresentBoard : Number max of logical present board to retrieve
// - PmMaxMissingBoard : Number max of logical missing board to retrieve
//
// Output parameters :
// *******************
// - PmpPresentBoardLogicalNumTable : Logical number array of present board
// - PmpMissingBoardLogicalNumTable : Logical number array of missing board
// - PmpActualPresentBoard : Actual number of present board
// - PmpActualMissingBoard : Actual number of missing board
//
// Return value :
// **************
// - An error code
//
// ************************************************************************************
// This function return logical number infos about board managed by PCX driver.
// Used for the new API
// ************************************************************************************
WORD _CDECL_ PCXGetBoardLogicalNumber(
    IN WORD   PmMaxPresentBoard,
    IN WORD   PmMaxMissingBoard,
   OUT LPWORD PmpPresentBoardLogicalNumTable,
   OUT LPWORD PmpMissingBoardLogicalNumTable,
   OUT LPWORD PmpActualPresentBoard,
   OUT LPWORD PmpActualMissingBoard )
{
    return PCXGetBoardLogicalNumberEx( PmMaxPresentBoard,
                                       PmMaxMissingBoard,
                                       PmpPresentBoardLogicalNumTable,
                                       PmpMissingBoardLogicalNumTable,
                                       PmpActualPresentBoard,
                                       PmpActualMissingBoard,
                                       NULL,
                                       NULL ) ;
}
                                

// ************************************************************************************
// WORD PCXGetBoardLogicalNumberEx
//
// Input parameters :
// ******************
// - PmMaxPresentBoard : Number max of logical present board to retrieve
// - PmMaxMissingBoard : Number max of logical missing board to retrieve
//
// Output parameters :
// *******************
// - PmpPresentBoardLogicalNumTable : Logical number array of present board
// - PmpMissingBoardLogicalNumTable : Logical number array of missing board
// - PmpActualPresentBoard : Actual number of present board
// - PmpActualMissingBoard : Actual number of missing board
// - PmpPresentBoardNumTable : Logical number array of present board
// - PmpMissingBoardNumTable : Mask array of missing board
//
// Return value :
// **************
// - An error code
//
// ************************************************************************************
// This function return logical number infos about board managed by PCX driver.
// Used for the new DISPATCH
// ************************************************************************************

#define NEW_DRV_DISPATCH_BASE 1050	// maj HRC pour integration cartes LXES dans nouvelle api 


WORD _CDECL_ PCXGetBoardLogicalNumberEx(
    IN WORD   PmMaxPresentBoard,
    IN WORD   PmMaxMissingBoard,
   OUT LPWORD PmpPresentBoardLogicalNumTable,
   OUT LPWORD PmpMissingBoardLogicalNumTable,
   OUT LPWORD PmpActualPresentBoard,
   OUT LPWORD PmpActualMissingBoard,
   OUT LPWORD PmpPresentBoardNumTable,
   OUT LPWORD PmpMissingBoardNumTable )
{  
    LPSYSTEM_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPSYSTEM_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD i;
    WORD LcActualPresentBoard = 0;
    WORD LcActualMissingBoard = 0;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    SYSTEM_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( SYSTEM_REQ_INFO ),
                    sizeof( SYSTEM_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );
    
    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    if ( !(LcPReply->RespHeader.rhCptr & ERROR_MASK) )
    {
        LcActualPresentBoard = GetNumberOfBoardInMask( LcPReply->sfPresentBoard );
    }

    if ( PmMaxPresentBoard > LcActualPresentBoard ) PmMaxPresentBoard = LcActualPresentBoard ;

    if ( PmpPresentBoardLogicalNumTable ) 
        for(i=0;i<PmMaxPresentBoard;i++)
        {
            PmpPresentBoardLogicalNumTable[i] = GetNumFrom(i,LcPReply->sfPresentBoard)
                                              + NEW_DRV_DISPATCH_BASE;
        }

    if ( PmpPresentBoardNumTable )
        for(i=0;i<PmMaxPresentBoard;i++)
        {
            PmpPresentBoardNumTable[i] = GetNumFrom(i,LcPReply->sfPresentBoard);
        }

    if ( PmpActualPresentBoard ) *PmpActualPresentBoard = LcActualPresentBoard ;
    if ( PmpActualMissingBoard ) *PmpActualMissingBoard = LcActualMissingBoard ;

    return SUCCESS;
}




// ****************************************************************************
// STATIC BOOLEAN AllocWaitErrorCmdEntry()
// ***************************
//
// Input parameters :
// ****************
//      PmAppHandle:    the handle of the application we are looking for
//                       a free entry in the WaitErrorCmds tab.
//
// Output parameters :
// *******************
//      NONE
// Return value :
// **************
//      TRUE if no entry matches the given handle, FALSE otherwise
// ****************************************************************************
//
// Performs a simple scan through the tab, and find out a free entry.
// Assign this entry to the application identified by PmApphandle
//
// ****************************************************************************
STATIC BOOLEAN AllocWaitErrorCmdEntry(
    IN PCX_HANDLE   PmAppHandle )
{
    WORD    i ;

    // First of all, find a free slot in WaitErrorCmd table for the
    // application (assuming an application never asks twice for
    // buffers without deallocating, hence a free slot IS available)
    i = 0;
    while ( (i < MAX_PCX_HANDLE)    // just in case...
           && (WaitErrorCmds[i].acbAppHandle != 0) )
    {
        i++;
    }

    if ( i == MAX_PCX_HANDLE ) return FALSE;

    // one slot is available, initialize it
    // let's zeroe all the entry

    MEMSET2( &(WaitErrorCmds[i]), 0, APPLI_CMD_BLK_INFO, 1 );

    // Mark the entry as reserved by this application
    WaitErrorCmds[i].acbAppHandle = PmAppHandle;
    WaitErrorCmds[i].acbWaitErrorReply.RespHeader.rhT = PROCESSED;
    WaitErrorCmds[i].acbWaitBoardReply.RespHeader.rhT = PROCESSED;
    WaitErrorCmds[i].acbWaitGpioReply.RespHeader.rhT = PROCESSED;

    return TRUE;
}


// FS - 02/10/1996
//   this function has been added with WaitErrorCmds[]
//
// ****************************************************************************
// STATIC BOOLEAN LookupWaitErrorCmdTab()
// ***************************
//
// Input parameters :
// ****************
//      PmAppHandle:    the handle of the application we are looking for
//                       an entry in the WaitErrorCmds tab.
// Output parameters :
// *******************
//      PmAppIndex:     the index corresponding to the application identified
//                       by PmAppHandle, if any.
// Return value :
// **************
//      TRUE if an entry matches the given handle, FALSE otherwise
// ****************************************************************************
//
// Performs a simple scan through the tab, and a test of each entry handle
// field against PmAppHandle
//
// ****************************************************************************
STATIC BOOLEAN LookupWaitErrorCmdTab(
    IN PCX_HANDLE   PmAppHandle,
    OUT LPWORD       PmAppIndex )
{
    WORD i = 0;

    while ( i < MAX_PCX_HANDLE )
    {
        if ( WaitErrorCmds[i].acbAppHandle == PmAppHandle )
        {
            *PmAppIndex = i;
            return TRUE;
        }
        i++;
    }
    return FALSE;
}

// FS - 02/10/1996
//   this function has been added with WaitErrorCmds[]
//
// ****************************************************************************
// STATIC VOID FreeAllWaitCmdEntry()
// ***************************
//
// Input parameters :
// ****************
//      PmAppHandle:    the handle of the application that has
//                       an entry to be freed in the WaitErrorCmds tab.
//                       and in the WaitCmds tab as well
//
// ****************************************************************************
//
// Performs a simple scan through the tab, finds out the entry for
// the application identified by PmApphandle, and frees it.
//
// ****************************************************************************
STATIC BOOLEAN FreeAllWaitCmdEntry(
    IN PCX_HANDLE   PmAppHandle )
{
    WORD    i;

    // First, locate the entry
    // in the tab of asynchronous command blocks
    if ( !LookupWaitErrorCmdTab( PmAppHandle, &i ) )
    {
        return FALSE;   //   The application gave a wrong handle
                        //   or did not reserve any buffer
    }

    // And finally, mark the entry as free
    WaitErrorCmds[i].acbAppHandle = 0;

    // find related slots in WaitCmd table for the
    // application
    // --------------------------------------------------------
    for( i = 0; i < (MAX_INPIPE + MAX_OUTPIPE); i++ )
    {
        WORD LcLoop = 0;

        if(WaitCmds[i].wcbAppHandle != PmAppHandle)
        {
            continue;
        }

#ifdef WIN32
        // before we FREE wcbAAP and wcbOverlappedTab,
        // we must take care, that all waiting PlayStream/RecordStream terminate !!
        //
        while( LcLoop++ < 10 )
        {
            WORD j;
            for( j=0; j<WaitCmds[i].wcbNbBuff; j++ )
            {
                // Test PlayStream or RecordStream ! (common data rhT)
                if( WaitCmds[i].wcbAAP[j].aapPlayStream.rhT != PROCESSED )
                {
#ifdef _DEBUG
                    OutputDebugString(_T("LXESAPI : ReleasePipe : Ooops, Buffer not PROCESSED ! wait\n"));
#endif //_DEBUG
                    Sleep(0);
                    break;  // break for
                }
            }
            if( j == WaitCmds[i].wcbNbBuff )
            {
                // all PlayStream/RecordStream PROCESSED successfully
                // now we can FREE the memory !
               break;  // break while
            }
        }
#endif  // WIN32

        // Now free what has been allocated..
        //
        FREE(WaitCmds[i].wcbAAR);
        FREE(WaitCmds[i].wcbAAP);

        FREE(WaitCmds[i].wcbWNR);
        FREE(WaitCmds[i].wcbWEPR);
        FREE(WaitCmds[i].wcbWERR);

#ifdef WIN32
        FREE(WaitCmds[i].wcbOverlappedTab );
#endif
        BZERO2(&WaitCmds[i], WAIT_CMD_BLK_INFO, 1);
    }

    return TRUE;
}


// ****************************************************************************
// BOOLEAN GENLookupWaitCmdTab()
// STATIC BOOLEAN LookupWaitCmdTab()
// ********************************
//
// Input parameters :
// ****************
//      PmOutPipeMask:    the mask of output pipe we are looking for
//                        an entry in the WaitCmds tab, or 0
//      PmInPipeMask:     the mask of input pipe we are looking for
//                        an entry in the WaitCmds tab, or 0
// Output parameters :
// *******************
//      PmIndex:        the index of corresponding entry to the pipe, if any.
//
// Return value :
// **************
//      TRUE if an entry matches the given handle, FALSE otherwise
// ****************************************************************************
//
// Performs a simple scan through the tab, and a test of each entry handle
// field against PmPipeMask
//
// FS - 06/02/97
// We must separate output and input pipe masks
//
// ****************************************************************************
STATIC BOOLEAN LookupWaitCmdTab(
    IN  DWORD       PmOutPipeMask,
    IN  DWORD       PmInPipeMask,
    OUT LPWORD      PmIndex )
{
    WORD i = 0;

    while ( i < (MAX_INPIPE + MAX_OUTPIPE) )
    {
        if (   ( WaitCmds[i].wcbOutPipeMask == PmOutPipeMask )
            && ( WaitCmds[i].wcbInPipeMask == PmInPipeMask ) )
        {
            *PmIndex = i;
            return TRUE;
        }
        i++;
    }

    return FALSE;
}

BOOLEAN GENLookupWaitCmdTab(
    IN  DWORD       PmOutPipeMask,
    IN  DWORD       PmInPipeMask,
    OUT LPWORD      PmIndex )
{
  return( LookupWaitCmdTab(PmOutPipeMask,PmInPipeMask,PmIndex) );
}

WORD GENGetStreamInternalBufferInfo(
    IN  WORD        PmWaitCmdIndex,
 INOUT  LPDWORD     PmNbStreamPtr,
   OUT  LPDWORD     PmStreamInternalBufferSizeArray )
{
   if ( *PmNbStreamPtr > MAX_STREAM_PIPE ) *PmNbStreamPtr = MAX_STREAM_PIPE ;

   if ( PmStreamInternalBufferSizeArray )
   {
       // master DMA cards : always 0
       BZERO2( PmStreamInternalBufferSizeArray, DWORD, *PmNbStreamPtr );
   }

   if ( *PmNbStreamPtr < WaitCmds[PmWaitCmdIndex].wcbNbStreams )
   {
       *PmNbStreamPtr = WaitCmds[PmWaitCmdIndex].wcbNbStreams ;
       return( WA_MORE_DATA );
   }
   else
   {
       return( SUCCESS );
   }
}

// ****************************************************************************
// STATIC VOID FreeWaitCmdEntry()
// ***************************
//
// Input parameters :
// ****************
//      PmOutPipeMask:    the deleted ouput pipe mask or 0
//
//      PmInPipeMask :    the deleted input pipe mask or 0
//
// ****************************************************************************
//
// Free the dynamically allocated command blocks and reinitializes the
// static ones.
//
// FS - 06/02/97
// We must separate output and input pipe masks
//
// ****************************************************************************
STATIC VOID FreeWaitCmdEntry(
    IN DWORD    PmOutPipeMask,
    IN DWORD    PmInPipeMask,
    IN BOOLEAN  PmWaitForProcessed)
{
    WORD    i;

    // First, locate the entry
    // in the tab of asynchronous command blocks
    if ( LookupWaitCmdTab( PmOutPipeMask, PmInPipeMask, &i ) )
    {
#ifdef WIN32
        // before we FREE wcbAAP and wcbOverlappedTab,
        // we must take care, that all waiting PlayStream/RecordStream terminate !!
        //
        WORD LcLoop = 0;
        while( PmWaitForProcessed && (LcLoop++ < 10) )
        {
            WORD j;
            for( j=0; j<WaitCmds[i].wcbNbBuff; j++ )
            {
                // Test PlayStream or RecordStream ! (common data rhT)
                if( WaitCmds[i].wcbAAP[j].aapPlayStream.rhT != PROCESSED )
                {
#ifdef _DEBUG
                    OutputDebugString(_T("LXESAPI : ReleasePipe : Ooops, Buffer not PROCESSED ! wait\n"));
#endif //_DEBUG
                    Sleep(0);
                    break;  // break for
                }
            }
            if( j == WaitCmds[i].wcbNbBuff )
            {
                // all PlayStream/RecordStream PROCESSED successfully
                // now we can FREE the memory !
               break;  // break while
            }
        }
#endif  // WIN32

        // Now free what has been allocated..
        FREE(WaitCmds[i].wcbAAR);
        FREE(WaitCmds[i].wcbAAP);

        FREE(WaitCmds[i].wcbWNR);
        FREE(WaitCmds[i].wcbWEPR);
        FREE(WaitCmds[i].wcbWERR);

#ifdef WIN32
        FREE(WaitCmds[i].wcbOverlappedTab );
#endif
        BZERO2(&WaitCmds[i], WAIT_CMD_BLK_INFO, 1);
    }
}


// ****************************************************************************
// STATIC BOOLEAN AllocWaitCmdEntry()
// ***************************
//
// Input parameters :
// ****************
//      PmMyProcess:    the application handle
//      PmPipeNum:      the created pipe number
//      PmNbBuffers:    how many buffers the application has successfully
//                       reserved.
//      PmFakeFrequency: pipe requires special handling case
//
// Return value :
// **************
//      TRUE if the allocation of wait command blocks completes successfully,
//  FALSE otherwise
// ****************************************************************************
//
// initializes request/reply blocks for WaitEndOfPlay
// allocates request/reply blocks for asynchronous commands on PmNbBuffers.
//
// FS - 06/02/97
// We must separate output and input pipe masks
//
// ****************************************************************************
STATIC BOOLEAN AllocWaitCmdEntry(
    IN PCX_HANDLE   PmMyProcess,
    IN DWORD        PmOutPipeMask,
    IN DWORD        PmInPipeMask,
    IN WORD         PmNbBuffers,
    IN DWORD        PmMaxBufferSize,
    IN DWORD        PmNbStreams )
{
    WORD    i, j ;

    // First of all, find a free slot in WaitCmd table for the
    // pipe
    // --------------------------------------------------------
    i = 0;
    while ( (i < (MAX_INPIPE + MAX_OUTPIPE) )    // just in case...
           && (  (WaitCmds[i].wcbOutPipeMask != 0)
               || (WaitCmds[i].wcbInPipeMask != 0) ) )
    {
        // are the same masks already used in this slot ???
        // can only happen if system comes out from sleep and the driver suppressed the pipes.
        // a ReleasePipe does not work and a new DefinePipe gives the same masks !
        if( (WaitCmds[i].wcbOutPipeMask == PmOutPipeMask) &&
            (WaitCmds[i].wcbInPipeMask == PmInPipeMask) )
        {
            FreeWaitCmdEntry( PmOutPipeMask, PmInPipeMask, FALSE );
            NPNTAPITRACE("PCXHRAPI : Reuse Waitcommand Entry\n");
            break;  // re-use this entry
        }
        i++;
    }

    if ( i == (MAX_INPIPE + MAX_OUTPIPE) ) return FALSE;

    // one slot is available, initialize it and try the
    // dynamic allocation

    // let's zeroe all the entry
    MEMSET2( &(WaitCmds[i]), 0, WAIT_CMD_BLK_INFO, 1 );

    // Mark the entry as reserved by this application
    WaitCmds[i].wcbOutPipeMask = PmOutPipeMask;
    WaitCmds[i].wcbInPipeMask = PmInPipeMask;
    WaitCmds[i].wcbAppHandle = PmMyProcess;

    // Remember stream specific info
    WaitCmds[i].wcbNbStreams = PmNbStreams;

    // and try the allocations
    //
    WaitCmds[i].wcbWNR  = MALLOC2(ANY_ASYNC_REQUEST_INFO,1);
    WaitCmds[i].wcbWNP  = &WaitCmdsNofifResp[i];

    WaitCmds[i].wcbWEPR = MALLOC2(ANY_ASYNC_REQUEST_INFO,1);
    WaitCmds[i].wcbWEPP = &WaitCmdsPlayEndResp[i];

    WaitCmds[i].wcbWERR = MALLOC2(ANY_ASYNC_REQUEST_INFO,1);
    WaitCmds[i].wcbWERP = &WaitCmdsRecEndResp[i];

    if(PmNbBuffers == 0)
    {
        WaitCmds[i].wcbAAR  = NULL;
        WaitCmds[i].wcbAAP  = NULL;
#ifdef WIN32
        WaitCmds[i].wcbOverlappedTab = NULL;
#endif
    }
    else
    {
        WaitCmds[i].wcbAAR  = MALLOC2(ANY_ASYNC_REQUEST_INFO,PmNbBuffers);
        WaitCmds[i].wcbAAP  = MALLOC2(ANY_ASYNC_REPLY_INFO,PmNbBuffers);
#ifdef WIN32
        WaitCmds[i].wcbOverlappedTab = MALLOC2(OVERLAPPED, PmNbBuffers);
#endif
    }

    if (   ((WaitCmds[i].wcbAAR == 0) && (PmNbBuffers >0))
        || ((WaitCmds[i].wcbAAP == 0) && (PmNbBuffers >0))
        || (WaitCmds[i].wcbWNR == 0)
        || (WaitCmds[i].wcbWEPR == 0)
#ifdef WIN32
        || ((WaitCmds[i].wcbOverlappedTab == 0) && (PmNbBuffers >0))
#endif
        || (WaitCmds[i].wcbWERR == 0) )
    {
        // Allocation failure, free what has been allocated..
        // FS - 06/02/97
        // We must separate output and input pipe masks
        FreeWaitCmdEntry( PmOutPipeMask, PmInPipeMask, FALSE );
        return FALSE;
    }

    BZERO2(WaitCmds[i].wcbWNR, ANY_ASYNC_REQUEST_INFO, 1);
    BZERO2(WaitCmds[i].wcbWNP, ANY_ASYNC_REPLY_INFO,   1);
    BZERO2(WaitCmds[i].wcbWEPR, ANY_ASYNC_REQUEST_INFO, 1);
    BZERO2(WaitCmds[i].wcbWEPP, ANY_ASYNC_REPLY_INFO,   1);
    BZERO2(WaitCmds[i].wcbWERR, ANY_ASYNC_REQUEST_INFO, 1);
    BZERO2(WaitCmds[i].wcbWERP, ANY_ASYNC_REPLY_INFO,   1);

    if( PmNbBuffers )
    {
        BZERO2(WaitCmds[i].wcbAAR, ANY_ASYNC_REQUEST_INFO, PmNbBuffers);
        BZERO2(WaitCmds[i].wcbAAP, ANY_ASYNC_REPLY_INFO, PmNbBuffers);
#ifdef WIN32
        BZERO2(WaitCmds[i].wcbOverlappedTab, OVERLAPPED, PmNbBuffers);
#endif
    }

    // Init blocks for tests in PlayFinished :
    //  - set request blocks' handle field to application handle
    //  - set reply blocks' bit T to PROCESSED
    //
    (WaitCmds[i].wcbWaitEndPlayRequest).Header.hdHandle = (APP_HANDLE) PmMyProcess;
    (WaitCmds[i].wcbWaitEndPlayReply).rhT = PROCESSED;

    (WaitCmds[i].wcbWaitEndRecordRequest).Header.hdHandle = (APP_HANDLE) PmMyProcess;
    (WaitCmds[i].wcbWaitEndRecordReply).rhT = PROCESSED;


    // FS - (25/04/1997)
    (WaitCmds[i].wcbWaitNotifyRequest).Header.hdHandle = (APP_HANDLE)PmMyProcess;
    (WaitCmds[i].wcbWaitNotifyReply).rhT = PROCESSED;

    // Remember how many buffers have been reserved by the appli
    // i.e the size of the dynamically-allocated table.
    WaitCmds[i].wcbNbBuff = PmNbBuffers;
    WaitCmds[i].wcbMaxBuffSize = PmMaxBufferSize;

    // 09/10/1996 - FS
    // Initialize all reply blocks to tell the application that
    // all buffers are available (or not used by driver bit T = 1)
    for ( j = 0; j < PmNbBuffers; j++ )
    {
        // Well, let's set the T bit through a play request
        // in order to avoid an ugly cast.
        // However, it doesn't really matter since rhT is member
        // of the common header for play/record and all commands
        //
        WaitCmds[i].wcbRequestTab[j].aarPlayStream.Header.hdHandle =
                                                     (APP_HANDLE) PmMyProcess;
        WaitCmds[i].wcbReplyTab[j].aapPlayStream.rhT = PROCESSED ;

        // FS - 04/02/97
        // Zeroe the length of a record buffer
        // -----------------------------------
        WaitCmds[i].wcbReplyTab[j].aapRecordStream.rrpDataLength = 0 ;
    }

    return TRUE;
}

// ****************************************************************************
// WORD GENGetDspFilename()
// ***************************
//
// Input parameters :
// ****************
//
// WORD PmSoftwareNumber: Version of the concerned software.
//
// Output parameters :
// *******************
//
// LPTSTR PmDspPathName: Points to a null terminating string containing the path
//                      to driver NP folder.
// LPTSTR PmDspFileName: Points to a null terminating string containing the name
//                      of the DSP file requested (if PmSoftwareNumber is not
//                      null), not significant otherwise.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function returns the name of a DSP software which is used
// for downloading on a certain type of DSP.
//
// ****************************************************************************

WORD GENGetDspFileName(
    IN  WORD PmSoftwareNumber,
    OUT LPTSTR PmDspPathName,
    OUT LPTSTR PmDspFileName)
{
    LPDRIVER_PATH_REQ_INFO  LcPReq      = 0;    // will point to the beginning
                                                // of the request block
    LPDRIVER_PATH_RESP_INFO LcPResp     = 0;    // will point to the beginning
                                                // of the response block
    WORD                    LcLength    ;
    WORD                    LcRet       ;
#ifdef WIN32
    OSVERSIONINFO           LcovOsInfos ;
#endif

    _TCHAR                  LcszDriverPathName[_MAX_PATH];
    _TCHAR                  LccDrive;


    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        return LcRet;
    }

    // Ask the driver what's the path for driver components
    // ----------------------------------------------------

    // filling the header of the request block
    COMFillBcHeader( GENE_FAM,
                    DRIVER_PATH_CMD,
                    (APP_HANDLE) 0,
                    sizeof( DRIVER_PATH_REQ_INFO ),
                    sizeof( DRIVER_PATH_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq);

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // processing the response
    // -----------------------
    if ( !(LcPResp->RespHeader.rhCptr & ERROR_MASK) )
    {
        LcLength = LcPResp->dnpSize ;

#if defined(_UNICODE) || defined(UNICODE)
        _snwprintf( PmDspPathName, LcLength, L"%hs", LcPResp->dnpFileName );
#else
        strncpy( PmDspPathName, LcPResp->dnpFileName, LcLength );
#endif
        PmDspPathName[LcLength] = _T('\0');


#ifdef WIN32
       // Dynamically check on which OS we are running
       //
       LcovOsInfos.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
       GetVersionEx( &LcovOsInfos );
       if (LcovOsInfos.dwPlatformId == VER_PLATFORM_WIN32_NT)
       {

           // Intepret SystemRoot if it's present as the first directory name in the path

            _tcslwr(PmDspPathName);
            if (_tcsncmp(PmDspPathName, _T("\\systemroot"), 11) == 0)
            {                                               // path relative to system root
                LcszDriverPathName[0] = _T('\0');

                GetWindowsDirectory(LcszDriverPathName,_MAX_PATH);

                _tcsnccat(LcszDriverPathName, PmDspPathName + 11, _MAX_PATH - 11);

                LcszDriverPathName[_MAX_PATH-1] = _T('\0');

                _tcscpy(PmDspPathName, LcszDriverPathName);
            }                                               // path relative to system root
            else
            {
                _TCHAR  LcszDrive[3] ;

                _tcscpy( LcszDrive, _T("A:") );
                // Find correspondence between arcname ("\Device\Partition0...") and DOS name ("C:\").
                for ( LccDrive = _T('A') ; LccDrive <= _T('Z'); LccDrive++ )
                {
                    LcszDrive[0] = LccDrive;

                    // Get arcname of drive.
                    if ( QueryDosDevice( LcszDrive, LcszDriverPathName, _MAX_PATH))
                    {
                        size_t iDriverDirectoryLength = _tcslen( LcszDriverPathName );

                        // Drive's arcname matches the BinariesRootPath's one.
                        if ( !_tcsncicmp( PmDspPathName, LcszDriverPathName, iDriverDirectoryLength ))
                        {
                            // Replace arcname with drive.
                            _tcscpy( PmDspPathName, LcszDrive );
                            _tcscat( PmDspPathName, &PmDspPathName[iDriverDirectoryLength] );
                            break;
                        }
                    }
                }
            }
       }
#endif // WIN32
    }

    // Copy the DSP name into the PmFileName string
    // --------------------------------------------
    if ( LcPResp->RespHeader.rhCptr == SUCCESS )
    {
        LPTSTR   LcSoftName = DspCapabilities[PmSoftwareNumber].dciFileName;

        _tcscpy( PmDspFileName, LcSoftName );
    }

    // returning the error code
    return LcPResp->RespHeader.rhCptr;
}

// ****************************************************************************
// BOOLEAN GENCanGetNotification()
// *******************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE   PmMyProcess:    Handle of the calling process.
// DWORD        PmInPipeMask:   Mask of the Input pipe concerned by the notify
//                      (if it is significant).
// DWORD        PmOutPipeMask:  Mask of the Input pipe concerned by the notify
//                      (if it is significant).
//
// Return value :
// **************
//
//  O:  if (PmMyProcess,PmInPipeMask) or (PmMyProcess,PmOutPipeMask) does
//      not correspond to a valid pipe and a WaitForNotification request,
//      or it corresponds but the WaitForNotification request is still
//      waiting
//  1:  the (PmMyProcess,PmInPipeMask) or (PmMyProcess,PmOutPipeMask) does
//      correspond to a WaitForNotification request that has returned
//
// ****************************************************************************
//
// Function to know whether an async waitfornotifcation command
// is actually completed
//
// ****************************************************************************
BOOLEAN GENCanGetNotification(IN  PCX_HANDLE PmMyProcess,
                              IN  DWORD PmInPipeMask,
                              IN  DWORD PmOutPipeMask)
{
    LPWAIT_NOTIFY_REQ_INFO  LcWaitNotifyReq;
    LPRESP_HEADER_INFO      LcWaitNotifyResp;
    WORD                    LcPipeIndex;

    // Extract pipe number from the mask
    // ---------------------------------
    if ( ( PmOutPipeMask == 0 ) && ( PmInPipeMask == 0 ) )
    {
        return FALSE;
    }

    // Lookup the pipe in WaitCmds
    // ----------------------------
    if ( !LookupWaitCmdTab( PmOutPipeMask, PmInPipeMask, &LcPipeIndex ) )
    {
        return FALSE;
    }

    // Locate the request / reply command block assigned to the pipe
    // -------------------------------------------------------------
    LcWaitNotifyReq  = &(WaitCmds[LcPipeIndex].wcbWaitNotifyRequest) ;
    LcWaitNotifyResp = &(WaitCmds[LcPipeIndex].wcbWaitNotifyReply) ;

    return(LcWaitNotifyResp->rhT == PROCESSED);
}


// ****************************************************************************
// STATIC WORD Registration()
// ***************************
//
// Input parameters :
// ****************
//
// PAPP_DECL_INFO   PmAppDeclInfo: application descriptor for registering
// PCX_HANDLE PmHandle: application handle for a unregister command, 0xff for a
//                  register commande.
//
// Output parameters :
// *******************
//
// PPCX_HANDLE PmMyProcess: points to a BYTE wich will contains the handle value
//                       returned by the driver for the register command, not
//                       changed for the unregister command.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function send a register command or an unregister command to the
// driver. The command be performed by the driver depending on the value of the
// request block field handle.
//
// ****************************************************************************
STATIC WORD Registration(
    IN  PAPP_DECL_INFO_A    PmAppDeclInfo,
    IN  PCX_HANDLE          PmHandle,
    IN  BOOLEAN             PmIsAnsiRegistration,
    OUT PPCX_HANDLE         PmMyProcess )
{
    LPREG_REQ_INFO  LcPReq = 0;   // will point to the request
    LPREG_REP_INFO  LcPResp = 0;  // will point to the response
#ifdef WIN32
    LPWSTR          LcAppName = 0 ;
#endif
    LPSTR           LcAnsiAppName = 0 ;
    HANDLE          LcWinHandle = 0 ;
    size_t          LcSizeName = 0 ;            // application name size
    DWORD           LcWinMessage = 0 ;
    BYTE            LcAttribs = 0 ;

    if ( PmAppDeclInfo )
    {
#ifdef WIN32
        LcAppName = ((PAPP_DECL_INFO_W)PmAppDeclInfo)->adAppName ;
#endif
        LcAttribs = PmAppDeclInfo->adAppAttributes ;
        LcWinHandle = PmAppDeclInfo->adWinHandle ;
        LcWinMessage = PmAppDeclInfo->adWinMessage ;
        LcAnsiAppName = ((PAPP_DECL_INFO_A)PmAppDeclInfo)->adAppName ;

        // pre_condition control
        //
        if ( PmIsAnsiRegistration )
        {
            LcSizeName = strlen( LcAnsiAppName ) ;
        }
#ifdef WIN32
        else
        {
            LcSizeName = wcslen( LcAppName ) ;
        }
#endif
        if ( LcSizeName > SIZE_MAX_APP_NAME )
        {
            return EA_APPLICATION_NAME_SIZE;
        }
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,REG_CMD,
                     (APP_HANDLE)PmHandle,
                     sizeof(REG_REQ_INFO),
                     // FS - 11/02/1997 - 16/12/1997
                     // Under WindowsNT, we need a larger size
                     // since the reply block is use to unmap
                     // buffers of remaining pipes
                     // -------------------------------------------
                     sizeof(ANY_UNMAPPING_REPLY_INFO),
                     (BC_HEADER_HANDLE) &LcPReq );

    // filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->rgqAttributes   = LcAttribs;
    LcPReq->rgqNameSize     = (BYTE)LcSizeName;
    LcPReq->rgqWindowMessage= LcWinMessage;
    LcPReq->rgqWindowHandle = (DWORD) LcWinHandle;

    if ( PmAppDeclInfo )
    {
        LcPReq->rgqAppInfoEx    = PmAppDeclInfo->adAppInfoEx;

        if ( PmIsAnsiRegistration )
        {
            (LcPReq->rgqAppName).asStringType = APP_ANSI_STRING ;
            strcpy( (LcPReq->rgqAppName).asAnsiString, LcAnsiAppName );

            if( LcPReq->rgqAppInfoEx & APP_INFO_EX_DHS )
            {
                // test additionnel DHS
                if( (LcAnsiAppName[SIZE_MAX_APP_NAME-2] != '\0') ||
                    (LcAnsiAppName[SIZE_MAX_APP_NAME-1] != '') ||
                    (LcAnsiAppName[SIZE_MAX_APP_NAME  ] != '@') )
                {
                    LcPReq->rgqAppInfoEx &= ~APP_INFO_EX_DHS;
                }
            }
        }
#ifdef WIN32
        else
        {
            (LcPReq->rgqAppName).asStringType = APP_UNICODE_STRING ;
            wcscpy( (LcPReq->rgqAppName).asUnicodeString, LcAppName );

            if( LcPReq->rgqAppInfoEx & APP_INFO_EX_DHS )
            {
                // test additionnel DHS
                if( (LcAppName[SIZE_MAX_APP_NAME-2] != '\0') ||
                    (LcAppName[SIZE_MAX_APP_NAME-1] != '') ||
                    (LcAppName[SIZE_MAX_APP_NAME  ] != '@') )
                {
                    LcPReq->rgqAppInfoEx &= ~APP_INFO_EX_DHS;
                }
            }
        }
#endif
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // reading the response for a register commande
    // --------------------------------------------
    if ( PmHandle == REGISTER_CMD_CDE ) *PmMyProcess = LcPResp->rgpHandle;

    // save/clear the handle if it's the DHS application
    //
    if( LcPResp->RespHeader.rhCptr == SUCCESS )
    {
        if( PmHandle == REGISTER_CMD_CDE )
        {
            // register
            if( LcPReq->rgqAppInfoEx & APP_INFO_EX_DHS )
            {
                *g_pDHSAppHandleDLL = LcPResp->rgpHandle;
                NPNTAPITRACE("API DHS REGISTER\n");
            }
        }
        else
        {
            // unregister (attention PmAppDeclInfo == NULL !)
            if(PmHandle == *g_pDHSAppHandleDLL)
            {
                *g_pDHSAppHandleDLL = 0;
                NPNTAPITRACE("API DHS UNREGISTER\n");
            }
        }
    }

    // returning the error code
    return LcPResp->RespHeader.rhCptr;
}


// ****************************************************************************
// STATIC WORD ManageDsp()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// WORD PmCardMask: Mask of cards, one bit per card (least significant bit is
//                  first card, etc...).
// WORD PmDspMask: Mask of DSPs on the cards.
// DWORD PmDspFeatures :   specifies the DSP capabilities (unused if unload cmd)
// WORD PmBuffersNumber: if PmDspMask different 0 and different than 0xFF then
//                       Number of buffers containing the code of the DSP
//                       software to download, 0 otherwise.
//
// BYTE PmSoftwareNumber: 0 if PmDspMask egal 0 or 0xFF, otherwise version of
//                        DSP software to download.
// BUFFER_INFO PmBufferInfo[MAX_BUFFER]: NULL if PmDspMask egal 0 or 0xFF,
//                  otherwise Array of BUFFER_INFO structures. There is one
//                  structure per buffer.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function performs a DSP managing command. The real command performed (
// use , download or unload) will depend on the value of PmNbBuff:
// 0 : use command
// 0 < PmNbBuff < 0x7fff : download
// 0xffff: unload
//
// ****************************************************************************
STATIC WORD ManageDsp(
    IN PCX_HANDLE   PmMyProcess,
    IN WORD         PmCardMask,
    IN WORD         PmDspMask,
    IN DWORD        PmDspFeatures,
    IN WORD         PmBuffersNumber,
    IN BYTE         PmSoftwareNumber,
    IN BYTE         PmFirmwareNumber,
    IN PBUFFER_TO_LOAD_INFO PmBufferInfo )
{
    LPDSP_USE_REQ_INFO LcPReq = 0;     // will point to the beginning of the
                                       // request block
    LPRESP_HEADER_INFO LcPResp = 0;    // will point to the beginning of the
                                       // response block
    LPBUFF_DESC_INFO LcPBuffDesc;      // will point to the buffer descriptions
                                       // within the request block
    WORD                 i;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
             DSP_MANAGE_CMD,
             (APP_HANDLE)PmMyProcess,
             MAX_DSP_USE_REQ_SIZE,
             // FS - 11/02/1997 - 16/12/97
             // Under WindowsNT, we need a larger size
             // since the reply block is use to unmap
             // buffers reserved for DSP loads
             // -------------------------------------------
             sizeof(ANY_UNMAPPING_REPLY_INFO),
             (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->duqBoardMask = (WORD)PmCardMask;
    LcPReq->duqDspMask = (BYTE)PmDspMask;
    LcPReq->duqNbBuff= PmBuffersNumber;
    LcPReq->duqDspFeatures = PmDspFeatures;
    LcPReq->duqMustFakeFrequency = (DWORD)DspCapabilities[PmSoftwareNumber].dciCheckForHardware;

    if ( (PmBuffersNumber != DSP_FREE_CMD_CDE) && (PmBuffersNumber != 0) )
    {
        LcPReq->duqNumLog = PmSoftwareNumber;
        LcPReq->duqNumFirm= PmFirmwareNumber;
        for ( i = 0,
                  LcPBuffDesc = (LPBUFF_DESC_INFO)(LcPReq + 1);
              i < PmBuffersNumber;
              i ++, LcPBuffDesc++ )
        {
            LcPBuffDesc->bdNumBuff = PmBufferInfo[i].btlBufferNumber;
            LcPBuffDesc->bdDataSize = PmBufferInfo[i].btlDataLength;
        }
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // returning the error code
    return LcPResp->rhCptr;
}

// ****************************************************************************
// STATIC WORD ReleaseSynchronizedPipes()
// **************************************
//
// Input parameters :
// ****************
//      PCX_HANDLE          PmMyProcess: application handle.
//
// Input parameters :
// ****************
//      none
//
// Return value :
// **************
//      0 if all pipes are unsynced and released successfully
//
// ****************************************************************************
//
// This function is intended to clean up before unregistering
//
// ****************************************************************************
//
// 1. The function calls the driver to retrieve the list of pipes' groups that
//  remain allocated to the application
// 2. For each group, the function unsynchronizes the pipes
// 3. For each group, the function releases the pipes
// ****************************************************************************
STATIC  WORD        ReleaseSynchronizedPipes(
    IN  PCX_HANDLE  PmMyProcess )
{
    LPSYNCED_PIPE_LIST_REQ_INFO     LcPSyncRequest = 0; // will point to the beginning of the
                                                    // request block
    LPSYNCED_PIPE_LIST_RESP_INFO    LcPSyncReply = 0;   // will point to the beginning of the
                                                    // response block
    LPSYNCED_PIPE_DESC_INFO         LcPSyncPipeDesc = 0;
    DWORD                           LcNbOfPipeGroups = 0 ;
    DWORD                           LcGroupNumber;
    WORD                            LcRet;

    // Allocate an array of pipe descriptor to store the list
    // of pipe groups
    //
    LcPSyncPipeDesc = MALLOC2( SYNCED_PIPE_DESC_INFO, MAX_OUTPIPE+MAX_INPIPE );
    if ( !LcPSyncPipeDesc )
    {
        return EA_ALLOCATE_MEMORY_IMPOSSIBLE;
    }
    MEMSET2( LcPSyncPipeDesc, 0, SYNCED_PIPE_DESC_INFO,  MAX_OUTPIPE+MAX_INPIPE );

    if (!COMWaitExmut())
    {
		FREE( LcPSyncPipeDesc );
        return EA_CANT_TAKE_MUTEX;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    SYNCED_PIPES_CMD,
                    (APP_HANDLE) PmMyProcess,
                    sizeof( SYNCED_PIPE_LIST_REQ_INFO ),
                    MAX_SYNCED_PIPE_LIST_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPSyncRequest );

    LcPSyncRequest->splqHandle = (APP_HANDLE)PmMyProcess ;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPSyncRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPSyncReply,
                   0 );

    LcRet = (LcPSyncReply->RespHeader).rhCptr;
    if ( LcRet != SUCCESS )
	{
		COMReleaseExmut();
		FREE( LcPSyncPipeDesc );
		return LcRet;
	}

    // Make a secure copy of the results
    // ---------------------------------
    LcNbOfPipeGroups = LcPSyncReply->splpNbOfSyncedGroups;
    memcpy( LcPSyncPipeDesc, LcPSyncReply + 1, (size_t)(LcNbOfPipeGroups * sizeof(SYNCED_PIPE_DESC_INFO)) );

    // We do not need the mutex anymore since we have secured the results
    // ------------------------------------------------------------------
    COMReleaseExmut();

    // For each group of pipes
    // -----------------------
    for ( LcGroupNumber = 0 ; LcGroupNumber < LcNbOfPipeGroups ; LcGroupNumber++ )
    {
        LPSYNCED_PIPE_DESC_INFO    LcPPipeGroupDesc ;

        LcPPipeGroupDesc = &(LcPSyncPipeDesc[LcGroupNumber]);

        _ASSERT( LcPPipeGroupDesc->spdOutPipeMask64.HighPart == 0 ); // not handled yet
        _ASSERT( LcPPipeGroupDesc->spdInPipeMask64.HighPart == 0 );  // not handled yet

        PCXSetPipeNoSamplingClock( PmMyProcess,
                                   LcPPipeGroupDesc->spdOutPipeMask64.LowPart,
                                   LcPPipeGroupDesc->spdInPipeMask64.LowPart,
                                   NULL );
        PCXReleasePipe( PmMyProcess,
                        LcPPipeGroupDesc->spdOutPipeMask64.LowPart,
                        LcPPipeGroupDesc->spdInPipeMask64.LowPart );
    }

    FREE( LcPSyncPipeDesc );

    return LcRet ;
}

// ************************************************************************* //
//                                                                           //
//  API INTERNAL UTILITIES                                                   //
//                                                                           //
// ************************************************************************* //

// ****************************************************************************
// WORD GENInitAndCheckVersion()
// *********************************
//
// Input parameters :
// ****************
//      PmHoldMutex     :   whether the API local mutex is held or not
//
// Output parameters :
// *******************
//      PmErrorMessage  :   an error message or NULL
//
// Return value :
// **************
//      0 if initialization completes successfully and a matching driver
// has been found
//
// ****************************************************************************
//
// Retrieve the PCX driver and check version numbers match
//
// ****************************************************************************
WORD GENInitAndCheckVersion(
    IN  BOOLEAN  PmHoldMutex     ,
    OUT LPTSTR   PmErrorMessage  )
{
    SYSTEM_FEATURES_INFO        LcSystem ;
    WORD                        LcErrCode = SUCCESS ;
    STATIC BOOLEAN              Passed   = FALSE ;

    if ( Passed ) return SUCCESS ;

    // First of all, take mutex if necessary
    //
    if ( !PmHoldMutex )
    {
        if (!COMWaitExmut())
           return EA_CANT_TAKE_MUTEX;
    }

    // Retrieve the version number
    // -----------------------------
    LcErrCode = GetSystemFeature( &LcSystem );
    if ( LcErrCode != SUCCESS )
       return LcErrCode ;

    // First of all, take mutex if necessary has been taken
    // here, let's release it immediately before further processing
    //
    if ( !PmHoldMutex )
    {
        COMReleaseExmut();
    }

    if (   ( LcSystem.siReleaseNumber != PCX_API_VERSION_RELEASE )
        || ( LcSystem.siVersionNumber != PCX_API_VERSION_NUMBER )
        || ( LcSystem.siVersionType   != PCX_API_VERSION_TYPE   ) )
    {
        if ( PmErrorMessage != NULL )
        {
            _stprintf(
                PmErrorMessage,
                _T("API V%02dR%02d %s\ndoes not match\nDRV V%02dR%02d %s"),
                PCX_API_VERSION_NUMBER,
                PCX_API_VERSION_RELEASE,
                (!PCX_API_VERSION_TYPE) ? _T("Release") : ( PCX_API_VERSION_TYPE == 1 ) ? _T("Beta") : _T("Alpha"),
                LcSystem.siVersionNumber,
                LcSystem.siReleaseNumber,
                (!LcSystem.siVersionType) ? _T("Release") : ( LcSystem.siVersionType == 1 ) ? _T("Beta") : _T("Alpha") );
        }

        // Bypass if driver version is alpha
        // assuming DIGIGRAM internal use only
        // ------------------------------------
        if ( LcSystem.siVersionType != 2 )
        {
            return EA_VERSION_MISMATCH ;
        }
    }

    Passed = TRUE ;

    return SUCCESS ;

}

// ****************************************************************************
// VOID GENAudioDecl2PipeAudioInfo()
// *************************************
//
//  Input parameters :
// *******************
//
// PAUDIO_DECL_INFO PmAudioDeclInfo : The board audio description
//
// Output parameters :
// *******************
//
// LPPIPE_AUDIO_INFO PmPipeAudioInfo : Declaration of a pipe audio
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function converts an board audio user-decriptor into a driver declaration
// of a pipe audio
//
// ****************************************************************************
VOID GENAudioDecl2PipeAudioInfo(
    IN  PAUDIO_DECL_INFO PmAudioDeclInfo,
    OUT LPPIPE_AUDIO_INFO PmPipeAudioInfo )
{
    PmPipeAudioInfo->paBoard = PmAudioDeclInfo->adBoardNumber ;
    PmPipeAudioInfo->paDsp = PmAudioDeclInfo->adDspNumber ;
    PmPipeAudioInfo->paAudioNum = PmAudioDeclInfo->adAudioNumber ;
    PmPipeAudioInfo->paAudioAttributes = PmAudioDeclInfo->adAudioAttributes ;
}

// ****************************************************************************
// WORD GENGetDspBuffers()
// ********************
//
// Output parameters:
// *******************
//
//  WORD PmPBuffNum:            Number of buffers reserved
//  PADD_DESC_INFO PmAddrTab:   Points to a tab of buffer address descriptions
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This command asks the driver for the number of buffers reserved
// to downloading of DSP
//
// ****************************************************************************
WORD GENGetDspBuffers(
    OUT LPWORD           PmPBuffNum,
    OUT PADD_DESC_INFO  PmAddrTab )
{
    LPDSP_BUFF_REQ_INFO LcPReq = 0;     // will point to the beginning of the
                                        // request block
    LPDSP_BUFF_RESP_INFO LcPResp = 0;   // will point to the beginning of the
                                        // response block
    LPADD_DESC_INFO LcPAddDesc;         // will point to the address
                                        // descriptions within the response
    WORD                i, LcRet;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    DSP_BUFF_ADDR_CMD,
                    (APP_HANDLE) 0,
                    sizeof( DSP_BUFF_REQ_INFO ),
                    MAX_DSP_BUFF_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // processing the reply
    // ---------------------
    if ( LcPResp->RespHeader.rhCptr == SUCCESS )
    {
        *PmPBuffNum = LcPResp->dbReservedBuffers;

        LcPAddDesc = (LPADD_DESC_INFO)(LcPResp + 1);
        for ( i = 0 ; i < LcPResp->dbReservedBuffers ; i++, LcPAddDesc++ )
        {
            PmAddrTab[i].adAddr1 = LcPAddDesc->adAddr1 ;
            PmAddrTab[i].adAddr2 = LcPAddDesc->adAddr2 ;
            PmAddrTab[i].adAddr3 = LcPAddDesc->adAddr3 ;
        }
    }

    // returning the error code
    return( LcPResp->RespHeader.rhCptr );
}

// ************************************************************************* //
//                                                                           //
//  API EXPORTED FUNCTIONS : GENERAL COMMAND FAMILY                          //
//                                                                           //
// ************************************************************************* //
// ****************************************************************************
// WORD PCXRegisterW()
// WORD PCXRegisterA()
// WORD PCXRegister()
// ********************
//
// Input parameters :
// ****************
//
// PAPP_DECL_INFO PmAppDeclInfo: point to a structure PAPP_DECL_INFO.
//
// Output parameters :
// *******************
//
// PPCX_HANDLE PmMyProcess: points to a BYTE wich will contains the handle value
//                          returned by the driver.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This is the first function to be called by an application that want to use
// the driver
//
// ****************************************************************************
#ifdef WIN32
WORD _CDECL_ PCXRegisterW(
    IN  PAPP_DECL_INFO_W    PmAppDeclInfo,
    OUT PPCX_HANDLE         PmMyProcess  )
{
    WORD LcCr;
    int  i=0;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This is the first call to the api: we going to put the interupt vector
    // number of the driver in a global static variable.
    // ----------------------------------------------------------------------
    if ( (LcCr = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcCr;
    }

    // call to Registration
    LcCr = Registration(  (PAPP_DECL_INFO_A)PmAppDeclInfo,
                          REGISTER_CMD_CDE,
                          FALSE,
                          PmMyProcess );

    if ( !(LcCr & ERROR_MASK) )
    {
        // Allocate an entry for async commands
        // ------------------------------------
        if ( !AllocWaitErrorCmdEntry( *PmMyProcess ) )
        {
            COMReleaseExmut();
// !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
            PCXUnRegister( *PmMyProcess );
            return EA_ALLOCATE_ASYNC_IMPOSSIBLE;
        }

        // Find a empty entry and fill it
        while( (i<MAX_PCX_HANDLE) && ( RegisterTab[i] != 0 ) ) i++;
           if ( i != MAX_PCX_HANDLE )
               RegisterTab[i] = *PmMyProcess ;
    }

    COMReleaseExmut();
    return LcCr;
}
#endif
WORD _CDECL_ PCXRegisterA(
    IN  PAPP_DECL_INFO_A    PmAppDeclInfo,
    OUT PPCX_HANDLE         PmMyProcess  )
{
    WORD LcCr;
    int  i=0;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This is the first call to the api: we going to put the interupt vector
    // number of the driver in a global static variable.
    // ----------------------------------------------------------------------
    if ( (LcCr = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcCr;
    }

    // call to Registration
    LcCr = Registration(  (PAPP_DECL_INFO_A)PmAppDeclInfo,
                          REGISTER_CMD_CDE,
                          TRUE,
                          PmMyProcess );

    if ( !(LcCr & ERROR_MASK) )
    {
        // Allocate an entry for async commands
        // ------------------------------------
        if ( !AllocWaitErrorCmdEntry( *PmMyProcess ) )
        {
            COMReleaseExmut();
// !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
            PCXUnRegister( *PmMyProcess );
            return EA_ALLOCATE_ASYNC_IMPOSSIBLE;
        }

        // Find a empty entry and fill it
        while( (i<MAX_PCX_HANDLE) && ( RegisterTab[i] != 0 ) ) i++;
           if ( i != MAX_PCX_HANDLE )
               RegisterTab[i] = *PmMyProcess ;
   }

    COMReleaseExmut();
    return LcCr;
}

// ****************************************************************************
// WORD PCXUnRegister()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: application handle.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This is the last function to be called by an application that used the
// driver.
//
// ****************************************************************************
// The function call the local function Registration with the application
// handle.

WORD _CDECL_ PCXUnRegister( IN PCX_HANDLE PmMyProcess )
{
    WORD LcRet;
    int  i=0;

    // ## FS (10/12/97) -- bug fix for NULL Unregister
    if ( PmMyProcess == 0 ) return EA_INVALID_PCX_HANDLE;

    // Before calling this actual Unregister function
    // of the driver, we shall unsynchronize and release
    // the potential pipes that remain allocated
    // -------------------------------------------------
    ReleaseSynchronizedPipes( PmMyProcess );

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // call to Registration
    LcRet =  Registration( NULL, PmMyProcess, FALSE, NULL );
    if ( LcRet == SUCCESS )
    {
        FreeAllWaitCmdEntry( PmMyProcess );
            // Find the corresponding handle and remove it.
        while( (i<MAX_PCX_HANDLE) && ( RegisterTab[i] != PmMyProcess ) ) i++;
        if ( i != MAX_PCX_HANDLE )
            RegisterTab[i] = 0 ;
    }

    COMReleaseExmut();
    return( LcRet );
}



/**
*
*	@brief	dispatcher init function supported for first-step 
*			compatibility with dispatcher-based applications.
*	@return SUCCESS, or WA_COMMAND_NOT_AVAILABLE if new api mode is wanted. 
*
*/
WORD _CDECL_ PCXInitialize( IN  PINIT_INFO PmInitInfo )
{
	 
	if ( PmInitInfo == NULL )  return SUCCESS;

	if (	(( PmInitInfo->iiFlags &= ~LX_FLAG ) != 0 )
		 || (  PmInitInfo->iiMode == DISP_NEW_MODE) )
		 return WA_COMMAND_NOT_AVAILABLE;

    return SUCCESS ;

} // PCXInitialize


/**
*
*	@brief	dispatcher uninit function supported for first-step 
*			compatibility with dispatcher-based applications.
*	@return SUCCESS
*
*/
WORD _CDECL_ PCXUnInitialize( IN  PUNINIT_INFO PmUnInitInfo )
{
	return SUCCESS;

} // PCXUnInitialize



// ****************************************************************************
// WORD PCXGetSystemFeature()
// ***************************
//
// Output parameters :
// *******************
//
// PSYSTEM_FEATURES_INFO PmSystemInfo : Description of system features of the driver
//
// Return value :
// **************
//          SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Returns to the application the system characteristics of the driver.
//
// ****************************************************************************
WORD _CDECL_ PCXGetSystemFeature(
    OUT PSYSTEM_FEATURES_INFO PmSystemInfo )
{
    WORD                LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }
    else
    {
        LcRet = GetSystemFeature( PmSystemInfo );
    }

    COMReleaseExmut();
    return( LcRet );
}

// ****************************************************************************
// WORD PCXGetHardwareBoardFeature()
// *********************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:    the mask of boards in the scope of the request
//
// Output parameters :
// *******************
//
// PCARD_FEATURES_INFO   PmCardInfo: Description of board features
//
// Return value :
// **************
//          SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Returns to the application the hardware characteristics of boards
// managed by the driver within the requested set
//
// ****************************************************************************
WORD _CDECL_ PCXGetHardwareBoardFeature(
    IN WORD                 PmBoardMask,
    OUT PCARD_FEATURES_INFO PmCardInfo )
{
    LPBOARD_HARD_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_HARD_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD                    LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_HARD_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_HARD_REQ_INFO ),
                    sizeof( BOARD_HARD_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bhrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcRet = LcPReply->RespHeader.rhCptr;

    if ( !( LcRet & ERROR_MASK) )
    {
        // ## FS (09/06/1997) - make sure the application will not read
        //                      rubbish data
        // ------------------------------------------------------------
        MEMSET2( PmCardInfo, 0, CARD_FEATURES_INFO, 1 );

        PmCardInfo->ciBoardType = LcPReply->bhfBoardType;
        PmCardInfo->ciExtendedBoardType = LcPReply->bhfExtendedType;

        if ( LcPReply->bhfBoardType != PCX_NO_PRESENT )
        {
            // copy the other fields as well
            // ------------------------------
            PmCardInfo->ciBusType       = LcPReply->bhfBus;
            PmCardInfo->ciDigitalSynchroInNumber = LcPReply->bhfSn;
            PmCardInfo->ciNbWordClock   = LcPReply->bhfNbWcl;
            PmCardInfo->ciMiscFeatures  = LcPReply->bhfMiscFeatures;
            PmCardInfo->ciDspNumber     = LcPReply->bhfDspNum;
            PmCardInfo->ciBaseAddress1  = LcPReply->bhfIO[0].ioStart;
            PmCardInfo->ciEndAddress1   = LcPReply->bhfIO[0].ioEnd;
            PmCardInfo->ciBaseAddress2  = LcPReply->bhfIO[1].ioStart;
            PmCardInfo->ciEndAddress2   = LcPReply->bhfIO[1].ioEnd;
            PmCardInfo->ciEtherSound0   = LcPReply->bhfEtherSound0;
            PmCardInfo->ciEtherSound1   = LcPReply->bhfEtherSound1;
            PmCardInfo->ciIrqNumber     = LcPReply->bhfIrqNum;
            PmCardInfo->ciMiscFeatures2 = LcPReply->bhfMiscFeatures2;

            memcpy( PmCardInfo->ciMACAddress,
                    LcPReply->bhfMACAddress,
                    sizeof(PmCardInfo->ciMACAddress) );

            memcpy( PmCardInfo->ciPhysicalOutFeature,
                    LcPReply->bhfOutAudio,
                    MAX_STEREO_OUTAUDIO_BOARD );

            memcpy( PmCardInfo->ciPhysicalOutFeature2,
                    LcPReply->bhfOutAudio2,
                    MAX_STEREO_OUTAUDIO_BOARD);

            memcpy( PmCardInfo->ciPhysicalInFeature,
                    LcPReply->bhfInAudio,
                    MAX_STEREO_INAUDIO_BOARD );

            memcpy( PmCardInfo->ciExtraPhysicalInFeature,
                    LcPReply->bhfExtraInAudio,
                    MAX_STEREO_INAUDIO_BOARD );

            memcpy( PmCardInfo->ciPhysicalInFeature2,
                    LcPReply->bhfInAudio2,
                    MAX_STEREO_INAUDIO_BOARD);

			PmCardInfo->ciLineConnections = 0;

            // analog outputs
            //
            if ( LcPReply->bhfOutAudio[0] & AUDIO_ANALOG_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= ANALOG_OUTPUTS_BALANCED;
            }
            // analog inputs
            //
            if ( LcPReply->bhfInAudio[0] & AUDIO_ANALOG_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= ANALOG_INPUTS_BALANCED;
            }
            // digital outputs
            //
            if ( LcPReply->bhfOutAudio[0] & AUDIO_DIGITAL_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= DIGITAL_OUTPUTS_BALANCED;
            }
            // digital inputs
            //
            if ( LcPReply->bhfInAudio[0] & AUDIO_DIGITAL_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= DIGITAL_INPUTS_BALANCED;
            }

            if ( LcPReply->bhfOutAudio2[0] & AUDIO_ETHERSOUND_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= ETHERSOUND_OUTPUTS_BALANCED;
            }

            if ( LcPReply->bhfInAudio2[0] & AUDIO_ETHERSOUND_PRESENT_MASK )
            {
                PmCardInfo->ciLineConnections |= ETHERSOUND_INPUTS_BALANCED;
            }

        }  // end of if there is such a board

    } // end of if there was no error

    COMReleaseExmut();
    return( LcRet );

}



WORD _CDECL_ PCXGetAudioFeatures(    
	PAUDIO_DECL_INFO 			IN		PmPAudioDeclInfo,
	PAUDIO_FEATURES_INFO		INOUT	PmPAudioFeaturesInfo)
{
    LPBOARD_HARD_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_HARD_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD                    LcRet;
	BOOL					LcCapture = FALSE;
	BYTE					LcStereoIndex = 0;
	BYTE					LcExtra = 0;
	BYTE					LcPhy	= 0;
	BYTE					LcPhy2	= 0;

	if( PmPAudioDeclInfo == NULL )
        return EA_INVALID_POINTER;

	PmPAudioFeaturesInfo->afiLevelFeatures = 0;

    if ( PmPAudioDeclInfo->adBoardNumber >= MAX_BOARD)  
        return ED_INVALID_BOARD;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }


	// 
	if (PmPAudioDeclInfo->adAudioAttributes & AUDIO_IN) LcCapture = TRUE;
	LcStereoIndex = PmPAudioDeclInfo->adAudioNumber / 2;



    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_HARD_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_HARD_REQ_INFO ),
                    sizeof( BOARD_HARD_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bhrBoardMask = (1 << PmPAudioDeclInfo->adBoardNumber); 

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcRet = LcPReply->RespHeader.rhCptr;

    if ( !( LcRet & ERROR_MASK) )
    {
        if ( LcPReply->bhfBoardType != PCX_NO_PRESENT )
        {
			if (LcCapture)
			{
				LcPhy	= LcPReply->bhfOutAudio[LcStereoIndex];
				LcPhy2	= LcPReply->bhfOutAudio2[LcStereoIndex];
			}
			else
			{
				LcPhy	= LcPReply->bhfInAudio[LcStereoIndex];
				LcPhy2	= LcPReply->bhfInAudio2[LcStereoIndex];
				LcExtra	= LcPReply->bhfExtraInAudio[LcStereoIndex];
			}

            PmPAudioFeaturesInfo->afiPhysicalFeatures   = LcPhy;
            PmPAudioFeaturesInfo->afiPhysicalFeatures2  = LcPhy2;
            PmPAudioFeaturesInfo->afiExtraFeatures      = LcExtra;

			// level features
			//
			if ((LcPReply->bhfBoardType > LXES_LAST) || (LcPReply->bhfBoardType < LXES_FIRST))
			{
				// usual digital gain and meters features
				//
				PmPAudioFeaturesInfo->afiLevelFeatures |= AFI_GAIN_DIGITAL_PIPE;
				PmPAudioFeaturesInfo->afiLevelFeatures |= AFI_VU_METERS;
			}

			PmPAudioFeaturesInfo->afiLevelFeatures |= AFI_PEAK_METERS;

        }  // end of if there is such a board

    } // end of if there was no error

    COMReleaseExmut();
    return( LcRet );

}


// ****************************************************************************
// WORD PCXBoardNameStringW()
// WORD PCXBoardNameStringA()
// WORD PCXBoardNameString()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:        the mask of boards in the scope of the request
//
// Output parameters :
// *******************
//
//  LPTSTR   PmBoardTypeName:    the name of the board type
//
// ****************************************************************************
//
// Function to get back a description string for a board type
//   and a description type for the board variant if
//   several flavors exist for the board type
//
// ****************************************************************************
WORD GenericBoardNameString(
    IN    WORD      PmBoardMask,
    IN    WORD      PmBoardNameMaxSize,
    INOUT LPTSTR    PmBoardNameString )
{
    LPBOARD_HARD_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_HARD_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD                    LcError;

    // The input string must hold at least one character !!
    //
    if ( (!PmBoardNameMaxSize) || (!PmBoardNameString) ) return WA_MORE_DATA ;

    if (!COMWaitExmut()) return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcError = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcError;
    }

    // Send a board Hardware features request to the driver
    //
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_HARD_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_HARD_REQ_INFO ),
                    sizeof( BOARD_HARD_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bhrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcError = LcPReply->RespHeader.rhCptr;

    if ( !( LcError & ERROR_MASK) )
    {
        BYTE                        LcFlavorNum;
        PASSOC_BOARD_FLAVOR_INFO    LcBoardFlavorInfo;
        LPTSTR                      LcString;
        size_t                      LcLength;

        // Initialise default reply
        //
        LcString = _T("") ;
        LcLength = 0;

        // Extract Flavor number From CSID
        //
        LcFlavorNum = (BYTE) EXTRACT_FLAVOR_NUM( LcPReply->bhfCSID );

        if ( LookupBoardFlavor( LcPReply->bhfBoardType,
                                LcPReply->bhfExtendedType,
                                LcFlavorNum,
                                &LcBoardFlavorInfo ) )
        {
            LcString = LcBoardFlavorInfo->abFlavorName;
            LcLength = _tcslen( LcString );
        }

        _tcsncpy( PmBoardNameString, LcString, PmBoardNameMaxSize - 1 );
        PmBoardNameString[PmBoardNameMaxSize - 1] = _T('\0');

        if ( PmBoardNameMaxSize < LcLength )
        {
            LcError = WA_MORE_DATA ;
        }

    } // end of if there was no error

    COMReleaseExmut();
    return( LcError );
}
#ifdef WIN32
WORD _CDECL_ PCXBoardNameStringW(
    IN    WORD      PmBoardMask,
    IN    WORD      PmBoardNameMaxSize,
    INOUT LPWSTR    PmBoardNameString )
{
    LPTSTR  LcGenericString ;
    WORD    LcError ;
    size_t  LcLength ;

    // The input string must hold at least one character !!
    //
    if ( (!PmBoardNameMaxSize) || (!PmBoardNameString) ) return WA_MORE_DATA ;

    // Let's call the UNICODE equivalent
    //
    LcGenericString = (LPTSTR) MALLOC2( _TCHAR, PmBoardNameMaxSize ) ;

    if ( LcGenericString )
    {
        LcError = GenericBoardNameString( PmBoardMask,
                                          PmBoardNameMaxSize,
                                          LcGenericString ) ;
        if ( !( LcError & ERROR_MASK ) )
        {
#if defined(UNICODE) || defined(_UNICODE)
            // Provide Unicode 2 ANSI Conversion if necessary
            //
            wcsncpy( PmBoardNameString, LcGenericString, PmBoardNameMaxSize );
            LcLength = wcslen( PmBoardNameString );
#else
            LcLength = _snwprintf( PmBoardNameString, PmBoardNameMaxSize, L"%hs", LcGenericString );
#endif
            PmBoardNameString[PmBoardNameMaxSize - 1] = _T('\0');

            // Check that conversion do not cut the string
            //
            if ( LcLength >= PmBoardNameMaxSize )
            {
                // Do not overwrite a previous error
                //
                if ( LcError == SUCCESS ) LcError = WA_MORE_DATA ;
            }
        }

        FREE(LcGenericString);
    }
    else
    {
        LcError = EA_ALLOCATE_MEMORY_IMPOSSIBLE ;
    }

    return( LcError );
}
#endif
WORD _CDECL_ PCXBoardNameStringA(
    IN    WORD      PmBoardMask,
    IN    WORD      PmBoardNameMaxSize,
    INOUT LPSTR     PmBoardNameString )
{
    LPTSTR  LcGenericString ;
    WORD    LcError ;
    DWORD   LcLength ;

    // The input string must hold at least one character !!
    //
    if ( (!PmBoardNameMaxSize) || (!PmBoardNameString) ) return WA_MORE_DATA ;

    // Let's call the UNICODE equivalent
    //
    LcGenericString = (LPTSTR) MALLOC2( _TCHAR, PmBoardNameMaxSize ) ;

    if ( LcGenericString )
    {
        LcError = GenericBoardNameString( PmBoardMask,
                                          PmBoardNameMaxSize,
                                          LcGenericString ) ;
        if ( !( LcError & ERROR_MASK ) )
        {
#if defined(UNICODE) || defined(_UNICODE)
            // Provide Unicode 2 ANSI Conversion if necessary
            //
            LcLength = _snprintf( PmBoardNameString, PmBoardNameMaxSize, "%ls", LcGenericString );
#else
            strncpy( PmBoardNameString, LcGenericString, PmBoardNameMaxSize );
            LcLength = strlen( PmBoardNameString );
#endif
            PmBoardNameString[PmBoardNameMaxSize - 1] = _T('\0');

            // Check that conversion do not cut the string
            //
            if ( LcLength >= PmBoardNameMaxSize )
            {
                // Do not overwrite a previous error
                //
                if ( LcError == SUCCESS ) LcError = WA_MORE_DATA ;
            }
        }

        FREE( LcGenericString );
    }
    else
    {
        LcError = EA_ALLOCATE_MEMORY_IMPOSSIBLE ;
    }

    return( LcError );
}

// ****************************************************************************
// WORD PCXGetSoftwareBoardFeature()
// *********************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:    the mask of boards in the scope of the request
//  WORD    PmMaxDspInfo:   the maximum number of Dsp that should be described
//
// Output parameters :
// *******************
//
// LPWORD    PmDspInfoNumber:        Number of Dsp actually described
// PDSP_FEATURES_INFO PmDspInfo:    Description of each dsp
//
// Return value :
// **************
//          SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Returns to the application the software capabilities of a DSP
//
// ****************************************************************************
WORD _CDECL_ PCXGetSoftwareBoardFeature(
    IN WORD                 PmBoardMask,
    IN WORD                 PmMaxDspInfo,
    OUT LPWORD              PmDspInfoNumber,
    OUT PDSP_FEATURES_INFO  PmDspInfo )
{
    LPBOARD_SOFT_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_SOFT_RESP_INFO  LcPReply = 0;     // will point to the reply
    LPDRV_DSP_INFO          LcPDspInfo;
    LPDSP_SOFT_INFO         LcPIoDesc;
    WORD                    LcDspNum;
    WORD                    LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_SOFT_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_SOFT_REQ_INFO ),
                    MAX_BOARD_SOFT_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bsrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcRet = LcPReply->RespHeader.rhCptr;
    if ( !( LcRet & ERROR_MASK) )
    {
        // ## FS (09/06/1997) - make sure the application will not read
        //                      rubbish data
        // ------------------------------------------------------------
        MEMSET2( PmDspInfo, 0, DSP_FEATURES_INFO, PmMaxDspInfo );

        *PmDspInfoNumber = LcPReply->bsfDspNum;

        // Set position to the 1st Dsp description
        // ----------------------------------------
        LcDspNum = 0;
        LcPDspInfo = (LPDRV_DSP_INFO)(LcPReply + 1);

        // Iterate till we reach the max of Dsp present on board and
        // requested Dsp number
        // ----------------------------------------------------------
        while (  (LcDspNum < LcPReply->bsfDspNum)
               && (LcDspNum < PmMaxDspInfo) )
        {
            PmDspInfo[LcDspNum].dfDspState        = LcPDspInfo->ddiDspState;
            PmDspInfo[LcDspNum].dfNbAppli         = LcPDspInfo->ddiAppNum;

            // support of DHS applications
            PmDspInfo[LcDspNum].dfFeaturesEx      = DHS_PRESENT_MASK;

            // if DSP gets a software loaded, copy its capabilities
            // -----------------------------------------------------
            if ( LcPDspInfo->ddiDspState != 0 )
            {
                // The DRV_DSP_INFO structure is followed by a DSP_SOFT_INFO
                // structure
                // ----------------------------------------------------------
                LcPIoDesc = (LPDSP_SOFT_INFO)(LcPDspInfo + 1);
                LcPDspInfo = (LPDRV_DSP_INFO)(LcPIoDesc + 1);

                PmDspInfo[LcDspNum].dfManagedPhysicalOutMask    = 
                    (LcPIoDesc->dsiManagedPhysOut >= 32) ? 0xffffffff : (((DWORD)1 << LcPIoDesc->dsiManagedPhysOut)-1);
                PmDspInfo[LcDspNum].dfManagedVirtualOutMask     =
                                                0; //LcPIoDesc->dsiManagedVirtOut;
                PmDspInfo[LcDspNum].dfManagedPhysicalInMask =
                    (LcPIoDesc->dsiManagedPhysIn >= 32) ? 0xffffffff : (((DWORD)1 << LcPIoDesc->dsiManagedPhysIn)-1);
                PmDspInfo[LcDspNum].dfManagedVirtualInMask =
                                                0; //LcPIoDesc->dsiManagedVirtIn;

				// New fields
				PmDspInfo[LcDspNum].dfManagedPhysicalOutputFirst	= 0;
    			PmDspInfo[LcDspNum].dfManagedPhysicalOutputNumber	= LcPIoDesc->dsiManagedPhysOut;
				PmDspInfo[LcDspNum].dfManagedPhysicalInputFirst		= 0;
    			PmDspInfo[LcDspNum].dfManagedPhysicalInputNumber	= LcPIoDesc->dsiManagedPhysIn;

                // ATTENTION : informations perdue 64 bit mask -> 32 bit mask
                PmDspInfo[LcDspNum].dfPhysicalOutMask = (DWORD)LcPIoDesc->dsiPhysOut;
                PmDspInfo[LcDspNum].dfVirtualOutMask = (DWORD)LcPIoDesc->dsiVirtOut;
                PmDspInfo[LcDspNum].dfPhysicalInMask = (DWORD)LcPIoDesc->dsiPhysIn;
                PmDspInfo[LcDspNum].dfVirtualInMask = (DWORD)LcPIoDesc->dsiVirtIn;

                PmDspInfo[LcDspNum].dfMaxBoardOutStream = LcPIoDesc->dsiMaxBoardOutStream;
                PmDspInfo[LcDspNum].dfMaxPipeOutStream = LcPIoDesc->dsiMaxPipeOutStream;
                PmDspInfo[LcDspNum].dfFirmwareVersion = LcPIoDesc->dsiFirmwareVersion;
            }
            else
            {
                LcPDspInfo++ ;
            }

            LcDspNum++;
        }
    } // end of if there was no error

    COMReleaseExmut();
    return( LcRet );
}

// ****************************************************************************
// WORD PCXGetBoardExternalClocks()
// ********************************
//
// Input parameters :
// ****************
//
// WORD PmBoardMask: The mask for the requested board
//
// WORD PmMaxExternalClockInfo: the maximum number of entries returned
//
// Output parameters :
// *******************
//
// OUT PCLOCK_INFO PmExternalClockInfo: points onto a table of CLOCK_INFO entries
//
// Return value :
// **************
// 0 if no error occured, a warning if PmMaxExternalClockInfo was to short to allow
// a full description of all external clocks, an error code otherwise.
//
// ****************************************************************************
//
// This function returns information (if available) about all the external clocks
//
// ****************************************************************************
WORD _CDECL_ PCXGetBoardExternalClocks(
    IN  WORD PmBoardMask,
    IN  WORD PmMaxExternalClockInfo,
    OUT PCLOCK_INFO PmExternalClockInfo)
{
    LPGET_BOARD_CLOCK_REQ_INFO  LcPRequest = 0;   // will point to the request
    LPGET_BOARD_CLOCK_RESP_INFO LcPReply = 0;     // will point to the reply
    WORD                        LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_CLOCK_CMD,
                    (APP_HANDLE) 0,
                    sizeof( GET_BOARD_CLOCK_REQ_INFO ),
                    sizeof( GET_BOARD_CLOCK_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bcrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // make sure the application will not read rubbish data
    // -----------------------------------------------------
    MEMSET2( PmExternalClockInfo, 0, CLOCK_INFO, PmMaxExternalClockInfo );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcRet = LcPReply->RespHeader.rhCptr;
    if ( LcRet == SUCCESS )
    {
      if( LcPReply->bcpClock[BOARD_CLOCK_ETHERSOUND].bcType == CLOCK_TYPE_ETHERSOUND )
      {
        if ( PmMaxExternalClockInfo > ETHERSOUND_CLOCK_INDEX )
        {
            PmExternalClockInfo[ETHERSOUND_CLOCK_INDEX].ckType = CLOCK_TYPE_ETHERSOUND;
            PmExternalClockInfo[ETHERSOUND_CLOCK_INDEX].ckSync = LcPReply->bcpClock[BOARD_CLOCK_ETHERSOUND].bcSync;
            PmExternalClockInfo[ETHERSOUND_CLOCK_INDEX].ckFrequency = LcPReply->bcpClock[BOARD_CLOCK_ETHERSOUND].bcFrequency;
        }
        else
        {
            LcRet = WA_MORE_DATA ;
        }
      }

      if( LcPReply->bcpClock[BOARD_CLOCK_VIDEO].bcType == CLOCK_TYPE_VIDEO )
      {
        if ( PmMaxExternalClockInfo > VIDEO_CLOCK_INDEX )
        {
            PmExternalClockInfo[VIDEO_CLOCK_INDEX].ckType = CLOCK_TYPE_VIDEO;
            PmExternalClockInfo[VIDEO_CLOCK_INDEX].ckSync = LcPReply->bcpClock[BOARD_CLOCK_VIDEO].bcSync;
            PmExternalClockInfo[VIDEO_CLOCK_INDEX].ckFrequency = LcPReply->bcpClock[BOARD_CLOCK_VIDEO].bcFrequency;
        }
        else
        {
            LcRet = WA_MORE_DATA ;
        }
      }

      if( LcPReply->bcpClock[BOARD_CLOCK_WCL].bcType == CLOCK_TYPE_WORD_CLOCK )
      {
        if ( PmMaxExternalClockInfo > WCL_CLOCK_INDEX )
        {
            PmExternalClockInfo[WCL_CLOCK_INDEX].ckType = CLOCK_TYPE_WORD_CLOCK;
            PmExternalClockInfo[WCL_CLOCK_INDEX].ckSync = LcPReply->bcpClock[BOARD_CLOCK_WCL].bcSync;
            PmExternalClockInfo[WCL_CLOCK_INDEX].ckFrequency = LcPReply->bcpClock[BOARD_CLOCK_WCL].bcFrequency;
        }
        else
        {
            LcRet = WA_MORE_DATA ;
        }
      }

      if( LcPReply->bcpClock[BOARD_CLOCK_UER].bcType == CLOCK_TYPE_UER_SYNCHRO )
      {
        if ( PmMaxExternalClockInfo > UER_CLOCK_INDEX )
        {
            PmExternalClockInfo[UER_CLOCK_INDEX].ckType = CLOCK_TYPE_UER_SYNCHRO;
            PmExternalClockInfo[UER_CLOCK_INDEX].ckSync = LcPReply->bcpClock[BOARD_CLOCK_UER].bcSync;
            PmExternalClockInfo[UER_CLOCK_INDEX].ckFrequency = LcPReply->bcpClock[BOARD_CLOCK_UER].bcFrequency;
        }
        else
        {
            LcRet = WA_MORE_DATA ;
        }
      }

    } // end of if there was no error

    COMReleaseExmut();
    return( LcRet );
}

// ****************************************************************************
// WORD PCXGetBoardExternalUerClocks()
// **********************************
//
// Input parameters :
// ****************
//
// WORD PmBoardMask: The mask for the requested board
//
// WORD PmMaxUerClockInfo: the maximum number of entries returned
//
// Output parameters :
// *******************
//
// OUT PCLOCK_INFO PmUerClockInfo: points onto a table of CLOCK_INFO entries
//
// Return value :
// **************
// 0 if no error occured, a warning if PmMaxExternalClockInfo was to short to allow
// a full description of all external clocks, an error code otherwise.
//
// ****************************************************************************
//
// This function returns information (if available) about all the external clocks
//
// ****************************************************************************
WORD _CDECL_ PCXGetBoardExternalUerClocks(
    IN  WORD PmBoardMask,
    IN  WORD PmMaxUerClockInfo,
    OUT PCLOCK_INFO PmUerClockInfo)
{
    LPGET_BOARD_CLOCK_REQ_INFO  LcPRequest = 0;   // will point to the request
    LPGET_BOARD_CLOCK_RESP_INFO LcPReply = 0;     // will point to the reply
    WORD                        LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_CLOCK_CMD,
                    (APP_HANDLE) 0,
                    sizeof( GET_BOARD_CLOCK_REQ_INFO ),
                    sizeof( GET_BOARD_CLOCK_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bcrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // make sure the application will not read rubbish data
    // -----------------------------------------------------
    MEMSET2( PmUerClockInfo, 0, CLOCK_INFO, PmMaxUerClockInfo );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcRet = LcPReply->RespHeader.rhCptr;
    if ( LcRet == SUCCESS )
    {
        WORD    LcUerIndex ;
        WORD    LcUerUserIndex ;

        LcUerUserIndex = 0;

        // Read all clock inputs described for the board
        //
        for ( LcUerIndex = 0 ; LcUerIndex < LcPReply->bcpNbClockInputs ; LcUerIndex++ )
        {
            // Let's skip the word clock sync input that is not relevant
            // here. Note that we must leave LcUerUserIndex as is
            //
            if( LcPReply->bcpClock[LcUerIndex].bcType != CLOCK_TYPE_UER_SYNCHRO )
				continue;

            if(  LcUerUserIndex < PmMaxUerClockInfo )
			{
				PmUerClockInfo[LcUerUserIndex].ckType = CLOCK_TYPE_UER_SYNCHRO;
				PmUerClockInfo[LcUerUserIndex].ckSync = LcPReply->bcpClock[LcUerIndex].bcSync;
				PmUerClockInfo[LcUerUserIndex].ckFrequency = LcPReply->bcpClock[LcUerIndex].bcFrequency;
			}
			else
			{
				LcRet = WA_MORE_DATA ;
				break;
			}

            LcUerUserIndex++;
        }

    } // end of if there was no error

    COMReleaseExmut();
    return( LcRet );
}

#ifdef WIN32
// ****************************************************************************
// WORD PCXGetPublicApplicationW()
// WORD PCXGetPublicApplicationA()
// WORD PCXGetPublicApplication()
// ***************************
//
// Input parameters :
// ****************
//
// WORD PmMaxAppInfo: Maximum application structures allocated.
//
// Output parameters :
// *******************
//
// OUT LPBYTE PmAppsNumber: Points to a byte that contains in return the
// number of applications registered as public.
//
// OUT PAPP_INFO PmAppInfo[MAX_APPS_NUMBER]: Array of pointers on APP_INFO
// structures that will contain informations on all the applications registered
// as public.
//
// Return value :
// **************
// 0 if no error occured, a warning if PmMaxAppInfo was to short to allow
// a full description of all applications, an error code otherwise.
//
// ****************************************************************************
//
// This function returns informations concerning all the
// applications registered as public.
//
// ****************************************************************************
WORD _CDECL_ PCXGetPublicApplicationW(
    IN WORD PmMaxAppInfo,
    OUT LPBYTE PmAppsNumber,
    OUT PAPP_INFO_W PmAppInfo )
{
    LPAPP_LIST_REQ_INFO  LcPReq = 0;   // will point to the beginning of the
                                        // request block
    LPAPP_LIST_RESP_INFO LcPResp = 0;   // will point to the beginning of the
                                        // response block
    LPAPP_DESC           LcPAppDesc;    // will point to the application
                                        // description structures
                                        // within the response block
    WORD                i, LcRet;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    APP_LIST_CMD,
                    (APP_HANDLE) 0,
                    sizeof( APP_LIST_REQ_INFO ),
                    MAX_APP_LIST_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcRet = LcPResp->RespHeader.rhCptr;
    if ( !( LcRet & ERROR_MASK) )
    {
        // reading the application number
        *PmAppsNumber = LcPResp->alpNbHandle;

        if ( PmAppInfo && PmMaxAppInfo )
        {
            // ## FS (09/06/1997) - make sure the application will not read
            //                      rubbish data
            // ------------------------------------------------------------
            MEMSET2( PmAppInfo, 0, APP_INFO_W, PmMaxAppInfo );

            // filling each application descriptions
            // -------------------------------------
            for ( i = 0,
                      LcPAppDesc = (LPAPP_DESC)(LcPResp + 1);
                  (i < LcPResp->alpNbHandle) && (i < PmMaxAppInfo);
                  i ++,
                      LcPAppDesc++ )
            {
                PmAppInfo[i].aiProcess = LcPAppDesc->adHandle;

                // Handle the string according to its original type
                // ------------------------------------------------
                if ( (LcPAppDesc->adAppName).asStringType == APP_UNICODE_STRING )
                {
                    wcscpy(PmAppInfo[i].aiAppName, (LcPAppDesc->adAppName).asUnicodeString);
                }
                else
                {
                    _snwprintf(PmAppInfo[i].aiAppName,SIZE_MAX_APP_NAME+1,L"%hs",(LcPAppDesc->adAppName).asAnsiString);
                }

                // TL 19/06/97 Add the two next lines
                PmAppInfo[i].aiWinHandle  = (HANDLE)LcPAppDesc->adWindowHandle;
                PmAppInfo[i].aiWinMessage = LcPAppDesc->adWindowMessage;
            }

            // Returns a warning if more data is availabe but cannot be copied
            // to PmAppInfo
            if ( PmMaxAppInfo < LcPResp->alpNbHandle )
            {
                COMReleaseExmut();
                return WA_MORE_DATA;
            }
        }
    }

    // returning the error code
    COMReleaseExmut();
    return LcRet;
}
WORD _CDECL_ PCXGetPublicApplicationA(
    IN  WORD        PmMaxAppInfo,
    OUT LPBYTE      PmAppsNumberPtr,
    OUT PAPP_INFO_A PmAppInfo )
{
    PAPP_INFO_W     LcAppInfo = 0 ;
    WORD            LcError ;
    WORD            i ;

    if ( PmAppInfo && PmMaxAppInfo )
    {
        // make sure the application will not read rubbish data
        // ------------------------------------------------------------
        MEMSET2( PmAppInfo, 0, APP_INFO_A, PmMaxAppInfo );

        // If ever the application works with more than the max
        // number of application, let's cut it down
        //
        if ( PmMaxAppInfo > MAX_PCX_HANDLE ) PmMaxAppInfo = MAX_PCX_HANDLE;
        LcAppInfo = MALLOC2( APP_INFO_W, PmMaxAppInfo );

        if ( !LcAppInfo ) return EA_ALLOCATE_MEMORY_IMPOSSIBLE ;
    }

    LcError = PCXGetPublicApplicationW(PmMaxAppInfo,PmAppsNumberPtr,LcAppInfo);

    if ( PmAppInfo && PmMaxAppInfo )
    {
        if ( !(LcError & ERROR_MASK) )
        {
            // filling each application descriptions
            //
            for ( i = 0; (i < (*PmAppsNumberPtr)) && (i < PmMaxAppInfo); i ++ )
            {
                PmAppInfo[i].aiProcess      = LcAppInfo[i].aiProcess;
                PmAppInfo[i].aiWinHandle    = LcAppInfo[i].aiWinHandle;
                PmAppInfo[i].aiWinMessage   = LcAppInfo[i].aiWinMessage;

                // Translate Unicode string to ANSI
                //
                sprintf( PmAppInfo[i].aiAppName, "%ls", LcAppInfo[i].aiAppName );
            }
        }
    }

    // ## FS (26/01/1999) -- FA #250 Memory leak may
    // cause the application to crash when malloc() will fail.
    //
    if ( LcAppInfo ) FREE( LcAppInfo );
    return( LcError );
}
#endif // WIN32

// ****************************************************************************
// WORD PCXGetPipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmRequestedProcess: Handle of the process on which pipes informations
//                          are required
// WORD PmMaxPipeInfo: maximum number of PIPE_INFO structure to fill.
//
// Output parameters :
// *******************
//
// LPBYTE PmOutPipesNumber: Points to a byte which will contain the number of
//                          output pipes of the process.
// LPBYTE PmInPipesNumber: Points to a byte which will contain the number of
//                         input pipes of the process.
// PPIPE_INFO PmPipeInfo: Array of pointers on PIPE_INFO
//                          structures that will contain informations on all
//                          the pipes of the process.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function returns the list of pipes (and there
// characteristics) belonging to the application which handle is
// given in parameter. This application must have been registered
// as public.
//
// ****************************************************************************

WORD _CDECL_ PCXGetPipe(
    IN PCX_HANDLE PmRequestedProcess,
    IN WORD PmMaxPipeInfo,
    OUT LPBYTE PmOutPipesNumber,
    OUT LPBYTE PmInPipesNumber,
    OUT PPIPE_INFO PmPipeInfo )
{
    LPPIPE_LIST_REQ_INFO LcPReq = 0;     // will point to the beginning of the
                                         // request block
    LPPIPE_LIST_RESP_INFO LcPResp = 0;   // will point to the beginning of the
                                         // response block
    LPPIPE_DESC_INFO      LcPAppDesc;    // will point to the pipe description
                                         // structures within the response block
    WORD                 i, LcRet;

    if( (PmRequestedProcess == 0) || (PmRequestedProcess > MAX_PCX_HANDLE) || (PmRequestedProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcRet = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    PIPES_LIST_CMD,
                    (APP_HANDLE) 0,
                    sizeof( PIPE_LIST_REQ_INFO ),
                    MAX_PIPE_LIST_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->plqHandle = (APP_HANDLE) PmRequestedProcess;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // processing the response
    LcRet = LcPResp->RespHeader.rhCptr;
    if ( !( LcRet & ERROR_MASK) )
    {
        // ## FS (09/06/1997) - make sure the application will not read
        //                      rubbish data
        // ------------------------------------------------------------
        MEMSET2( PmPipeInfo, 0, PIPE_INFO, PmMaxPipeInfo );

        *PmOutPipesNumber = LcPResp->plpNbOut;
        *PmInPipesNumber = LcPResp->plpNbIn;

        for( i = 0,
             LcPAppDesc = (LPPIPE_DESC_INFO)(LcPResp + 1);
             (  (i < (WORD)(LcPResp->plpNbOut + LcPResp->plpNbIn))
              && (i < PmMaxPipeInfo) );
             i ++, LcPAppDesc ++ )
        {
            PmPipeInfo[i].piPipeMask = LcPAppDesc->pdPipeMask64.LowPart;

            _ASSERT( LcPAppDesc->pdPipeMask64.HighPart == 0 );   // not handled yet

            PmPipeInfo[i].piStreamNumber = LcPAppDesc->pdMaxStream;
            PmPipeInfo[i].piAudioInOutNumber = LcPAppDesc->pdNbIO;
            PmPipeInfo[i].piPipeAttributes = LcPAppDesc->pdPipeAttrib;
        }

        // Returns a warning if more data is availabe but cannot be copied
        // to PmPipeInfo
        if ( PmMaxPipeInfo < (WORD)(LcPResp->plpNbOut+ LcPResp->plpNbIn) )
            LcRet = WA_MORE_DATA;
    }

    // returning the error code
    COMReleaseExmut();
    return LcRet;
}

// ## FM (09/25/97) -- Creation
// ****************************************************************************
// STATIC WORD OpenD56File()
// *************************
//
// Input parameters :
// ****************
//
//  IN  LPTSTR   PmPathName      : the path to the various driver NP files
//  IN  LPTSTR   PmShortFileName : the name of the .D56 file
//
// Output parameters :
// ******************
//
//  LPINT    PmFileDescPtr   :   the descriptor of the file
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
// This function handles .D56 files
// ****************************************************************************
STATIC WORD OpenD56File(
    IN  LPTSTR   PmPathName,
    IN  LPTSTR   PmShortFileName,
    OUT LPINT    PmFileDescPtr)
{
    INT         LcFile ;

    // We must build the path PmPathName\DSP_____PATH\PmShortFileName
    //
    _tcscat(PmPathName,DSP_____PATH)     ;
    _tcscat(PmPathName,PmShortFileName)  ;

    // Open the .D56 table file
    // --------------------------
    LcFile = _topen(PmPathName,O_RDONLY|O_BINARY) ;

    if ( LcFile < 0 )
    {
        return(EA_OPEN_FILE_IMPOSSIBLE) ;
    }

    *PmFileDescPtr = LcFile ;
    return(SUCCESS)         ;
}


// ****************************************************************************
// STATIC WORD DspDownload()
// ***************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess  :   Handle of the calling process
//  WORD PmBoardMask            :   specifies One board
//  WORD PmDspMask              :   specifies One Dsp to download
//  DWORD PmDspFeatures         :   specifies the DSP capabilities
//  WORD PmSoftwareIndex    :   Index in DSP_CAPABILITIES_INFO table
//
// Output parameters :
// *******************
//
//  none
//
// Return value :
// **************
//  SUCCESS if no error occured, a warning if filename is truncated,
//   an error code otherwise.
//
// ****************************************************************************
//
//  Handles the loading process of a DSP software
//
// ****************************************************************************
STATIC WORD DspDownload(
    IN PCX_HANDLE   PmMyProcess,
    IN WORD     PmBoardMask,
    IN WORD     PmDspMask,
    IN DWORD    PmDspFeatures,
    IN BYTE     PmSoftwareIndex,
    IN BYTE     PmFirmwareIndex )
{
    // FS: 21/11/96
    // New specifications for DspDownload: the buffers are filled
    // by the API layer and not by the application anymore
    // ------------------------------------------------------------
    WORD            LcReservedBuffers, LcBuffNum ;
    WORD            LcRet ;
    int             LcReadRet ;
    DWORD           LcReadSize ;
    DWORD           LcBuffSize ;
    LPBYTE          LcBuffAddr;
    STATIC ADD_DESC_INFO       LcAddrDescTab[MAX_DOWNLOAD_BUFFERS];
    STATIC BUFFER_TO_LOAD_INFO LcBufferToLoad[MAX_DOWNLOAD_BUFFERS];
    STATIC _TCHAR              LcDspFileName[MAX_SHORT_FILE_NAME_LEN];
    INT             LcFile  = -1 ;
    STATIC _TCHAR              LcPathName[MAX_DSP_FILE_NAME_LEN];

    if (!COMWaitExmut())
    {
        return EA_CANT_TAKE_MUTEX;
    }

    // First of all, get the file name of the DSP software requested
    // --------------------------------------------------------------
    LcRet = GENGetDspFileName(PmSoftwareIndex,LcPathName,LcDspFileName);

    if ( LcRet != SUCCESS )
    {
        goto clean_exit ;
    }

    // Then get the addresses of buffers reserved for downloading purposes
    // --------------------------------------------------------------------
    LcRet = GENGetDspBuffers( &LcReservedBuffers, LcAddrDescTab );
    if ( LcRet != SUCCESS )
    {
        goto clean_exit ;
    }

    // Open the DSP software file
    // --------------------------
    LcRet = OpenD56File(LcPathName,LcDspFileName,&LcFile) ;

    // Give up if the binary file cannot be opened
    // -------------------------------------------
    if ( LcRet != SUCCESS )
    {
        // Free the buffers before quit
        // ----------------------------
        LcBufferToLoad[0].btlDataLength = 0;

        // FS - 11/02/97
        // Let's keep the above error code
        // -------------------------------
        (VOID) ManageDsp( PmMyProcess,
              PmBoardMask,
              PmDspMask,
              PmDspFeatures,
              1,
              PmSoftwareIndex,
              PmFirmwareIndex,
              LcBufferToLoad );

        goto clean_exit ;
    }

    // Read the file and fills as many buffers as necessary
    // -----------------------------------------------------
    LcBuffNum = 0;
    LcReadSize = 0 ;

    do
    {
        LcBuffAddr = (LPBYTE) LcAddrDescTab[LcBuffNum].adAddr1; // was LinearMem.aiLinearAddress ;

        // The size of a buffer must be a number a DSP word
        // in order to ease downloading process
        // --------------------------------------------------
        LcBuffSize = ( LcAddrDescTab[LcBuffNum].adAddr3 / 3 ) * 3 ;

        LcReadRet = _read( LcFile, LcBuffAddr, LcBuffSize );

        if ( LcReadRet != -1 )
                // Cast to unsigned before adjusting size
                //
            LcReadSize = (unsigned int) LcReadRet ;
        else
            LcReadSize = 0 ;
        LcBufferToLoad[LcBuffNum].btlBufferNumber = LcBuffNum ;
        LcBufferToLoad[LcBuffNum].btlDataLength   = LcReadSize ;

        // Next buffer if needed
        // ----------------------
        LcBuffNum ++ ;

        // ## FM (08/20/97) -- If we used the last reserved buffers, but the
        //                     file has been totally read, THIS IS NOT A BUG!
        //
        if ( LcBuffNum == LcReservedBuffers )
        {
            // It the file has been entirely read, the returned size should be 0
            //
            LcReadRet = _read( LcFile, LcBuffAddr, 1 ) ;

            if ( LcReadRet != -1 )
                // Cast to unsigned before adjusting size
                //
                LcReadSize = (unsigned int) LcReadRet ;
            else
                LcReadSize = 0 ;
        }
    } while ( (LcReadRet != 0) && (LcReadRet != -1 ) &&
                                    (LcBuffNum < LcReservedBuffers) ) ;

    // if the buffer loading did not finish with end of file
    // an error occured
    // ------------------------------------------------------
    if ( LcReadRet != 0 )
    {
        LcBufferToLoad[0].btlDataLength = 0;
        // FS - 11/02/97
        // Let's keep the above error code
        // -------------------------------
        (VOID) ManageDsp( PmMyProcess,
                           PmBoardMask,
                           PmDspMask,
                           PmDspFeatures,
                           1,
                           PmSoftwareIndex,
						   PmFirmwareIndex,
                           LcBufferToLoad );
        if ( LcBuffNum == LcReservedBuffers )
        {
            LcRet = EA_FILE_TOO_LARGE ;
            goto clean_exit ;
        }

        LcRet = EA_READ_FILE_IMPOSSIBLE ;
        goto clean_exit ;
    }

    // Finally call the local function ManageDsp
    // that will finish up the job
    // -------------------------------------------
    LcRet = ManageDsp( PmMyProcess,
                       PmBoardMask,
                       PmDspMask,
                       PmDspFeatures,
                       LcBuffNum,
                       PmSoftwareIndex,
                       PmFirmwareIndex,
                       LcBufferToLoad );

clean_exit:

    if ( LcFile >= 0 ) _close( LcFile );

    COMReleaseExmut();

    return LcRet ;
}


// ****************************************************************************
// STATIC WORD IsDspMatchingCapabilities()
// ***************************************
//
// Input parameters :
// ****************
//
//  PDSP_CAPABILITY_INFO    PmDspCapabilities :  the capabilities of this software
//  PDSP_TO_LOAD_INFO       PmDspToLoadInfo   :  the user requirements for the sw
//
// Return value :
// **************
//  SUCCESS if the DSP can fulfill the requirements,
//   an error code otherwise.
//
// ****************************************************************************
//
//  Handles the loading process of a DSP software
//
// ****************************************************************************
STATIC WORD IsDspMatchingCapabilities(
    IN  PDSP_CAPABILITY_INFO    PmDspCapabilities,
    IN  PDSP_TO_LOAD_INFO   PmDspToLoadInfo )
{
    if ( (   PmDspCapabilities->dciFeatures
       & PmDspToLoadInfo->dtlDspFeatures ) != PmDspToLoadInfo->dtlDspFeatures )
    {
        return EA_BAD_DSP_FEATURES;
    }
    else if ( (  PmDspCapabilities->dciPlayFormats
           & PmDspToLoadInfo->dtlPlayFormats ) != PmDspToLoadInfo->dtlPlayFormats )
    {
        return EA_BAD_PLAY_FORMATS;
    }
    else if ( (  PmDspCapabilities->dciRecordFormats
           & PmDspToLoadInfo->dtlRecordFormats ) != PmDspToLoadInfo->dtlRecordFormats )
    {
        return EA_BAD_RECORD_FORMATS;
    }
    else if ( (  PmDspCapabilities->dciPlayEffects
               & PmDspToLoadInfo->dtlPlayEffects ) != PmDspToLoadInfo->dtlPlayEffects )
    {
        return EA_BAD_PLAY_EFFECTS;
    }
    else if ( (  PmDspCapabilities->dciRecordEffects
           & PmDspToLoadInfo->dtlRecordEffects ) != PmDspToLoadInfo->dtlRecordEffects )
    {
        return EA_BAD_RECORD_EFFECTS;
    }
    else
    {
        // Match OK !!
        // -----------
        return SUCCESS ;
    }
}

// ****************************************************************************
// STATIC WORD CheckDspCapabilities()
// **********************************
//
// Input parameters :
// ****************
//
// BYTE PmDspSoftwareNumber:        the Software code number to check or 0
// BYTE PmBoardType:            the board typ
// PDSP_TO_LOAD_INFO PmDspToLoad:   description of requested capabilities
//
// Output parameters :
// *****************
//
// LPWORD PmSoftIndex:           the index of an entry matching the
//                                   requested capabilities
//
// Return value :
// **************
//
// SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// If PmDspSoftwareNumber is null this function looks up the DSP capabilities
//   table for a software matching requested capabilities and board type
// Else this function checks the specified Dsp software matches the requested
//   capabilities for the specified board type
//
// ****************************************************************************
STATIC WORD CheckDspCapabilities(
    IN  BYTE            PmDspSoftwareNumber,
    IN  BYTE            PmBoardType,
    IN  BYTE            PmExtendedBoardType,
    IN  PDSP_TO_LOAD_INFO   PmDspToLoadInfo,
    OUT LPBYTE              PmSoftIndexPtr,
    OUT LPBYTE              PmFirmIndexPtr )
{
    WORD    LcFileIndex ;
    BYTE    LcSoftIndex ;
    BYTE    LcFirmIndex ;
    BOOLEAN LcMatchFound    = FALSE ;
    WORD    LcErrorCode = SUCCESS ;

    // What kind of request: check a specified DSP or lookup for one matching ?
    // ------------------------------------------------------------------------
    if ( PmDspSoftwareNumber )
    {
        // Verify the specified entry matches
        // ----------------------------------
        return IsDspMatchingCapabilities( &(DspCapabilities[PmDspSoftwareNumber]),
                      PmDspToLoadInfo );
    }
    else
    {
        // Lookup entries for this type of board
        // -------------------------------------
        for ( LcFileIndex = 0 ; LcFileIndex < DSP_FILE_INFO_ENTRIES ; LcFileIndex ++ )
        {
            if ( DspFiles[LcFileIndex].dfBoardType != PmBoardType )
                continue ;
            if ( DspFiles[LcFileIndex].dfExtendedBoardType != PmExtendedBoardType)
                continue ;

            // Verify this software has the right capabilities
            // -----------------------------------------------
            LcSoftIndex = DspFiles[LcFileIndex].dfSoftwareIndex;
            LcFirmIndex = DspFiles[LcFileIndex].dfFirmwareIndex;

            LcErrorCode = IsDspMatchingCapabilities(
                      &(DspCapabilities[LcSoftIndex]),
                      PmDspToLoadInfo );

            if ( LcErrorCode == SUCCESS )
            {
                LcMatchFound = TRUE ;
                break;
            }

            // if not found search for another software...
            // -------------------------------------------
        }

        if ( LcMatchFound )
        {
            *PmSoftIndexPtr = LcSoftIndex ;
            *PmFirmIndexPtr = LcFirmIndex ;
        }
        else
        {
            *PmSoftIndexPtr = 0 ;
            *PmFirmIndexPtr = 0 ;
        }

        return LcErrorCode ;
    }

}

// ****************************************************************************
// STATIC WORD DspAllocFct()
// ************************
//
// Input parameters:
// *****************
//
//  PCX_HANDLE        PmMyProcess:      Handle of the calling process.
//  PDSP_TO_LOAD_INFO PmDspToLoad:      Descriptor of application requirements
//  WORD              PmMaxSizeDspName: Max length of the supplied string
//                                      PmDspFileName
// Output parameters:
// *******************
//
//  LPTSTR            PmDspFileName:    Filename of DSP Software loaded if any
//  LPBOOLEAN         PmDspAlreadyOwned: Flag indicating that the application
//                                      already owned *this* (must be only one)
//                                      DSP before this call
//
// Return value :
// **************
//
//  SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function downloads a DSP (or many ones) with a software suitable
// to the requirements specified in DSP_TO_LOAD_INFO structure
//
// ****************************************************************************
STATIC WORD DspAllocFct(
    IN  PCX_HANDLE          PmMyProcess,
    IN  PDSP_TO_LOAD_INFO   PmDspToLoad,
    IN  WORD                PmMaxSizeDspName,
    OUT LPTSTR              PmDspFileName,
    OUT LPBOOLEAN           PmDspAlreadyOwned )
{
    WORD                LcErrorCode = SUCCESS   ;
    BOOLEAN             LcDspLoad   = FALSE ;   // whether a reload is needed
    BOOLEAN             LcDspPassed = TRUE  ;   // whether the DSP is not incompatible
    WORD                LcBoard     ;
    WORD                LcDsp       ;
    DWORD               LcBoardMask ;
    DWORD               LcDspMask   ;
    WORD                LcDspInfoNumber;
    DSP_FEATURES_INFO   LcDspInfo[MAX_BOARD_DSP];
    CARD_FEATURES_INFO  LcBoardInfo;
    BYTE                LcSoftwareIndex;
    BYTE                LcFirmwareIndex;
    size_t              LcStringSize;
    WORD                LcAppNumBeforeDspLoad = 0;

    // First of all, let's get the global mutex
    // in order to have exclusive access to the
    // driver
    // ----------------------------------------
    if (!COMWaitGlobalExmut())
       return EA_CANT_TAKE_MUTEX;

    if (!COMWaitExmut())
    {
        COMReleaseGlobalExmut();
        return EA_CANT_TAKE_MUTEX;
    }

    // ## FS (22/09/1997) -- Return DSP file names
    LcStringSize = PmMaxSizeDspName ;
    if ( PmDspFileName && ( LcStringSize != 0 ) )
    {
        *PmDspFileName = _T('\0');
    }

    if ( PmDspAlreadyOwned )
    {
        *PmDspAlreadyOwned = TRUE ;
    }

    // For all boards ...
    // -------------------
    LcBoardMask = PmDspToLoad->dtlCardMask ;
    while ( LcBoardMask )
    {
        LcBoard = UTIDWMask2Word( LcBoardMask );
        LcBoardMask &= ~UTIMaskDWord( LcBoard );

        // !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
        COMReleaseExmut();

        // Let's get the board type
        // ------------------------
        LcErrorCode = PCXGetHardwareBoardFeature( UTIMaskWord( LcBoard ),
                                                  &LcBoardInfo );

        if ( LcErrorCode != SUCCESS )
        {
            COMReleaseGlobalExmut();
            return LcErrorCode;
        }

        // Let's checked whether the DSP is already loaded or not
        // ------------------------------------------------------
        LcErrorCode = PCXGetSoftwareBoardFeature( UTIMaskWord( LcBoard ),
                                                  MAX_BOARD_DSP,
                                                  &LcDspInfoNumber,
                                                  LcDspInfo );

        if ( LcErrorCode != SUCCESS )
        {
            COMReleaseGlobalExmut();
            return LcErrorCode;
        }

        // Take Exmut back
        // ---------------
        if (!COMWaitExmut())
        {
            COMReleaseGlobalExmut();
            return EA_CANT_TAKE_MUTEX;
        }

        // ... and for all DSPs
        // --------------------
        LcDspMask = PmDspToLoad->dtlDspMask ;
        while ( LcDspMask )
        {
            LcDsp = UTIDWMask2Word( LcDspMask );
            LcDspMask &= ~UTIMaskDWord( LcDsp );

            // Let's verify if the DSP loaded is suitable
            // ## FM (08/27/97) -- If there is no application to use a loaded
            //                     DSP, then force the DSP to be loaded again.
            // ------------------------------------------
            if ( (LcDspInfo[LcDsp].dfDspState >= DSP_FIRST_SOFT) &&
                 (LcDspInfo[LcDsp].dfDspState <= DSP_LAST_SOFT) )
            {
                // Dsp already loaded
                // ------------------

                // Remember how many applications have already reserved
                // the DSP software (if any)
                // ----------------------------------------------------
                LcAppNumBeforeDspLoad = LcDspInfo[LcDsp].dfNbAppli ;

                // Verify that the DSP software match the application
                // requirements:
                // If it matches there is nothing to do unless
                //     no application is using the DSP AND the FAST_REALLOC
                //     flag is set
                // else
                //     if no application is using the DSP
                //         we can reload it
                //     else
                //         we cannot
                // -------------------------------------------------
                if ( CheckDspCapabilities( LcDspInfo[LcDsp].dfDspState,
                                           LcBoardInfo.ciBoardType,
                                           LcBoardInfo.ciExtendedBoardType,
                                           PmDspToLoad      ,
                                           NULL,
					   NULL ) == SUCCESS )
                {
                    if (   ( LcDspInfo[LcDsp].dfNbAppli != 0 )
                        || ( PmDspToLoad->dtlLoadPolicy == DSP_FAST_REALLOC ) )
                    {
                        LcDspLoad = FALSE ;
                    }
                    else
                    {
                        LcDspLoad = TRUE ;
                    }
                }
                else
                {
                    if ( LcDspInfo[LcDsp].dfNbAppli != 0 )
                    {
                        // sharing violation
                        // -----------------
                        LcErrorCode = EA_LOAD_DSP_REFUSED ;
                        LcDspPassed = FALSE ;
                    }
                    else
                    {
                        // Let's reload
                        // ------------
                        LcDspLoad = TRUE;
                    }
                }

            }
            // Dsp free (eventually crashed)
            // -----------------------------
            else if ( LcDspInfo[LcDsp].dfDspState == STATE_DSP_FREE )
            {
                // Dsp not really free because an application has reserved the
                // board through a PCXSetPipeProgrammableClock call.
                //
                if ( LcDspInfo[LcDsp].dfNbAppli != 0 )
                {
                    // sharing violation
                    // -----------------
                    LcErrorCode = EA_LOAD_DSP_REFUSED ;
                    LcDspPassed = FALSE ;
                }
                else
                {
                    LcDspLoad = TRUE ;
                }
            }
            else
            {
                // Dsp crashed but not released yet
                // --------------------------------
                LcErrorCode = EA_LOAD_DSP_REFUSED ;
                LcDspPassed = FALSE ;
            }

            // Do we need to reload a DSP ?
            // ----------------------------
            if ( LcDspPassed && LcDspLoad )
            {
                // Let's download the software if a suitable one
                // exists
                // ---------------------------------------------
                LcErrorCode = CheckDspCapabilities( 0,
                                                    LcBoardInfo.ciBoardType,
                                                    LcBoardInfo.ciExtendedBoardType,
                                                    PmDspToLoad,
                                                    &LcSoftwareIndex,
													&LcFirmwareIndex );

                if ( LcErrorCode == SUCCESS )
                {

                    // !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
                    COMReleaseExmut();

                    LcErrorCode = DspDownload( PmMyProcess,
                                               UTIMaskWord( LcBoard ),
                                               UTIMaskWord( LcDsp ),
                                               PmDspToLoad->dtlDspFeatures,
                                               LcSoftwareIndex,
											   LcFirmwareIndex);
                    // Take Exmut back
                    // ---------------
                    if (!COMWaitExmut())
                    {
                        COMReleaseGlobalExmut();
                        return EA_CANT_TAKE_MUTEX;
                    }
                }
            }
            else if ( LcDspPassed && !LcDspLoad )
            {
                // simply mark the application as using the DSP
                // --------------------------------------------
                LcErrorCode =  ManageDsp( PmMyProcess,
                                          UTIMaskWord( LcBoard ),
                                          UTIMaskWord( LcDsp ),
                                          PmDspToLoad->dtlDspFeatures,
                                          DSP_USE_CMD_CDE,
                                          0,
					  0,
                                          NULL );
            }
        }
        // while ( LcDspMask )
        // -------------------

        // Fill in the parameters sent back to the application
        // Beware that these params are valid only in ONE Dsp
        // is specified
        // ---------------------------------------------------
        if ( LcErrorCode != SUCCESS )
        {
            break ;
        }
        else
        {
            // !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
            COMReleaseExmut();

            // Did the application own the DSP before ?
            // ----------------------------------------
            PCXGetSoftwareBoardFeature( UTIMaskWord( LcBoard ),
                                        MAX_BOARD_DSP,
                                        &LcDspInfoNumber,
                                        LcDspInfo );
            // Take Exmut back
            // ---------------
            if (!COMWaitExmut())
            {
                COMReleaseGlobalExmut();
                return EA_CANT_TAKE_MUTEX;
            }

            if ( LcDspInfo[LcDsp].dfNbAppli > LcAppNumBeforeDspLoad )
            {
                *PmDspAlreadyOwned = FALSE ;
            }

            // What's the filename for DSP software ?
            // --------------------------------------

			if( LcDspLoad == FALSE )
			{
				// if we didn't load the DSP, get the SoftwareIndex by it's DSP state
				LcSoftwareIndex = LcDspInfo[LcDsp].dfDspState;
			}

            if( (LcSoftwareIndex != 0) &&
				(LcSoftwareIndex < DSP_CAPABILITIES_INFO_ENTRIES) &&
				PmDspFileName &&
				( LcStringSize != 0 ) )
            {
                if ( LcStringSize != PmMaxSizeDspName )
                {
                    _tcscat( PmDspFileName, _T(",") );
                    LcStringSize -- ;
                }

                if ( LcStringSize >= _tcslen( DspCapabilities[LcSoftwareIndex].dciFileName ) )
                {
                    _tcscat( PmDspFileName,
                            DspCapabilities[LcSoftwareIndex].dciFileName );
                    LcStringSize -= _tcslen( DspCapabilities[LcSoftwareIndex].dciFileName );
                }
            }
        }
    }
    // while (LcBoardMask )
    // --------------------

    COMReleaseExmut();
    COMReleaseGlobalExmut();
    return LcErrorCode;
}

// ****************************************************************************
// WORD PCXGetSoftwareBoardAbilities()
// ***********************************
//
// Input parameters :
// ****************
//
//  WORD              PmBoardMask:              Mask for naming a board
//
// Output parameters:
// *******************
//
//  PDSP_TO_LOAD_INFO PmDspToLoad:              Array of descriptor for DSP softwares
//  LPWORD            PmActualDspInfoNumPtr:    Actual number of DSP software available
//
// Return value:
// **************
//
//  SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function retrieves thes DSP softwares available for the board and prepare
// DSP_TO_LOAD_INFO structures ready-to-use for calling DspAlloc() if desired.
//
// ****************************************************************************
WORD _CDECL_ PCXGetSoftwareBoardAbilities(
    IN  WORD                PmBoardMask,
    IN  WORD                PmMaxDspInfo,
    OUT PDSP_TO_LOAD_INFO   PmDspToLoadArray,
    OUT LPWORD              PmActualDspInfoNumPtr )
{
    LPBOARD_HARD_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_HARD_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD                    LcFileIndex ;
    WORD                    LcError = SUCCESS ;

#define MAX_DSP_SOFTWARE_PER_BOARD    4
    DSP_TO_LOAD_INFO        LcDspToLoadArray[MAX_DSP_SOFTWARE_PER_BOARD];

    *PmActualDspInfoNumPtr = 0 ;

    // Reset the reply structure
    // --------------------------
    MEMSET2( LcDspToLoadArray, 0, DSP_TO_LOAD_INFO, MAX_DSP_SOFTWARE_PER_BOARD );

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcError = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcError;
    }

    // Send a board Hardware features request to the driver
    //
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_HARD_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_HARD_REQ_INFO ),
                    sizeof( BOARD_HARD_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bhrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcError = LcPReply->RespHeader.rhCptr;

    if ( !( LcError & ERROR_MASK) )
    {
        // Lookup entries for this type of board
        // -------------------------------------
        for ( LcFileIndex = 0 ; LcFileIndex < DSP_FILE_INFO_ENTRIES ; LcFileIndex ++ )
        {
            PDSP_CAPABILITY_INFO    LcDspCaps ;
            PDSP_TO_LOAD_INFO       LcDspInfo ;
            UINT                    LcSoftIndex ;
            WORD                    LcDspNum ;
            BYTE                    i ;

            if ( DspFiles[LcFileIndex].dfBoardType != LcPReply->bhfBoardType )
                continue ;
            if ( DspFiles[LcFileIndex].dfExtendedBoardType != LcPReply->bhfExtendedType)
                continue ;

            // Add this software to local list
            // -------------------------------

            // Copy the soft descriptor in the array
            // -------------------------------------
            LcSoftIndex = DspFiles[LcFileIndex].dfSoftwareIndex;
            LcDspInfo = &(LcDspToLoadArray[*PmActualDspInfoNumPtr]) ;
            LcDspCaps = &(DspCapabilities[LcSoftIndex]) ;

            LcDspInfo->dtlCardMask = UTIMaskWord( UTIDWMask2Word( PmBoardMask ) );
            for ( LcDspNum = 0 ; LcDspNum < LcPReply->bhfDspNum ; LcDspNum++ )
            {
                LcDspInfo->dtlDspMask |= (BYTE) UTIMaskWord( LcDspNum ) ;
            }
            LcDspInfo->dtlDspFeatures   = LcDspCaps->dciFeatures ;
            LcDspInfo->dtlPlayFormats   = LcDspCaps->dciPlayFormats ;
            LcDspInfo->dtlRecordFormats = LcDspCaps->dciRecordFormats ;
            LcDspInfo->dtlPlayEffects   = LcDspCaps->dciPlayEffects ;
            LcDspInfo->dtlRecordEffects = LcDspCaps->dciRecordEffects ;

			// hardware or eeprom protection limitations:
			//
            if ( ! ( LcPReply->bhfMiscFeatures & SMPTE_PRESENT_MASK ) )
                LcDspInfo->dtlDspFeatures &= ~DSP_CAN_TIME_CODE_MASK ;

            if ( ! ( LcPReply->bhfOptions & OPT_EQUALIZER_EFFECTS ) )
            {
                LcDspInfo->dtlPlayEffects &= ~(DSP_CAN_EQUALIZER_MASK | DSP_CAN_STREAM_EQ_LINEAR_MASK);
                LcDspInfo->dtlRecordEffects &= ~DSP_CAN_IN_AUDIO_EQ_MASK;
            }
            if ( ! ( LcPReply->bhfOptions & OPT_MAXIMIZER_EFFECTS ) )
            {
                LcDspInfo->dtlPlayEffects &= ~DSP_CAN_MAXIMIZER_MASK;
                LcDspInfo->dtlRecordEffects &= ~DSP_CAN_IN_AUDIO_MX_MASK;
            }
            if ( ! ( LcPReply->bhfOptions & OPT_MPEG123_FORMATS ) )
            {
                LcDspInfo->dtlPlayFormats &= ~(DSP_CAN_PR_MPEGI_MASK | DSP_CAN_PR_MPEGII_MASK | DSP_CAN_PR_MPEGIII_MASK);
                LcDspInfo->dtlRecordFormats &= ~(DSP_CAN_PR_MPEGI_MASK | DSP_CAN_PR_MPEGII_MASK );
            }
            if ( ! ( LcPReply->bhfOptions & OPT_TS_CF_EFFECTS ) )
            {
                LcDspInfo->dtlDspFeatures &= ~DSP_CAN_OFFLINE_MASK;
                LcDspInfo->dtlPlayEffects &= ~(DSP_CAN_PITCH_SHIFT_MASK | DSP_CAN_TIME_STRETCH_MASK | DSP_CAN_FREQUENCY_CONV_MASK);
            }

            for(i=0;i<MAX_BOARD_OUTPUTS;i++) if (LcPReply->bhfOutAudio[i]) break;

            if (i==MAX_BOARD_OUTPUTS)
            {
                LcDspInfo->dtlDspFeatures &= ~DSP_CAN_PLAY_MASK ;
                LcDspInfo->dtlPlayFormats  = 0 ;
            }

            for(i=0;i<MAX_BOARD_INPUTS;i++) if (LcPReply->bhfInAudio[i]) break;

            if (i==MAX_BOARD_INPUTS)
            {
                LcDspInfo->dtlDspFeatures   &= ~DSP_CAN_RECORD_MASK ;
                LcDspInfo->dtlRecordFormats  = 0 ;
            }

            // One more entry in array
            // -----------------------
            (*PmActualDspInfoNumPtr)++;
        }

        // The input array must hold at least one entry !!
        //
        if ( (!PmMaxDspInfo) || (!PmDspToLoadArray) ) LcError = WA_MORE_DATA ;
        if ( PmMaxDspInfo < *PmActualDspInfoNumPtr ) LcError = WA_MORE_DATA ;

        if ( PmMaxDspInfo )
        {
            memcpy(PmDspToLoadArray,
                   LcDspToLoadArray,
                   MIN(PmMaxDspInfo,*PmActualDspInfoNumPtr)*sizeof(DSP_TO_LOAD_INFO));
        }
    } // end of if there was no error

#undef MAX_DSP_SOFTWARE_PER_BOARD

    COMReleaseExmut();

    return( LcError );
}

// ****************************************************************************
// WORD PCXDspAllocW()
// WORD PCXDspAllocA()
// WORD PCXDspAlloc()
// ******************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE        PmMyProcess:      Handle of the calling process.
//  PDSP_TO_LOAD_INFO PmDspToLoad:      Descriptor of application requirements
//  WORD              PmMaxSizeDspName: Max length of the supplied string
//                                      PmDspFileName
// Output parameters:
// *******************
//
//  LPTSTR            PmDspFileName:    Filename of DSP Software loaded if any
//
// Return value:
// **************
//
//  SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function downloads a DSP (or many ones) with a software suitable
// to the requirements specified in DSP_TO_LOAD_INFO structure
//
// ****************************************************************************
#if WIN32
WORD _CDECL_ PCXDspAllocW(
    IN  PCX_HANDLE          PmMyProcess,
    IN  PDSP_TO_LOAD_INFO   PmDspToLoad,
    IN  WORD                PmMaxSizeDspName,
    OUT LPWSTR              PmDspFileName )
{
    LPTSTR  LcGenericString = 0 ;
    WORD    LcError;
    BOOLEAN LcDspAlreadyOwned;  // dummy variable

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if ( PmMaxSizeDspName && PmDspFileName )
    {
        LcGenericString = (LPTSTR) MALLOC2( _TCHAR, PmMaxSizeDspName );
        if ( LcGenericString )
        {
            _tcscpy( LcGenericString, _T("") );
        }
        else
        {
            return EA_ALLOCATE_MEMORY_IMPOSSIBLE;
        }
    }

    LcError = DspAllocFct( PmMyProcess,
                           PmDspToLoad,
                           PmMaxSizeDspName,
                           LcGenericString,
                           &LcDspAlreadyOwned );
    if ( PmMaxSizeDspName && PmDspFileName && LcGenericString )
    {
#if defined(_UNICODE) || defined(UNICODE)
        _tcsncpy( PmDspFileName, LcGenericString, PmMaxSizeDspName );
#else
        _snwprintf( PmDspFileName, PmMaxSizeDspName, L"%S", LcGenericString);
#endif
    }
    if( LcGenericString )
    {
        FREE( LcGenericString );
    }

    return( LcError );
}
#endif
WORD _CDECL_ PCXDspAllocA(
    IN  PCX_HANDLE          PmMyProcess,
    IN  PDSP_TO_LOAD_INFO   PmDspToLoad,
    IN  WORD                PmMaxSizeDspName,
    OUT LPSTR               PmDspFileName )
{
    LPTSTR  LcGenericString = 0 ;
    WORD    LcError;
    BOOLEAN LcDspAlreadyOwned;  // dummy variable

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if ( PmMaxSizeDspName && PmDspFileName )
    {
        LcGenericString = (LPTSTR) MALLOC2( _TCHAR, PmMaxSizeDspName );
        if ( LcGenericString )
        {
            _tcscpy( LcGenericString, _T("") );
        }
        else
        {
            return EA_ALLOCATE_MEMORY_IMPOSSIBLE;
        }
    }

    LcError = DspAllocFct( PmMyProcess,
                           PmDspToLoad,
                           PmMaxSizeDspName,
                           LcGenericString,
                           &LcDspAlreadyOwned );
    if ( PmMaxSizeDspName && PmDspFileName && LcGenericString )
    {
#if defined(_UNICODE) || defined(UNICODE)
        _snprintf( PmDspFileName, PmMaxSizeDspName, "%ls", LcGenericString);
#else
        _tcsncpy( PmDspFileName, LcGenericString, PmMaxSizeDspName );
#endif
    }
    if( LcGenericString )
    {
        FREE( LcGenericString );
    }

    return( LcError );
}

// ****************************************************************************
// WORD PCXDspFree()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PDSP_TO_LOAD_INFO PmDspToLoadInfo: Points to a DSP_TO_LOAD_INFO structure.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function must be used to release DSP that have previously
// been reserved by the application.

// All resources (such as clock, pipes, physical audio inputs /
// outputs) allocated to each DSP concerned by the command must
// be first released.
//
// ****************************************************************************
WORD _CDECL_ PCXDspFree(
    IN PCX_HANDLE PmMyProcess,
    IN PDSP_TO_LOAD_INFO PmDspToLoadInfo )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // calling the local function ManageDsp
    LcValret = ManageDsp( PmMyProcess,
                          PmDspToLoadInfo->dtlCardMask,
                          PmDspToLoadInfo->dtlDspMask,
                          0,  // unused
                          DSP_FREE_CMD_CDE,
                          0,
			  0,
                          NULL );
    COMReleaseExmut();
	
    return LcValret;
}

// ****************************************************************************
// WORD PCXGetDspResources()
// ***************************
//
// Input parameters :
// ****************
//
// WORD					PmBoardMask
// PDSP_RESOURCES_INFO	PmDspResourcesInfo
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
//	returns the % of DSP consumption and total memory size and actually free memory size
//
// ****************************************************************************
WORD _CDECL_ PCXGetDspResources(
    IN WORD                 PmBoardMask,
    IN PDSP_RESOURCES_INFO  PmDspResourcesInfo )
{

	LPDSP_RESOURCES_REQ_INFO LcPReq = 0;     // will point to the beginning of the
                                       // request block
    LPDSP_RESOURCES_RESP_INFO LcPResp = 0;    // will point to the beginning of the
                                       // response block

	if (!PmDspResourcesInfo)
        return EA_INVALID_POINTER;
	
    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;


    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
             DSP_RESOURCES_CMD,
			(APP_HANDLE) 1,        // pseudo handle, not used (is 1 to avoid error message)!
             sizeof( DSP_RESOURCES_REQ_INFO ),
             sizeof( DSP_RESOURCES_RESP_INFO ),
             (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->drqBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
		sizeof( DSP_RESOURCES_REQ_INFO ),
        (RESP_HEADER_HANDLE) &LcPResp,
        0 );

    COMReleaseExmut();

	if ( (LcPResp->drpDspRefCounter) && (LcPResp->Header.rhCptr == SUCCESS )) 
	{
		PmDspResourcesInfo->driDspUsage = (FLOAT)( 1 - (FLOAT)(LcPResp->drpDspCurrentCounter)/(FLOAT)(LcPResp->drpDspRefCounter))*100.0f;
		if (PmDspResourcesInfo->driDspUsage < 0)
			PmDspResourcesInfo->driDspUsage = 0.0f;
	}
	else
		PmDspResourcesInfo->driDspUsage = -1.0f;

	PmDspResourcesInfo->driMemAvailable		= LcPResp->drpMemAvailable;
	PmDspResourcesInfo->driMemSize			= LcPResp->drpMemSize;
	
    // returning the error code
    return LcPResp->Header.rhCptr;

}



// ****************************************************************************
// STATIC WORD DefinePipeSubFct()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PPIPE_DECL_INFO PmPipe: general description of pipe
//
// PSTREAM_DECL_INFO PmStream: description of pipe streams
//
// PAUDIO_DECL_INFO PmAudio: description of pipe audios
//
// LPWORD  PmBufferNumber: the number of buffers requested
//
// Output parameters :
// *****************
//
// LPWORD  PmBufferNumber: the actual number of buffers allocated
//
// LPDWORD PmPipeInOutMask: the mask of the pipe which will be used as a handle
//
// PBUFFER_ADDRESS_INFO: the address description of allocated buffers
//
// Return value :
// **************
//      SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to define a pipe (input pipe or output
// pipe) and the associated audio inputs or outputs.
// This function allows under-allocation with the flag
// P_ALLOW_UNDER_ALLOCATION_MASK of PmPipe->PdPipeManagement
//
// CCA : The original PCXDefinePipe has been indirected to prevent potential weirds
// behaviours with application which only set the (ON/OFF)LINE and (IN/OUT)PUT flags over
// a non initialized value (the documentation does not ask explicitly to reset the others bits).
//
// ****************************************************************************
STATIC WORD DefinePipeSubFct(
    IN PCX_HANDLE PmMyProcess,
    IN PPIPE_DECL_INFO PmPipe,
    IN PSTREAM_DECL_INFO PmStream,
    IN PAUDIO_DECL_INFO PmAudio,
    INOUT LPWORD PmBufferNumber,
    OUT LPDWORD PmPipeInOutMask,
    OUT PBUFFER_ADDRESS_INFO PmBufferAddressInfo)
{
    LPPIPE_DEF_REQ_INFO  LcPReq = 0;    // will point to the beginning of the
                                        // request block
    LPPIPE_DEF_RESP_INFO LcPResp = 0;   // will point to the beginning of the
                                        // response block
    LPPIPE_STREAM_INFO   LcPStreamDesc; // will point to each stream description
                                        // within the request block
    LPPIPE_AUDIO_INFO    LcPAudioDesc;  // will point to each audio description
                                        // within the request block
    WORD                 i;
    WORD                 LcError;
    BOOLEAN LcAllocOk ;

    DWORDLONG LcAudioMask = 0x0;
    DWORD LcChannelMask = 0x0;
    DWORD LcAudioNum = 0;

    LcChannelMask   = PmPipe->pdChannelMask;
    LcAudioNum      = PmPipe->pdAudioNumber;

    // Audio parameters
    // -----------------
    if (  ( ( LcAudioNum == 0 ) && ( LcChannelMask == 0 ))
        || ( LcAudioNum > MAX_AUDIO_PIPE ) )
        return EA_BAD_AUDIO_NUMBER ;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    PIPE_DEF_CMD,
                    (APP_HANDLE)PmMyProcess,
                    MAX_PIPE_DEF_REQ_SIZE,
                    MAX_PIPE_DEF_RESP_SIZE,
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------

    // General pipe info
    // -----------------
    LcPReq->pdqMaxStream = PmPipe->pdStreamNumber;
    LcPReq->pdqChannelMask = 0; // init

    if ( LcChannelMask != 0)
    {
        switch (LcChannelMask) {
        case SPEAKER_FRONT_LEFT:
        case SPEAKER_FRONT_CENTER:
            // mono in fact
            LcPReq->pdqChannelMask  = SPEAKER_FRONT_LEFT;
            LcAudioNum     = 1;
            break;
        case SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT:
            // stereo in fact
            LcPReq->pdqChannelMask  = SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT;
            LcAudioNum    = 2;
            break;
        case DIGIGRAM_5POINT1:
            LcPReq->pdqChannelMask  = DIGIGRAM_5POINT1;
            LcAudioNum     = 6;
            break;
        case DIGIGRAM_5POINT1_PLUS2:
            // 5.1 with stereo downmix
            LcPReq->pdqChannelMask  = DIGIGRAM_5POINT1_PLUS2;
            LcAudioNum     = 8;
            break;
        case DIGIGRAM_7POINT1:
            LcPReq->pdqChannelMask  = DIGIGRAM_7POINT1;
            LcAudioNum     = 8;
            break;
        case DIGIGRAM_7POINT1_PLUS2:
            // 7.1 with stereo downmix
            LcPReq->pdqChannelMask  = DIGIGRAM_7POINT1_PLUS2;
            LcAudioNum    = 10;
            break;
        default:
            COMReleaseExmut();
            return EA_BAD_MULTICHANNEL_FORMAT ;
        } // check channel mask
    } 
    else
        for ( i = 0 ; i < LcAudioNum; i++)  { 
            LcPReq->pdqChannelMask <<=1;
            LcPReq->pdqChannelMask++;
        }

    LcPReq->pdqAudioNum = (BYTE)LcAudioNum;

    LcPReq->pdqAttributes = PmPipe->pdPipeAttributes;

    // pas de pipe offline en m/c
    if ((PmPipe->pdPipeAttributes & PIPE_OFFLINE)&&(LcAudioNum>2))
    {
        COMReleaseExmut();
        return EA_BAD_MULTICHANNEL_FORMAT ;
    }

    // ## FS (16/06/98 ) -- sanity bound enforcement
    //
    if ( * PmBufferNumber > MAX_BUFFER ) *PmBufferNumber = MAX_BUFFER ;

    LcPReq->pdqBuffersNum = *PmBufferNumber;
    LcPReq->pdqManagement = PmPipe->pdPipeManagement;
    LcPReq->pdqBufferSize = PmPipe->pdBufferSize;

    // Stream parameters
    // ------------------
    if (   ( (PmPipe->pdStreamNumber == 0) && (*PmBufferNumber != 0) )
        || ( PmPipe->pdStreamNumber > MAX_STREAM_PIPE ) )
    {
        COMReleaseExmut();
        return EA_BAD_STREAM_NUMBER ;
    }

    for ( i = 0, LcPStreamDesc = (LPPIPE_STREAM_INFO)(LcPReq + 1);
          i < PmPipe->pdStreamNumber;
          i ++, LcPStreamDesc++)
    {
        LcPStreamDesc->psFormat = PmStream[i].sdFormat;
        // ## FS -- Enforce pipe to prepare for MPEG Layer III
        // if requested
		// SML: this case is obsolete when appli use HostNp
        // ---------------------------------------------------
        if ( LcPStreamDesc->psFormat & STREAM_FMT_MPEGIII )
        {
            LcPReq->pdqManagement |= P_PREPARE_FOR_MPEG3_MASK;
        }
        LcPStreamDesc->psBitRate = PmStream[i].sdBitrate;
    }

    // If a pipe has an even number of audios
    // the first audio number must be even as well
    // -------------------------------------------
// MTR   if ( !( LcAudioNum % 2) && (PmAudio[0].adAudioNumber % 2) )
// MTR   {
// MTR        COMReleaseExmut();
// MTR        return EA_BAD_AUDIO_NUMBER ;
// MTR   }

    for ( i = 0, LcPAudioDesc = (LPPIPE_AUDIO_INFO)(LcPStreamDesc);
          i < LcAudioNum;
          i ++, LcPAudioDesc++)
    {
        // Force the application to give the audio description in
        // the right order
        // ------------------------------------------------------
        if (   ( i > 0 )
            && ( PmAudio[i].adAudioNumber <= PmAudio[i-1].adAudioNumber ) )
        {
            COMReleaseExmut();
            return EA_BAD_AUDIO_NUMBER ;
        }

        GENAudioDecl2PipeAudioInfo( &(PmAudio[i]), LcPAudioDesc);

        // FS - (15/05/97) - Let's ignore the bit IN/OUT
        // ---------------------------------------------
        if (PmAudio[i].adAudioAttributes & AUDIO_PHYSICAL)
        {
            LcPAudioDesc->paAudioAttributes = AUDIO_PHYSICAL;
            LcAudioMask |= (((DWORDLONG) 1 ) << i);
        }
        else
        {
            LcPAudioDesc->paAudioAttributes = AUDIO_VIRTUAL;
        }

        // flag to control the parameters of the pipe or not
        if((PmMyProcess == *g_pDHSAppHandleDLL) && (PmAudio[i].adAudioAttributes & AUDIO_DHS_CONTROL_MASK))
        {
            LcPAudioDesc->paAudioAttributes |= AUDIO_DHS_CONTROL_MASK;
        }
    }

    // sending the request to the driver
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // processing the response
    LcError = LcPResp->RespHeader.rhCptr;

    if ( !( LcError & ERROR_MASK) )
    {
        LPADD_DESC_INFO LcPAddDesc = (LPADD_DESC_INFO)(LcPResp + 1);

        PMY_PIPE_INFO LcPipeInfo;
        DWORD LcMaxBufferSize = 0xFFFFFFFF; // max buffer size of all allocated buffers

        _ASSERT( LcPResp->pdpPipeMask64.HighPart == 0 ); // not handled yet

        *PmPipeInOutMask = LcPResp->pdpPipeMask64.LowPart;
        *PmBufferNumber = LcPResp->pdpBuffersNum;

		LcPAddDesc = (LPADD_DESC_INFO)(LcPResp + 1);
		for(i=0; i<(*PmBufferNumber); i++)
		{
			PmBufferAddressInfo[i].LinearMem.aiLinearAddress = LcPAddDesc[i].adAddr1;
			PmBufferAddressInfo[i].LinearMem.aiReserved1     = LcPAddDesc[i].adAddr2;

            if(LcPAddDesc[i].adAddr3 < LcMaxBufferSize)
                LcMaxBufferSize = LcPAddDesc[i].adAddr3;   // min buffer size of the allocated buffers
		}

        // CCA (20 sep 1999)
        // The pipe info structure is updated with the nb of streams allocated
        // -------------------------------------------------------------------
        PmPipe->pdStreamNumber = (BYTE) LcPResp->pdpMaxAllocatedStream ;

        // pipe allocation for DHS application! do not allocate Wait command entries
        //
        if( PmMyProcess == *g_pDHSAppHandleDLL )
        {
            // returning the error code
            COMReleaseExmut();
            return LcError; // SUCCESS or warning
        }

        // FS (FA #175) - 03/09/98
        // We should better allocate memory here only if pipe
        // allocation succeeds
        //

        // FS - 22/11/96
        // Allocates a request/reply command block per buffer
        // for async commands
        // FS - 06/02/97
        // We must separate output and input pipe masks
        // -------------------------------------------------------
        
        if (LcAudioMask <= 0x3) LcChannelMask = (DWORD)LcAudioMask;    // if not multichannel

        if ( PmPipe->pdPipeAttributes & PIPE_IN )
        {
            LcPipeInfo = &TbPipeInfo_IN[ UTIDWMask2Word( *PmPipeInOutMask) ];

            LcAllocOk = AllocWaitCmdEntry( PmMyProcess,
                                           0L,
                                           *PmPipeInOutMask,
                                           *PmBufferNumber,
                                           LcMaxBufferSize,
                                           PmPipe->pdStreamNumber );
        }
        else
        {
            LcPipeInfo = &TbPipeInfo_OUT[ UTIDWMask2Word( *PmPipeInOutMask) ];

            LcAllocOk = AllocWaitCmdEntry( PmMyProcess,
                                           *PmPipeInOutMask,
                                           0L,
                                           *PmBufferNumber,
                                           LcMaxBufferSize,
                                           PmPipe->pdStreamNumber );
        }

        if ( !LcAllocOk )
        {
            // cannot allocate, no need to go further on
            // -----------------------------------------

            // release the pipe
            // -----------------
            COMReleaseExmut();
            // !!!!!!!! MUST RELEASE EXMUT BEFORE CALLING ANOTHER PCX FCT !!!!
            if ( PmPipe->pdPipeAttributes & PIPE_IN )
                PCXReleasePipe( PmMyProcess, 0, *PmPipeInOutMask );
            else
                PCXReleasePipe( PmMyProcess, *PmPipeInOutMask, 0 );

            return EA_ALLOCATE_ASYNC_IMPOSSIBLE;
        }

        LcPipeInfo->piPipeAudioMask = LcAudioMask;
        LcPipeInfo->piPipeChannelMask = LcChannelMask;
        LcPipeInfo->piPipeNbChannels = LcAudioNum;
    }

    // returning the error code
    COMReleaseExmut();

    return LcError;
}


// ****************************************************************************
// WORD PCXDefinePipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PPIPE_DECL_INFO PmPipe: general description of pipe
//
// PSTREAM_DECL_INFO PmStream: description of pipe streams
//
// PAUDIO_DECL_INFO PmAudio: description of pipe audios
//
// LPWORD  PmBufferNumber: the number of buffers requested
//
// Output parameters :
// *****************
//
// LPWORD  PmBufferNumber: the actual number of buffers allocated
//
// LPDWORD PmPipeInOutMask: the mask of the pipe which will be used as a handle
//
// PBUFFER_ADDRESS_INFO: the address description of allocated buffers
//
// Return value :
// **************
//      SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to define a pipe (input pipe or output
// pipe) and the associated audio inputs or outputs.
//
// ****************************************************************************
WORD _CDECL_ PCXDefinePipe(
    IN    PCX_HANDLE           PmMyProcess,
    IN    PPIPE_DECL_INFO      PmPipe,
    IN    PSTREAM_DECL_INFO    PmStream,
    IN    PAUDIO_DECL_INFO     PmAudio,
    INOUT LPWORD               PmBufferNumber,
    OUT   LPDWORD              PmPipeInOutMask,
    OUT   PBUFFER_ADDRESS_INFO PmBufferAddressInfo)
{
// Force the under allocation bit to 0 to make sure the function does the
    // right operation.
    PmPipe->pdPipeManagement &= ~P_ALLOW_UNDER_ALLOCATION_MASK ;

    // Call the function
    return ( DefinePipeSubFct( PmMyProcess,
                               PmPipe,
                               PmStream,
                               PmAudio,
                               PmBufferNumber,
                               PmPipeInOutMask,
                               PmBufferAddressInfo ) ) ;
}


// ****************************************************************************
WORD ReleaseOnePipe(IN PCX_HANDLE PmMyProcess,
                    IN DWORD PmPipeOutMask,
                    IN DWORD PmPipeInMask,
					IN BOOLEAN PmDoStopPipe );
// ****************************************************************************
// WORD PCXDefinePipeEx()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE           PmMyProcess:    Handle of the calling process.
//
// PPIPE_DECL_INFO      PmPipe:         General description of pipe
//
// PSTREAM_DECL_INFO    PmStream:       Description of pipe streams
//
// PAUDIO_DECL_INFO     PmAudio:        Description of pipe audios
//
// LPWORD               PmBufferNumber: The number of buffers requested
//
// POUT_PIPE_SET_EFFECT PmEffect:       Description of the effect
//
// Output parameters :
// *****************
//
// LPWORD  PmBufferNumber:     the actual number of buffers allocated
//
// LPDWORD PmPipeInOutMask:    the mask of the pipe which will be used as a handle
//
// PBUFFER_ADDRESS_INFO:       the address description of allocated buffers
//
// PWORD PmAllocatedNbStream:  number of streams allocated
//
// Return value :
// **************
//      SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to define a pipe (input pipe or output
// pipe) and the associated audio inputs or outputs.
// All streams have the same format specified by PmStream.
// The function try to allocate the nb (<= requested one) of streams requested.
// If the function succeed, the specified effect is allocated.
// Note : PCXStopPipe remove this effect.
//
// ****************************************************************************
WORD _CDECL_ PCXDefinePipeEx(
    IN PCX_HANDLE PmMyProcess,
    INOUT PPIPE_DECL_INFO PmPipe,
    IN PSTREAM_DECL_INFO PmStream,
    IN PAUDIO_DECL_INFO PmAudio,
    IN POUT_PIPE_SET_EFFECT_INFO PmEffect,
    INOUT LPWORD PmBufferNumber,
    OUT LPDWORD PmPipeInOutMask,
    OUT PBUFFER_ADDRESS_INFO PmBufferAddressInfo,
    IN DWORD PmReserved)
{
    WORD             LcError,i;
    STREAM_DECL_INFO LcStreamFormat[MAX_STREAM_PIPE];
    TIME_INFO        LcDelayCondition;	
	BOOLEAN			 LcbNoEffect;
	
    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    MEMSET2(&LcDelayCondition,0,TIME_INFO,1);

    // Only output pipes are allowed
    if ( PmPipe->pdPipeAttributes & PIPE_IN) return ED_INVALID_PIPE ;

    // Check the number of streams (0 is not allowed)
    if (    ( PmPipe->pdStreamNumber > MAX_STREAM_PIPE )
         || ( PmPipe->pdStreamNumber == 0 ) )
    {
        return EA_BAD_STREAM_NUMBER;
    }

	LcbNoEffect				= ((!PmEffect) || (PmEffect->opseIdentifier == 0));
    
	// if No Effects where requested, remove under_allocation
	// don't pass to DSP
	if  (LcbNoEffect)	PmPipe->pdPipeManagement &= ~P_ALLOW_UNDER_ALLOCATION_MASK;
	else				PmPipe->pdPipeManagement |= P_ALLOW_UNDER_ALLOCATION_MASK;

    // The streams have the same format
    for (i=0;i<PmPipe->pdStreamNumber;i++)
    {
        memcpy(LcStreamFormat+i,PmStream,sizeof(STREAM_DECL_INFO));
    }

    // Find the maximum number of streams (we must take into account the effect)
    while (  PmPipe->pdStreamNumber )
    {
        LcError = DefinePipeSubFct( PmMyProcess,
                                    PmPipe,
                                    LcStreamFormat,
                                    PmAudio,
                                    PmBufferNumber,
                                    PmPipeInOutMask,
                                    PmBufferAddressInfo );

        if( LcbNoEffect &&
            ( ( LcError == EB_ALLOCATE_ALL_STREAM_TRANSFERT_BUFFERS_IMPOSSIBLE  ) ||
              ( LcError == EB_ALLOCATE_CONTEXT_LIII_IMPOSSIBLE ) ||
              ( LcError == EB_ALLOCATE_STREAM_IMPOSSIBLE ) ) )  // MBR (14/02/03) Defect 2596 : EB_ALLOCATE_STREAM_IMPOSSIBLE is also possible here
		{   
			ReleaseOnePipe( PmMyProcess, *PmPipeInOutMask, 0, FALSE );

            (PmPipe->pdStreamNumber)--;
			continue;
		}

        if ( LcError != SUCCESS ) return LcError;

        LcError = PCXSetOutPipeEffect( PmMyProcess,
                                       *PmPipeInOutMask,
                                       &LcDelayCondition,
                                       PmEffect );

        if ( ( LcError != SUCCESS ) || ( PmPipe->pdStreamNumber == 0 ) )
        {
            ReleaseOnePipe( PmMyProcess, *PmPipeInOutMask, 0, FALSE );

            // Exit if PCXDefinePipe Allocate a pipe with 0 stream
            if ( PmPipe->pdStreamNumber == 0 ) return EA_BAD_STREAM_NUMBER;

            // Exit if an error occured during PCXSetOutPipeEffect
            if ( LcError != EB_ALLOCATE_ALL_STREAM_TRANSFERT_BUFFERS_IMPOSSIBLE ) return LcError;

            // If there is not enough DSP memory to create an effect, we try
            // with less streams until there is enough
            (PmPipe->pdStreamNumber)--;
        }
        else
        {
            // All right, exit successfully
            return SUCCESS;
        }
    }

    // No configuration has been found
    return EA_BAD_STREAM_NUMBER;
}


// ****************************************************************************
// STATIC WORD ReleaseOnePipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: output pipe mask.
//
// DWORD PmPipeInMask: input pipe mask.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to removing only one pipe at a time
//
// ****************************************************************************
WORD ReleaseOnePipe(
    IN PCX_HANDLE   PmMyProcess,
    IN DWORD        PmPipeOutMask,
    IN DWORD        PmPipeInMask,
	IN BOOLEAN      PmDoStopPipe )
{
    LPPIPE_REM_REQ_INFO LcPReq = 0;      // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    DWORD       LcMask ;
    WORD        LcPipeNum ;
    PCX_TIME    LcPipeSample ;
    WORD        LcError ;

    // Try to stop un-synchronized pipes
    //
	if( PmDoStopPipe )
	{
		LcError = PCXStopPipe( PmMyProcess,
                        PmPipeOutMask,
                        PmPipeInMask,
                        NULL,
                        &LcPipeSample );

		if( LcError != SUCCESS )
			return LcError;
	}

    if( !COMWaitExmut() )
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    PIPE_DEL_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PIPE_REM_REQ_INFO ),
                    // FS - 11/02/1997 - 16/12/1997
                    // Under WindowsNT, we need a larger size
                    // since the reply block is used to unmap
                    // pipe's buffers
                    // -------------------------------------------
                    sizeof(ANY_UNMAPPING_REPLY_INFO),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->prqOutMask64.QuadPart = PmPipeOutMask;
    LcPReq->prqInMask64.QuadPart = PmPipeInMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;

    if ( !( LcError & ERROR_MASK) && (PmMyProcess != *g_pDHSAppHandleDLL) )
    {
        for ( LcMask = PmPipeOutMask ;
              LcMask != 0 ;
              LcMask &= ~UTIMaskDWord( LcPipeNum ) )
        {
            LcPipeNum = UTIDWMask2Word( LcMask );

            FreeWaitCmdEntry( UTIMaskDWord( LcPipeNum ), 0L, TRUE );

			TbPipeInfo_OUT[ LcPipeNum ].piPipeAudioMask = 0;
			TbPipeInfo_OUT[ LcPipeNum ].piPipeChannelMask = 0;
			TbPipeInfo_OUT[ LcPipeNum ].piPipeNbChannels = 0;
        }

        for ( LcMask = PmPipeInMask ;
              LcMask != 0 ;
              LcMask &= ~UTIMaskDWord( LcPipeNum ) )
        {
            LcPipeNum = UTIDWMask2Word( LcMask );

            FreeWaitCmdEntry( 0L, UTIMaskDWord( LcPipeNum ), TRUE );

			TbPipeInfo_IN[ LcPipeNum ].piPipeAudioMask = 0;
			TbPipeInfo_IN[ LcPipeNum ].piPipeChannelMask = 0;
			TbPipeInfo_IN[ LcPipeNum ].piPipeNbChannels = 0;
        }
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXReleasePipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: output pipe mask.
//
// DWORD PmPipeInMask: input pipe mask.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to removing pipes.
//
// ****************************************************************************
WORD _CDECL_ PCXReleasePipe(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask )
{
    LPPIPE_REM_REQ_INFO LcPReq = 0;      // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    DWORD       LcMask ;
    WORD        LcPipeNum ;
    PCX_TIME    LcPipeSample ;
    WORD        LcError ;
    WORD        LcReturnCode   = SUCCESS ;
	BOOLEAN     LcDoStopSinglePipes = FALSE;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if(PmMyProcess != *g_pDHSAppHandleDLL)
    {
        // Try to stop synchronized pipes
        //
        LcError = PCXStopPipe( PmMyProcess,
                            PmPipeOutMask,
                            PmPipeInMask,
                            NULL,
                            &LcPipeSample );

	    if( LcError != SUCCESS )
		    LcDoStopSinglePipes = TRUE;
    }

    // FS (FA #175) - 03/09/98
    // In order to handle correctly error cases, we should better release
    // pipes only one at a time
    //
    for ( LcMask = PmPipeOutMask ;
          LcMask != 0 ;
          LcMask &= ~UTIMaskDWord( LcPipeNum ) )
    {
        LcPipeNum = UTIDWMask2Word( LcMask );
        LcError = ReleaseOnePipe( PmMyProcess, UTIMaskDWord( LcPipeNum ), 0L, LcDoStopSinglePipes );
        if ( ( LcReturnCode == SUCCESS ) && ( LcError != SUCCESS ) )
        {
            LcReturnCode = LcError ;
        }
    }

    for ( LcMask = PmPipeInMask ;
          LcMask != 0 ;
          LcMask &= ~UTIMaskDWord( LcPipeNum ) )
    {
        LcPipeNum = UTIDWMask2Word( LcMask );
       
        LcError = ReleaseOnePipe( PmMyProcess, 0L, UTIMaskDWord( LcPipeNum ), LcDoStopSinglePipes );
        if ( ( LcReturnCode == SUCCESS ) && ( LcError != SUCCESS ) )
        {
            LcReturnCode = LcError ;
        }
    }

    // returning the error code
    return LcReturnCode;
}


// ****************************************************************************
// WORD PCXGetDspCrashError()
// **************************
//
// Input parameters :
// ****************
//
// WORD PmCarteNum  : the number of the targetted board
// WORD PmDspNum    : the number of the targetted DSP on PmCarteNum board
//
// Output parameters:
// ******************
//
// LPWORD PmDspFatalError:   The fatal error wich occur afert a DSP crash
//                          or 0 if the DSP wasn't crash.
//
// Return value :
// **************
//
// This function returns :
//           0 : the request completes successfully
//
// ****************************************************************************
//
// For debugging purpose only
//
// ****************************************************************************
WORD _CDECL_ PCXGetDspCrashError(
    IN  WORD PmCarteNum,
    IN  WORD PmDspNum,
    OUT LPWORD PmPtrDspFatalError )
{
    LPDSP_ERROR_REQ_INFO    LcRequest =  0;
    LPDSP_ERROR_RESP_INFO   LcReply   =  0;
    WORD LcError;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    COMFillBcHeader(    GENE_FAM,
                        DSP_RECOVER_ERROR_CMD,
                        0,
                        (WORD)(sizeof( DSP_ERROR_REQ_INFO )),
                        (WORD)(sizeof( DSP_ERROR_RESP_INFO )),
                        (BC_HEADER_HANDLE) &LcRequest );

    // Initialize the rest of the request block
    // ---------------------------------------
    LcRequest->derqCarteNum = PmCarteNum;
    LcRequest->derqDspNum = PmDspNum;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcReply,
                   0 );

    // Copy the fatal error from the response block
    // ----------------------------------------------
    *PmPtrDspFatalError = LcReply->derspDspError;

    LcError = LcReply->RespHeader.rhCptr;
    COMReleaseExmut();

    return LcError;
}

#if WIN32
// ****************************************************************************
// WORD PCXGetOwnerApplicationW()
// WORD PCXGetOwnerApplicationA()
// WORD PCXGetOwnerApplication()
// **********************************
//
// Input parameters :
// ****************
//
// PAUDIO_DECL_INFO PmAudio: Description of audio IO
//
//
// Output parameters :
// *****************
//
// OUT PAPP_INFO PmAppInfo: Pointer on APP_INFO structure that will contain informations
//                          on the application wich own the requested audio IO.
//
// Return value :
// **************
//      SUCCESS if no error occured, an error code otherwise.
//
// ****************************************************************************
// FH 26/02/97 V3 management
// ****************************************************************************
WORD _CDECL_ PCXGetOwnerApplicationW(
    IN  PAUDIO_DECL_INFO    PmAudio,
    OUT PAPP_INFO_W         PmAppInfo )
{
    LPAPP_AUDIO_OWNER_REQ_INFO LcPReq = 0;      // will point to the beginning of the
                                                // request block  '
    LPAPP_AUDIO_OWNER_RESP_INFO LcPResp = 0;    // will point to the beginning of the
                                                // response block
    WORD LcError;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader(    GENE_FAM,
                        AUDIO_INFO_CMD,
                        0,
                        sizeof(APP_AUDIO_OWNER_REQ_INFO),
                        sizeof(APP_AUDIO_OWNER_RESP_INFO),
                        (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmAudio, &(LcPReq->AudioInf) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // processing the response
    // -----------------------
    LcError = LcPResp->RespHeader.rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        PmAppInfo->aiProcess    = LcPResp->AppliInf.adHandle;
        PmAppInfo->aiWinHandle  = (HANDLE) LcPResp->AppliInf.adWindowHandle;
        PmAppInfo->aiWinMessage = LcPResp->AppliInf.adWindowMessage;

        if ( ((LcPResp->AppliInf).adAppName).asStringType == APP_UNICODE_STRING )
        {
            wcscpy(PmAppInfo->aiAppName, ((LcPResp->AppliInf).adAppName).asUnicodeString);
        }
        else
        {
            _snwprintf(PmAppInfo->aiAppName, SIZE_MAX_APP_NAME+1, L"%hs", ((LcPResp->AppliInf).adAppName).asAnsiString);
        }
    }

    // returning the error code
    COMReleaseExmut();
    return LcError;
}
WORD _CDECL_ PCXGetOwnerApplicationA(
    IN  PAUDIO_DECL_INFO    PmAudio,
    OUT PAPP_INFO_A         PmAppInfo )
{
    APP_INFO_W      LcAppInfo ;
    WORD            LcError ;

    LcError = PCXGetOwnerApplicationW( PmAudio,
                                       &LcAppInfo  );

    if ( !(LcError & ERROR_MASK) )
    {
        PmAppInfo->aiProcess    = LcAppInfo.aiProcess;
        PmAppInfo->aiWinHandle  = LcAppInfo.aiWinHandle;
        PmAppInfo->aiWinMessage = LcAppInfo.aiWinMessage;

        sprintf( PmAppInfo->aiAppName, "%ls", LcAppInfo.aiAppName );
    }

    return( LcError );
}
#endif // WIN32

// ************************************************************************* //
//                                                                           //
//  API EXPORTED FUNCTIONS : ASYNCHRONOUS COMMAND FAMILY                     //
//                                                                           //
// ************************************************************************* //


// ****************************************************************************
// WORD PCXPlayStream()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PPLAY_BUFFER_INFO PmPlayBufferInfo: Points to a PLAY_BUFFER_INFO structure.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function transfers a buffer to a playing stream.
//
// ****************************************************************************

WORD _CDECL_ PCXPlayStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmStreamMask,
    IN PPLAY_BUFFER_INFO PmPlayBufferInfo,
    IN HANDLE PmEvent )

{
    WORD                LcNumBuff;
    LPPLAY_REQ_INFO     LcPlayReq;     // FS - 02/10/1996
    LPRESP_HEADER_INFO  LcPlayResp;    //   ReqTab[] is dynamically
                                       //   allocated when an appli
                                       //   makes some buffer reservation
    WORD                LcIndex;
    WORD                LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // FS - remove test against MAX_BUFFER and replace by a test
    // against hte number of buffers the application has really reserved
    // see further...

    // Look up the application handle
    // in the tab of asynchronous command blocks
    if ( !LookupWaitCmdTab( PmPipeOutMask, 0L, &LcIndex ) )
    {
        // The application either gave a wrong handle
        //  or did not reserve any buffer
        // ------------------------------------------
       COMReleaseExmut();
       return EA_INVALID_PCX_HANDLE;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    if( (PmPlayBufferInfo->pbBufferNumber >= WaitCmds[LcIndex].wcbNbBuff)   ||
        (PmPlayBufferInfo->pbDataLength > WaitCmds[LcIndex].wcbMaxBuffSize) )
    {
        COMReleaseExmut();
        return EA_INVALID_BUFFER;
    }

    LcNumBuff = PmPlayBufferInfo->pbBufferNumber;

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcPlayReq  = &( WaitCmds[LcIndex].wcbRequestTab[LcNumBuff].aarPlayStream);
    LcPlayResp = &( WaitCmds[LcIndex].wcbReplyTab[LcNumBuff].aapPlayStream );


    // Beware the buffer must be actually empty
    // -----------------------------------------

    if (LcPlayResp->rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_BUFFER_REFUSED;
    }

		// filling the header of the request block
		// ---------------------------------------
		COMFillBcHeader( PLAY_RECORD_FAM,
						PLAY_CMD,
						(APP_HANDLE)PmMyProcess,
						sizeof( PLAY_REQ_INFO ),
						sizeof( RESP_HEADER_INFO ),
						(BC_HEADER_HANDLE) &LcPlayReq );

		// filling the other fields of the request
		// ---------------------------------------
		LcPlayReq->plqOutPipeMask64.QuadPart = PmPipeOutMask;
		LcPlayReq->plqStreamMask    = PmStreamMask;
		LcPlayReq->plqBuffNum       = LcNumBuff;
		LcPlayReq->plqDataLength    = PmPlayBufferInfo->pbDataLength;
		LcPlayReq->plqPause         = PmPlayBufferInfo->pbPause;
		LcPlayReq->plqHint          = PmPlayBufferInfo->pbHint;
        
        // setting this tag allows to be notified for this buffer
        // the tag value can be read later in NOTIFY_INFO.niCommandParam2
        //
		LcPlayReq->plq9BitsTag      = PmPlayBufferInfo->pb9BitsTag; 

		LcPlayReq->plqMisc          = PmPlayBufferInfo->pbMisc; 

		// sending the request to the driver
		// ---------------------------------
	#if defined (WIN32)
		WaitCmds[LcIndex].wcbOverlappedTab[LcNumBuff].hEvent = PmEvent;
		COMCallDriver( (LPBC_HEADER_INFO) LcPlayReq,
					   sizeof( PLAY_REQ_INFO ),
					   &LcPlayResp,
					   &(WaitCmds[LcIndex].wcbOverlappedTab[LcNumBuff]) );
	#else
		COMCallDriver( (LPBC_HEADER_INFO) LcPlayReq,
					   sizeof( PLAY_REQ_INFO ),
					   &LcPlayResp,
					   PmEvent );
	#endif
		LcError = LcPlayResp->rhCptr;
		COMReleaseExmut();

		return( LcError );
}


// ****************************************************************************
// WORD PCXRecordStream()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Output pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// IN PRECORD_BUFFER_INFO PmRecordBufferInfo: Points to a RECORD_BUFFER_INFO
//                              structure.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function transfers a buffer to a recording stream.
//
// ****************************************************************************

WORD _CDECL_ PCXRecordStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PRECORD_BUFFER_INFO PmRecordBufferInfo,
    IN HANDLE PmEvent)
{
    WORD                LcNumBuff;
    LPRECORD_REQ_INFO   LcRecordReq;   // FS - 02/10/1996
    LPRECORD_RESP_INFO  LcRecordResp;  //   ReqTab[] is dynamically
                                       //   allocated when an appli
                                       //   makes some buffer reservation
    WORD                LcAppliIdx;
    WORD                LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // FS - remove test against MAX_BUFFER and replace by a test
    // against the number of buffers the application has really reserved
    // see further...

    // Look up the application handle
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitCmdTab( 0L, PmPipeInMask, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        // or did not reserve any buffer
        //
        COMReleaseExmut();
        return EA_INVALID_PCX_HANDLE;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    LcNumBuff = PmRecordBufferInfo->rbBufferNumber;

    if( (LcNumBuff >= WaitCmds[LcAppliIdx].wcbNbBuff) ||
        (PmRecordBufferInfo->rbDataLength > WaitCmds[LcAppliIdx].wcbMaxBuffSize) )
    {
        COMReleaseExmut();
        return EA_INVALID_BUFFER;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcRecordReq  = &( WaitCmds[LcAppliIdx].wcbRequestTab[LcNumBuff].aarRecordStream);
    LcRecordResp = &( WaitCmds[LcAppliIdx].wcbReplyTab[LcNumBuff].aapRecordStream);

    // Beware the buffer must be actually free
    // -----------------------------------------
    if ((LcRecordResp->RespHeader).rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_BUFFER_REFUSED;
    }

		// filling the header of the request block
		// ---------------------------------------
		COMFillBcHeader( PLAY_RECORD_FAM,
						RECORD_CMD,
						(APP_HANDLE)PmMyProcess,
						sizeof( RECORD_REQ_INFO ),
						sizeof( RECORD_RESP_INFO ),
						(BC_HEADER_HANDLE) &LcRecordReq );

		// filling the other fields of the request
		// ---------------------------------------
		LcRecordReq->rcqInPipeMask64.QuadPart = PmPipeInMask;
		LcRecordReq->rcqStreamMask   = PmStreamMask;
		LcRecordReq->rcqBuffNum      = LcNumBuff;
		LcRecordReq->rcqDataLength   = PmRecordBufferInfo->rbDataLength ;

        // setting this tag allows to be notified for this buffer
        // the tag value can be read later in NOTIFY_INFO.niCommandParam2
        //
		LcRecordReq->rcq9BitsTag     = PmRecordBufferInfo->rb9BitsTag; 

		LcRecordReq->rcqMisc         = PmRecordBufferInfo->rbMisc; 

		// sending the request to the driver
		// ---------------------------------
	#if defined (WIN32)
		WaitCmds[LcAppliIdx].wcbOverlappedTab[LcNumBuff].hEvent = PmEvent;
		COMCallDriver( (LPBC_HEADER_INFO) LcRecordReq,
					   sizeof( RECORD_REQ_INFO ),
					   (RESP_HEADER_HANDLE) &LcRecordResp,
					   &(WaitCmds[LcAppliIdx].wcbOverlappedTab[LcNumBuff]) );
	#else
		COMCallDriver( (LPBC_HEADER_INFO) LcRecordReq,
					   sizeof( RECORD_REQ_INFO ),
					   (RESP_HEADER_HANDLE) &LcRecordResp,
					   PmEvent );
	#endif

		LcError = (LcRecordResp->RespHeader).rhCptr;
		COMReleaseExmut();
		return( LcError );
}

// ****************************************************************************
// WORD PCXCancelStreamBuffers()
// *****************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// DWORD PmPipeOutMask: Output pipe mask.
// DWORD PmPipeInMask: Input pipe mask.
// DWORD PmStreamMask: Stream mask.
// WORD  PmBufferNumber: Number of first buffer to cancel
//
// Output parameters :
// ****************
//
// LPWORD PmCancelledNumber: Number of buffers actually cancelled
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function cancels a buffer and the following for a given stream if
// the no-return point has not been crossed yet. In the later case, no
// buffer wille be cancelled and the function returns a warning
//
// Note that only one stream can be specified.
//
// ****************************************************************************
WORD _CDECL_ PCXCancelStreamBuffers(
    IN  PCX_HANDLE  PmMyProcess,
    IN  DWORD       PmPipeOutMask,
    IN  DWORD       PmPipeInMask,
    IN  DWORD       PmStreamMask,
    IN  WORD        PmBufferNumber,
   OUT  LPWORD      PmCancelledNumber )
{
    LPRESP_HEADER_INFO           LcPlayResp;
    LPCANCEL_BUFFER_REQ_INFO     LcCancelReq = NULL;
    LPCANCEL_BUFFER_RESP_INFO    LcCancelResp = NULL;
    WORD                         LcIndex;
    WORD                         LcError = SUCCESS;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // No buffer cancelled yet
    //
    if ( PmCancelledNumber != NULL ) *PmCancelledNumber = 0 ;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Look up the application handle
    // in the tab of asynchronous command blocks
    if ( !LookupWaitCmdTab( PmPipeOutMask, PmPipeInMask, &LcIndex ) )
    {
        // The application either gave a wrong handle
        //  or did not reserve any buffer
        // ------------------------------------------
        LcError = EA_INVALID_PIPE;
        goto clean_exit;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    if ( PmBufferNumber >= WaitCmds[LcIndex].wcbNbBuff )
    {
        LcError = EA_INVALID_BUFFER;
        goto clean_exit;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcPlayResp = &( WaitCmds[LcIndex].wcbReplyTab[PmBufferNumber].aapPlayStream);


    // Beware the buffer must be actually in the driver's hands
    // --------------------------------------------------------
    if (LcPlayResp->rhT == PROCESSED)
    {
        LcError = WA_CANNOT_CANCEL;
        goto clean_exit;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    CANCEL_STR_BUFFER_CMD,
                    (APP_HANDLE) PmMyProcess,
                    sizeof( CANCEL_BUFFER_REQ_INFO ),
                    sizeof( CANCEL_BUFFER_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcCancelReq );

    // filling the other fields of the request
    // ---------------------------------------
    LcCancelReq->cbqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcCancelReq->cbqInPipeMask64.QuadPart  = PmPipeInMask;
    LcCancelReq->cbqStreamMask    = PmStreamMask;
    LcCancelReq->cbqBuffNum       = PmBufferNumber;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcCancelReq,
                   sizeof( CANCEL_BUFFER_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcCancelResp,
                   NULL );

    LcError = (LcCancelResp->RespHeader).rhCptr;

    // Retrieve the number of buffers actually cancelled
    // -------------------------------------------------
    if ( ( LcError == SUCCESS ) && ( PmCancelledNumber != NULL ) )
    {
        *PmCancelledNumber = (WORD) LcCancelResp->cbrCancelledNumber;
    }

clean_exit:
    COMReleaseExmut();
    return( LcError );
}

// ****************************************************************************
// WORD PCXCancelPauseAtBufferEnd()
// *****************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// DWORD PmPipeOutMask: Output pipe mask.
// DWORD PmStreamMask: Stream mask.
// WORD  PmBufferNumber: Number of first buffer to cancel
// PTIME_INFO PmDelayType: Flags for notification handling
//
// Output parameters :
// ****************
//
// none
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function cancels a pause at end of buffer if
// the no-return point has not been crossed yet.
// In the later case, the pause will take place anyway, and
// the function returns a warning.
//
// If a pause is cancelled, the notification might be cancelled as well
// if not specified in the DelayType or kept active if specified.
//
// ****************************************************************************
WORD _CDECL_ PCXCancelPauseAtBufferEnd(
    IN  PCX_HANDLE      PmMyProcess,
    IN  DWORD           PmPipeOutMask,
    IN  DWORD           PmStreamMask,
    IN  WORD            PmBufferNumber,
    IN  PTIME_INFO      PmTimeInfo )
{
    LPRESP_HEADER_INFO           LcPlayResp;
    LPCANCEL_PAUSE_REQ_INFO      LcCancelReq = NULL;
    LPRESP_HEADER_INFO           LcCancelResp = NULL;
    WORD                         LcIndex;
    WORD                         LcError;
	
    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Look up the application handle
    // in the tab of asynchronous command blocks
    //
    if ( !LookupWaitCmdTab( PmPipeOutMask, 0L, &LcIndex ) )
    {
        // The application either gave a wrong handle
        //  or did not reserve any buffer
        // ------------------------------------------
        LcError = EA_INVALID_PIPE;
        goto clean_exit;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    if ( PmBufferNumber >= WaitCmds[LcIndex].wcbNbBuff )
    {
        LcError = EA_INVALID_BUFFER;
        goto clean_exit;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcPlayResp = &( WaitCmds[LcIndex].wcbReplyTab[PmBufferNumber].aapPlayStream);

    // Beware the buffer must be actually in the driver's hands
    // --------------------------------------------------------
    if (LcPlayResp->rhT == PROCESSED)
    {
        LcError = WA_CANNOT_CANCEL;
        goto clean_exit;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    CANCEL_BUFFER_PAUSE_CMD,
                    (APP_HANDLE) PmMyProcess,
                    sizeof( CANCEL_PAUSE_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcCancelReq );

    // filling the other fields of the request
    // ---------------------------------------
    if ( PmTimeInfo )
    {
        LcCancelReq->cbpqDiffered      = (BYTE)PmTimeInfo->tiDelayed;;
    }
    else
    {
        LcCancelReq->cbpqDiffered      = 0;
    }
    LcCancelReq->cbpqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcCancelReq->cbpqStreamMask    = PmStreamMask;
    LcCancelReq->cbpqBuffNum       = PmBufferNumber;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcCancelReq,
                   sizeof( CANCEL_PAUSE_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcCancelResp,
                   NULL );

    LcError = LcCancelResp->rhCptr;

clean_exit:
    COMReleaseExmut();
    return( LcError );
}


// ****************************************************************************
// WORD PCXWaitPipePlayEnd()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Asked pipe mask.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Asynchronous function to ask the driver to tell when the play of a
// specified pipe is finished.
//
// ****************************************************************************

WORD _CDECL_ PCXWaitPipePlayEnd(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN HANDLE PmEvent)
{
    LPWAIT_ENDOF_PLAY_REQ_INFO  LcWaitEndOfPlayReq;
    LPRESP_HEADER_INFO          LcWaitEndOfPlayResp;
    WORD                        LcPipeIndex;
    WORD                        LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // FS - 03/10/1996
    //  Removal of the global WaitEndOfPlayReq / Resp variables and
    //  replacement by per-pipe allocated command blocks
    //

    // Extract pipe number from the mask
    // ---------------------------------
    if ( PmPipeOutMask == 0 )
    {
        COMReleaseExmut();
        return EA_INVALID_PIPE;
    }

    // FS - 24/02/1997
    // Lookup the pipe in WaitCmds
    // ----------------------------
    if ( !LookupWaitCmdTab( PmPipeOutMask, 0L, &LcPipeIndex ) )
    {
        COMReleaseExmut();
        return EA_INVALID_PIPE;
    }

    // Locate the request / reply command block assigned to the pipe
    // -------------------------------------------------------------
    LcWaitEndOfPlayReq  = &(WaitCmds[LcPipeIndex].wcbWaitEndPlayRequest) ;
    LcWaitEndOfPlayResp = &(WaitCmds[LcPipeIndex].wcbWaitEndPlayReply) ;

    if (LcWaitEndOfPlayResp->rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_WAIT_REQUEST_REFUSED;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( ASYNC_EVT_FAM,
                    WAIT_ENDOF_PLAY_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( WAIT_ENDOF_PLAY_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcWaitEndOfPlayReq );

    // filling the other fields of the request
    // ---------------------------------------
    LcWaitEndOfPlayReq->wpqOutPipeMask64.QuadPart = PmPipeOutMask;

    // sending the request to the driver
    // ---------------------------------
#if defined (WIN32)
    WaitCmds[LcPipeIndex].wcbWaitEndPlayOverlapped.hEvent = PmEvent;
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitEndOfPlayReq,
                   sizeof( WAIT_ENDOF_PLAY_REQ_INFO ),
                   &LcWaitEndOfPlayResp,
                   &(WaitCmds[LcPipeIndex].wcbWaitEndPlayOverlapped) );
#else
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitEndOfPlayReq,
                   sizeof( WAIT_ENDOF_PLAY_REQ_INFO ),
                   &LcWaitEndOfPlayResp,
                   PmEvent );
#endif

    LcError =  LcWaitEndOfPlayResp->rhCptr;
    COMReleaseExmut();
    return( LcError );
}


// ****************************************************************************
// WORD PCXWaitPipeRecordEnd()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Asked pipe mask.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Asynchronous function to ask the driver to tell when a conversion
// of a specified record pipe is finished.
//
// ****************************************************************************
WORD _CDECL_ PCXWaitPipeRecordEnd(
    IN PCX_HANDLE PmMyProcess   ,
    IN DWORD PmPipeInMask       ,
    IN HANDLE PmEvent           )
{
    LPWAIT_ENDOF_RECORD_REQ_INFO    LcWaitEndOfRecordReq;
    LPRESP_HEADER_INFO              LcWaitEndOfRecordResp;
    WORD                            LcPipeIndex;
    WORD                            LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if ( PmPipeInMask == 0 )
        return EA_INVALID_PIPE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // FS - 24/02/1997
    // Lookup the pipe in WaitCmds
    // ----------------------------
    if ( !LookupWaitCmdTab( 0L, PmPipeInMask, &LcPipeIndex ) )
    {
        COMReleaseExmut();
        return EA_INVALID_PIPE;
    }

    // Locate the request / reply command block assigned to the pipe
    // -------------------------------------------------------------
    LcWaitEndOfRecordReq  = &(WaitCmds[LcPipeIndex].wcbWaitEndRecordRequest) ;
    LcWaitEndOfRecordResp = &(WaitCmds[LcPipeIndex].wcbWaitEndRecordReply) ;

    if (LcWaitEndOfRecordResp->rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_WAIT_REQUEST_REFUSED;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( ASYNC_EVT_FAM,
                    WAIT_ENDOF_RECORD_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( WAIT_ENDOF_RECORD_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcWaitEndOfRecordReq );

    // filling the other fields of the request
    // ---------------------------------------
    LcWaitEndOfRecordReq->wrqInPipeMask64.QuadPart = PmPipeInMask;

    // sending the request to the driver
    // ---------------------------------
#if defined (WIN32)
    WaitCmds[LcPipeIndex].wcbWaitEndRecordOverlapped.hEvent = PmEvent;
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitEndOfRecordReq,
                   sizeof( WAIT_ENDOF_RECORD_REQ_INFO ),
                   &LcWaitEndOfRecordResp,
                   &(WaitCmds[LcPipeIndex].wcbWaitEndRecordOverlapped) );
#else
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitEndOfRecordReq,
                   sizeof( WAIT_ENDOF_RECORD_REQ_INFO ),
                   &LcWaitEndOfRecordResp,
                   PmEvent );
#endif

    LcError = LcWaitEndOfRecordResp->rhCptr;
    COMReleaseExmut();
    return( LcError );
}

// ****************************************************************************
// BOOLEAN PCXBufferEmpty()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD    PmPipeOutMask: the pipe mask
//
// WORD PmBufferNumber: number of the asked buffer.
//
// Return value :
// **************
//           1 : This buffer is available for the application.
//           0 : The parameters (PmMyProcess, PmPipeOutMask, PmBufferNumber)
//               does not correspond to a buffer or it corresponds and the
//               buffer is managed by the driver.
//
// ****************************************************************************
//
// Functions to know whether a buffer is empty (play).
//
// ****************************************************************************

BOOLEAN _CDECL_ PCXBufferEmpty(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN WORD PmBufferNumber)
{
    LPRESP_HEADER_INFO  LcPlayResp;
    WORD                LcIndex;
    BOOLEAN             LcRet;

    if (!COMWaitExmut())
       return FALSE;
    // FS - 03/10/1996
    // 1. look up the result of last async cmd (assuming it is a play)
    //   in the reply block assigned to the application and its buffer
    // 2. remove test against MAX_BUFFER and replace by a test
    //   against hte number of buffers the application has really reserved
    //   see further...

    // Look up the pipe
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitCmdTab( PmPipeOutMask, 0L, &LcIndex ) )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    if ( PmBufferNumber >= WaitCmds[LcIndex].wcbNbBuff )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Then retrieve the reply block allocated
    // for the application
    // ---------------------------------------
    LcPlayResp = &( WaitCmds[LcIndex].wcbReplyTab[PmBufferNumber].aapPlayStream );

    LcRet = (LcPlayResp->rhT == PROCESSED);
    COMReleaseExmut();
    return LcRet;
}


// ****************************************************************************
// BOOLEAN PCXBufferFull()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD    PmPipeInMask: the pipe mask
//

// WORD PmBufferNumber: number of the asked buffer.
//
// Output parameters
// *****************
//
// LPDWORD PmDataLength: number of byte in the buffer.
//
// Return value :
// **************
//           1 : This buffer is available for the application.
//           0 : The parameters (PmMyProcess, PmPipeOutMask, PmBufferNumber)
//               does not correspond to a buffer or it corresponds and the
//               buffer is managed by the driver.
//
// ****************************************************************************
//
// Functions to know whether a buffer is full (record).
//
// ****************************************************************************

BOOLEAN _CDECL_ PCXBufferFull(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN WORD PmBufferNumber,
    OUT LPDWORD PmDataLength)
{
    LPRECORD_RESP_INFO  LcRecordResp;
    WORD                LcIndex;
    BOOLEAN             LcTest ;

    // FS - 18/02/1997
    // DataLength must be coherent
    // if ever the buffer is not valid
    // --------------------------------
    *PmDataLength = 0 ;

    if (!COMWaitExmut())
       return FALSE;
    // FS - 03/10/1996
    // 1. look up the result of last async cmd (assuming it is a record)
    //   in the reply block assigned to the application and its buffer
    // 2. remove test against MAX_BUFFER and replace by a test
    //   against the number of buffers the application has really reserved
    //   see further...

    // Look up the pipe mask
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitCmdTab( 0L, PmPipeInMask, &LcIndex ) )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Check that the application has really reserved a buffer
    // with such a number
    // -------------------------------------------------------
    if ( PmBufferNumber >= WaitCmds[LcIndex].wcbNbBuff )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Then retrieve the reply block allocated
    // for the application
    // ---------------------------------------
    LcRecordResp = &( WaitCmds[LcIndex].wcbReplyTab[PmBufferNumber].aapRecordStream);

    // FS - 06/02/97
    // we must test only once the T bit to make
    // sure, the reply is coherent
    // ----------------------------------------
    LcTest = ( (LcRecordResp->RespHeader).rhT == PROCESSED ) ;

    if ( LcTest )
    {
        *PmDataLength = LcRecordResp->rrpDataLength;
    }

    COMReleaseExmut();
    return( LcTest );
}

// ****************************************************************************
// WORD PCXPlayFinished()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmOutPipeMask: Asked pipe mask.
//
// Return value :
// **************
// Functions to know whether a play of a pipe is finished.
// This function return :
//           1 : This pipe exists and there is no PCXWaitPipePlayEnd running.
//               If no PCXWaitPipePlayEnd was called on this pipe since the
//               creation or the last stop, returns 1.
//           0 : The couple (PmMyProcess, PmOutPipeMask) does not correspond
//               to a pipe or it corresponds and a PCXWaitPipePlayEnd was
//               called by the application and the play is not finished.
// ****************************************************************************
BOOLEAN _CDECL_ PCXPlayFinished(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmOutPipeMask)
{
    WORD    LcIndex;
    BOOLEAN LcRet;

    if (!COMWaitExmut())
       return FALSE;

    // Look up the pipe mask
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitCmdTab( PmOutPipeMask, 0L, &LcIndex ) )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Check the application owns this pipe and it has issued
    // a PCXWaitPipePlayEnd() command.
    // ------------------------------------------------------
    if ((WaitCmds[LcIndex].wcbWaitEndPlayRequest).Header.hdHandle != PmMyProcess)
    {
        COMReleaseExmut();
        return FALSE;
    }

    // 1st check passed, now let's have a look
    // on the command completion status
    // ---------------------------------------
    LcRet = ((WaitCmds[LcIndex].wcbWaitEndPlayReply).rhT == PROCESSED);
    COMReleaseExmut();
    return ( LcRet );

}

// ****************************************************************************
// BOOLEAN PCXRecordFinished()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmInPipeMask: Asked pipe mask.
//
// Return value :
// **************
// Functions to know whether a conversion (off-line record)
// of a pipe is finished.
//
// This function returns :
//           1 : This pipe exists and there is no PCXWaitPipeRecordEnd running.
//               If no PCXWaitPipeRecordEnd was called on this pipe since the
//               creation or the last stop, returns 1.
//           0 : The couple (pmMyProcess, pmInPipeMask) does not correspond
//               to a pipe or it corresponds and a PCXWaitPipeRecordEnd was
//               called by the application and the conversion is not finished.
// *****************************************************************************
BOOLEAN _CDECL_ PCXRecordFinished(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmInPipeMask)
{
    WORD    LcIndex;
    BOOLEAN LcRet;

    if (!COMWaitExmut()) 
       return FALSE;
    // Look up the pipe mask
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitCmdTab( 0L, PmInPipeMask, &LcIndex ) )
    {
        COMReleaseExmut();
        return FALSE;
    }

    // Check the application owns this pipe and it has issued
    // a PCXWaitPipeRecordEnd() command.
    // ------------------------------------------------------
    if ( (   WaitCmds[LcIndex].wcbWaitEndRecordRequest).Header.hdHandle
          != PmMyProcess)
    {
        COMReleaseExmut();
        return FALSE;
    }

    // 1st check passed, now let's have a look
    // on the command completion status
    // ---------------------------------------
    LcRet = ( (WaitCmds[LcIndex].wcbWaitEndRecordReply).rhT == PROCESSED );
    COMReleaseExmut();
    return ( LcRet );

}


// ****************************************************************************
// WORD PCXWaitForError()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if if the command has been sent successfully to the driver,
//  an error code otherwise.
//
// ****************************************************************************
//
// Asynchronous function to ask the driver to tell when an error occured.
// Register a request/reply block for the application.
//
// ****************************************************************************

WORD _CDECL_ PCXWaitForError(
    IN PCX_HANDLE PmMyProcess,
    IN HANDLE PmEvent)
{
    LPBC_HEADER_INFO        LcRequest;      //  modified to allow many
    LPWAIT_ERROR_RESP_INFO  LcReply;        //  applications waiting
                                            //  for errors
    WORD                    LcAppliIdx;     // FS - 02/10/1996
    WORD                    LcError;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // Look up the application handle
    // in the tab of asynchronous command blocks
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        //
        COMReleaseExmut();
        return EA_INVALID_PCX_HANDLE;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    LcRequest = (LPBC_HEADER_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitErrorRequest );
    LcReply   = (LPWAIT_ERROR_RESP_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitErrorReply );

    if ((LcReply->RespHeader).rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_WAIT_REQUEST_REFUSED;
    }

    // filling the header of the request block
    COMFillBcHeader( ASYNC_EVT_FAM,
                    WAIT_ERROR_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( WAIT_ERROR_REQ_INFO ),
                    sizeof( WAIT_ERROR_RESP_INFO ),
                    &LcRequest );

    // sending the request to the driver
    // ---------------------------------
#if defined (WIN32)
    WaitErrorCmds[LcAppliIdx].acbWaitErrorOverlapped.hEvent = PmEvent;
    COMCallDriver( LcRequest,
                   sizeof( WAIT_ERROR_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   &(WaitErrorCmds[LcAppliIdx].acbWaitErrorOverlapped) );
#else
    COMCallDriver( LcRequest,
                   sizeof( WAIT_ERROR_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   PmEvent );
#endif

    LcError = (LcReply->RespHeader).rhCptr;
    COMReleaseExmut();
    return( LcError );
}

// ****************************************************************************
// WORD PCXGetError()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// Output parameters:
// ******************
//
// LPDWORD PmInPipeMask: Mask of the Input pipe concerned by the error
//                      (if it is significant).
//
// LPDWORD PmOutPipeMask: Mask of the Output pipe concerned by the error
//                      (if it is significant).
//
// LPDWORD PmStreamMask: Mask of the stream concerned by the error
//                      (if it is significant).
//
// Return value :
// **************
//
// This function returns :
//           0 : The parameter PmMyProcess does not correspond to a
//               PCXWaitForError or it corresponds and there is no error.
//           else : The error.
//
// ****************************************************************************
//
// Functions to know whether a buffer is an error has occured.
//
// ****************************************************************************

WORD _CDECL_ PCXGetError(
    IN PCX_HANDLE PmMyProcess,
    OUT LPDWORD PmInPipeMask,
    OUT LPDWORD PmOutPipeMask,
    OUT LPDWORD PmStreamMask)
{
    LPBC_HEADER_INFO        LcRequest;  //  modified to allow many
    LPWAIT_ERROR_RESP_INFO  LcReply;    //  applications waiting
    WORD                    LcAppliIdx; // FS - 02/10/1996
    WORD                    LcError;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // Look up the application handle
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        // or did not reserve any buffer
        //
        COMReleaseExmut();
        return SUCCESS;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcRequest = (LPBC_HEADER_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitErrorRequest );
    LcReply   = (LPWAIT_ERROR_RESP_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitErrorReply );

    *PmInPipeMask = 0;
    *PmOutPipeMask = 0;
    *PmStreamMask = 0;

    // Check PCXWaitForError command completion
    // ----------------------------------------
    if ( (LcReply->RespHeader).rhT == PROCESSED )
    {
      if(LcReply->RespHeader.rhCptr == ED_CANCELLED)
      {
        NPNTAPITRACE("LXESAPI: PCXWaitForError was CANCELLED (Thread exit) !\n");
        LcError = SUCCESS;
      }
      else
      {
         _ASSERT( LcReply->wepInPipeMask64.HighPart == 0 );  // not handled yet
         _ASSERT( LcReply->wepOutPipeMask64.HighPart == 0 ); // not handled yet

        *PmInPipeMask = LcReply->wepInPipeMask64.LowPart;
        *PmOutPipeMask = LcReply->wepOutPipeMask64.LowPart;
        *PmStreamMask = LcReply->wepStreamMask;

        LcError = LcReply->wepError ;
      }
    }
    else
    {
        LcError = SUCCESS ;
    }

    COMReleaseExmut();
    return LcError;
}

// ****************************************************************************
// WORD PCXWaitForNotification()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE   PmMyProcess:    Handle of the calling process.
// DWORD        PmInPipeMask:   Mask of the Input pipe concerned by the notify
//                      (if it is significant).
// DWORD        PmOutPipeMask:  Mask of the Input pipe concerned by the notify
//                      (if it is significant).
// HANDLE       PmEvent:        Event triggered by the notification
//
// Return value :
// **************
//
// ****************************************************************************
//
// Function to know whether an async command is actually completed
//
// ****************************************************************************
WORD _CDECL_ PCXWaitForNotification(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmInPipeMask,
    IN  DWORD PmOutPipeMask,
    IN  HANDLE PmEvent)
{
    LPWAIT_NOTIFY_REQ_INFO  LcWaitNotifyReq;
    LPRESP_HEADER_INFO      LcWaitNotifyResp;
    WORD                    LcPipeIndex;
    WORD                    LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Extract pipe number from the mask
    // ---------------------------------
    if ( ( PmOutPipeMask == 0 ) && ( PmInPipeMask == 0 ) )
    {
        COMReleaseExmut();
        return EA_INVALID_PIPE;
    }

    // FS - 24/02/1997
    // Lookup the pipe in WaitCmds
    // ----------------------------
    if ( !LookupWaitCmdTab( PmOutPipeMask, PmInPipeMask, &LcPipeIndex ) )
    {
        COMReleaseExmut();
        return EA_INVALID_PIPE;
    }

    // Locate the request / reply command block assigned to the pipe
    // -------------------------------------------------------------
    LcWaitNotifyReq  = &(WaitCmds[LcPipeIndex].wcbWaitNotifyRequest) ;
    LcWaitNotifyResp = &(WaitCmds[LcPipeIndex].wcbWaitNotifyReply) ;

    if (LcWaitNotifyResp->rhT != PROCESSED)
    {
            COMReleaseExmut();
            return EA_WAIT_REQUEST_REFUSED;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( ASYNC_EVT_FAM,
                    WAIT_NOTIFICATION_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( WAIT_NOTIFY_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcWaitNotifyReq );

    // filling the other fields of the request
    // ---------------------------------------
    LcWaitNotifyReq->wnrOutPipeMask64.QuadPart   = PmOutPipeMask;
    LcWaitNotifyReq->wnrInPipeMask64.QuadPart    = PmInPipeMask;

    // sending the request to the driver
    // ---------------------------------
#if defined (WIN32)
    WaitCmds[LcPipeIndex].wcbWaitNotifyOverlapped.hEvent = PmEvent;
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitNotifyReq,
                   sizeof( WAIT_NOTIFY_REQ_INFO ),
                   &LcWaitNotifyResp,
                   &(WaitCmds[LcPipeIndex].wcbWaitNotifyOverlapped) );
#else
    COMCallDriver( (LPBC_HEADER_INFO) LcWaitNotifyReq,
                   sizeof( WAIT_NOTIFY_REQ_INFO ),
                   &LcWaitNotifyResp,
                   PmEvent );
#endif

    LcError = LcWaitNotifyResp->rhCptr;
    COMReleaseExmut();
    return( LcError );
}


// ****************************************************************************
// WORD PCXWaitForBoardEvent()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if if the command has been sent successfully to the driver,
//  an error code otherwise.
//
// ****************************************************************************
//
// Asynchronous function to ask the driver to tell when a PCCARD event such as
// insertion of a new PCXPocket or removal of a board occured.
//
// Register a request/reply block for the application.
//
// ****************************************************************************
WORD _CDECL_ PCXWaitForBoardEvent(
    IN PCX_HANDLE           PmMyProcess,
    IN HANDLE               PmEvent)
{
    WORD                       LcError = WA_COMMAND_NOT_AVAILABLE;
    LPWAIT_BOARD_PNP_REQ_INFO  LcRequest;
    LPWAIT_BOARD_PNP_RESP_INFO LcReply;
    WORD                       LcAppliIdx;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Look up the application handle
    // in the tab of asynchronous command blocks
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        //
        COMReleaseExmut();
        return EA_INVALID_PCX_HANDLE;
    }

    // Initialize reply bloc
    //
    LcError = SUCCESS;

    // Then retrieve the request / reply blocks allocated
    // for the application
    LcRequest = (LPWAIT_BOARD_PNP_REQ_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitBoardRequest );
    LcReply   = (LPWAIT_BOARD_PNP_RESP_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitBoardReply );

        // Send the wait request if none has been already
        // sent
        //
        if ((LcReply->RespHeader).rhT != PROCESSED)
        {
                COMReleaseExmut();
                return EA_WAIT_REQUEST_REFUSED;
        }

        // filling the header of the request block
        COMFillBcHeader( ASYNC_EVT_FAM,
                    WAIT_BOARD_PNP_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( WAIT_BOARD_PNP_REQ_INFO ),
                    sizeof( WAIT_BOARD_PNP_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcRequest );

        // sending the request to the driver
        // ---------------------------------
#if defined(WIN32)
        WaitErrorCmds[LcAppliIdx].acbWaitBoardPnpOverlapped.hEvent = PmEvent;
        COMCallDriver( (LPBC_HEADER_INFO) LcRequest,
                   sizeof( WAIT_BOARD_PNP_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   &(WaitErrorCmds[LcAppliIdx].acbWaitBoardPnpOverlapped) );
#else
        COMCallDriver( (LPBC_HEADER_INFO) LcRequest,
                   sizeof( WAIT_BOARD_PNP_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   PmEvent );
#endif

        LcError = (LcReply->RespHeader).rhCptr;

    COMReleaseExmut();

    return( LcError );
}


// ****************************************************************************
// WORD PCXIsBoardEventSet()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// Output parameters:
// ******************
//
// LPWORD  PmBoardMask: Mask of the board concerned by the event
//
// LPDWORD PmBoardEvent: Event occured
//
// Return value :
// **************
//
// This function returns :
//           1 : The parameter PmMyProcess does correspond to a
//              PCXWaitForBoardEvent call, and an event has occured
//              since this call.
//           0 : The parameter PmMyProcess does not correspond to
//              a PCXWaitForBoardEvent call or it corresponds but
//              no event has occured yet.
// ****************************************************************************
//
// Functions to know whether a PCCARD board has been inserted/removed.
//
// ****************************************************************************

BOOLEAN _CDECL_ PCXIsBoardEventSet(
    IN PCX_HANDLE           PmMyProcess,
    OUT LPWORD               PmBoardMask,
    OUT LPDWORD              PmBoardEvent )
{
    LPWAIT_BOARD_PNP_REQ_INFO  LcRequest;
    LPWAIT_BOARD_PNP_RESP_INFO LcReply;
    WORD                    LcAppliIdx;

    if (!COMWaitExmut())
       return FALSE;
    // Look up the application handle
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        // or did not reserve any buffer
        //
        COMReleaseExmut();
        return FALSE;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcRequest = (LPWAIT_BOARD_PNP_REQ_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitBoardRequest );
    LcReply   = (LPWAIT_BOARD_PNP_RESP_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitBoardReply );

    // Check PCXWaitForError command completion
    // ----------------------------------------
    if ( (LcReply->RespHeader).rhT == PROCESSED )
    {
      if(LcReply->RespHeader.rhCptr == ED_CANCELLED)
      {
        NPNTAPITRACE("LXESAPI: PCXWaitForBoardEvent was CANCELLED (Thread exit) !\n");
        *PmBoardMask = 0;
        *PmBoardEvent = BOARD_EVENT_NOTHING;
      }
      else
      {
        *PmBoardMask = LcReply->wprBoardMask;
        *PmBoardEvent = LcReply->wprEvent;
      }

        COMReleaseExmut();
        return ( TRUE );
    }
    else
    {
        COMReleaseExmut();
        return( FALSE );
    }
}

// ***************************************************************************
// WORD PCXGetSerialNumberA()
// WORD PCXGetSerialNumberW()
// WORD PCXGetSerialNumber()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:           the mask of boards in the scope of the request
//  WORD    PmSerialNumberMaxSize: length of the PmPSerialNumber string
//
// Output parameters :
// *******************
//
//  LPCHAR   PmPSerialNumber:    the serial number of the board (array of
//                              SERIAL_NUMBER_LEN+1 characters)
//
// ***************************************************************************
//
// Function to get back the board serial number.
//
// ***************************************************************************
WORD _CDECL_ PCXGetSerialNumberA(
    IN  WORD   PmBoardMask,
    IN  WORD   PmSerialNumberMaxSize,
    OUT LPSTR  PmPSerialNumber )
{
    return EA_COMMAND_NOT_AVAILABLE;
}


WORD _CDECL_ PCXGetSerialNumberW(
    IN  WORD   PmBoardMask,
    IN  WORD   PmSerialNumberMaxSize,
    OUT LPWSTR PmPSerialNumber )
{
    return EA_COMMAND_NOT_AVAILABLE;
}

// ****************************************************************************
// WORD PCXGetOEMInformation()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:        the mask of boards in the scope of the request
//  BYTE    PmOEMNumber:        the OEM number (from 0 to 127)
//
// Output parameters :
// *******************
//
//  LPBYTE   PmPOEMInfo:         the OEM information (bits 4 to 7 are irrelevant).
//
// ****************************************************************************
//
// Function to get back the board OEM information.
//
// ****************************************************************************
WORD _CDECL_ PCXGetOEMInformation(
    IN  WORD   PmBoardMask,
    IN  BYTE   PmOEMNumber,
    OUT LPBYTE  PmPOEMInfo )
{
    return EA_COMMAND_NOT_AVAILABLE;
}


// ****************************************************************************
// WORD PCXGetChecksum()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:        the mask of boards in the scope of the request
//
// Output parameters :
// *******************
//
//  PBYTE    PmPChecksum:       the checksum
//
// ****************************************************************************
WORD _CDECL_ PCXGetChecksum(
    IN  WORD    PmBoardMask,
    OUT LPBYTE  PmPChecksum )
{
    return EA_COMMAND_NOT_AVAILABLE;

}

//******************************************************************************************
//                          DIAGNOSTIC FUNCTIONS
//******************************************************************************************

// ****************************************************************************
// WORD PCXGetTimeCode()
// *************************
//
//  Input parameters :
// *******************
//
//  PCX_HANDLE      PmMyProcess:       Handle of the calling process.
//  WORD            PmBoardMask      : the mask of boards in the scope of the request
//
// Output parameters :
// *******************
//
//  LPBOOLEAN       PmPNewTimeCode    : if TRUE, indicates that this is a new
//                                      time-code.
//  LPBOOLEAN       PmPBackward       : if TRUE, indicates that the time-code
//                                      direction is backward.
//  LPBOOLEAN       PmPWaitingTimeCode: if TRUE, indicates that the board is
//                                     scanning for a specific time-code
//  PPCX_TIME       PmPPcxTime        : the board reference time.
//  PPCX_TIME_CODE  PmPTimeCode       : the time-code.
//
// ****************************************************************************
//
// Read the time-code on a board.
//
// ****************************************************************************
WORD _CDECL_ PCXGetTimeCode(
    IN  PCX_HANDLE      PmMyProcess,
    IN  WORD            PmBoardMask,
    OUT LPBOOLEAN       PmPNewTimeCode,
    OUT LPBOOLEAN       PmPBackward,
    OUT LPBOOLEAN       PmPWaitingTimeCode,
    OUT PPCX_TIME       PmPPcxTime,
    OUT PPCX_TIME_CODE  PmPTimeCode )
{
    PTIME_CODE_REQ_INFO   LcPRequest = 0;   // will point to the request
    PTIME_CODE_RESP_INFO  LcPReply   = 0;   // will point to the reply
    DSP_TO_LOAD_INFO      LcDspLoadInfo;
    WORD                  LcError;
    BOOLEAN               LcDspAlreadyOwned;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // Before sending TIME_CODE request to the driver
    // we must make sure that TimeCode is actually valid
    // on DSP software
    // -------------------------------------------------
    MEMSET2( (&LcDspLoadInfo), 0, DSP_TO_LOAD_INFO, 1 );
    LcDspLoadInfo.dtlCardMask    = PmBoardMask ;
    LcDspLoadInfo.dtlDspMask     = 0x01 ;          // Only first DSP is concerned
    LcDspLoadInfo.dtlDspFeatures = DSP_CAN_TIME_CODE_MASK ;

    LcError = DspAllocFct( PmMyProcess, &LcDspLoadInfo, 0, NULL, &LcDspAlreadyOwned );

    if ( LcError != SUCCESS )
    {
        return( LcError );
    }

    // Send a TIME_CODE request to the driver
    // --------------------------------------
    if (!COMWaitExmut()) return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                     TIME_CODE_CMD,
                     (APP_HANDLE) PmMyProcess,
                     sizeof( TIME_CODE_REQ_INFO ),
                     sizeof( TIME_CODE_RESP_INFO ),
                     (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->tcoBoardMask    = PmBoardMask;
    LcPRequest->tcoRequest      = TIME_CODE_CMD_GET;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcError = LcPReply->RespHeader.rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        // Fill optional output parameters
        // -------------------------------
        if ( PmPNewTimeCode )
        {
            *PmPNewTimeCode     = (BOOLEAN) LcPReply->tcoNewTimeCode;
        }
        if ( PmPBackward )
        {
            *PmPBackward        = (BOOLEAN) LcPReply->tcoBackward;
        }
        if ( PmPPcxTime )
        {
            *PmPPcxTime         = LcPReply->tcoPcxTime;
        }
        if ( PmPTimeCode )
        {
            *PmPTimeCode        = LcPReply->tcoTimeCode;
        }
        if ( PmPWaitingTimeCode )
        {
            *PmPWaitingTimeCode = (BOOLEAN) LcPReply->tcoConfigInProgress;
        }
    } // end of if there was no error

    COMReleaseExmut();

    // Free the DSP if it has been allocated here
    //
    if ( !LcDspAlreadyOwned )
    {
        (VOID) PCXDspFree( PmMyProcess, &LcDspLoadInfo );
    }

    return( LcError );
}

// ****************************************************************************
// WORD PCXGetBoardLevelsFeatures()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmBoardMask:        the mask of boards in the scope of the request
//  DWORD   PmLevelTypeMask:    the mask giving the features we want to know
//
// Output parameters :
// *******************
//
//  PLEVEL_FEATURE_INFO        an array where the results are stored
//
// ****************************************************************************
//
// Function to get back a level description of a board inputs and outputs.
//
// CAUTION: the caller expects to find in the resultat array:
//          - all the entry with the lfValidity field equal to TRUE,
//          and then
//          - all the entry with the lfValidity field equal to FALSE,
//
// ****************************************************************************
WORD _CDECL_ PCXGetBoardLevelsFeatures(
    IN  WORD                 PmBoardMask,
    IN  DWORD                PmLevelTypeMask,
    OUT PLEVEL_FEATURE_INFO PmPLevelsFeatures)
{
    LPBOARD_HARD_REQ_INFO   LcPRequest = 0;   // will point to the request
    LPBOARD_HARD_RESP_INFO  LcPReply = 0;     // will point to the reply
    WORD                    LcError;

    if (!COMWaitExmut()) return EA_CANT_TAKE_MUTEX;

    // This might be the first call to the api
    // Check the driver is installed
    // ----------------------------------------
    if ( (LcError = GENInitAndCheckVersion( TRUE, NULL ) ) != SUCCESS )
    {
        COMReleaseExmut();
        return LcError;
    }

    // Send a board Hardware features request to the driver
    //

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOARD_HARD_FEATURE_CMD,
                    (APP_HANDLE) 0,
                    sizeof( BOARD_HARD_REQ_INFO ),
                    sizeof( BOARD_HARD_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPRequest );

    // filling the other fields of the request
    // ---------------------------------------
    LcPRequest->bhrBoardMask = PmBoardMask;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   0,
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // Reading the response and filling the out parameter
    // if no error has been set
    // --------------------------------------------------
    LcError = LcPReply->RespHeader.rhCptr;
    if ( !( LcError & ERROR_MASK) )
    {
        BOOLEAN LcFound    = FALSE; // Board found in TabLevelFeature array.
        BYTE    LcIndex    = 0;     // Nb of entry copied in PmPLevelsFeatures.
        BYTE    LcBoardType;
        BYTE    LcExtendedBoardType;
        WORD    i;
        BYTE    LcFlavorNum;
        PLEVEL_FEATURE_INFO LcPLevel;

        LcBoardType = LcPReply->bhfBoardType;
        LcExtendedBoardType = LcPReply->bhfExtendedType;

        // Extract Flavor number From CSID
        //
        LcFlavorNum = (BYTE) EXTRACT_FLAVOR_NUM( LcPReply->bhfCSID );

		if( (LcBoardType > HR_FIRST) && (LcBoardType < HR_LAST) )
		{
			LcBoardType = HR_FIRST;	// HR boards have all same level features !!!
		}

        // Search in the TabLevelFeature array the entries related to
        // our board, and if thoses entries are in the PmLevelTypeMask mask,
        // fill with them the PmPLevelsFeatures array.
        //
        for ( i = 0 ; i < BOARD_LEVEL_INFO_ENTRIES ; i++ )
        {
            LcPLevel = & (TabLevelFeature[i].blLevelFeatures);

            // The entry is related to our board.
            //
            if (    ( TabLevelFeature[i].blBoardType == LcBoardType )
                 && ( TabLevelFeature[i].blExtendedBoardType == LcExtendedBoardType )
                 && (    ( TabLevelFeature[i].blFlavorNum == ALL_FLAVOR )
                      || ( TabLevelFeature[i].blFlavorNum == LcFlavorNum ) ) )
            {
                LcFound = TRUE;

                // The entry is wanted by the caller.
                //
                if ( ( (LcPLevel->lfLevelType) & PmLevelTypeMask ) != 0 )
                {
					// HR boards have all same level features, but
					// handle digital only boards with analog daughter board (VX881, PCX881, VX1221, PCX1221)
					// handle analog+digital boards without analog daughter board  (VX882, PCX882, VX1222, PCX1222)
					if( LcPLevel->lfLevelType == LEVEL_FEATURE_OUTPUT_LINE_MASK )
						if( (LcPReply->bhfOutAudio[0] & AUDIO_LEVEL_PRESENT_MASK) == 0)
							continue;	// for(i)
					if( LcPLevel->lfLevelType == LEVEL_FEATURE_INPUT_LINE_MASK )
						if( (LcPReply->bhfInAudio[0] & AUDIO_LEVEL_PRESENT_MASK) == 0)
							continue;	// for(i)

                    // Retrieve the entry from the entry mask given by the caller.
                    //
                    PmLevelTypeMask &= ~(LcPLevel->lfLevelType);

                    // Copy the entry in the PmPLevelsFeatures array.
                    //
                    memcpy( PmPLevelsFeatures+LcIndex,
                            LcPLevel,
                            sizeof(LEVEL_FEATURE_INFO) );

                    // Update the number of copied entries.
                    //
                    LcIndex += 1;
                }
            }
        } // end search in TabLevelFeature array.

        if ( LcFound == FALSE )
        {
            LcError = EA_UNKNOWN_BOARD;
        }
        else
        {
            // For each  entry remaining in PmLevelTypeMask, clear an entry
            // in PmPLevelsFeatures with the field lfValidity equal to
            // FALSE.
            //
            while ( PmLevelTypeMask != 0 )
            {
                WORD LcBoardIndex = UTIDWMask2Word( PmLevelTypeMask ); // from 0 to ...

                // Add an null entry in the PmPLevelsFeatures array.
                //
                BZERO2( PmPLevelsFeatures+LcIndex,
                        LEVEL_FEATURE_INFO,
                        1 );
                PmPLevelsFeatures[LcIndex].lfLevelType = UTIMaskDWord( LcBoardIndex );

                // Update the number of copied entries.
                //
                LcIndex += 1;

                // Retrieve the entry from the mask.
                //
                PmLevelTypeMask &= ~UTIMaskDWord( LcBoardIndex );
            }
        }
    } // end of if there was no error

    COMReleaseExmut();

    return( LcError );
}

// ****************************************************************************
// WORD PCXGetBoardLevelsFeaturesEx()
// *************************
//
//  Input parameters :
// *******************
//
//  WORD    PmAudioInfo:        pointer to AUDIO_DECL_INFO
//  DWORD   PmLevelTypeMask:    the mask giving the features we want to know
//
// Output parameters :
// *******************
//
//  PLEVEL_FEATURE_INFO        an array where the results are stored
//
// ****************************************************************************
//
// Function to get back a level description of a board inputs and outputs.
//
// ****************************************************************************
WORD _CDECL_ PCXGetBoardLevelsFeaturesEx(
    IN  PAUDIO_DECL_INFO     PmAudioInfo,
    IN  DWORD                PmLevelTypeMask,
   OUT  PLEVEL_FEATURE_INFO  PmPLevelsFeatures )
{
    if(PmAudioInfo)
    {
        WORD LcBoardMask = (((WORD) 1) << PmAudioInfo->adBoardNumber);

        // in PCX driver we can ignore the AudioNumber because all AudioNumbers have the same fix gains, min max and step
        return PCXGetBoardLevelsFeatures(LcBoardMask,PmLevelTypeMask,PmPLevelsFeatures);
    }
    else
    {
        return EA_INVALID_POINTER;
    }
}

// ****************************************************************************
// WORD PCXGetManagingDsp()
// ************************
//
//  Input parameters :
// *******************
//
// PAUDIO_DECL_INFO PmAudioInfo:    description of pipe audios
//
// Output parameters :
// *******************
//
// LPBYTE PmDspMask                 DspMask for the one DSP managing the audio
//                                  or 0 if result is unknown
//
// ****************************************************************************
//
// Function to get back the number of DSP to load for using the specified
// physical audio. Note that we cannot give a reply for virtual audio if
// the DSP is not loaded yet
//
// ****************************************************************************
WORD _CDECL_ PCXGetManagingDsp(
    IN  PAUDIO_DECL_INFO    PmAudioInfo,
   OUT  LPBYTE              PmDspMask )
{
    CARD_FEATURES_INFO  LcBoardInfo;

    WORD                LcErrorCode = SUCCESS;
    WORD                LcBoard     = PmAudioInfo->adBoardNumber;

    *PmDspMask = 0;

    // Let's checked whether the DSP is already loaded or not
    // ------------------------------------------------------
    LcErrorCode = PCXGetHardwareBoardFeature( UTIMaskWord( LcBoard ),
                                              &LcBoardInfo );

    if ( LcErrorCode != SUCCESS )
    {
        return LcErrorCode;
    }

    if ( ( PmAudioInfo->adAudioAttributes & AUDIO_PHYSICAL ) == AUDIO_PHYSICAL )
    {
        // Physical audio
        // --------------
        // Only PCX80 boards require special cases
        //
        if ( LcBoardInfo.ciBoardType == PCX80 )
        {
            *PmDspMask = (BYTE) UTIMaskWord( (WORD)( PmAudioInfo->adAudioNumber / 4 ) ) ;
        }
        else
        {
            *PmDspMask = 0x01;
        }
    }
    else
    {
        // Virtual audio
        // --------------
        // In order to decode DSP number for managing a virtual audio
        // a DSP software must be already loaded. This feature cannot help
        // in any way the application...
        //
        LcErrorCode = WA_COMMAND_NOT_AVAILABLE ;
    }

    return LcErrorCode;
}

// ****************************************************************************
// WORD GENGetPipeClock()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// Output parameters
// *****************
//
// PCLOCK_INFO PmPipeClockInfo: the pipe clock description
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Only one pipe can be specified.
//
// ****************************************************************************
WORD GENGetPipeClock(
    IN  BOOLEAN         PmForceActual, /*force read of board clock */
    IN  PCX_HANDLE      PmMyProcess,
    IN  DWORD           PmPipeOutMask,
    IN  DWORD           PmPipeInMask,
    OUT PCLOCK_INFO     PmPipeClockInfo)
{
    LPPIPE_GET_CLOCK_REQ_INFO  LcPReq = 0;  // will point to the beginning
                                            // of the request block
    LPPIPE_GET_CLOCK_RESP_INFO LcPResp = 0; // will point to the beginning
                                            // of the response block
    WORD                       LcError = 0;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_STATUS_FAM,
                    PIPE_GET_CLOCK_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PIPE_GET_CLOCK_REQ_INFO ),
                    sizeof( PIPE_GET_CLOCK_RESP_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq);

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->scqOutputPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->scqInputPipeMask64.QuadPart  = PmPipeInMask;
    LcPReq->scqForceActual      = PmForceActual;


    // sending the request to the driver
    // ---------------------------------
    COMCallDriver(
        (LPBC_HEADER_INFO) LcPReq       ,
        0                               ,
        (RESP_HEADER_HANDLE) &LcPResp   ,
        NULL                           );

    LcError = LcPResp->RespHeader.rhCptr ;

    // Reset the reply structure
    // --------------------------
    BZERO2( PmPipeClockInfo, CLOCK_INFO, 1);

    // reading the response
    // --------------------
    if ( LcError == SUCCESS )
    {
        // transmit type and frequency information
        // ---------------------------------------
        PmPipeClockInfo->ckType = LcPResp->pcrSource;
        PmPipeClockInfo->ckFrequency = LcPResp->pcrFrequency;
        PmPipeClockInfo->ckSync= LcPResp->pcrSync;
        PmPipeClockInfo->ckStatusFlags = LcPResp->pcrFlags;
        PmPipeClockInfo->ckMisc = LcPResp->pcrMisc;

        if (LcPResp->pcrSRCRatio != 0) 
        {
            double LcTemp = (double)LcPResp->pcrFrequency * 64.0 / (double)(LcPResp->pcrSRCRatio);
            
            if (LcTemp > 32000*0.9) {
                if (LcTemp <= 44100*0.9)        PmPipeClockInfo->ckAudioClock = 32000;
                else if (LcTemp <= 48000*0.95)  PmPipeClockInfo->ckAudioClock = 44100;
                else if (LcTemp <= 50000)       PmPipeClockInfo->ckAudioClock = 48000;
                else if (LcTemp <= 64000*1.1)   PmPipeClockInfo->ckAudioClock = 64000;
                else if (LcTemp <= 88200*1.05)  PmPipeClockInfo->ckAudioClock = 88200;
                else if (LcTemp <= 96000*1.1)   PmPipeClockInfo->ckAudioClock = 96000;
                else if (LcTemp <= 176400*1.05) PmPipeClockInfo->ckAudioClock = 176400;
                else if (LcTemp <= 192000*1.1)  PmPipeClockInfo->ckAudioClock = 192000;
            }
        } // si les src sont en route
    }

    // returning the error code
    return( LcError);
}


// ****************************************************************************
// EXTERN VOID GENMpegEqualBandToIndex(...)
// ***********************************
//
// Input parameters :
// ****************
//
//  DWORD       PmBandNumber : band number
//
// Output parameters :
// *******************
//
//  LPDWORD     PmPMinIndex : min index of the band
//  LPDWORD     PmPMedIndex : medium index of the band
//  LPDWORD     PmPMaxIndex : max index of the band
//
// Return value :
// **************
//
//  None
//
// ****************************************************************************
//
// Give the indexes associated with a band.
// NOTA: all the explanations are given with the declaration of the
//       TabMpegEqual array.
//
// ****************************************************************************
EXTERN VOID    GENMpegEqualBandToIndex(
    IN      DWORD       PmBandNumber,
    OUT     LPDWORD     PmPMinIndex,
    OUT     LPDWORD     PmPMedIndex,
    OUT     LPDWORD     PmPMaxIndex )
{
    *PmPMinIndex = TabMpegEqual[PmBandNumber].meMinIndex;
    *PmPMedIndex = TabMpegEqual[PmBandNumber].meMedIndex;
    *PmPMaxIndex = TabMpegEqual[PmBandNumber].meMaxIndex;
}


// ****************************************************************************
// STATIC VOID MpegEqualLevelToCoeff(...)
// ****************************************
//
// Input parameters :
// ****************
//
//  FLOAT   PmLevel:      level to set (in dB)
//
// Output parameters :
// *******************
//
//  None.
//
// Return value :
// **************
//
// the coeff
//
// ****************************************************************************
//
// Translate a level into a coeff.
// NOTA: all the explanations are given with the declaration of the
//       TabMpegEqual array.
//
// ****************************************************************************
static DWORD MpegEqualLevelToCoeff(
    IN      FLOAT   PmLevel )
{
    double LcTmp   = (double)MPEG_EQUAL_A * pow( (double)MPEG_EQUAL_B, (double)PmLevel / (double)MPEG_EQUAL_C );
    DWORD  LcCoeff = (DWORD)LcTmp;

#if (MPEG_EQUAL_COEFF_MIN_VALUE != 0)
    LcCoeff = MAX( MPEG_EQUAL_COEFF_MIN_VALUE, LcCoeff );
#endif
    LcCoeff = MIN( MPEG_EQUAL_COEFF_MAX_VALUE, LcCoeff );

    return ( (DWORD)LcCoeff );
}


// ****************************************************************************
// EXTERN FLOAT GENMpegEqualCoeffToLevel(...)
// *************************************
//
// Input parameters :
// ****************
//
//  DWORD   PmCoeff
//
// Output parameters :
// *******************
//
//  None.
//
// Return value :
// **************
//
// the level (in dB)
//
// ****************************************************************************
//
// Translate a coeff into a level.
// NOTA: all the explanations are given with the declaration of the
//       TabMpegEqual array.
//
// ****************************************************************************
FLOAT    GENMpegEqualCoeffToLevel(
    IN      DWORD       PmCoeff )
{
#ifdef _MSC_VER
#pragma warning(disable:4723)       /* disable unwanted C warning */
#endif
    // Nota : we use log10 => we use that MPEG_EQUAL_B = 10.
    double LcLevel = log10( (double)PmCoeff / (double)MPEG_EQUAL_A ) * (double) MPEG_EQUAL_C;
#ifdef _MSC_VER
#pragma warning(default:4723)       /* reset warning behaviour */
#endif
    return ( (FLOAT)LcLevel );
}


// ****************************************************************************
// EXTERN VOID GENMpegEqualLevelToCoeffForUserBand(...)
// ***********************************************
//
// Input parameters :
// ****************
//
//  DWORD   PmBandNumber: user band number.
//  FLOAT   PmLevel:      level to set (in dB)
//
// Input/Output parameters :
// *************************
//
//  LPDWORD  PmCoeff:      array of MAX_EQUALIZATION_MPEG_COEFF coeff.
//
// Return value :
// **************
//
// None.
//
// ****************************************************************************
//
// Update all the coeff related to the user band with the level.
// NOTA: all the explanations are given with the declaration of the
//       TabMpegEqual array.
//
// ****************************************************************************
void GENMpegEqualLevelToCoeffForUserBand(
    IN      DWORD       PmBandNumber,
    IN      FLOAT       PmLevel,
    INOUT   LPDWORD     PmCoeff )
{
    DWORD               i;
    PMPEG_EQUAL_INFO    LcPInfo    = TabMpegEqual + PmBandNumber;
    DWORD               LcMinIndex = LcPInfo->meMinIndex;
    DWORD               LcMaxIndex = LcPInfo->meMaxIndex;
    DWORD               LcMedIndex = LcPInfo->meMedIndex;
    FLOAT               LcDelta;

    if ( PmBandNumber == MPEG_EQUAL_MAX_BANDS-1 )
    {
        LcDelta = PmLevel / ( (FLOAT)LcMedIndex - (FLOAT)LcMinIndex + 1 );
    }
    else
    {
        LcDelta = PmLevel / ( ( (FLOAT)LcMaxIndex - (FLOAT)LcMinIndex ) / 2 + 1 );
    }

    // Loop on the coeff to update in the first part of the user band.
    // ---------------------------------------------------------------
    //
    for ( i = LcMinIndex ; i <= LcMedIndex ; i++ )
    {
        PmCoeff[i] = MpegEqualLevelToCoeff( ( i - LcMinIndex + 1 ) * LcDelta );
    }

    // Loop on the coeff to update in the second part of the user band.
    // ----------------------------------------------------------------
    //
    if ( PmBandNumber == MPEG_EQUAL_MAX_BANDS-1 )
    {
        for ( i = LcMedIndex + 1 ; i <= LcMaxIndex ; i++ )
        {
            PmCoeff[i] = PmCoeff[LcMedIndex];
        }
    }
    else
    {
        for ( i = LcMedIndex + 1 ; i <= LcMaxIndex ; i++ )
        {
            PmCoeff[i] = MpegEqualLevelToCoeff( ( LcMaxIndex - i + 1 ) * LcDelta );
        }
    }
}


// ****************************************************************************
// EXTERN WORD GENGetOutStreamEqualization(...)
// *******************************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmPipeOutMask: Output pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//
// Output parameters :
// *****************
//
//  LPGET_OUT_STREAM_EFFECTS_RESP_INFO  *PmPPResp: the response block
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to get the equalization settings
//
// ****************************************************************************
WORD GENGetOutStreamEqualization(
    IN  PCX_HANDLE                          PmMyProcess,
    IN  DWORD                               PmPipeOutMask,
    IN  DWORD                               PmStreamMask,
    IN  PCX_EFFECT_HANDLE                   PmEffectID,
   OUT LPGET_OUT_STREAM_EFFECTS_RESP_INFO  *PmPPResp )
{
    LPGET_OUT_STREAM_EFFECTS_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                                  // request block
    WORD LcError;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_STATUS_FAM,
                     GET_OUT_STREAM_EFFECTS_CMD,
                     (APP_HANDLE)PmMyProcess,
                     sizeof( GET_OUT_STREAM_EFFECTS_REQ_INFO ),
                     sizeof( GET_OUT_STREAM_EFFECTS_RESP_INFO ),
                     (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->gseqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->gseqStreamMask    = PmStreamMask;
    LcPReq->gseqEffectId      = PmEffectID;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) PmPPResp,
                   0 );

    LcError = (*PmPPResp)->RespHeader.rhCptr;

    // The equalization is not active in the DSP. Return the default values.
    // Nota: the DSP return the error BERR_MPEG_EQUAL_NOT_ACTIVE, which is
    //       translated into the error EB_CMD_REFUSED by the driver function
    //       called TraitErr (see virtio.c).
    //
    if ( LcError == EB_CMD_REFUSED )
    {
		WORD    i;

        if (PmEffectID == EFFECT_EQUALIZATION_STREAM_MPEG)
		{
			DWORD   LcCoeff = MpegEqualLevelToCoeff( MPEG_EQUAL_DEFAULT_LEVEL );

			for ( i = 0 ; i < MAX_EQUALIZATION_MPEG_COEFF; i++ )
			{
				(*PmPPResp)->gserParam[i] = LcCoeff;
			}
		} else
		{
			for ( i = 0 ; i < MAX_EQUALIZATION_LINEAR_COEFF; i++ )
			{
				(*PmPPResp)->gserParam[i] = 0;	// gain 0dB
			}
		}

        LcError = SUCCESS;
    }

    // returning the error code
    return LcError;
}

// GPIO Functions 
// **********************

// ****************************************************************************
//
// ****************************************************************************

#define MAX_GPIO_PER_BOARD  32

WORD GPIOFct(
    IN  	PCX_HANDLE	    PmMyProcess,
    IN      DWORD           PmGpioRequest,
    INOUT   PGPIO_INFO   	PmTabGpioInfo,
    IN	    DWORD	   	    PmTabSize )
{
	LPGPIO_REQ_INFO     LcPRequest;   // will point to the request
    LPGPIO_RESP_INFO    LcPReply;     // will point to the reply
    WORD                LcError = SUCCESS;
	WORD			    LcMySize;
    DWORD               i;
    DWORD               j;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // --------------------------------------------------
    // Mark all the elements of the array as not treated.
    // --------------------------------------------------

    for ( i = 0 ; i < PmTabSize ; i++ )
    {
        if( &PmTabGpioInfo[i] == NULL )     // validate GPIO_INFO array
            return EA_INVALID_POINTER;

	    PmTabGpioInfo[i].gpioUnused0 = FALSE;
    }

    // -----------------------------------------------------------------------
    // Send to the driver only one request per NCX, because:
    // - PmTabGpioInfo can be greater than the maximum size of a
    //   request to the driver.
    // - All the GPIO related to a board must be read/written at the same time.
    // ------------------------------------------------------------------------

    // Loop on all the array elements.
    //
    for ( i = 0 ; i < PmTabSize ; i++ )
    {
        // The current element is not yet treated.
        //
        if ( PmTabGpioInfo[i].gpioUnused0 == FALSE )
        {
            WORD    LcBoardNum    = PmTabGpioInfo[i].gpioBoardNumber;
            DWORD   LcArrayInd[MAX_GPIO_PER_BOARD];
            DWORD   LcArrayIndNum = 0;  // Number of used elements in LcArrayInd/

            // ---------------------------------------------------
            // Search all the array elements having the same board
            // number as the current element.
            // ---------------------------------------------------

            for ( j = i ; j < PmTabSize ; j++ )
            {                   
                if ( PmTabGpioInfo[j].gpioBoardNumber == LcBoardNum )
                {
                    if ( LcArrayIndNum >= MAX_GPIO_PER_BOARD )
                    {
                        LcError = EA_TOO_MUCH_DATA;
                        break;
                    }
                    else
                    {
                        // Memorize the position of this element.
                        LcArrayInd[LcArrayIndNum]       = j;
                        LcArrayIndNum                  += 1;
                        PmTabGpioInfo[j].gpioUnused0    = TRUE; // Mark the element as treated
                    }
                }
            }

            if ( LcError != SUCCESS )
            {
                break;
            }
            if (!COMWaitExmut())
            {              
                LcError = EA_CANT_TAKE_MUTEX;
                break;
            }

            // ----------------------------------------------
            // Create the request to be sended to the driver.
            // ----------------------------------------------

	        LcMySize = (WORD)(sizeof( GPIO_REQ_INFO ) + sizeof( GPIO_INFO ) * LcArrayIndNum);

            LcPRequest = NULL;
            LcPReply   = NULL;

            COMFillBcHeader( GENE_FAM,
                             GPIO_CMD,
                             (APP_HANDLE) PmMyProcess,
                             LcMySize, 
                             LcMySize,
                             (BC_HEADER_HANDLE) &LcPRequest );

            LcPRequest->gpioRequest = PmGpioRequest;
	        LcPRequest->gpioTabSize = LcArrayIndNum;

            for ( j = 0 ; j < LcArrayIndNum ; j++ )
            {                   
                (void)memcpy( LcPRequest->gpioTabInfo + j, PmTabGpioInfo + LcArrayInd[j], sizeof( GPIO_INFO ) );
            }

            // ------------------------------
	        // Send the request to the driver
            // ------------------------------

            COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                           LcMySize,
                           (RESP_HEADER_HANDLE) &LcPReply,
                           0 );

            // ------------------
            // Read the response.
            // ------------------

            LcError = LcPReply->RespHeader.rhCptr;
            // if there is no error we copy back data if it was a PCXReadGpio (warning WD_MORE_DATA accepted!)
            if ( ( PmGpioRequest == GPIO_CMD_READ ) && ((LcError & ERROR_MASK) == 0) )
            {
                for ( j = 0 ; j < LcArrayIndNum ; j++ )
                {                   
                    (void)memcpy( PmTabGpioInfo + LcArrayInd[j], LcPReply->gpioTabInfo + j, sizeof( GPIO_INFO ) );
                    PmTabGpioInfo[j].gpioUnused0 = TRUE; // To be sure that the element stay marked as treated...
                }
            }

	        COMReleaseExmut();

            // break on error, but warnings are accepted (-> WD_MORE_DATA) !
            if ( LcError & ERROR_MASK )
            {
                break;
            }
        }
    }

    // ----------------------------
    // Restore the previous values.
    // ----------------------------

    for ( j = 0 ; j < PmTabSize ; j++ )
    {
	    PmTabGpioInfo[j].gpioUnused0 = 0;
    }

    // Close only the GPIOs correctly opened.
    // --------------------------------------
    if ( ( PmGpioRequest == GPIO_CMD_OPEN ) && ( LcError != SUCCESS ) && ( i > 0 ) )
    {
        (void)GPIOFct(
            PmMyProcess,
            GPIO_CMD_CLOSE,
            PmTabGpioInfo,
            i );
    }

    return( LcError );
}

// ****************************************************************************
//
// ****************************************************************************

WORD _CDECL_ PCXOpenGPIO(
    IN  PCX_HANDLE  PmMyProcess,
    IN  PGPIO_INFO  PmTabGpioInfo,
    IN  DWORD       PmTabSize,
    IN  PVOID       PmUnused )
{
    WORD                LcError = SUCCESS;

    LcError = GPIOFct(
        PmMyProcess,
        GPIO_CMD_OPEN,
        PmTabGpioInfo,
        PmTabSize );

    return( LcError );
}

// ****************************************************************************
//
// ****************************************************************************

WORD _CDECL_ PCXReadGPIO(
    IN      PCX_HANDLE  PmMyProcess,
    INOUT   PGPIO_INFO      PmTabGpioInfo,
    IN      DWORD           PmTabSize
 )
{
    WORD LcError = SUCCESS;

    LcError = GPIOFct(
        PmMyProcess,
        GPIO_CMD_READ,
        PmTabGpioInfo,
        PmTabSize );

    if( LcError == WD_MORE_DATA )   // not yet known to higher levels!
        LcError =  WA_MORE_DATA;

    return( LcError );
}

// ****************************************************************************
//
// ****************************************************************************

WORD _CDECL_ PCXWriteGPIO(
    IN  PCX_HANDLE	PmMyProcess,
    IN  PGPIO_INFO   	PmTabGpioInfo,
    IN	DWORD	   	PmTabSize
 )
{
    WORD LcError = SUCCESS;

    LcError = GPIOFct(
        PmMyProcess,
        GPIO_CMD_WRITE,
        PmTabGpioInfo,
        PmTabSize );

    return( LcError );
}

// ****************************************************************************
//
// ****************************************************************************

WORD _CDECL_ PCXCloseGPIO(
    IN  PCX_HANDLE	PmMyProcess,
    IN  PGPIO_INFO 	PmTabGpioInfo,
    IN	DWORD	   	PmTabSize,
    IN  PVOID       PmUnused )
{
    WORD LcError = SUCCESS;

    LcError = GPIOFct(
        PmMyProcess,
        GPIO_CMD_CLOSE,
        PmTabGpioInfo,
        PmTabSize );

    return( LcError );
}

// ****************************************************************************
// WORD PCXWaitForGpioEvent()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// HANDLE PmEvent: Handle of the event to rise under NT
//
// Return value :
// **************
// 0 if if the command has been sent successfully to the driver,
//  an error code otherwise.
//
// ****************************************************************************
//
// Asynchronous function to ask the driver to tell when a GPIO event occured.
//
// Register a request/reply block for the application.
//
// ****************************************************************************
WORD _CDECL_ PCXWaitForGpioEvent(
    IN  PCX_HANDLE          PmMyProcess,
    IN  HANDLE              PmEvent )
{
    WORD                    LcError = WA_COMMAND_NOT_AVAILABLE;
    LPWAIT_GPIO_REQ_INFO   LcRequest;
    LPWAIT_GPIO_RESP_INFO  LcReply;
    WORD                    LcAppliIdx;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Look up the application handle
    // in the tab of asynchronous command blocks
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        //
        COMReleaseExmut();
        return EA_INVALID_PCX_HANDLE;
    }

    // Initialize reply bloc
    //
    LcError = SUCCESS;

    // Then retrieve the request / reply blocks allocated
    // for the application
    LcRequest = (LPWAIT_GPIO_REQ_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitGpioRequest );
    LcReply   = (LPWAIT_GPIO_RESP_INFO)
                   &( WaitErrorCmds[LcAppliIdx].acbWaitGpioReply );

    // Send the wait request if none has been already
    // sent
    //
    if ((LcReply->RespHeader).rhT != PROCESSED)
    {
        COMReleaseExmut();
        return EA_WAIT_REQUEST_REFUSED;
    }

    // filling the header of the request block
    COMFillBcHeader( ASYNC_EVT_FAM,
                WAIT_GPIO_EVENT_CMD,
                (APP_HANDLE)PmMyProcess,
                sizeof( WAIT_GPIO_REQ_INFO ),
                sizeof( WAIT_GPIO_RESP_INFO ),
                (BC_HEADER_HANDLE) &LcRequest );

    // sending the request to the driver
    // ---------------------------------
#ifdef WIN32
        WaitErrorCmds[LcAppliIdx].acbWaitGpioOverlapped.hEvent = PmEvent;
        COMCallDriver( (LPBC_HEADER_INFO) LcRequest,
                   sizeof( WAIT_GPIO_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   &(WaitErrorCmds[LcAppliIdx].acbWaitGpioOverlapped) );
#else
        COMCallDriver( (LPBC_HEADER_INFO) LcRequest,
                   sizeof( WAIT_GPIO_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcReply,
                   PmEvent );
#endif

        LcError = (LcReply->RespHeader).rhCptr;

    COMReleaseExmut();

    return( LcError );
}

// ****************************************************************************
// WORD PCXIsGpioEventSet()
// ***************************
//
// ****************************************************************************
//
// Function to know which board has one GPIO changed.
//
// ****************************************************************************

BOOLEAN _CDECL_ PCXIsGpioEventSet(
    IN  PCX_HANDLE          PmMyProcess,
   OUT  LPWORD              PmPBoardNum )
{ 
    LPWAIT_GPIO_REQ_INFO   LcRequest;
    LPWAIT_GPIO_RESP_INFO  LcReply;
    WORD                   LcAppliIdx;

    if (!COMWaitExmut())
       return FALSE;
    // Look up the application handle
    // in the tab of asynchronous command blocks
    // -----------------------------------------
    if ( !LookupWaitErrorCmdTab( PmMyProcess, &LcAppliIdx ) )
    {
        // The application gave a wrong handle
        // or did not reserve any buffer
        //
        COMReleaseExmut();
        return FALSE;
    }

    // Then retrieve the request / reply blocks allocated
    // for the application
    // --------------------------------------------------
    LcRequest = (LPWAIT_GPIO_REQ_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitGpioRequest );
    LcReply   = (LPWAIT_GPIO_RESP_INFO)
                  &( WaitErrorCmds[LcAppliIdx].acbWaitGpioReply );

    // Check PCXWaitForError command completion
    // ----------------------------------------
    if ( (LcReply->RespHeader).rhT == PROCESSED )
    {
        *PmPBoardNum   = UTIDWMask2Word(LcReply->wgrBoardMask);
        COMReleaseExmut();
        return ( TRUE );
    }
    else
    {
        COMReleaseExmut();
        return( FALSE );
    }
}


// ****************************************************************************
// WORD PCXGetGPIOFeature()
// ***************************
//
// ****************************************************************************
//
// Function to know which GPIO's exist on this board.
//
// ****************************************************************************
WORD _CDECL_ PCXGetGPIOFeature(
    IN  WORD                PmBoardNum, 
    IN  WORD                PmMaxGPIOInfo,
   OUT  PGPIO_INFO          PmGpioInfo,
   OUT  LPWORD              PmActualGPIOInfo )
{
	LPGPIO_REQ_INFO     LcPRequest;   // will point to the request
    LPGPIO_RESP_INFO    LcPReply;     // will point to the reply
    WORD                LcError;

    if (!COMWaitExmut())
    {              
        return EA_CANT_TAKE_MUTEX;
    }

    // ----------------------------------------------
    // Create the request to be sent to the driver.
    // ----------------------------------------------

    LcPRequest = NULL;
    LcPReply   = NULL;

    COMFillBcHeader( GENE_FAM,
                     GPIO_CMD,
                     (APP_HANDLE) 0xFF,            // pseudo handle, not used!
                     sizeof( GPIO_REQ_INFO ), 
                     sizeof( GPIO_RESP_INFO ),
                     (BC_HEADER_HANDLE) &LcPRequest );

    LcPRequest->gpioRequest = GPIO_CMD_FEATURE;
	LcPRequest->gpioTabSize = 1;
    LcPRequest->gpioTabInfo[0].gpioBoardNumber = PmBoardNum;

    // ------------------------------
	// Send the request to the driver
    // ------------------------------

    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   sizeof( GPIO_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // ------------------
    // Read the response.
    // ------------------

    LcError = LcPReply->RespHeader.rhCptr;

    if ( LcError == SUCCESS  )
    {
        WORD  j;
        DWORD LcMask;
        WORD  LcActualGPIOInfo  = 0;
        DWORD LcReadAccess      = LcPReply->gpioTabInfo[0].gpioUnused1;
        DWORD LcWriteAccess     = LcPReply->gpioTabInfo[0].gpioUnused2;
        DWORD LcProgrammable    = LcPReply->gpioTabInfo[0].gpioUnused3;

        for ( j = 0 ; j < 8*sizeof(DWORD) ; j++ )   // 32 times
        {
            LcMask = 1 << j;

            if(((LcReadAccess | LcWriteAccess) & LcMask) == 0)
                continue;                           // no GPIO at pos. j

            // if ELEM_GPIO_INFO available
            if( (PmMaxGPIOInfo > LcActualGPIOInfo) && (PmGpioInfo != NULL) )
            {
                BZERO2(PmGpioInfo, GPIO_INFO, 1);

                PmGpioInfo->gpioBoardNumber = PmBoardNum;
                PmGpioInfo->gpioNum         = j;

                if( (LcReadAccess & LcMask) != 0 )
                {
                    PmGpioInfo->gpioType   |=  GPIO_READ_ACCESS;
                }
                if( (LcWriteAccess & LcMask) != 0 )
                {
                    PmGpioInfo->gpioType   |=  GPIO_WRITE_ACCESS;
                }

                if( PmGpioInfo->gpioType   ==  GPIO_FULLDUPLEX_ACCESS )
                {
                    if( (LcProgrammable & LcMask) != 0 )
                    {
                        PmGpioInfo->gpioType = GPIO_PROGRAMMABLE_ACCESS;
                    }
                }
            }
            else
                LcError = WA_MORE_DATA;             // not enough ELEM_GPIO_INFO

            LcActualGPIOInfo++;                     // one more GPIO found
            PmGpioInfo++;                           // switch to next ELEM_GPIO_INFO
        }

        // number of available GPIO's of this board
        if ( PmActualGPIOInfo )
        {
            *PmActualGPIOInfo = LcActualGPIOInfo;
        }
        else
        {
            LcError = EA_INVALID_POINTER;
        }
    }

	COMReleaseExmut();    

    return LcError;
}


// ****************************************************************************
// WORD PCXSetInternalLatency()
// ***************************
//
// ****************************************************************************
//
// Function to Dynamicely set the pace of sound tranfers, and Min ASIO buffer size
//
// ****************************************************************************
WORD _CDECL_ PCXSetInternalLatency(
	IN  WORD				PmBoardMask,
 INOUT  PIBL_INFO			PmIBLInfoPtr)
{ 
	
	LPIBL_REQ_INFO     LcPRequest;   // will point to the request
    LPIBL_RESP_INFO    LcPReply;     // will point to the reply
    WORD                LcError;

    if (!COMWaitExmut())
    {              
        return EA_CANT_TAKE_MUTEX;
    }

	if (!PmIBLInfoPtr)
    {              
        return EA_INVALID_POINTER;
    }

    // ----------------------------------------------
    // Create the request to be sent to the driver.
    // ----------------------------------------------

    LcPRequest = NULL;
    LcPReply   = NULL;

    COMFillBcHeader( GENE_FAM,
                     IBL_CMD,
                     (APP_HANDLE) 1,            // pseudo handle, not used (is 1 to avoid error message)!
                     sizeof( IBL_REQ_INFO ), 
                     sizeof( IBL_RESP_INFO ),
                     (BC_HEADER_HANDLE) &LcPRequest );

    LcPRequest->iblSamples = PmIBLInfoPtr->iblSamples;
    LcPRequest->iblBoardMask = PmBoardMask;

   
    // ------------------------------
	// Send the request to the driver
    // ------------------------------

    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   sizeof( IBL_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // ------------------
    // Read the response.
    // ------------------

    LcError = LcPReply->RespHeader.rhCptr;

    if (!(LcError & ERROR_MASK))
    {
		PmIBLInfoPtr->iblSamples = LcPReply->iblSamples;
		PmIBLInfoPtr->iblMax = LcPReply->iblMax;
		PmIBLInfoPtr->iblMin = LcPReply->iblMin;
		PmIBLInfoPtr->iblGranularity = LcPReply->iblGranularity;
	} else  PmIBLInfoPtr->iblSamples = 0;

	COMReleaseExmut();    

    return LcError;

}


// ****************************************************************************
// WORD PCXGetInternalLatency()
// ***************************
//
// ****************************************************************************
//
// Function to retrieve the pace of sound tranfers, and Min/Max ASIO buffer size
//
// ****************************************************************************
WORD _CDECL_ PCXGetInternalLatency(
	IN	WORD				PmBoardMask,
 INOUT	PIBL_INFO			PmIBLInfoPtr )
{
	LPIBL_REQ_INFO     LcPRequest;   // will point to the request
    LPIBL_RESP_INFO    LcPReply;     // will point to the reply
    WORD                LcError;

    if (!PmIBLInfoPtr)
    {
        return EA_INVALID_POINTER;
    }

    if (!COMWaitExmut())
    {
        return EA_CANT_TAKE_MUTEX;
    }

    // ----------------------------------------------
    // Create the request to be sent to the driver.
    // ----------------------------------------------

    LcPRequest = NULL;
    LcPReply   = NULL;

    COMFillBcHeader( GENE_FAM,
                     IBL_CMD,
                     (APP_HANDLE) 1,            // pseudo handle, not used (is 1 to avoid error message)!
                     sizeof( IBL_REQ_INFO ), 
                     sizeof( IBL_RESP_INFO ),
                     (BC_HEADER_HANDLE) &LcPRequest );

    LcPRequest->iblSamples = (WORD)0;					// sinon c'est un set
    LcPRequest->iblBoardMask = PmBoardMask;
 
   
    // ------------------------------
	// Send the request to the driver
    // ------------------------------

    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   sizeof( IBL_REQ_INFO ),
                   (RESP_HEADER_HANDLE) &LcPReply,
                   0 );

    // ------------------
    // Read the response.
    // ------------------

    LcError = LcPReply->RespHeader.rhCptr;

	PmIBLInfoPtr->iblSamples		= LcPReply->iblSamples;
	PmIBLInfoPtr->iblMax			= LcPReply->iblMax;
	PmIBLInfoPtr->iblMin			= LcPReply->iblMin;
	PmIBLInfoPtr->iblGranularity	= LcPReply->iblGranularity;

	COMReleaseExmut();    

    return LcError;

}


WORD _CDECL_ ESHardwareConfSet (
    IN  WORD                PmBoardMask,
    IN  PES_HW_CFG_INFO     PmHardwareCfg )
{
	LES_HW_CFG_REQ_INFO LcPRequest = NULL; // will point to the request
    LPRESP_HEADER_INFO  LcPReply = NULL;   // will point to the reply
    WORD                LcError;

    if (!PmHardwareCfg)
    {
        return EA_INVALID_POINTER;
    }

    if (!COMWaitExmut())
    {
        return EA_CANT_TAKE_MUTEX;
    }

    // ----------------------------------------------
    // Create the request to be sent to the driver.
    // ----------------------------------------------

    COMFillBcHeader( GENE_FAM,
                     ES_HW_CFG_CMD,
                     (APP_HANDLE) 1,            // pseudo handle, not used (is 1 to avoid error message)!
                     sizeof( ES_HW_CFG_REQ_INFO ),
                     sizeof( RESP_HEADER_INFO ),
                     (BC_HEADER_HANDLE) &LcPRequest );

    LcPRequest->ehcBoardMask   = PmBoardMask;
    LcPRequest->ehcEtherSound0 = PmHardwareCfg->cfgEtherSound0;
    LcPRequest->ehcEtherSound1 = PmHardwareCfg->cfgEtherSound1;

    // ------------------------------
	// Send the request to the driver
    // ------------------------------

    COMCallDriver( (LPBC_HEADER_INFO) LcPRequest,
                   sizeof( ES_HW_CFG_REQ_INFO ),
                   &LcPReply,
                   0 );

    // ------------------
    // Read the response.
    // ------------------

    LcError = LcPReply->rhCptr;

	COMReleaseExmut();

    return LcError;
}
