// *************************************************************************
//
//  COPYRIGHT 1996-2000 DIGIGRAM. ALL RIGHTS RESERVED.
//
//  DIGIGRAM
//
// **************************************************************************

// Include files
// *************
#include <fcntl.h>
#include <io.h>
#include <errno.h>
#include <math.h>

#include "apidef.h"
#include "pcxapi_e.h"   // prototypes of functions
#include "pcxerr_e.h"   // shared list of errror codes

#include "common.h"     // common objects shared between all API family
                        // functions
#include "genasync.h"   // shared driver commands of general/async family
#include "util.h"

#include <crtdbg.h> /*for ASSERT*/

#include "GainsLUT.c"		// define static array of DSP gains, in steps of 0.1 dB (+-18dB)	


#ifdef _DEBUG
#define NPNTAPITRACE(x)     OutputDebugString(_T(x))
#else
#define NPNTAPITRACE(x)
#endif //_DEBUG


// Constants
// *********

// generic file names for frequency conversion tables
// --------------------------------------------------
#define CF_TABLE_FILENAME  _T("T_1_E%2X.T56")

STATIC const DWORD CFFrequencies[] = {
       8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000, 64000, 88200, 96000,128000,176400,192000 };

#define MAX_TABLE_ENTRIES   (sizeof(CFFrequencies)/sizeof(CFFrequencies[0]))

STATIC const BYTE CFTable[MAX_TABLE_ENTRIES][MAX_TABLE_ENTRIES] =
{
	// Frequence Dest:
	// 8000  11025  12000  16000  22050  24000  32000  44100  48000  64000  88200  96000 128000 176400 192000
	{ 0x11 ,   0  ,   0  , 0x12 ,   0  , 0x13 , 0x14 ,   0  , 0x16 ,   0  ,   0  ,   0  ,   0  ,   0  ,   0  }, // Source 8000
	{   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  ,   0  , 0x14 ,   0  ,   0  ,   0  ,   0  ,   0  ,   0  ,   0  }, // Source 11025
	{   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  ,   0  , 0x14 ,   0  ,   0  ,   0  ,   0  ,   0  ,   0  }, // Source 12000
	{ 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  , 0x13 , 0x14 ,   0  , 0x16 ,   0  ,   0  ,   0  }, // Source 16000
	{   0  , 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  ,   0  , 0x14 ,   0  ,   0  ,   0  ,   0  }, // Source 22050
	{ 0x31 ,   0  , 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  ,   0  , 0x14 ,   0  ,   0  ,   0  }, // Source 24000
	{ 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  , 0x13 , 0x14 ,   0  , 0x16 }, // Source 32000
	{   0  , 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 , 0xBA ,   0  , 0x12 ,   0  ,   0  , 0x14 ,   0  }, // Source 44100
	{ 0x61 ,   0  , 0x41 , 0x31 ,   0  , 0x21 ,   0  , 0xAB , 0x11 ,   0  ,   0  , 0x12 ,   0  ,   0  , 0x14 }, // Source 48000
	{   0  ,   0  ,   0  , 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  , 0x12 ,   0  , 0x13 }, // Source 64000
	{   0  ,   0  ,   0  ,   0  , 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 , 0xBA ,   0  , 0x12 ,   0  }, // Source 88200
	{   0  ,   0  ,   0  , 0x61 ,   0  , 0x41 , 0x31 ,   0  , 0x21 ,   0  , 0xAB , 0x11 ,   0  ,   0  , 0x12 }, // Source 96000
	{   0  ,   0  ,   0  ,   0  ,   0  ,   0  , 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 ,   0  ,   0  }, // Source 128000
	{   0  ,   0  ,   0  ,   0  ,   0  ,   0  ,   0  , 0x41 ,   0  ,   0  , 0x21 ,   0  ,   0  , 0x11 , 0xBA }, // Source 176400
	{   0  ,   0  ,   0  ,   0  ,   0  ,   0  , 0x61 ,   0  , 0x41 , 0x31 ,   0  , 0x21 ,   0  , 0xAB , 0x11 }  // Source 192000
	// do not use the contexts AB or BA for frequencies lower than 44100
};

#define CONTEXT_FREQCODE_MASK   0x0000000f
#define CONTEXT_FREQCODE_WIDTH  4

// Bounds for effects
// ------------------
#define SC_MIN_SPEED    ((FLOAT) 0.0)
// ## FS (13/11/97) -- the max speed for a scrub is theoretically
// 5.1, but we would better cut it off since it does not work
// well over 2.0
// --------------------------------------------------------------
#define SC_MAX_SPEED    ((FLOAT) 5.0)       // ## FM (19/12/97)

// ## FM (08/20/97) -- For TS
#define TS_MIN_SPEED    ((FLOAT) 0.5)
#define TS_MAX_SPEED    ((FLOAT) 2.0)


// List of available MPEG bit rates in LAYER_1, LAYER_2 and LAYER_3 for regular frequencies.
STATIC const DWORD BITRATES[3][17] =
{
 {0, 32000UL, 64000UL, 96000UL, 128000UL, 160000UL, 192000UL, 224000UL, 256000UL, 288000UL, 320000UL, 352000UL, 384000UL, 416000UL, 448000UL, 65535, 0}, // BITRATES[0][?] for LAYER_1.
 {0, 32000UL, 48000UL, 56000UL, 64000UL, 80000UL, 96000UL, 112000UL, 128000UL, 160000UL, 192000UL, 224000UL, 256000UL, 320000UL, 384000UL, 65535, 0},    // BITRATES[1][?] for LAYER_2.
 {0, 32000UL, 40000UL, 48000UL, 56000UL, 64000UL, 80000UL, 96000UL, 112000UL, 128000UL, 160000UL, 192000UL, 224000UL, 256000UL, 320000UL, 65535, 0}      // BITRATES[2][?] for LAYER_3.
};

// List of available MPEG bit rates in LAYER_1, LAYER_2 and LAYER_3 for low frequencies.
STATIC const DWORD LOW_BITRATES[3][17] =
{
 {0, 32000UL, 48000UL, 56000UL, 64000UL, 80000UL, 96000UL, 112000UL, 128000UL, 144000UL, 160000UL, 176000UL, 192000UL, 224000UL, 256000UL, 65535, 0}, // LOW_BITRATES[0][?] for LAYER_1.
 {0,  8000UL, 16000UL, 24000UL, 32000UL, 40000UL, 48000UL, 56000UL, 64000UL, 80000UL, 96000UL, 112000UL, 128000UL, 144000UL, 160000UL, 65535, 0},     // LOW_BITRATES[1][?] for LAYER_2.
 {0,  8000UL, 16000UL, 24000UL, 32000UL, 40000UL, 48000UL, 56000UL, 64000UL, 80000UL, 96000UL, 112000UL, 128000UL, 144000UL, 160000UL, 65535, 0}      // LOW_BITRATES[2][?] for LAYER_3.
};

// Local functions
// ******************

STATIC WORD  Fill_Set_MAXIMIZER(    INOUT  PEFFECT_MX_PARAM_INFO       PmepMX,
                                    INOUT  LPDWORD  PmpDspParams,
                                    IN     DWORD    PmBoardFrequency,
                                    INOUT  LPWORD   PmpAudioMask);

STATIC WORD  Fill_Set_EQUALIZER(    INOUT  PEFFECT_EQ_PARAM_INFO       PmepEQ,
                                    INOUT  LPDWORD                     PmpDspParams,
                                    IN     DWORD  PmBoardFrequency);



// *************************************************************************
// STATIC DWORD ReverseDW
// ************************************
//
// Input Parameters :
// ******************
//      PmdwElmt : DWORD to translate
//
// Output Parameters :
// *******************
//      None
//
// Return value :
// **************
//      Reversed DWORD
//
// *************************************************************************
//
// This function reverse bytes order of a DWORD (usefull for Little<->Big
// endian conversion)
//
// *************************************************************************

STATIC DWORD ReverseDW(IN DWORD PmdwElmt)
{
 int   i;
 BYTE  LccTemp;
 LPBYTE LcpcData;
 DWORD LcdwValRet;
 int   LciSize = sizeof(PmdwElmt);

 LcdwValRet = PmdwElmt;
 LcpcData = (LPBYTE)&LcdwValRet;
 for (i = 0;i < LciSize/2; i++)
 {
  LccTemp = LcpcData[i];
  LcpcData[i] = LcpcData[LciSize - i - 1];
  LcpcData[LciSize - i - 1] = LccTemp;
 }
 return(LcdwValRet);
}

// *************************************************************************
// STATIC WORD ReverseW
// ************************************
//
// Input Parameters :
// ******************
//      PmwElmt : WORD to translate
//
// Output Parameters :
// *******************
//      None
//
// Return value :
// **************
//    Reversed WORD
//
// *************************************************************************
//
// This function reverse bytes order of a WORD (usefull for Little<->Big
// endian conversion)
//
// *************************************************************************

STATIC WORD ReverseW(IN WORD PmwElmt)
{
 int   i;
 BYTE  LccTemp;
 LPBYTE LcpcData;
 WORD  LcwValRet;
 int   LciSize = sizeof(PmwElmt);

 LcwValRet = PmwElmt;
 LcpcData = (LPBYTE)&LcwValRet;
 for (i = 0;i < LciSize/2; i++)
 {
  LccTemp = LcpcData[i];
  LcpcData[i] = LcpcData[LciSize - i - 1];
  LcpcData[LciSize - i - 1] = LccTemp;
 }
 return(LcwValRet);
}

// *************************************************************************
// STATIC WORD AUDFormatToHeader
// *******************************
//
// Input Parameters :
// ******************
//      PmpsfSoundFormat  : Sound's format
//      PmbHeaderMotorola : TRUE if the header must be at motorola format
//
// Output Parameters :
// *******************
//      PmdwHeader        : MPEG header corresponding to PmpsfSoundFormat
//
// Error codes :
// **************
//      EA_BAD_SOUND_FORMAT
//
// *************************************************************************
//
// Translate a format to a header
//
// *************************************************************************
STATIC WORD AUDFormatToHeader(IN PSOUND_FORMAT_INFO PmpsfSoundFormat,
                              IN BOOL               PmbHeaderMotorola,
                              OUT LPDWORD           PmpdwHeader)
{
    DWORD LcdwHeader;

    // Let's first handle the very special and simple case of GSM
    //
    if ( PmpsfSoundFormat->sfHandle == CODEC_GSM )
    {
         LcdwHeader = 0xFBD00000L;

         if (PmpsfSoundFormat->sfChannels != CHANNELS_MONO)
             return EA_BAD_SOUND_FORMAT;

         if (PmpsfSoundFormat->sfFrequency != 8000)
             return EA_BAD_SOUND_FORMAT;

         if ( PmpsfSoundFormat->sfHasHeader || PmpsfSoundFormat->sfGsmIsMagic )
         {
             PmpsfSoundFormat->sfHasHeader = FALSE ;
             PmpsfSoundFormat->sfGsmIsMagic = TRUE ;
             LcdwHeader |= 00000002L;
         }
         else
         {
             switch ( PmpsfSoundFormat->sfGsmEndianNess )
             {
             case GSM_LITTLE_ENDIAN :
                 LcdwHeader |= 00000000L;
                 break;
             case GSM_BIG_ENDIAN :
                 LcdwHeader |= 00000001L;
                 break;
             case GSM_MSFT :
                 LcdwHeader |= 00000003L;
                 break;
             default :
                 return EA_BAD_SOUND_FORMAT;
             }
         }

         goto clean_exit;
    }

    // Set default emphasis... if format is PCM
    //
    LcdwHeader = (PmpsfSoundFormat->sfHandle == CODEC_MPEG) ? 0x00000000L : 0x00010004L;

    // Check frequency.
    if          (PmpsfSoundFormat->sfFrequency % 11025 != 0) 
        if      (PmpsfSoundFormat->sfFrequency % 12000 == 0)              LcdwHeader |= 0x00000400L;
        else if (PmpsfSoundFormat->sfFrequency % 8000  == 0)              LcdwHeader |= 0x00000800L;
        else return EA_BAD_SOUND_FORMAT;

    // Check mode MONO / STEREO
    //
    if ( PmpsfSoundFormat->sfChannels == CHANNELS_MONO )
        LcdwHeader |= 0x000000C0L;

    // Always Layer 1
    //
    if ( PmpsfSoundFormat->sfHandle != CODEC_MPEG )
    {
        LcdwHeader |= 0x00060000;     // Layer 1
    }
    else
    {
        switch ( PmpsfSoundFormat->sfMpegLayer )
        {
        case 1 :
            LcdwHeader |= 0x00060000;
            break;
        case 2 :
            LcdwHeader |= 0x00040000;
            break;
        case 3 :
            LcdwHeader |= 0x00020000;
            break;
        default:
            return EA_BAD_SOUND_FORMAT;
        }
    }

    if ( PmpsfSoundFormat->sfHandle == CODEC_LINEAR )
    {
		if( PmpsfSoundFormat->sfLinearIsFloat )
		{
			if( (PmpsfSoundFormat->sfLinearNbBits != 32) ||
				(PmpsfSoundFormat->sfLinearEndianNess != LINEAR_LITTLE_ENDIAN) )
			{	return EA_BAD_SOUND_FORMAT; }

			LcdwHeader |= 0xFAD00000L;	// float 32 bit
		}
		else
		{
			LcdwHeader |= 0xFED00000L;	// linear fix 8..24 bit

			switch ( PmpsfSoundFormat->sfLinearNbBits )
			{
			case 8  : break;
			case 16 : LcdwHeader |= 0x00002000; break;
			case 24 : LcdwHeader |= 0x00004000; break;
			default : return EA_BAD_SOUND_FORMAT;
			}
		}

        if ((PmpsfSoundFormat->sfFrequency < 32000L) && (PmpsfSoundFormat->sfFrequency > 11025L))
            LcdwHeader |= 0x00000100L;
        else if (PmpsfSoundFormat->sfFrequency <= 11025L)
            LcdwHeader |= 0x00000200L;

        switch ( PmpsfSoundFormat->sfLinearEndianNess )
        {
        case LINEAR_LITTLE_ENDIAN :
            LcdwHeader |= 0x00008000;
            break;
        case LINEAR_BIG_ENDIAN    :
            break;
        default                   :
            // ## FS (12/08/97) -- Endianness not significant
            //  for linear 8 bits
            // -----------------------------------------------
            if (PmpsfSoundFormat->sfLinearNbBits != 8)
                return(EA_BAD_SOUND_FORMAT) ;
        }
    }
    else // Pure MPEG format
    {
        WORD LcwBitRateIndex;
        BOOL LcbBitRateFound = FALSE;

        LcdwHeader |= 0xFFF00000L;

        // Check JointStereo / DualChannel
        // ## TL (24/09/97) -- Add Joint-Stereo choice depending on layer and
        //                     bitrate (same as in KDos1)
        //
        if ( (PmpsfSoundFormat->sfMpegMode == MPEG_MODE_JOINT_STEREO) &&
            (PmpsfSoundFormat->sfMpegModeExtension == MPEG_JSTEREO_MODE_AUTO) )
        {
            if ( PmpsfSoundFormat->sfMpegLayer == 2 )
            {
                if ( PmpsfSoundFormat->sfBitRate <= 192000 )
                    PmpsfSoundFormat->sfMpegModeExtension = MPEG_JSTEREO_MODE_1;
                else if ( PmpsfSoundFormat->sfBitRate <= 256000 )
                    PmpsfSoundFormat->sfMpegModeExtension = MPEG_JSTEREO_MODE_2;
                else
                    PmpsfSoundFormat->sfMpegModeExtension = MPEG_JSTEREO_MODE_3;
            }
            else
            {
                PmpsfSoundFormat->sfMpegModeExtension = MPEG_JSTEREO_MODE_1;
            }
        }
        // ## TL (24/09/97) -- End modification

        switch ( PmpsfSoundFormat->sfMpegMode )
        {
        case MPEG_MODE_MONO         :
        case MPEG_MODE_STEREO       :
            break;
        case MPEG_MODE_DUAL_CHANNEL :
            LcdwHeader |= 0x00000080L;
            break;
        case MPEG_MODE_JOINT_STEREO :
            if (PmpsfSoundFormat->sfMpegLayer == 3)
            {
                switch (PmpsfSoundFormat->sfMpegModeExtension)
                {
                case MPEG_LIII_JSTEREO_MODE_INTENSITY   :
                    LcdwHeader |= 0x00000050L;
                    break;
                case MPEG_JSTEREO_MODE_MSSTEREO         :
                    LcdwHeader |= 0x00000060L;
                    break;
                case MPEG_JSTEREO_MODE_MS_AND_INTENSITY :
                    LcdwHeader |= 0x00000070L;
                    break;
                case MPEG_JSTEREO_MODE_AUTO             :
                case MPEG_LIII_JSTEREO_MODE_STEREO      :
                default                                 :
                    LcdwHeader |= 0x00000040L;
                }
            }
            else
            {
                switch (PmpsfSoundFormat->sfMpegModeExtension)
                {
                case MPEG_JSTEREO_MODE_1 :
                    LcdwHeader |= 0x00000040L;
                    break; //  4->31 <=> 00
                case MPEG_JSTEREO_MODE_2 :
                    LcdwHeader |= 0x00000050L;
                    break; //  8->31 <=> 01
                case MPEG_JSTEREO_MODE_3 :
                    LcdwHeader |= 0x00000060L;
                    break; // 12->31 <=> 10
                case MPEG_JSTEREO_MODE_4 :
                    LcdwHeader |= 0x00000070L;
                    break; // 16->31 <=> 11
                case MPEG_JSTEREO_MODE_AUTO:
                default :
                    if (PmpsfSoundFormat->sfMpegLayer == 2)
                    {
                        if (PmpsfSoundFormat->sfBitRate <= 192000)
                            LcdwHeader |= 0x00000040L; //  4->31 <=> 00
                        else if (PmpsfSoundFormat->sfBitRate <= 256000)
                            LcdwHeader |= 0x00000050L; //  8->31 <=> 01
                        else
                            LcdwHeader |= 0x00000060L; // 12->31 <=> 10
                    }
                    else
                        LcdwHeader |= 0x00000040L; // For Layer 1
                }
            }
            break;
        default:
            return(EA_BAD_SOUND_FORMAT);
        }

        if ( PmpsfSoundFormat->sfFrequency >= 32000L )
            LcdwHeader |= 0x00080000L;

        // Padding bit.
        //
        if ( PmpsfSoundFormat->sfMpegHasPaddingBit )
        {
            LcdwHeader |= 0x00000200L;

            if ( (PmpsfSoundFormat->sfFrequency != 44100L) &&
                                    (PmpsfSoundFormat->sfFrequency != 22050L) )
                return(EA_BAD_SOUND_FORMAT);
        }

        // Check bit rate.
        //
        LcbBitRateFound = FALSE;

        for ( LcwBitRateIndex = 0;
              (LcwBitRateIndex < 17) && (!LcbBitRateFound);
              LcwBitRateIndex++)
        {
            DWORD dwBitRate;

            if ( PmpsfSoundFormat->sfFrequency >= 32000L )
            {
                dwBitRate =
                    BITRATES[PmpsfSoundFormat->sfMpegLayer-1][LcwBitRateIndex];
            }
            else
            {
                dwBitRate =
                    LOW_BITRATES[PmpsfSoundFormat->sfMpegLayer-1][LcwBitRateIndex];
            }

            if ( dwBitRate == PmpsfSoundFormat->sfBitRate )
            {
                LcdwHeader |= ((DWORD)LcwBitRateIndex) << 12;
                LcbBitRateFound = TRUE;
            }
        } // End for

        if ( (!LcbBitRateFound) || (LcwBitRateIndex == 0) ||
                                                    (LcwBitRateIndex == 17) )
            return(EA_BAD_SOUND_FORMAT) ;

        if ( !(PmpsfSoundFormat->sfMpegHasCrc) )    LcdwHeader |= 0x00010000L;
        if ( PmpsfSoundFormat->sfMpegHasExtension ) LcdwHeader |= 0x00000100L;
        if ( PmpsfSoundFormat->sfMpegHasCopyright ) LcdwHeader |= 0x00000008L;
        if ( PmpsfSoundFormat->sfMpegIsOriginal )   LcdwHeader |= 0x00000004L;

        switch ( PmpsfSoundFormat->sfMpegPreEmphasis )
        {
        case MPEG_PREEMP_50_15: LcdwHeader |= 0x00000001L; break;
        case MPEG_PREEMP_J17  : LcdwHeader |= 0x00000003L; break;
        default: break ;
        }
    }

clean_exit:
    if ( PmbHeaderMotorola )
        *PmpdwHeader = ReverseDW(LcdwHeader);
    else
        *PmpdwHeader = LcdwHeader;

    return(SUCCESS) ;
}


// ****************************************************************************
// STATIC FLOAT Api2DrvALevel()
// **********************************
//
// Input parameters :
// ****************
//
//  FLOAT       PmUserLevelValue:  the user level in dB
//
// Output parameters :
// ****************
//
//  WORD        PmDrvLevelValue:    the driver code for the analog level
//
// Return value :
// **************
//
// SUCCESS if conversion is successful,
// WA_LEVEL_OUT_OF_RANGE otherwise
//
// ****************************************************************************
//
//  Converts driver absolute integer codes standing for 1/4 dB steps
//  into a floating point value in dB
//
// ****************************************************************************
STATIC WORD Api2DrvALevel(
    IN  FLOAT   PmUserLevelValue,
    OUT LPWORD   PmDrvLevelValue )
{
    FLOAT   LcDrvLevel ;
    WORD    LcErrorCode ;

    // Ceil for analog level is 0dB
    // Bring upper values back to this ceil
    // and issue a warning
    // ------------------------------------
    if ( PmUserLevelValue > MAX_ANALOG_LEVEL_VALUE )
    {
        PmUserLevelValue = MAX_ANALOG_LEVEL_VALUE ;
        LcErrorCode = WA_LEVEL_OUT_OF_RANGE ;
    }
    else
    {
        LcErrorCode = SUCCESS ;
    }

    // Floor for analog level
    // Silently set lower values back to this floor
    // --------------------------------------------
    if ( PmUserLevelValue <= LEVEL_NEGATIVE_INFINITY )
    {
        LcDrvLevel = (FLOAT) 0.0 ;
    }
    else
    {
        LcDrvLevel = ((FLOAT)MAX_ANALOG_LEVEL_DRV_VALUE)
                    + ( PmUserLevelValue * ((FLOAT)DRV_LEVEL_STEP) );
    }

    // Round float value to closest integer
    // -------------------------------------
    *PmDrvLevelValue = (WORD) LcDrvLevel ;

    return LcErrorCode ;
}

// ****************************************************************************
// STATIC FLOAT Api2DrvDLevel()
// **********************************
//
// Input parameters :
// ****************
//
//  FLOAT       PmUserLevelValue:  the user level in dB
//
// Output parameters :
// ****************
//
//  WORD        PmDrvLevelValue:    the driver code for the digital level
//
// Return value :
// **************
//
// SUCCESS if conversion is successful,
// WA_LEVEL_OUT_OF_RANGE otherwise
//
// ****************************************************************************
//
//  Converts driver absolute integer codes standing for 1/4 dB steps
//  into a floating point value in dB
//
// ****************************************************************************
STATIC WORD Api2DrvDLevel(
    IN  FLOAT   PmUserLevelValue,
    OUT LPWORD   PmDrvLevelValue )
{
    FLOAT   LcDrvLevel ;
    WORD    LcErrorCode ;

    // Ceil for digital level is +18dB
    // Bring upper values back to this ceil
    // and issue a warning
    // ------------------------------------
    if ( PmUserLevelValue > MAX_LEVEL_VALUE )
    {
        PmUserLevelValue = MAX_LEVEL_VALUE ;
        LcErrorCode = WA_LEVEL_OUT_OF_RANGE ;
    }
    else
    {
        LcErrorCode = SUCCESS ;
    }

    // Floor for digital level
    // Silently set lower values back to this floor
    // --------------------------------------------
    if ( PmUserLevelValue <= LEVEL_NEGATIVE_INFINITY )
    {
        LcDrvLevel = ((FLOAT)0.0) ;
    }
    else
    {
        LcDrvLevel = ((FLOAT)MAX_LEVEL_DRV_VALUE)
                + ( (PmUserLevelValue - ((FLOAT)MAX_LEVEL_VALUE)) * ((FLOAT)DRV_LEVEL_STEP) );
    }

    *PmDrvLevelValue = (WORD) LcDrvLevel ;

    return LcErrorCode ;
}

// ****************************************************************************
// STATIC WORD FillOutAudioSetLevelInfo()
// ***************************************
//
// Input parameters :
// ****************
//
//  POUT_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo: User information block of
//                                              a set out audio level command
//
// Output parameters :
// ****************
//
//  POUT_AUDIO_SET_LEVEL_REQ_INFO PmLevelRequestInfo: Driver request block of
//                                              a set out audio level command
//
// Return value :
// **************
//
// SUCCESS if conversion is successful, WA_LEVEL_OUT_OF_RANGE otherwise
//
// ****************************************************************************
//
//  Fill a driver request block for a set out audio levels command
//
// ****************************************************************************
STATIC WORD FillOutAudioSetLevelInfo(
    IN  POUT_AUDIO_SET_LEVEL_INFO       PmSetOutLevelInfo   ,
    OUT POUT_AUDIO_SET_LEVEL_REQ_INFO   PmLevelRequestInfo  )
{
    WORD    LcErrCode   = SUCCESS ;
    WORD    LcRetCode   = SUCCESS ;

    PmLevelRequestInfo->oasliPar1 = PmSetOutLevelInfo->oaslValidationMask1;
    PmLevelRequestInfo->oasliPar2 = PmSetOutLevelInfo->oaslValidationMask2;

    if ( PmLevelRequestInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_ANALOG_MASK )
    {
        // Converts floating user levels into integers values
        // ---------------------------------------------------
        LcErrCode = Api2DrvALevel( PmSetOutLevelInfo->oaslAnalogLevel,
                                &(PmLevelRequestInfo->oasliAnalogLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( PmLevelRequestInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_DIGITAL_MASK )
    {
        LcErrCode = Api2DrvDLevel( PmSetOutLevelInfo->oaslDigitalLevel,
                                &(PmLevelRequestInfo->oasliDigitalLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( PmLevelRequestInfo->oasliPar1 & OUT_AUDIO_SET_LEVEL_MONITOR_MASK )
    {
        LcErrCode = Api2DrvDLevel( PmSetOutLevelInfo->oaslMonitoringLevel,
                                  &(PmLevelRequestInfo->oasliMonitorLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    PmLevelRequestInfo->oasliMute         = PmSetOutLevelInfo->oaslMute;
    PmLevelRequestInfo->oasliMonitorMute1 = PmSetOutLevelInfo->oaslMonitoringMute1;
    PmLevelRequestInfo->oasliMonitorMute2 = PmSetOutLevelInfo->oaslMonitoringMute2;

    return LcRetCode ;
}

// ****************************************************************************
// STATIC WORD FillInAudioSetLevelInfo()
// ***************************************
//
// Input parameters :
// ****************
//
//  PIN_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo: User information block of
//                                              a set in audio level command
//
// Output parameters :
// ****************
//
//  PIN_AUDIO_SET_LEVEL_REQ_INFO PmLevelRequestInfo: Driver request block of
//                                              a set in audio level command
//
// Return value :
// **************
//
// SUCCESS if conversion is successful, WA_LEVEL_OUT_OF_RANGE otherwise
//
// ****************************************************************************
//
//  Fill a driver request block for a set in audio levels command
//
// ****************************************************************************
STATIC WORD FillInAudioSetLevelInfo(
    IN  PIN_AUDIO_SET_LEVEL_INFO        PmSetInLevelInfo    ,
    OUT PIN_AUDIO_SET_LEVEL_REQ_INFO    PmLevelRequestInfo  )
{
    WORD    LcErrCode   = SUCCESS ;
    WORD    LcRetCode   = SUCCESS ;

    PmLevelRequestInfo->iasliPar1 = PmSetInLevelInfo->iaslValidationMask1;
    PmLevelRequestInfo->iasliPar2 = PmSetInLevelInfo->iaslValidationMask2;

    if ( PmLevelRequestInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_ANALOG_MASK )
    {
        LcRetCode = Api2DrvALevel( PmSetInLevelInfo->iaslAnalogLevel,
                                  &(PmLevelRequestInfo->iasliAnalogLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if( PmLevelRequestInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_MICRO_MASK )
    {
        LcRetCode = Api2DrvALevel( PmSetInLevelInfo->iaslMicroLevel,
                                  &(PmLevelRequestInfo->iasliMicroLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( PmLevelRequestInfo->iasliPar1 & IN_AUDIO_SET_LEVEL_DIGITAL_MASK )
    {
        LcErrCode = Api2DrvDLevel( PmSetInLevelInfo->iaslDigitalLevel,
                                  &(PmLevelRequestInfo->iasliDigitalLevel) ) ;
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    PmLevelRequestInfo->iasliAnalogMute = (BYTE)(PmSetInLevelInfo->iaslAnalogMute != 0);
    PmLevelRequestInfo->iasliMicroMute  = (BYTE)(PmSetInLevelInfo->iaslMicroMute != 0);

    return LcErrCode ;
}

// ****************************************************************************
// STATIC WORD CmdPipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Audio Outputs mask.
//
// DWORD PmPipeInMask: Audio Inputs mask.
//
// BYTE PmCondition: one of the START_PAUSE_* constants
//
// BYTE PmCmd: Pipe command to execute (START_PIPES_CMD/PAUSE_PIPES_CMD/
//             STOP_PIPES_CMD)
//
// PTIME_CODE_START_INFO : PmPTimeCodeStart (can be NULL, typically if PmCondition
//                         is different from START_PAUSE_ON_TIME_CODE)
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function manages the pipe start, pause and stop commands.
//
// ****************************************************************************

STATIC WORD CmdPipe(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmPipeInMask,
    IN  WORD PmCondition,
    IN  BYTE PmCmd,
    IN  PTIME_CODE_START_INFO PmPTimeCodeStart,
    OUT PPCX_TIME PmTimePtr )
{
    LPPIPE_CMD_REQ_INFO  LcPReq = 0;   // will point to the beginning of the
                                       // request block
    LPPIPE_CMD_RESP_INFO LcPResp = 0;  // will point to the beginning of the
                                       // response block

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                 PmCmd,
                 (APP_HANDLE)PmMyProcess,
                 sizeof( PIPE_CMD_REQ_INFO ),
                 sizeof( PIPE_CMD_RESP_INFO ),
                 (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block

    LcPReq->spqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->spqInPipeMask64.QuadPart  = PmPipeInMask;

    if ( PmCmd != STOP_PIPES_CMD )
    {
        LcPReq->spqAttributes = (BYTE) PmCondition;
    }

    if ( PmPTimeCodeStart != NULL )
    {
        memcpy( &(LcPReq->spqTimeCodeStart), PmPTimeCodeStart, sizeof( LcPReq->spqTimeCodeStart ) );
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( PmCmd == STOP_PIPES_CMD )
    {
        if( PmTimePtr )
        {
            *PmTimePtr = LcPResp->sprHour ;
        }
    }

    // returning the error code
    return (LcPResp->RespHeader).rhCptr;
}

// ****************************************************************************
// STATIC WORD CmdStream()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: stream mask.
//
// PTIME_INFO PmTimeInfo: Points to a TIME_INFO structure.
//
//.BYTE PmCmd: Stream command to execute.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function manages the stream start, pause and stop commands.
//
// ****************************************************************************
STATIC WORD CmdStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmTimeInfo,
    IN BYTE PmCmd )
{
    LPSTREAM_CMD_REQ_INFO LcPReq = 0;   // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO    LcPResp = 0;  // will point to the beginning of the
                                        // response block

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                 PmCmd,
                 (APP_HANDLE)PmMyProcess,
                 sizeof( STREAM_CMD_REQ_INFO ),
                 sizeof( RESP_HEADER_INFO ),
                 (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->scqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->scqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->scqStreamMask = PmStreamMask;

    if ( PmCmd != STOP_STREAM_CMD )
    {
        // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
        // if not NULL initialize, otherwise let as is since
        // COMFillBcHeader has reset to 0 the request block
        // ------------------------------------------------------
        if ( PmTimeInfo )
        {
            LcPReq->scqDiffered = (BYTE)PmTimeInfo->tiDelayed;
            LcPReq->scqHour = PmTimeInfo->tiScheduler;
        }
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0);

    // returning the error code
    return LcPResp->rhCptr;
}

// ****************************************************************************
// STATIC WORD GetCFTableFileName()
// ******************************
//
//  Input parameters :
// *******************
//
// DWORD        PmSrcFreq   : The source frequency for the conversion
// DWORD        PmDestFreq  : The destination frequency for the conversion
//
// Output parameters :
// *******************
//
// LPTSTR        PmFileName  : The file nmae containing the effect context
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function manages the table names for frequency conversions
//
// ****************************************************************************
STATIC WORD GetCFTableFileName(
    OUT LPTSTR PmFileName,
    IN  DWORD  PmSrcFreq,
    IN  DWORD  PmDestFreq,
	OUT LPDWORD PmIntermedFreq)
{
    int		LcIndexSRC;
    int		LcIndexDEST;
	BYTE	LcVal;

	for ( LcIndexSRC = 0; LcIndexSRC < MAX_TABLE_ENTRIES ; LcIndexSRC++ )
	{
		if( PmSrcFreq == CFFrequencies[LcIndexSRC] )
			break;
	}
	if( LcIndexSRC >= MAX_TABLE_ENTRIES)
		return EA_BAD_FREQUENCY_VALUE;

	for ( LcIndexDEST = 0; LcIndexDEST < MAX_TABLE_ENTRIES ; LcIndexDEST++ )
	{
		if( PmDestFreq == CFFrequencies[LcIndexDEST] )
			break;
	}
	if( LcIndexDEST >= MAX_TABLE_ENTRIES)
		return EA_BAD_FREQUENCY_VALUE;

	LcVal = CFTable[LcIndexSRC][LcIndexDEST];

	if( LcVal == 0)
	{
		// no direct conversion possible !
		// look for possibility of intermediate frequency conversion:
		//
		int i;
		for( i = MIN(LcIndexDEST, LcIndexSRC) + 1; i < MAX_TABLE_ENTRIES; i++ )
		{
			if( (CFTable[LcIndexSRC][i] != 0) &&
				(CFTable[i][LcIndexDEST] != 0) )
			{
				break;	// FOUND !!  conversion with intermediate frequency is possible !
			}
		}

		if( (i >= MAX_TABLE_ENTRIES) || (PmIntermedFreq == NULL) )
		{
			return EA_BAD_FREQUENCY_VALUE;
		}

		*PmIntermedFreq = CFFrequencies[i];

		if( PmFileName != NULL )
			PmFileName[0] = '\0';

#ifdef _DEBUG
		{
			_TCHAR LcText[256];
			_stprintf(LcText, _T("LXESAPI : FREQUENCY_CHANGE : use intermediate Freq %d to convert %d -> %d\n"), *PmIntermedFreq, PmSrcFreq, PmDestFreq );
			OutputDebugString(LcText);
		}
#endif //_DEBUG

		return WA_PARAMETER_ADJUSTED;
	}

	if( PmIntermedFreq != NULL )
		*PmIntermedFreq = 0;

    // Build the string for file name
    // of the frequency table
    // ------------------------------
	if( PmFileName != NULL )
	{
		_stprintf( PmFileName, CF_TABLE_FILENAME, LcVal );

#ifdef _DEBUG
		{
			_TCHAR LcText2[256];
			_stprintf(LcText2, _T("LXESAPI : FREQUENCY_CHANGE : use file %s to convert %d -> %d\n"), PmFileName, PmSrcFreq, PmDestFreq );
			OutputDebugString(LcText2);
		}
#endif //_DEBUG
	}

    return SUCCESS ;
}


// ****************************************************************************
// STATIC WORD OpenT56File()
// *************************
//
// Input parameters :
// ****************
//
//  IN  LPTSTR   PmPathName      : the path to the various driver NP files
//  IN  LPTSTR   PmShortFileName : the name of the .T56 file
//
// Output parameters :
// ******************
//
//  LPINT    PmFileDescPtr   :   the descriptor of the file
//  LPDWORD  PmLengthPtr     :   the length of the significative section within
//                              the file
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function handles .T56 files
//
// ****************************************************************************
// Open, analyze and verify the file
//  - check its label
//  - if  it's ok :
//     . set the read position to the start of the program section
//     . tell the length of the program section
//     . give back a descriptor onto the file
//  - otherwise, close the file and return an error code
// ****************************************************************************

STATIC WORD OpenT56File(
    IN  LPTSTR   PmPathName,
    IN  LPTSTR   PmShortFileName,
    OUT LPINT    PmFileDescPtr,
    OUT LPDWORD  PmLengthPtr )
{
    INT         LcFile ;
    INT         LcLength = 0 ;
    INT         LcRet ;
    WORD        LcErrorCode = SUCCESS ;

#define SIZE_DSP_WORD       3
#define SIZE_READ_BUFFER    16

    CHAR        LcBuffer[SIZE_READ_BUFFER];

    // File markers
    // ------------
#define     ESPP        0x000004  // P
#define     ESPINF      0x000008  // INFORMATION
#define     STARTADDR   0x000000

#define     DSP_WORD_2_DWORD(readbuffer) \
    (  (((DWORD)readbuffer[0] & 0x000000FF) << 16)      \
     | (((DWORD)readbuffer[1] & 0x000000FF) << 8 )      \
     | (((DWORD)readbuffer[2] & 0x000000FF)) )

    // ## FM -- We must build the path PmPathName\CONTEXT_PATH\PmShortFileName
    //
    _tcscat(PmPathName,CONTEXT_PATH)     ;
    _tcscat(PmPathName,PmShortFileName)  ;

    // Open the .T56 table file
    // --------------------------
    LcFile = _topen( PmPathName, O_RDONLY|O_BINARY );
    if ( LcFile < 0 )
    {
        if ( errno == ENOENT )
            LcErrorCode = EA_CONTEXT_NOT_AVAILABLE ;
        else
            LcErrorCode = EA_OPEN_FILE_IMPOSSIBLE ;

        return( LcErrorCode );
    }

    // read DSP words instead of bytes
    // -------------------------------

    // 1. Check start address ( = 0 )
    // ------------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if (   ( LcRet != SIZE_DSP_WORD )
        || ( DSP_WORD_2_DWORD( LcBuffer ) != STARTADDR ) )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }

    // 2. Get length of next section (in DSP words)
    // --------------------------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if ( LcRet != SIZE_DSP_WORD )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }
    LcLength = (INT) DSP_WORD_2_DWORD( LcBuffer );

    // 3. Check information marker
    // ---------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if (   ( LcRet != SIZE_DSP_WORD )
        || ( DSP_WORD_2_DWORD( LcBuffer ) != ESPINF ) )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }

    // 4. Check file name
    // ## FM (09/15/97) -- We use the short name because this is what is
    //                     stored in the .T56 file.
    // ------------------
    LcRet = _read( LcFile, LcBuffer, LcLength * SIZE_DSP_WORD );

    if ( LcRet != (LcLength * SIZE_DSP_WORD) )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }
    else
    {
        _TCHAR        LcConvertBuffer[SIZE_READ_BUFFER];

        _sntprintf(LcConvertBuffer, _tcsclen(PmShortFileName), _T("%hs"), LcBuffer);
        if ( _tcsncicmp(LcConvertBuffer, PmShortFileName, _tcsclen(PmShortFileName)) != 0 )
        {
            LcErrorCode = EA_FILE_CORRUPTED ;
            goto clean_exit ;
        }
    }

    // 5. Check start address ( = 0 )
    // ------------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if (   ( LcRet != SIZE_DSP_WORD )
        || ( DSP_WORD_2_DWORD( LcBuffer ) != STARTADDR ) )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }

    // 6. Get length of next section (in DSP words)
    // --------------------------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if ( LcRet != SIZE_DSP_WORD )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }
    LcLength = (INT) DSP_WORD_2_DWORD( LcBuffer );

    // 7. Check information marker
    // ---------------------------
    LcRet = _read( LcFile, LcBuffer, SIZE_DSP_WORD );

    if (   ( LcRet != SIZE_DSP_WORD )
        || ( DSP_WORD_2_DWORD( LcBuffer ) != ESPP ) )
    {
        LcErrorCode = EA_FILE_CORRUPTED ;
        goto clean_exit ;
    }

    // Everything clear, hence set the output parameters
    // and return successfully
    // -------------------------------------------------
    *PmFileDescPtr = LcFile ;
    *PmLengthPtr = LcLength * SIZE_DSP_WORD ;

clean_exit:
    if ( LcErrorCode != SUCCESS )
    {
        _close( LcFile );
    }

    return( LcErrorCode );
}


// ****************************************************************************
// STATIC WORD CmdEffectContext()
// ******************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// DWORD PmPipeOutMask: Output pipe mask.
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
// OUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo: Descriptor of effect request
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function manages the contexts (if any) for effects
//
// ****************************************************************************

#define EFFECT_FREQUENCY_CHANGE2    ((PCX_EFFECT_HANDLE) 6)


STATIC WORD CmdEffectContext(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  PTIME_INFO PmDelayCondition,
    IN  POUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo,
    OUT LPDWORD PmContextNumPtr )
{
    LPEFFECT_CTXT_LOAD_REQ_INFO LcPReq = 0;     // will point to the beginning
                                                // of request block
    LPEFFECT_CTXT_LOAD_RESP_INFO LcPResp = 0;   // will point to the beginning
                                                // of response block
    LPBUFF_DESC_INFO LcPBuffDesc;      // will point to the buffer descriptions
                                       // within the request block
    LPBYTE                      LcBuffAddr  ;
    INT                         LcFile      ;
    INT                         LcContextId;
    DWORD                       LcFileLength= 0 ;
    int                         LcReadRet ;
    DWORD                       LcReadSize ;
    DWORD                       LcBuffSize  ;
    WORD                        LcNameLength= 0 ;
    WORD                        LcErrCode   ;
    WORD                        LcReservedBuffers, LcBuffNum ;

    STATIC ADD_DESC_INFO        LcAddrDescTab[MAX_DOWNLOAD_BUFFERS];
    STATIC _TCHAR               LcPathName[MAX_DSP_FILE_NAME_LEN];
    STATIC _TCHAR               LcContextFileName[MAX_SHORT_FILE_NAME_LEN];

    // Only frequency change need some context to be loaded
    // Just bypass this step for any other effect
    // ----------------------------------------------------
    if (   ( PmPipeOutEffectInfo->opseIdentifier != EFFECT_FREQUENCY_CHANGE )
        && ( PmPipeOutEffectInfo->opseIdentifier != EFFECT_FREQUENCY_CHANGE2 ) )
    {
        return SUCCESS ;
    }

    // Frequency conversion
    // --------------------

    // The only available mode as of today is PARM_CF_MODE_CODE
    // ------------------------------------------------------
    if ( PmPipeOutEffectInfo->opseMode != PARM_CF_MODE_CODE )
    {
        return EA_MODE_NOT_AVAILABLE ;
    }

    // Identify the context table to be loaded
    // ---------------------------------------

    // Retrieve the driver path
    // -------------------------
    LcErrCode = GENGetDspFileName(0,LcPathName,LcContextFileName );

    if ( LcErrCode != SUCCESS )
       return LcErrCode ;

    // Convert the parameters to frequency
    // codes for mode CODE
    // -------------------------------------
    LcErrCode = GetCFTableFileName(
                    LcContextFileName,
                    (PmPipeOutEffectInfo->opseParams).epCF.cfpSrcFrequency,
                    (PmPipeOutEffectInfo->opseParams).epCF.cfpDestFrequency,
					NULL);

    if ( LcErrCode != SUCCESS )
       return LcErrCode ;

    // Open, analyze and verify the file
    //  - check its label
    //  - if  it's ok :
    //     . set the read position to the start of the program section
    //     . tell the length of the program section
    //     . give back a descriptor onto the file
    //  - otherwise, close the file and return an error code
    // ------------------------------------------------------------
    // ## FM (09/15/97) -- Add a short file name as 2nd parameter
    //

    LcErrCode = OpenT56File(LcPathName,LcContextFileName,&LcFile,&LcFileLength);

    // Give up if the binary file is corrupted
    // ---------------------------------------
    if ( LcErrCode != SUCCESS )
    {
        return( LcErrCode );
    }

    // Then get the addresses of buffers reserved for downloading purposes
    // --------------------------------------------------------------------
    LcErrCode = GENGetDspBuffers( &LcReservedBuffers, LcAddrDescTab );
    if ( LcErrCode != SUCCESS )
    {
        goto clean_exit;
    }

    // filling the header of the request block
    // ---------------------------------------
    // ## FM (09/12/97) -- Must be done BEFORE any bloc req. initialization!
    //
    COMFillBcHeader( GENE_FAM,
                     EFFECT_CTXT_LOAD_CMD,
                     (APP_HANDLE) PmMyProcess,
                     MAX_EFFECT_CTXT_LOAD_REQ_SIZE,

                     // Under WindowsNT, we need a larger size
                     // since the reply block is used to unmap
                     // pipe's buffers
                     // -------------------------------------------
                     sizeof(ANY_UNMAPPING_REPLY_INFO),

                     (BC_HEADER_HANDLE) &LcPReq );

    // Read the file and fills as many buffers as necessary
    // -----------------------------------------------------
    LcBuffNum = 0;
    LcPBuffDesc = (LPBUFF_DESC_INFO)(LcPReq + 1);
    LcReadRet = 1 ;
    LcReadSize = 0 ;

    while (   ( LcFileLength > 0 )
           && ( LcReadRet != 0 )        // EOF
           && ( LcReadRet != -1 )       // ReadError
           && ( LcBuffNum < LcReservedBuffers ) )
    {
        LcBuffAddr = (LPBYTE) LcAddrDescTab[LcBuffNum].adAddr1; // was LinearMem.aiLinearAddress ;

        // The size of a buffer must be a number a DSP word
        // in order to ease downloading process
        // --------------------------------------------------
        LcBuffSize = ( LcAddrDescTab[LcBuffNum].adAddr3 / 3 ) * 3 ;

        LcReadRet = _read( LcFile, LcBuffAddr, LcBuffSize );
        if ( LcReadRet != -1 )
            // Cast to unsigned before adjusting size
            //
            LcReadSize = (unsigned int) LcReadRet ;
        else
            LcReadSize = 0 ;

        LcPBuffDesc->bdNumBuff  = LcBuffNum ;
        LcPBuffDesc->bdDataSize = ( LcFileLength < LcReadSize )
                                  ? LcFileLength : LcReadSize ;

        LcFileLength -= LcPBuffDesc->bdDataSize ;

        // Next buffer if needed
        // ----------------------
        LcBuffNum ++ ;
        LcPBuffDesc++ ;
    }

    // if the buffer loading did not finish with end of file
    // an error occured
    // ------------------------------------------------------
    if ( LcFileLength != 0 )
    {
        if ( LcBuffNum == LcReservedBuffers )
        {
            LcErrCode = EA_FILE_TOO_LARGE ;
        }

        LcErrCode = EA_READ_FILE_IMPOSSIBLE ;
    }
    else
    {
        // Let's send the request to the driver
        // ------------------------------------

        // Filling the other fields of the request block
        // ---------------------------------------------
        LcPReq->ecqOutPipeMask64.QuadPart = PmPipeOutMask ;
        LcPReq->ecqNbBuffers        = LcBuffNum     ;   // ## FM (09/12/97) -- was missing

        // Copy identifier
        // ---------------
        LcPReq->ecqEffectID = PmPipeOutEffectInfo->opseIdentifier;

        // Set the Context ID ( 4bits SrcFreq | 4bits DestFreq )
        _stscanf( LcContextFileName, CF_TABLE_FILENAME, &LcContextId);
        LcPReq->ecqContextID = (WORD)LcContextId;

        // sending the request to the driver
        // ---------------------------------
        COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                       0,
                       (RESP_HEADER_HANDLE) &LcPResp,
                       0 );

        LcErrCode = (LcPResp->RespHeader).rhCptr ;
        if ( LcErrCode == SUCCESS )
        {
            *PmContextNumPtr = (DWORD) LcPResp->ecrContextNum;
        }
    }

clean_exit:
    _close( LcFile );

    return(LcErrCode) ;
}


// ****************************************************************************
// STATIC WORD CmdOutPipeEffect()
// ******************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
// DWORD PmPipeOutMask: Output pipe mask.
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
// OUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo: Descriptor of effect request
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function actually send an effect request to the driver
//
// ****************************************************************************
STATIC WORD CmdOutPipeEffect(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo,
    IN DWORD PmContextNum )
{
    LPOUT_PIPE_EFFECTS_REQ_INFO LcPReq = 0; // will point to the beginning of
                                            // request block
    LPRESP_HEADER_INFO          LcPResp = 0; // will point to the beginning of
                                            // response block
    FLOAT                       LcFloatValue;
    double                      LcBaseValue;
    DWORD                       LcBoardFrequency = 48000;
    LPDWORD                     LcpDspParams;

	(void)PmDelayCondition;

    // Nothing to do if effect is NULL
    // -------------------------------
    if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_NONE )   return SUCCESS ;

    // we need the clock information for some effects
    if( (PmPipeOutEffectInfo->opseIdentifier == EFFECT_EQUALIZATION_PIPE)    ||
        (PmPipeOutEffectInfo->opseIdentifier == EFFECT_MAXIMIZER)       )
    {
        CLOCK_INFO  LcClo;
        DWORD *Lc_pdw_Clo_Freq = &(LcClo.ckFrequency);

        *Lc_pdw_Clo_Freq	= 0;

        GENGetPipeClock(    TRUE, //force read 
                            PmMyProcess,
                            PmPipeOutMask,
                            0,
                            &LcClo );

        if (*Lc_pdw_Clo_Freq) LcBoardFrequency = *Lc_pdw_Clo_Freq;
    }

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                     OUT_PIPES_EFFECTS_CMD,
                     (APP_HANDLE) PmMyProcess,
                     sizeof( OUT_PIPE_EFFECTS_REQ_INFO ),
                     sizeof( RESP_HEADER_INFO ),
                     (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->oeqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->oeqEffectID = PmPipeOutEffectInfo->opseIdentifier;
    LcPReq->oeqMode = (WORD) PmPipeOutEffectInfo->opseMode;

    // initialize parameters
    // ---------------------------------------------
    LcpDspParams = &(LcPReq->oeqParams[0]);
    memset( LcpDspParams, 0, FX_MAX_USER_PARAMETERS*sizeof(DWORD)); 

    // Handle parameters for each effect
    // and parameter mode
    // -------------------------------------
    if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_PITCH_SHIFTING )
    {
        // The only available mode as of today is PARM_PS_MODE_PITCH_FACTOR
        // ------------------------------------------------------
        if ( PmPipeOutEffectInfo->opseMode != PARM_PS_MODE_PITCH_FACTOR )
        {
            return EA_MODE_NOT_AVAILABLE ;
        }

        // Pitch-shifting is obtain by doing simultaneously a
        // scrub and a time-stretching effect
        // Since the resulting file MUST be of the same size than
        // the original, we first compute the most precise effect
        // (Time-Stretch step is 1/104876), and then the less precise one
        // (Scrub effective step is 1/128)
        //
        // ## FS (12/02/98) -- FA #125
        // Use a base common value to get the same precision in
        // both effects
        // ------------------------------------------------------

        // First send the scrub command, overwrite mode and effectID
        // ---------------------------------------------------------
        LcPReq->oeqEffectID = EFFECT_TIME_STRETCHING ;
        LcPReq->oeqMode = (WORD) PARM_TS_MODE_SPEED_FACTOR ;

        // Update command parameters
        // -------------------------

        // Convert the parameter from FLOAT to double
        // for mode Factor
        // (Time-Stretch step is 1/104876(8192*128) resolution)
        // ------------------------------------------------------------
        LcBaseValue = (double) (PmPipeOutEffectInfo->opseParams).epPS.pspPitchFactor ;
        LcBaseValue *= 128.0 ;
        LcBaseValue = ceil( LcBaseValue );

        LcpDspParams[0] = (DWORD) LcBaseValue * 8192 ;

		// sending the request to the driver
        // ---------------------------------
        COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                       0,
                       (RESP_HEADER_HANDLE) &LcPResp,
                       0 );

        // returning the error code if any
        if ( LcPResp->rhCptr != SUCCESS )
        {
            return( LcPResp->rhCptr );
        }

        // AGAIN, let's
        // fill the header of the request block
        // for a 2nd effect call
        // ---------------------------------------
        LcPReq = 0 ;
        LcPResp = 0 ;
        COMFillBcHeader( PIPE_CTRL_FAM,
                         OUT_PIPES_EFFECTS_CMD,
                         (APP_HANDLE) PmMyProcess,
                         sizeof( OUT_PIPE_EFFECTS_REQ_INFO ),
                         sizeof( RESP_HEADER_INFO ),
                         (BC_HEADER_HANDLE) &LcPReq );

        // Filling the other fields of the request block
        // ---------------------------------------------
        LcPReq->oeqOutPipeMask64.QuadPart = PmPipeOutMask;

        // Compute scrub speed
        // -------------------

        // Retrieve the base common value
        // and multiply to reach the so-called 1/16384 (128*128)precision
        // ------------------------------------------------------------

        // Update command parameters
        // -------------------------
        LcpDspParams[0] = (DWORD) LcBaseValue * 128 ;

        // Then send the scrub command, overwrite mode and effectID
        // ---------------------------------------------------------
        LcPReq->oeqEffectID = EFFECT_SCRUB ;
        LcPReq->oeqMode = (WORD) PARM_SC_MODE_SPEED_FACTOR ;


    }
    else if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_TIME_STRETCHING )
    {
        // Convert the parameter from FLOAT to integer type
        // for mode SPEED
        // -------------------------------------
        LcFloatValue = (PmPipeOutEffectInfo->opseParams).epTS.tspSpeedFactor;

        // The only available mode as of today is PARM_SC_MODE_SPEED_FACTOR
        // ------------------------------------------------------
        if ( PmPipeOutEffectInfo->opseMode != PARM_TS_MODE_SPEED_FACTOR )
        {
            return EA_MODE_NOT_AVAILABLE ;
        }

        if ( (LcFloatValue < TS_MIN_SPEED) || (LcFloatValue > TS_MAX_SPEED) )
        {
            return EA_BAD_EFFECT_PARAMETER ;
        }
        LcFloatValue = 1048576.0f / LcFloatValue ;

        LcpDspParams[0] = (DWORD) LcFloatValue ;

    }
    else if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_SCRUB )
    {
        // Scrub
        // -----

        // The only available mode as of today is PARM_SC_MODE_SPEED_FACTOR
        // ------------------------------------------------------
        if ( PmPipeOutEffectInfo->opseMode != PARM_SC_MODE_SPEED_FACTOR )
        {
            return EA_MODE_NOT_AVAILABLE ;
        }

        // Convert the parameter from FLOAT to integer type
        // for mode SPEED
        // -------------------------------------
        LcFloatValue = (PmPipeOutEffectInfo->opseParams).epSC.scpSpeedFactor;
        if ( (LcFloatValue < SC_MIN_SPEED) || (LcFloatValue > SC_MAX_SPEED) )
        {
            return EA_BAD_EFFECT_PARAMETER ;
        }
        LcFloatValue *= 16384.0f ;

        LcpDspParams[0] = (DWORD) LcFloatValue ;
    }
    else if (   ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_FREQUENCY_CHANGE )
             || ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_FREQUENCY_CHANGE2 ) )
    {
        // Frequency conversion
        // --------------------

        // The only available mode as of today is PARM_MODE_CODE
        // ------------------------------------------------------
        if ( PmPipeOutEffectInfo->opseMode != PARM_CF_MODE_CODE )
        {
            return EA_MODE_NOT_AVAILABLE ;
        }

        LcpDspParams[0] = PmContextNum ;
    }
    else if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_EQUALIZATION_PIPE )
    {
        PEFFECT_EQ_PARAM_INFO   LcEQ = &PmPipeOutEffectInfo->opseParams.epEQ;
        WORD LcRet;
        
        LcRet = Fill_Set_EQUALIZER( LcEQ, LcpDspParams, LcBoardFrequency);
            if (LcRet & ERROR_MASK) return LcRet;
    }
    else if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_MAXIMIZER )
    {
        PEFFECT_MX_PARAM_INFO   LcMX = &PmPipeOutEffectInfo->opseParams.epMX;
        WORD LcRet;

        LcRet = Fill_Set_MAXIMIZER( LcMX, LcpDspParams, LcBoardFrequency, 0 /*not known here*/);
            if (LcRet & ERROR_MASK) return LcRet;
    }
    else
    {
        return EA_BAD_EFFECT_ID;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // returning the error code
    return LcPResp->rhCptr;
}


// Exported functions
// ******************


// ****************************************************************************
// WORD PCXStartPipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Audio Output mask.
//
// DWORD PmPipeInMask: Audio Input mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// BYTE PmStartCondition: Command can be immediate (PmCondition = 0), or
//                      triggered by an external source(PmCondition = 1).
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to set outputs and input pipes in START
// state.
//
// ****************************************************************************

WORD _CDECL_ PCXStartPipe(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN WORD PmStartCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    LcValret = CmdPipe( PmMyProcess,
                    PmPipeOutMask,
                    PmPipeInMask,
                    PmStartCondition,
                    START_PIPES_CMD,
                    NULL,
                    NULL );
    COMReleaseExmut();
    return LcValret;
}

// ****************************************************************************
// WORD PCXStartPipeOnTimeCode()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Audio Output mask.
//
// DWORD PmPipeInMask: Audio Input mask.
//
// BYTE  PmBoardNum  : the board on which the time-code is read
//
// DWORD PmSamplesCount: the time-code sample count
//
// BYTE  PmNbOfTimeCode: the number of time-code pointed by PmPTimeCode
//
// PPCX_TIME_CODE PmPTimeCode: an array of time-code
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to set outputs and input pipes in START
// state. The start condition is that the time-code received is equal to one of
// the PmNbOfTimeCode time-code given.
//
// ****************************************************************************

WORD _CDECL_ PCXStartPipeOnTimeCode(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN WORD  PmBoardMask,
    IN DWORD PmSamplesCount,
    IN BYTE  PmNbOfTimeCode,
    IN PPCX_TIME_CODE PmPTimeCode )
{
    WORD                 LcValret;
    TIME_CODE_START_INFO LcTimeCodeStart;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // Check that the number of time-code is right.
    // --------------------------------------------
    if ( ( PmNbOfTimeCode < 1 ) || ( PmNbOfTimeCode > MAX_TIME_CODE ) )
    {
        COMReleaseExmut();
        return EA_BAD_TIME_CODE_NUMBER;
    }

    LcTimeCodeStart.tcsBoardNum = (BYTE) UTIDWMask2Word( (DWORD) PmBoardMask );
    LcTimeCodeStart.tcsNbOfTimeCode = PmNbOfTimeCode;
    LcTimeCodeStart.tcsSamplesCount = PmSamplesCount;
    memcpy( LcTimeCodeStart.tcsTabTimeCode, PmPTimeCode, PmNbOfTimeCode * sizeof( PmPTimeCode[0] ) );

    LcValret = CmdPipe( PmMyProcess,
                        PmPipeOutMask,
                        PmPipeInMask,
                        START_PAUSE_ON_TIME_CODE,
                        START_PIPES_CMD,
                        & LcTimeCodeStart,
                        NULL );
    COMReleaseExmut();
    return LcValret;
}

// ****************************************************************************
// WORD PCXPausePipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Audio Output mask.
//
// DWORD PmPipeInMask: Audio Input mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// BYTE PmStartCondition: Command can be immediate (PmCondition = 0), or
//                      triggered by an external source(PmCondition = 1).
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to set outputs and input pipes in PAUSE
// state.
//
// ****************************************************************************

WORD _CDECL_ PCXPausePipe(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN WORD PmPauseCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    LcValret = CmdPipe( PmMyProcess,
                    PmPipeOutMask,
                    PmPipeInMask,
                    PmPauseCondition,
                    PAUSE_PIPES_CMD,
                    NULL,
                    NULL );
    COMReleaseExmut();
    return LcValret;
}

// ****************************************************************************
// WORD PCXStopPipe()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Audio Output mask.
//
// DWORD PmPipeInMask: Audio Input mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to stop the sound on all specified
// pipes.
//
// ****************************************************************************

WORD _CDECL_ PCXStopPipe(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    OUT PPCX_TIME PmSampleNumber )
{
    WORD  LcValRet;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    LcValRet = CmdPipe( PmMyProcess,
                    PmPipeOutMask,
                    PmPipeInMask,
                    0,
                    STOP_PIPES_CMD,
                    NULL,
                    PmSampleNumber);

    COMReleaseExmut();

    return LcValRet;
}


// ****************************************************************************
// WORD PCXSetOutPipeParameter()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// IN DWORD PmPipeOutMask: Output pipe mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_PIPE_SET_PARAM_INFO PmPipeOutParamInfo: Points to a OUT_PIPE_PARAM_INFO.
//
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
// ## FM (09/23/97) -- Now, take into account the backward mode.
//
// ****************************************************************************

WORD _CDECL_ PCXSetOutPipeParameter(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_PIPE_SET_PARAM_INFO PmPipeOutParamInfo )
{
    LPOUT_PIPE_SET_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                 OUT_PIPES_SETTINGS_CMD,
                 (APP_HANDLE)PmMyProcess,
                 sizeof( OUT_PIPE_SET_REQ_INFO ),
                 sizeof( RESP_HEADER_INFO ),
                 (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->opqOutPipeMask64.QuadPart = PmPipeOutMask;

    // ## FM -- Check that if the command is deffered, only the backward
    //          attribut is specified
    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition && PmDelayCondition->tiDelayed )
    {
        DWORD   LcValidationMask = PmPipeOutParamInfo->opspValidationMask1 ;

        if (    (LcValidationMask & OUT_PIPE_SET_PARAM_UER_MASK)
             || (LcValidationMask & OUT_PIPE_SET_PARAM_MPEG_MASK)
             || (LcValidationMask & OUT_PIPE_SET_PARAM_UER_EXTRA_MASK) )
        {
            COMReleaseExmut();
            return (EA_MODE_NOT_AVAILABLE) ;
        }

        LcPReq->opqDiffered = (BYTE)PmDelayCondition->tiDelayed;
        LcPReq->opqHour     = PmDelayCondition->tiScheduler;
    }

    LcPReq->opqPar1         = PmPipeOutParamInfo->opspValidationMask1 ;
    LcPReq->opqPar2         = 0;
    LcPReq->opqUer          = PmPipeOutParamInfo->opspUer;
    LcPReq->opqFrequency    = PmPipeOutParamInfo->opspFrequency;
    LcPReq->opqMpeg         = PmPipeOutParamInfo->opspMpeg;
    LcPReq->opqBackward     = PmPipeOutParamInfo->opspBackward;     // ## FM
    LcPReq->opqUerExtra     = PmPipeOutParamInfo->opspUerExtra;     // ## MBR

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXSetOutPipeEffect()
// ***************************
//
// Input parameters :
// ****************
//
// APP_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO.
//
// POUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo: Points to a OUT_PIPE_SET_EFFECT_INFO.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to define an effect on pipes.
//
// ****************************************************************************

WORD _CDECL_ PCXSetOutPipeEffect(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_PIPE_SET_EFFECT_INFO PmPipeOutEffectInfo )
{
    OUT_PIPE_SET_EFFECT_INFO        LcPipeOutEffectInfo ;
    DWORD   LcContextNum            = 0 ;
    DWORD   LcSourceFrequency       = 0;
    DWORD   LcFinalDestFrequency    = 0;
    DWORD   LcIntermediateFrequency = 0;
    WORD    LcErrCode               = SUCCESS ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // Filter multichannel
    if (TbPipeInfo_OUT[ UTIDWMask2Word( PmPipeOutMask) ].piPipeNbChannels > 2)
        switch (PmPipeOutEffectInfo->opseIdentifier)
        {
        case EFFECT_FREQUENCY_CHANGE:
        case EFFECT_SCRUB:
            // effets autoriss
            break;
        default:
            return EA_BAD_MULTICHANNEL_FORMAT;
        }
    
    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // ## FS -- Ugly (but temporary of course) emergency workaround
    // to allow frequency conversion that need two passes
    // ------------------------------------------------------------
    if ( PmPipeOutEffectInfo->opseIdentifier == EFFECT_FREQUENCY_CHANGE )
    {
        LcSourceFrequency       = (PmPipeOutEffectInfo->opseParams).epCF.cfpSrcFrequency  ;
        LcFinalDestFrequency    = (PmPipeOutEffectInfo->opseParams).epCF.cfpDestFrequency ;

        // By default conversion can be done directly in one pass
        //
        LcIntermediateFrequency = 0 ;

		// Check if the conversion is possible, or if it's possible to
		// use an intermediate frequency !
		// -------------------------------------
		LcErrCode = GetCFTableFileName(
						NULL,
						(PmPipeOutEffectInfo->opseParams).epCF.cfpSrcFrequency,
						(PmPipeOutEffectInfo->opseParams).epCF.cfpDestFrequency,
						&LcIntermediateFrequency);

		if( LcErrCode & ERROR_VALUE )
		{
			COMReleaseExmut();
			return( LcErrCode );
		}

        // Handle first pass here if intermediate conversion is needed
        //
        LcPipeOutEffectInfo = *PmPipeOutEffectInfo ;

        if ( LcIntermediateFrequency != 0 )
        {
            (LcPipeOutEffectInfo.opseParams).epCF.cfpDestFrequency = LcIntermediateFrequency ;

            // Verify and load effect context if needed
            // ----------------------------------------
            LcErrCode = CmdEffectContext(
                PmMyProcess,
                PmPipeOutMask,
                PmDelayCondition,
                &LcPipeOutEffectInfo,
                &LcContextNum );

            // If context verification passed, let's
            // send a request for the effect
            // -------------------------------------
            if ( LcErrCode == SUCCESS )
            {
                LcErrCode = CmdOutPipeEffect(
                    PmMyProcess,
                    PmPipeOutMask,
                    PmDelayCondition,
                    &LcPipeOutEffectInfo,
                    LcContextNum );
            }

            // Prepare for second pass
            //
            LcPipeOutEffectInfo = *PmPipeOutEffectInfo ;
            (LcPipeOutEffectInfo.opseParams).epCF.cfpDestFrequency = LcFinalDestFrequency ;
            (LcPipeOutEffectInfo.opseParams).epCF.cfpSrcFrequency = LcIntermediateFrequency ;
            LcPipeOutEffectInfo.opseIdentifier = EFFECT_FREQUENCY_CHANGE2;
        }

    }
    else
    {
        LcPipeOutEffectInfo = *PmPipeOutEffectInfo ;
    }

    // Verify and load effect context if needed
    // ----------------------------------------
    if ( LcErrCode == SUCCESS )
    {
        LcErrCode = CmdEffectContext(
            PmMyProcess,
            PmPipeOutMask,
            PmDelayCondition,
            &LcPipeOutEffectInfo,
            &LcContextNum );
    }

    // If context verification passed, let's
    // send a request for the effect
    // -------------------------------------
    if ( LcErrCode == SUCCESS )
    {
        LcErrCode = CmdOutPipeEffect(
                        PmMyProcess,
                        PmPipeOutMask,
                        PmDelayCondition,
                        &LcPipeOutEffectInfo,
                        LcContextNum );
    }

    if(LcErrCode != EA_CANT_TAKE_MUTEX)
    {
        COMReleaseExmut();
    }

    return( LcErrCode );
}


// ****************************************************************************
// WORD PCXSetInPipeParameter()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: input pipe mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PIN_PIPE_SET_PARAM_INFO PmPipeInParamInfo: Points to a IN_PIPE_SET_PARAM_INFO
//                              structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to define some parameters on input
// pipes, among which the selection of analogue or digital input.
//
// ****************************************************************************
WORD _CDECL_ PCXSetInPipeParameter(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN PTIME_INFO PmDelayCondition,
    IN PIN_PIPE_SET_PARAM_INFO PmPipeInParamInfo )
{
    LPIN_PIPE_SET_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                    IN_PIPES_SETTINGS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_PIPE_SET_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ipqInPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->ipqPar1 = PmPipeInParamInfo->ipspValidationMask1 ;
    LcPReq->ipqPar2 = 0;
    LcPReq->ipqSource = PmPipeInParamInfo->ipspADSelect;
    LcPReq->ipqADProperties = PmPipeInParamInfo->ipspADProperties;  // IN_PROPERTIES_MIC_48V used here

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();
    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXOfflineLoop()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// BYTE PmLoopNumber: Number of audio output - audio input couples. One couple
//                     is a loop.
//
// PLOOP_AUDIO_INFO PmLoopAudioInfo: Points to a LOOP_AUDIO_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function allows to define loops between output pipes and
// inputs pipe that are used for offline processing.
//
// ****************************************************************************

WORD _CDECL_ PCXOfflineLoop(
    IN PCX_HANDLE PmMyProcess,
    IN BYTE PmLoopNumber,
    IN PLOOP_AUDIO_INFO PmLoopAudioInfo )
{
    LPOFFLINE_LOOPS_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block
    LPLOOPS_DESC_INFO  LcPLoopDesc;      // will point to the loop descriptions
                                         // within the request block
    DWORD i;
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // no offline processing available in m/c yet
    // -----------------------------------------
    if (    (PmLoopAudioInfo->laPipeAudioOutMask > 0x03)
        ||  (PmLoopAudioInfo->laPipeAudioInMask > 0x03)
        )
       return EA_BAD_MULTICHANNEL_FORMAT;


    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                    OFFLINE_LOOPS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    MAX_OFFLINE_LOOPS_REQ_SIZE,
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->olqNbLoops = PmLoopNumber;

    // FH BUG correction 12/02/97
    // We must increment PmLoopAudioInfo

    for ( i = 0,
          LcPLoopDesc = (LPLOOPS_DESC_INFO)(LcPReq + 1);
          i < PmLoopNumber;
          i ++, LcPLoopDesc ++, PmLoopAudioInfo++ )
    {
        LcPLoopDesc->ldOutPipeMask64.QuadPart = PmLoopAudioInfo->laPipeOutMask;
        LcPLoopDesc->ldOutMask = PmLoopAudioInfo->laPipeAudioOutMask;
        LcPLoopDesc->ldInPipeMask64.QuadPart = PmLoopAudioInfo->laPipeInMask;
        LcPLoopDesc->ldInMask = PmLoopAudioInfo->laPipeAudioInMask;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();
    // returning the error code
    return LcError;
}

// ## FM (10/28/98) -- New function
// ****************************************************************************
// WORD PCXChangeMonitoringSource()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// BYTE PmChangesNumber: Number of CHANGE_MONITORING_INFO structures.
//
// PCHANGE_MONITORING_INFO PmChangeMonitInfoPtr: Points to a CHANGE_MONITORING_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function allows to change the monitoring source (audio inputs) of one
// or more audio outputs.
//
// ****************************************************************************

WORD _CDECL_ PCXChangeMonitoringSource(
    IN PCX_HANDLE               PmMyProcess,
    IN BYTE                     PmChangesNumber,
    IN PCHANGE_MONITORING_INFO  PmChangeMonitInfoPtr)
{
    LPMONIT_CHANGES_REQ_INFO LcReqPtr = 0;// will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcRespPtr = 0;     // will point to the beginning of the
                                          // response block
    LPCHANGE_DESC_INFO  LcChangeDescPtr;  // will point to the monit changes
                                          // desc. within the request block
    DWORD   i;
    WORD    LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                    CHANGE_MONIT_SOURCE_CMD,
                    (APP_HANDLE) PmMyProcess,
                    MAX_MONIT_CHANGES_REQ_SIZE,
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcReqPtr );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcReqPtr->mcqNbChanges =  PmChangesNumber ;

    for ( i = 0,
          LcChangeDescPtr = (LPCHANGE_DESC_INFO)(LcReqPtr + 1);
          i < PmChangesNumber;
          i++, LcChangeDescPtr++, PmChangeMonitInfoPtr++ )
    {
        LcChangeDescPtr->cdOutPipeMask64.QuadPart = PmChangeMonitInfoPtr->cmPipeOutMask;
        LcChangeDescPtr->cdOutMask         = PmChangeMonitInfoPtr->cmPipeAudioOutMask;
        LcChangeDescPtr->cdInPipeMask64.QuadPart  = PmChangeMonitInfoPtr->cmPipeInMask;
        LcChangeDescPtr->cdInMask          = PmChangeMonitInfoPtr->cmPipeAudioInMask;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcReqPtr,
                   0,
                   (RESP_HEADER_HANDLE) &LcRespPtr,
                   0 );

    LcError = LcRespPtr->rhCptr;
    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXNotifyPipeTime()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmOutPipeMask: Output pipe mask
//
// DWORD PmInPipeMask: Input pipe mask
//
// PTIME_INFO PmDelayCondition: the time of notification
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to ask the driver to tell when a pipe reached
//  a given time or sample count
//
// ****************************************************************************
WORD _CDECL_ PCXNotifyPipeTime(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmOutPipeMask,
    IN DWORD PmInPipeMask,
    IN PTIME_INFO PmDelayCondition)
{
    LPNOTIFY_PIPE_TIME_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                         // request block
    LPRESP_HEADER_INFO LcPResp = 0;      // will point to the beginning of the
                                         // response block

    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                    NOTIFY_TIME_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( NOTIFY_PIPE_TIME_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->nptqOutPipeMask64.QuadPart = PmOutPipeMask;
    LcPReq->nptqInPipeMask64.QuadPart  = PmInPipeMask;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->nptqDiffered = (BYTE) NOTIFIED_COMMAND_MASK ;
        LcPReq->nptqHour = PmDelayCondition->tiScheduler;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();

    // returning the error code
    return LcError;
}


// ****************************************************************************
// WORD PCXCancelDifferedCommands()
// ********************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmOutPipeMask: Output pipe mask
//
// DWORD PmInPipeMask: Input pipe mask
//
// WORD  PmCommandCode: the code of the commands to cancel
//
// PTIME_INFO PmDelayType: the type of the commands to cancel
//
// DWORD PmCommandParam1: additionnal parameters
//
// DWORD PmCommandParam2: additionnal parameters
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to ask the driver to cancel some differed commands.
//
// ****************************************************************************
WORD _CDECL_ PCXCancelDifferedCommands(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeOutMask,
    IN  DWORD               PmPipeInMask,
    IN  WORD                PmCommandCode,
    IN  PTIME_INFO          PmDelayType,
    IN  DWORD               PmCommandParam1,
    IN  DWORD               PmCommandParam2 )
{
    LPPURGE_DCMDS_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                    PURGE_DCMDS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( PURGE_DCMDS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    if ( PmDelayType != NULL )
    {
        LcPReq->pdqDiffered = PmDelayType->tiDelayed;
    }
    LcPReq->pdqParam1        = PmCommandParam1;
    LcPReq->pdqParam2        = PmCommandParam2;
    LcPReq->pdqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->pdqInPipeMask64.QuadPart  = PmPipeInMask;
    LcPReq->pdqCommandCode   = PmCommandCode;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );
    LcError = LcPResp->rhCptr;

    COMReleaseExmut();

    // returning the error code
    return LcError;
}


// ****************************************************************************
// WORD PCXSetOutAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: output pipe mask.
//
// WORD PmPipeAudioOutMask: Audio outputs pipe mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo: Points to a
//               OUT_AUDIO_SET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to  adjust the analogue and digital
// levels of an audio output, the digital level of the monitoring
// on this output, and to set the mute of this output and the
// mute of the monitoring on this output.
//
// ****************************************************************************

WORD _CDECL_ PCXSetOutAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN WORD PmPipeAudioOutMask,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo )
{
    LPOUT_AUDIO_SET_LEVELS_REQ_INFO LcPReq = 0;
                                          // will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcPResp = 0;       // will point to the beginning of the
                                          // response block
    WORD               LcErrCode = 0 ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_CTRL_FAM,
                    OUT_AUDIO_SET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_AUDIO_SET_LEVELS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->oslqPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->oslqAudioMask = PmPipeAudioOutMask;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->oslqDiffered = PmDelayCondition->tiDelayed;
        LcPReq->oslqHour = PmDelayCondition->tiScheduler;
    }
    // Fill the audio levels description
    // ---------------------------------
    LcErrCode = FillOutAudioSetLevelInfo(
                            PmSetOutLevelInfo,
                            &(LcPReq->oslqLevelInfo) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcErrCode =  LcPResp->rhCptr;
    }

    COMReleaseExmut();
    // returning the error code
    return LcErrCode;
}

// ****************************************************************************
// WORD PCXSetInAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// WORD PmPipeAudioInMask: Audio inputs pipe mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PIN_AUDIO_SET_LEVEL_INFO PmSetInLevelInfo: Points to
//        an IN_AUDIO_SET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to adjust the analog and digital input
// levels of audio input(s).
//
// ****************************************************************************

WORD _CDECL_ PCXSetInAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN WORD PmPipeAudioInMask,
    IN PTIME_INFO PmDelayCondition,
    IN PIN_AUDIO_SET_LEVEL_INFO PmSetInLevelInfo )
{
    LPIN_AUDIO_SET_LEVELS_REQ_INFO LcPReq = 0;
                                          // will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcPResp = 0;       // will point to the beginning of the
                                          // response block
    WORD               LcErrCode ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_CTRL_FAM,
                    IN_AUDIO_SET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_AUDIO_SET_LEVELS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->islqPipeMask64.QuadPart = PmPipeInMask;
    LcPReq->islqAudioMask = PmPipeAudioInMask;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->islqDiffered = PmDelayCondition->tiDelayed;
        LcPReq->islqHour = PmDelayCondition->tiScheduler;
    }

    // Fill the audio levels description
    // ---------------------------------
    LcErrCode = FillInAudioSetLevelInfo(
                            PmSetInLevelInfo,
                            &(LcPReq->islqLevelInfo) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcErrCode =  LcPResp->rhCptr;
    }

    // returning the error code
    COMReleaseExmut();

    return LcErrCode;
} // PCXSetInAudioLevel

/* -------------------------------------------------------------------------
                    HELPER FUNCTIONS FOR INPUT EFFECTS
   ---------------------vvvvvvvvvvvvvvvvvvvvv------------------------------*/

// tables with valid input float values for our effects!
// values must always grow in the array !!!!!
STATIC FLOAT gfCompThresholdTab[] = { -28.0f, -26.0f, -24.0f, -22.0f, -20.0f, -18.0f, -16.0f, -14.0f, -12.0f, -9.0f, -6.0f, -3.0f, -0.0f };
STATIC FLOAT gfCompRatioTab[] =     { 1.0f, 1.5f, 1.8f, 2.1f, 2.5f, 3.0f, 3.5f, 4.0f, 4.5f, 5.0f};     // further possible : 10.0f, 13.0f !
STATIC FLOAT gfNoiseThreshldTab[] = { -52.0f, -42.0f, -32.0f };
STATIC FLOAT gfLimiterGainTab[] =   { 0.0f, 2.0f, 4.0f, 6.0f, 8.0f, 10.0f, 12.0f, 14.0f, 16.0f };

// used by PCXSetInAudioEffect() :
// returns PmInputFloat or nearest value in the table !
// use table with only growing values !!!
// parameter PmPCheckTab is not checked!
//
STATIC FLOAT CheckFloatWithTab(FLOAT PmInputFloat, FLOAT* PmPCheckTab, WORD PmTabSize)
{
    FLOAT   LcMiddle;
    WORD    i;

    for(i=0; i<PmTabSize; i++)
    {
        if( PmInputFloat <= PmPCheckTab[i] )    // value is smaller or equal than current value in the table
            break;

        if( i >= ( PmTabSize - 1 ) )            // value is greater than all values in the table
            break;
        else
        {
            LcMiddle = ( PmPCheckTab[i] + PmPCheckTab[i+1] ) / 2.0f;
            if( PmInputFloat < LcMiddle )       // value is smaller than middle between current and next value in the table
                break;
        }
    }

    return PmPCheckTab[i];
}

// subfunction called by PCXSetInAudioEffect
// checks PIN_AUDIO_SET_EFFECT_INFO and fills in LPIN_AUDIO_EFFECT_REQ_INFO
//
STATIC WORD  Fill_Set_COMPRESSOR_LIMITER(    INOUT  PEFFECT_CL_PARAM_INFO       PmepCL,
                                         OUT    LPIN_AUDIO_EFFECT_REQ_INFO  PmPReq                  )
{
    WORD    LcRet = SUCCESS;
    FLOAT   LcCompThreshold;
    FLOAT   LcCompRatio;
    FLOAT   LcNoiseThreshold;
    FLOAT   LcOutputGain;

    // check Compressor Threshold parameter
    LcCompThreshold =   CheckFloatWithTab(  PmepCL->clpCompThreshold,
                                            gfCompThresholdTab,
                                            sizeof(gfCompThresholdTab) / sizeof(FLOAT)  );
    // check Compressor Ratio parameter
    LcCompRatio =       CheckFloatWithTab(  PmepCL->clpCompRatio,
                                            gfCompRatioTab,
                                            sizeof(gfCompRatioTab) / sizeof(FLOAT)  );
    // check Noise Threshold parameter
    LcNoiseThreshold =  CheckFloatWithTab(  PmepCL->clpNoiseThreshold,
                                            gfNoiseThreshldTab,
                                            sizeof(gfNoiseThreshldTab) / sizeof(FLOAT)  );
    // check Limiter Output Gain parameter
    LcOutputGain =      CheckFloatWithTab(  PmepCL->clpOutputGain,
                                            gfLimiterGainTab,
                                            sizeof(gfLimiterGainTab) / sizeof(FLOAT)  );

    // if min. one float value had to be rounded, inform application about this!
    //
    if( ( LcCompThreshold   != PmepCL->clpCompThreshold )  ||
        ( LcCompRatio       != PmepCL->clpCompRatio )      ||
        ( LcNoiseThreshold  != PmepCL->clpNoiseThreshold ) ||
        ( LcOutputGain      != PmepCL->clpOutputGain )     )
    {
        PmepCL->clpCompThreshold =     LcCompThreshold;
        PmepCL->clpCompRatio =         LcCompRatio;
        PmepCL->clpNoiseThreshold =    LcNoiseThreshold;
        PmepCL->clpOutputGain =        LcOutputGain;

        LcRet = WA_PARAMETER_ADJUSTED;
    }

#define FLOAT_2_DRVPARAM(a) ((LONG)((a) * 100.0f + ((a)<0 ? -0.5f : 0.5f)))

    // transform (-28.0dB .. 0.0dB)  ->  (-2800..0)
    PmPReq->iaqParams.hw.iaqCompThreshold  = FLOAT_2_DRVPARAM( LcCompThreshold );
    // transform (1/1 .. 13/1)  ->  (100..1300)
    PmPReq->iaqParams.hw.iaqCompRatio      = FLOAT_2_DRVPARAM( LcCompRatio );
    // transform (-52.0dB .. -32.0dB)  ->  (-5200 .. -3200)
    PmPReq->iaqParams.hw.iaqNoiseThreshold = FLOAT_2_DRVPARAM( LcNoiseThreshold );
    // transform (0.0dB .. 16.0dB)  ->  (0 .. 1600)
    PmPReq->iaqParams.hw.iaqOutputGain     = FLOAT_2_DRVPARAM( LcOutputGain );

    return LcRet;
}; // Fill_Set_COMPRESSOR_LIMITER


// Process GET command only
STATIC WORD  Fill_Get_InAudioEffectParam(    INOUT  PIN_AUDIO_SET_EFFECT_INFO   PmPInAudioSetEffectInfo ,
                                             OUT    LPIN_AUDIO_EFFECT_REQ_INFO  PmPReq
                                         )
{
    PEFFECT_MX_PARAM_INFO   LcMX;
    PEFFECT_EQ_PARAM_INFO   LcEQ;
    WORD        LcPipeNum = 0;

#ifdef WIN32
    _ASSERT(PmPReq->iaqHandle.iaqInPipeMask);
#endif


    LcPipeNum = UTIDWMask2Word(PmPReq->iaqHandle.iaqInPipeMask);
   
    switch(PmPInAudioSetEffectInfo->iaseIdentifier)
    {
    case HW_EFFECT_MAXIMIZER:
        LcMX            = &(PmPInAudioSetEffectInfo->iaseParams.epMX);
        /* Retrieve MX SETTINGS */
        memcpy( LcMX, &(TbAudioInMXSettings[LcPipeNum]),sizeof(PEFFECT_MX_PARAM_INFO));

        /* check wheter this effect was ever set for this application*/
        if (LcMX->mxpRelease == 0 ) return EA_BAD_EFFECT_ID;
        break;
        
    case HW_EFFECT_EQUALIZER:
        LcEQ            = &(PmPInAudioSetEffectInfo->iaseParams.epEQ);

#ifdef WIN32
        _ASSERT(LcEQ->eqpBand > 0);
        _ASSERT(LcEQ->eqpBand < 4);
#endif

        /* Retrieve EQ SETTINGS */
        memcpy( LcEQ, &(TbAudioInEQSettings[LcPipeNum][LcEQ->eqpBand-1]), sizeof(PEFFECT_EQ_PARAM_INFO));

        /* check wheter this effect was ever set for this application*/
        if (LcEQ->eqpFrequency != 0 ) break;
        
    default:
        return EA_BAD_EFFECT_ID;
    }

    return SUCCESS;
} // Fill_Get_InAudioEffectParam
//

// Process SET command only
//
STATIC WORD  Fill_Set_InAudioEffectParam(    IN  PCX_HANDLE          PmMyProcess,
                                             INOUT  PIN_AUDIO_SET_EFFECT_INFO   PmPInAudioSetEffectInfo ,
                                             OUT    LPIN_AUDIO_EFFECT_REQ_INFO  PmPReq
                                         )
{
    return EA_BAD_EFFECT_ID;
} // Fill_Set_InAudioEffect


// Process Remove command only
//
STATIC WORD  Fill_Remove_InAudioEffectParam(OUT    LPIN_AUDIO_EFFECT_REQ_INFO  PmPReq)
{
    return EA_BAD_EFFECT_ID;
} // Fill_Remove_InAudioEffect



STATIC WORD  Fill_Set_MAXIMIZER(    INOUT  PEFFECT_MX_PARAM_INFO       PmepMX,
                                    INOUT  LPDWORD  PmpDspParams,
                                    IN     DWORD    PmBoardFrequency,
                                    INOUT  LPWORD  PmpAudioMask)
{
    return EA_BAD_EFFECT_ID;
}; // Fill_Set_MAXIMIZER

#undef DOUBLE_2_DSP


STATIC WORD  Fill_Set_EQUALIZER(    INOUT  PEFFECT_EQ_PARAM_INFO       PmepEQ,
                                    INOUT  LPDWORD                      PmpDspParams,
                                    IN     DWORD  PmBoardFrequency)
{
    return EA_BAD_EFFECT_ID;
}; // Fill_Set_EQUALIZER

/* --------------------------^^^^^^^^^^^^^^^^^^-----------------------------
                END OF HELPER FUNCTIONS FOR INPUT EFFECTS
   --------------------------------------------------------------------------*/


// ****************************************************************************
// WORD PCXInAudioEffect()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess:      Handle of the calling process.
//
// DWORD PmPipeInMask:          Input pipe mask
//
// DWORD  PmPipeAudioInMask:    Input audio mask
//
// PIN_AUDIO_SET_EFFECT_INFO    PmPInAudioSetEffectInfo: effect parameters
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to adjust (hardware) input effects
//
// ****************************************************************************
WORD _CDECL_ PCXInAudioEffect(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeInMask, 
    IN  WORD                PmPipeAudioInMask, 
    IN  PIN_AUDIO_SET_EFFECT_INFO   PmPInAudioEffectInfo )
{
    WORD    LcError = SUCCESS;
    WORD    LcPipeNumber;
//    _TCHAR temp[256];

    LPIN_AUDIO_EFFECT_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                            // request block
    LPRESP_HEADER_INFO LcPResp = 0;         // will point to the beginning of the
                                            // response block
    if( PmPInAudioEffectInfo == NULL )
        return EA_INVALID_POINTER;

    if( PmPipeInMask == 0 )
        return EA_INVALID_PIPE;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    // only one pipe accepted, this is checked by Driver for now
    //
    LcPipeNumber = UTIDWMask2Word(PmPipeInMask); // LSBit pos

        // Filter multichannel
    if  (TbPipeInfo_IN[LcPipeNumber].piPipeNbChannels > 2)
          return EA_BAD_MULTICHANNEL_FORMAT;
    
    // MX : all audios of this pipe, because they are correlated 
    //  
  /*  _stprintf(temp,_T("LXESAPI : PCXInAudioEffect : PmPipeAudioInMask = %lx, Pipe = %lx\n"),PmPipeAudioInMask,TbPipeInfo_IN[LcPipeNumber].piPipeAudioMask);
    OutputDebugString(temp);*/

    if(   ( PmPipeAudioInMask == 0 )
        ||(     (PmPipeAudioInMask != TbPipeInfo_IN[LcPipeNumber].piPipeAudioMask )
             && (PmPInAudioEffectInfo->iaseIdentifier == HW_EFFECT_MAXIMIZER)
          )
      )
      return EA_INVALID_PARAMETER;
    
    if( !COMWaitExmut() )
        return EA_CANT_TAKE_MUTEX;
    
   
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_PIPE_CTRL_FAM,
                    IN_AUDIO_SET_EFFECT_CMD, // SetInPipeAudioEffect
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_AUDIO_EFFECT_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->iaqHandle.iaqInPipeMask    = PmPipeInMask;
    LcPReq->iaqInAudioMask          = PmPipeAudioInMask;
    LcPReq->iaqIdentifier           = PmPInAudioEffectInfo->iaseIdentifier;
    LcPReq->iaqCommand              = PmPInAudioEffectInfo->iaseCommand;

    switch(PmPInAudioEffectInfo->iaseCommand)
    {
    case EFFECT_CMD_SET:
        LcError =   Fill_Set_InAudioEffectParam   (PmMyProcess, PmPInAudioEffectInfo, LcPReq);
        break;
    case EFFECT_CMD_GET:
        LcError =   Fill_Get_InAudioEffectParam   (PmPInAudioEffectInfo, LcPReq);
        // does no go to Driver.
        goto _exit;
    case EFFECT_CMD_REMOVE:
        LcError =   Fill_Remove_InAudioEffectParam(/*PInAudioEffectInfo,*/ LcPReq);
        break;
    default:
        LcError = EA_INVALID_PARAMETER;
    };

    // continue if there was no error ( warnings accepted )
    //
    if( ! (LcError & ERROR_MASK ) )
    {
        // sending the request to the driver
        // ---------------------------------
        COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                       0,
                       (RESP_HEADER_HANDLE) &LcPResp,
                       0 );

        // keep the possible warning if driver reported SUCCESS
        //
        if(LcPResp->rhCptr != SUCCESS)
        {
            LcError = LcPResp->rhCptr;
        }
    }
_exit:

    COMReleaseExmut();

    // returning the error code
    return LcError;
} // PCXInAudioEffect



// ****************************************************************************
// WORD PCXSetInAudioEffect()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess:      Handle of the calling process.
//
// DWORD PmPipeInMask:          Input pipe mask
//
// DWORD  PmPipeAudioInMask:    Input audio mask
//
// PIN_AUDIO_SET_EFFECT_INFO    PmPInAudioSetEffectInfo: effect parameters
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to adjust (hardware) input effects
//
// ****************************************************************************
WORD _CDECL_ PCXSetInAudioEffect(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeInMask, 
    IN  WORD                PmPipeAudioInMask, 
    IN  PTIME_INFO          PmPDelayCondition,
    IN  PIN_AUDIO_SET_EFFECT_INFO   PmPInAudioSetEffectInfo )
{
    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if( PmPInAudioSetEffectInfo == NULL )
        return EA_INVALID_POINTER;

    if ( (PmPDelayCondition != NULL) && (PmPDelayCondition->tiDelayed !=0 ) )
        return EA_INVALID_PARAMETER;

    if(PmPInAudioSetEffectInfo->iaseCommand != EFFECT_CMD_SET) 
        return EA_INVALID_PARAMETER;

    return PCXInAudioEffect(PmMyProcess, PmPipeInMask, PmPipeAudioInMask, PmPInAudioSetEffectInfo);
} // PCXSetInAudioEffect



// ****************************************************************************
// WORD PCXSetInBoardAudioEffect()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess:      Handle of the calling process.
//
// PAUDIO_DECL_INFO             AudioDeclInfo
//
// PTIME_INFO                   TIME_INFO not used
//
// PIN_AUDIO_SET_EFFECT_INFO    PmPInAudioSetEffectInfo: effect parameters
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to adjust board (hardware) input effects
//
// ****************************************************************************
WORD _CDECL_ PCXSetInBoardAudioEffect(
    IN PCX_HANDLE                   PmMyProcess,
    IN PAUDIO_DECL_INFO             PmPAudioDeclInfo,
    IN PTIME_INFO                   PmPDelayCondition,
    IN PIN_AUDIO_SET_EFFECT_INFO    PmPInAudioSetEffectInfo )
{
    WORD    LcError = SUCCESS;

    LPIN_AUDIO_EFFECT_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                            // request block
    LPRESP_HEADER_INFO LcPResp = 0;         // will point to the beginning of the
                                            // response block
    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if( PmPInAudioSetEffectInfo == NULL )
        return EA_INVALID_POINTER;

    if( PmPAudioDeclInfo == NULL )
        return EA_INVALID_POINTER;

    if( !(PmPAudioDeclInfo->adAudioAttributes & AUDIO_IN) || (PmPAudioDeclInfo->adDspNumber != 0) )
        return EA_INVALID_PARAMETER;

    if ( (PmPDelayCondition != NULL) && (PmPDelayCondition->tiDelayed !=0 ) )
        return EA_INVALID_PARAMETER;

    if ( PmPAudioDeclInfo->adBoardNumber >= MAX_BOARD)  
        return ED_INVALID_BOARD;

    if( !COMWaitExmut() )
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_CTRL_FAM,
                    IN_AUDIO_SET_EFFECT_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_AUDIO_EFFECT_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------

    LcPReq->iaqHandle.iaqInBoardNum  = PmPAudioDeclInfo->adBoardNumber; 
    LcPReq->iaqInAudioMask        = UTIMaskWord(  PmPAudioDeclInfo->adAudioNumber ); // not forget to pass a mask !
    LcPReq->iaqIdentifier         = PmPInAudioSetEffectInfo->iaseIdentifier;
    LcPReq->iaqCommand            = EFFECT_CMD_SET;

    switch ( PmPInAudioSetEffectInfo->iaseIdentifier) 
    {
    case HW_EFFECT_COMPRESSOR_LIMITER:
        // only supported for this kind of effects.
        LcError = Fill_Set_COMPRESSOR_LIMITER( &(PmPInAudioSetEffectInfo->iaseParams.epCL), LcPReq);
        break;
    case HW_EFFECT_NONE:
        LcError = SUCCESS;
        break;
    default:
        LcError = EA_BAD_EFFECT_ID;
    };

    // continue if there was no error ( warnings accepted )
    //
    if( ! (LcError & ERROR_MASK ) )
    {
        // sending the request to the driver
        // ---------------------------------
        COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                       0,
                       (RESP_HEADER_HANDLE) &LcPResp,
                       0 );

        // keep the possible warning if driver reported SUCCESS
        //
        if(LcPResp->rhCptr != SUCCESS)
        {
            LcError = LcPResp->rhCptr;
        }
    }
   
    COMReleaseExmut();

    // returning the error code
    return LcError;
} // PCXSetInBoardAudioEffect

// ****************************************************************************
// WORD PCXSetInBoardAudioParameter()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess:      Handle of the calling process.
//
// PAUDIO_DECL_INFO             AudioDeclInfo
//
// PIN_AUDIO_SET_PARAM_INFO     PmPInAudioSetParamInfo: parameters for in pipe
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to switch on/off Microphone 48 Volt without pipe dependency !
//
// ****************************************************************************
WORD _CDECL_ PCXSetInBoardAudioParameter(
    IN PCX_HANDLE           PmMyProcess,
    IN PAUDIO_DECL_INFO     PmPAudioDeclInfo,
    IN PIN_AUDIO_SET_PARAM_INFO  PmPInAudioSetParamInfo)
{
    WORD    LcError = SUCCESS;

    LPIN_AUDIO_PARAM_REQ_INFO LcPReq = 0;   // will point to the beginning of the
                                            // request block
    LPRESP_HEADER_INFO LcPResp = 0;         // will point to the beginning of the
                                            // response block
    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if( PmPInAudioSetParamInfo == NULL )
        return EA_INVALID_POINTER;

    if( PmPAudioDeclInfo == NULL )
        return EA_INVALID_POINTER;

    if( !(PmPAudioDeclInfo->adAudioAttributes & AUDIO_IN) || (PmPAudioDeclInfo->adDspNumber != 0) )
        return EA_INVALID_PARAMETER;

    if( !COMWaitExmut() )
        return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_CTRL_FAM,
                    IN_AUDIO_SET_PARAM_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_AUDIO_EFFECT_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the fields of the request block
    // ---------------------------------------------
    LcPReq->iapqInBoardMask = UTIMaskDWord( PmPAudioDeclInfo->adBoardNumber ); // not forget to pass a mask !
    LcPReq->iapqInAudioMask = UTIMaskWord(  PmPAudioDeclInfo->adAudioNumber ); // not forget to pass a mask !
    LcPReq->iapqValidationMask = PmPInAudioSetParamInfo->iaspValidationMask;
    LcPReq->iapqMic48V = PmPInAudioSetParamInfo->iaspMic48V;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    // keep the possible warning if driver reported SUCCESS
    //
    if(LcPResp->rhCptr != SUCCESS)
    {
        LcError = LcPResp->rhCptr;
    }

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXSetOutBoardAudioLevel()
// ********************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmOutAudio: Audio output description.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo: Points to a
//               OUT_AUDIO_SET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to  adjust the analogue and digital
// levels of an audio output, the digital level of the monitoring
// on this output, and to set the mute of this output and the
// mute of the monitoring on this output.
//
// ****************************************************************************
WORD _CDECL_ PCXSetOutBoardAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN PAUDIO_DECL_INFO PmOutAudio,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_AUDIO_SET_LEVEL_INFO PmSetOutLevelInfo )
{
    LPOUT_BOARD_AUDIO_SET_LEVELS_REQ_INFO LcPReq = 0;
                                          // will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcPResp = 0;       // will point to the beginning of the
                                          // response block
    WORD               LcErrCode = SUCCESS ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_CTRL_FAM,
                    OUT_AUDIO_SET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_BOARD_AUDIO_SET_LEVELS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmOutAudio, &(LcPReq->obslqAudio) );

    // Filling the other fields of the request block
    // ---------------------------------------------
    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->obslqDiffered = PmDelayCondition->tiDelayed;
        LcPReq->obslqHour = PmDelayCondition->tiScheduler;
    }

    // Fill the audio levels description
    // ---------------------------------
    LcErrCode = FillOutAudioSetLevelInfo(
                            PmSetOutLevelInfo,
                            &(LcPReq->obslqLevelInfo) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcErrCode =  LcPResp->rhCptr;
    }

    COMReleaseExmut();
    // returning the error code
    return LcErrCode;
}

// ****************************************************************************
// WORD PCXSetInBoardAudioLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// PAUDIO_DECL_INFO PmOutAudio: Audio input description.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PIN_AUDIO_SET_LEVEL_INFO PmSetInLevelInfo: Points to
//        an IN_AUDIO_SET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function is used to adjust the analog and digital input
// levels of audio input(s).
//
// ****************************************************************************
WORD _CDECL_ PCXSetInBoardAudioLevel(
    IN PCX_HANDLE PmMyProcess,
    IN PAUDIO_DECL_INFO PmInAudio,
    IN PTIME_INFO PmDelayCondition,
    IN PIN_AUDIO_SET_LEVEL_INFO PmSetInLevelInfo )
{
    LPIN_BOARD_AUDIO_SET_LEVELS_REQ_INFO LcPReq = 0;
                                          // will point to the beginning of the
                                          // request block
    LPRESP_HEADER_INFO LcPResp = 0;       // will point to the beginning of the
                                          // response block
    WORD               LcErrCode ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( IO_BOARD_CTRL_FAM,
                    IN_AUDIO_SET_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_BOARD_AUDIO_SET_LEVELS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Copy the audio description into the request block
    // -------------------------------------------------
    GENAudioDecl2PipeAudioInfo( PmInAudio, &(LcPReq->ibslqAudio) );

    // Filling the other fields of the request block
    // ---------------------------------------------
    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->ibslqDiffered = PmDelayCondition->tiDelayed;
        LcPReq->ibslqHour = PmDelayCondition->tiScheduler;
    }

    // Fill the audio levels description
    // ---------------------------------
    LcErrCode = FillInAudioSetLevelInfo(
                            PmSetInLevelInfo,
                            &(LcPReq->ibslqLevelInfo) );

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcErrCode =  LcPResp->rhCptr;
    }

    // returning the error code
    COMReleaseExmut();

    return LcErrCode;
}

// ****************************************************************************
// WORD PCXStartStream()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function sets stream(s) in START state. The playback of
// the stream(s) effectively starts if the pipe(s) to which
// belong(s) the stream(s) is (are) in START state.
// Remarks :
// This function doesnt do any inter-pipe synchronization. The
// start is consecutively performed on all the pipes given in
// parameters.
//
// ****************************************************************************

WORD _CDECL_ PCXStartStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    LcValret = CmdStream( PmMyProcess,
                     PmPipeOutMask,
                     PmPipeInMask,
                     PmStreamMask,
                     PmDelayCondition,
                     START_STREAM_CMD );
    COMReleaseExmut();
    return LcValret;
}

// ****************************************************************************
// WORD PCXPauseStream()
// ***************************
//
// Input parameters :
// ****************
//
// APP_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function sets stream(s) in PAUSE state.
// Remarks :
// This function doesnt do any inter-pipe synchronization. The
// pause is consecutively performed on all the pipes given in
// parameters.
//
// ****************************************************************************

WORD _CDECL_ PCXPauseStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    LcValret = CmdStream( PmMyProcess,
                     PmPipeOutMask,
                     PmPipeInMask,
                     PmStreamMask,
                     PmDelayCondition,
                     PAUSE_STREAM_CMD );
    COMReleaseExmut();
    return LcValret;
}

// ****************************************************************************
// WORD PCXStopStream()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function stops all the streams it receives in parameter.
// Remarks:
// This function doesnt do any inter-pipe synchronization. The
// stop is consecutively performed on all the pipes given in
// parameters.
// First of all, a PAUSE is done on all the streams given in
// parameter. Then a STOP is done; this means that:
// - if the pipe is in record, all the sound stored on the card
// is transfered in the sound buffers (if some of them are
// available for the driver).
// - all the pipe and stream effects are disabled (time-
// stretching, pitch-shifting, frequency change, equalization).
// - stream levels are set to their default values:
//
// - LEVEL_LC_O1 = 0db
//
// - LEVEL_LC_O2 = -100 dB
//
// - LEVEL_RC_O1 = -100db
//
// - LEVEL_RC_O2 = 0db
//
// - LEVEL_S_1 = 0db
//
// - LEVEL_S_2 = 0db
// - The stream mutes are disabled.
// - The stream is in pause.
// - The differed commands on the concerned streams are disabled.
//
// ****************************************************************************

WORD _CDECL_ PCXStopStream(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition )
{
    WORD LcValret;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    LcValret = CmdStream( PmMyProcess,
                     PmPipeOutMask,
                     PmPipeInMask,
                     PmStreamMask,
                     NULL,
                     STOP_STREAM_CMD );

    COMReleaseExmut();

#ifdef WIN32
    if ( PmPipeInMask ) {
        // ## FM (07/17/98) -- Wait for 3 ms to allow the other threads completion
        //                     routines (in a multi-threaded application) to
        //                     terminate before this thread is scheduled again.
        //
        Sleep(3) ;
    }
#endif

    return LcValret;
}

// ****************************************************************************
// WORD PCXSetOutStreamFormat()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PSOUND_FORMAT_INFO PmOutStreamFormatInfo: Points to an
//                                          SOUND_FORMAT_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function (which can have a delayed effect), defines the
// type of data that the output streams will receive.
//
// ****************************************************************************


//
WORD  BuildMapping(  MY_PIPE_INFO*   IN  pPipeInfo,
                     PDWORD  INOUT      pStreamChannelMask,
                     PDWORD  INOUT      pChannels,
                     PSTREAM_SETTINGS   INOUT   pStreamSettings)
{
    WORD LcRet = SUCCESS;
    DWORD LcChannels = *pChannels;
    DWORD LcChannelMask; 
    DWORD LcPipeChannelMask; 

	LcPipeChannelMask = pPipeInfo->piPipeChannelMask;

    switch (LcChannels) {
    case 0:
        LcChannelMask = *pStreamChannelMask;
        break;
    case 1:
        LcChannelMask = 0;
        break;
    case 2:
        LcChannelMask = SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT;
        break;
    default:
        LcChannelMask = 0;
    }

    // check stream/pipe format dependendy 
    //
    switch (LcChannelMask) {
    case 0:
        // mapping proprietaire
        if (LcChannels == 0) {
                        NPNTAPITRACE("no channelmask specified, and null number of channels wanted\n");
                        return EA_BAD_MULTICHANNEL_FORMAT;
        };

        if (LcChannels > 2) {
            pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        }
        break;

    case SPEAKER_FRONT_LEFT:
    case SPEAKER_FRONT_CENTER:
        // mono in fact 
        LcChannels = 1;
      //  pStreamSettings->ChannelMapping[0] = (1 << BIT_mapping_direct); 
        break;
        
    case SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT:
    case SPEAKER_STEREO_LEFT|SPEAKER_STEREO_RIGHT:
        // stereo in fact 
        LcChannels = 2;

        if (LcPipeChannelMask == DIGIGRAM_5POINT1_PLUS2) {
            pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        }
        
        if (LcPipeChannelMask == DIGIGRAM_7POINT1_PLUS2)    {
            pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        }
        break;

    case DIGIGRAM_5POINT1:
        if (LcPipeChannelMask == DIGIGRAM_7POINT1_PLUS2)   // downmix missing
        {
            NPNTAPITRACE("5.1 stream over 7.1+2 pipe!\n");
            return EA_BAD_MULTICHANNEL_FORMAT;
        }
        LcChannels = 6;
        pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        break;

    case DIGIGRAM_7POINT1:
            if (   ( LcPipeChannelMask == DIGIGRAM_5POINT1) 
                 ||( LcPipeChannelMask == DIGIGRAM_5POINT1_PLUS2)   // downmix missing
                 ||( LcPipeChannelMask <= 0x3))                     // downmix missing
           {
                NPNTAPITRACE("7.1 stream over 5xx ou less!\n");
                return EA_BAD_MULTICHANNEL_FORMAT;
            }
            LcChannels = 8;
            pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
            break;
    
    case DIGIGRAM_5POINT1_PLUS2:
        LcChannels = 8;
        pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        break;
    
    case DIGIGRAM_7POINT1_PLUS2:
        if (    (LcPipeChannelMask == DIGIGRAM_5POINT1)
             || (LcPipeChannelMask == DIGIGRAM_5POINT1_PLUS2))
        {
            NPNTAPITRACE("7.1+2 stream over 5xx pipe!\n");
            return EA_BAD_MULTICHANNEL_FORMAT;
        }
        LcChannels = 10;
        pStreamSettings->DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
        break;
    default:
        // mapping propritaires
        return EA_BAD_MULTICHANNEL_FORMAT;
        } // switch
        
        // tell the DSP howmany channels
         pStreamSettings->DeCodeForm1 |= ( LcChannels & 0x01FF);

        *pChannels = LcChannels;
        *pStreamChannelMask = LcChannelMask;


		// make sure the stream is not bigger than the pipe 
		// (for now)
		if (LcChannels > pPipeInfo->piPipeNbChannels ) 
			return EA_BAD_SOUND_FORMAT;


        return LcRet;
}; // BuildMapping

WORD _CDECL_ PCXSetOutStreamFormat(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition,
    IN PSOUND_FORMAT_INFO PmStreamFormatInfo )
{
    LPOUT_STREAM_SET_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                           // request block
    LPRESP_HEADER_INFO  LcPResp = 0;       // will point to the beginning of the
                                           // response block
    WORD        LcRet ;
    DWORD       LcHeader = 0L;
    WORD        LcStreamNum, LcPipeNum;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    OUT_STREAM_SETTINGS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_STREAM_SET_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssq.PipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssq.StreamMask = PmStreamMask;

    LcStreamNum = UTIDWMask2Word(PmStreamMask);
    LcPipeNum = UTIDWMask2Word(PmPipeOutMask);

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->ssq.Differed = PmDelayCondition->tiDelayed ;
        LcPReq->ssq.Hour = PmDelayCondition->tiScheduler;
    }

    LcPReq->ssq.Par1 = OUT_STREAM_SET_PARAM_FORMAT_MASK ;
    LcPReq->ssq.Par2 = 0L ;

    LcPReq->ssq.DeCodeForm1 = 0L;

    // le mapping est maintenant calcul dans le DSP, 
    // pour l'instant on laisse cette fonction
    // pour avoir la gestion des erreur, et pour le debug
    //
    LcRet = BuildMapping(   &(TbPipeInfo_OUT[LcPipeNum]), 
                            &(PmStreamFormatInfo->sfChannelMask), 
                            &(PmStreamFormatInfo->sfChannels),
                            &(LcPReq->ssq)
                            );

    if (LcRet != SUCCESS)
    {    
            COMReleaseExmut();
            return LcRet;
    }

    ///
    if ( PmStreamFormatInfo->sfHasHeader )
    {
        LcPReq->ssq.DeCodeForm1 |= (1 << BIT_FMP_HEADER);
        if ( PmStreamFormatInfo->sfHasSuffix )
        {
            LcPReq->ssq.DeCodeForm1 |= (1 << BIT_FMP_SD);
        }
    }
    else
    {
        LcRet = AUDFormatToHeader( PmStreamFormatInfo,
                                   FALSE,               // Intel platform
                                   &LcHeader );

        if ( LcRet != SUCCESS )
        {
            COMReleaseExmut();
            return LcRet;
        }
    }

    LcPReq->ssq.DeCodeForm2 = LcHeader;
    LcPReq->ssq.dwChannelMask = PmStreamFormatInfo->sfChannelMask;
    LcPReq->ssq.dwSamplesPerChannel = PmStreamFormatInfo->sfSamplesPerChannel;


    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

   /* if (LcRet==SUCCESS) 
        // store stream channel mask for later compatibility check 
        TbPipeInfo_OUT[LcPipeNum].piStreamChannelMaskArray[LcStreamNum] = PmStreamFormatInfo->sfChannelMask;*/

    LcRet = LcPResp->rhCptr;
    COMReleaseExmut();
    // returning the error code
    return LcRet;
}

// ****************************************************************************
// WORD PCXSetOutStreamLevel()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_STREAM_SET_LEVEL_INFO PmOutStreamLevelInfo: Points to an
//                                    OUTPUT_STREAM_SET_LEVEL_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function allows to adjust the output levels on the
// streams (digital levels and mutes).
//
// ****************************************************************************

WORD _CDECL_ PCXSetOutStreamLevel(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeOutMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition,
    IN POUT_STREAM_SET_LEVEL_INFO PmOutStreamLevelInfo )
{
    LPOUT_STREAM_LEVELS_REQ_INFO LcPReq = 0;  // will point to the beginning of
                                              // request block
    LPRESP_HEADER_INFO LcPResp = 0;           // will point to the beginning of
                                              // response block
    WORD    LcErrCode ;
    WORD    LcRetCode = SUCCESS ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    OUT_STREAM_LEVELS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_STREAM_LEVELS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->slqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->slqStreamMask = PmStreamMask;
    LcPReq->slqPar1 = PmOutStreamLevelInfo->osslValidationMask1;
    LcPReq->slqPar2 = PmOutStreamLevelInfo->osslValidationMask2;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->slqDiffered = PmDelayCondition->tiDelayed ;
        LcPReq->slqHour = PmDelayCondition->tiScheduler;
    }

    // Converts all digital levels from user to driver format
    // -------------------------------------------------------

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_LEFT_AUDIO1 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslLeftChannelToOut1DigitalLevel,
                         &(LcPReq->slqLeftChannelToOutput1DigitalLevel) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_LEFT_AUDIO2 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslLeftChannelToOut2DigitalLevel,
                         &(LcPReq->slqLeftChannelToOutput2DigitalLevel) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_RIGHT_AUDIO1 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslRightChannelToOut1DigitalLevel,
                         &(LcPReq->slqRightChannelToOutput1DigitalLevel) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_RIGHT_AUDIO2 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslRightChannelToOut2DigitalLevel,
                         &(LcPReq->slqRightChannelToOutput2DigitalLevel) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_STREAM_1 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslDigitalLevel1,
                         &(LcPReq->slqDigitalLevel1) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    if ( LcPReq->slqPar1 & OUT_STREAM_SET_LEVEL_STREAM_2 )
    {
        LcErrCode = Api2DrvDLevel(
                         PmOutStreamLevelInfo->osslDigitalLevel2,
                         &(LcPReq->slqDigitalLevel2) );
        if ( LcErrCode != SUCCESS )
        {
            LcRetCode = LcErrCode ;
        }
    }

    LcPReq->slqMute1 = PmOutStreamLevelInfo->osslMute1;
    LcPReq->slqMute2 = PmOutStreamLevelInfo->osslMute2;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcRetCode = LcPResp->rhCptr;
    }
    COMReleaseExmut();
    // returning the error code
    return LcRetCode;
}

// ****************************************************************************
// WORD PCXSetOutStreamLevelCurve()
// ********************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// WORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_STREAM_SET_LCURVE_INFO PmSetOutStreamLCurveInfo: Points to an
//                                    OUT_STREAM_SET_LCURVE_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function allows to adjust the output levels on the
// streams (digital levels and mutes).
//
// ****************************************************************************
WORD _CDECL_ PCXSetOutStreamLevelCurve(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmStreamMask,
    IN  PTIME_INFO PmDelayCondition,
    IN  POUT_STREAM_SET_LCURVE_INFO PmSetOutStreamLCurveInfo)
{
    LPOUT_STREAM_LCURVE_REQ_INFO LcPReq = 0;  // will point to the beginning of
                                              // request block
    LPRESP_HEADER_INFO LcPResp = 0;           // will point to the beginning of
                                              // response block
    WORD    LcErrCode ;
    WORD    LcRetCode = SUCCESS ;
    WORD    i ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;
    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    OUT_STREAM_LCURVE_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_STREAM_LCURVE_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->oslqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->oslqStreamMask = PmStreamMask;
    LcPReq->oslqPar1 = PmSetOutStreamLCurveInfo->osnlValidationMask1;
    LcPReq->oslqPar2 = 0;
    LcPReq->oslqLevelNumber = PmSetOutStreamLCurveInfo->osnlNumberOfValues;
    LcPReq->oslqStep = PmSetOutStreamLCurveInfo->osnlStep;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->oslqDiffered = PmDelayCondition->tiDelayed ;
        LcPReq->oslqHour = PmDelayCondition->tiScheduler;
    }

    // Only stream level 1 can be selected
    // -----------------------------------
    if ( LcPReq->oslqPar1 == 0 )
    {
        COMReleaseExmut();
        return SUCCESS ;
    }
    else if (   ( LcPReq->oslqPar1 & OUT_STREAM_SET_LEVEL_STREAM_2 )
             || ( LcPReq->oslqPar1 & OUT_STREAM_SET_LEVEL_STREAM_1 ) )
    {
        // Converts all digital levels from user to driver format
        // -------------------------------------------------------
        for ( i = 0 ; i < PmSetOutStreamLCurveInfo->osnlNumberOfValues ; i++ )
        {
            // Convert
            // -------
            LcErrCode = Api2DrvDLevel(
                         PmSetOutStreamLCurveInfo->osnlLevelValues[i],
                         &(LcPReq->oslqLevelValue[i]) );

            if ( LcErrCode != SUCCESS )
            {
                LcRetCode = LcErrCode ;
            }
        }
        // Reset all remaining values
        // --------------------------
        for ( i = PmSetOutStreamLCurveInfo->osnlNumberOfValues ; i < MAX_CURVE_LEVELS ; i++ )
        {
            LcPReq->oslqLevelValue[i] = 0 ;
        }

    }
    else
    {
        COMReleaseExmut();
        return WA_COMMAND_NOT_AVAILABLE ;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    if ( LcPResp->rhCptr != SUCCESS )
    {
        LcRetCode = LcPResp->rhCptr;
    }
    COMReleaseExmut();
    // returning the error code
    return LcRetCode;
}

// ****************************************************************************
// WORD PCXSetOutStreamExtraParameters()
// *************************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// WORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// POUT_STREAM_EPARAM_INFO PmStreamExtraParamInfo: Points to an
//                                    OUT_STREAM_EPARAM_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// This function allows to set some parameters (other than format) on the
// stream
//
// ****************************************************************************
WORD _CDECL_ PCXSetOutStreamExtraParameters(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmStreamMask,
    IN  PTIME_INFO PmDelayCondition,
    IN  POUT_STREAM_EPARAM_INFO PmStreamExtraParameters)
{
    LPOUT_STREAM_SET_REQ_INFO LcPReq = 0;  // will point to the beginning of
                                           // request block
    LPRESP_HEADER_INFO LcPResp = 0;        // will point to the beginning of
                                           // response block
    WORD    LcRetCode = SUCCESS ;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    OUT_STREAM_SETTINGS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( OUT_STREAM_SET_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->ssq.Differed = PmDelayCondition->tiDelayed ;
        LcPReq->ssq.Hour = PmDelayCondition->tiScheduler;
    }

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->ssq.PipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->ssq.StreamMask = PmStreamMask;
    LcPReq->ssq.Par1 = 0;

    // Only stream level 1 can be selected
    // -----------------------------------
    if ( PmStreamExtraParameters->osseValidationMask1 == OUT_STREAM_SET_SYNC_LEVEL1_MASK )
    {
        LcPReq->ssq.Par1 = OUT_STREAM_SET_PARAM_AUTOMAT_MASK;
        LcPReq->ssq.Automation = (BYTE) PmStreamExtraParameters->osseSyncLevelSettings;
    }
    else
    {
        LcRetCode = WA_COMMAND_NOT_AVAILABLE;
    }

    // If something has been selected...
    // ---------------------------------
    if ( LcPReq->ssq.Par1 != 0 )
    {

        // sending the request to the driver
        // ---------------------------------
        COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                       0,
                       (RESP_HEADER_HANDLE) &LcPResp,
                       0 );
    }

    COMReleaseExmut();
    // returning the error code
    return LcRetCode;
}

// ****************************************************************************
// WORD PCXSetInStreamFormat()
// ***************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// PTIME_INFO PmDelayCondition: Points to a TIME_INFO structure.
//
// PSOUND_FORMAT_INFO PmStreamFormatInfo: Points to an
//                                        SOUND_FORMAT_INFO structure.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
//
// ****************************************************************************
WORD _CDECL_ PCXSetInStreamFormat(
    IN PCX_HANDLE PmMyProcess,
    IN DWORD PmPipeInMask,
    IN DWORD PmStreamMask,
    IN PTIME_INFO PmDelayCondition,
    IN PSOUND_FORMAT_INFO PmStreamFormatInfo )
{
    LPIN_STREAM_SETTINGS_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                               // request block
    LPRESP_HEADER_INFO LcPResp = 0;            // will point to the beginning of the
                                               // response block

    WORD        LcRet, i, LcPipeNum; 
    DWORD       LcChannelMask, LcPipeChannels, LcChannels, LcHeader = 0L;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!PmStreamFormatInfo)
       return EA_INVALID_POINTER;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    IN_STREAM_SETTINGS_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( IN_STREAM_SETTINGS_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->isq.PipeMask64.QuadPart = PmPipeInMask;
    LcPReq->isq.StreamMask = PmStreamMask;
    LcPReq->isq.Par1 = IN_STREAM_SET_PARAM_FORMAT_MASK;
    LcPReq->isq.Par2 = 0L;

    LcPipeNum           =       UTIDWMask2Word(PmPipeInMask);
    LcChannels          =       PmStreamFormatInfo->sfChannels;
    LcPipeChannels      =       TbPipeInfo_IN[LcPipeNum].piPipeNbChannels; 
    LcHeader = 0L;

    // Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
        LcPReq->isq.Differed = PmDelayCondition->tiDelayed ;
        LcPReq->isq.Hour = PmDelayCondition->tiScheduler;
    }

    LcRet = AUDFormatToHeader( PmStreamFormatInfo,
                               FALSE,               // Intel platform
                               &LcHeader );
    if ( LcRet != SUCCESS )
    {
        COMReleaseExmut();
        return LcRet;
    }

    LcPReq->isq.DeCodeForm1 = 0L;
    if ( PmStreamFormatInfo->sfHasHeader )
    {
        // tester si la frequence est autorisee
        if (PmStreamFormatInfo->sfFrequency > 48000)
        {
            COMReleaseExmut();
            return EA_BAD_FREQUENCY_VALUE;
        }

        LcPReq->isq.DeCodeForm1 |= (1 << BIT_FMP_HEADER);
        if ( PmStreamFormatInfo->sfHasSuffix )
        {
            LcPReq->isq.DeCodeForm1 |= (1 << BIT_FMP_SD);
        }
    }

    // sfChannelMask prime sur channelmask
    //
    switch (LcChannels) {
    case 0:
        // if a channelmask was given, it must ba a standard one.
        LcChannelMask = PmStreamFormatInfo->sfChannelMask;
        break;
    case 1:
        LcChannelMask = SPEAKER_FRONT_LEFT;
        break;
    case 2:
        LcChannelMask = SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT;
        break;
    default:
        LcChannelMask = 0;
    }

    // check stream/pipe format dependendy 
    //
    switch (LcChannelMask) {
    case SPEAKER_FRONT_LEFT:
    case SPEAKER_FRONT_CENTER:
        // mono in fact 
        LcChannels = 1;
        break;
    case SPEAKER_FRONT_LEFT|SPEAKER_FRONT_RIGHT:
    case SPEAKER_STEREO_LEFT|SPEAKER_STEREO_RIGHT:
        // stereo in fact 
        LcChannels = 2;
        break;
    case DIGIGRAM_5POINT1:
        LcChannels = 6;
        break;
    case DIGIGRAM_7POINT1:
    case DIGIGRAM_5POINT1_PLUS2:
        LcChannels = 8;
        break;
    case DIGIGRAM_7POINT1_PLUS2:
        LcChannels = 10;
     } // switch
        

    // at this point, if a non standard channel mas k was given, LcChannels may be zero ==>  error case
    //
    if (LcChannels == 0) {
            OutputDebugStringA("no channelmask specified and null number of channels wanted\n");
            COMReleaseExmut();
            return EA_BAD_MULTICHANNEL_FORMAT;
    };


    // LcChannels may be > 2, and ChannelMAsk = 0, because not standard
    if (LcChannels > 2) {
            LcPReq->isq.DeCodeForm1 |= (1 << BIT_FMP_MULTICHANNEL);
            if (LcChannelMask == 0)  
                for ( i = 0; i < LcChannels; i++)
                { 
                    LcChannelMask <<=1; 
                    LcChannelMask++;
                };
    }

    if (LcChannels != LcPipeChannels)        {
        CHAR test[200];
        sprintf(test,"INPUT Stream w/ %dCh cannot be recorded on pipe w/ %dch\n",LcChannels,LcPipeChannels);
            OutputDebugStringA(test);
            COMReleaseExmut();
            return EA_BAD_MULTICHANNEL_FORMAT;
    };

    // tell the DSP howmany channels
    LcPReq->isq.dwChannelMask = LcChannelMask;
    LcPReq->isq.DeCodeForm1 |= (LcChannels & 0x01FF);
    LcPReq->isq.DeCodeForm2 = LcHeader;
    LcPReq->isq.DeCodeForm3 = 0L;

	LcPReq->isq.dwSamplesPerChannel = PmStreamFormatInfo->sfSamplesPerChannel;

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcRet = LcPResp->rhCptr;
    COMReleaseExmut();
    // returning the error code
    return LcRet;
}

// ****************************************************************************
// WORD PCXPurgeStreamCommands()
// *****************************
//
// Input parameters :
// ****************
//
// PCX_HANDLE PmMyProcess: Handle of the calling process.
//
// DWORD PmPipeOutMask: Output pipe mask.
//
// DWORD PmPipeInMask: Input pipe mask.
//
// DWORD PmStreamMask: Stream mask.
//
// Return value :
// **************
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
//
// ****************************************************************************
WORD _CDECL_ PCXPurgeStreamCommands(
    IN  PCX_HANDLE PmMyProcess,
    IN  DWORD PmPipeOutMask,
    IN  DWORD PmPipeInMask,
    IN  DWORD PmStreamMask)
{
    LPPURGE_DCMDS_REQ_INFO LcPReq = 0;  // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( PIPE_CTRL_FAM,
                     PURGE_DCMDS_CMD,
                     (APP_HANDLE)PmMyProcess,
                     sizeof( PURGE_DCMDS_REQ_INFO ),
                     sizeof( RESP_HEADER_INFO ),
                     (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->pdqDiffered      = STREAM_TIME_MASK;    // Requests use stream time.
    LcPReq->pdqParam1        = PmStreamMask;
    LcPReq->pdqParam2        = 0;
    LcPReq->pdqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReq->pdqInPipeMask64.QuadPart  = PmPipeInMask;
    LcPReq->pdqCommandCode   = ALL_CMDS_NOTIFIED; // All differed cmds will be purged.

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );
    LcError = LcPResp->rhCptr;

    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXNotifyStreamTime()
// ***************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmOutPipeMask: Output pipe mask
//  DWORD PmInPipeMask: Input pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//  PTIME_INFO PmDelayCondition: the time of notification
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to ask the driver to tell when a stream reached
//  a given time or sample count
//
// ****************************************************************************
WORD _CDECL_ PCXNotifyStreamTime(
    IN  PCX_HANDLE  PmMyProcess,
    IN  DWORD       PmOutPipeMask,
    IN  DWORD       PmInPipeMask,
    IN  DWORD       PmStreamMask,
    IN  PTIME_INFO  PmDelayCondition)
{
    LPNOTIFY_STREAM_TIME_REQ_INFO LcPReq = 0; // will point to the beginning of the
                                              // request block
    LPRESP_HEADER_INFO LcPResp = 0;           // will point to the beginning of the
                                              // response block
    WORD LcError;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( STREAM_PIPE_CTRL_FAM,
                    NOTIFY_STREAM_TIME_CMD,
                    (APP_HANDLE)PmMyProcess,
                    sizeof( NOTIFY_STREAM_TIME_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->nstqOutPipeMask64.QuadPart = PmOutPipeMask;
    LcPReq->nstqInPipeMask64.QuadPart = PmInPipeMask;
    LcPReq->nstqStreamMask = PmStreamMask;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    // ------------------------------------------------------
    if ( PmDelayCondition )
    {
		LcPReq->nstqDiffered = (BYTE) (PmDelayCondition->tiDelayed | NOTIFIED_COMMAND_MASK | STREAM_TIME_MASK) ;
        LcPReq->nstqHour = PmDelayCondition->tiScheduler;
    }

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();

    // returning the error code
    return LcError;
}

// ****************************************************************************
// WORD PCXSetOutStreamEqualization()
// **************************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmOutPipeMask: Output pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//  PTIME_INFO PmDelayCondition: the time of notification
//  PCX_EFFECT_HANDLE  PmEffectID: Effect Identifier
//  DWORD PmStreamEqualInfoSize: the entries of the array
//  POUT_STREAM_EQUAL_INFO PmStreamEqualInfo: array of effect descriptors
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to tell the driver the requested equalization settings
//
// ****************************************************************************
WORD _CDECL_ PCXSetOutStreamEqualization(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeOutMask,
    IN  DWORD               PmStreamMask,
    IN  PTIME_INFO          PmDelayCondition,
    IN  PCX_EFFECT_HANDLE   PmEffectID,
    IN  DWORD               PmStreamEqualInfoSize,
    IN  POUT_STREAM_EQUAL_INFO PmStreamEqualInfo )
{
    LPGET_OUT_STREAM_EFFECTS_RESP_INFO LcPRespGet    = 0; // will point to the beginning of
                                                          // response block
    LPSET_OUT_STREAM_EFFECTS_REQ_INFO  LcPReqSet     = 0; // will point to the beginning of the
                                                          // request block
    LPRESP_HEADER_INFO                 LcPRespSet    = 0; // will point to the beginning of the
                                                          // response block
    DWORD                              LcBoardFrequency = 0;

    WORD                               LcError = SUCCESS;
    WORD                               i = 0;

    if( (PmMyProcess == 0) || (PmMyProcess > MAX_PCX_HANDLE) || (PmMyProcess == *g_pDHSAppHandleDLL) )
        return EA_INVALID_PCX_HANDLE;

    if(!PmStreamEqualInfo)
        return EA_INVALID_POINTER;

    // we need the clock information for some effects
    if( PmEffectID == EFFECT_EQUALIZATION_STREAM_LINEAR )
    {
        CLOCK_INFO  LcClo;
        LcClo.ckFrequency = 0;

        // Filter multichannel
        if (TbPipeInfo_OUT[ UTIDWMask2Word( PmPipeOutMask) ].piPipeNbChannels > 2)
            return EA_BAD_MULTICHANNEL_FORMAT;

        // get the current clock frequency on the board
        PCXGetPipeClock( PmMyProcess, PmPipeOutMask, 0, &LcClo );

        LcBoardFrequency = LcClo.ckFrequency;
    }

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

	
	// Read the current coeff for this stream.
    // ---------------------------------------
    LcError = GENGetOutStreamEqualization(	PmMyProcess,
											PmPipeOutMask,
											PmStreamMask,
											PmEffectID,
											&LcPRespGet );
   
											if ( LcError != SUCCESS )
											{
												COMReleaseExmut();
												return LcError;
											}

   // filling the header of the request block
   // ---------------------------------------
	COMFillBcHeader(	STREAM_PIPE_CTRL_FAM,
						SET_OUT_STREAM_EFFECTS_CMD,
						(APP_HANDLE)PmMyProcess,
						sizeof( SET_OUT_STREAM_EFFECTS_REQ_INFO ),
						sizeof( RESP_HEADER_INFO ),
						(BC_HEADER_HANDLE) &LcPReqSet );

   
	switch(PmEffectID)
    {
    case EFFECT_EQUALIZATION_STREAM_MPEG:


        // Initialize all the coeff with their current values.
        //
        for ( i = 0 ; i < MAX_EQUALIZATION_MPEG_COEFF; i++ )
        {
            LcPReqSet->sseqParam[i] = LcPRespGet->gserParam[i];
        }

        // Overwrite some coeff with the values given by the user.
        //
        for ( i = 0 ; ( LcError == SUCCESS ) && ( i < PmStreamEqualInfoSize ) ; i++ )
        {
            if ( PmStreamEqualInfo[i].seqValidity == TRUE )
            {
                DWORD LcBandNumber = PmStreamEqualInfo[i].seqBandNumber;

                if ( LcBandNumber >= MPEG_EQUAL_MAX_BANDS )
                {
                    LcError = EA_BAD_BAND_NUMBER;
                }
                else
                {
                    FLOAT   LcLevel = PmStreamEqualInfo[i].seqLevel;

                    LcLevel = MAX( LcLevel, MPEG_EQUAL_LEVEL_MIN );
                    LcLevel = MIN( LcLevel, MPEG_EQUAL_LEVEL_MAX );

                    GENMpegEqualLevelToCoeffForUserBand( LcBandNumber,
                                                         LcLevel,
                                                         LcPReqSet->sseqParam );
                }
            }
        }
        break;

    case EFFECT_EQUALIZATION_STREAM_LINEAR:

        if( PmStreamEqualInfoSize != 3 )
        {
            LcError = EA_INVALID_DATA_LENGTH;
            break;  // switch
        }
        if( LcBoardFrequency < 8000L )
        {
            LcError = EA_BAD_FREQUENCY_VALUE ;
            break;  // switch
        }

		// Initialize all the coeff with their current values.
        //
        for ( i = 0 ; i < MAX_EQUALIZATION_LINEAR_COEFF; i++ )
        {
            LcPReqSet->sseqParam[i] = LcPRespGet->gserParam[i];
        }

		// Overwrite some coeff with the values given by the user.
        //
        for ( i = 0 ; ( LcError == SUCCESS ) && ( i < PmStreamEqualInfoSize ) ; i++ )
        {
		  FLOAT LcFloatValue;

          if ( PmStreamEqualInfo[i].seqValidity == TRUE )
            {
                DWORD LcBandNumber = PmStreamEqualInfo[i].seqBandNumber;

                if ( LcBandNumber >= MPEG_EQUAL_MAX_BANDS )	// assuming it will remain 3 for both linear and mpeg
                {
                    LcError = EA_BAD_BAND_NUMBER;
                }
                else
                {
					// compute Gain:
					// it is converted from -18.0..+18.0 to 0..361
					// + conversion to DSP linear values
					//
					LcFloatValue = (PmStreamEqualInfo[i].seqLevel + 18.0f) * 10.0f + 0.5f;  // transform <-18..+18> to <0..2*18*10>
					if (LcFloatValue > 360.0f)	LcFloatValue = 360.0f;
					if (LcFloatValue < 0.0f)	LcFloatValue = 0.0f;
                    LcPReqSet->sseqParam[i] = Linear_DSP_Gain_LUT[ (DWORD)LcFloatValue] ;
                }
            }
        }


        break;  // switch

    default:
        LcError = EA_BAD_EFFECT_ID ;
    }

    // sending the request to the driver
    // ---------------------------------
    if ( LcError == SUCCESS )
    {
	
		// Filling the other fields of the request block
		//
    LcPReqSet->sseqOutPipeMask64.QuadPart = PmPipeOutMask;
    LcPReqSet->sseqStreamMask    = PmStreamMask;
    LcPReqSet->sseqEffectId      = PmEffectID;

    // ## FS (11/12/97) -- Accept NULL pointers for TIME_INFO
    // if not NULL initialize, otherwise let as is since
    // COMFillBcHeader has reset to 0 the request block
    //
    if ( PmDelayCondition )
    {
        LcPReqSet->sseqDiffered = 0;   // Not differed in that release
        LcPReqSet->sseqHour     = PmDelayCondition->tiScheduler;
    }


        COMCallDriver( (LPBC_HEADER_INFO) LcPReqSet,
                       0,
                       (RESP_HEADER_HANDLE) &LcPRespSet,
                       0 );

        LcError = LcPRespSet->rhCptr;
    
	} // no error

    COMReleaseExmut();

    // returning the error code
    return LcError;
}


// ****************************************************************************
// WORD PCXSetOutStreamMpegEqualization()
// **************************************
//
// Input parameters :
// ****************
//
//  PCX_HANDLE PmMyProcess: Handle of the calling process.
//  DWORD PmOutPipeMask: Output pipe mask
//  DWORD PmStreamMask: Stream mask for all pipes
//  PTIME_INFO PmDelayCondition: the time of notification
//  DWORD PmStreamEqualInfoSize: the entries of the abova array
//  POUT_STREAM_MPEG_EQUAL_INFO PmStreamEqualInfo: array of equal descriptors
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
//
// Function to tell the driver the requested equalization settings
//
// ****************************************************************************
WORD _CDECL_ PCXSetOutStreamMpegEqualization(
    IN  PCX_HANDLE          PmMyProcess,
    IN  DWORD               PmPipeOutMask,
    IN  DWORD               PmStreamMask,
    IN  PTIME_INFO          PmDelayCondition,
    IN  DWORD               PmStreamEqualInfoSize,
    IN  POUT_STREAM_MPEG_EQUAL_INFO PmStreamEqualInfo )
{
    // protection
    if( sizeof(PmStreamEqualInfo) != sizeof(POUT_STREAM_EQUAL_INFO))
    {
        return EA_INVALID_DATA_LENGTH;
    }
    
    return PCXSetOutStreamEqualization( PmMyProcess,
                                        PmPipeOutMask,
                                        PmStreamMask,
                                        PmDelayCondition,
                                        EFFECT_EQUALIZATION_STREAM_MPEG,
                                        PmStreamEqualInfoSize,
                                        (POUT_STREAM_EQUAL_INFO) PmStreamEqualInfo );
}


// ****************************************************************************
// WORD PCXBoostThisThread()
// ***************************
//
// Input parameters :
// ****************
//
// Return value :
// **************
//
// 0 if no error occured, an error code otherwise.
//
// ****************************************************************************
WORD _CDECL_ PCXBoostThisThread()
{
    LPSYSTEM_REQ_INFO LcPReq = 0;       // will point to the beginning of the
                                        // request block
    LPRESP_HEADER_INFO LcPResp = 0;     // will point to the beginning of the
                                        // response block
    WORD LcError;

    if (!COMWaitExmut())
       return EA_CANT_TAKE_MUTEX;

    // filling the header of the request block
    // ---------------------------------------
    COMFillBcHeader( GENE_FAM,
                    BOOST_THREAD_CMD,
                    (APP_HANDLE)0,
                    sizeof( SYSTEM_REQ_INFO ),
                    sizeof( RESP_HEADER_INFO ),
                    (BC_HEADER_HANDLE) &LcPReq );

    // Filling the other fields of the request block
    // ---------------------------------------------
    LcPReq->srqZero = 31;   // new priority of the thread (31 == HIGH_PRIORITY)

    // sending the request to the driver
    // ---------------------------------
    COMCallDriver( (LPBC_HEADER_INFO) LcPReq,
                   0,
                   (RESP_HEADER_HANDLE) &LcPResp,
                   0 );

    LcError = LcPResp->rhCptr;
    COMReleaseExmut();

    // returning the error code
    return LcError;
}

