//
// CreateFileMapping.cpp : This file implements a private, VISTA-compatible version of
//                         CreateFileMapping. The implementation requires Digigram shared
//                         memory allocation service.
//

#include <windows.h>
#include "ShMemAllocator.h"

extern "C" HANDLE DigigramCreateFileMapping(
    HANDLE hFile,
    LPSECURITY_ATTRIBUTES lpFileMappingAttributes,
    DWORD flProtect,
    DWORD dwMaximumSizeHigh,
    DWORD dwMaximumSizeLow,
    LPCSTR lpName)
{
    HANDLE hOrderSem = NULL;
    BOOL bReleaseOrderSem = FALSE;
    HANDLE hOrderShMem = NULL;
    HANDLE hOrderReadyEvent = NULL;
    HANDLE hDeliveredEvent = NULL;

    tShMemOrder* pShMemOrder = NULL;

    HANDLE hShMem = NULL;
    BOOL bReleaseShMem = FALSE;

    DWORD lastError = 0;
    __try
    {
        // Security attributes for IPC creations
        SECURITY_DESCRIPTOR Lc_SecurityDescriptor;
        SECURITY_ATTRIBUTES Lc_SecurityAttributes;

        Lc_SecurityAttributes.nLength = sizeof(SECURITY_ATTRIBUTES);
        Lc_SecurityAttributes.lpSecurityDescriptor = &Lc_SecurityDescriptor;
        Lc_SecurityAttributes.bInheritHandle = FALSE;

        InitializeSecurityDescriptor(&Lc_SecurityDescriptor, SECURITY_DESCRIPTOR_REVISION);
        SetSecurityDescriptorDacl(&Lc_SecurityDescriptor, TRUE, NULL, FALSE);

        // SHMEM_ORDER_SEM semaphore protects the processing of a shared memory order.
        // It assures only one shared memory order is processed at a time.
        // It is created by the service and used by the applications.
        hOrderSem =    CreateSemaphore(&Lc_SecurityAttributes,
                                       1, // InitialCount
                                       1, // MaximumCount
                                       SHMEM_ORDER_SEM);
        if (hOrderSem == NULL)
            goto _DigigramCreateFileMapping_error;
        if (GetLastError() != ERROR_ALREADY_EXISTS)
            // The object must have been created by the service
            goto _DigigramCreateFileMapping_error;

        // SHMEM_ORDER_SHMEM is the shared memory where the shared memory orders are located.
        // It is written by the applications and read by the serice.
        hOrderShMem = CreateFileMapping((HANDLE)0xFFFFFFFF,
                                        &Lc_SecurityAttributes,
                                        PAGE_READWRITE,
                                        0,
                                        sizeof(tShMemOrder),
                                        SHMEM_ORDER_SHMEM);
        if (hOrderShMem == NULL)
            goto _DigigramCreateFileMapping_error;
        if (GetLastError() != ERROR_ALREADY_EXISTS)
            // The object must have been created by the service
            goto _DigigramCreateFileMapping_error;

        pShMemOrder = (tShMemOrder*)MapViewOfFile(hOrderShMem,
                                                  FILE_MAP_READ |FILE_MAP_WRITE,
                                                  0,  // high offset: map from
                                                  0,  // low offset: beginning
                                                  0); // default: map entire file
        if (pShMemOrder == NULL)
            goto _DigigramCreateFileMapping_error;

        // SHMEM_ORDER_READY_EVT is the event set by the application to wake up the
        // service when a share memory order has been laced in SHMEM_ORDER_SHMEM
        hOrderReadyEvent = CreateEvent(&Lc_SecurityAttributes,
                                       TRUE,  // Manual reset
                                       FALSE, // Initial state
                                       SHMEM_ORDER_READY_EVT);
        if (hOrderReadyEvent == NULL)
            goto _DigigramCreateFileMapping_error;
        if (GetLastError() != ERROR_ALREADY_EXISTS)
            // The object must have been created by the service
            goto _DigigramCreateFileMapping_error;

        // SHMEM_DELIVERED_EVT is the event set by the service to wake up the
        // application when a share memory order has been delivered
        hDeliveredEvent = CreateEvent(&Lc_SecurityAttributes,
                                      TRUE,  // Manual reset
                                      FALSE, // Initial state
                                      SHMEM_DELIVERED_EVT);
        if (hDeliveredEvent == NULL)
            goto _DigigramCreateFileMapping_error;
        if (GetLastError() != ERROR_ALREADY_EXISTS)
            // The object must have been created by the service
            goto _DigigramCreateFileMapping_error;

        if (strlen(lpName) >= _SHMEM_NAME_MAX_SIZE_)
            goto _DigigramCreateFileMapping_error;

        WaitForSingleObject(hOrderSem, INFINITE);
        bReleaseOrderSem = TRUE;

        // Check version compatibility
        if (pShMemOrder->mySize < sizeof(tShMemOrder))
            goto _DigigramCreateFileMapping_error;

        strcpy(pShMemOrder->name, lpName);
        pShMemOrder->flags = flProtect;
        pShMemOrder->sizeHigh = dwMaximumSizeHigh;
        pShMemOrder->sizeLow = dwMaximumSizeLow;

        SetEvent(hOrderReadyEvent);

        WaitForSingleObject(hDeliveredEvent, INFINITE);
        ResetEvent(hDeliveredEvent);

        lastError = pShMemOrder->lastError;

        ReleaseSemaphore(hOrderSem, 1L, NULL);
        bReleaseOrderSem = FALSE;

        if ( (lastError == ERROR_SUCCESS) ||
             (lastError == ERROR_ALREADY_EXISTS) )
        {
            hShMem = CreateFileMapping((HANDLE)0xFFFFFFFF,
                                       &Lc_SecurityAttributes,
                                       flProtect,
                                       dwMaximumSizeHigh,
                                       dwMaximumSizeLow,
                                       lpName);
            if (GetLastError() != ERROR_ALREADY_EXISTS)
            {
                // The requested shared memory must have been created by the service
                bReleaseShMem = TRUE;
            }
        }

_DigigramCreateFileMapping_error:
        {
            // nop
        }

    } // __try

    __finally
    {
        if ( (bReleaseShMem) && (hShMem) )
        {
            CloseHandle(hShMem);
            hShMem = NULL;
        }
        if (hDeliveredEvent) CloseHandle(hDeliveredEvent);
        if (hOrderReadyEvent) CloseHandle(hOrderReadyEvent);
        if (pShMemOrder) UnmapViewOfFile(pShMemOrder);
        if (hOrderShMem) CloseHandle(hOrderShMem);
        if (bReleaseOrderSem) ReleaseSemaphore(hOrderSem, 1L, NULL);
        if (hOrderSem) CloseHandle(hOrderSem);
    }

    SetLastError(lastError);
    return(hShMem);
}
